/*
Copyright 1990-2008 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package com.sun.g11n.vkb.geom.xkb;

import static com.sun.g11n.vkb.Main.*;
import com.sun.g11n.vkb.*;
import com.sun.g11n.vkb.geom.*;
import com.sun.g11n.vkb.geom.xml.XMLGeometryWriter;
import java.io.*;
import java.util.*;

/**
 *
 * @author naoyuki
 */
public class XKBGeometryCollection implements GeometryCollection {
    private static String USAGE = "Usage: java -cp VKB.jar com.sun.g11n.vkb.XKBGeometry -geomdir <xkb geometry dir>";
    private static void usage() {
        System.err.println(USAGE);
        System.exit(1);
    }
    private Map<String, Map<String, Geometry>> geometryMap;
    
    @Override public Set<Geometry> getGeometrySet() {
        Set<Geometry> geoms = new HashSet<Geometry>();
        Set<String> keys = geometryMap.keySet();
        for (String fileName : keys) {
            Map<String, Geometry> geomMap = geometryMap.get(fileName);
            Collection<Geometry> values = geomMap.values();
            geoms.addAll(values);
        }
        return geoms;
    }
    
    @Override public Geometry getGeometryByName(String name) {
        if (name == null) {
            return null;
        }
        Geometry geom = null;
        Collection<String> files = geometryMap.keySet();
        for (String file : files) {
            Map<String, Geometry> map = geometryMap.get(file);
            geom = map.get(name);
            if (geom != null) {
                break;
            }
        }
        return geom;
    }

    private static GeometryCollection geometryCollection;
    
    /*
     * Entry point of this class and returns all of keyboard
     * geometry data as GeometryCollection which is just 
     * Set<Geometry>. Each Geometry represents one geometry like
     * 'Generic 101', or 'Sun Type5 Unix'. Geometry is represented
     * by GeomInst inner class of this class.
     */
    public static GeometryCollection getInstance(String[] args) {
        main(args);
        return geometryCollection;
    }
    
    public static void main(String[] args) {
        File geomDir = null;
        File outDir = null;
        Set<String> readTypes = new HashSet<String>();
        
        initProperties();
        
        for (int i = 0, len = args.length; i < len; i++) {
            if (args[i].equals("-help")) {
                usage();
            }
            // -geomdir specifies XKB geometry data directry
            if (args[i].equals("-geomdir")) {
                if (++i < len) {
                    geomDir = new File(args[i]);
                    continue;
                }
                usage();
            }
            // -types specifies file name under XKB geometry 
            // directly like pc or sun
            if (args[i].equals("-type")) {
                if (++i < len) {
                    readTypes.add(args[i]);
                    continue;
                }
                usage();
            }
            if (args[i].equals("-outdir")) {
                if (++i < len) {
                    outDir = new File(args[i]);
                    continue;
                }
                usage();
            }
        }
        if (geomDir == null || !geomDir.isDirectory()) {
            usage();
        }
        
        geometryCollection = new XKBGeometryCollection(geomDir, readTypes);
        if (outDir != null) {
            new XMLGeometryWriter(outDir, geometryCollection).writeOut();
        }
    }
    
    // for debug
    static String curTypeName;
    
    /**
     * Represents all of geometry data.
     * Create Set of Geometry (GeomInst)
     * 
     * @param dir: XKB geometry data directry
     * @param types: file name in XKB geometry directory
     */
    private XKBGeometryCollection(File dir, Set<String> types) {
        geometryMap = new HashMap<String, Map<String, Geometry>>();
        //ParseUtil.setScale(Main.getDefaultScaleFactor());
        
        String[] fileList = dir.list();
        for (String fname : fileList) {
            // TODO: should support sub-directories types later
            if (types.size() != 0 && !types.contains((fname))) {
                continue;
            }
            try {
                Map<String, Geometry> typeGeomMap = new HashMap<String, Geometry>();
                geometryMap.put(fname, typeGeomMap);
                
                FileInputStream fis = new FileInputStream(new File(dir, fname));
                InputStreamReader isr = new InputStreamReader(fis);
                BufferedReader br = new BufferedReader(isr);
                String line = null;
                boolean defaultType = false;
                while ((line = br.readLine()) != null) {
                    line = line.trim();
                    if (line.startsWith(COMMENT_TAG)) {
                        // skip comments
                        continue;
                    }
                    if (line.startsWith(DEFAULT_TAG)) {
                        defaultType = true;
                        line = line.substring(DEFAULT_TAG.length()).trim();
                    }
                    if (line.startsWith(BEGIN_TAG)) {
                        // getTypeName assumes xkb_geometry line
                        // contains its name and begin mark '{'
                        String typeName = getTypeName(line);
                        // new GeomInst will consume rest of type
                        // definition. Otherwise it must something
                        // went wrong...
                        if (typeName == null) {
                            error(" typeName is null");
                        }                        
                        curTypeName = typeName;
                        // This makes one geometry like 'Generic 101' as 
                        // Geometry object (GeomInst)
                        GeomInst inst = new GeomInst(br, geometryMap);
                        inst.setDefault(defaultType);
                        inst.setScale(Main.getDefaultScaleFactor());
                        typeGeomMap.put(typeName, inst);
                    }
                }
            } catch (IOException e) {
                e.printStackTrace();
                continue;
            }
        }
    }

    private String getTypeName(String line) {
        StringTokenizer st = new StringTokenizer(line);
        int numToken = st.countTokens();
        if (numToken != 3) {
            error("The token count at xkb_geometry line is not 3.");
        }
        st.nextElement(); // skip
        String name = (String)st.nextElement();
        if (name.startsWith("\"") && name.endsWith("\"")) {
            return name.substring(1, name.length() - 1);
        }
        return null;
    }
    
    static private class GeomInst implements Geometry, Comparable {
        private boolean defaultType;
        private String description;
        private float width;
        private float height;
        private float scale;
        private Map<String, KeyShape> shapeMap;
        private float section_left;
        private float row_left;
        private String default_shape;
        private float key_gap;
        private Map<String, KeySection> sectionMap;
        private Map<String, Map<String, Geometry>> includeMap;
        private boolean inclusion;
        private String includeGeom;
        private String includeType;
        private Geometry inclusionGeom;
        
        /**
         * Read type infomation and make layout structure
         * 
         * @param reader
         * @throws java.io.IOException
         */
        GeomInst(BufferedReader reader, Map<String, Map<String, Geometry>> includeMap) throws IOException {
            this.includeMap = includeMap;
            defaultType = false;
            description = null;
            shapeMap = new HashMap<String, KeyShape>();
            sectionMap = new HashMap<String, KeySection>();
            inclusion = false;
            
            String line = null;
            while ((line = reader.readLine()) != null) {
                line = line.trim();
                if (line.startsWith(COMMENT_TAG)) {
                    // skip comments
                    continue;
                }
                if (line.startsWith(INCLUDE_TAG)) {
                    inclusion = true;
                    /*
                     * incluce "pc(pc101)" -> includeGeom = pc, includeType = pc101
                     */
                    includeGeom = XKBParser.parseIncludeGeom(line);
                    includeType = XKBParser.parseIncludeType(line);
                    inclusionGeom = null;
                }
                if (line.startsWith(DESCRIPTION_TAG)) {
                    description = XKBParser.parseString(line, reader);
                    continue;
                }
                if (line.startsWith(WIDTH_TAG)) {
                    width = XKBParser.parseFloat(line, reader);
                    continue;
                }
                if (line.startsWith(HEIGHT_TAG)) {
                    height = XKBParser.parseFloat(line, reader);
                    continue;
                }
                if (line.startsWith(SHAPE_TAG)) {
                    XKBParser.parseShape(shapeMap, line, reader);
                    continue;
                }
                if (line.startsWith(OUTLINE_TAG)) {
                    XKBParser.parseOutline(line, reader);
                    continue;
                }
                if (line.startsWith(SECTION_LEFT_TAG)) {
                    section_left = XKBParser.parseFloat(line, reader);
                    continue;
                }
                if (line.startsWith(ROW_LEFT_TAG)) {
                    row_left = XKBParser.parseFloat(line, reader);
                    continue;
                }
                if (line.startsWith(ROW_LEFT_TAG)) {
                    row_left = XKBParser.parseFloat(line, reader);
                    continue;
                }
                if (line.startsWith(KEY_SHAPE_TAG)) {
                    default_shape = XKBParser.parseString(line, reader);
                    continue;
                }
                if (line.startsWith(KEY_GAP_TAG)) {
                    key_gap = XKBParser.parseFloat(line, reader);
                    continue;
                }
                if (line.startsWith(SECTION_TAG)) {
                    XKBParser.parseSection(sectionMap, line, reader);
                    continue;
                }
                if (line.startsWith(SOLID_TAG)) {
                    XKBParser.parseOutline(line, reader);
                    continue;
                }
                if (line.startsWith(END_STATEMENT2_TAG)) {
                    break;
                }
                // do nothing for other tags like indicator, text
            }
        }
        void setDefault(boolean def) {
            defaultType = def;
        }
        boolean isDefault() {
            return defaultType;
        }
        void setDescription(String desc) {
            description = desc;
        }
        @Override public String getDescription() {
            return description;
        }

        private void checkInclude() {
            if (inclusion && inclusionGeom == null) {
                Map<String, Geometry> gmap = includeMap.get(includeGeom);
                // System.out.println(" INCLUDE!" + includeGeom + ", type = " + includeType);
                if (gmap == null) {
                    error("\"" + includeGeom + "\" inclusion in " + description + " is not appropriate.");
                }
                inclusionGeom = gmap.get(includeType);      
                if (inclusionGeom == null) {
                    error(includeType + " inclusion in " + description + " is not appropriate");
                }
                width = inclusionGeom.getWidth();
                height = inclusionGeom.getHeight();
                sectionMap = inclusionGeom.getSectionMap();
                shapeMap = inclusionGeom.getShapeMap();
                default_shape = inclusionGeom.getDefaultShape();
                section_left = inclusionGeom.getSectionLeft();
                row_left = inclusionGeom.getRowLeft();
                key_gap = inclusionGeom.getDefaultGap();
            }
        }
        
        @Override public float getWidth() {
            checkInclude();
            return width;                
        }
        @Override public float getHeight() {
            checkInclude();
            return height;
        }
        @Override public Map<String, KeySection> getSectionMap() {
            checkInclude();
            return sectionMap;
        }
        @Override public KeySection getDefaultSection() {
            checkInclude();
            return sectionMap.get(ALPHA_TAG);
        }
        @Override public Collection<KeySection> getActiveSections() {
            Collection<KeySection> set = sectionMap.values();
            return set;
        }
        @Override public Map<String, KeyShape> getShapeMap() {
            checkInclude();
            return shapeMap;
        }
        @Override public KeyShape getShape(String name) {
            checkInclude();
            return shapeMap.get(name);
        }
        @Override public String getDefaultShape() {
            checkInclude();
            return default_shape;
        }
        @Override public float getSectionLeft() {
            checkInclude();
            return section_left;
        }
        @Override public float getRowLeft() {
            checkInclude();
            return row_left;
        }
        @Override public float getDefaultGap() {
            checkInclude();
            return key_gap;
        }
        @Override public void setScale(float scale) {
            this.scale = scale;
        }
        @Override public float getScale() {
            return scale;
        }
        
        @Override public String toString() {
            return description;
        }
        
        @Override public int compareTo(Object g) {
            return toString().compareTo(g.toString());
        }
    }
    
    static final String COMMENT_TAG = "//";
    static final String BEGIN_TAG = "xkb_geometry";
    static final String DEFAULT_TAG = "default";
    static final String INCLUDE_TAG = "include";
    static final String DESCRIPTION_TAG = "description";
    static final String WIDTH_TAG = "width";
    static final String HEIGHT_TAG = "height";
    static final String SHAPE_TAG = "shape";
    static final String CORNER_RADIUS_TAG = "cornerRadius";
    static final String APPROX_TAG = "approx";
    static final String SECTION_TAG = "section";
    static final String SECTION_LEFT_TAG = "section.left";
    static final String ROW_LEFT_TAG = "row.left";
    static final String KEY_SHAPE_TAG = "key.shape";
    static final String KEY_GAP_TAG = "key.gap";
    static final String KEY_COLOR_TAG = "key.color";
    static final String OUTLINE_TAG = "outline";
    static final String LEFT_TAG = "left";
    static final String TOP_TAG = "top";
    static final String GAP_TAG = "gap";
    static final String ALPHA_TAG = "Alpha";
    static final String ROW_TAG = "row";
    static final String KEYS_TAG = "keys";
    static final String SOLID_TAG = "solid";
    static final String INDICATOR_TAG = "indicator";
    static final String ON_COLOR_TAG = "onColor";
    static final String OFF_COLOR_TAG = "offColor";
    static final String TEXT_TAG = "text";
    static final String COLOR_TAG = "color";
    static final String ROW_VERTICAL_TAG = "row.vertical";
    static final String END_STATEMENT_TAG = ";";
    static final String END_STATEMENT2_TAG = "};";
    static final String START_STATEMENT_TAG = "{";
    static final String END_STATEMENT3_TAG = "}";
    static final String KEY_LABEL_START_TAG = "<";
    static final String KEY_LABEL_END_TAG = ">";
}
