/*
Copyright 1990-2006 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/
#include <stdlib.h>
#include <libxml/parser.h>

#include "kbltrans.h"
#include "kbltrans_impl.h"

extern UTF16_CHAR *utf8_to_utf16(char *);

static KBLTR_layout *all_layouts;
static int additional_type_id = _MAX_LAYOUT;
static int current_size;
#define INCREMENT_SIZE 10
#define DATAFILE "layoutdata.xml"

static KBLTR_layout *ext_layouts;
static int ext_size = 0;

/*
 * return modifier level
 *   0: -> no shift / no altgraph
 *   1: -> shift / no altgraph
 *   2: -> no shift / altgraph
 *   3: -> shft / altgraph
 */
static int
get_level (int modifier) {
  int level = 0;

  if (modifier & KBLTR_SHIFT) {
    level++;
  }
  if (modifier & KBLTR_ALT_GRAPH) {
    level += 2;
  }
  return level;
}

static int
is_numpad (int keycode) {
  if (keycode >= 0x60 && keycode <= 0x6f) {
    return 1;
  }
  return 0;
}

static int
capitalize (int letter) {
  int upper_case = letter;

  if (letter >= 0x61 && letter <= 0x7a) {
    /* Basic Latin */
    upper_case -= 0x20;
  } else if (letter >= 0xe0 && letter <= 0xfe) {
    /* Latin-1 Supplement */
    upper_case -= 0x20;
  } else if ((letter >= 0x101 && letter <= 0x137) ||
	     (letter >= 0x14b && letter <= 0x177)) {
    /* Latin Extended-A part-1 */
    if (letter % 2 == 1) {
      upper_case--;
    }
  } else if ((letter >= 0x13a && letter <= 0x148) ||
	     (letter >= 0x17a && letter <= 0x17e)) {
    /* Latin Extended-A part-2 */
    if (letter % 2 == 0) {
      upper_case--;
    }
  } else if (letter >= 0x3ac && letter <= 0x3fb) {
    /* Greek and Coptic */
    if ((letter >= 0x3b1 && letter <= 0x3c1) ||
	(letter >= 0x3c3 && letter <= 0x3ce)) {
      upper_case -= 0x20;
    } else {
      switch (letter) {
      case 0x3c2:
	upper_case = 0x3a3;
	break;
      case 0x3ac:
	upper_case = 0x386;
	break;
      case 0x3ad:
	upper_case = 0x388;
	break;
      case 0x3ae:
	upper_case = 0x389;
	break;
      case 0x3af:
	upper_case = 0x38a;
	break;
      case 0x3d1:
	upper_case = 0x3f4;
	break;
      case 0x3d9:
	upper_case = 0x3d8;
	break;
      case 0x3db:
	upper_case = 0x3da;
	break;
      case 0x3dd:
	upper_case = 0x3dc;
	break;
      case 0x3df:
	upper_case = 0x3df;
	break;
      case 0x3e3:
	upper_case = 0x3e2;
	break;
      case 0x3e5:
	upper_case = 0x3e4;
	break;
      case 0x3e7:
	upper_case = 0x3e6;
	break;
      case 0x3e9:
	upper_case = 0x3e8;
	break;
      case 0x3eb:
	upper_case = 0x3ea;
	break;
      case 0x3ed:
	upper_case = 0x3ec;
	break;
      case 0x3ef:
	upper_case = 0x3ee;
	break;
      case 0x3f2:
	upper_case = 0x3f9;
	break;
      case 0x3f8:
	upper_case = 0x3f7;
	break;
      case 0x3fb:
	upper_case = 0x3fa;
	break;
      }
    }
  } else if (letter >= 0x430 && letter <= 0x44f) {
    /* Basic Russian alphabet */
    upper_case -= 0x20;
  } else if (letter >= 0x450 && letter <= 0x45f) {
    /* Cyrillic extensions */
    upper_case -= 0x50;
  } else if (letter >= 0x461 && letter <= 0x47f) {
    /* Historic cyrillic letters */
    if (letter % 2 == 1) {
      upper_case--;
    }
  }

  return upper_case;
}

static char *
get_node_value(xmlDocPtr doc, xmlNodePtr node) {
  xmlChar *val = xmlNodeListGetString (doc, node->xmlChildrenNode, 1);
  return xstrdup ((const char *)val);
}

static CODES *
get_codes (xmlDocPtr doc, xmlNodePtr node) {
  CODES *codes = XCALLOC(CODES, 1);
  codes->location_code_name =
    xstrdup ((const char *)xmlGetProp(node, (xmlChar *)"sym"));
  xmlNodePtr cur = node->xmlChildrenNode;
  while (cur != NULL) {
    int val = (int)strtol(get_node_value (doc, cur), (char **)NULL, 16);
    if (!strcmp("c1", (const char *)cur->name)) {
      codes->g1l1 = val;
    } else if (!strcmp("c2", (const char *)cur->name)) {
      codes->g1l2 = val;
    } else if (!strcmp("c3", (const char *)cur->name)) {
      codes->g2l1 = val;
    } else if (!strcmp("c4", (const char *)cur->name)) {
      codes->g2l2 = val;
    } else if (!strcmp("code", (const char *)cur->name)) {
      codes->code = val;
    } else if (!strcmp("altcode", (const char *)cur->name)) {
      codes->altcode = val;
    }
    cur = cur->next;
  }

  if (codes->g1l2 == 0 && codes->g1l1 != 0)
    codes->g1l2 = codes->g1l1;
  if (codes->g2l1 == 0 && codes->g1l1 != 0)
    codes->g2l1 = codes->g1l1;
  if (codes->g2l2 == 0 && codes->g1l2 != 0)
    codes->g2l2 = codes->g1l2;

  return codes;
}

static int
get_index_from_name (const char *name)
{
  int i;
  int len = _MAX_LAYOUT * 2;
  for (i = 0; i < len; i++) {
    if (!strcasecmp (name, layout_name[i])) {
      return i / 2;
    }
  }

  return additional_type_id++;
}

/*---- public interface ----*/ 

char ** predefined_layout_list;

int
kbltrans_init()
{
  /* get data xml */
  xmlDocPtr dataDoc;
  xmlNodePtr cur, codeNode, charNode;
  List *typeList;
  KBLTR_layout layout;
  int i;

  set_program_name("kbltrans");

  dataDoc = xmlParseFile (KBCONFDIR "/" DATAFILE);
  if (dataDoc == NULL) {
    /* init failed and the client can not use iiimkb functioanlity */
    kbltrans_error(DATAFILE " is not found");
    return -1;
  }
  cur = xmlDocGetRootElement (dataDoc);
  if (cur == NULL) {
    xmlFreeDoc (dataDoc);
    kbltrans_error(DATAFILE " is corrupted");
    return -1;
  }
  
  if (xmlStrcmp (cur->name, (const xmlChar *)"kbltrans")) {
    xmlFreeDoc (dataDoc);
    kbltrans_error(DATAFILE " is corrupted");
    return -1;
  }
    
  all_layouts = XCALLOC(KBLTR_layout, _MAX_LAYOUT);
  current_size = _MAX_LAYOUT;

  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    CODES *codes;
    if (!xmlStrcmp(cur->name, (const xmlChar *)"lo")) {
      xmlChar *typename = xmlGetProp(cur, (xmlChar *)"type");
      int typeID = get_index_from_name ((const char *)typename);
      xmlChar *alg = xmlGetProp(cur, (xmlChar *)"alg");

      codeNode = cur->xmlChildrenNode;
      if (typeID >= current_size) {
	all_layouts = XREALLOC(KBLTR_layout, all_layouts,
			       (current_size = typeID + INCREMENT_SIZE));
      }
      typeList = NULL;
      while (codeNode != NULL &&
	     !xmlStrcmp(codeNode->name, (xmlChar *)"key")) {
	codes = get_codes (dataDoc, codeNode);
	if (typeList == NULL) {
	  typeList = list_new(codes);
	} else {
	  list_add (typeList, list_new(codes));
	}
	codeNode = codeNode->next;
      }
      layout = XCALLOC(struct _KBLTR_layout, 1);
      layout->type = BUITIN;
      layout->name = xstrdup ((const char *)typename);
      layout->alg = (alg != NULL && !xmlStrcmp(alg, (xmlChar *)"true")) ? 1 : 0;
      layout->list = typeList;
      all_layouts[typeID] = layout;

      cur = cur->next;
    } else {
      kbltrans_warning(DATAFILE " constins unknown elements");
    }
  }
  predefined_layout_list = XCALLOC(char *, _MAX_LAYOUT + 1);
  for (i = 0; i < _MAX_LAYOUT; i++) {
    predefined_layout_list[i] = layout_name[i * 2];
  }

  return 0;
}

KBLTR_context
kbltrans_get_context (const KBLTR_layout src, const KBLTR_layout tgt)
{
  KBLTR_context ctx = XCALLOC (struct _KBLTR_context, 1);
  ctx->source = src;
  ctx->target = tgt;
  return ctx;
}

void
kbltrans_free_context (KBLTR_context ctx)
{
  /* do not free layout parts, they are allocated once and re-used */
  free (ctx);
}

void
kbltrans_free_keyevent (KBLTR_keyevent *kev)
{
  /* kev->str should not be freed, they are allocated only once
     at ext_layout initialazatioon timing */
  free (kev);
}

const KBLTR_layout
kbltrans_get_layout_from_id (int id)
{
  if (id < 0 || id > _MAX_LAYOUT) {
    id = DEFAULT_ID;
  }
  return kbltrans_get_layout (layout_name[id * 2]);
}

static KBLTR_layout
find_custom_layout (const char *typename)
{
  int i;
  
  for (i = 0; i < ext_size; i++) {
    if (!strcmp (ext_layouts[i]->name, typename)) {
      return ext_layouts[i];
    }
  }
  return NULL;
}

const KBLTR_layout
kbltrans_get_layout (const char *typename)
{
  int i, id, len;

  KBLTR_layout layout = NULL;

  if (KBLTRANS_USER_LT (typename, strlen(typename))) {
    layout = find_custom_layout (typename);
  }
  if (layout == NULL) {
    id = get_index_from_name(typename);
    if (all_layouts[id] == NULL) {
      kbltrans_error ("There is no requested data");
      layout = all_layouts[get_index_from_name("US/English")];
    } else {
      layout = all_layouts[id];
    }
  }

  return layout;
}

int
kbltrans_trans_char_to_code (const KBLTR_layout layout, int keychar, int modifier)
{
  List *l = NULL;
  List *data = layout->list;
  for (l = data; l != NULL; l = l->next) {
    CODES *c = (CODES *)l->userdata;
    int ch = 0;
    int level = get_level(modifier);

    switch(level) {
    case 0:
      ch = c->g1l1;
      break;
    case 1:
      ch = c->g1l2;
      break;
    case 2:
      if (c->g2l1 != 0)
	ch = c->g2l1;
      else
	ch = c->g1l1;
      break;
    case 3:
      if (c->g2l2 != 0)
	ch = c->g2l2;
      else
	ch = c->g1l2;
    }
    if (keychar == ch)
      return c->code;
  }
  /* not found */
  return 0;
}

/*
 * return 1 means, this source layout default is AltGraph
 * position (new Russian/Arabic/Greek variant layout)
 */
static int
check_altgr_swap (KBLTR_layout layout, int modifier, int keyChar) {
  if (layout->alg == 1 && (modifier & KBLTR_ALT_GRAPH) == 0 &&
      keyChar > 0xff) {
    /* This means non AltGraph state produses AltGraph position char
       with Arabic/Russian/Greek variant layout keyboard */
    return 1;
  }

  return 0;
}

static char *
find_location_name (KBLTR_layout source, const KBLTR_keyevent *kev)
{
  int srcChar = kev->keychar;
  int srcCode = kev->keycode;
  int level;
  List *srcList;
  List *l = NULL;
  char *lcn = NULL;
  int caps = (kev->modifier & CAPS_LOCK_MASK) ? 1 : 0;
  int swap_mode = check_altgr_swap (source, kev->modifier, kev->keychar);
  int modifier = kev->modifier;

  srcList = source->list;
  
  if (swap_mode == 1) {
    modifier ^= KBLTR_ALT_GRAPH;
  }

  level = get_level (modifier);

  /* search source location_code_name with srcCode first
     for duplicate defined character on one keyboard.
     Currently it's only EuroSign (0x20ac). */
  if (srcChar == 0x20ac) {
    for (l = srcList; l != NULL; l = l->next) {
      CODES *c = (CODES *)l->userdata;
      if (c->code == srcCode) {
	lcn = c->location_code_name;
	break;
      }
    }
  } else {
    /* search soruce location_code_name */
    for (l = srcList; l != NULL; l = l->next) {
      CODES *c = (CODES *)l->userdata;
      int charData = 0;
      switch(level) {
      case 0:
	charData = c->g1l1;
	break;
      case 1:
	charData = c->g1l2;
	break;
      case 2:
	charData = c->g2l1;
	if (charData == 0)
	  charData = c->g1l1;
	break;
      case 3:
	charData = c->g2l2;
	if (charData == 0)
	  charData = c->g1l2;
      default:
	charData = c->g1l1;
      }

      if (srcChar == charData ||
	  (caps == 1 && capitalize (charData) == srcChar)) {
	lcn = c->location_code_name;
	break;
      }
    }
  }
  
  if (lcn == NULL) {
    /* search different level of char for unset Caps by client */
    for (l = srcList; l != NULL; l = l->next) {
      CODES *c = (CODES *)l->userdata;
      int charData = 0;
      switch (level) {
      case 0:
	if (c->g1l2 == srcChar) {
	  lcn = c->location_code_name;
	  break;
	}
      case 1:
	if (c->g1l1 == srcChar) {
	  lcn = c->location_code_name;
	  break;
	}
      case 2:
	if (c->g2l2 == srcChar) {
	  lcn = c->location_code_name;
	  break;
	}
      case 3:
	if (c->g2l1 == srcChar) {
	  lcn = c->location_code_name;
	  break;
	}
      }
      if (lcn != NULL) {
	caps = 1;
	break;
      }
    }
  }

  if (lcn == NULL) {
    /* search all level of char */
    for (l = srcList; l != NULL; l = l->next) {
      CODES *c = (CODES *)l->userdata;
      if (srcChar == c->g1l1 || srcChar == c->g1l2 ||
	  srcChar == c->g2l1 || srcChar == c->g2l2) {
	lcn = c->location_code_name;
      }
    }
  }

  return lcn;
}

static CODES_EXT *
get_custom_trans_data (KBLTR_layout target, char *location_name)
{
  List *l;
    
  if (target->type != CUSTOM) {
    return NULL;
  }

  for (l = target->list; l != NULL; l = l->next) {
    CODES_EXT *c = (CODES_EXT *)l->userdata;

    if (!strcmp (c->location_code_name, location_name)) {
      return c;
    }
  }

  return NULL;
}

static int
trans_custom_layout (KBLTR_context ctx, const KBLTR_keyevent *kev, KBLTR_keyevent *new_kev)
{
  char *lcn;
  CODES_EXT *trans_data;
  int level = get_level (kev->modifier);

  lcn = find_location_name (ctx->source, kev);
  trans_data = get_custom_trans_data (ctx->target, lcn);
  if (trans_data == NULL) {
    /* not processed */
    return 0;
  }

  switch (level) {
  case 0:
    new_kev->str = trans_data->g1l1;
    break;
  case 1:
    new_kev->str = trans_data->g1l2;
    break;
  case 2:
    new_kev->str = trans_data->g2l1;
    break;
  case 3:
    new_kev->str = trans_data->g2l2;
    break;
  }
  if (new_kev->str != NULL) {
    new_kev->type = STRING;
    return 1;
  }

  /*
   * translation should be fall back to built-in source layout
   */
  return 0;
}


/*
 * main function of this module. convert keyevent so that it's generated
 * on the emulation target layout keyboard.
 */
KBLTR_keyevent *
kbltrans_trans_keyevent (KBLTR_context ctx, const KBLTR_keyevent *kev)
{
  int srcChar = kev->keychar;
  int srcCode = kev->keycode;
  int level = get_level (kev->modifier);
  List *tgtList;
  List *l;
  int caps = (kev->modifier & CAPS_LOCK_MASK) ? 1 : 0;
  int num_pad = is_numpad (kev->keycode);
  int tgtCode, tgtChar;
  char *lcn;
  KBLTR_keyevent *new_kev = XCALLOC (KBLTR_keyevent, 1);

  new_kev->modifier = kev->modifier;
  new_kev->type = KEYEVENT; /* may be overriden by trans_custom_layout () */

  if (srcChar == 0 || num_pad == 1 || !ctx || !(ctx->source) || !(ctx->target)) {
    /* if context is invalid, returns with no translation with error */
    new_kev->keycode = kev->keycode;
    new_kev->keychar = kev->keychar;
    return new_kev;
  }

  if (ctx->target->alg == 1 && kev->modifier & (KBLTR_CONTROL|KBLTR_ALT) &&
      kev->keychar < 127) {
    new_kev->keycode = kev->keycode;
    new_kev->keychar = kev->keychar;
    return new_kev;
  }

  lcn = find_location_name (ctx->source, kev);

  if (lcn == NULL) {
    kbltrans_error ("Can not find source keyevent in " DATAFILE);
    new_kev->keycode = kev->keycode;
    new_kev->keychar = kev->keychar;
    return new_kev;
  }

  /* if target layout is custom, then process it first. And if no translaion
     done in custom part, change target to source layout and do normal translation */
  if (ctx->target->type == CUSTOM) {
    int process = trans_custom_layout (ctx, kev, new_kev);
    if (process) {
      return new_kev;
    }
    ctx->target = kbltrans_get_layout (ctx->target->source);
    if (!ctx->target) {
      new_kev->keycode = kev->keycode;
      new_kev->keychar = kev->keychar;
      return new_kev;
    }
    /* continue translation with not customized part of layout */
  }

  tgtList = ctx->target->list;

  /* if layout data has 'alg' attribute, then 2nd group of character
     will be treated as 1st group, and 'AltGraph/ModeSwitch' works
     to switch 2nd groupt to 1st group. */
  if (ctx->target->alg == 1 && strcmp (ctx->source->name, ctx->target->name)) {
    level = (level + 2) % 4;
  }

  /* search target keyinfo */
  for (l = tgtList; l != NULL; l = l->next) {
    CODES *c = (CODES *)l->userdata;
      
    if (!strcmp(c->location_code_name, lcn)) {
      new_kev->keycode = c->code;
      switch(level) {
      case 0:
	new_kev->keychar = c->g1l1;
	break;
      case 1:
	new_kev->keychar = c->g1l2;
	break;
      case 2:
	new_kev->keychar = c->g2l1;
	break;
      case 3:
	new_kev->keychar = c->g2l2;
	break;
      default:
	new_kev->keychar = c->g1l1;
	kbltrans_error("level has invalid value");
      }
      break;
    }
  }

  if (caps == 1) {
    /* ensure caps treatment, because not all keyboard has
     * caps character at shift position
     */
    new_kev->keychar = capitalize (new_kev->keychar);
  }
  
  return new_kev;
}

char *
kbltrans_id_to_text (int id)
{
  if (id < 0 || id > _MAX_LAYOUT) {
    id = DEFAULT_ID;
  }
  return layout_name[id * 2];
}

int
kbltrans_text_to_id (const char *typename)
{
  int i;
  for (i = 0; i < _MAX_LAYOUT; i++) {
    if (!strcmp (layout_name[i * 2], typename)) {
      return i;
    }
  }

  
  return 0;
}

/* dynamic layout extension mechanism */
void
kbltrans_add_custom_layout (char *typename, char *source_typename, int mod_key_num,
			    char *key_str_info[], int key_code_info[]) {
  int i, j;
  List *list = NULL;
  struct _KBLTR_layout *layout;

  for (i = 0; i < mod_key_num; i++) {
    CODES_EXT *codes = XCALLOC(CODES_EXT, 1);
    j = i * 5;

    codes->location_code_name = xstrdup (key_str_info[j]);
    codes->g1l1 = utf8_to_utf16 (key_str_info[j + 1]);
    codes->g1l2 = utf8_to_utf16 (key_str_info[j + 2]);
    codes->g2l1 = utf8_to_utf16 (key_str_info[j + 3]);
    codes->g2l2 = utf8_to_utf16 (key_str_info[j + 4]);
    codes->code = key_code_info[i];

    if (list == NULL) {
      list = list_new (codes);
    } else {
      list_add (list, list_new (codes));
    }
  }
  layout = XCALLOC (struct _KBLTR_layout, 1);
  layout->type = CUSTOM;
  layout->name = xstrdup ((const char *)typename);
  layout->source = xstrdup ((const char *)source_typename);
  layout->list = list;

  ext_layouts = xrealloc (ext_layouts, sizeof (struct _KBLTR_layout *) * (ext_size + 1));
  ext_layouts[ext_size] = layout;
  ext_size++;
  
}

#if sun
/*
 * translate platform specific id to kbltrans layout id
 */
static int
normalize_id (const int pid, const int type)
{
  int id;
  if (type == 4 || type == 5 || type == 101) {
    if (id > TYPE45_MAX) {
      return DEFAULT_ID;
    }
    id = platform_layout_type45[pid];
    if (id == _NO_DEF_) {
      return DEFAULT_ID;
    }
    return id;
  }
  id = DEFAULT_ID;
  if (pid <= TYPE6_1_MAX) {
    id = platform_layout_type6_1[pid];
  } else if (pid >= TYPE6_2_MIN && pid <= TYPE6_2_MAX) {
    id = platform_layout_type6_2[pid - TYPE6_2_MIN];
  } 
  if (id == _NO_DEF_) {
    id = DEFAULT_ID;
  }

  return id;
}
#endif /* sun */

/*
 * keyboard layout detect code
 * This info is not always correct in remote display env, as
 * this is executed on server machine, but we need the keyboard
 * layout info for client side.
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stropts.h>

#if sun
#include <sys/kbio.h>
#endif

#include "kbltrans.h"

int
kbltrans_get_platform_id ()
{
  int layout = kbltrans_get_default_id ();
#if sun
  int ret = -1;
  int type;
  int fd = open("/dev/kbd", O_RDONLY);
  ret = ioctl(fd, KIOCLAYOUT, &layout);

  if (ret == -1)
    return layout;

  ret = ioctl(fd, KIOCTYPE, &type);

  if (ret == -1)
    return layout;

  return normalize_id (layout, type);
#else
  return layout;
#endif /* sun */

}

void
kbltrans_get_supported_layout_names (int *num, char ***layouts)
{
  *num = _MAX_LAYOUT;
  *layouts = predefined_layout_list;
}

int
kbltrans_get_default_id ()
{
  return DEFAULT_ID;
}

char *
kbltrans_get_layout_name_by_locale (const char *lc_name)
{
  static const char *lc2kbl_mappings[] = {
    "sq_AL", "al",
    "cs_CZ", "cz",
    "da_DK", "dk",
    "et_EE", "ee",
    "sh_BA", "bs",
    "sl_SI", "si",
    "uk_UA", "ua",
    "pt_BR", "br",
    "kk_KZ", "kz",
    "sv_SE", "se",
    "fr_CA", "ca_enhanced",
    "nb_NO", "no",
    "nn_NO", "no",
    "be_BY", "by",
    NULL
  };
  static const int MAX_NAME_LEN = 32;
  static const char *KBL_PREFIX = "x-kbl-";

  char kbl[MAX_NAME_LEN], *ret;
  char *ll = (char *)xmalloc (MAX_NAME_LEN - strlen (KBL_PREFIX));

  if (lc_name == NULL)
    return NULL;
  
  strcpy (ll, lc_name);
  char *p = strchr(ll, '.');
  if (p) *p = '\0';

  int i;
  int found = 0;
  for (i=0; lc2kbl_mappings[i]; i+=2) {
    if (!strncmp (ll, lc2kbl_mappings[i], 5)) {
      strcpy (ll, lc2kbl_mappings[i+1]);
      found = 1;
      break;
    }
  }
  if (found == 0) {
    ll[2] = '\0';
  }

  sprintf (kbl, "%s%s", KBL_PREFIX, ll);
  ret = (get_index_from_name (kbl) >= _MAX_LAYOUT)? NULL: strdup (kbl);

  free (ll);
  return ret;
}
