#include <locale.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>

#include <iiimp.h>
#include "iiimp-opcode.h"

#define	IO_READ		(0xfe)
#define	IO_WRITE	(0xef)
#define MAGIC		"IIIMP DUMP DATA"
#define	DATA_VERSION	(1)
#define BIG_ENDIAN	(1)
#define SMALL_ENDIAN	(2)


typedef enum {
	unknown_endian = 0,
	big_endian = 1,
	little_endian = 2,
} endian_t;

typedef struct {
	char		fds;
	int		r_seq_no;
	int		w_seq_no;
	int		r_left;
	int		w_left;
	char		r_op;
	char		w_op;
	unsigned char *	buf;
	off_t	 	buf_off;
	size_t		buf_size;
	endian_t	byte_order;
	int		protocol_version;
	IIIMP_data_s *	data_s;
} client_t;

static client_t	client[1024];

uint_t		print_flag;
int		abort_on_error;
int		follow;
int		partial;
int		dump_fd = -1;
int		input_fd;

#define	UINTEGER32(a, e)	((BIG_ENDIAN == (e)) ?                    \
				 ((*(((unsigned char *)(a)) + 0) << 24) + \
				  (*(((unsigned char *)(a)) + 1) << 16) + \
				  (*(((unsigned char *)(a)) + 2) << 8)  + \
				  (*(((unsigned char *)(a)) + 3) << 0)) : \
				 ((*(((unsigned char *)(a)) + 3) << 24) + \
				  (*(((unsigned char *)(a)) + 2) << 16) + \
				  (*(((unsigned char *)(a)) + 1) << 8)  + \
				  (*(((unsigned char *)(a)) + 0) << 0)))


#define	INTEGER16(a, f)	((big_endian == client[f].byte_order) ? \
			 ((*((unsigned char *)(a)) << 8) +  \
			  *(((unsigned char *)(a)) + 1)) : \
			 ((*(((unsigned char *)(a)) + 1) << 8) + \
			  *((unsigned char *)(a))))

#define	INTEGER32(a, f)	((big_endian == client[f].byte_order) ? \
			 ((*(((unsigned char *)(a)) + 0) << 24) +  \
			  (*(((unsigned char *)(a)) + 1) << 16) + \
			  (*(((unsigned char *)(a)) + 2) << 8) + \
			  (*(((unsigned char *)(a)) + 3) << 0)) : \
			 ((*(((unsigned char *)(a)) + 3) << 24) +  \
			  (*(((unsigned char *)(a)) + 2) << 16) + \
			  (*(((unsigned char *)(a)) + 1) << 8) + \
			  (*(((unsigned char *)(a)) + 0) << 0)))

static void	dump(int thread_id, int io_type, int fildes,
		     const unsigned char * buf, size_t nbyte);


void
buf_free_all()
{
	int	i;
	for (i = 0; i < 1024; i++) {
		if (NULL != client[i].buf) {
			free(client[i].buf);
		}
	}
	return;
}


void
dump(int thread_id, int io_type, int fildes, const unsigned char * buf, size_t nbyte)
{
	const unsigned char *	p;
	size_t			rest;
	IIIMP_message *		m;
	const unsigned char *	p0;
	size_t			rest0;

	p = buf;

	if (0 == nbyte) {
		fprintf(stdout, "%d:%d -- connection closed\n", thread_id, fildes);
		iiimp_data_s_delete(client[fildes].data_s);
		free(client[fildes].buf);
		memset(&(client[fildes]), 0, sizeof (client[fildes]));
		return;
	}

	if (NULL == client[fildes].buf) {
		client[fildes].buf_size = (2 * 1024 * 1024);
		client[fildes].buf = malloc(2 * 1024 * 1024);
		client[fildes].data_s = iiimp_data_s_new();
	}
	memcpy(client[fildes].buf + client[fildes].buf_off, buf, nbyte);
	client[fildes].buf_off += nbyte;
	client[fildes].w_seq_no += 1;

	if ((0 == client[fildes].w_left) && (4 <= client[fildes].buf_off)) {
		p = client[fildes].buf;

		client[fildes].w_op = (0x7f & *p);
		client[fildes].w_seq_no = 0;
		client[fildes].w_left = (((*(p + 1) << 16) +
					  (*(p + 2) << 8) +
					  (*(p + 3) << 0)) * 4);
		client[fildes].w_left += 4;
	}

	if (client[fildes].w_left != client[fildes].buf_off) {
		if (partial <= client[fildes].buf_off) {
			fprintf(stdout, "%d:%d", thread_id, fildes);
			if (IO_READ == io_type) {
				fprintf(stdout, "<-");
			} else {
				fprintf(stdout, "->");
			}
			fprintf(stdout, " partial +%d = %d / %d\n",
				nbyte, client[fildes].buf_off, client[fildes].w_left);
		}
		return;
	}

	p = client[fildes].buf;

	fprintf(stdout, "%d:%d", thread_id, fildes);
	if (IO_READ == io_type) {
		fprintf(stdout, "<-");
	} else {
		fprintf(stdout, "->");
	}
	p0 = p + 4;
	rest0 = (client[fildes].w_left - 4);

	m = iiimp_message_unpack(client[fildes].data_s,
				 client[fildes].w_op, &rest0, &p0);
	if (NULL != m) {
		iiimp_message_print(client[fildes].data_s, m);
		iiimp_message_delete(client[fildes].data_s, m);
	} else if (0 != abort_on_error) {
		p0 = p + 4;
		rest0 = (client[fildes].w_left - 4);
		m = iiimp_message_unpack(client[fildes].data_s,
					 client[fildes].w_op, &rest0, &p0);
		abort();
	}

	client[fildes].buf_off = 0;
	client[fildes].w_left = 0;
	client[fildes].w_seq_no = 0;

	return;
}


int
pad_size(int nbyte)
{
	int	i;

	i = (nbyte % 16);
	return ((0 == i) ? 0 : (16 - i));
}


void
usage()
{
	fprintf(stderr, "%s [-a] [-j] [-f] [-p nbyte] [iiim-data-file]\n", "iiim-printer2");
}


void
parse_opt(int argc, char ** argv)
{
	int		c;
	extern char *	optarg;
	extern int	optind;
	extern int	opterr;
	extern int	optopt;

	print_flag = 0;
	abort_on_error = 0;
	follow = 0;
	partial = 8;

	while (EOF != (c = getopt(argc, argv, "ajfd:p:"))) {
		switch (c) {
		case 'a':
			abort_on_error = 1;
			break;
		case 'j':
			print_flag |= IIIMP_PRINT_JARFILE;
			break;
		case 'f':
			follow = 1;
			break;
		case 'd':
			dump_fd = atol(optarg);
			break;
		case 'p':
			partial = atol(optarg);
			break;
		default:
			usage(c);
			exit(1);
			break;
		}
	}
	if (optind == argc) {
		input_fd = 0;
	} else {
		input_fd = open(*(argv + optind), O_RDONLY, 0);
		if (-1 == input_fd) {
			perror("open");
			exit(1);
		}
	}
}

int
main(int argc, char ** argv)
{
	char		header[256];
	char		buf[1024 * 1024];
	char *		p;
	int		i;
	int		r;
	int		pad;
	uint32_t	thread_id;
	uint32_t	fildes;
	uint32_t	io_type;
	uint32_t	data_size;
	uint32_t	byte_order;
	uint32_t	version;

	setlocale(LC_ALL, "");

	parse_opt(argc, argv);

	r = read(input_fd, header, 32);
	if (r < 0) {
		perror("read");
		exit(1);
	}
	if (32 != r) {
		fprintf(stderr, "initial read error\n");
		exit(1);
	}

	if (0 != memcmp(header, MAGIC, 16)) {
		fprintf(stderr, "not IIIMP data\n");
		exit(1);
	}

	byte_order = ((1 == header[16]) ? SMALL_ENDIAN : BIG_ENDIAN);
	p = (header + 16 + 4);
	version = UINTEGER32(p, byte_order);

	for (;;) {
		for (i = 0; i < 16;) {
			r = read(input_fd, header + i, 16 - i);
			if (r < 0) {
				perror("read");
				exit(1);
			}
			if (0 == r) {
				if (0 == follow) {
					buf_free_all();
					exit(0);
				}
				sleep(1);
				continue;
			}
			i += r;
		}
		p = header;
		thread_id = UINTEGER32(p, byte_order);
		p += 4;
		io_type = UINTEGER32(p, byte_order);
		p += 4;
		fildes = UINTEGER32(p, byte_order);
		p += 4;
		data_size = UINTEGER32(p, byte_order);
		if ((sizeof (buf)) < data_size) {
			fprintf(stderr, "too large data size %u\n", data_size);
			exit(1);
		}

		pad = pad_size(data_size);

		data_size += pad;

		for (i = 0; i < data_size;) {
			r = read(input_fd, buf + i, data_size - i);
			if (r < 0) {
				perror("read");
				exit(1);
			}
			if (0 == r) {
				if (0 == follow) {
					buf_free_all();
					exit(0);
				}
				sleep(1);
				continue;
			}
			i += r;
		}

		if ((dump_fd < 0) || (dump_fd == fildes)) {
			dump(thread_id, io_type, fildes,
			     (unsigned const char *)buf, data_size - pad);
		}
	}

	return 0;
}
