/* 
 * Copyright (C) 2008 Sun Microsystems
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Aijin Kim <Aijin.Kim@Sun.COM>
 */

#include <stdio.h>

#include "ime.h"
#include "ime_buffer.h"

/**************************************************************************
* IME Methods according to the IME.h
***************************************************************************/
ImeResult hex_ko_Initialize(ImeInfo ime_info);
ImeResult hex_ko_Destroy(ImeInfo ime_info);
ImeResult hex_ko_Create_Session(ImeInputContext ic);
ImeResult hex_ko_Destroy_Session(ImeInputContext ic);
ImeResult hex_ko_Process_Key_Event(ImeInputContext ic, ImeKey key_event);
ImeResult hex_ko_FocusIn(ImeInputContext ic);
ImeResult hex_ko_FocusOut(ImeInputContext ic);
ImeResult hex_ko_Create_User(ImeInputContext ic);
ImeResult hex_ko_Destroy_User(ImeInputContext ic);
ImeResult hex_ko_Create_Desktop(ImeInputContext ic);
ImeResult hex_ko_Destroy_Desktop(ImeInputContext ic);

ImeMethodsRec hex_ko_methods = {
    10,                                /* version */
    hex_ko_Initialize,                 /* ImeInitialize */
    hex_ko_Destroy,                    /* ImeDestroy  */
    hex_ko_Process_Key_Event,          /* ImeProcessKeyEvent */
    NULL,                              /* ImeProcessAuxEvent  */
    hex_ko_Create_Session,             /* ImeAttachSession */
    hex_ko_Destroy_Session,            /* ImeDetachSession */
    hex_ko_FocusIn,                    /* ImeFocusIn  */
    hex_ko_FocusOut,                   /* ImeFocusOut */
    hex_ko_Create_User,                /* ImeAttachUser */
    hex_ko_Destroy_User,               /* ImeDetachUser */
    hex_ko_Create_Desktop,             /* ImeAttachDesktop */
    hex_ko_Destroy_Desktop,            /* ImeDeta chDesktop */
    NULL,                              /* ImeGetErrorMsg */
};

ImmServices imm_services = NULL;

// Typedefine for data saved in user scope
// do not used bit mask, just for demostration
typedef struct _TUserPreference {
    int    show_utf8_value;
    int    show_unicode_value;
} TUserPreference;

#define LANG_KOKR	ENCODE_EUCKR

#ifdef	WIN32
#define EXPORT extern __declspec(dllexport)
EXPORT
#endif

ImeResult RegisterIME(ImmServices srvs, ImeInfo* ppinfo, ImeMethods* pmthds, int argc, char **argv)
{
    int enc, lang = LANG_KOKR;
    ImeInfoRec *ime_info = NULL;

    DEBUG_printf("    ====>Register Hex_Ko IM: argc: %d\n", argc);

    ime_info = (ImeInfoRec *)malloc(sizeof(ImeInfoRec)); //suppose success

    ime_info->version          = 10;
    ime_info->mt_safe          = 0;
    ime_info->encoding         = lang;
    ime_info->uuid             = "hex-ko-21d76e189-9a54-4a24-8cf7-5d611f3d555f";
    ime_info->author           = "Jeffrey Chen <Pu.Chen@Sun.COM>\nAijin Kim <Aijin.Kim@Sun.COM>";
    ime_info->copyright        = "Copyright (c) 2008, 2010 Oracle and/or its affiliates";;
    ime_info->icon_file        = "hex_ko.xpm";
    ime_info->pl               = NULL;
    ime_info->specific_data    = (void*)lang;
    ime_info->hinting          = NULL;

    ime_info->name       = "Hex (EUC)";
    ime_info->support_locales = "ko_KR.EUC,ko_KR.UTF-8";
    *ppinfo = ime_info;
    *pmthds = &hex_ko_methods;
    imm_services = srvs;

    return (IME_OK);
}

static
int get_ime_language(ImeInputContext ic)
{
    ImeInfo ime_info = imm_services->ImmGetImeInfo(ic);
    return (ime_info)?((int)ime_info->specific_data):(LANG_KOKR);
}

static
ImHandle get_candidate_aux(ImeInputContext ic)
{
    return (ImHandle)imm_services->ImmGetData(ic, IME_SCOPE_DESKTOP);
}

ImeResult hex_ko_Initialize(ImeInfo ime_info)
{
    DEBUG_printf("    ====>hex_ko_Initialize for language %d\n", ime_info->specific_data);

    return (IME_OK);
}

ImeResult hex_ko_Destroy(ImeInfo ime_info)
{
    DEBUG_printf("    ====>hex_ko_Destroy\n");

    if (ime_info != NULL)
        free(ime_info);

    return (IME_OK);
}

static
ImeResult my_update_candidates(ImeInputContext ic, ImeBufferRec *ime_buffer)
{
    int sz = 0;
    ImeEventRec         event;

    ime_buffer->candidates.horizental = 0;

    event.any_event.ic = ic;
    event.any_event.peer = get_candidate_aux(ic);
    if (ime_buffer) sz = ime_buffer->candidates.count;
    if (event.any_event.peer != 0) { //make sure the aux start successfully
        if (sz == 0) {
            event.type = IME_EVENT_EXPOSE;
            event.any_event.param = 0;    //hide candidates
            imm_services->ImmSendUiMessage(&event);
        } else {
            event.type = IME_EVENT_EXPOSE;
            event.any_event.param = IMUI_STATE_SHOW;    //show candidates
            imm_services->ImmSendUiMessage(&event);
            event.type = IME_EVENT_CANDI_DATA;
            event.any_event.param = ENCODE_EUCKR;
            event.candidate_data_event.candidates = &(ime_buffer->candidates);
            imm_services->ImmSendUiMessage(&event);
        }
    } else {
        if (sz == 0)
            imm_services->ImmHideCandidates(ic);
        else {
            imm_services->ImmShowCandidates(ic);
            imm_services->ImmUpdateCandidates(ic, &ime_buffer->candidates);
        }
    }

    return IME_OK;
}

//infact 9/16/5/1/5
#define MAX_PREEDIT_BYTES               16
#define MAX_CANDIDATES_NUM              16
#define MAX_CANDIDATE_BYTES             16
#define MAX_NUMBERING_BYTES             4
#define MAX_COMMIT_BYTES                16

ImeResult hex_ko_Create_Session(ImeInputContext ic)
{
    int i;
    ImmResult imm_result;
    ImeBufferRec *ime_buffer = NULL;

    DEBUG_printf("    ====>hex_ko_Create_Session ======= begin calloc for ime_buffer\n");
    ime_buffer = alloc_ime_buffer(MAX_PREEDIT_BYTES, MAX_CANDIDATES_NUM, MAX_CANDIDATE_BYTES, MAX_NUMBERING_BYTES, MAX_COMMIT_BYTES);
    if (ime_buffer == NULL)
        return (IME_FAIL);

    imm_result = imm_services->ImmSetData(ic, IME_SCOPE_SESSION, ime_buffer);
    if (imm_result == IMM_FAIL) {
        free(ime_buffer);
        return (IME_FAIL);
    }

    return (IME_OK);
}

ImeResult hex_ko_Destroy_Session(ImeInputContext ic)
{
    ImeBufferRec *ime_buffer = (ImeBufferRec *)imm_services->ImmGetData(ic, IME_SCOPE_SESSION);
    DEBUG_printf("    ====>hex_ko_Destroy_Session ======= begin get ime_session_data: 0x%x\n", ime_buffer);

    free_ime_buffer(ime_buffer);

    return (IME_OK);
}

ImeResult hex_ko_Create_User(ImeInputContext ic)
{
    int  sz;
    unsigned char* buf = NULL;
    TUserPreference* pup = NULL;

    if (pup = (TUserPreference*)malloc(sizeof(TUserPreference))) {
        memset(pup, 0, sizeof(TUserPreference));
        buf = (unsigned char*)imm_services->ImmLoadUserProfile(ic, "preference", &sz);
        if (buf) {
            if (sscanf(buf, "%d", &sz) == 1) {
                pup->show_utf8_value = (sz & 1);
                pup->show_unicode_value = (sz & 2);
            }
            imm_services->ImmFreeUserProfile(buf);
        }
        imm_services->ImmSetData(ic, IME_SCOPE_USER, pup);
        return IME_OK;
    }
    return IME_FAIL;
}

ImeResult hex_ko_Destroy_User(ImeInputContext ic)
{
    int  pref = 0;
    char valstr[32];
    TUserPreference* pup = (TUserPreference*)imm_services->ImmGetData(ic, IME_SCOPE_USER);
    if (pup) {
        if (pup->show_utf8_value) pref |= 1;
        if (pup->show_unicode_value) pref |= 2;
        snprintf(valstr, 32, "%d\n", pref);
        return ImmSaveUserProfile(ic, "preference", valstr, strlen(valstr));
    }
    return IME_OK;
}

ImeResult hex_ko_Create_Desktop(ImeInputContext ic)
{
    static const char *aux_name = "com.sun.iiim.hangul.aux.hex";

    DEBUG_printf("    ====>hex_ko: call hex_ko_Create_Desktop()\n");

    ImHandle candi_aux = imm_services->ImmStartUI(ic, aux_name);
    imm_services->ImmSetData(ic, IME_SCOPE_DESKTOP, (void*)candi_aux);
}

ImeResult hex_ko_Destroy_Desktop(ImeInputContext ic)
{
    DEBUG_printf("    ====>hex_ko: call hex_ko_Destroy_Desktop()\n");

    ImHandle candi_aux = (ImHandle)imm_services->ImmGetData(ic, IME_SCOPE_DESKTOP);
    imm_services->ImmCloseUI(ic, candi_aux);
}

ImeResult hex_ko_FocusIn(ImeInputContext ic)
{
    DEBUG_printf("    ====>hex_ko: call hex_ko_FocusIn()\n");

    ImeBufferRec *ime_buffer = NULL;
    ime_buffer = (ImeBufferRec *)imm_services->ImmGetData(ic, IME_SCOPE_SESSION);
    if (ime_buffer && ime_buffer->candidates.count)
        return my_update_candidates(ic, ime_buffer);
   
    imm_services->ImmUpdatePreedit(ic, &ime_buffer->preedit);

    return IME_OK;
}

ImeResult hex_ko_FocusOut(ImeInputContext ic)
{
    ImeEventRec event;

    DEBUG_printf("    ====>hex_ko: call hex_ko_FocusOut()\n");

    event.type              = IME_EVENT_EXPOSE;
    event.any_event.peer    = get_candidate_aux(ic);
    if (event.any_event.peer) {
        event.any_event.ic      = ic;
        event.any_event.param   = 0;    //hide it
        imm_services->ImmSendUiMessage(&event);
    }
    return(IME_OK);
}

/* Process key input event */
/* Return value:  
 * IME_UNUSED_KEY : if IME not use this key, return this key to systerm directly
 * IME_OK         : if IME has used this key */
ImeResult hex_ko_Process_Key_Event(ImeInputContext ic, ImeKey key_event)
{
    unsigned char   key;
    int             lang, ret;
    ImeBufferRec   *ime_buffer = NULL;

    DEBUG_printf("    ====>hex_ko_Process_Key_Event: ic: 0x%x\n", ic);
    ime_buffer = (ImeBufferRec *)imm_services->ImmGetData(ic, IME_SCOPE_SESSION);
    if (ime_buffer == NULL){
        DEBUG_printf("      ====>hex_ko: ime_buffer is null.\n");
        return (IME_UNUSED_KEY);
    }

    ime_buffer->return_status = 0;

    lang = get_ime_language(ic);

    key = imm_services->ImmPrefilterKey(key_event);
    if (key == IME_FILTERED_KEY_UNUSED)
	return (IME_UNUSED_KEY);

    ret = hex_ko_filter(key, ime_buffer);
    if (ret == IME_UNUSED_KEY) {
        DEBUG_printf("      ====>hex_ko: key is not used.\n");
	return (IME_UNUSED_KEY);
    }

    DEBUG_printf("      ====>hex_ko: key is used.\n");
    if (ime_buffer->return_status & IME_PREEDIT_AREA) {
        if (ime_buffer->preedit.preedit.text[0] != '\0') {
            imm_services->ImmShowPreedit(ic);
            imm_services->ImmUpdatePreedit(ic, &ime_buffer->preedit);
        } else {
            imm_services->ImmHidePreedit(ic);
        }
    }

    if (ime_buffer->return_status & IME_LOOKUP_AREA)
        my_update_candidates(ic, ime_buffer);

    if (ime_buffer->return_status & IME_COMMIT)
        imm_services->ImmCommit(ic, ime_buffer->commit_buf);

    return (IME_OK);
}
