/*

Copyright (c) 2006 Yuuhei TERAMOTO <tera@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <stdio.h>
#include "iconv.h"
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <getopt.h>

#include "phonogram_conv.h"

/*
 * phonogram_conv console tool.
 */
static void usage( int argc, char *argv[] );
static int output( pgc_trans_list lst, char * inbuf );

int main(
int argc,
char *argv[]
)
{
    int exit_status = EXIT_SUCCESS;
    pgc_descriptor pgcd = NULL;
    iconv_t ic_2utf8 = NULL, ic_2utf16 = NULL;
    size_t utf16bom=1;
    int eot=1;
    char encode[32];
    char table_path[2048];
    encode[0] = 0;
    table_path[0] = 0;

    while( 1 ){
	int res, option_index = 0;
	static struct option long_options[] = {
		{"unit-table", 1, 0, 'u'},
		{"encode", 1, 0, 'e'},
		{"incomplete-mode", 1, 0, 't' },
		{0, 0, 0, 0}
               };
	res = getopt_long (argc, argv, "tu:e:", long_options, &option_index);
	if( res == -1 ){
	    break;
	}
	switch( res ){
	  case 'u':
	   strncpy( table_path, optarg, sizeof(table_path)-1 );
	   break;
	  case 'e':
	   strcpy( encode, optarg );
	   break;
	  case 't':
	   eot = 0;
	   break;
	  case '?':
	  default:
	   usage( argc, argv );
	   exit( 1 );
	}
    }
    if( encode[0] == 0 ){
	strcpy( encode, "EUC-JP" );
    }
    if( table_path[0] == 0 ){
	fprintf( stderr,"No unit table used for convert specified.\n");
	usage( argc, argv );
	exit( 1 );
    }
    
    ic_2utf8 = iconv_open( "UTF-8", encode );
    if( ic_2utf8 == (iconv_t)-1 ){
	fprintf( stderr, "iconv_open() %s to UTF-8 fail. errno:%d\n", encode, errno );
	goto err;
    }
    ic_2utf16 = iconv_open( "UTF-16", "UTF-8" );
    if( ic_2utf16 == (iconv_t)-1 ){
	fprintf( stderr, "iconv_open() UTF-8 to UTF-16 fail. errno:%d\n", errno );
	goto err;
    }

    pgcd = pgc_open( table_path, encode );
    if( pgcd == NULL ){
	fprintf( stderr, "phongram_conv_open() fail. errno:%d\n", errno );
	goto err;
    }
    while( 1 ){
	char buf[256];
	char buf2[256*6];
	char buf3[256*2];
	char *pp, *inbuf, *outbuf;
	PGC_UTFCHAR *inbuf2;
	size_t inlen, inbytesleft, outbytesleft;
	pgc_trans_list trans_list;
	int res;

	if( fgets( buf, sizeof(buf), stdin ) == NULL ){
	    break;
	}
 	pp = strrchr( buf, '\n' );
	if( pp != NULL ){
	    *pp = 0;
	}
	inbuf = buf;
	inbytesleft = strlen(buf);
	if( inbytesleft == 0 ){
	    continue;
	}
	outbuf = buf2;
	outbytesleft = sizeof(buf2);
	if( iconv( ic_2utf8, &inbuf, &inbytesleft, &outbuf, &outbytesleft ) == -1 ){
	    fprintf( stderr, "iconv() EUC to UTF-8 fail. errno:%d\n", errno );
	    goto err;
	}
	inbuf = buf2;
	inbytesleft = sizeof(buf2) - outbytesleft;
	outbuf = buf3;
	outbytesleft = sizeof(buf3);
	if( iconv( ic_2utf16, &inbuf, &inbytesleft, &outbuf, &outbytesleft ) == -1 ){
	    fprintf( stderr, "iconv() UTF-8 to UTF-16 fail. errno:%d\n", errno );
	    goto err;
	}
	inbuf2 = (PGC_UTFCHAR*)buf3 + utf16bom;
	inlen = ( sizeof(buf3) - outbytesleft ) / sizeof(PGC_UTFCHAR) - utf16bom;
	res = pgc_convert( pgcd, inbuf2, inlen, eot, &trans_list );
	if( res < 0 ){
#ifdef DEBUG
	    fprintf( stderr, "pgc_convert() fail. errno:%d\n", errno );
#endif
	    goto err;
	}
#ifdef DEBUG	
	if( res == PGC_ACCEPTED ){
	    fprintf( stderr, "pgc_convert() ACCEPTED\n" );
	}else if( res == PGC_INPROCESS ){
	    fprintf( stderr, "pgc_convert() INPROCESS\n" );
	}else if( res == PGC_REJECTED ){
	    fprintf( stderr, "pgc_convert() REJECTED\n" );
	}
#endif
	if( output( trans_list, buf ) != 0 ){
	    pgc_trans_list_destroy( trans_list );
	    goto err;
	}
	pgc_trans_list_destroy( trans_list );
	utf16bom = 0;
	fprintf( stdout, "\n" );
    }
end:
    if( ic_2utf8 != (iconv_t)-1 ){
	iconv_close( ic_2utf8 );
    }
    if( ic_2utf16 != (iconv_t)-1 ){
	iconv_close( ic_2utf16 );
    }
    if( pgcd != NULL ){
	pgc_close( pgcd );
    }
    exit( exit_status );
err:
    if( exit_status == EXIT_SUCCESS ){
	exit_status = EXIT_FAILURE;
    }
    goto end;
}

static int
output(
    pgc_trans_list lst,
    char * inbuf
)
{
    pgc_trans trans;
    for( trans = pgc_trans_list_next( lst, NULL ); 
	 trans != NULL; trans = pgc_trans_list_next( lst, trans ) ){
	unsigned char *pp = NULL,*qq = NULL,*s1,*s2;
	size_t size, sp, ep;
	if( pgc_trans_get_string( trans, 0, PGC_TRANS_LAST, &sp, &ep, NULL, 0 ) != 0 ){
	    return -1;
	}
	size = ep-sp;
	if( size > 0  ){
	    pp = malloc( size + 1 );
	    if( pp == NULL ){
		return -1;
	    }
	    pgc_trans_get_string( trans, 0, PGC_TRANS_LAST, &sp, &ep, pp, size );
	    pp[size] = 0;
	    if( pgc_trans_get_source_pos( trans, 0, PGC_TRANS_LAST, &sp, &ep ) != 0 ){
		return -1;
	    }
	}else{
	    qq = inbuf + ep;
	}
	if( pp == NULL ){
	    s1 = "";
	}else{
	    s1 = pp;
	}
	if( qq == NULL ){
	    s2 = "";
	}else{
	    s2 = qq;
	}
	fprintf( stdout, "%s\t%s\n", s1, s2 );
	free(pp);
#ifdef DEBUG
	pgc_trans_print( trans, stderr );
#endif
    }
    return 0;
}

static void
usage(
int argc,
char *argv[]
)
{
    fprintf( stdout, "Usage: %s -u file [-e encode] [-t]\n", argv[0] );
    fprintf( stdout, "Options:\n");
    fprintf( stdout, "-u, --unit-table=FILE\tSpecify unit table file to be used for conversion\n");
    fprintf( stdout, "-e, --encode=ENCODE\tSpecify input encode\n"); 
    fprintf( stdout, "-t, --incomplete-mode\tAssume that input is incomplete, some more charcters may follow.\n");
}



/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */



