/*

Copyright (c) 2006 Kenji ABE <abek@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/

#include <stdlib.h>

#include "imcaret.h"
#include "imcommit.h"
#include "imconv.h"
#include "imeventhandler.h"
#include "iminput.h"
#include "imlookup.h"
#include "impreedit.h"
#include "imstatus.h"
#include "imtext.h"


typedef struct {
    int key_code;
    int modifier;
    KeyEventHandler handler;
} KeyMapEntry;

typedef struct {
    KeyEventHandler fallback_handler;
    int n_entry;
    const KeyMapEntry *entry;
} KeyMap;

static const KeyMapEntry EditEmptyKeyMapEntries[] = {
	{IM_VK_SPACE, 0, handle_input_space},

	{IM_VK_COMMA, IM_CTRL_MASK, handle_input_mode_prev},

	{IM_VK_PERIOD, IM_CTRL_MASK, handle_input_mode_next},
};

static const KeyMapEntry EditingKeyMapEntries[] = {
	{IM_VK_SPACE, 0, handle_convert},
	{IM_VK_CONVERT, 0, handle_convert},

	{IM_VK_ENTER, 0, handle_commit},
	{IM_VK_ACCEPT, 0, handle_commit},
	{IM_VK_J, IM_CTRL_MASK, handle_commit},
	{IM_VK_M, IM_CTRL_MASK, handle_commit},

	{IM_VK_ESCAPE, 0, handle_cancel},
	{IM_VK_G, IM_CTRL_MASK, handle_cancel},
	{IM_VK_OPEN_BRACKET, IM_CTRL_MASK, handle_cancel},

	{IM_VK_BACK_SPACE, 0, handle_delete_prev},
	{IM_VK_H, IM_CTRL_MASK, handle_delete_prev},

	{IM_VK_DELETE, 0, handle_delete_next},
	{IM_VK_D, IM_CTRL_MASK, handle_delete_next},

	{IM_VK_HOME, 0, handle_caret_move_begin},
	{IM_VK_A, IM_CTRL_MASK, handle_caret_move_begin},

	{IM_VK_END, 0, handle_caret_move_end},
	{IM_VK_E, IM_CTRL_MASK, handle_caret_move_end},

	{IM_VK_LEFT, 0, handle_caret_move_left},
	{IM_VK_B, IM_CTRL_MASK, handle_caret_move_left},

	{IM_VK_RIGHT, 0, handle_caret_move_right},
	{IM_VK_F, IM_CTRL_MASK, handle_caret_move_right},

	{IM_VK_F6, 0, handle_convert_to_hiragana},
	{IM_VK_F7, 0, handle_convert_to_katakana},
	{IM_VK_F8, 0, handle_convert_to_halfwidth_katakana},
	{IM_VK_F9, 0, handle_convert_to_fullwidth_alpha},
	{IM_VK_F10, 0, handle_convert_to_alpha},

	{IM_VK_TAB, 0, handle_predict},
	{IM_VK_I, IM_CTRL_MASK, handle_predict},
};

static const KeyMapEntry LetterTypeConvKeyMapEntries[] = {
	{IM_VK_SPACE, 0, handle_convert},
	{IM_VK_CONVERT, 0, handle_convert},

	{IM_VK_ENTER, 0, handle_commit},
	{IM_VK_ACCEPT, 0, handle_commit},
	{IM_VK_J, IM_CTRL_MASK, handle_commit},
	{IM_VK_M, IM_CTRL_MASK, handle_commit},

	{IM_VK_ESCAPE, 0, handle_cancel},
	{IM_VK_G, IM_CTRL_MASK, handle_cancel},
	{IM_VK_OPEN_BRACKET, IM_CTRL_MASK, handle_cancel},

	{IM_VK_BACK_SPACE, 0, handle_step_back},
	{IM_VK_H, IM_CTRL_MASK, handle_step_back},

	{IM_VK_DELETE, 0, handle_cancel},
	{IM_VK_D, IM_CTRL_MASK, handle_cancel},

	{IM_VK_F6, 0, handle_convert_to_hiragana},
	{IM_VK_F7, 0, handle_convert_to_katakana},
	{IM_VK_F8, 0, handle_convert_to_halfwidth_katakana},
	{IM_VK_F9, 0, handle_convert_to_fullwidth_alpha},
	{IM_VK_F10, 0, handle_convert_to_alpha},
};

static const KeyMapEntry ConvKeyMapEntries[] =
{
	{IM_VK_SPACE, 0, handle_convert},
	{IM_VK_CONVERT, 0, handle_convert},

	{IM_VK_DOWN, 0, handle_lookup_choice_next},
	{IM_VK_N, IM_CTRL_MASK, handle_lookup_choice_next},

	{IM_VK_UP, 0, handle_lookup_choice_prev},
	{IM_VK_P, IM_CTRL_MASK, handle_lookup_choice_prev},

	{IM_VK_ENTER, 0, handle_commit},
	{IM_VK_ACCEPT, 0, handle_commit},
	{IM_VK_J, IM_CTRL_MASK, handle_commit},
	{IM_VK_M, IM_CTRL_MASK, handle_commit},

	{IM_VK_HOME, 0, handle_caret_move_begin},
	{IM_VK_A, IM_CTRL_MASK, handle_caret_move_begin},

	{IM_VK_END, 0, handle_caret_move_end},
	{IM_VK_E, IM_CTRL_MASK, handle_caret_move_end},

	{IM_VK_LEFT, 0, handle_caret_move_left},
	{IM_VK_B, IM_CTRL_MASK, handle_caret_move_left},

	{IM_VK_RIGHT, 0, handle_caret_move_right},
	{IM_VK_F, IM_CTRL_MASK, handle_caret_move_right},

	{IM_VK_RIGHT, IM_SHIFT_MASK, handle_expand_current_segment},
	{IM_VK_O, IM_CTRL_MASK, handle_expand_current_segment},

	{IM_VK_LEFT, IM_SHIFT_MASK, handle_shrink_current_segment},
	{IM_VK_I, IM_CTRL_MASK, handle_shrink_current_segment},

	{IM_VK_ESCAPE, 0, handle_cancel},
	{IM_VK_G, IM_CTRL_MASK, handle_cancel},
	{IM_VK_OPEN_BRACKET, IM_CTRL_MASK, handle_cancel},

	{IM_VK_BACK_SPACE, 0, handle_step_back},
	{IM_VK_H, IM_CTRL_MASK, handle_step_back},

	{IM_VK_F6, 0, handle_convert_to_hiragana},
	{IM_VK_F7, 0, handle_convert_to_katakana},
	{IM_VK_F8, 0, handle_convert_to_halfwidth_katakana},
	{IM_VK_F9, 0, handle_convert_to_fullwidth_alpha},
	{IM_VK_F10, 0, handle_convert_to_alpha},
};

static const KeyMapEntry LookupChoiceKeyMapEntries[] =
{
 	{IM_VK_ENTER, 0, handle_commit},
 	{IM_VK_ACCEPT, 0, handle_commit},
 	{IM_VK_J, IM_CTRL_MASK, handle_commit},
 	{IM_VK_M, IM_CTRL_MASK, handle_commit},

	{IM_VK_SPACE, 0, handle_lookup_choice_next},
	{IM_VK_CONVERT, 0, handle_lookup_choice_next},
	{IM_VK_DOWN, 0, handle_lookup_choice_next},
	{IM_VK_N, IM_CTRL_MASK, handle_lookup_choice_next},

	{IM_VK_SPACE, IM_SHIFT_MASK, handle_lookup_choice_prev},
	{IM_VK_CONVERT, IM_SHIFT_MASK, handle_lookup_choice_prev},
	{IM_VK_UP, 0, handle_lookup_choice_prev},
	{IM_VK_P, IM_CTRL_MASK, handle_lookup_choice_prev},

	{IM_VK_PAGE_DOWN, 0, handle_lookup_choice_next_page},
	{IM_VK_PAGE_UP, 0, handle_lookup_choice_prev_page},

	{IM_VK_0, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_1, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_2, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_3, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_4, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_5, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_6, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_7, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_8, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_9, 0, handle_lookup_choice_select_by_choice},

	{IM_VK_HOME, 0, handle_caret_move_begin},
	{IM_VK_A, IM_CTRL_MASK, handle_caret_move_begin},

	{IM_VK_END, 0, handle_caret_move_end},
	{IM_VK_E, IM_CTRL_MASK, handle_caret_move_end},

	{IM_VK_LEFT, 0, handle_caret_move_left},
	{IM_VK_B, IM_CTRL_MASK, handle_caret_move_left},

	{IM_VK_RIGHT, 0, handle_caret_move_right},
	{IM_VK_F, IM_CTRL_MASK, handle_caret_move_right},

	{IM_VK_LEFT, IM_SHIFT_MASK, handle_shrink_current_segment},
	{IM_VK_I, IM_CTRL_MASK, handle_shrink_current_segment},

	{IM_VK_RIGHT, IM_SHIFT_MASK, handle_expand_current_segment},
	{IM_VK_O, IM_CTRL_MASK, handle_expand_current_segment},

	{IM_VK_ESCAPE, 0, handle_cancel},
	{IM_VK_G, IM_CTRL_MASK, handle_cancel},
	{IM_VK_OPEN_BRACKET, IM_CTRL_MASK, handle_cancel},

	{IM_VK_BACK_SPACE, 0, handle_step_back},
	{IM_VK_H, IM_CTRL_MASK, handle_step_back},

	{IM_VK_F6, 0, handle_convert_to_hiragana},
	{IM_VK_F7, 0, handle_convert_to_katakana},
	{IM_VK_F8, 0, handle_convert_to_halfwidth_katakana},
	{IM_VK_F9, 0, handle_convert_to_fullwidth_alpha},
	{IM_VK_F10, 0, handle_convert_to_alpha},
};

static const KeyMapEntry PredictionKeyMapEntries[] =
{
 	{IM_VK_ENTER, 0, handle_commit},
 	{IM_VK_ACCEPT, 0, handle_commit},
 	{IM_VK_J, IM_CTRL_MASK, handle_commit},
 	{IM_VK_M, IM_CTRL_MASK, handle_commit},

	{IM_VK_SPACE, 0, handle_lookup_choice_next},
	{IM_VK_CONVERT, 0, handle_lookup_choice_next},
	{IM_VK_DOWN, 0, handle_lookup_choice_next},
	{IM_VK_N, IM_CTRL_MASK, handle_lookup_choice_next},
	{IM_VK_TAB, 0, handle_lookup_choice_next},
	{IM_VK_I, IM_CTRL_MASK, handle_lookup_choice_next},

	{IM_VK_SPACE, IM_SHIFT_MASK, handle_lookup_choice_prev},
	{IM_VK_CONVERT, IM_SHIFT_MASK, handle_lookup_choice_prev},
	{IM_VK_UP, 0, handle_lookup_choice_prev},
	{IM_VK_P, IM_CTRL_MASK, handle_lookup_choice_prev},

	{IM_VK_PAGE_DOWN, 0, handle_lookup_choice_next_page},
	{IM_VK_PAGE_UP, 0, handle_lookup_choice_prev_page},

	{IM_VK_0, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_1, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_2, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_3, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_4, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_5, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_6, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_7, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_8, 0, handle_lookup_choice_select_by_choice},
	{IM_VK_9, 0, handle_lookup_choice_select_by_choice},

	{IM_VK_ESCAPE, 0, handle_cancel},
	{IM_VK_G, IM_CTRL_MASK, handle_cancel},
	{IM_VK_OPEN_BRACKET, IM_CTRL_MASK, handle_cancel},

	{IM_VK_BACK_SPACE, 0, handle_step_back},
	{IM_VK_H, IM_CTRL_MASK, handle_step_back},
};

enum KEY_MAP_INDEX {
	NO_CONV_KEY_MAP = 0,
	EDIT_EMPTY_KEY_MAP,
	EDITING_KEY_MAP,
	LETTER_TYPE_CONVERSION_KEY_MAP,
	CONV_KEY_MAP,
	LOOKUP_CHOICE_KEY_MAP,
	PREDICTION_KEY_MAP,
};

#define DECLARE_EMPTY_KEY_MAP(fallback) {fallback, 0, NULL}
#define DECLARE_KEY_MAP(fallback, entries) {fallback, sizeof(entries)/sizeof(entries[0]), entries}

static const KeyMap KeyMaps[] = {
	DECLARE_EMPTY_KEY_MAP(handle_send_back_key),
	DECLARE_KEY_MAP(handle_preedit_append, EditEmptyKeyMapEntries),
	DECLARE_KEY_MAP(handle_preedit_append, EditingKeyMapEntries),
	DECLARE_KEY_MAP(handle_preedit_append, LetterTypeConvKeyMapEntries),
	DECLARE_KEY_MAP(handle_preedit_append, ConvKeyMapEntries),
	DECLARE_KEY_MAP(handle_preedit_append, LookupChoiceKeyMapEntries),
	DECLARE_KEY_MAP(handle_preedit_append, PredictionKeyMapEntries),
};

UTFCHAR SPACE[] = {0x0020, 0x0000};
UTFCHAR W_SPACE[] = {0x3000, 0x0000};


KeyEventHandler
lookup_key_event_handler(
    iml_session_t *session,
    IMKeyEventStruct *key_event
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    const KeyMap *key_map;
    const KeyMapEntry *key_map_entry;
    int i, n, key_map_index;

    switch (psd->im_state) {
      case IM_STATE_EDIT_EMPTY:
       key_map_index = EDIT_EMPTY_KEY_MAP;
       break;

      case IM_STATE_EDITING:
       key_map_index = EDITING_KEY_MAP;
       break;

      case IM_STATE_LETTER_TYPE_CONVERSION:
       key_map_index = LETTER_TYPE_CONVERSION_KEY_MAP;
       break;

      case IM_STATE_CONVERSION:
       if (psd->lookup_choice_p) {
	   key_map_index = LOOKUP_CHOICE_KEY_MAP;
	   break;
       }
       /* not break, but fall through. */
      case IM_STATE_CHANGE_SEG:
       key_map_index = CONV_KEY_MAP;
       break;

      case IM_STATE_PREDICTION:
       key_map_index = PREDICTION_KEY_MAP;
       break;

      default:
       ASSERT(0);
       /* not break, but fall through. */
      case IM_STATE_OFF:
       key_map_index = NO_CONV_KEY_MAP;
       break;
    }

    key_map = &KeyMaps[key_map_index];
    key_map_entry = key_map->entry;
    for (i=0, n = key_map->n_entry;
	 i < n;
	 ++i, ++key_map_entry) {
	if (key_map_entry->key_code == key_event->keyCode &&
	    key_map_entry->modifier == key_event->modifier) {
	    return key_map_entry->handler;
	}
    }

    return key_map->fallback_handler;
}

Bool
handle_noop(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    return True;
}

Bool
handle_send_back_key(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    iml_methods_t *mtbl;
    iml_inst *inst;

    mtbl = METHODS_TBL(session);
    inst = mtbl->iml_make_keypress_inst(session, key_event);
    mtbl->iml_link_inst_tail(inst_list, inst);
    return True;
}

Bool
handle_input_space(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->im_state != IM_STATE_EDIT_EMPTY) {
	if (!handle_commit(session, key_event, inst_list)) {
	    return False;
	}
    }

    ASSERT(psd->im_state == IM_STATE_EDIT_EMPTY);

    make_commit_inst_from_text(
	session,
	make_im_text(
	    session,
	    im_input_wide_p(psd->im_input) ? W_SPACE : SPACE,
	    1,
	    0
	),
	inst_list
    );
    return True;
}

Bool
handle_conversion_off(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    change_im_state(session, IM_STATE_OFF, inst_list, True);
    return True;
}


Bool
handle_convert(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_EDITING:
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
      case IM_STATE_PREDICTION:
       change_im_state(session, IM_STATE_CONVERSION, inst_list, True);
       return True;

      case IM_STATE_CONVERSION:
       if (psd->lookup_choice_p) {
	   return handle_lookup_choice_next(session, key_event, inst_list);
       }

       if (im_conv_cur_select_next(psd->im_conv)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_conv_make_preedit_text(psd->im_conv, session),
	       im_conv_cur_pos(psd->im_conv),
	       inst_list
	   );
       }

       start_lookup_choice(session, inst_list);
       return True;

      default:
       ASSERT(0); /* cannot happen */
       break;
    }

    return False;
}

Bool
handle_predict(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_EDITING:
       if (change_im_state(session, IM_STATE_PREDICTION, inst_list, True)) {
	   start_lookup_choice(session, inst_list);
	   return True;
       }
       break;

      case IM_STATE_PREDICTION:
       return handle_lookup_choice_next(session, key_event, inst_list);

      default:
       ASSERT(0); /* cannot happen */
       /* not break, but fall through. */
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
       break;
    }

    return False;
}

Bool
handle_cancel(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_EDITING:
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
      case IM_STATE_PREDICTION:
       return change_im_state(session, IM_STATE_EDIT_EMPTY, inst_list, True);

      default:
       ASSERT(0); /* cannot happen */
    }
    return False;
}

Bool
handle_step_back(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       return handle_send_back_key(session, key_event, inst_list);

      case IM_STATE_EDITING:
       return change_im_state(session, IM_STATE_EDIT_EMPTY, inst_list, True);

      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
      case IM_STATE_PREDICTION:
       return change_im_state(session, IM_STATE_EDITING, inst_list, True);

      default:
       ASSERT(0); /* cannot happen */
    }
    return False;
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
