/*

Copyright (c) 2006 Kenji ABE <abek@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/

#include <stdlib.h>
#include <string.h>

#include <anthy/anthy.h>

#include "dynbuff.h"
#include "convdef.h"
#include "imconv.h"
#include "impreedit.h"
#include "imtext.h"


#define SPECIAL_CAND_RAW (-1)
#define SPECIAL_CAND_ANY (-2)
#define IS_SPECIAL_CAND_NO(n) ((n) < 0)


typedef struct im_conv_seg_rec {
    struct im_conv_seg_rec *prev;
    struct im_conv_seg_rec *next;

    int index;

    int n_cand;
    int cand_no;

    dyn_buffer_t buff;
    int length;
    int org_length;
} im_conv_seg;

typedef struct im_conv_rec {
    anthy_context_t actx;

    IM_CONV_MODE mode;

    im_conv_seg* head;
    im_conv_seg* tail;
    im_conv_seg* cur;

    int length;

    dyn_buffer_t buff;
} im_conv;


static im_conv_seg *im_conv_create_seg(im_conv_t conv, int index, int n_cand, int cand_no, int org_len);
static void im_conv_free_seg(im_conv_seg *seg);

static void im_conv_insert_seg(im_conv_t conv, im_conv_seg *new_seg, im_conv_seg *prev, im_conv_seg *next);
static void im_conv_cut_seg(im_conv_t conv, im_conv_seg *seg);

static Bool im_conv_init_seg(im_conv_t conv,  im_conv_seg *seg, int index, int n_cand, int cand_no, int org_len);
static Bool im_conv_update_seg(im_conv_t conv, im_conv_seg *seg, int new_cand_no);

static int im_conv_seg_pos(im_conv_seg *seg);

static Bool im_conv_cur_select_sub(im_conv_t conv, int new_cand_no);

static IMText *im_conv_make_text_sub(im_conv_t conv, iml_session_t *session, int deco, int *cur_head, int *cur_length);
/* static IMText *im_conv_make_seg_text(im_conv_t conv, im_conv_seg* seg, int deco, iml_session_t *session); */

static int im_conv_get_seg_cand_string(im_conv_t conv, im_conv_seg *seg, int cand_no, dyn_buffer_t buff, Bool force_p);


static const char ANTHY_ENCODE[] = "EUC-JP";


static Bool init_p = False;
static conv_handle enc_conv = NULL;


Bool
im_conv_init(void)
{
    if (!init_p) {
	if (anthy_init() != 0) {
	    goto ERR_EXIT;
	}
	init_p = True;
    }

    if (enc_conv == NULL) {
	enc_conv = create_converter(ANTHY_ENCODE, CONVERT_BOTH);
	if (enc_conv == NULL) {
	    goto ERR_EXIT;
	}
    }

    return True;

ERR_EXIT:
    im_conv_term();
    return False;
}

void
im_conv_term(void)
{
    if (init_p) {
/*
	anthy_quit();
	init_p = False;
*/
    }

    if (enc_conv) {
	destroy_converter(enc_conv);
	enc_conv = NULL;
    }
}

im_conv_t
im_conv_create(void)
{
    im_conv_t conv;

    if (!init_p || enc_conv==NULL) {
	return False;
    }

    conv = calloc(1, sizeof(im_conv));
    if (conv) {
	conv->actx = anthy_create_context();
	if (conv->actx) {
	    conv->mode = IM_CONV_MODE_NORMAL;
	    conv->buff = dyn_buffer_create();
	    if (conv->buff) {
		return conv;		    
	    }
	}
	
	im_conv_free(conv);
    }
    return NULL;
}

void
im_conv_free(
    im_conv_t conv
)
{
    if (conv) {
	im_conv_reset(conv);

	if (conv->actx) {
	    anthy_release_context(conv->actx);
	}

	if (conv->buff) {
	    dyn_buffer_free(conv->buff);
	}

	free(conv);
    }
}

void
im_conv_reset(
    im_conv_t conv
)
{
    if (conv) {
	im_conv_seg *seg, *next;

	anthy_reset_context(conv->actx);

	seg = conv->head;

	conv->head = conv->tail = conv->cur = NULL;
	conv->length = 0;

	while (seg) {
	    next = seg->next;
	    im_conv_free_seg(seg);
	    seg = next;
	}
    }
}

Bool
im_conv_set_string(
    im_conv_t conv,
    const UTFCHAR *str,
    int len,
    IM_CONV_MODE mode
)
{
    int buff_len;
    conv_result r;
    char *p;
    im_conv_seg *seg;

    if (conv == NULL) {
	return False;
    }

    im_conv_reset(conv);

    if (!convert_from_utf16(enc_conv, str, len, conv->buff, &r)) {
	return False;
    }

    buff_len = r.dest_length;

    if (!DYN_BUFFER_ENSURE_SIZE(conv->buff, char, buff_len + 1)) {
	return False;
    }
    p = DYN_BUFFER_GET_BUFFER(conv->buff, char);
    p[buff_len] = '\0';

    conv->mode = mode;
    switch (mode) {
      case IM_CONV_MODE_NORMAL:
       {
	   struct anthy_conv_stat stat;
	   struct anthy_segment_stat seg_stat;
	   int i;

	   if ((anthy_set_string(conv->actx, p) != 0) ||
	       (anthy_get_stat(conv->actx, &stat) != 0)) {
	       return False;
	   }

	   for (i=0; i<stat.nr_segment; ++i) {
	       anthy_get_segment_stat(conv->actx, i, &seg_stat);

	       seg = im_conv_create_seg(conv, i, seg_stat.nr_candidate, 0, -1);
	       if (seg == NULL) {
		   return False;
	       }

	       im_conv_insert_seg(conv, seg, conv->tail, NULL);
	   }
       }
       break;

      case IM_CONV_MODE_PREDICT:
#ifdef HAS_ANTHY_PREDICTION
       {
	   struct anthy_prediction_stat stat;

	   if ((anthy_set_prediction_string(conv->actx, p) != 0) ||
	       (anthy_get_prediction_stat(conv->actx, &stat) != 0) ||
	       (stat.nr_prediction <= 0)) {
	       return False;
	   }

	   seg = im_conv_create_seg(conv, 0, stat.nr_prediction, 0, len);
	   if (seg == NULL) {
	       return False;
	   }

	   im_conv_insert_seg(conv, seg, conv->tail, NULL);
       }
       break;
#else
       return False;
#endif

      default:
       ASSERT(0);
       return False;
    }

    conv->cur = conv->head;

    return True;
}

static
im_conv_seg *
im_conv_create_seg(
    im_conv_t conv,
    int index,
    int n_cand,
    int cand_no,
    int org_len
)
{
    im_conv_seg *seg = calloc(1, sizeof(im_conv_seg));
    if (seg) {
	seg->buff = dyn_buffer_create();
	if (seg->buff) {
	    if (im_conv_init_seg(conv, seg, index, n_cand, cand_no, org_len)) {
		return seg;
	    }
	}

	im_conv_free_seg(seg);
    }
    return NULL;
}

static
void
im_conv_free_seg(
    im_conv_seg *seg
)
{
    ASSERT(seg);
    if (seg->buff) {
	dyn_buffer_free(seg->buff);
    }
    free(seg);
}

static
void
im_conv_insert_seg(
    im_conv_t conv,
    im_conv_seg *new_seg,
    im_conv_seg *prev,
    im_conv_seg *next
)
{
    ASSERT(conv);
    ASSERT(new_seg);

    if (prev) {
	prev->next = new_seg;
    } else {
	conv->head = new_seg;
    }

    new_seg->prev = prev;
    new_seg->next = next;

    if (next) {
	next->prev = new_seg;
    } else {
	conv->tail = new_seg;
    }

    conv->length += new_seg->length;
}

static
void
im_conv_cut_seg(
    im_conv_t conv,
    im_conv_seg *seg
)
{
    im_conv_seg *prev, *next;

    ASSERT(seg);
    prev = seg->prev;
    next = seg->next;

    seg->prev = seg->next = NULL;

    if (prev) {
	prev->next = next;
    } else {
	conv->head = next;
    }

    if (next) {
	next->prev = prev;
    } else {
	conv->tail = prev;
    }

    conv->length -= seg->length;
}

static
Bool
im_conv_init_seg(
    im_conv_t conv,
    im_conv_seg *seg,
    int index,
    int n_cand,
    int cand_no,
    int org_len
)
{
    ASSERT(conv);
    ASSERT(seg);

    seg->index = index;
    seg->n_cand = n_cand;
    seg->cand_no = -1;
    seg->org_length = org_len;

    if (conv->mode == IM_CONV_MODE_NORMAL &&
	org_len <= 0) {
	if (!im_conv_update_seg(conv, seg, SPECIAL_CAND_RAW)) {
	    return False;
	}
	seg->org_length = seg->length;
    }

    return im_conv_update_seg(conv, seg, cand_no);
}

static
Bool
im_conv_update_seg(
    im_conv_t conv,
    im_conv_seg *seg,
    int new_cand_no
)
{
    int org_length = seg->length;

    ASSERT(conv);
    ASSERT(seg);

    seg->length = im_conv_get_seg_cand_string(conv, seg, new_cand_no, seg->buff, True);
    seg->cand_no = new_cand_no;

    if (org_length != seg->length) {
	if (seg->prev ||
	    seg->next ||
	    seg == conv->head) {
	    conv->length -= org_length;
	    conv->length += seg->length;
	}
    }

    return True;
}

int
im_conv_length(
    im_conv_t conv
)
{
    return conv ? conv->length : 0;
}

int
im_conv_cur_pos(
    im_conv_t conv
)
{
    return conv ? im_conv_seg_pos(conv->cur) : 0;
}

int
im_conv_cur_cand_index(
    im_conv_t conv
)
{
    if (conv &&
	conv->cur &&
	!IS_SPECIAL_CAND_NO(conv->cur->cand_no)) {
	return conv->cur->cand_no;
    }

    return 0;
}

int
im_conv_cur_cand_count(
    im_conv_t conv
)
{
    return (conv && conv->cur) ? conv->cur->n_cand : 0;
}

static
int
im_conv_seg_pos(
    im_conv_seg *seg
)
{
    int offset = 0;
    if (seg) {
	seg = seg->prev;
	while (seg) {
	    offset += seg->length;
	    seg = seg->prev;
	}
    }
    return offset;
}

Bool
im_conv_cur_get_unconverted_range(
    im_conv_t conv,
    int *start,
    int *end
)
{
    if (conv && conv->cur && start && end) {
	im_conv_seg *seg = conv->cur->prev;
	int offset = 0;
	while (seg) {
	    offset += seg->org_length;
	    seg = seg->prev;
	}

	*start = offset;
	*end = offset + conv->cur->org_length;
    }

    return False;
}


Bool
im_conv_cur_move_to_head(
    im_conv_t conv
)
{
    return conv && 
	conv->cur &&
	conv->cur != conv->head &&
	(conv->cur = conv->head, True);
}

Bool
im_conv_cur_move_to_tail(
    im_conv_t conv
)
{
    return conv && 
	conv->cur &&
	conv->cur != conv->tail &&
	(conv->cur = conv->tail, True);
}

Bool
im_conv_cur_move_prev(
    im_conv_t conv
)
{
    return conv && 
	conv->cur &&
	conv->cur->prev &&
	(conv->cur = conv->cur->prev, True);
}

Bool
im_conv_cur_move_next(
    im_conv_t conv
)
{
    return conv && 
	conv->cur &&
	conv->cur->next &&
	(conv->cur = conv->cur->next, True);
}

Bool
im_conv_cur_resize(
    im_conv_t conv,
    int delta
)
{
    if (conv &&
	conv->mode == IM_CONV_MODE_NORMAL &&
	conv->cur) {
	struct anthy_conv_stat stat;
	struct anthy_segment_stat seg_stat;
	im_conv_seg *seg, *next;
	int i;

	anthy_resize_segment(conv->actx, conv->cur->index, delta);

	anthy_get_stat(conv->actx, &stat);

	i = conv->cur->index;
	anthy_get_segment_stat(conv->actx, i, &seg_stat);
	im_conv_init_seg(conv, conv->cur, i, seg_stat.nr_candidate, SPECIAL_CAND_RAW, -1);
	++i;
	seg = conv->cur->next;
	for (; i<stat.nr_segment; ++i, seg = seg->next) {
	    anthy_get_segment_stat(conv->actx, i, &seg_stat);
	    if (seg) {
		if (!im_conv_init_seg(conv, seg, i, seg_stat.nr_candidate, 0, -1)) {
		    return False;
		}
	    } else {
		seg = im_conv_create_seg(conv, i, seg_stat.nr_candidate, 0, -1);
		if (seg == NULL) {
		    return False;
		}

		im_conv_insert_seg(conv, seg, conv->tail, NULL);
	    }
	}

	if (seg) {
	    while (seg) {
		next = seg->next;
		im_conv_cut_seg(conv, seg);
		im_conv_free_seg(seg);
		seg = next;
	    }
	}

	return True;
    }

    return False;
}

Bool
im_conv_cur_select_next(
    im_conv_t conv
)
{
    if (conv && conv->cur) {
	if (conv->cur->n_cand > 1) {
	    int new_cand_no = conv->cur->cand_no;

	    if (IS_SPECIAL_CAND_NO(new_cand_no)) {
		new_cand_no = 0;
	    } else {
		++new_cand_no;
		if (conv->cur->n_cand <= new_cand_no) {
		    new_cand_no = 0;
		}
	    }

	    return im_conv_cur_select_sub(conv, new_cand_no);
	}
    }

    return False;
}

Bool
im_conv_cur_select_prev(
    im_conv_t conv
)
{
    if (conv && conv->cur) {
	if (conv->cur->n_cand > 1) {
	    int new_cand_no = conv->cur->cand_no;

	    if (IS_SPECIAL_CAND_NO(new_cand_no)) {
		new_cand_no = conv->cur->n_cand -1;
	    } else {
		--new_cand_no;
		if (new_cand_no < 0) {
		    new_cand_no = conv->cur->n_cand -1;
		}
	    }


	    return im_conv_cur_select_sub(conv, new_cand_no);
	}
    }

    return False;
}

Bool
im_conv_cur_select_cand(
    im_conv_t conv,
    int cand_no
)
{
    if (conv && conv->cur &&
	(IS_SPECIAL_CAND_NO(cand_no) ||
	 (0 <= cand_no && cand_no < conv->cur->n_cand))) {
	return im_conv_cur_select_sub(conv, cand_no);
    }
    
    return False;
}

Bool
im_conv_cur_select_unconverted(
    im_conv_t conv
)
{
    return im_conv_cur_select_cand(conv, SPECIAL_CAND_RAW);
}

static
Bool
im_conv_cur_select_sub(
    im_conv_t conv,
    int new_cand_no
)
{
    ASSERT(conv);
    ASSERT(conv->cur);
    ASSERT(IS_SPECIAL_CAND_NO(new_cand_no) || (0 <= new_cand_no && new_cand_no < conv->cur->n_cand));

    if (conv->cur->cand_no == new_cand_no) {
	return True;
    }

    return im_conv_update_seg(conv, conv->cur, new_cand_no);
}

Bool
im_conv_cur_set_string(
    im_conv_t conv,
    const UTFCHAR *str,
    int length
)
{
    if (conv && conv->cur) {
	if (!DYN_BUFFER_ENSURE_SIZE(conv->cur->buff, UTFCHAR, length)) {
	    return False;
	}

	conv->length -= conv->cur->length;

	memmove(DYN_BUFFER_GET_BUFFER(conv->cur->buff, UTFCHAR), str, length * sizeof(UTFCHAR));
	conv->cur->cand_no = SPECIAL_CAND_ANY;
	conv->cur->length = length;

	conv->length += conv->cur->length;

	return True;
    }
    
    return False;
}

IMText *
im_conv_commit(
    im_conv_t conv,
    iml_session_t *session
)
{
    int h, l;
    IMText *txt;
    im_conv_seg *seg;

    txt = im_conv_make_text_sub(conv, session, 0, &h, &l);

    for (seg=conv->head; seg; seg = seg->next) {
	anthy_commit_segment(conv->actx, seg->index, seg->cand_no);
    }

    im_conv_reset(conv);

    return txt;
}

IMText *
im_conv_make_preedit_text(
    im_conv_t conv,
    iml_session_t *session
)
{
    IMText *txt;
    int cur_head, cur_len;

    txt = im_conv_make_text_sub(conv, session, IMUnderline, &cur_head, &cur_len);
    if (txt && txt->char_length > 0) {
	IMFeedbackList *fbl = txt->feedback + cur_head;
	int i;

	for (i=0; i<cur_len; ++i, ++fbl) {
	    ASSERT(fbl->count_feedbacks == 1);
	    fbl->feedbacks[0].type = IM_DECORATION_FEEDBACK;
	    fbl->feedbacks[0].value = IMReverse;
	}
    }
    return txt;
}

static
IMText *
im_conv_make_text_sub(
    im_conv_t conv,
    iml_session_t *session,
    int deco,
    int *cur_head,
    int *cur_length
)
{
    IMText *txt;
    int len;

    ASSERT(conv);
    ASSERT(session);
    ASSERT(cur_head);
    ASSERT(cur_length);

    len = im_conv_length(conv);
    txt = make_im_text(session, NULL, len, deco);
    if (txt && len > 0) {
	UTFCHAR *p = txt->text.utf_chars;
	im_conv_seg *seg;
	int offset = 0;

	for (seg=conv->head; seg; seg = seg->next) {
	    memmove(p, DYN_BUFFER_GET_BUFFER(seg->buff, UTFCHAR), seg->length * sizeof(UTFCHAR));
	    p += seg->length;

	    if (seg == conv->cur) {
		*cur_head = offset;
		*cur_length = seg->length;
	    }

	    offset += seg->length;
	    ASSERT(offset<=len);
	}
    }
    return txt;
}

/* static */
/* IMText * */
/* im_conv_make_seg_text( */
/*     im_conv_t conv, */
/*     im_conv_seg* seg, */
/*     int deco, */
/*     iml_session_t *session */
/* ) */
/* { */
/*     ASSERT(conv); */
/*     ASSERT(seg); */
/*     ASSERT(session); */
/*     return make_im_text(session, DYN_BUFFER_GET_BUFFER(seg->buff, UTFCHAR), seg->length, deco); */
/* } */

IMText *
im_conv_make_cur_cand_text(
    im_conv_t conv,
    int cand_no,
    iml_session_t *session
)
{
    int len = 0;
    if (conv && session) {
	len = im_conv_get_seg_cand_string(conv, conv->cur, cand_no, conv->buff, False);
    }
    return make_im_text(session, DYN_BUFFER_GET_BUFFER(conv->buff, UTFCHAR), len, 0);
}

static
int
im_conv_get_seg_cand_string(
    im_conv_t conv,
    im_conv_seg *seg,
    int cand_no,
    dyn_buffer_t buff,
    Bool force_p
)
{
    int len;
    char *p;
    conv_result r;

    ASSERT(conv);
    ASSERT(seg);
    ASSERT(IS_SPECIAL_CAND_NO(cand_no) || (0 <= cand_no && cand_no < seg->n_cand));
    ASSERT(buff);

    if (!force_p &&
	cand_no == seg->cand_no &&
	seg->length > 0) {
	if (buff != seg->buff) {
	    if (!DYN_BUFFER_ENSURE_SIZE(buff, UTFCHAR, seg->length)) {
		return 0;
	    }

	    memmove(DYN_BUFFER_GET_BUFFER(buff, UTFCHAR), DYN_BUFFER_GET_BUFFER(seg->buff, UTFCHAR), seg->length * sizeof(UTFCHAR));
	}
	return seg->length;
    }

    switch (conv->mode) {
      case IM_CONV_MODE_NORMAL:
       if (cand_no == SPECIAL_CAND_RAW) {
	   cand_no = NTH_UNCONVERTED_CANDIDATE;
       } else if (IS_SPECIAL_CAND_NO(cand_no)) {
	   return 0;
       }

       len = anthy_get_segment(conv->actx, seg->index, cand_no, NULL, 0);
       ASSERT(len > 0);
       p = alloca(len+1);
       ASSERT(p);
       anthy_get_segment(conv->actx, seg->index, cand_no, p, len+1);
       break;

      case IM_CONV_MODE_PREDICT:
#ifdef HAS_ANTHY_PREDICTION
       len = anthy_get_prediction(conv->actx, cand_no, NULL, 0);
       ASSERT(len > 0);
       p = alloca(len+1);
       ASSERT(p);
       anthy_get_prediction(conv->actx, cand_no, p, len+1);
       break;
#else
       return 0;
#endif

      default:
       ASSERT(0);
       return 0;
    }

    convert_to_utf16(enc_conv, p, len, buff, &r);
    return r.dest_length;
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
