/*

Copyright (c) 2006 Kenji ABE <abek@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/

#include <iconv.h>
#include <stdlib.h>
#include <string.h>

#include "SunIM.h"

#include "convdef.h"

#ifdef sun
#define NO_DIRECT_CONVERT_TO_UTF16
#endif

#define INVALID_ICONV_T ((iconv_t)-1)
#define MAX_BYTE_OF_ONE_CHAR (4)

static const UTFCHAR BOM_MARK = 0xFEFF;
#ifdef NO_DIRECT_CONVERT_TO_UTF16
static const char UTF8_ENC[] = "UTF-8";
#endif
static const char UTF16_ENC[] = "UTF-16";

struct conv_data_t {
#ifdef NO_DIRECT_CONVERT_TO_UTF16
    iconv_t native_to_utf8;
    iconv_t utf8_to_utf16;

    iconv_t utf16_to_utf8;
    iconv_t utf8_to_native;
#else
    iconv_t to_utf16;
    iconv_t from_utf16;
#endif
};


conv_handle
create_converter(
    const char *encoding,
    enum CONVERT_TYPE type
)
{
    conv_handle p = malloc(sizeof(struct conv_data_t));
    if (p) {
#ifdef NO_DIRECT_CONVERT_TO_UTF16
	p->native_to_utf8 = INVALID_ICONV_T;
	p->utf8_to_utf16 = INVALID_ICONV_T;

	p->utf16_to_utf8 = INVALID_ICONV_T;
	p->utf8_to_native = INVALID_ICONV_T;

	if (type & CONVERT_TO_UTF16) {
	    p->native_to_utf8 = iconv_open(UTF8_ENC, encoding);
	    if (p->native_to_utf8 == INVALID_ICONV_T) {
		goto ON_ERROR;
	    }

	    p->utf8_to_utf16 = iconv_open(UTF16_ENC, UTF8_ENC);
	    if (p->utf8_to_utf16 == INVALID_ICONV_T) {
		goto ON_ERROR;
	    }
	}

	if (type & CONVERT_FROM_UTF16) {
	    p->utf16_to_utf8 = iconv_open(UTF8_ENC, UTF16_ENC);
	    if (p->utf16_to_utf8 == INVALID_ICONV_T) {
		goto ON_ERROR;
	    }

	    p->utf8_to_native = iconv_open(encoding, UTF8_ENC);
	    if (p->utf8_to_native == INVALID_ICONV_T) {
		goto ON_ERROR;
	    }
	}
#else
	p->to_utf16 = INVALID_ICONV_T;
	p->from_utf16 = INVALID_ICONV_T;

	if (type & CONVERT_TO_UTF16) {
	    p->to_utf16 = iconv_open(UTF16_ENC, encoding);
	    if (p->to_utf16 == INVALID_ICONV_T) {
		goto ON_ERROR;
	    }
	}

	if (type & CONVERT_FROM_UTF16) {
	    p->from_utf16 = iconv_open(encoding, UTF16_ENC);
	    if (p->from_utf16 == INVALID_ICONV_T) {
		goto ON_ERROR;
	    }
	}
#endif
	return p;

ON_ERROR:
	destroy_converter(p);
    }
    return NULL;
}

void
destroy_converter(
    conv_handle conv
)
{
    if (conv) {
#ifdef NO_DIRECT_CONVERT_TO_UTF16
	if (conv->native_to_utf8 != INVALID_ICONV_T) {
	    iconv_close(conv->native_to_utf8);
	}

	if (conv->utf8_to_utf16 != INVALID_ICONV_T) {
	    iconv_close(conv->utf8_to_utf16);
	}

	if (conv->utf16_to_utf8 != INVALID_ICONV_T) {
	    iconv_close(conv->utf16_to_utf8);
	}

	if (conv->utf8_to_native != INVALID_ICONV_T) {
	    iconv_close(conv->utf8_to_native);
	}
#else
	if (conv->to_utf16 != INVALID_ICONV_T) {
	    iconv_close(conv->to_utf16);
	}

	if (conv->from_utf16 != INVALID_ICONV_T) {
	    iconv_close(conv->from_utf16);
	}
#endif
	free(conv);
    }
}

enum CONVERT_RESULT
convert_to_utf16(
    conv_handle conv,
    const char *src_buff,
    int src_buff_len,
    dyn_buffer_t dest_buff,
    conv_result *result
)
{
#ifdef NO_DIRECT_CONVERT_TO_UTF16
    const char *in, *in0;
    char *out, *out0;
    size_t in_len, out_len, in_converted, out_converted;
    char buff[1024];
    conv_result r;

    ASSERT(conv);
    ASSERT(conv->native_to_utf8 != INVALID_ICONV_T);
    ASSERT(conv->utf8_to_utf16 != INVALID_ICONV_T);
    ASSERT(src_buff);
    ASSERT(src_buff_len > 0);
    ASSERT(dest_buff);

    if (result == NULL) {
	result = &r;
    }

    result->result = CONVERT_FAIL;
    result->src_length = 0;
    result->dest_length = 0;

    while (src_buff_len > 0) {
	in = src_buff;
	in_len = src_buff_len;
	out = buff;
	out_len = sizeof(buff);
	if (iconv(conv->native_to_utf8, &in, &in_len, &out, &out_len) == -1) {
	    return result->result;
	}

	in_converted = in - src_buff;
	out_converted = out - buff;

	src_buff += in_converted;
	src_buff_len -= in_converted;
	result->src_length += in_converted;

	in = buff;
	in_len = out_converted;
	while (in_len > 0) {
	    if (!DYN_BUFFER_ENSURE_SIZE(dest_buff, UTFCHAR, result->dest_length + in_len)) {
		return result->result;
	    }

	    in0 = in;
	    out0 = out = (char*)(DYN_BUFFER_GET_BUFFER(dest_buff, UTFCHAR) + result->dest_length);
	    out_len = (DYN_BUFFER_GET_SIZE(dest_buff, UTFCHAR) - result->dest_length) * sizeof(UTFCHAR);
	    if (iconv(conv->utf8_to_utf16, &in, &in_len, &out, &out_len) == -1) {
		return result->result;
	    }

	    in_converted = in - in0;
	    out_converted = (out - out0) / sizeof(UTFCHAR);

	    if (((UTFCHAR*)out0)[0] == BOM_MARK) {
		--out_converted;
		memmove(out0, out0 + sizeof(UTFCHAR), out_converted * sizeof(UTFCHAR));
	    }

	    result->dest_length += out_converted;
	}
    }

    result->result = CONVERT_SUCCESS;

    return result->result;
#else
    char *in, *in0, *out, *out0;
    size_t in_len, out_len, in_converted, out_converted;
    conv_result r;

    ASSERT(conv);
    ASSERT(conv->to_utf16 != INVALID_ICONV_T);
    ASSERT(src_buff);
    ASSERT(src_buff_len);
    ASSERT(dest_buff);

    if (result == NULL) {
	result = &r;
    }

    result->result = CONVERT_FAIL;
    result->src_length = 0;
    result->dest_length = 0;

    in = (char*)src_buff;
    in_len = src_buff_len;
    while (in_len > 0) {
	if (!DYN_BUFFER_ENSURE_SIZE(dest_buff, UTFCHAR, result->dest_length + in_len)) {
	    return result->result;
	}

	in0 = in;
	out0 = out = (char*)(DYN_BUFFER_GET_BUFFER(dest_buff, UTFCHAR) + result->dest_length);
	out_len = (DYN_BUFFER_GET_SIZE(dest_buff, UTFCHAR) - result->dest_length) * sizeof(UTFCHAR);
	if (iconv(conv->to_utf16, &in, &in_len, &out, &out_len) == -1) {
	    return result->result;
	}

	in_converted = in - in0;
	out_converted = (out - out0) / sizeof(UTFCHAR);

	if (((UTFCHAR*)out0)[0] == BOM_MARK) {
	    --out_converted;
	    memmove(out0, out0 + sizeof(UTFCHAR), out_converted * sizeof(UTFCHAR));
	}

	result->src_length += in_converted;
	result->dest_length += out_converted;
    }

    result->result = CONVERT_SUCCESS;

    return result->result;
#endif
}

enum CONVERT_RESULT
convert_from_utf16(
    conv_handle conv,
    const UTFCHAR *src_buff,
    int src_buff_len,
    dyn_buffer_t dest_buff,
    conv_result *result
)
{
#ifdef NO_DIRECT_CONVERT_TO_UTF16
    const char *in;
    char *out, *out0;
    size_t in_len, out_len, in_converted, out_converted;
    char buff[1024];
    conv_result r;

    ASSERT(conv);
    ASSERT(conv->utf16_to_utf8 != INVALID_ICONV_T);
    ASSERT(conv->utf8_to_native != INVALID_ICONV_T);
    ASSERT(src_buff);
    ASSERT(src_buff_len);
    ASSERT(dest_buff);

    if (result == NULL) {
	result = &r;
    }

    result->result = CONVERT_FAIL;
    result->src_length = 0;
    result->dest_length = 0;

    while (src_buff_len > 0) {
	in = (char*)src_buff;
	in_len = src_buff_len * sizeof(UTFCHAR);
	out = buff;
	out_len = sizeof(buff);
	if (iconv(conv->utf16_to_utf8, &in, &in_len, &out, &out_len) == -1) {
	    return result->result;
	}

	in_converted = ((const UTFCHAR*)in) - src_buff;
	out_converted = out - buff;

	src_buff += in_converted;
	src_buff_len -= in_converted;
	result->src_length += in_converted;

	in = buff;
	in_len = out_converted;
	while (in_len > 0) {
	    if (!DYN_BUFFER_ENSURE_SIZE(dest_buff, char, result->dest_length + in_len * MAX_BYTE_OF_ONE_CHAR)) {
		return result->result;
	    }

	    out0 = out = DYN_BUFFER_GET_BUFFER(dest_buff, char) + result->dest_length;
	    out_len = DYN_BUFFER_GET_SIZE(dest_buff, char) - result->dest_length;
	    if (iconv(conv->utf8_to_native, &in, &in_len, &out, &out_len) == -1) {
		return result->result;
	    }

	    out_converted = out - out0;
	    result->dest_length += out_converted;
	}
    }
    result->result = CONVERT_SUCCESS;

    return result->result;
#else
    char *in, *in0, *out, *out0;
    size_t in_len, out_len, in_converted, out_converted;
    conv_result r;

    ASSERT(conv);
    ASSERT(conv->from_utf16 != INVALID_ICONV_T);
    ASSERT(src_buff);
    ASSERT(src_buff_len);
    ASSERT(dest_buff);

    if (result == NULL) {
	result = &r;
    }

    result->result = CONVERT_FAIL;
    result->src_length = 0;
    result->dest_length = 0;

    in = (char*)src_buff;
    in_len = src_buff_len * sizeof(UTFCHAR);
    while (in_len > 0) {
	if (!DYN_BUFFER_ENSURE_SIZE(dest_buff, char, result->dest_length + in_len * MAX_BYTE_OF_ONE_CHAR)) {
	    return result->result;
	}

	in0 = in;
	out0 = out = DYN_BUFFER_GET_BUFFER(dest_buff, char) + result->dest_length;
	out_len = DYN_BUFFER_GET_SIZE(dest_buff, char) - result->dest_length;
	if (iconv(conv->from_utf16, &in, &in_len, &out, &out_len) == -1) {
	    return result->result;
	}

	in_converted = (in - in0) / sizeof(UTFCHAR);
	out_converted = out - out0;

	result->src_length += in_converted;
	result->dest_length += out_converted;
    }
    result->result = CONVERT_SUCCESS;

    return result->result;
#endif
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
