/*
Copyright 1999-2004 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/

/* my_stpcpy is a copy of g_stpcpy() from glib-2, hence
   glib-2 copyright is included here.
   	Commentted by Hidetoshi Tajima <hidetoshi.tajima@sun.com>
 */

/* GLIB - Library of useful routines for C programming
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GLib Team and others 1997-2000.  See the AUTHORS
 * file for a list of people on the GLib Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GLib at ftp://ftp.gtk.org/pub/gtk/. 
 */

#include <string.h>
#include "iiimpIM.h"
#include "iiimpSwitcher.h"
#include "keysyms.h"

struct _SwitcherInfo
{
  Window switcher_window;
  Atom  selection_atom;
  Atom  set_current_input_language_atom;
  Atom  set_current_client_atom;
  Atom  set_status_text_atom;
  Atom  set_input_language_list_atom;
  Atom  set_conversion_mode_atom;
  Atom  set_hotkey_atom;
};

/* A listener window for input method switcher */
struct _SwitcherContext
{
  Window invisible;
};

static Atom selection = None;

/**
 * my_stpcpy: 
 * @dest: destination buffer.
 * @src: source string.
 * 
 * Copies a nul-terminated string into the dest buffer, include the
 * trailing nul, and return a pointer to the trailing nul byte.
 * This is useful for concatenating multiple strings together
 * without having to repeatedly scan for the end.
 * 
 * Return value: a pointer to trailing nul byte.
 **/
char *
my_stpcpy (char *dest, const char *src)
{
  if (dest == NULL || src == NULL)
    return NULL;
#ifdef HAVE_STPCPY
  return stpcpy (dest, src);
#else
  register char *d = dest;
  register const char *s = src;

  do
    *d++ = *s;
  while (*s++ != '\0');

  return d - 1;
#endif
}

Bool
im_switcher_active (XimCommon im)
{
  SwitcherInfo *sw_info = im->switcher_info;

  return (sw_info && (sw_info->switcher_window));
}

static Bool
filter_destroy_event(Display *d, Window w, XEvent *ev,
		     XPointer client_data)
{
  XimCommon im = (XimCommon)client_data;
  SwitcherInfo *sw_info = im->switcher_info;

  if (sw_info && sw_info->switcher_window == w)
    {
      sw_info->switcher_window = None;
      Xfree (sw_info);
      im->switcher_info = NULL;
      return True;
    }
  return False;
}

Bool
im_switcher_new (XimCommon im)
{
  Display *display = im->core.display;
  SwitcherInfo *sw_info = im->switcher_info;
  Window owner = None;

  if (sw_info == NULL)
    {
      sw_info = Xmalloc (sizeof(SwitcherInfo));
      if (sw_info == NULL)
	return False;
      memset(sw_info, 0, sizeof(SwitcherInfo));
      im->switcher_info = sw_info;
    }

  if (selection == None)
    selection  = XInternAtom (display, "_IIIM_SWITCHER", False);

  sw_info->selection_atom = selection;

  owner = XGetSelectionOwner (display, selection);
  if (owner == None)
    return False;

  sw_info->switcher_window = owner;

  _XRegisterFilterByType (display, owner,
			  DestroyNotify, DestroyNotify,
			  filter_destroy_event, (XPointer)im);

  XSelectInput (display, owner, StructureNotifyMask);

  sw_info->set_current_input_language_atom
    = XInternAtom (display, "_IIIM_SWITCHER_CURRENT_INPUT_LANGUAGE", False);

  sw_info->set_current_client_atom =
    XInternAtom (display, "_IIIM_SWITCHER_CURRENT_CLIENT", False);

  sw_info->set_status_text_atom =
    XInternAtom (display, "_IIIM_SWITCHER_STATUS_TEXT", False);

  sw_info->set_input_language_list_atom =
    XInternAtom (display, "_IIIM_SWITCHER_INPUT_LANGUAGE_LIST", False);

  sw_info->set_conversion_mode_atom = 
    XInternAtom (display, "_IIIM_SWITCHER_SET_CONVERSION_MODE", False);

  sw_info->set_hotkey_atom =
    XInternAtom (display, "_IIIM_SWITCHER_SET_HOTKEY", False);

  return True;
}

void
im_switcher_shutdown (XimCommon im)
{
  SwitcherInfo *sw_info = im->switcher_info;
  Display *display = im->core.display;

  if (sw_info && sw_info->switcher_window)
    _XUnregisterFilter (display, sw_info->switcher_window,
			filter_destroy_event, (XPointer)im);
}

#ifdef HAS_IIIM_PROPERTIES
static Atom desktop_lang_atom = None;

char *get_desktop_lang (XicCommon ic)
{
  XTextProperty text_props;
  Display *display = ic->core.im->core.display;

  text_props.value = NULL;
  if (desktop_lang_atom == None) {
    desktop_lang_atom = XInternAtom (display,
				     "_IIIM_SWITCHER_DESKTOP_INPUT_LANGUAGE",
				     True);
  }
  if (desktop_lang_atom != NULL) {
    XGetTextProperty (display,
		      RootWindow (display, DefaultScreen (display)),
		      &text_props,
		      desktop_lang_atom);
    if (text_props.value != NULL) {
      return (char *)text_props.value;
    }
  }
  return NULL;
}
#endif /* HAS_IIIM_PROPERTIES */
static Bool
property_notify_switcher_window(Display *d, Window w, XEvent *ev,
				XPointer client_data)
{
  XicCommon ic = (XicCommon)client_data;
  XimCommon im = (XimCommon)ic->core.im;
  Atom type;
  unsigned char *data = NULL;
  int format;
  unsigned long length = 0, bytes_after;
  Display *display = ic->core.im->core.display;
  SwitcherInfo *sw_info = im->switcher_info;

  if (!im_switcher_active (im))
      return False;

  if (ev->xproperty.atom == sw_info->set_current_input_language_atom)
    {
      XGetWindowProperty(display, w, ev->xproperty.atom, 0L, 1000000L,
			 False, AnyPropertyType, &type, &format,
			 &length, &bytes_after, (unsigned char **)&data);
      ic_initialize_with_input_language (ic, data);
      /* This callback to switcher is needed for ensuring switcher
	 always shows correct client language indicator.
	 The timing of focus-in event sometimes causes
	 switcher's language indicator inconsistency problem without this. */
      ic_switcher_set_input_language (ic, NULL);
      Xfree (data);
      return True;
    }
  if (ev->xproperty.atom == sw_info->set_conversion_mode_atom)
    {
      XGetWindowProperty(display, w, ev->xproperty.atom, 0L, 1000000L,
			 False, AnyPropertyType, &type, &format,
			 &length, &bytes_after, (unsigned char **)&data);
      ic_change_conversion_mode (ic, data);
      Xfree (data);
      return True;
    }
  return False;
}

static Bool
destroy_switcher_window(Display *d, Window window, XEvent *ev,
			XPointer client_data)
{
  XicCommon ic = (XicCommon)client_data;
  SwitcherContext *w = ic->switcher_context;

  if (!w)
    return False;
  XDestroyWindow (d, w->invisible);
  Xfree (w);
  ic->switcher_context = NULL;
  return False;
}

void
ic_switcher_new (XicCommon ic)
{
  SwitcherContext *w = Xmalloc(sizeof(SwitcherContext));
  XimCommon im = (XimCommon)ic->core.im;
  Display *display = im->core.display;

  if (w == NULL)
    return;

  memset(w, 0, sizeof(SwitcherContext));

  w->invisible = XCreateSimpleWindow(display, DefaultRootWindow(display),
				     0, 0, 1, 1, 0, 0, 0);
  XSelectInput(display, w->invisible,
	       PropertyChangeMask| StructureNotifyMask);

  _XRegisterFilterByType(display, w->invisible,
			 PropertyNotify, PropertyNotify,
			 property_notify_switcher_window, (XPointer)ic);
  _XRegisterFilterByType(display, w->invisible,
			 DestroyNotify, DestroyNotify,
			 destroy_switcher_window, (XPointer)ic);
  ic->switcher_context = w;
}

void
ic_switcher_finalize (XicCommon ic)
{
  SwitcherContext *w = ic->switcher_context;
  XimCommon im = (XimCommon)ic->core.im;
  Display *display = im->core.display;

  if (w == NULL)
    return;

  _XUnregisterFilter(display, w->invisible,
		     property_notify_switcher_window, (XPointer)ic);
  _XUnregisterFilter(display, w->invisible,
		     destroy_switcher_window, (XPointer)ic);

  XDestroyWindow (display, w->invisible);

  Xfree (w);
  ic->switcher_context = NULL;
  return;

}

void
ic_switcher_set_language_list (XicCommon ic,
			       IIIMCF_language *lang_list,
			       int n_lang)
{
  Display *display = ic->core.im->core.display;
  XimCommon im = (XimCommon)ic->core.im;
  SwitcherInfo *sw_info = im->switcher_info;
  char *languages;
  char *ptr;
  IIIMF_status st;
  char *langid;
  int i;
  size_t len;
  const char *separator = ";";
  size_t separator_len;

  if (!im_switcher_active (im))
    return;

  if (lang_list == NULL || n_lang == 0)
    return;

  /* First part, getting length */
  st = iiimcf_get_language_id (lang_list[0],
			       (const char **) &langid);
  if (st != IIIMF_STATUS_SUCCESS)
    return;

  separator_len = strlen (separator);
  len = strlen (langid);
  for (i = 1; i < n_lang; i++)
    {
      st = iiimcf_get_language_id (lang_list[i],
				   (const char **) &langid);
      if (st != IIIMF_STATUS_SUCCESS)
	continue;
      len += strlen (langid);
    }
  len += separator_len * (i - 1);

  /* Second part, building string */
  languages = Xmalloc (sizeof(char) * (len + 1));
  if (languages == NULL)
    return;
  memset(languages, 0, sizeof(char) * (len + 1));

  st = iiimcf_get_language_id (lang_list[0],
			       (const char **) &langid);
  ptr = my_stpcpy (languages, langid);
  for (i = 1; i < n_lang; i++)
    {
      ptr = my_stpcpy (ptr, separator);
      st = iiimcf_get_language_id (lang_list[i],
				   (const char **) &langid);
      if (st != IIIMF_STATUS_SUCCESS)
	continue;
      ptr = my_stpcpy (ptr, langid);
    }

  XChangeProperty (display, sw_info->switcher_window,
		   sw_info->set_input_language_list_atom,
		   sw_info->set_input_language_list_atom,
		   8,
		   PropModeReplace,
		   (unsigned char*)languages, len);
  Xfree (languages);
}

void
ic_switcher_set_status_text (XicCommon ic, char *utf8)
{
  Display *display = ic->core.im->core.display;
  XimCommon im = (XimCommon)ic->core.im;
  SwitcherInfo *sw_info;

  if (!im_switcher_active (im))
    {
      im_switcher_new (im);
      if (!im_switcher_active (im))
	return;
    }

  if (ic->switcher_context == NULL)
    ic_switcher_new (ic);

  sw_info = im->switcher_info;

  if (utf8 == NULL)
    return;

  XChangeProperty (display,
		   sw_info->switcher_window,
		   sw_info->set_status_text_atom,
		   sw_info->set_status_text_atom,
		   8,
		   PropModeReplace,
		   (unsigned char *)utf8,
		   strlen (utf8));
}

void
ic_switcher_set_focus (XicCommon ic)
{
  Display *display = ic->core.im->core.display;
  XimCommon im = (XimCommon)ic->core.im;
  SwitcherInfo *sw_info;
  SwitcherContext *w;
  extern char *im_canonicalize_langname(XimCommon);

  if (!im_switcher_active (im))
    {
      im_switcher_new (im);
      if (!im_switcher_active (im))
	return;
    }

  if (ic->switcher_context == NULL)
    ic_switcher_new (ic);

  w = ic->switcher_context;
  sw_info = im->switcher_info;
  
  /* to reflect current lang_id to gimlet */
  if (ic->current_language == NULL) {
    ic->current_language = im_canonicalize_langname (im); 
  }
  ic_switcher_set_input_language (ic, NULL);
  
  if (w && w->invisible)
    XConvertSelection (display,
		       sw_info->selection_atom,
		       sw_info->set_current_client_atom,
		       sw_info->set_current_client_atom,
		       w->invisible,
		       CurrentTime);
}

static void
ic_switcher_set_hotkey (XicCommon ic, char *hotkey)
{
  Display *display = ic->core.im->core.display;
  XimCommon im = (XimCommon)ic->core.im;
  SwitcherInfo *sw_info = im->switcher_info;

  if (!sw_info || !hotkey) return;

  XChangeProperty (display,
		   sw_info->switcher_window,
		   sw_info->set_hotkey_atom,
		   sw_info->set_hotkey_atom,
		   8,
		   PropModeReplace,
		   (unsigned char *)hotkey,
		   strlen (hotkey));
}

void
ic_switcher_set_input_language (XicCommon ic, char *input_lang)
{
  Display *display = ic->core.im->core.display;
  XimCommon im = (XimCommon)ic->core.im;
  SwitcherInfo *sw_info = im->switcher_info;

  if (!im_switcher_active(im))
    return;

  if (input_lang == NULL)
    input_lang = ic->current_language;

  if (input_lang == NULL)
    return;

  XChangeProperty (display,
		   sw_info->switcher_window,
		   sw_info->set_current_input_language_atom,
		   sw_info->set_current_input_language_atom,
		   8,
		   PropModeReplace,
		   (unsigned char *)input_lang,
		   strlen (input_lang) + 1);
}

void
ic_switcher_set_conversion_mode (XicCommon ic)
{
  Display *display = ic->core.im->core.display;
  XimCommon im = (XimCommon)ic->core.im;
  SwitcherInfo *sw_info = im->switcher_info;
  IIIMF_status st;
  int conversion_mode = CONV_OFF;
  long conversion_mode_long;

  if (!im_switcher_active (im))
    return;

  st = iiimcf_get_current_conversion_mode (XIC_IIIMP(ic, context),
					   &conversion_mode);
  conversion_mode_long = conversion_mode;
  XChangeProperty (display,
		   sw_info->switcher_window,
		   sw_info->set_conversion_mode_atom,
		   sw_info->set_conversion_mode_atom,
		   32,
		   PropModeReplace,
		   (unsigned char *)&conversion_mode_long,
		   1);
}

void
my_strfreev (char **string)
{
  char **p = string;

  if (!string) return;

  while (*p)
    {
      char *t = *p++;
      if (t) free (t);
    }

  free (string);
}

static int
my_strv_length (char **string)
{
  int n = 0;

  if (!string) return n;

  while (string[n]) ++n;

  return n;
}

char **
my_strsplit (const char *string,
	    const char *single_delimiter)
{
  char *remainder;
  char **str_array, *s;
  int n = 0;

  remainder = string;
  s = strstr (remainder, single_delimiter);
  while (s)
    {
      remainder = s + 1;
      s = strstr (remainder, single_delimiter);
      ++n;
    }
  if (*string) ++n;

  str_array = (char **) malloc (sizeof(char *) * (n + 1));

  n = 0;
  remainder = string;
  s = strstr (remainder, single_delimiter);
  while (s)
    {
      int len;
      char *new_string;

      len = s - remainder;
      str_array[n++] = new_string = malloc (len + 1);
      strncpy (new_string, remainder, len);
      new_string[len] = 0;

      remainder = s + 1;
      s = strstr (remainder, single_delimiter);
    }
  if (*string)
    str_array[n++] = strdup (remainder);

  str_array[n] = 0;
  
  return str_array;
}

static char *
ic_switcher_get_hotkey_with_type (char *string,
				  char *type)
{
  char *label_delimiter = ":";
  char **hotkeys = my_strsplit (string, label_delimiter);
  int num_hotkeys = 0, i;
  char *keys = NULL;

  num_hotkeys = my_strv_length (hotkeys) / 2;

  for (i = 0; i < num_hotkeys; ++i)
    {
      if (!strcasecmp (hotkeys[i * 2], type))
	{
	  keys = strdup (hotkeys[i * 2 + 1]);
	  break;
	}
    }

  my_strfreev (hotkeys);

  return keys;
}

static char *
ic_switcher_get_hotkey (XicCommon ic)
{
  IIIMCF_context context = XIC_IIIMP (ic, context);
  int num_hotkey = 0, i;
  IIIMCF_hotkey *hotkeys = NULL;
  char s[512], *p = s;

  iiimcf_get_hotkeys (context, &num_hotkey, &hotkeys);

  if (num_hotkey == 0) return NULL;

  memset (s, 0, 512);

  for (i = 0; i < num_hotkey; ++i)
    {
      char label_delimiter = ':';
      char key_delimiter = ',';
      int k;

      strcpy (p, hotkeys[i].hotkey_label);
      p += strlen (hotkeys[i].hotkey_label);
      *p++ = label_delimiter;

      for (k = 0; k < hotkeys[i].nkeys; ++k)
	{
	  IIIMCF_keyevent *key = hotkeys[i].keys + k;
	  unsigned int keyval;
	  char *keyname;

	  if (k) *p++ = key_delimiter;

	  if (key->modifier & IIIMF_CONTROL_MODIFIER)
	    {
	      strcpy (p, "Ctrl+");
	      p += strlen ("Ctrl+");
	    }
	  if (key->modifier & IIIMF_SHIFT_MODIFIER)
	    {
	      strcpy (p, "Shift+");
	      p += strlen ("Shift+");
	    }
	  if (key->modifier & IIIMF_ALT_MODIFIER)
	    {
	      strcpy (p, "Alt+");
	      p += strlen ("Alt+");
	    }

	  keyval = i2gcode (key->keycode);
	  keyname = x11_keyval_name (keyval);

	  if (keyname)
	    {
	      strcpy (p, keyname);
	      p += strlen (keyname);
	    }
	}

      if (i < num_hotkey - 1) *p++ = label_delimiter;
    }

  return strdup (s);
}

static void
change_hotkey_with_type (XicCommon ic,
			 char *type,
			 int num_keys,
			 IIIMCF_keyevent *keys)
{
  IIIMCF_context context = XIC_IIIMP (ic, context);
  XimCommon im = (XimCommon)ic->core.im;
  IIIMCF_handle handle = XIM_IIIMP (im, handle);
  int num_hotkey = 0, i;
  IIIMCF_hotkey *hotkeys = NULL;

  iiimcf_get_hotkeys (context, &num_hotkey, &hotkeys);

  if (num_hotkey == 0) return;

  for (i = 0; i < num_hotkey; ++i)
    {
      if (!strcasecmp (type, hotkeys[i].hotkey_label))
	{
	  int size = sizeof (IIIMCF_keyevent) * num_keys;

	  /*
	   * return if the hotkey setting remains unchanged
	   */
	  if (num_keys == hotkeys[i].nkeys)
	    {
	      int k;
	      Bool changed = False;
	      IIIMCF_keyevent *pk = keys, *ph = hotkeys[i].keys;

	      for (k=0; k < num_keys; ++k)
		if (pk[k].keycode != ph[k].keycode ||
		    pk[k].modifier != ph[k].modifier)
		  {
		    changed = True;
		    break;
		  }

              /* when the 1st focus-in within composeIM, the default trigger key is just ctrl+space,
                 we still need update the pkev_on data structure even the hotkeys are not changed */
              if (!strcasecmp (type, TRIGGER_KEY_LABEL) && num_keys != XIM_IIIMP(im, on_keys_num))
                break;

	      if (!strcasecmp (type, LANGLIST_MENU_LABEL) && num_keys > 0 &&
		  XIM_IIIMP (im, le_select_keys_num) == 0)
		break;
	      if (!strcasecmp (type, CYCLE_LANG_SWITCH_LABEL) && num_keys > 0 &&
		  XIM_IIIMP (im, le_cycle_keys_num) == 0)
		break;
	      if (!strcasecmp (type, RE_CYCLE_LANG_SWITCH_LABEL) && num_keys > 0 &&
		  XIM_IIIMP (im, le_reverse_cycle_keys_num) == 0)
		break;

	      if (!changed) return;
	    }

	  if (hotkeys[i].nkeys < num_keys)
	    {
	      free (hotkeys[i].keys);

	      hotkeys[i].keys = (char *)malloc (size);
	      memset (hotkeys[i].keys, 0, size);
	    }

	  hotkeys[i].nkeys = num_keys;
	  memcpy (hotkeys[i].keys, keys, size);

          /* hotkey type matched, no more loop needed */
          break;
	}
    }

  /* change the onkeys and off keys if the target type is TRIGGER_KEY_LABEL */
  if (handle && !strcasecmp (type, TRIGGER_KEY_LABEL))
    {
      int num_on_keys = 0, num_off_keys = 0;
      static int initial_num_on_keys = 0, initial_num_off_keys = 0;
      IIIMCF_keyevent *onkeys, *offkeys;

      iiimcf_get_trigger_keys (handle, &num_on_keys, &onkeys,
			       &num_off_keys, &offkeys);

      if (!initial_num_on_keys && !initial_num_off_keys)
	{
	  initial_num_on_keys = num_on_keys;
	  initial_num_off_keys = num_off_keys;
	}

      num_on_keys = initial_num_on_keys < num_keys ? initial_num_on_keys : num_keys;
      memcpy (onkeys, keys, sizeof (IIIMCF_keyevent) * num_on_keys);
      memset (onkeys + num_on_keys, 0, sizeof (IIIMCF_keyevent) * (initial_num_on_keys - num_on_keys));

      num_off_keys = initial_num_off_keys < num_keys ? initial_num_off_keys : num_keys;
      memcpy (offkeys, keys, sizeof (IIIMCF_keyevent) * num_off_keys);
      memset (offkeys + num_off_keys, 0, sizeof (IIIMCF_keyevent) * (initial_num_off_keys - num_off_keys));

      /* 
       * XIIIMP client also puts the on and off keys setting to pkev_on and pkev_off
       * in XIMIIimpIMRec struct, SwitchFilter uses the data to check whether the X key
       * event is trigger key or not, so the data should be modified also.
       */
      if (XIM_IS_IIIMP (im))
	{
	  IIIMCF_keyevent *pkev;

	  if (XIM_IIIMP (im, pkev_on)) 
	    {
	      Xfree (XIM_IIIMP (im, pkev_on));
	      XIM_IIIMP (im, on_keys_num) = 0;
	    }
	  if (XIM_IIIMP (im, pkev_off))
	    {
	      Xfree (XIM_IIIMP (im, pkev_off));
	      XIM_IIIMP (im, off_keys_num) = 0;
	    }

	  pkev = (IIIMCF_keyevent *)malloc (sizeof (IIIMCF_keyevent) * num_keys);
	  if (pkev)
	    {
	      memcpy (pkev, keys, sizeof (IIIMCF_keyevent) * num_keys);
	      XIM_IIIMP (im, on_keys_num) = num_keys;
	      XIM_IIIMP (im, pkev_on) = pkev;
	    }
	  pkev = (IIIMCF_keyevent *) malloc (sizeof (IIIMCF_keyevent) * num_keys);
	  if (pkev)
	    {
	      memcpy (pkev, keys, sizeof (IIIMCF_keyevent) * num_keys);
	      XIM_IIIMP (im, off_keys_num) = num_keys;
	      XIM_IIIMP (im, pkev_off) = pkev;
	    }
	}
    }
#ifdef HAS_IIIM_PROPERTIES
  else if (handle && !strcasecmp (type, LANGLIST_MENU_LABEL)) {
    if (XIM_IS_IIIMP (im)) {
      IIIMCF_keyevent *pkev;
      if (XIM_IIIMP (im, pkev_le_select)) {
	Xfree (XIM_IIIMP (im, pkev_le_select));
	XIM_IIIMP (im, le_select_keys_num) = 0;
      }
      pkev = (IIIMCF_keyevent *)malloc (sizeof (IIIMCF_keyevent) * num_keys);
      if (pkev) {
	memcpy (pkev, keys, sizeof (IIIMCF_keyevent) * num_keys);
	XIM_IIIMP (im, le_select_keys_num) = num_keys;
	XIM_IIIMP (im, pkev_le_select) = pkev;
      }
    }
  } else if (handle && !strcasecmp (type, CYCLE_LANG_SWITCH_LABEL)) {
      IIIMCF_keyevent *pkev;
      if (XIM_IIIMP (im, pkev_le_cycle)) {
	Xfree (XIM_IIIMP (im, pkev_le_cycle));
	XIM_IIIMP (im, le_cycle_keys_num) = 0;
      }
      pkev = (IIIMCF_keyevent *)malloc (sizeof (IIIMCF_keyevent) * num_keys);
      if (pkev) {
	memcpy (pkev, keys, sizeof (IIIMCF_keyevent) * num_keys);
	XIM_IIIMP (im, le_cycle_keys_num) = num_keys;
	XIM_IIIMP (im, pkev_le_cycle) = pkev;
      }
  } else if (handle && !strcasecmp (type, RE_CYCLE_LANG_SWITCH_LABEL)) {
      IIIMCF_keyevent *pkev;
      if (XIM_IIIMP (im, pkev_le_reverse_cycle)) {
	Xfree (XIM_IIIMP (im, pkev_le_reverse_cycle));
	XIM_IIIMP (im, le_reverse_cycle_keys_num) = 0;
      }
      pkev = (IIIMCF_keyevent *)malloc (sizeof (IIIMCF_keyevent) * num_keys);
      if (pkev) {
	memcpy (pkev, keys, sizeof (IIIMCF_keyevent) * num_keys);
	XIM_IIIMP (im, le_reverse_cycle_keys_num) = num_keys;
	XIM_IIIMP (im, pkev_le_reverse_cycle) = pkev;
      }
  }
#endif /* HAS_IIIM_PROPERTIES */
}

/*
 * the triggerkey is set by gimlet with set_hotkey_atom, gimlet will validate
 * its content before change the content of atom.
 * Usually the format of hotkey would take "Ctrl+space,Kanji,Shift+Kanji".
 */
static void
convert_hotkey_to_IIIMCF_keyevent (char *triggerkey,
				   int *num_keys,
				   IIIMCF_keyevent **keys)
{
  char *key_delimiter = ",";
  char **hotkeys = my_strsplit (triggerkey, key_delimiter);
  int i;

  if (hotkeys == NULL) return;

  *num_keys = my_strv_length (hotkeys);

  *keys = (IIIMCF_keyevent *)malloc (sizeof (IIIMCF_keyevent) * (*num_keys));
  memset (*keys, 0, sizeof (IIIMCF_keyevent) * (*num_keys));

  for (i=0; i < *num_keys; ++i)
    {
      char *key_separator = "+";
      char **k, **keys_text = my_strsplit (hotkeys[i], key_separator);

      k = keys_text;
      for (; *k; ++k)
	{
	  if (!strcasecmp (*k, "Ctrl"))
	    (*keys)[i].modifier |= IIIMF_CONTROL_MODIFIER;
	  else if (!strcasecmp (*k, "Shift"))
	    (*keys)[i].modifier |= IIIMF_SHIFT_MODIFIER;
	  else if (!strcasecmp (*k, "Alt"))
	    (*keys)[i].modifier |= IIIMF_ALT_MODIFIER;
	  else if (!strcasecmp (*k, "Control_L"))
            (*keys)[i].modifier |= IIIMCF_CONTROL_L_STATE;
          else if (!strcasecmp (*k, "Control_R"))
            (*keys)[i].modifier |= IIIMCF_CONTROL_R_STATE;
          else if (!strcasecmp (*k, "Shift_L"))
            (*keys)[i].modifier |= IIIMCF_SHIFT_L_STATE;
          else if (!strcasecmp (*k, "Shift_R"))
            (*keys)[i].modifier |= IIIMCF_SHIFT_R_STATE;
          else if (!strcasecmp (*k, "Alt_L"))
            (*keys)[i].modifier |= IIIMCF_ALT_L_STATE;
          else if (!strcasecmp (*k, "Alt_R"))
            (*keys)[i].modifier |= IIIMCF_ALT_R_STATE;
          else if (!strcasecmp (*k, "Meta_L"))
            (*keys)[i].modifier |= IIIMCF_META_L_STATE;
          else if (!strcasecmp (*k, "Meta_R"))
            (*keys)[i].modifier |= IIIMCF_META_R_STATE;
          else
	    {
	      unsigned int keyval = x11_keyval_from_name (*k);
	      int keycode = g2icode (keyval);
	      (*keys)[i].keycode = keycode < 0 ? -keycode : keycode;
	    }
	}

      my_strfreev (keys_text);
    }

  my_strfreev (hotkeys);
}

static void
ic_switcher_change_hotkey_with_type (XicCommon ic,
				     char *type,
				     char *string)
{
  int num_keys;
  IIIMCF_keyevent *keys;

  convert_hotkey_to_IIIMCF_keyevent (string, &num_keys, &keys);

  change_hotkey_with_type (ic, type, num_keys, keys);

  free (keys);
}

static char *
ic_switcher_get_hotkey_with_atom (XicCommon ic)
{
  Display *display = ic->core.im->core.display;
  XimCommon im = (XimCommon)ic->core.im;
  SwitcherInfo *sw_info = im->switcher_info;
  int format;
  unsigned long length, nitem;
  unsigned char *data = NULL;
  Atom type;
 
#ifdef HAS_IIIM_PROPERTIES
 {
   Atom hotkey_list_atom;
   XTextProperty text_props;

   text_props.value = NULL;
   hotkey_list_atom = XInternAtom (im->core.display,
				   "_IIIM_SWITCHER_HOTKEY_LIST", True);
   if (hotkey_list_atom != NULL)
     {
       XGetTextProperty (display,
			 RootWindow (display, DefaultScreen (display)),
			 &text_props,
			 hotkey_list_atom);
       data = text_props.value;
     }
 }
#else /* HAS_IIIM_PROPERTIES */
  if (!im_switcher_active (im))
    return NULL;

  XGetWindowProperty (display,
		      sw_info->switcher_window,
		      sw_info->set_hotkey_atom,
		      0L, 1000000L, False,
		      sw_info->set_hotkey_atom,
		      &type, &format, &nitem, &length, &data);
#endif /* HAS_IIIM_PROPERTIES */

  return data;
}

void
IIIMSetHotkey (XicCommon ic)
{
  XimCommon im = (XimCommon)ic->core.im;
  char *hotkey;

  if ((hotkey = ic_switcher_get_hotkey_with_atom (ic)) ||
      /* dirty hack - change the trigger key in Solaris CDE environment */
      (hotkey = ic_switcher_get_hotkey (ic)))
    {
      char *conversion_keys;
      char *langlist_menu_keys;
      char *cycle_lang_switch_keys;
      char *reverse_cycle_lang_switch_keys;

      conversion_keys = ic_switcher_get_hotkey_with_type (hotkey,
							  TRIGGER_KEY_LABEL);
      ic_switcher_change_hotkey_with_type (ic,
					   TRIGGER_KEY_LABEL,
					   conversion_keys);
      free (conversion_keys);
      
      langlist_menu_keys = ic_switcher_get_hotkey_with_type (hotkey,
							     LANGLIST_MENU_LABEL);
      if (langlist_menu_keys != NULL) {
	ic_switcher_change_hotkey_with_type (ic,
					     LANGLIST_MENU_LABEL,
					     langlist_menu_keys);
	free (langlist_menu_keys);
      }

      cycle_lang_switch_keys = ic_switcher_get_hotkey_with_type (hotkey,
								 CYCLE_LANG_SWITCH_LABEL);
      if (cycle_lang_switch_keys != NULL) {
	ic_switcher_change_hotkey_with_type (ic,
					     CYCLE_LANG_SWITCH_LABEL,
					     cycle_lang_switch_keys);
	free (cycle_lang_switch_keys);
      }

      reverse_cycle_lang_switch_keys = ic_switcher_get_hotkey_with_type (hotkey,
									 RE_CYCLE_LANG_SWITCH_LABEL);
      if (reverse_cycle_lang_switch_keys != NULL) {
	ic_switcher_change_hotkey_with_type (ic,
					     RE_CYCLE_LANG_SWITCH_LABEL,
					     reverse_cycle_lang_switch_keys);
	free (reverse_cycle_lang_switch_keys);
      }

    }

  if (hotkey) free (hotkey);
}
