#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#include <poll.h>
#include <sys/time.h>
#include <X11/Xmd.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include "Xlcint.h"

#include <iiimcf.h>

#include "iiimpAuxProxyClient.h"


/* XAUX PROXY service number */
#define XAUX_PROXY_SERVICE_NONE			(0)
#define XAUX_PROXY_SERVICE_AUX_SETVALUE		(1)
#define XAUX_PROXY_SERVICE_IM_ID		(2)
#define XAUX_PROXY_SERVICE_IC_ID		(3)
#define XAUX_PROXY_SERVICE_DATA_SET		(4)
#define XAUX_PROXY_SERVICE_DATA_GET		(5)
#define XAUX_PROXY_SERVICE_DISPLAY		(6)
#define XAUX_PROXY_SERVICE_WINDOW		(7)
#define XAUX_PROXY_SERVICE_POINT		(8)
#define XAUX_PROXY_SERVICE_POINT_CARET		(9)
#define XAUX_PROXY_SERVICE_UTF16_MB		(10)
#define XAUX_PROXY_SERVICE_MB_UTF16		(11)
#define XAUX_PROXY_SERVICE_COMPOSE		(12)
#define XAUX_PROXY_SERVICE_COMPOSE_SIZE		(13)
#define XAUX_PROXY_SERVICE_DECOMPOSE		(14)
#define XAUX_PROXY_SERVICE_DECOMPOSE_FREE	(15)
#define XAUX_PROXY_SERVICE_REGISTER_X_FILTER	(16)
#define XAUX_PROXY_SERVICE_UNREGISTER_X_FILTER	(17)
#define XAUX_PROXY_SERVICE_SERVER		(18)
#define XAUX_PROXY_SERVICE_CLIENT_WINDOW	(19)
#define XAUX_PROXY_SERVICE_FOCUS_WINDOW		(20)
#define XAUX_PROXY_SERVICE_SCREEN_NUMBER	(21)
#define XAUX_PROXY_SERVICE_POINT_SCREEN		(22)
#define XAUX_PROXY_SERVICE_POINT_CARET_SCREEN	(23)
#define XAUX_PROXY_SERVICE_GET_CONVERSION_MODE	(24)
#define XAUX_PROXY_SERVICE_SET_CONVERSION_MODE	(25)
#define XAUX_PROXY_SERVICE_AUX_GETVALUE		(26)
#define XAUX_PROXY_SERVICE_AUX_GET_FROM_ID	(27)
#define XAUX_PROXY_SERVICE_MAXIMUM		(27)

/* XAUX PROXY message type */
#define	XAUX_PROXY_MESSAGE_TYPE_NONE		(0)
#define	XAUX_PROXY_MESSAGE_TYPE_RECEIPT		(1)
#define	XAUX_PROXY_MESSAGE_TYPE_SERVICE		(2)
#define	XAUX_PROXY_MESSAGE_TYPE_SIMPLE		(3)
#define	XAUX_PROXY_MESSAGE_TYPE_VALUE		(4)

/* XAUX PROXY Window Name */
#define XAUX_PROXY_WINDOW_NAME_SERVER	"IIIM XAUX Proxy Server"
#define XAUX_PROXY_WINDOW_NAME_CLIENT	"IIIM XAUX Proxy Client"


/*
 * data type definition
 */

typedef struct xaux_proxy_message_queue {
    unsigned int			server_id;
    unsigned short			im_id;
    unsigned short			ic_id;
    Display *				display;
    Atom				atom;
    Window				window_target;
    Window				window_self;
    int					message_type;
    unsigned char *			message;
    size_t				message_size;
    time_t				timestamp;
    struct xaux_proxy_message_queue *	next;
} xaux_proxy_message_queue_t;

typedef struct xaux_proxy_xevent_list {
    XEvent				event;
    struct xaux_proxy_xevent_list *	next;
} xaux_proxy_xevent_list_t;

typedef struct xaux_proxy_ic {
    Display *			display;
    Atom			atom;
    Window			window_target;
    Window			window_self;
    int				server_id;
    int				im_id;
    int				ic_id;
    aux_t *			aux;
    Bool			aux_self;
    IIIMP_data_s *		data_s;
    struct xaux_proxy_ic *	next;
} xaux_proxy_ic_t;


/*
 * local function prototype
 */

static int
xaux_proxy_xerror_handler(
    Display *		d,
    XErrorEvent *	e);

static Status
xaux_proxy_XSendEvent(
    Display *	display,
    Window	w,
    Bool	propagate,
    int		event_mask,
    XEvent *	event_send);

static int
xaux_proxy_XChangeProperty(
    Display *		display,
    Window		w,
    Atom		property,
    Atom		type,
    int			format,
    int			mode,
    unsigned char *	data,
    int			nelements);

static int
xaux_proxy_XGetWindowProperty(
    Display *		display,
    Window		w,
    Atom		property,
    long		long_offset,
    long		long_length,
    Bool		delete,
    Atom		req_type,
    Atom *		actual_type_return,
    int *		actual_format_return,
    unsigned long *	nitems_return,
    unsigned long *	bytes_after_return,
    unsigned char **	prop_return);

static int
xaux_proxy_XSelectInput(
    Display *	display,
    Window	w,
    long	event_mask);

static int
xaux_proxy_XStoreName(
    Display *	display,
    Window	w,
    char *	window_name);

static Status
xaux_proxy_XFetchName(
    Display *	display,
    Window	w,
    char **	window_name_return);

static int
xaux_string_length(const IIIMP_card16 * p);

static xaux_proxy_message_queue_t *
xaux_proxy_message_queue_new(
    unsigned int	server_id,
    unsigned short	im_id,
    unsigned short	ic_id,
    Display *		display,
    Atom		atom,
    Window		window_target,
    Window		window_self,
    int			message_type,
    unsigned char *	message,
    size_t		message_size);

static void
xaux_proxy_message_queue_delete(
    xaux_proxy_message_queue_t *	m);

static void
xaux_proxy_message_queue_delete_all(void);

static void
xaux_proxy_message_queue_delete_display(
    Display *	display);

static void
xaux_proxy_message_queue_delete_window_self(
    Window	window);

static void
xaux_proxy_message_queue_delete_window_target(
    Window	window);

static void
xaux_proxy_message_queue_unsent_purge(void);

static void
xaux_proxy_message_queue_append(
    xaux_proxy_message_queue_t *	m);

static xaux_proxy_message_queue_t *
xaux_proxy_message_queue_pop(void);

static void
xaux_proxy_message_data_to_cm(
    Atom			atom,
    Window			window_receiver,
    Window			window_sender,
    int				server_id,
    unsigned short		im_id,
    unsigned short		ic_id,
    int				message_type,
    unsigned long		data0,
    unsigned long		data1,
    XClientMessageEvent *	cm);

static void
xaux_proxy_message_cm_to_data(
    XClientMessageEvent *	cm,
    unsigned long *		data0,
    unsigned long *		data1);

static Bool
xaux_proxy_message_short_exchange(
    Display *			display,
    Window			window_target,
    Window			window_self,
    Atom			atom,
    int				server_id,
    unsigned short		im_id,
    unsigned short		ic_id,
    int				message_type,
    unsigned long		data0,
    unsigned long		data1,
    unsigned long *		data0_ret,
    unsigned long *		data1_ret);

static void
xaux_proxy_message_send(
    Display *		display,
    Window		window_target,
    Window		window_self,
    Atom		atom,
    unsigned int	server_id,
    unsigned short	im_id,
    unsigned short	ic_id,
    int			message_type,
    unsigned char *	message,
    size_t		message_size);

static void
xaux_proxy_message_ping(
    Display *	display);

static void
xaux_proxy_message_send_cleanup(void);

static void
xaux_proxy_message_process(
    const xaux_proxy_ic_t *	ic,
    IIIMP_message *		m);

static Bool
xaux_proxy_window_target_filter(
    Display *	display,
    Window	window,
    XEvent *	event,
    XPointer	client_data);

static Bool
xaux_proxy_window_self_filter(
    Display *	display,
    Window	window,
    XEvent *	event,
    XPointer	client_data);

static xaux_proxy_ic_t *
xaux_proxy_ic_new(
    Display *		display,
    Window		window_target,
    Window		window_self,
    int			server_id,
    int			im_id,
    int			ic_id,
    aux_t *		aux);

static void
xaux_proxy_ic_delete_alone(xaux_proxy_ic_t * ic);

static void
xaux_proxy_ic_delete(
    xaux_proxy_ic_t *	ic,
    Window		window_target,
    Window		window_self);

static void
xaux_proxy_ic_delete_all();

static xaux_proxy_ic_t *
xaux_proxy_ic_get(
    Display *		display,
    Window		window_target,
    Window		window_self,
    int			server_id,
    IIIMP_card16	im_id,
    IIIMP_card16	ic_id,
    aux_t *		aux);

static void
xaux_proxy_ic_destroy(
    aux_t *	aux);

static void
xaux_proxy_destroy(
    Display *	d);

void
xaux_proxy_aux_start_process(
    aux_t *		aux,
    IIIMCF_event	ev);

static void
xaux_proxy_aux_simple(
    aux_t *		aux,
    int			type,
    int			server_id,
    int			im_id,
    int			ic_id,
    IIIMCF_event	ev);

static void
xaux_proxy_aux_value(
    aux_t *		ic,
    int			type,
    int			server_id,
    int			im_id,
    int			ic_id,
    IIIMCF_event	ev);

#define xaux_proxy_aux_start(aux, sv, im, ic, ev) \
	xaux_proxy_aux_simple((aux), IM_AUX_START, (sv), (im), (ic), (ev))
#define xaux_proxy_aux_done(aux, sv, im, ic, ev) \
	xaux_proxy_aux_simple((aux), IM_AUX_DONE, (sv), (im), (ic), (ev))
#define xaux_proxy_aux_draw(aux, sv, im, ic, ev) \
	xaux_proxy_aux_value((aux), IM_AUX_DRAW, (sv), (im), (ic), (ev))
#define xaux_proxy_aux_setvalues(aux, sv, im, ic, ev) \
	xaux_proxy_aux_value((aux), IM_AUX_SETVALUES, (sv), (im), (ic), (ev))
#define xaux_proxy_aux_getvalues(aux, sv, im, ic, ev) \
	xaux_proxy_aux_value((aux), IM_AUX_GETVALUES, (sv), (im), (ic), (ev))


/*
 * X auxiliary object service method - client
 */

typedef	Bool	(xaux_proxy_service_client_t)(aux_t *		aux,
					      int		message_type,
					      unsigned long	data0,
					      unsigned long	data1,
					      IIIMP_message *	message,
					      unsigned long *	data0_ret,
					      unsigned long *	data1_ret,
					      IIIMP_message **	message_ret);

static xaux_proxy_service_client_t	service_none;
static xaux_proxy_service_client_t	service_aux_setvalue;
static xaux_proxy_service_client_t	service_aux_getvalue;
static xaux_proxy_service_client_t	service_im_id;
static xaux_proxy_service_client_t	service_ic_id;
static xaux_proxy_service_client_t	service_data_set;
static xaux_proxy_service_client_t	service_data_get;
static xaux_proxy_service_client_t	service_display;
static xaux_proxy_service_client_t	service_window;
static xaux_proxy_service_client_t	service_point;
static xaux_proxy_service_client_t	service_point_caret;
static xaux_proxy_service_client_t	service_utf16_mb;
static xaux_proxy_service_client_t	service_mb_utf16;
static xaux_proxy_service_client_t	service_compose;
static xaux_proxy_service_client_t	service_compose_size;
static xaux_proxy_service_client_t	service_decompose;
static xaux_proxy_service_client_t	service_decompose_free;
static xaux_proxy_service_client_t	service_register_X_filter;
static xaux_proxy_service_client_t	service_unregister_X_filter;
static xaux_proxy_service_client_t	service_server;
static xaux_proxy_service_client_t	service_client_window;
static xaux_proxy_service_client_t	service_focus_window;
static xaux_proxy_service_client_t	service_screen_number;
static xaux_proxy_service_client_t	service_point_screen;
static xaux_proxy_service_client_t	service_point_caret_screen;
static xaux_proxy_service_client_t	service_get_conversion_mode;
static xaux_proxy_service_client_t	service_set_conversion_mode;
static xaux_proxy_service_client_t	service_aux_get_from_id;

static xaux_proxy_service_client_t *	xaux_proxy_service[] = {
	service_none, /* none */
	service_aux_setvalue,
	service_none, /* service_im_id */
	service_none, /* service_ic_id */
	service_none, /* service_data_set */
	service_none, /* service_data_get */
	service_none, /* service_display */
	service_window,
	service_point,
	service_point_caret,
	service_none, /* service_utf16_mb */
	service_none, /* service_mb_utf16 */
	service_none, /* service_compose */
	service_none, /* service_compose_size */
	service_none, /* service_decompose */
	service_none, /* service_decompose_free */
	service_none, /* service_register_X_filter */
	service_none, /* service_unregister_X_filter */
	service_none, /* service_server */
	service_client_window,
	service_focus_window,
	service_screen_number,
	service_point_screen,
	service_point_caret_screen,
	service_get_conversion_mode,
	service_set_conversion_mode,
	service_aux_getvalue,
	service_none, /* service_aux_get_from_id */
};


/*
 * static variables
 */

static xaux_proxy_message_queue_t *	xaux_proxy_message_queue_head;
static xaux_proxy_message_queue_t *	xaux_proxy_message_queue_tail;
static xaux_proxy_message_queue_t *	xaux_proxy_message_queue_last;

static xaux_proxy_ic_t *		xaux_proxy_ic;


/*
 * X error handler
 */

static int
xaux_proxy_xerror_handler(
    Display *		d,
    XErrorEvent *	e
)
{
    /* ignore all error event */
    return 0;
}


/*
 * X call wrappers to avoid BadWindow error
 */

static Status
xaux_proxy_XSendEvent(
    Display *	display,
    Window	w,
    Bool	propagate,
    int		event_mask,
    XEvent *	event_send
)
{
    int		(*oeh)(Display *, XErrorEvent *);
    Status	status;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    status = XSendEvent(display, w, propagate, event_mask, event_send);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return status;
}


static int
xaux_proxy_XChangeProperty(
    Display *		display,
    Window		w,
    Atom		property,
    Atom		type,
    int			format,
    int			mode,
    unsigned char *	data,
    int			nelements
)
{
    int	(*oeh)(Display *, XErrorEvent *);
    int	r;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    r = XChangeProperty(display, w, property, type,
			format, mode, data, nelements);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return r;
}


static int
xaux_proxy_XGetWindowProperty(
    Display *		display,
    Window		w,
    Atom		property,
    long		long_offset,
    long		long_length,
    Bool		delete,
    Atom		req_type,
    Atom *		actual_type_return,
    int *		actual_format_return,
    unsigned long *	nitems_return,
    unsigned long *	bytes_after_return,
    unsigned char **	prop_return
)
{
    int	(*oeh)(Display *, XErrorEvent *);
    int	r;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    r = XGetWindowProperty(display, w, property, long_offset, long_length,
			   delete, req_type,
			   actual_type_return, actual_format_return,
			   nitems_return, bytes_after_return, prop_return);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return r;
}


static int
xaux_proxy_XSelectInput(
    Display *	display,
    Window	w,
    long	event_mask
)
{
    int	(*oeh)(Display *, XErrorEvent *);
    int	r;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    r = XSelectInput(display, w, event_mask);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return r;
}

static int
xaux_proxy_XStoreName(
    Display *	display,
    Window	w,
    char *	window_name
)
{
    int	(*oeh)(Display *, XErrorEvent *);
    int	r;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    r = XStoreName(display, w, window_name);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return r;
}


static Status
xaux_proxy_XFetchName(
    Display *	display,
    Window	w,
    char **	window_name_return
)
{
    int		(*oeh)(Display *, XErrorEvent *);
    Status	s;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    s = XFetchName(display, w, window_name_return);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return s;
}


/*
 * miscellaneous utility
 */

static int
xaux_string_length(const IIIMP_card16 * p)
{
    int n;
    n = 0;
    for (; 0 != *p; p++) n++;
    return n;
}


/*
 * xaux_proxy_message_queue_t
 */

static xaux_proxy_message_queue_t *
xaux_proxy_message_queue_new(
    unsigned int	server_id,
    unsigned short	im_id,
    unsigned short	ic_id,
    Display *		display,
    Atom		atom,
    Window		window_target,
    Window		window_self,
    int			message_type,
    unsigned char *	message,
    size_t		message_size
)
{
    xaux_proxy_message_queue_t *	m;

    m = (xaux_proxy_message_queue_t *)
	malloc(sizeof (xaux_proxy_message_queue_t));
    if (NULL == m) return NULL;

    m->server_id = server_id;
    m->im_id = im_id;
    m->ic_id = ic_id;
    m->display = display;
    m->atom = atom;
    m->window_target = window_target;
    m->window_self = window_self;
    m->message_type = message_type;
    m->message = (unsigned char *)malloc(message_size);
    if (NULL == m->message) {
	free(m);
	return NULL;
    }
    memcpy(m->message, message, message_size);
    m->message_size = message_size;
    m->timestamp = 0;
    m->next = NULL;

    return m;
}


static void
xaux_proxy_message_queue_delete(
    xaux_proxy_message_queue_t *	m
)
{
    if (NULL == m) return;
    free(m->message);
    free(m);
}


static void
xaux_proxy_message_queue_delete_all(void)
{
    xaux_proxy_message_queue_t *	m;
    xaux_proxy_message_queue_t **	m_prev;

    for (m_prev = (&xaux_proxy_message_queue_head); NULL != (m = *m_prev); ) {
	*m_prev = m->next;
	xaux_proxy_message_queue_delete(m);
    }

    xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
    xaux_proxy_message_queue_head = NULL;
    xaux_proxy_message_queue_last = NULL;
}


static void
xaux_proxy_message_queue_delete_display(
    Display *	display
)
{
    xaux_proxy_message_queue_t *	m;
    xaux_proxy_message_queue_t **	m_prev;

    for (m_prev = (&xaux_proxy_message_queue_head); NULL != (m = *m_prev); ) {
	if (display == m->display) {
	    *m_prev = m->next;
	    xaux_proxy_message_queue_delete(m);
	} else {
	    m_prev = (&(m->next));
	}
    }

    if ((NULL != xaux_proxy_message_queue_last) &&
	(display == xaux_proxy_message_queue_last->display)) {
	xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
	xaux_proxy_message_queue_last = NULL;
    }
}


static void
xaux_proxy_message_queue_delete_window_self(
    Window	window
)
{
    xaux_proxy_message_queue_t *	m;
    xaux_proxy_message_queue_t **	m_prev;

    for (m_prev = (&xaux_proxy_message_queue_head); NULL != (m = *m_prev); ) {
	if (window == m->window_self) {
	    *m_prev = m->next;
	    xaux_proxy_message_queue_delete(m);
	} else {
	    m_prev = (&(m->next));
	}
    }

    if ((NULL != xaux_proxy_message_queue_last) &&
	(window == xaux_proxy_message_queue_last->window_self)) {
	xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
	xaux_proxy_message_queue_last = NULL;
    }
}


static void
xaux_proxy_message_queue_delete_window_target(
    Window	window
)
{
    xaux_proxy_message_queue_t *	m;
    xaux_proxy_message_queue_t **	m_prev;

    for (m_prev = (&xaux_proxy_message_queue_head); NULL != (m = *m_prev); ) {
	if (window == m->window_target) {
	    *m_prev = m->next;
	    xaux_proxy_message_queue_delete(m);
	} else {
	    m_prev = (&(m->next));
	}
    }

    if ((NULL != xaux_proxy_message_queue_last) &&
	(window == xaux_proxy_message_queue_last->window_target)) {
	xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
	xaux_proxy_message_queue_last = NULL;
    }
}


static void
xaux_proxy_message_queue_unsent_purge(void)
{
    xaux_proxy_message_queue_t *	m;
    xaux_proxy_message_queue_t **	m_prev;

    if (NULL == xaux_proxy_message_queue_last) return;

    for (m_prev = (&xaux_proxy_message_queue_head); NULL != (m = *m_prev); ) {
	if (xaux_proxy_message_queue_last->window_target == m->window_target) {
	    *m_prev = m->next;
	    xaux_proxy_message_queue_delete(m);
	} else {
	    m_prev = (&(m->next));
	}
    }

    xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
    xaux_proxy_message_queue_last = NULL;
}


static void
xaux_proxy_message_queue_append(
    xaux_proxy_message_queue_t *	m
)
{
    if (NULL == xaux_proxy_message_queue_head) {
	xaux_proxy_message_queue_head = m;
    } else {
	xaux_proxy_message_queue_tail->next = m;
    }
    xaux_proxy_message_queue_tail = m;
}


static xaux_proxy_message_queue_t *
xaux_proxy_message_queue_pop(void)
{
    xaux_proxy_message_queue_t *	m;

    if (NULL == xaux_proxy_message_queue_head) return NULL;

    m = xaux_proxy_message_queue_head;
    xaux_proxy_message_queue_head = m->next;
    if (NULL == xaux_proxy_message_queue_head) {
	xaux_proxy_message_queue_tail = NULL;
    }

    return m;
}


/*
 * xaux_proxy_message simple data handling
 */

static void
xaux_proxy_message_data_to_cm(
    Atom			atom,
    Window			window_receiver,
    Window			window_sender,
    int				server_id,
    unsigned short		im_id,
    unsigned short		ic_id,
    int				message_type,
    unsigned long		data0,
    unsigned long		data1,
    XClientMessageEvent *	cm
)
{
    cm->type = ClientMessage;
    cm->message_type = atom;
    cm->format = 32;
    cm->window = window_receiver;
    cm->data.l[0] = window_sender;
    cm->data.l[1] = ((im_id << 16) | ic_id);
    cm->data.l[2] = ((message_type << 16) | server_id);
    cm->data.l[3] = data0;
    cm->data.l[4] = data1;
}


static void
xaux_proxy_message_cm_to_data(
    XClientMessageEvent *	cm,
    unsigned long *		data0,
    unsigned long *		data1
)
{
    if (NULL != data0) *data0 = cm->data.l[3];
    if (NULL != data1) *data1 = cm->data.l[4];
}


/*
 * short message exchange
 */

static Bool
xaux_proxy_message_short_exchange(
    Display *			display,
    Window			window_target,
    Window			window_self,
    Atom			atom,
    int				server_id,
    unsigned short		im_id,
    unsigned short		ic_id,
    int				message_type,
    unsigned long		data0,
    unsigned long		data1,
    unsigned long *		data0_ret,
    unsigned long *		data1_ret
)
{
    XEvent			e;
    XClientMessageEvent		cm;
    struct pollfd		fds;
    struct timeval		timeval;
    struct timeval		timeval_current;
    Bool			b;
    long			ms;
    xaux_proxy_xevent_list_t *	el;
    xaux_proxy_xevent_list_t *	el0;

    xaux_proxy_message_data_to_cm(atom, window_target, window_self,
				  server_id, im_id, ic_id, message_type,
				  data0, data1, &cm);

    xaux_proxy_XSendEvent(display, window_target, True, 0, (XEvent *)(&cm));

    if (NULL == data0_ret) return True;

    fds.fd = ConnectionNumber(display);
    fds.events = POLLIN;
    fds.revents = 0;

    gettimeofday(&timeval, NULL);
    timeval_current = timeval;

    b = False;
    el = NULL;

    for (;;) {
	/* timeout == 10 seconds */
	gettimeofday(&timeval_current, NULL);
	ms = (((timeval_current.tv_sec - timeval.tv_sec) * 1000000) +
	      (timeval_current.tv_usec - timeval.tv_usec));
	if (10000000 <= ms) break;

	b = XCheckTypedWindowEvent(display, window_self, ClientMessage, &e);
	if (True == b) {
	    if (XAUX_PROXY_MESSAGE_TYPE_RECEIPT != e.xclient.data.l[2]) {
		el0 = (xaux_proxy_xevent_list_t *)
		    malloc(sizeof (xaux_proxy_xevent_list_t));
		el0->event = e;
		el0->next = el;
		el = el0;
		continue;
	    }

	    xaux_proxy_message_cm_to_data(&cm, data0_ret, data1_ret);

	    break;
	}
	b = XCheckTypedWindowEvent(display, window_target, DestroyNotify, &e);
	if (True == b) {
	    b = False;
	    break;
	}
	poll(&fds, 1, 1000);
    }

    for (; NULL != el; el = el0) {
	XPutBackEvent(display, &(el->event));
	el0 = el->next;
	free(el);
    }

    return b;
}


/*
 * send long message.
 *
 * add a message to the quque.  if pending message does not
 * exist, send out a message at the top of the queue.
 */

static void
xaux_proxy_message_send(
    Display *		display,
    Window		window_target,
    Window		window_self,
    Atom		atom,
    unsigned int	server_id,
    unsigned short	im_id,
    unsigned short	ic_id,
    int			message_type,
    unsigned char *	message,
    size_t		message_size
)
{
    XClientMessageEvent		cm;
    xaux_proxy_message_queue_t *	m;
    time_t			time_current;

    if (NULL != message) {
	m = xaux_proxy_message_queue_new(server_id, im_id, ic_id, display, atom,
					 window_target, window_self,
					 message_type, message, message_size);
	if (NULL == m) return;

	xaux_proxy_message_queue_append(m);
    }

    time_current = time(NULL);

    if (NULL != xaux_proxy_message_queue_last) {
	if (time_current < (xaux_proxy_message_queue_last->timestamp + 10)) {
	    return;
	}
	xaux_proxy_message_queue_unsent_purge();
    }

    m = xaux_proxy_message_queue_pop();
    if (NULL == m) return;

    xaux_proxy_message_queue_last = m;
    m->timestamp = time_current;

    if (8 < m->message_size) {
	xaux_proxy_XChangeProperty(m->display, m->window_self, m->atom,
				   XA_CARDINAL, 8, PropModeReplace,
				   m->message, m->message_size);
    }

    xaux_proxy_message_short_exchange(m->display, m->window_target, m->window_self,
				      m->atom, m->server_id, m->im_id, m->ic_id,
				      m->message_type, 0, 0, NULL, NULL);

    return;
}


/*
 * send pending message if left on the queue.
 */

static void
xaux_proxy_message_ping(Display * display
)
{
    xaux_proxy_message_send(display, None, None, None, 0, 0, 0,
			    XAUX_PROXY_MESSAGE_TYPE_NONE, NULL, 0);
}


static void
xaux_proxy_message_send_cleanup(void)
{
    xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
    xaux_proxy_message_queue_last = NULL;
}


static void
xaux_proxy_message_process(
    const xaux_proxy_ic_t *	ic,
    IIIMP_message *		m
)
{
    if ((NULL == ic) || (NULL == m)) return;

    switch (m->opcode) {
    case IM_AUX_START:
    case IM_AUX_DONE:
    case IM_AUX_DRAW:
	/* never happen */
	break;
    case IM_AUX_SETVALUES:
    case IM_AUX_GETVALUES:
	break;
    default:
	break;
    }
    return;
}


/*
 * filter to handle message
 */

static Bool
xaux_proxy_window_target_filter(
    Display *	display,
    Window	window,
    XEvent *	event,
    XPointer	client_data
)
{
    if (DestroyNotify != event->type) return False;

    _XUnregisterFilter(display, window,
		       xaux_proxy_window_target_filter, client_data);

    xaux_proxy_ic_delete(NULL, window, None);

    return True;
}


static Bool
xaux_proxy_window_self_filter(
    Display *	display,
    Window	window,
    XEvent *	event,
    XPointer	client_data
)
{
    xaux_proxy_message_queue_t *	m;
    XClientMessageEvent			cm;
    Atom				atom;
    Window				window_sender;
    unsigned int			server_id;
    unsigned short			im_id;
    unsigned short			ic_id;
    int					byte_order;
    int					message_type;
    unsigned char *			message;
    Atom				actual_type_return;
    int					actual_format_return;
    unsigned long			nitem_return;
    unsigned long			bytes_after_return;
    unsigned char *			prop;
    unsigned int			buf[8];
    IIIMP_message *			p;
    size_t				nbyte;
    const uchar_t *			ptr;
    const xaux_proxy_ic_t *		ic;
    int					r;

    if ((ClientMessage != event->type) || (event->xclient.window != window)) {
	return False;
    }

    atom = event->xclient.message_type;
    window_sender = event->xclient.data.l[0];
    im_id = ((0xffff0000 & event->xclient.data.l[1]) >> 16);
    ic_id = (0x0000ffff & event->xclient.data.l[1]);
    byte_order = ((0x80000000 & event->xclient.data.l[2]) >> 31);
    message_type = ((0x7fff0000 & event->xclient.data.l[2]) >> 16);
    server_id = (0x0000ffff & event->xclient.data.l[2]);

    switch (message_type) {
    case XAUX_PROXY_MESSAGE_TYPE_RECEIPT: /* receipt */
	xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
	xaux_proxy_message_queue_last = NULL;
	xaux_proxy_message_ping(display);

	if (NULL != xaux_proxy_message_queue_last) return True;

    /*
     * 2006/02/10
     * xaux proxy partial implementation.
     * Once the feature is fully implemented, xaux_proxy_ic_t
     * data must be destroyed when the real IC is destroyed.
     */
	ic = xaux_proxy_ic_get(display, window_sender, window,
			       server_id, im_id, ic_id, NULL);
	if (NULL == ic) return True;
	xaux_proxy_ic_delete(ic, None, None);

	break;

    case XAUX_PROXY_MESSAGE_TYPE_SERVICE: /* service routines */
	xaux_proxy_message_short_exchange(display, window_sender, window,
					  atom, server_id, im_id, ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_RECEIPT,
					  0, 0, NULL, NULL);
	break;

    case XAUX_PROXY_MESSAGE_TYPE_SIMPLE: /* IM_AUX_START/DONE */
    case XAUX_PROXY_MESSAGE_TYPE_VALUE: /* IM_AUX_DRAW/SETVALUES/GETVALUES */
	ic = xaux_proxy_ic_get(display, window_sender, window,
			       server_id, im_id, ic_id, NULL);
	if (NULL == ic) return True;

	r = xaux_proxy_XGetWindowProperty(display, window_sender, atom,
					  0, 0xffff, True, XA_CARDINAL,
					  &actual_type_return,
					  &actual_format_return,
					  &nitem_return,
					  &bytes_after_return,
					  &prop);

	xaux_proxy_message_short_exchange(display, window_sender, window,
					  atom, server_id, im_id, ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_RECEIPT,
					  0, 0, NULL, NULL);

	if ((Success != r) || (NULL == prop)) return True;

	nbyte = (nitem_return - 4);
	ptr = (prop + 4);
	p = iiimp_message_unpack(ic->data_s, *prop, &nbyte, &ptr);

	XFree(prop);

	if (NULL == p) return True;

	xaux_proxy_message_process(ic, p);

	xaux_proxy_message_ping(display);

	switch (message_type) {
	case XAUX_PROXY_MESSAGE_TYPE_SIMPLE:
	    /* IM_AUX_START/DONE */
	    break;
	case XAUX_PROXY_MESSAGE_TYPE_VALUE:
	    /* IM_AUX_DRAW/SETVALUES/GETVALUES */
	    break;
	}

	iiimp_message_delete(ic->data_s, p);

	break;

    default:
	break;
    }

    return True;
}


/*
 * xaux_proxy_ic_t
 */

static xaux_proxy_ic_t *
xaux_proxy_ic_new(
    Display *		display,
    Window		window_target,
    Window		window_self,
    int			server_id,
    int			im_id,
    int			ic_id,
    aux_t *		aux
)
{
    xaux_proxy_ic_t *	ic;
    char		buf[256];

    ic = (xaux_proxy_ic_t *)malloc(sizeof (xaux_proxy_ic_t));
    if (NULL == ic) return NULL;

    ic->display = display;
    ic->atom = None;
    ic->window_target = window_target;
    ic->window_self = window_self;
    ic->server_id = server_id;
    ic->im_id = im_id;
    ic->ic_id = ic_id;
    ic->aux = aux;
    ic->aux_self = False;
    ic->data_s = NULL;
    ic->next = NULL;

    if (NULL == display) {
	xaux_proxy_ic_delete_alone(ic);
	return NULL;
    }

    ic->atom = XInternAtom(display, "IIIM_XAUX_PROXY", True);
    if (None == ic->atom) {
	xaux_proxy_ic_delete_alone(ic);
	return NULL;
    }

    if (None == window_target) {
	ic->window_target = XGetSelectionOwner(display, ic->atom);
	if (None == ic->window_target) {
	    xaux_proxy_ic_delete_alone(ic);
	    return NULL;
	}
    }

    if (None == window_self) {
	ic->window_self = XCreateSimpleWindow(display, RootWindow(display, 0),
					      0, 0, 1, 1, 0, 0, 0);
	if (None == ic->window_self) {
	    xaux_proxy_ic_delete_alone(ic);
	    return NULL;
	}

	snprintf(buf, sizeof (buf),
		 "%s %d", XAUX_PROXY_WINDOW_NAME_CLIENT, getpid());
	XStoreName(display, ic->window_self, buf);

	_XRegisterFilterByType(display, ic->window_self,
			       ClientMessage, ClientMessage,
			       xaux_proxy_window_self_filter, (XPointer)ic);
    }

    ic->data_s = iiimp_data_s_new();
    if (NULL == ic->data_s) {
	xaux_proxy_ic_delete_alone(ic);
	return NULL;
    }

    return ic;
}


static void
xaux_proxy_ic_delete_alone(
    xaux_proxy_ic_t *	ic
)
{
    if (NULL == ic) return;

    if (None != ic->window_self) {
	_XUnregisterFilter(ic->display, ic->window_self,
			   xaux_proxy_window_self_filter, (XPointer)ic);
	XDestroyWindow(ic->display, ic->window_self);
    }

    iiimp_data_s_delete(ic->data_s);

    free(ic);
}


static void
xaux_proxy_ic_delete(
    xaux_proxy_ic_t *	ic,
    Window		window_target,
    Window		window_self
)
{
    Display *		d;
    xaux_proxy_ic_t *	p;
    xaux_proxy_ic_t **	p_prev;

    if ((NULL == ic) && (None == window_self)) return;

    d = NULL;

    for (p_prev = (&xaux_proxy_ic); NULL != (p = *p_prev); ) {
	if ((ic == p) ||
	    ((None != window_target) && (window_target == p->window_target)) ||
	    ((None != window_self) && (window_self == p->window_self))) {
	    *p_prev = p->next;
	    d = p->display;
	    xaux_proxy_ic_delete_alone(p);
	} else {
	    p_prev = (&(p->next));
	}
    }

    if ((NULL == xaux_proxy_ic) && (NULL != d)) xaux_proxy_destroy(d);
}


static void
xaux_proxy_ic_delete_all(
)
{
    xaux_proxy_ic_t *	p;
    xaux_proxy_ic_t **	p_prev;

    for (p_prev = (&xaux_proxy_ic); NULL != (p = *p_prev); ) {
	*p_prev = p->next;
	xaux_proxy_ic_delete_alone(p);
    }
}


static xaux_proxy_ic_t *
xaux_proxy_ic_get(
    Display *		display,
    Window		window_target,
    Window		window_self,
    int			server_id,
    IIIMP_card16	im_id,
    IIIMP_card16	ic_id,
    aux_t *		aux
)
{
    xaux_proxy_ic_t *	ic;
    xaux_proxy_ic_t **	ic_prev;
    char *		p;
    char *		n;
    Status		s;
    Window		w;
    char		buf[256];

    ic = NULL;
    ic_prev = (&xaux_proxy_ic);
    w = None;

    while (NULL != (ic = *ic_prev)) {
	if ((aux == ic->aux) ||
	    (((im_id == ic->im_id) &&
	      (window_self == ic->window_self) &&
	      (server_id == ic->server_id) &&
	      (ic_id == ic->ic_id)))) {
	    *ic_prev = ic->next;
	    break;
	} else {
	    ic_prev = (&(ic->next));
	}
    }

    if (NULL == ic) {
	ic = xaux_proxy_ic_new(display, window_target, window_self,
			       server_id, im_id, ic_id, aux);
	if (NULL == ic) return NULL;
    }

    if (None == ic->window_target) {
	if (None != window_target) {
	    ic->window_target = window_target;
	} else {
	    if (None == ic->atom) return NULL;
	    ic->window_target = XGetSelectionOwner(display, ic->atom);
	}
	if (None == ic->window_target) {
	    xaux_proxy_ic_delete(ic, None, None);
	    return NULL;
	}
    }

    xaux_proxy_XSelectInput(ic->display, ic->window_target,
			    StructureNotifyMask);

    n = XAUX_PROXY_WINDOW_NAME_SERVER;
    s = xaux_proxy_XFetchName(ic->display, ic->window_target, &p);

    if (0 == s) {
	return NULL;
    } else if ((NULL == p) || (0 != strncmp(p, n, strlen(n)))) {
	xaux_proxy_XSelectInput(ic->display, ic->window_self, NoEventMask);
	if (NULL != p) XFree(p);
	return NULL;
    }
    XFree(p);

    _XRegisterFilterByType(ic->display, ic->window_target,
			   DestroyNotify, DestroyNotify,
			   xaux_proxy_window_target_filter, NULL);

    ic->next = xaux_proxy_ic;
    xaux_proxy_ic = ic;

    return ic;
}


static void
xaux_proxy_ic_destroy(
    aux_t *	aux
)
{
    xaux_proxy_ic_t *	ic;
    Display *		display;
    IIIMP_card16	im_id;
    IIIMP_card16	ic_id;

    if ((NULL == aux) || (NULL == aux->service)) return;

    display = aux->service->display(aux);
    im_id = aux->service->im_id(aux);
    ic_id = aux->service->ic_id(aux);

    ic = xaux_proxy_ic_get(display, None, None, 0, im_id, ic_id, aux);

    xaux_proxy_ic_delete(ic, None, None);

    if (NULL == xaux_proxy_ic) {
	xaux_proxy_destroy(display);
    }
}


static void
xaux_proxy_destroy(
    Display *	d
)
{
    Atom	a;
    Window	w;

    if (NULL == d) return;

    a = XInternAtom(d, "IIIM_XAUX_PROXY", True);
    if (None != a) {
	w = XGetSelectionOwner(d, a);
	if (None != w) {
	    xaux_proxy_XSelectInput(d, w, NoEventMask);
	    _XUnregisterFilter(d, w, xaux_proxy_window_target_filter, NULL);
	}
    }
}


void
iiimxcf_xaux_proxy_aux_start(
    aux_t *		aux,
    IIIMCF_event	ev
)
{
    xaux_proxy_ic_t *	ic;
    int			server_id;
    int			im_id;
    int			ic_id;

    if ((NULL == aux) || (NULL == aux->service)) return;

    server_id = 0;
    im_id = aux->service->im_id(aux);
    ic_id = aux->service->ic_id(aux);

    xaux_proxy_aux_start(aux, server_id, im_id, ic_id, ev);
}


static void
xaux_proxy_aux_simple(
    aux_t *		aux,
    int			type,
    int			server_id,
    int			im_id,
    int			ic_id,
    IIIMCF_event	ev
)
{
    IIIMP_data_s *		data_s;
    IIIMF_status		st;
    const IIIMP_card16 *	aux_name;
    IIIMP_card32		class_index;
    IIIMP_string *		input_method_name;
    size_t			len;
    uchar_t *			buf;
    size_t			buf_size;
    const xaux_proxy_ic_t *	ic;

    ic = xaux_proxy_ic_get(aux->service->display(aux), None, None,
			   server_id, im_id, ic_id, aux);
    if (NULL == ic) return;

    st = iiimcf_get_aux_event_value(ev, &aux_name, &class_index,
				    NULL, NULL, NULL, NULL);
    if (IIIMF_STATUS_SUCCESS != st) return;

    len = xaux_string_length(aux_name);

    input_method_name = iiimp_string_new(ic->data_s, len, aux_name);
    if (NULL == input_method_name) return;

    buf = iiimp_aux_simple_pack(ic->data_s,
				type,
				im_id,
				ic_id,
				class_index,
				input_method_name,
				&buf_size);
    iiimp_string_delete(ic->data_s, input_method_name);
    if (NULL ==buf) return;

    xaux_proxy_message_send(ic->display, ic->window_target, ic->window_self,
			    ic->atom, server_id, im_id, ic_id,
			    XAUX_PROXY_MESSAGE_TYPE_SIMPLE, buf, buf_size);

    free(buf);

    return;
}


static void
xaux_proxy_aux_value(
    aux_t *		ic,
    int			type,
    int			server_id,
    int			im_id,
    int			ic_id,
    IIIMCF_event	ev
)
{
    IIIMP_data_s *		data_s;
    IIIMF_status		st;
    const IIIMP_card16 *	aux_name;
    const IIIMP_card16 *	p;
    IIIMP_card32		class_index;
    int				num_intvals;
    const IIIMP_card32 *	pintvals;
    int				num_strvals;
    const IIIMP_card16 **	pstrs;
    IIIMP_string *		input_method_name;
    size_t			len;
    IIIMP_card32_list *		integer_value;
    IIIMP_string *		string_value;
    IIIMP_string *		string_value_last;
    IIIMP_string *		string_value_cur;
    uchar_t *			buf;
    size_t			buf_size;
    int				i;

    st = iiimcf_get_aux_event_value(ev, &aux_name, &class_index,
				    &num_intvals, &pintvals, &num_strvals, &pstrs);
    if (IIIMF_STATUS_SUCCESS != st) return;

    len = xaux_string_length(aux_name);
    input_method_name = iiimp_string_new(data_s, len, aux_name);
    if (NULL == input_method_name) return;

    integer_value = NULL;
    iiimp_card32_list_new(data_s, num_intvals, pintvals);

    string_value = NULL;
    for (i = 0; i < num_strvals; i++) {
	len = xaux_string_length(*(pstrs + i));
	string_value_cur =
	    iiimp_string_new(data_s, len, *(pstrs + i));
	if (NULL == string_value_last) {
	    string_value_last = string_value_cur;
	} else {
	    string_value_last->next = string_value_cur;
	}
    }

    buf = iiimp_aux_value_pack(data_s,
			       type,
			       im_id,
			       ic_id,
			       class_index,
			       input_method_name,
			       integer_value,
			       string_value,
			       &buf_size);
    iiimp_string_delete(data_s, input_method_name);
    iiimp_card32_list_delete(data_s, integer_value);
    iiimp_string_list_delete(data_s, string_value);
    if (NULL == buf) return;

    free(buf);
}


/*
 * aux service function - client
 */

static Bool
service_none(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_aux_setvalue(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_aux_getvalue(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_window(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_point(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_point_caret(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_client_window(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_focus_window(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_screen_number(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_point_screen(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_point_caret_screen(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_get_conversion_mode(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


static Bool
service_set_conversion_mode(
    aux_t *		aux,
    int			message_type,
    unsigned long	data0,
    unsigned long	data1,
    IIIMP_message *	message,
    unsigned long *	data0_ret,
    unsigned long *	data1_ret,
    IIIMP_message **	message_ret
)
{
    return True;
}


/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
