#include <vsg/io/VSG.h>
#include <vsg/io/mem_stream.h>
static auto brick_phong_frag = []() {
static const char str[] = 
R"(#vsga 1.1.2
Root id=1 vsg::ShaderStage
{
  userObjects 0
  mask 18446744073709551615
  stage 16
  entryPointName "main"
  module id=2 vsg::ShaderModule
  {
    userObjects 0
    hints id=0
    source "#version 450
#extension GL_ARB_separate_shader_objects : enable
#pragma import_defines (VSG_POINT_SPRITE, VSG_DIFFUSE_MAP, VSG_GREYSCALE_DIFFUSE_MAP, VSG_EMISSIVE_MAP, VSG_LIGHTMAP_MAP, VSG_NORMAL_MAP, VSG_SPECULAR_MAP, VSG_TWO_SIDED_LIGHTING, SHADOWMAP_DEBUG)

#define VIEW_DESCRIPTOR_SET 0
#define MATERIAL_DESCRIPTOR_SET 1

#ifdef VSG_DIFFUSE_MAP
layout(set = MATERIAL_DESCRIPTOR_SET, binding = 0) uniform sampler2D diffuseMap;
#endif

#ifdef VSG_NORMAL_MAP
layout(set = MATERIAL_DESCRIPTOR_SET, binding = 2) uniform sampler2D normalMap;
#endif

#ifdef VSG_LIGHTMAP_MAP
layout(set = MATERIAL_DESCRIPTOR_SET, binding = 3) uniform sampler2D aoMap;
#endif

#ifdef VSG_EMISSIVE_MAP
layout(set = MATERIAL_DESCRIPTOR_SET, binding = 4) uniform sampler2D emissiveMap;
#endif

#ifdef VSG_SPECULAR_MAP
layout(set = MATERIAL_DESCRIPTOR_SET, binding = 5) uniform sampler2D specularMap;
#endif

layout(set = MATERIAL_DESCRIPTOR_SET, binding = 10) uniform MaterialData
{
    vec4 ambientColor;
    vec4 diffuseColor;
    vec4 specularColor;
    vec4 emissiveColor;
    float shininess;
    float alphaMask;
    float alphaMaskCutoff;
} material;

layout(set = VIEW_DESCRIPTOR_SET, binding = 0) uniform LightData
{
    vec4 values[2048];
} lightData;


layout(set = VIEW_DESCRIPTOR_SET, binding = 2) uniform sampler2DArrayShadow shadowMaps;

layout(location = 0) in vec3 eyePos;
layout(location = 1) in vec3 normalDir;
layout(location = 2) in vec4 vertexColor;
#ifndef VSG_POINT_SPRITE
layout(location = 3) in vec2 texCoord0;
#endif
layout(location = 5) in vec3 viewDir;

layout(location = 0) out vec4 outColor;

// Find the normal for this fragment, pulling either from a predefined normal map
// or from the interpolated mesh normal and tangent attributes.
vec3 getNormal()
{
    vec3 result;
#ifdef VSG_NORMAL_MAP
    // Perturb normal, see http://www.thetenthplanet.de/archives/1180
    vec3 tangentNormal = texture(normalMap, texCoord0).xyz * 2.0 - 1.0;

    //tangentNormal *= vec3(2,2,1);

    vec3 q1 = dFdx(eyePos);
    vec3 q2 = dFdy(eyePos);
    vec2 st1 = dFdx(texCoord0);
    vec2 st2 = dFdy(texCoord0);

    vec3 N = normalize(normalDir);
    vec3 T = normalize(q1 * st2.t - q2 * st1.t);
    vec3 B = -normalize(cross(N, T));
    mat3 TBN = mat3(T, B, N);

    result = normalize(TBN * tangentNormal);
#else
    result = normalize(normalDir);
#endif
#ifdef VSG_TWO_SIDED_LIGHTING
    if (!gl_FrontFacing)
        result = -result;
#endif
    return result;
}

vec3 computeLighting(vec3 ambientColor, vec3 diffuseColor, vec3 specularColor, vec3 emissiveColor, float shininess, float ambientOcclusion, vec3 ld, vec3 nd, vec3 vd)
{
    vec3 color = vec3(0.0);
    color.rgb += ambientColor;

    float diff = max(dot(ld, nd), 0.0);
    color.rgb += diffuseColor * diff;

    if (diff > 0.0)
    {
        vec3 halfDir = normalize(ld + vd);
        color.rgb += specularColor * pow(max(dot(halfDir, nd), 0.0), shininess);
    }

    vec3 result = color + emissiveColor;
    result *= ambientOcclusion;

    return result;
}


void main()
{
    float brightnessCutoff = 0.001;

#ifdef VSG_POINT_SPRITE
    vec2 texCoord0 = gl_PointCoord.xy;
#endif

    vec4 diffuseColor = vertexColor * material.diffuseColor;
#ifdef VSG_DIFFUSE_MAP
    #ifdef VSG_GREYSCALE_DIFFUSE_MAP
        float v = texture(diffuseMap, texCoord0.st).s;
        diffuseColor *= vec4(v, v, v, 1.0);
    #else
        diffuseColor *= texture(diffuseMap, texCoord0.st);
    #endif
#endif

    vec4 ambientColor = diffuseColor * material.ambientColor * material.ambientColor.a;
    vec4 specularColor = material.specularColor;
    vec4 emissiveColor = material.emissiveColor;
    float shininess = material.shininess;
    float ambientOcclusion = 1.0;

    if (material.alphaMask == 1.0f)
    {
        if (diffuseColor.a < material.alphaMaskCutoff)
            discard;
    }

#ifdef VSG_EMISSIVE_MAP
    emissiveColor *= texture(emissiveMap, texCoord0.st);
#endif

#ifdef VSG_LIGHTMAP_MAP
    ambientOcclusion *= texture(aoMap, texCoord0.st).r;
#endif

#ifdef VSG_SPECULAR_MAP
    specularColor *= texture(specularMap, texCoord0.st);
#endif

    vec3 nd = getNormal();
    vec3 vd = normalize(viewDir);

    vec3 color = vec3(0.0, 0.0, 0.0);

    vec4 lightNums = lightData.values[0];
    int numAmbientLights = int(lightNums[0]);
    int numDirectionalLights = int(lightNums[1]);
    int numPointLights = int(lightNums[2]);
    int numSpotLights = int(lightNums[3]);
    int index = 1;

    if (numAmbientLights>0)
    {
        // ambient lights
        for(int i = 0; i<numAmbientLights; ++i)
        {
            vec4 lightColor = lightData.values[index++];
            color += (ambientColor.rgb * lightColor.rgb) * (lightColor.a);
        }
    }

    // index used to step through the shadowMaps array
    int shadowMapIndex = 0;

    if (numDirectionalLights>0)
    {
        // directional lights
        for(int i = 0; i<numDirectionalLights; ++i)
        {
            vec4 lightColor = lightData.values[index++];
            vec3 direction = -lightData.values[index++].xyz;
            vec4 shadowMapSettings = lightData.values[index++];

            float brightness = lightColor.a;

            // check shadow maps if required
            bool matched = false;
            while ((shadowMapSettings.r > 0.0 && brightness > brightnessCutoff) && !matched)
            {
                mat4 sm_matrix = mat4(lightData.values[index++],
                                      lightData.values[index++],
                                      lightData.values[index++],
                                      lightData.values[index++]);

                vec4 sm_tc = (sm_matrix) * vec4(eyePos, 1.0);

                if (sm_tc.x >= 0.0 && sm_tc.x <= 1.0 && sm_tc.y >= 0.0 && sm_tc.y <= 1.0 && sm_tc.z >= 0.0 /* && sm_tc.z <= 1.0*/)
                {
                    matched = true;

                    float coverage = texture(shadowMaps, vec4(sm_tc.st, shadowMapIndex, sm_tc.z)).r;
                    brightness *= (1.0-coverage);

#ifdef SHADOWMAP_DEBUG
                    if (shadowMapIndex==0) color = vec3(1.0, 0.0, 0.0);
                    else if (shadowMapIndex==1) color = vec3(0.0, 1.0, 0.0);
                    else if (shadowMapIndex==2) color = vec3(0.0, 0.0, 1.0);
                    else if (shadowMapIndex==3) color = vec3(1.0, 1.0, 0.0);
                    else if (shadowMapIndex==4) color = vec3(0.0, 1.0, 1.0);
                    else color = vec3(1.0, 1.0, 1.0);
#endif
                }

                ++shadowMapIndex;
                shadowMapSettings.r -= 1.0;
            }

            if (shadowMapSettings.r > 0.0)
            {
                // skip lightData and shadowMap entries for shadow maps that we haven't visited for this light
                // so subsequent light pointions are correct.
                index += 4 * int(shadowMapSettings.r);
                shadowMapIndex += int(shadowMapSettings.r);
            }

            // if light is too dim/shadowed to effect the rendering skip it
            if (brightness <= brightnessCutoff ) continue;

            float unclamped_LdotN = dot(direction, nd);

            float diff = max(unclamped_LdotN, 0.0);
            color.rgb += (diffuseColor.rgb * lightColor.rgb) * (diff * brightness);

            if (shininess > 0.0 && diff > 0.0)
            {
                vec3 halfDir = normalize(direction + vd);
                color.rgb += specularColor.rgb * (pow(max(dot(halfDir, nd), 0.0), shininess) * brightness);
            }
        }
    }

    if (numPointLights>0)
    {
        // point light
        for(int i = 0; i<numPointLights; ++i)
        {
            vec4 lightColor = lightData.values[index++];
            vec3 position = lightData.values[index++].xyz;

            float brightness = lightColor.a;

            // if light is too dim/shadowed to effect the rendering skip it
            if (brightness <= brightnessCutoff ) continue;

            vec3 delta = position - eyePos;
            float distance2 = delta.x * delta.x + delta.y * delta.y + delta.z * delta.z;
            vec3 direction = delta / sqrt(distance2);
            float scale = brightness / distance2;

            float unclamped_LdotN = dot(direction, nd);

            float diff = scale * max(unclamped_LdotN, 0.0);

            color.rgb += (diffuseColor.rgb * lightColor.rgb) * diff;
            if (shininess > 0.0 && diff > 0.0)
            {
                vec3 halfDir = normalize(direction + vd);
                color.rgb += specularColor.rgb * (pow(max(dot(halfDir, nd), 0.0), shininess) * scale);
            }
        }
    }

    if (numSpotLights>0)
    {
        // spot light
        for(int i = 0; i<numSpotLights; ++i)
        {
            vec4 lightColor = lightData.values[index++];
            vec4 position_cosInnerAngle = lightData.values[index++];
            vec4 lightDirection_cosOuterAngle = lightData.values[index++];

            float brightness = lightColor.a;

            // if light is too dim/shadowed to effect the rendering skip it
            if (brightness <= brightnessCutoff ) continue;

            vec3 delta = position_cosInnerAngle.xyz - eyePos;
            float distance2 = delta.x * delta.x + delta.y * delta.y + delta.z * delta.z;
            vec3 direction = delta / sqrt(distance2);

            float dot_lightdirection = dot(lightDirection_cosOuterAngle.xyz, -direction);
            float scale = (brightness  * smoothstep(lightDirection_cosOuterAngle.w, position_cosInnerAngle.w, dot_lightdirection)) / distance2;

            float unclamped_LdotN = dot(direction, nd);

            float diff = scale * max(unclamped_LdotN, 0.0);
            color.rgb += (diffuseColor.rgb * lightColor.rgb) * diff;
            if (shininess > 0.0 && diff > 0.0)
            {
                vec3 halfDir = normalize(direction + vd);
                color.rgb += specularColor.rgb * (pow(max(dot(halfDir, nd), 0.0), shininess) * scale);
            }
        }
    }


    outColor.rgb = (color * ambientOcclusion) + emissiveColor.rgb;
    outColor.a = diffuseColor.a;
}
"
    code 3794
     119734787 65536 524299 632 0 131089 1 393227 1 1280527431 1685353262 808793134
     0 196622 0 1 720911 4 4 1852399981 0 14 27 83
     242 613 631 196624 4 7 196611 2 450 589828 1096764487 1935622738
     1918988389 1600484449 1684105331 1868526181 1667590754 29556 262149 4 1852399981 0 327685 9
     1316250983 1634562671 10348 262149 12 1970496882 29804 327685 14 1836216174 1766091873 114
     458757 21 1734963810 1701737576 1967354739 1717989236 0 393221 25 1717987684 1130722165 1919904879
     0 327685 27 1953654134 1866692709 7499628 393221 29 1702125901 1818323314 1635017028 0
     458758 29 0 1768058209 1131703909 1919904879 0 458758 29 1 1717987684 1130722165
     1919904879 0 458758 29 2 1667592307 1918987381 1869377347 114 458758 29 3
     1936289125 1702259059 1869377347 114 393222 29 4 1852401779 1936027241 115 393222 29
     5 1752198241 1935756641 107 458758 29 6 1752198241 1935756641 1953842027 6710895 327685
     31 1702125933 1818323314 0 393221 38 1768058209 1131703909 1919904879 0 393221 50
     1667592307 1918987381 1869377347 114 393221 54 1936289125 1702259059 1869377347 114 327685 58
     1852401779 1936027241 115 458757 62 1768058209 1333030501 1970037603 1852795251 0 196613 80
     25710 196613 82 25718 262149 83 2003134838 7498052 262149 86 1869377379 114
     327685 89 1751607660 1836404340 115 327685 92 1751607628 1952531572 97 327686 92
     0 1970037110 29541 327685 94 1751607660 1952531572 97 458757 98 1097692526 1701405293
     1766618222 1937008743 0 524293 103 1148024174 1667592809 1852795252 1766616161 1937008743 0 393221
     108 1349350766 1953393007 1751607628 29556 393221 113 1399682414 1282699120 1952999273 115 262149
     117 1701080681 120 196613 122 105 327685 131 1751607660 1819231092 29295 393221
     148 1684105331 1632466799 1684949360 30821 196613 153 105 327685 162 1751607660 1819231092
     29295 327685 167 1701996900 1869182051 110 458757 174 1684105331 1632466799 1952797552 1735289204
     115 327685 179 1734963810 1701737576 29555 262149 183 1668571501 6579560 327685 202
     1834970483 1769108577 120 262149 240 1952410995 99 262149 242 1348827493 29551 327685
     279 1702260579 1701273970 0 327685 283 1684105331 1632466799 29552 393221 329 1818455669
     1701866849 1682726756 5141615 262149 333 1717987684 0 262149 354 1718378856 7498052 196613
     378 105 327685 387 1751607660 1819231092 29295 327685 392 1769172848 1852795252 0
     327685 398 1734963810 1701737576 29555 262149 407 1953260900 97 327685 411 1953720676
     1701015137 50 327685 429 1701996900 1869182051 110 262149 435 1818321779 101 393221
     439 1818455669 1701866849 1682726756 5141615 262149 443 1717987684 0 262149 464 1718378856
     7498052 196613 488 105 327685 497 1751607660 1819231092 29295 524293 502 1769172848
     1852795252 1936679775 1701736009 1735278962 25964 655365 507 1751607660 1919501428 1769235301 1667198575 1968141167
     1098016116 1701603182 0 327685 512 1734963810 1701737576 29555 262149 521 1953260900 97
     327685 526 1953720676 1701015137 50 327685 544 1701996900 1869182051 110 458757 550
     1601466212 1751607660 1919509620 1769235301 28271 262149 556 1818321779 101 393221 567 1818455669
     1701866849 1682726756 5141615 262149 571 1717987684 0 262149 592 1718378856 7498052 327685
     613 1131705711 1919904879 0 327685 631 1131963764 1685221231 48 262215 14 30
     1 262215 27 30 2 327752 29 0 35 0 327752 29
     1 35 16 327752 29 2 35 32 327752 29 3 35
     48 327752 29 4 35 64 327752 29 5 35 68 327752
     29 6 35 72 196679 29 2 262215 31 34 1 262215
     31 33 10 262215 83 30 5 262215 91 6 16 327752
     92 0 35 0 196679 92 2 262215 94 34 0 262215
     94 33 0 262215 242 30 0 262215 283 34 0 262215
     283 33 2 262215 613 30 0 262215 631 30 3 131091
     2 196641 3 2 196630 6 32 262167 7 6 3 196641
     8 7 262176 11 7 7 262176 13 1 7 262203 13
     14 1 262176 20 7 6 262187 6 22 981668463 262167 23
     6 4 262176 24 7 23 262176 26 1 23 262203 26
     27 1 589854 29 23 23 23 23 6 6 6 262176
     30 2 29 262203 30 31 2 262165 32 32 1 262187
     32 33 1 262176 34 2 23 262187 32 40 0 262165
     44 32 0 262187 44 45 3 262176 46 2 6 262187
     32 51 2 262187 32 55 3 262187 32 59 4 262187
     6 63 1065353216 262187 32 64 5 131092 67 262187 32 73
     6 262203 13 83 1 262187 6 87 0 393260 7 88
     87 87 87 262187 44 90 2048 262172 91 23 90 196638
     92 91 262176 93 2 92 262203 93 94 2 262176 97
     7 32 262187 44 99 0 262187 44 104 1 262187 44
     109 2 262176 182 7 67 196650 67 184 262168 200 23
     4 262176 201 7 200 262203 13 242 1 196649 67 278
     589849 280 6 1 1 1 0 1 0 196635 281 280
     262176 282 0 281 262203 282 283 0 262167 285 6 2
     262176 612 3 23 262203 612 613 3 262176 620 3 6
     262176 630 1 285 262203 630 631 1 327734 2 4 0
     3 131320 5 262203 20 21 7 262203 24 25 7 262203
     24 38 7 262203 24 50 7 262203 24 54 7 262203
     20 58 7 262203 20 62 7 262203 11 80 7 262203
     11 82 7 262203 11 86 7 262203 24 89 7 262203
     97 98 7 262203 97 103 7 262203 97 108 7 262203
     97 113 7 262203 97 117 7 262203 97 122 7 262203
     24 131 7 262203 97 148 7 262203 97 153 7 262203
     24 162 7 262203 11 167 7 262203 24 174 7 262203
     20 179 7 262203 182 183 7 262203 201 202 7 262203
     24 240 7 262203 20 279 7 262203 20 329 7 262203
     20 333 7 262203 11 354 7 262203 97 378 7 262203
     24 387 7 262203 11 392 7 262203 20 398 7 262203
     11 407 7 262203 20 411 7 262203 11 429 7 262203
     20 435 7 262203 20 439 7 262203 20 443 7 262203
     11 464 7 262203 97 488 7 262203 24 497 7 262203
     24 502 7 262203 24 507 7 262203 20 512 7 262203
     11 521 7 262203 20 526 7 262203 11 544 7 262203
     20 550 7 262203 20 556 7 262203 20 567 7 262203
     20 571 7 262203 11 592 7 196670 21 22 262205 23
     28 27 327745 34 35 31 33 262205 23 36 35 327813
     23 37 28 36 196670 25 37 262205 23 39 25 327745
     34 41 31 40 262205 23 42 41 327813 23 43 39
     42 393281 46 47 31 40 45 262205 6 48 47 327822
)"
R"(     23 49 43 48 196670 38 49 327745 34 52 31 51
     262205 23 53 52 196670 50 53 327745 34 56 31 55
     262205 23 57 56 196670 54 57 327745 46 60 31 59
     262205 6 61 60 196670 58 61 196670 62 63 327745 46
     65 31 64 262205 6 66 65 327860 67 68 66 63
     196855 70 0 262394 68 69 70 131320 69 327745 20 71
     25 45 262205 6 72 71 327745 46 74 31 73 262205
     6 75 74 327864 67 76 72 75 196855 78 0 262394
     76 77 78 131320 77 65788 131320 78 131321 70 131320 70
     262201 7 81 9 196670 80 81 262205 7 84 83 393228
     7 85 1 69 84 196670 82 85 196670 86 88 393281
     34 95 94 40 40 262205 23 96 95 196670 89 96
     327745 20 100 89 99 262205 6 101 100 262254 32 102
     101 196670 98 102 327745 20 105 89 104 262205 6 106
     105 262254 32 107 106 196670 103 107 327745 20 110 89
     109 262205 6 111 110 262254 32 112 111 196670 108 112
     327745 20 114 89 45 262205 6 115 114 262254 32 116
     115 196670 113 116 196670 117 33 262205 32 118 98 327853
     67 119 118 40 196855 121 0 262394 119 120 121 131320
     120 196670 122 40 131321 123 131320 123 262390 125 126 0
     131321 127 131320 127 262205 32 128 122 262205 32 129 98
     327857 67 130 128 129 262394 130 124 125 131320 124 262205
     32 132 117 327808 32 133 132 33 196670 117 133 393281
     34 134 94 40 132 262205 23 135 134 196670 131 135
     262205 23 136 38 524367 7 137 136 136 0 1 2
     262205 23 138 131 524367 7 139 138 138 0 1 2
     327813 7 140 137 139 327745 20 141 131 45 262205 6
     142 141 327822 7 143 140 142 262205 7 144 86 327809
     7 145 144 143 196670 86 145 131321 126 131320 126 262205
     32 146 122 327808 32 147 146 33 196670 122 147 131321
     123 131320 125 131321 121 131320 121 196670 148 40 262205 32
     149 103 327853 67 150 149 40 196855 152 0 262394 150
     151 152 131320 151 196670 153 40 131321 154 131320 154 262390
     156 157 0 131321 158 131320 158 262205 32 159 153 262205
     32 160 103 327857 67 161 159 160 262394 161 155 156
     131320 155 262205 32 163 117 327808 32 164 163 33 196670
     117 164 393281 34 165 94 40 163 262205 23 166 165
     196670 162 166 262205 32 168 117 327808 32 169 168 33
     196670 117 169 393281 34 170 94 40 168 262205 23 171
     170 524367 7 172 171 171 0 1 2 262271 7 173
     172 196670 167 173 262205 32 175 117 327808 32 176 175
     33 196670 117 176 393281 34 177 94 40 175 262205 23
     178 177 196670 174 178 327745 20 180 162 45 262205 6
     181 180 196670 179 181 196670 183 184 131321 185 131320 185
     262390 187 188 0 131321 189 131320 189 327745 20 190 174
     99 262205 6 191 190 327866 67 192 191 87 262205 6
     193 179 262205 6 194 21 327866 67 195 193 194 327847
     67 196 192 195 262205 67 197 183 262312 67 198 197
     327847 67 199 196 198 262394 199 186 187 131320 186 262205
     32 203 117 327808 32 204 203 33 196670 117 204 393281
     34 205 94 40 203 262205 23 206 205 262205 32 207
     117 327808 32 208 207 33 196670 117 208 393281 34 209
     94 40 207 262205 23 210 209 262205 32 211 117 327808
     32 212 211 33 196670 117 212 393281 34 213 94 40
     211 262205 23 214 213 262205 32 215 117 327808 32 216
     215 33 196670 117 216 393281 34 217 94 40 215 262205
     23 218 217 327761 6 219 206 0 327761 6 220 206
     1 327761 6 221 206 2 327761 6 222 206 3 327761
     6 223 210 0 327761 6 224 210 1 327761 6 225
     210 2 327761 6 226 210 3 327761 6 227 214 0
     327761 6 228 214 1 327761 6 229 214 2 327761 6
     230 214 3 327761 6 231 218 0 327761 6 232 218
     1 327761 6 233 218 2 327761 6 234 218 3 458832
     23 235 219 220 221 222 458832 23 236 223 224 225
     226 458832 23 237 227 228 229 230 458832 23 238 231
     232 233 234 458832 200 239 235 236 237 238 196670 202
     239 262205 200 241 202 262205 7 243 242 327761 6 244
     243 0 327761 6 245 243 1 327761 6 246 243 2
     458832 23 247 244 245 246 63 327825 23 248 241 247
     196670 240 248 327745 20 249 240 99 262205 6 250 249
     327870 67 251 250 87 196855 253 0 262394 251 252 253
     131320 252 327745 20 254 240 99 262205 6 255 254 327868
     67 256 255 63 131321 253 131320 253 458997 67 257 251
     186 256 252 196855 259 0 262394 257 258 259 131320 258
     327745 20 260 240 104 262205 6 261 260 327870 67 262
     261 87 131321 259 131320 259 458997 67 263 257 253 262
     258 196855 265 0 262394 263 264 265 131320 264 327745 20
     266 240 104 262205 6 267 266 327868 67 268 267 63
     131321 265 131320 265 458997 67 269 263 259 268 264 196855
     271 0 262394 269 270 271 131320 270 327745 20 272 240
     109 262205 6 273 272 327870 67 274 273 87 131321 271
     131320 271 458997 67 275 269 265 274 270 196855 277 0
     262394 275 276 277 131320 276 196670 183 278 262205 281 284
     283 262205 23 286 240 458831 285 287 286 286 0 1
     262205 32 288 148 262255 6 289 288 327745 20 290 240
     109 262205 6 291 290 327761 6 292 287 0 327761 6
     293 287 1 458832 23 294 292 293 289 291 327761 6
     295 294 3 393305 6 296 284 294 295 196670 279 296
     262205 6 297 279 327811 6 298 63 297 262205 6 299
     179 327813 6 300 299 298 196670 179 300 131321 277 131320
     277 262205 32 301 148 327808 32 302 301 33 196670 148
     302 327745 20 303 174 99 262205 6 304 303 327811 6
     305 304 63 327745 20 306 174 99 196670 306 305 131321
     188 131320 188 131321 185 131320 187 327745 20 307 174 99
     262205 6 308 307 327866 67 309 308 87 196855 311 0
     262394 309 310 311 131320 310 327745 20 312 174 99 262205
     6 313 312 262254 32 314 313 327812 32 315 59 314
     262205 32 316 117 327808 32 317 316 315 196670 117 317
     327745 20 318 174 99 262205 6 319 318 262254 32 320
     319 262205 32 321 148 327808 32 322 321 320 196670 148
     322 131321 311 131320 311 262205 6 323 179 262205 6 324
     21 327868 67 325 323 324 196855 327 0 262394 325 326
     327 131320 326 131321 157 131320 327 262205 7 330 167 262205
     7 331 80 327828 6 332 330 331 196670 329 332 262205
     6 334 329 458764 6 335 1 40 334 87 196670 333
     335 262205 23 336 25 524367 7 337 336 336 0 1
     2 262205 23 338 162 524367 7 339 338 338 0 1
     2 327813 7 340 337 339 262205 6 341 333 262205 6
     342 179 327813 6 343 341 342 327822 7 344 340 343
     262205 7 345 86 327809 7 346 345 344 196670 86 346
     262205 6 347 58 327866 67 348 347 87 262205 6 349
     333 327866 67 350 349 87 327847 67 351 348 350 196855
     353 0 262394 351 352 353 131320 352 262205 7 355 167
     262205 7 356 82 327809 7 357 355 356 393228 7 358
     1 69 357 196670 354 358 262205 23 359 50 524367 7
     360 359 359 0 1 2 262205 7 361 354 262205 7
     362 80 327828 6 363 361 362 458764 6 364 1 40
     363 87 262205 6 365 58 458764 6 366 1 26 364
     365 262205 6 367 179 327813 6 368 366 367 327822 7
     369 360 368 262205 7 370 86 327809 7 371 370 369
     196670 86 371 131321 353 131320 353 131321 157 131320 157 262205
     32 372 153 327808 32 373 372 33 196670 153 373 131321
     154 131320 156 131321 152 131320 152 262205 32 374 108 327853
     67 375 374 40 196855 377 0 262394 375 376 377 131320
     376 196670 378 40 131321 379 131320 379 262390 381 382 0
     131321 383 131320 383 262205 32 384 378 262205 32 385 108
     327857 67 386 384 385 262394 386 380 381 131320 380 262205
     32 388 117 327808 32 389 388 33 196670 117 389 393281
     34 390 94 40 388 262205 23 391 390 196670 387 391
     262205 32 393 117 327808 32 394 393 33 196670 117 394
     393281 34 395 94 40 393 262205 23 396 395 524367 7
     397 396 396 0 1 2 196670 392 397 327745 20 399
     387 45 262205 6 400 399 196670 398 400 262205 6 401
     398 262205 6 402 21 327868 67 403 401 402 196855 405
     0 262394 403 404 405 131320 404 131321 382 131320 405 262205
     7 408 392 262205 7 409 242 327811 7 410 408 409
     196670 407 410 327745 20 412 407 99 262205 6 413 412
     327745 20 414 407 99 262205 6 415 414 327813 6 416
     413 415 327745 20 417 407 104 262205 6 418 417 327745
     20 419 407 104 262205 6 420 419 327813 6 421 418
     420 327809 6 422 416 421 327745 20 423 407 109 262205
     6 424 423 327745 20 425 407 109 262205 6 426 425
     327813 6 427 424 426 327809 6 428 422 427 196670 411
     428 262205 7 430 407 262205 6 431 411 393228 6 432
     1 31 431 393296 7 433 432 432 432 327816 7 434
     430 433 196670 429 434 262205 6 436 398 262205 6 437
     411 327816 6 438 436 437 196670 435 438 262205 7 440
     429 262205 7 441 80 327828 6 442 440 441 196670 439
     442 262205 6 444 435 262205 6 445 439 458764 6 446
     1 40 445 87 327813 6 447 444 446 196670 443 447
     262205 23 448 25 524367 7 449 448 448 0 1 2
     262205 23 450 387 524367 7 451 450 450 0 1 2
     327813 7 452 449 451 262205 6 453 443 327822 7 454
     452 453 262205 7 455 86 327809 7 456 455 454 196670
     86 456 262205 6 457 58 327866 67 458 457 87 262205
     6 459 443 327866 67 460 459 87 327847 67 461 458
     460 196855 463 0 262394 461 462 463 131320 462 262205 7
     465 429 262205 7 466 82 327809 7 467 465 466 393228
     7 468 1 69 467 196670 464 468 262205 23 469 50
     524367 7 470 469 469 0 1 2 262205 7 471 464
     262205 7 472 80 327828 6 473 471 472 458764 6 474
     1 40 473 87 262205 6 475 58 458764 6 476 1
     26 474 475 262205 6 477 435 327813 6 478 476 477
     327822 7 479 470 478 262205 7 480 86 327809 7 481
     480 479 196670 86 481 131321 463 131320 463 131321 382 131320
     382 262205 32 482 378 327808 32 483 482 33 196670 378
     483 131321 379 131320 381 131321 377 131320 377 262205 32 484
     113 327853 67 485 484 40 196855 487 0 262394 485 486
     487 131320 486 196670 488 40 131321 489 131320 489 262390 491
     492 0 131321 493 131320 493 262205 32 494 488 262205 32
     495 113 327857 67 496 494 495 262394 496 490 491 131320
     490 262205 32 498 117 327808 32 499 498 33 196670 117
     499 393281 34 500 94 40 498 262205 23 501 500 196670
     497 501 262205 32 503 117 327808 32 504 503 33 196670
     117 504 393281 34 505 94 40 503 262205 23 506 505
     196670 502 506 262205 32 508 117 327808 32 509 508 33
     196670 117 509 393281 34 510 94 40 508 262205 23 511
     510 196670 507 511 327745 20 513 497 45 262205 6 514
     513 196670 512 514 262205 6 515 512 262205 6 516 21
     327868 67 517 515 516 196855 519 0 262394 517 518 519
     131320 518 131321 492 131320 519 262205 23 522 502 524367 7
     523 522 522 0 1 2 262205 7 524 242 327811 7
     525 523 524 196670 521 525 327745 20 527 521 99 262205
     6 528 527 327745 20 529 521 99 262205 6 530 529
     327813 6 531 528 530 327745 20 532 521 104 262205 6
     533 532 327745 20 534 521 104 262205 6 535 534 327813
     6 536 533 535 327809 6 537 531 536 327745 20 538
     521 109 262205 6 539 538 327745 20 540 521 109 262205
     6 541 540 327813 6 542 539 541 327809 6 543 537
     542 196670 526 543 262205 7 545 521 262205 6 546 526
     393228 6 547 1 31 546 393296 7 548 547 547 547
     327816 7 549 545 548 196670 544 549 262205 23 551 507
     524367 7 552 551 551 0 1 2 262205 7 553 544
     262271 7 554 553 327828 6 555 552 554 196670 550 555
     262205 6 557 512 327745 20 558 507 45 262205 6 559
     558 327745 20 560 502 45 262205 6 561 560 262205 6
     562 550 524300 6 563 1 49 559 561 562 327813 6
     564 557 563 262205 6 565 526 327816 6 566 564 565
     196670 556 566 262205 7 568 544 262205 7 569 80 327828
     6 570 568 569 196670 567 570 262205 6 572 556 262205
     6 573 567 458764 6 574 1 40 573 87 327813 6
     575 572 574 196670 571 575 262205 23 576 25 524367 7
     577 576 576 0 1 2 262205 23 578 497 524367 7
     579 578 578 0 1 2 327813 7 580 577 579 262205
     6 581 571 327822 7 582 580 581 262205 7 583 86
     327809 7 584 583 582 196670 86 584 262205 6 585 58
     327866 67 586 585 87 262205 6 587 571 327866 67 588
     587 87 327847 67 589 586 588 196855 591 0 262394 589
     590 591 131320 590 262205 7 593 544 262205 7 594 82
     327809 7 595 593 594 393228 7 596 1 69 595 196670
     592 596 262205 23 597 50 524367 7 598 597 597 0
     1 2 262205 7 599 592 262205 7 600 80 327828 6
     601 599 600 458764 6 602 1 40 601 87 262205 6
     603 58 458764 6 604 1 26 602 603 262205 6 605
     556 327813 6 606 604 605 327822 7 607 598 606 262205
     7 608 86 327809 7 609 608 607 196670 86 609 131321
     591 131320 591 131321 492 131320 492 262205 32 610 488 327808
     32 611 610 33 196670 488 611 131321 489 131320 491 131321
     487 131320 487 262205 7 614 86 262205 6 615 62 327822
     7 616 614 615 262205 23 617 54 524367 7 618 617
     617 0 1 2 327809 7 619 616 618 327745 620 621
     613 99 327761 6 622 619 0 196670 621 622 327745 620
     623 613 104 327761 6 624 619 1 196670 623 624 327745
     620 625 613 109 327761 6 626 619 2 196670 625 626
     327745 20 627 25 45 262205 6 628 627 327745 620 629
     613 45 196670 629 628 65789 65592 327734 7 9 0 8
     131320 10 262203 11 12 7 262205 7 15 14 393228 7
     16 1 69 15 196670 12 16 262205 7 17 12 131326
     17 65592
  }
  NumSpecializationConstants 0
}
)";
vsg::VSG io;
return io.read_cast<vsg::ShaderStage>(reinterpret_cast<const uint8_t*>(str), sizeof(str));
};
