//! Just as much COFF object file support as is needed to write a resource data segment
//! for GNU Windows targets. Inspired by the `write::Object` code from the `object` crate.
//!
//! Integers are converted from u64 to u32 and added without checking because the manifest
//! data cannot get anywhere close to overflowing unless the supplied application name or
//! number of dependencies was extremely long. If this code was used more generally or if
//! the input was less trustworthy then more checked conversions and checked arithmetic
//! would be needed.

use std::io::{self, Seek, SeekFrom, Write};
use std::time::SystemTime;

#[derive(Debug, Clone, Copy)]
pub enum MachineType {
    I386,
    X86_64,
    Aarch64,
}

impl MachineType {
    pub fn machine(&self) -> u16 {
        match self {
            Self::I386 => 0x014c,
            Self::X86_64 => 0x8664,
            Self::Aarch64 => 0xaa64,
        }
    }

    pub fn relocation_type(&self) -> u16 {
        match self {
            Self::I386 => 7,
            Self::X86_64 => 3,
            Self::Aarch64 => 2,
        }
    }
}

pub struct CoffWriter<W> {
    /// wrapped writer or buffer
    writer: W,
    /// machine type for file header
    machine_type: MachineType,
    /// details for section table
    sections: Vec<Section>,
    /// details for non-section symbols in symbol table
    symbols: Vec<Symbol>,
    /// 0-based section currently being written
    current_section: u16,
}

impl<W> CoffWriter<W> {
    /// Create a new instance wrapping a writer.
    pub fn new(writer: W, machine_type: MachineType, section_count: u16) -> Self {
        let mut sections = Vec::new();
        sections.resize_with(section_count as usize, Section::new);
        Self {
            writer,
            machine_type,
            sections,
            symbols: Vec::new(),
            current_section: 0,
        }
    }

    fn section(&mut self) -> &mut Section {
        &mut self.sections[self.current_section as usize]
    }

    /// Add a symbol to be updated when its section and address are known.
    pub fn add_symbol(&mut self, name: &[u8; 8]) -> u32 {
        let index = self.symbols.len() as u32;
        self.symbols.push(Symbol {
            name: *name,
            value: 0,
            section: 0,
        });
        index
    }

    /// Set a symbol to the current offset in a section.
    pub fn set_symbol(&mut self, symbol_index: u32) {
        let mut symbol = &mut self.symbols[symbol_index as usize];
        symbol.value = self.sections[self.current_section as usize].size_of_raw_data;
        symbol.section = self.current_section + 1;
    }
}

impl<W: Write + Seek> CoffWriter<W> {
    pub fn start_section(&mut self, name: &[u8; 8]) -> io::Result<()> {
        // Add space for file header and section table if this is the first section.
        if self.current_section == 0 && self.sections[0].name == [0; 8] {
            self.writer.write_all(&[0; 20])?;
            for _ in 0..self.sections.len() {
                self.writer.write_all(&[0; 40])?;
            }
        } else {
            self.current_section += 1;
        }

        // Save the section name for later.
        self.section().name = *name;
        Ok(())
    }

    /// Pad the current file position to a multiple of four bytes
    /// and save it as the section's start position.
    fn align_section(&mut self) -> io::Result<()> {
        let position = self.writer.stream_position()? as u32;
        let offset = (position & 3) as u32;
        if offset != 0 {
            let padding = 4 - offset;
            self.writer.write_all(&[0; 4][0..(padding as usize)])?;
            self.section().pointer_to_raw_data = position + padding;
        } else {
            self.section().pointer_to_raw_data = position;
        }
        Ok(())
    }

    /// Add data to a section and return its offset within the section.
    pub fn add_data(&mut self, data: &[u8]) -> io::Result<u32> {
        let start = self.section().size_of_raw_data;
        if start == 0 {
            self.align_section()?;
        }
        self.writer.write_all(data)?;
        self.section().size_of_raw_data = start + data.len() as u32;
        Ok(start)
    }

    /// Write a relocation for a symbol at the end of the section.
    pub fn relocation(&mut self, address: u32, symbol_index: u32) -> io::Result<()> {
        self.section().number_of_relocations += 1;
        self.writer.write_all(&address.to_le_bytes())?;
        self.writer.write_all(&symbol_index.to_le_bytes())?;
        self.writer.write_all(&self.machine_type.relocation_type().to_le_bytes())
    }

    /// Write the object and section headers and write the symbol table.
    pub fn finish(mut self) -> io::Result<W> {
        // Copy values for the header.
        let number_of_sections = self.sections.len() as u16;
        let timestamp = SystemTime::now()
            .duration_since(SystemTime::UNIX_EPOCH)
            .map_or(0, |d| d.as_secs() as u32);
        let pointer_to_symbol_table = self.writer.stream_position()? as u32;
        let number_of_symbols = (2 * self.sections.len() + self.symbols.len()) as u32;

        // Write the normal symbols.
        for symbol in self.symbols {
            self.writer.write_all(&symbol.name)?;
            self.writer.write_all(&symbol.value.to_le_bytes())?;
            self.writer.write_all(&symbol.section.to_le_bytes())?;
            self.writer.write_all(&[0, 0, 6, 0])?;
        }

        // Write the symbols and auxiliary data for sections.
        for (i, section) in self.sections.iter().enumerate() {
            self.writer.write_all(&section.name)?;
            self.writer.write_all(&[0, 0, 0, 0])?;
            self.writer.write_all(&((i + 1) as u16).to_le_bytes())?;
            self.writer.write_all(&[0, 0, 3, 1])?;
            self.writer.write_all(&section.pointer_to_raw_data.to_le_bytes())?;
            self.writer.write_all(&section.number_of_relocations.to_le_bytes())?;
            self.writer.write_all(&[0; 12])?;
        }

        // Write the empty string table.
        self.writer.write_all(&[0; 4])?;

        // Write the object header.
        let end_of_file = self.writer.seek(SeekFrom::Start(0))?;
        self.writer.write_all(&self.machine_type.machine().to_le_bytes())?;
        self.writer.write_all(&number_of_sections.to_le_bytes())?;
        self.writer.write_all(&timestamp.to_le_bytes())?;
        self.writer.write_all(&pointer_to_symbol_table.to_le_bytes())?;
        self.writer.write_all(&number_of_symbols.to_le_bytes())?;
        self.writer.write_all(&[0; 4])?; // optional header size = 0, characteristics = 0

        // Update the section headers.
        for section in self.sections.iter() {
            self.writer.write_all(&section.name)?;
            self.writer.write_all(&[0; 8])?; // virtual size = 0 and virtual address = 0
            self.writer.write_all(&section.size_of_raw_data.to_le_bytes())?;
            self.writer.write_all(&section.pointer_to_raw_data.to_le_bytes())?;
            self.writer.write_all(&section.pointer_to_relocations().to_le_bytes())?;
            self.writer.write_all(&[0; 4])?; // pointer to line numbers
            self.writer.write_all(&section.number_of_relocations.to_le_bytes())?;
            self.writer.write_all(&[0; 2])?; // number of line numbers
            self.writer.write_all(&[0x40, 0, 0x30, 0xc0])?; // characteristics
        }

        // Return the inner writer and dispose of this object.
        self.writer.seek(SeekFrom::Start(end_of_file))?;
        Ok(self.writer)
    }
}

/// Details for writing the section table at the start of the file.
struct Section {
    name: [u8; 8],
    size_of_raw_data: u32,
    pointer_to_raw_data: u32,
    number_of_relocations: u16,
}

impl Section {
    fn new() -> Self {
        Section {
            name: [0; 8],
            size_of_raw_data: 0,
            pointer_to_raw_data: 0,
            number_of_relocations: 0,
        }
    }

    fn pointer_to_relocations(&self) -> u32 {
        if self.number_of_relocations == 0 {
            0
        } else {
            // assume there is some data if there are some relocations
            self.pointer_to_raw_data + self.size_of_raw_data
        }
    }
}

/// Details of a non-section symbol to write at the end of the file.
///
/// Note that `section` is a 1-based index here.
struct Symbol {
    name: [u8; 8],
    value: u32,
    section: u16,
}

/// Returns the bytes for a resource directory table.
///
/// Most of the fields are set to zero, including the timestamp, to aid
/// with making builds reproducible.
///
/// ```c
/// typedef struct {
///     DWORD Characteristics,
///     DWORD TimeDateStamp,
///     WORD MajorVersion,
///     WORD MinorVersion,
///     WORD NumberOfNamedEntries,
///     WORD NumberOfIdEntries
/// } IMAGE_RESOURCE_DIRECTORY;
/// ```
pub fn resource_directory_table(number_of_id_entries: u16) -> [u8; 16] {
    let mut table = [0; 16];
    table[14..16].copy_from_slice(&number_of_id_entries.to_le_bytes());
    table
}

/// Returns the bytes for a resource directory entry for an ID.
///
/// ```c
/// typedef struct {
///     DWORD Name,
///     DWORD OffsetToData
/// } IMAGE_RESOURCE_DIRECTORY_ENTRY;
/// ```
pub fn resource_directory_id_entry(id: u32, offset: u32, subdirectory: bool) -> [u8; 8] {
    let mut entry = [0; 8];
    entry[0..4].copy_from_slice(&id.to_le_bytes());
    let flag: u32 = if subdirectory { 0x80000000 } else { 0 };
    entry[4..8].copy_from_slice(&((offset & 0x7fffffff) | flag).to_le_bytes());
    entry
}

/// Returns the bytes for a resource data entry.
///
/// ```c
/// typedef struct {
///     DWORD OffsetToData,
///     DWORD Size,
///     DWORD CodePage,
///     DWORD Reserved
/// } IMAGE_RESOURCE_DATA_ENTRY;
/// ```
pub fn resource_data_entry(rva: u32, size: u32) -> [u8; 16] {
    let mut entry = [0; 16];
    entry[0..4].copy_from_slice(&rva.to_le_bytes());
    entry[4..8].copy_from_slice(&size.to_le_bytes());
    entry
}
