// =============================================================================
// PROJECT CHRONO - http://projectchrono.org
//
// Copyright (c) 2014 projectchrono.org
// All rights reserved.
//
// Use of this source code is governed by a BSD-style license that can be found
// in the LICENSE file at the top level of the distribution and at
// http://projectchrono.org/license-chrono.txt.
//
// =============================================================================
// Author: Milad Rakhsha
// =============================================================================
// This is a utulity to symbollically compute the inverse of a few small matrices
//===============================================================================

#ifndef CH_SOLVER6X6_H_
#define CH_SOLVER6X6_H_

#include <cuda_runtime.h>
#include "chrono_fsi/ChConfigFSI.h"

namespace chrono {
namespace fsi {

/// @addtogroup fsi_solver
/// @{

/// This function calculates the normalized coordinates of a point inside a bilinear element
/// i.e, P= N1*p1.+N2*p2.+N3*p3.+N4*p4. where Ni's are given based on eta and zeta
/// The solver returns the eta and zeta of point P from the coordinates of p1.,p2.,p3.,p4.
static __host__ __device__ void solver2x2(Real2 p1, Real2 p2, Real2 p3, Real2 p4, Real2 p, Real2& eta, Real2& zeta) {
    eta.x =
        -(-p.x * p1.y + p.x * p2.y - p.x * p3.y + p.x * p4.y + p.y * p1.x - p.y * p2.x + p.y * p3.x - p.y * p4.x -
          p1.x * p4.y + p2.x * p3.y - p3.x * p2.y + p4.x * p1.y +
          sqrt((p.x * p.x) * (p1.y * p1.y) + (p.x * p.x) * (p2.y * p2.y) + (p.x * p.x) * (p3.y * p3.y) +
               (p.x * p.x) * (p4.y * p4.y) + (p.y * p.y) * (p1.x * p1.x) + (p.y * p.y) * (p2.x * p2.x) +
               (p.y * p.y) * (p3.x * p3.x) + (p.y * p.y) * (p4.x * p4.x) + (p1.x * p1.x) * (p3.y * p3.y) +
               (p3.x * p3.x) * (p1.y * p1.y) + (p2.x * p2.x) * (p4.y * p4.y) + (p4.x * p4.x) * (p2.y * p2.y) -
               p.x * p1.x * (p3.y * p3.y) * 2.0 - p.x * p3.x * (p1.y * p1.y) * 2.0 - p.x * p2.x * (p4.y * p4.y) * 2.0 -
               p.x * p4.x * (p2.y * p2.y) * 2.0 - (p.x * p.x) * p1.y * p2.y * 2.0 + (p.x * p.x) * p1.y * p3.y * 2.0 -
               (p.x * p.x) * p1.y * p4.y * 2.0 - (p.x * p.x) * p2.y * p3.y * 2.0 + (p.x * p.x) * p2.y * p4.y * 2.0 -
               (p.x * p.x) * p3.y * p4.y * 2.0 - (p.y * p.y) * p1.x * p2.x * 2.0 + (p.y * p.y) * p1.x * p3.x * 2.0 -
               (p.y * p.y) * p1.x * p4.x * 2.0 - (p.y * p.y) * p2.x * p3.x * 2.0 + (p.y * p.y) * p2.x * p4.x * 2.0 -
               (p.y * p.y) * p3.x * p4.x * 2.0 - p.y * (p1.x * p1.x) * p3.y * 2.0 - p.y * (p3.x * p3.x) * p1.y * 2.0 -
               p.y * (p2.x * p2.x) * p4.y * 2.0 - p.y * (p4.x * p4.x) * p2.y * 2.0 - p1.x * p3.x * p1.y * p3.y * 2.0 -
               p1.x * p2.x * p3.y * p4.y * 2.0 + p1.x * p3.x * p2.y * p4.y * 4.0 - p1.x * p4.x * p2.y * p3.y * 2.0 -
               p2.x * p3.x * p1.y * p4.y * 2.0 + p2.x * p4.x * p1.y * p3.y * 4.0 - p3.x * p4.x * p1.y * p2.y * 2.0 -
               p2.x * p4.x * p2.y * p4.y * 2.0 - p.x * p.y * p1.x * p1.y * 2.0 + p.x * p.y * p1.x * p2.y * 2.0 +
               p.x * p.y * p2.x * p1.y * 2.0 - p.x * p.y * p1.x * p3.y * 2.0 - p.x * p.y * p2.x * p2.y * 2.0 -
               p.x * p.y * p3.x * p1.y * 2.0 + p.x * p.y * p1.x * p4.y * 2.0 + p.x * p.y * p2.x * p3.y * 2.0 +
               p.x * p.y * p3.x * p2.y * 2.0 + p.x * p.y * p4.x * p1.y * 2.0 - p.x * p.y * p2.x * p4.y * 2.0 -
               p.x * p.y * p3.x * p3.y * 2.0 - p.x * p.y * p4.x * p2.y * 2.0 + p.x * p.y * p3.x * p4.y * 2.0 +
               p.x * p.y * p4.x * p3.y * 2.0 - p.x * p.y * p4.x * p4.y * 2.0 + p.x * p1.x * p1.y * p3.y * 2.0 +
               p.x * p1.x * p2.y * p3.y * 2.0 - p.x * p2.x * p1.y * p3.y * 4.0 + p.x * p3.x * p1.y * p2.y * 2.0 -
               p.x * p1.x * p2.y * p4.y * 4.0 + p.x * p2.x * p1.y * p4.y * 2.0 + p.x * p3.x * p1.y * p3.y * 2.0 +
               p.x * p4.x * p1.y * p2.y * 2.0 + p.x * p1.x * p3.y * p4.y * 2.0 + p.x * p2.x * p2.y * p4.y * 2.0 +
               p.x * p3.x * p1.y * p4.y * 2.0 - p.x * p4.x * p1.y * p3.y * 4.0 + p.x * p2.x * p3.y * p4.y * 2.0 -
               p.x * p3.x * p2.y * p4.y * 4.0 + p.x * p4.x * p2.y * p3.y * 2.0 + p.x * p4.x * p2.y * p4.y * 2.0 +
               p.y * p1.x * p3.x * p1.y * 2.0 + p.y * p1.x * p2.x * p3.y * 2.0 - p.y * p1.x * p3.x * p2.y * 4.0 +
               p.y * p2.x * p3.x * p1.y * 2.0 + p.y * p1.x * p2.x * p4.y * 2.0 + p.y * p1.x * p3.x * p3.y * 2.0 +
               p.y * p1.x * p4.x * p2.y * 2.0 - p.y * p2.x * p4.x * p1.y * 4.0 - p.y * p1.x * p3.x * p4.y * 4.0 +
               p.y * p1.x * p4.x * p3.y * 2.0 + p.y * p2.x * p4.x * p2.y * 2.0 + p.y * p3.x * p4.x * p1.y * 2.0 +
               p.y * p2.x * p3.x * p4.y * 2.0 - p.y * p2.x * p4.x * p3.y * 4.0 + p.y * p3.x * p4.x * p2.y * 2.0 +
               p.y * p2.x * p4.x * p4.y * 2.0)) /
        (p3.x * (p1.y - p2.y) - p4.x * (p1.y - p2.y) - p1.x * (p3.y - p4.y) + p2.x * (p3.y - p4.y));
    eta.y =
        (p.x * p1.y - p.x * p2.y + p.x * p3.y - p.x * p4.y - p.y * p1.x + p.y * p2.x - p.y * p3.x + p.y * p4.x +
         p1.x * p4.y - p2.x * p3.y + p3.x * p2.y - p4.x * p1.y +
         sqrt((p.x * p.x) * (p1.y * p1.y) + (p.x * p.x) * (p2.y * p2.y) + (p.x * p.x) * (p3.y * p3.y) +
              (p.x * p.x) * (p4.y * p4.y) + (p.y * p.y) * (p1.x * p1.x) + (p.y * p.y) * (p2.x * p2.x) +
              (p.y * p.y) * (p3.x * p3.x) + (p.y * p.y) * (p4.x * p4.x) + (p1.x * p1.x) * (p3.y * p3.y) +
              (p3.x * p3.x) * (p1.y * p1.y) + (p2.x * p2.x) * (p4.y * p4.y) + (p4.x * p4.x) * (p2.y * p2.y) -
              p.x * p1.x * (p3.y * p3.y) * 2.0 - p.x * p3.x * (p1.y * p1.y) * 2.0 - p.x * p2.x * (p4.y * p4.y) * 2.0 -
              p.x * p4.x * (p2.y * p2.y) * 2.0 - (p.x * p.x) * p1.y * p2.y * 2.0 + (p.x * p.x) * p1.y * p3.y * 2.0 -
              (p.x * p.x) * p1.y * p4.y * 2.0 - (p.x * p.x) * p2.y * p3.y * 2.0 + (p.x * p.x) * p2.y * p4.y * 2.0 -
              (p.x * p.x) * p3.y * p4.y * 2.0 - (p.y * p.y) * p1.x * p2.x * 2.0 + (p.y * p.y) * p1.x * p3.x * 2.0 -
              (p.y * p.y) * p1.x * p4.x * 2.0 - (p.y * p.y) * p2.x * p3.x * 2.0 + (p.y * p.y) * p2.x * p4.x * 2.0 -
              (p.y * p.y) * p3.x * p4.x * 2.0 - p.y * (p1.x * p1.x) * p3.y * 2.0 - p.y * (p3.x * p3.x) * p1.y * 2.0 -
              p.y * (p2.x * p2.x) * p4.y * 2.0 - p.y * (p4.x * p4.x) * p2.y * 2.0 - p1.x * p3.x * p1.y * p3.y * 2.0 -
              p1.x * p2.x * p3.y * p4.y * 2.0 + p1.x * p3.x * p2.y * p4.y * 4.0 - p1.x * p4.x * p2.y * p3.y * 2.0 -
              p2.x * p3.x * p1.y * p4.y * 2.0 + p2.x * p4.x * p1.y * p3.y * 4.0 - p3.x * p4.x * p1.y * p2.y * 2.0 -
              p2.x * p4.x * p2.y * p4.y * 2.0 - p.x * p.y * p1.x * p1.y * 2.0 + p.x * p.y * p1.x * p2.y * 2.0 +
              p.x * p.y * p2.x * p1.y * 2.0 - p.x * p.y * p1.x * p3.y * 2.0 - p.x * p.y * p2.x * p2.y * 2.0 -
              p.x * p.y * p3.x * p1.y * 2.0 + p.x * p.y * p1.x * p4.y * 2.0 + p.x * p.y * p2.x * p3.y * 2.0 +
              p.x * p.y * p3.x * p2.y * 2.0 + p.x * p.y * p4.x * p1.y * 2.0 - p.x * p.y * p2.x * p4.y * 2.0 -
              p.x * p.y * p3.x * p3.y * 2.0 - p.x * p.y * p4.x * p2.y * 2.0 + p.x * p.y * p3.x * p4.y * 2.0 +
              p.x * p.y * p4.x * p3.y * 2.0 - p.x * p.y * p4.x * p4.y * 2.0 + p.x * p1.x * p1.y * p3.y * 2.0 +
              p.x * p1.x * p2.y * p3.y * 2.0 - p.x * p2.x * p1.y * p3.y * 4.0 + p.x * p3.x * p1.y * p2.y * 2.0 -
              p.x * p1.x * p2.y * p4.y * 4.0 + p.x * p2.x * p1.y * p4.y * 2.0 + p.x * p3.x * p1.y * p3.y * 2.0 +
              p.x * p4.x * p1.y * p2.y * 2.0 + p.x * p1.x * p3.y * p4.y * 2.0 + p.x * p2.x * p2.y * p4.y * 2.0 +
              p.x * p3.x * p1.y * p4.y * 2.0 - p.x * p4.x * p1.y * p3.y * 4.0 + p.x * p2.x * p3.y * p4.y * 2.0 -
              p.x * p3.x * p2.y * p4.y * 4.0 + p.x * p4.x * p2.y * p3.y * 2.0 + p.x * p4.x * p2.y * p4.y * 2.0 +
              p.y * p1.x * p3.x * p1.y * 2.0 + p.y * p1.x * p2.x * p3.y * 2.0 - p.y * p1.x * p3.x * p2.y * 4.0 +
              p.y * p2.x * p3.x * p1.y * 2.0 + p.y * p1.x * p2.x * p4.y * 2.0 + p.y * p1.x * p3.x * p3.y * 2.0 +
              p.y * p1.x * p4.x * p2.y * 2.0 - p.y * p2.x * p4.x * p1.y * 4.0 - p.y * p1.x * p3.x * p4.y * 4.0 +
              p.y * p1.x * p4.x * p3.y * 2.0 + p.y * p2.x * p4.x * p2.y * 2.0 + p.y * p3.x * p4.x * p1.y * 2.0 +
              p.y * p2.x * p3.x * p4.y * 2.0 - p.y * p2.x * p4.x * p3.y * 4.0 + p.y * p3.x * p4.x * p2.y * 2.0 +
              p.y * p2.x * p4.x * p4.y * 2.0)) /
        (p3.x * (p1.y - p2.y) - p4.x * (p1.y - p2.y) - p1.x * (p3.y - p4.y) + p2.x * (p3.y - p4.y));

    zeta.x =
        (p.x * p1.y - p.x * p2.y + p.x * p3.y - p.x * p4.y - p.y * p1.x + p.y * p2.x - p.y * p3.x + p.y * p4.x +
         p1.x * p2.y - p2.x * p1.y + p3.x * p4.y - p4.x * p3.y +
         sqrt((p.x * p.x) * (p1.y * p1.y) + (p.x * p.x) * (p2.y * p2.y) + (p.x * p.x) * (p3.y * p3.y) +
              (p.x * p.x) * (p4.y * p4.y) + (p.y * p.y) * (p1.x * p1.x) + (p.y * p.y) * (p2.x * p2.x) +
              (p.y * p.y) * (p3.x * p3.x) + (p.y * p.y) * (p4.x * p4.x) + (p1.x * p1.x) * (p3.y * p3.y) +
              (p3.x * p3.x) * (p1.y * p1.y) + (p2.x * p2.x) * (p4.y * p4.y) + (p4.x * p4.x) * (p2.y * p2.y) -
              p.x * p1.x * (p3.y * p3.y) * 2.0 - p.x * p3.x * (p1.y * p1.y) * 2.0 - p.x * p2.x * (p4.y * p4.y) * 2.0 -
              p.x * p4.x * (p2.y * p2.y) * 2.0 - (p.x * p.x) * p1.y * p2.y * 2.0 + (p.x * p.x) * p1.y * p3.y * 2.0 -
              (p.x * p.x) * p1.y * p4.y * 2.0 - (p.x * p.x) * p2.y * p3.y * 2.0 + (p.x * p.x) * p2.y * p4.y * 2.0 -
              (p.x * p.x) * p3.y * p4.y * 2.0 - (p.y * p.y) * p1.x * p2.x * 2.0 + (p.y * p.y) * p1.x * p3.x * 2.0 -
              (p.y * p.y) * p1.x * p4.x * 2.0 - (p.y * p.y) * p2.x * p3.x * 2.0 + (p.y * p.y) * p2.x * p4.x * 2.0 -
              (p.y * p.y) * p3.x * p4.x * 2.0 - p.y * (p1.x * p1.x) * p3.y * 2.0 - p.y * (p3.x * p3.x) * p1.y * 2.0 -
              p.y * (p2.x * p2.x) * p4.y * 2.0 - p.y * (p4.x * p4.x) * p2.y * 2.0 - p1.x * p3.x * p1.y * p3.y * 2.0 -
              p1.x * p2.x * p3.y * p4.y * 2.0 + p1.x * p3.x * p2.y * p4.y * 4.0 - p1.x * p4.x * p2.y * p3.y * 2.0 -
              p2.x * p3.x * p1.y * p4.y * 2.0 + p2.x * p4.x * p1.y * p3.y * 4.0 - p3.x * p4.x * p1.y * p2.y * 2.0 -
              p2.x * p4.x * p2.y * p4.y * 2.0 - p.x * p.y * p1.x * p1.y * 2.0 + p.x * p.y * p1.x * p2.y * 2.0 +
              p.x * p.y * p2.x * p1.y * 2.0 - p.x * p.y * p1.x * p3.y * 2.0 - p.x * p.y * p2.x * p2.y * 2.0 -
              p.x * p.y * p3.x * p1.y * 2.0 + p.x * p.y * p1.x * p4.y * 2.0 + p.x * p.y * p2.x * p3.y * 2.0 +
              p.x * p.y * p3.x * p2.y * 2.0 + p.x * p.y * p4.x * p1.y * 2.0 - p.x * p.y * p2.x * p4.y * 2.0 -
              p.x * p.y * p3.x * p3.y * 2.0 - p.x * p.y * p4.x * p2.y * 2.0 + p.x * p.y * p3.x * p4.y * 2.0 +
              p.x * p.y * p4.x * p3.y * 2.0 - p.x * p.y * p4.x * p4.y * 2.0 + p.x * p1.x * p1.y * p3.y * 2.0 +
              p.x * p1.x * p2.y * p3.y * 2.0 - p.x * p2.x * p1.y * p3.y * 4.0 + p.x * p3.x * p1.y * p2.y * 2.0 -
              p.x * p1.x * p2.y * p4.y * 4.0 + p.x * p2.x * p1.y * p4.y * 2.0 + p.x * p3.x * p1.y * p3.y * 2.0 +
              p.x * p4.x * p1.y * p2.y * 2.0 + p.x * p1.x * p3.y * p4.y * 2.0 + p.x * p2.x * p2.y * p4.y * 2.0 +
              p.x * p3.x * p1.y * p4.y * 2.0 - p.x * p4.x * p1.y * p3.y * 4.0 + p.x * p2.x * p3.y * p4.y * 2.0 -
              p.x * p3.x * p2.y * p4.y * 4.0 + p.x * p4.x * p2.y * p3.y * 2.0 + p.x * p4.x * p2.y * p4.y * 2.0 +
              p.y * p1.x * p3.x * p1.y * 2.0 + p.y * p1.x * p2.x * p3.y * 2.0 - p.y * p1.x * p3.x * p2.y * 4.0 +
              p.y * p2.x * p3.x * p1.y * 2.0 + p.y * p1.x * p2.x * p4.y * 2.0 + p.y * p1.x * p3.x * p3.y * 2.0 +
              p.y * p1.x * p4.x * p2.y * 2.0 - p.y * p2.x * p4.x * p1.y * 4.0 - p.y * p1.x * p3.x * p4.y * 4.0 +
              p.y * p1.x * p4.x * p3.y * 2.0 + p.y * p2.x * p4.x * p2.y * 2.0 + p.y * p3.x * p4.x * p1.y * 2.0 +
              p.y * p2.x * p3.x * p4.y * 2.0 - p.y * p2.x * p4.x * p3.y * 4.0 + p.y * p3.x * p4.x * p2.y * 2.0 +
              p.y * p2.x * p4.x * p4.y * 2.0)) /
        (p1.x * p2.y - p2.x * p1.y - p1.x * p3.y + p3.x * p1.y + p2.x * p4.y - p4.x * p2.y - p3.x * p4.y + p4.x * p3.y);

    zeta.y =
        -(-p.x * p1.y + p.x * p2.y - p.x * p3.y + p.x * p4.y + p.y * p1.x - p.y * p2.x + p.y * p3.x - p.y * p4.x -
          p1.x * p2.y + p2.x * p1.y - p3.x * p4.y + p4.x * p3.y +
          sqrt((p.x * p.x) * (p1.y * p1.y) + (p.x * p.x) * (p2.y * p2.y) + (p.x * p.x) * (p3.y * p3.y) +
               (p.x * p.x) * (p4.y * p4.y) + (p.y * p.y) * (p1.x * p1.x) + (p.y * p.y) * (p2.x * p2.x) +
               (p.y * p.y) * (p3.x * p3.x) + (p.y * p.y) * (p4.x * p4.x) + (p1.x * p1.x) * (p3.y * p3.y) +
               (p3.x * p3.x) * (p1.y * p1.y) + (p2.x * p2.x) * (p4.y * p4.y) + (p4.x * p4.x) * (p2.y * p2.y) -
               p.x * p1.x * (p3.y * p3.y) * 2.0 - p.x * p3.x * (p1.y * p1.y) * 2.0 - p.x * p2.x * (p4.y * p4.y) * 2.0 -
               p.x * p4.x * (p2.y * p2.y) * 2.0 - (p.x * p.x) * p1.y * p2.y * 2.0 + (p.x * p.x) * p1.y * p3.y * 2.0 -
               (p.x * p.x) * p1.y * p4.y * 2.0 - (p.x * p.x) * p2.y * p3.y * 2.0 + (p.x * p.x) * p2.y * p4.y * 2.0 -
               (p.x * p.x) * p3.y * p4.y * 2.0 - (p.y * p.y) * p1.x * p2.x * 2.0 + (p.y * p.y) * p1.x * p3.x * 2.0 -
               (p.y * p.y) * p1.x * p4.x * 2.0 - (p.y * p.y) * p2.x * p3.x * 2.0 + (p.y * p.y) * p2.x * p4.x * 2.0 -
               (p.y * p.y) * p3.x * p4.x * 2.0 - p.y * (p1.x * p1.x) * p3.y * 2.0 - p.y * (p3.x * p3.x) * p1.y * 2.0 -
               p.y * (p2.x * p2.x) * p4.y * 2.0 - p.y * (p4.x * p4.x) * p2.y * 2.0 - p1.x * p3.x * p1.y * p3.y * 2.0 -
               p1.x * p2.x * p3.y * p4.y * 2.0 + p1.x * p3.x * p2.y * p4.y * 4.0 - p1.x * p4.x * p2.y * p3.y * 2.0 -
               p2.x * p3.x * p1.y * p4.y * 2.0 + p2.x * p4.x * p1.y * p3.y * 4.0 - p3.x * p4.x * p1.y * p2.y * 2.0 -
               p2.x * p4.x * p2.y * p4.y * 2.0 - p.x * p.y * p1.x * p1.y * 2.0 + p.x * p.y * p1.x * p2.y * 2.0 +
               p.x * p.y * p2.x * p1.y * 2.0 - p.x * p.y * p1.x * p3.y * 2.0 - p.x * p.y * p2.x * p2.y * 2.0 -
               p.x * p.y * p3.x * p1.y * 2.0 + p.x * p.y * p1.x * p4.y * 2.0 + p.x * p.y * p2.x * p3.y * 2.0 +
               p.x * p.y * p3.x * p2.y * 2.0 + p.x * p.y * p4.x * p1.y * 2.0 - p.x * p.y * p2.x * p4.y * 2.0 -
               p.x * p.y * p3.x * p3.y * 2.0 - p.x * p.y * p4.x * p2.y * 2.0 + p.x * p.y * p3.x * p4.y * 2.0 +
               p.x * p.y * p4.x * p3.y * 2.0 - p.x * p.y * p4.x * p4.y * 2.0 + p.x * p1.x * p1.y * p3.y * 2.0 +
               p.x * p1.x * p2.y * p3.y * 2.0 - p.x * p2.x * p1.y * p3.y * 4.0 + p.x * p3.x * p1.y * p2.y * 2.0 -
               p.x * p1.x * p2.y * p4.y * 4.0 + p.x * p2.x * p1.y * p4.y * 2.0 + p.x * p3.x * p1.y * p3.y * 2.0 +
               p.x * p4.x * p1.y * p2.y * 2.0 + p.x * p1.x * p3.y * p4.y * 2.0 + p.x * p2.x * p2.y * p4.y * 2.0 +
               p.x * p3.x * p1.y * p4.y * 2.0 - p.x * p4.x * p1.y * p3.y * 4.0 + p.x * p2.x * p3.y * p4.y * 2.0 -
               p.x * p3.x * p2.y * p4.y * 4.0 + p.x * p4.x * p2.y * p3.y * 2.0 + p.x * p4.x * p2.y * p4.y * 2.0 +
               p.y * p1.x * p3.x * p1.y * 2.0 + p.y * p1.x * p2.x * p3.y * 2.0 - p.y * p1.x * p3.x * p2.y * 4.0 +
               p.y * p2.x * p3.x * p1.y * 2.0 + p.y * p1.x * p2.x * p4.y * 2.0 + p.y * p1.x * p3.x * p3.y * 2.0 +
               p.y * p1.x * p4.x * p2.y * 2.0 - p.y * p2.x * p4.x * p1.y * 4.0 - p.y * p1.x * p3.x * p4.y * 4.0 +
               p.y * p1.x * p4.x * p3.y * 2.0 + p.y * p2.x * p4.x * p2.y * 2.0 + p.y * p3.x * p4.x * p1.y * 2.0 +
               p.y * p2.x * p3.x * p4.y * 2.0 - p.y * p2.x * p4.x * p3.y * 4.0 + p.y * p3.x * p4.x * p2.y * 2.0 +
               p.y * p2.x * p4.x * p4.y * 2.0)) /
        (p1.x * p2.y - p2.x * p1.y - p1.x * p3.y + p3.x * p1.y + p2.x * p4.y - p4.x * p2.y - p3.x * p4.y + p4.x * p3.y);
}

/// Note that this is function calculates L=[L11 L12 L13 L22 L23 L33]' in B*L=[-1 0 0 -1 0 -1]' for a given B 6x6 matirx
/// This is is the core function for higher-order accurate schemes
static __host__ __device__ void inv6xdelta_mn(Real* B, Real* L) {
    Real DET = B[0] * B[7] * B[14] * B[21] * B[28] * B[35] - B[0] * B[7] * B[14] * B[21] * B[29] * B[34] -
               B[0] * B[7] * B[14] * B[22] * B[27] * B[35] + B[0] * B[7] * B[14] * B[22] * B[29] * B[33] +
               B[0] * B[7] * B[14] * B[23] * B[27] * B[34] - B[0] * B[7] * B[14] * B[23] * B[28] * B[33] -
               B[0] * B[7] * B[15] * B[20] * B[28] * B[35] + B[0] * B[7] * B[15] * B[20] * B[29] * B[34] +
               B[0] * B[7] * B[15] * B[22] * B[26] * B[35] - B[0] * B[7] * B[15] * B[22] * B[29] * B[32] -
               B[0] * B[7] * B[15] * B[23] * B[26] * B[34] + B[0] * B[7] * B[15] * B[23] * B[28] * B[32] +
               B[0] * B[7] * B[16] * B[20] * B[27] * B[35] - B[0] * B[7] * B[16] * B[20] * B[29] * B[33] -
               B[0] * B[7] * B[16] * B[21] * B[26] * B[35] + B[0] * B[7] * B[16] * B[21] * B[29] * B[32] +
               B[0] * B[7] * B[16] * B[23] * B[26] * B[33] - B[0] * B[7] * B[16] * B[23] * B[27] * B[32] -
               B[0] * B[7] * B[17] * B[20] * B[27] * B[34] + B[0] * B[7] * B[17] * B[20] * B[28] * B[33] +
               B[0] * B[7] * B[17] * B[21] * B[26] * B[34] - B[0] * B[7] * B[17] * B[21] * B[28] * B[32] -
               B[0] * B[7] * B[17] * B[22] * B[26] * B[33] + B[0] * B[7] * B[17] * B[22] * B[27] * B[32] -
               B[0] * B[8] * B[13] * B[21] * B[28] * B[35] + B[0] * B[8] * B[13] * B[21] * B[29] * B[34] +
               B[0] * B[8] * B[13] * B[22] * B[27] * B[35] - B[0] * B[8] * B[13] * B[22] * B[29] * B[33] -
               B[0] * B[8] * B[13] * B[23] * B[27] * B[34] + B[0] * B[8] * B[13] * B[23] * B[28] * B[33] +
               B[0] * B[8] * B[15] * B[19] * B[28] * B[35] - B[0] * B[8] * B[15] * B[19] * B[29] * B[34] -
               B[0] * B[8] * B[15] * B[22] * B[25] * B[35] + B[0] * B[8] * B[15] * B[22] * B[29] * B[31] +
               B[0] * B[8] * B[15] * B[23] * B[25] * B[34] - B[0] * B[8] * B[15] * B[23] * B[28] * B[31] -
               B[0] * B[8] * B[16] * B[19] * B[27] * B[35] + B[0] * B[8] * B[16] * B[19] * B[29] * B[33] +
               B[0] * B[8] * B[16] * B[21] * B[25] * B[35] - B[0] * B[8] * B[16] * B[21] * B[29] * B[31] -
               B[0] * B[8] * B[16] * B[23] * B[25] * B[33] + B[0] * B[8] * B[16] * B[23] * B[27] * B[31] +
               B[0] * B[8] * B[17] * B[19] * B[27] * B[34] - B[0] * B[8] * B[17] * B[19] * B[28] * B[33] -
               B[0] * B[8] * B[17] * B[21] * B[25] * B[34] + B[0] * B[8] * B[17] * B[21] * B[28] * B[31] +
               B[0] * B[8] * B[17] * B[22] * B[25] * B[33] - B[0] * B[8] * B[17] * B[22] * B[27] * B[31] +
               B[0] * B[9] * B[13] * B[20] * B[28] * B[35] - B[0] * B[9] * B[13] * B[20] * B[29] * B[34] -
               B[0] * B[9] * B[13] * B[22] * B[26] * B[35] + B[0] * B[9] * B[13] * B[22] * B[29] * B[32] +
               B[0] * B[9] * B[13] * B[23] * B[26] * B[34] - B[0] * B[9] * B[13] * B[23] * B[28] * B[32] -
               B[0] * B[9] * B[14] * B[19] * B[28] * B[35] + B[0] * B[9] * B[14] * B[19] * B[29] * B[34] +
               B[0] * B[9] * B[14] * B[22] * B[25] * B[35] - B[0] * B[9] * B[14] * B[22] * B[29] * B[31] -
               B[0] * B[9] * B[14] * B[23] * B[25] * B[34] + B[0] * B[9] * B[14] * B[23] * B[28] * B[31] +
               B[0] * B[9] * B[16] * B[19] * B[26] * B[35] - B[0] * B[9] * B[16] * B[19] * B[29] * B[32] -
               B[0] * B[9] * B[16] * B[20] * B[25] * B[35] + B[0] * B[9] * B[16] * B[20] * B[29] * B[31] +
               B[0] * B[9] * B[16] * B[23] * B[25] * B[32] - B[0] * B[9] * B[16] * B[23] * B[26] * B[31] -
               B[0] * B[9] * B[17] * B[19] * B[26] * B[34] + B[0] * B[9] * B[17] * B[19] * B[28] * B[32] +
               B[0] * B[9] * B[17] * B[20] * B[25] * B[34] - B[0] * B[9] * B[17] * B[20] * B[28] * B[31] -
               B[0] * B[9] * B[17] * B[22] * B[25] * B[32] + B[0] * B[9] * B[17] * B[22] * B[26] * B[31] -
               B[0] * B[10] * B[13] * B[20] * B[27] * B[35] + B[0] * B[10] * B[13] * B[20] * B[29] * B[33] +
               B[0] * B[10] * B[13] * B[21] * B[26] * B[35] - B[0] * B[10] * B[13] * B[21] * B[29] * B[32] -
               B[0] * B[10] * B[13] * B[23] * B[26] * B[33] + B[0] * B[10] * B[13] * B[23] * B[27] * B[32] +
               B[0] * B[10] * B[14] * B[19] * B[27] * B[35] - B[0] * B[10] * B[14] * B[19] * B[29] * B[33] -
               B[0] * B[10] * B[14] * B[21] * B[25] * B[35] + B[0] * B[10] * B[14] * B[21] * B[29] * B[31] +
               B[0] * B[10] * B[14] * B[23] * B[25] * B[33] - B[0] * B[10] * B[14] * B[23] * B[27] * B[31] -
               B[0] * B[10] * B[15] * B[19] * B[26] * B[35] + B[0] * B[10] * B[15] * B[19] * B[29] * B[32] +
               B[0] * B[10] * B[15] * B[20] * B[25] * B[35] - B[0] * B[10] * B[15] * B[20] * B[29] * B[31] -
               B[0] * B[10] * B[15] * B[23] * B[25] * B[32] + B[0] * B[10] * B[15] * B[23] * B[26] * B[31] +
               B[0] * B[10] * B[17] * B[19] * B[26] * B[33] - B[0] * B[10] * B[17] * B[19] * B[27] * B[32] -
               B[0] * B[10] * B[17] * B[20] * B[25] * B[33] + B[0] * B[10] * B[17] * B[20] * B[27] * B[31] +
               B[0] * B[10] * B[17] * B[21] * B[25] * B[32] - B[0] * B[10] * B[17] * B[21] * B[26] * B[31] +
               B[0] * B[11] * B[13] * B[20] * B[27] * B[34] - B[0] * B[11] * B[13] * B[20] * B[28] * B[33] -
               B[0] * B[11] * B[13] * B[21] * B[26] * B[34] + B[0] * B[11] * B[13] * B[21] * B[28] * B[32] +
               B[0] * B[11] * B[13] * B[22] * B[26] * B[33] - B[0] * B[11] * B[13] * B[22] * B[27] * B[32] -
               B[0] * B[11] * B[14] * B[19] * B[27] * B[34] + B[0] * B[11] * B[14] * B[19] * B[28] * B[33] +
               B[0] * B[11] * B[14] * B[21] * B[25] * B[34] - B[0] * B[11] * B[14] * B[21] * B[28] * B[31] -
               B[0] * B[11] * B[14] * B[22] * B[25] * B[33] + B[0] * B[11] * B[14] * B[22] * B[27] * B[31] +
               B[0] * B[11] * B[15] * B[19] * B[26] * B[34] - B[0] * B[11] * B[15] * B[19] * B[28] * B[32] -
               B[0] * B[11] * B[15] * B[20] * B[25] * B[34] + B[0] * B[11] * B[15] * B[20] * B[28] * B[31] +
               B[0] * B[11] * B[15] * B[22] * B[25] * B[32] - B[0] * B[11] * B[15] * B[22] * B[26] * B[31] -
               B[0] * B[11] * B[16] * B[19] * B[26] * B[33] + B[0] * B[11] * B[16] * B[19] * B[27] * B[32] +
               B[0] * B[11] * B[16] * B[20] * B[25] * B[33] - B[0] * B[11] * B[16] * B[20] * B[27] * B[31] -
               B[0] * B[11] * B[16] * B[21] * B[25] * B[32] + B[0] * B[11] * B[16] * B[21] * B[26] * B[31] -
               B[1] * B[6] * B[14] * B[21] * B[28] * B[35] + B[1] * B[6] * B[14] * B[21] * B[29] * B[34] +
               B[1] * B[6] * B[14] * B[22] * B[27] * B[35] - B[1] * B[6] * B[14] * B[22] * B[29] * B[33] -
               B[1] * B[6] * B[14] * B[23] * B[27] * B[34] + B[1] * B[6] * B[14] * B[23] * B[28] * B[33] +
               B[1] * B[6] * B[15] * B[20] * B[28] * B[35] - B[1] * B[6] * B[15] * B[20] * B[29] * B[34] -
               B[1] * B[6] * B[15] * B[22] * B[26] * B[35] + B[1] * B[6] * B[15] * B[22] * B[29] * B[32] +
               B[1] * B[6] * B[15] * B[23] * B[26] * B[34] - B[1] * B[6] * B[15] * B[23] * B[28] * B[32] -
               B[1] * B[6] * B[16] * B[20] * B[27] * B[35] + B[1] * B[6] * B[16] * B[20] * B[29] * B[33] +
               B[1] * B[6] * B[16] * B[21] * B[26] * B[35] - B[1] * B[6] * B[16] * B[21] * B[29] * B[32] -
               B[1] * B[6] * B[16] * B[23] * B[26] * B[33] + B[1] * B[6] * B[16] * B[23] * B[27] * B[32] +
               B[1] * B[6] * B[17] * B[20] * B[27] * B[34] - B[1] * B[6] * B[17] * B[20] * B[28] * B[33] -
               B[1] * B[6] * B[17] * B[21] * B[26] * B[34] + B[1] * B[6] * B[17] * B[21] * B[28] * B[32] +
               B[1] * B[6] * B[17] * B[22] * B[26] * B[33] - B[1] * B[6] * B[17] * B[22] * B[27] * B[32] +
               B[1] * B[8] * B[12] * B[21] * B[28] * B[35] - B[1] * B[8] * B[12] * B[21] * B[29] * B[34] -
               B[1] * B[8] * B[12] * B[22] * B[27] * B[35] + B[1] * B[8] * B[12] * B[22] * B[29] * B[33] +
               B[1] * B[8] * B[12] * B[23] * B[27] * B[34] - B[1] * B[8] * B[12] * B[23] * B[28] * B[33] -
               B[1] * B[8] * B[15] * B[18] * B[28] * B[35] + B[1] * B[8] * B[15] * B[18] * B[29] * B[34] +
               B[1] * B[8] * B[15] * B[22] * B[24] * B[35] - B[1] * B[8] * B[15] * B[22] * B[29] * B[30] -
               B[1] * B[8] * B[15] * B[23] * B[24] * B[34] + B[1] * B[8] * B[15] * B[23] * B[28] * B[30] +
               B[1] * B[8] * B[16] * B[18] * B[27] * B[35] - B[1] * B[8] * B[16] * B[18] * B[29] * B[33] -
               B[1] * B[8] * B[16] * B[21] * B[24] * B[35] + B[1] * B[8] * B[16] * B[21] * B[29] * B[30] +
               B[1] * B[8] * B[16] * B[23] * B[24] * B[33] - B[1] * B[8] * B[16] * B[23] * B[27] * B[30] -
               B[1] * B[8] * B[17] * B[18] * B[27] * B[34] + B[1] * B[8] * B[17] * B[18] * B[28] * B[33] +
               B[1] * B[8] * B[17] * B[21] * B[24] * B[34] - B[1] * B[8] * B[17] * B[21] * B[28] * B[30] -
               B[1] * B[8] * B[17] * B[22] * B[24] * B[33] + B[1] * B[8] * B[17] * B[22] * B[27] * B[30] -
               B[1] * B[9] * B[12] * B[20] * B[28] * B[35] + B[1] * B[9] * B[12] * B[20] * B[29] * B[34] +
               B[1] * B[9] * B[12] * B[22] * B[26] * B[35] - B[1] * B[9] * B[12] * B[22] * B[29] * B[32] -
               B[1] * B[9] * B[12] * B[23] * B[26] * B[34] + B[1] * B[9] * B[12] * B[23] * B[28] * B[32] +
               B[1] * B[9] * B[14] * B[18] * B[28] * B[35] - B[1] * B[9] * B[14] * B[18] * B[29] * B[34] -
               B[1] * B[9] * B[14] * B[22] * B[24] * B[35] + B[1] * B[9] * B[14] * B[22] * B[29] * B[30] +
               B[1] * B[9] * B[14] * B[23] * B[24] * B[34] - B[1] * B[9] * B[14] * B[23] * B[28] * B[30] -
               B[1] * B[9] * B[16] * B[18] * B[26] * B[35] + B[1] * B[9] * B[16] * B[18] * B[29] * B[32] +
               B[1] * B[9] * B[16] * B[20] * B[24] * B[35] - B[1] * B[9] * B[16] * B[20] * B[29] * B[30] -
               B[1] * B[9] * B[16] * B[23] * B[24] * B[32] + B[1] * B[9] * B[16] * B[23] * B[26] * B[30] +
               B[1] * B[9] * B[17] * B[18] * B[26] * B[34] - B[1] * B[9] * B[17] * B[18] * B[28] * B[32] -
               B[1] * B[9] * B[17] * B[20] * B[24] * B[34] + B[1] * B[9] * B[17] * B[20] * B[28] * B[30] +
               B[1] * B[9] * B[17] * B[22] * B[24] * B[32] - B[1] * B[9] * B[17] * B[22] * B[26] * B[30] +
               B[1] * B[10] * B[12] * B[20] * B[27] * B[35] - B[1] * B[10] * B[12] * B[20] * B[29] * B[33] -
               B[1] * B[10] * B[12] * B[21] * B[26] * B[35] + B[1] * B[10] * B[12] * B[21] * B[29] * B[32] +
               B[1] * B[10] * B[12] * B[23] * B[26] * B[33] - B[1] * B[10] * B[12] * B[23] * B[27] * B[32] -
               B[1] * B[10] * B[14] * B[18] * B[27] * B[35] + B[1] * B[10] * B[14] * B[18] * B[29] * B[33] +
               B[1] * B[10] * B[14] * B[21] * B[24] * B[35] - B[1] * B[10] * B[14] * B[21] * B[29] * B[30] -
               B[1] * B[10] * B[14] * B[23] * B[24] * B[33] + B[1] * B[10] * B[14] * B[23] * B[27] * B[30] +
               B[1] * B[10] * B[15] * B[18] * B[26] * B[35] - B[1] * B[10] * B[15] * B[18] * B[29] * B[32] -
               B[1] * B[10] * B[15] * B[20] * B[24] * B[35] + B[1] * B[10] * B[15] * B[20] * B[29] * B[30] +
               B[1] * B[10] * B[15] * B[23] * B[24] * B[32] - B[1] * B[10] * B[15] * B[23] * B[26] * B[30] -
               B[1] * B[10] * B[17] * B[18] * B[26] * B[33] + B[1] * B[10] * B[17] * B[18] * B[27] * B[32] +
               B[1] * B[10] * B[17] * B[20] * B[24] * B[33] - B[1] * B[10] * B[17] * B[20] * B[27] * B[30] -
               B[1] * B[10] * B[17] * B[21] * B[24] * B[32] + B[1] * B[10] * B[17] * B[21] * B[26] * B[30] -
               B[1] * B[11] * B[12] * B[20] * B[27] * B[34] + B[1] * B[11] * B[12] * B[20] * B[28] * B[33] +
               B[1] * B[11] * B[12] * B[21] * B[26] * B[34] - B[1] * B[11] * B[12] * B[21] * B[28] * B[32] -
               B[1] * B[11] * B[12] * B[22] * B[26] * B[33] + B[1] * B[11] * B[12] * B[22] * B[27] * B[32] +
               B[1] * B[11] * B[14] * B[18] * B[27] * B[34] - B[1] * B[11] * B[14] * B[18] * B[28] * B[33] -
               B[1] * B[11] * B[14] * B[21] * B[24] * B[34] + B[1] * B[11] * B[14] * B[21] * B[28] * B[30] +
               B[1] * B[11] * B[14] * B[22] * B[24] * B[33] - B[1] * B[11] * B[14] * B[22] * B[27] * B[30] -
               B[1] * B[11] * B[15] * B[18] * B[26] * B[34] + B[1] * B[11] * B[15] * B[18] * B[28] * B[32] +
               B[1] * B[11] * B[15] * B[20] * B[24] * B[34] - B[1] * B[11] * B[15] * B[20] * B[28] * B[30] -
               B[1] * B[11] * B[15] * B[22] * B[24] * B[32] + B[1] * B[11] * B[15] * B[22] * B[26] * B[30] +
               B[1] * B[11] * B[16] * B[18] * B[26] * B[33] - B[1] * B[11] * B[16] * B[18] * B[27] * B[32] -
               B[1] * B[11] * B[16] * B[20] * B[24] * B[33] + B[1] * B[11] * B[16] * B[20] * B[27] * B[30] +
               B[1] * B[11] * B[16] * B[21] * B[24] * B[32] - B[1] * B[11] * B[16] * B[21] * B[26] * B[30] +
               B[2] * B[6] * B[13] * B[21] * B[28] * B[35] - B[2] * B[6] * B[13] * B[21] * B[29] * B[34] -
               B[2] * B[6] * B[13] * B[22] * B[27] * B[35] + B[2] * B[6] * B[13] * B[22] * B[29] * B[33] +
               B[2] * B[6] * B[13] * B[23] * B[27] * B[34] - B[2] * B[6] * B[13] * B[23] * B[28] * B[33] -
               B[2] * B[6] * B[15] * B[19] * B[28] * B[35] + B[2] * B[6] * B[15] * B[19] * B[29] * B[34] +
               B[2] * B[6] * B[15] * B[22] * B[25] * B[35] - B[2] * B[6] * B[15] * B[22] * B[29] * B[31] -
               B[2] * B[6] * B[15] * B[23] * B[25] * B[34] + B[2] * B[6] * B[15] * B[23] * B[28] * B[31] +
               B[2] * B[6] * B[16] * B[19] * B[27] * B[35] - B[2] * B[6] * B[16] * B[19] * B[29] * B[33] -
               B[2] * B[6] * B[16] * B[21] * B[25] * B[35] + B[2] * B[6] * B[16] * B[21] * B[29] * B[31] +
               B[2] * B[6] * B[16] * B[23] * B[25] * B[33] - B[2] * B[6] * B[16] * B[23] * B[27] * B[31] -
               B[2] * B[6] * B[17] * B[19] * B[27] * B[34] + B[2] * B[6] * B[17] * B[19] * B[28] * B[33] +
               B[2] * B[6] * B[17] * B[21] * B[25] * B[34] - B[2] * B[6] * B[17] * B[21] * B[28] * B[31] -
               B[2] * B[6] * B[17] * B[22] * B[25] * B[33] + B[2] * B[6] * B[17] * B[22] * B[27] * B[31] -
               B[2] * B[7] * B[12] * B[21] * B[28] * B[35] + B[2] * B[7] * B[12] * B[21] * B[29] * B[34] +
               B[2] * B[7] * B[12] * B[22] * B[27] * B[35] - B[2] * B[7] * B[12] * B[22] * B[29] * B[33] -
               B[2] * B[7] * B[12] * B[23] * B[27] * B[34] + B[2] * B[7] * B[12] * B[23] * B[28] * B[33] +
               B[2] * B[7] * B[15] * B[18] * B[28] * B[35] - B[2] * B[7] * B[15] * B[18] * B[29] * B[34] -
               B[2] * B[7] * B[15] * B[22] * B[24] * B[35] + B[2] * B[7] * B[15] * B[22] * B[29] * B[30] +
               B[2] * B[7] * B[15] * B[23] * B[24] * B[34] - B[2] * B[7] * B[15] * B[23] * B[28] * B[30] -
               B[2] * B[7] * B[16] * B[18] * B[27] * B[35] + B[2] * B[7] * B[16] * B[18] * B[29] * B[33] +
               B[2] * B[7] * B[16] * B[21] * B[24] * B[35] - B[2] * B[7] * B[16] * B[21] * B[29] * B[30] -
               B[2] * B[7] * B[16] * B[23] * B[24] * B[33] + B[2] * B[7] * B[16] * B[23] * B[27] * B[30] +
               B[2] * B[7] * B[17] * B[18] * B[27] * B[34] - B[2] * B[7] * B[17] * B[18] * B[28] * B[33] -
               B[2] * B[7] * B[17] * B[21] * B[24] * B[34] + B[2] * B[7] * B[17] * B[21] * B[28] * B[30] +
               B[2] * B[7] * B[17] * B[22] * B[24] * B[33] - B[2] * B[7] * B[17] * B[22] * B[27] * B[30] +
               B[2] * B[9] * B[12] * B[19] * B[28] * B[35] - B[2] * B[9] * B[12] * B[19] * B[29] * B[34] -
               B[2] * B[9] * B[12] * B[22] * B[25] * B[35] + B[2] * B[9] * B[12] * B[22] * B[29] * B[31] +
               B[2] * B[9] * B[12] * B[23] * B[25] * B[34] - B[2] * B[9] * B[12] * B[23] * B[28] * B[31] -
               B[2] * B[9] * B[13] * B[18] * B[28] * B[35] + B[2] * B[9] * B[13] * B[18] * B[29] * B[34] +
               B[2] * B[9] * B[13] * B[22] * B[24] * B[35] - B[2] * B[9] * B[13] * B[22] * B[29] * B[30] -
               B[2] * B[9] * B[13] * B[23] * B[24] * B[34] + B[2] * B[9] * B[13] * B[23] * B[28] * B[30] +
               B[2] * B[9] * B[16] * B[18] * B[25] * B[35] - B[2] * B[9] * B[16] * B[18] * B[29] * B[31] -
               B[2] * B[9] * B[16] * B[19] * B[24] * B[35] + B[2] * B[9] * B[16] * B[19] * B[29] * B[30] +
               B[2] * B[9] * B[16] * B[23] * B[24] * B[31] - B[2] * B[9] * B[16] * B[23] * B[25] * B[30] -
               B[2] * B[9] * B[17] * B[18] * B[25] * B[34] + B[2] * B[9] * B[17] * B[18] * B[28] * B[31] +
               B[2] * B[9] * B[17] * B[19] * B[24] * B[34] - B[2] * B[9] * B[17] * B[19] * B[28] * B[30] -
               B[2] * B[9] * B[17] * B[22] * B[24] * B[31] + B[2] * B[9] * B[17] * B[22] * B[25] * B[30] -
               B[2] * B[10] * B[12] * B[19] * B[27] * B[35] + B[2] * B[10] * B[12] * B[19] * B[29] * B[33] +
               B[2] * B[10] * B[12] * B[21] * B[25] * B[35] - B[2] * B[10] * B[12] * B[21] * B[29] * B[31] -
               B[2] * B[10] * B[12] * B[23] * B[25] * B[33] + B[2] * B[10] * B[12] * B[23] * B[27] * B[31] +
               B[2] * B[10] * B[13] * B[18] * B[27] * B[35] - B[2] * B[10] * B[13] * B[18] * B[29] * B[33] -
               B[2] * B[10] * B[13] * B[21] * B[24] * B[35] + B[2] * B[10] * B[13] * B[21] * B[29] * B[30] +
               B[2] * B[10] * B[13] * B[23] * B[24] * B[33] - B[2] * B[10] * B[13] * B[23] * B[27] * B[30] -
               B[2] * B[10] * B[15] * B[18] * B[25] * B[35] + B[2] * B[10] * B[15] * B[18] * B[29] * B[31] +
               B[2] * B[10] * B[15] * B[19] * B[24] * B[35] - B[2] * B[10] * B[15] * B[19] * B[29] * B[30] -
               B[2] * B[10] * B[15] * B[23] * B[24] * B[31] + B[2] * B[10] * B[15] * B[23] * B[25] * B[30] +
               B[2] * B[10] * B[17] * B[18] * B[25] * B[33] - B[2] * B[10] * B[17] * B[18] * B[27] * B[31] -
               B[2] * B[10] * B[17] * B[19] * B[24] * B[33] + B[2] * B[10] * B[17] * B[19] * B[27] * B[30] +
               B[2] * B[10] * B[17] * B[21] * B[24] * B[31] - B[2] * B[10] * B[17] * B[21] * B[25] * B[30] +
               B[2] * B[11] * B[12] * B[19] * B[27] * B[34] - B[2] * B[11] * B[12] * B[19] * B[28] * B[33] -
               B[2] * B[11] * B[12] * B[21] * B[25] * B[34] + B[2] * B[11] * B[12] * B[21] * B[28] * B[31] +
               B[2] * B[11] * B[12] * B[22] * B[25] * B[33] - B[2] * B[11] * B[12] * B[22] * B[27] * B[31] -
               B[2] * B[11] * B[13] * B[18] * B[27] * B[34] + B[2] * B[11] * B[13] * B[18] * B[28] * B[33] +
               B[2] * B[11] * B[13] * B[21] * B[24] * B[34] - B[2] * B[11] * B[13] * B[21] * B[28] * B[30] -
               B[2] * B[11] * B[13] * B[22] * B[24] * B[33] + B[2] * B[11] * B[13] * B[22] * B[27] * B[30] +
               B[2] * B[11] * B[15] * B[18] * B[25] * B[34] - B[2] * B[11] * B[15] * B[18] * B[28] * B[31] -
               B[2] * B[11] * B[15] * B[19] * B[24] * B[34] + B[2] * B[11] * B[15] * B[19] * B[28] * B[30] +
               B[2] * B[11] * B[15] * B[22] * B[24] * B[31] - B[2] * B[11] * B[15] * B[22] * B[25] * B[30] -
               B[2] * B[11] * B[16] * B[18] * B[25] * B[33] + B[2] * B[11] * B[16] * B[18] * B[27] * B[31] +
               B[2] * B[11] * B[16] * B[19] * B[24] * B[33] - B[2] * B[11] * B[16] * B[19] * B[27] * B[30] -
               B[2] * B[11] * B[16] * B[21] * B[24] * B[31] + B[2] * B[11] * B[16] * B[21] * B[25] * B[30] -
               B[3] * B[6] * B[13] * B[20] * B[28] * B[35] + B[3] * B[6] * B[13] * B[20] * B[29] * B[34] +
               B[3] * B[6] * B[13] * B[22] * B[26] * B[35] - B[3] * B[6] * B[13] * B[22] * B[29] * B[32] -
               B[3] * B[6] * B[13] * B[23] * B[26] * B[34] + B[3] * B[6] * B[13] * B[23] * B[28] * B[32] +
               B[3] * B[6] * B[14] * B[19] * B[28] * B[35] - B[3] * B[6] * B[14] * B[19] * B[29] * B[34] -
               B[3] * B[6] * B[14] * B[22] * B[25] * B[35] + B[3] * B[6] * B[14] * B[22] * B[29] * B[31] +
               B[3] * B[6] * B[14] * B[23] * B[25] * B[34] - B[3] * B[6] * B[14] * B[23] * B[28] * B[31] -
               B[3] * B[6] * B[16] * B[19] * B[26] * B[35] + B[3] * B[6] * B[16] * B[19] * B[29] * B[32] +
               B[3] * B[6] * B[16] * B[20] * B[25] * B[35] - B[3] * B[6] * B[16] * B[20] * B[29] * B[31] -
               B[3] * B[6] * B[16] * B[23] * B[25] * B[32] + B[3] * B[6] * B[16] * B[23] * B[26] * B[31] +
               B[3] * B[6] * B[17] * B[19] * B[26] * B[34] - B[3] * B[6] * B[17] * B[19] * B[28] * B[32] -
               B[3] * B[6] * B[17] * B[20] * B[25] * B[34] + B[3] * B[6] * B[17] * B[20] * B[28] * B[31] +
               B[3] * B[6] * B[17] * B[22] * B[25] * B[32] - B[3] * B[6] * B[17] * B[22] * B[26] * B[31] +
               B[3] * B[7] * B[12] * B[20] * B[28] * B[35] - B[3] * B[7] * B[12] * B[20] * B[29] * B[34] -
               B[3] * B[7] * B[12] * B[22] * B[26] * B[35] + B[3] * B[7] * B[12] * B[22] * B[29] * B[32] +
               B[3] * B[7] * B[12] * B[23] * B[26] * B[34] - B[3] * B[7] * B[12] * B[23] * B[28] * B[32] -
               B[3] * B[7] * B[14] * B[18] * B[28] * B[35] + B[3] * B[7] * B[14] * B[18] * B[29] * B[34] +
               B[3] * B[7] * B[14] * B[22] * B[24] * B[35] - B[3] * B[7] * B[14] * B[22] * B[29] * B[30] -
               B[3] * B[7] * B[14] * B[23] * B[24] * B[34] + B[3] * B[7] * B[14] * B[23] * B[28] * B[30] +
               B[3] * B[7] * B[16] * B[18] * B[26] * B[35] - B[3] * B[7] * B[16] * B[18] * B[29] * B[32] -
               B[3] * B[7] * B[16] * B[20] * B[24] * B[35] + B[3] * B[7] * B[16] * B[20] * B[29] * B[30] +
               B[3] * B[7] * B[16] * B[23] * B[24] * B[32] - B[3] * B[7] * B[16] * B[23] * B[26] * B[30] -
               B[3] * B[7] * B[17] * B[18] * B[26] * B[34] + B[3] * B[7] * B[17] * B[18] * B[28] * B[32] +
               B[3] * B[7] * B[17] * B[20] * B[24] * B[34] - B[3] * B[7] * B[17] * B[20] * B[28] * B[30] -
               B[3] * B[7] * B[17] * B[22] * B[24] * B[32] + B[3] * B[7] * B[17] * B[22] * B[26] * B[30] -
               B[3] * B[8] * B[12] * B[19] * B[28] * B[35] + B[3] * B[8] * B[12] * B[19] * B[29] * B[34] +
               B[3] * B[8] * B[12] * B[22] * B[25] * B[35] - B[3] * B[8] * B[12] * B[22] * B[29] * B[31] -
               B[3] * B[8] * B[12] * B[23] * B[25] * B[34] + B[3] * B[8] * B[12] * B[23] * B[28] * B[31] +
               B[3] * B[8] * B[13] * B[18] * B[28] * B[35] - B[3] * B[8] * B[13] * B[18] * B[29] * B[34] -
               B[3] * B[8] * B[13] * B[22] * B[24] * B[35] + B[3] * B[8] * B[13] * B[22] * B[29] * B[30] +
               B[3] * B[8] * B[13] * B[23] * B[24] * B[34] - B[3] * B[8] * B[13] * B[23] * B[28] * B[30] -
               B[3] * B[8] * B[16] * B[18] * B[25] * B[35] + B[3] * B[8] * B[16] * B[18] * B[29] * B[31] +
               B[3] * B[8] * B[16] * B[19] * B[24] * B[35] - B[3] * B[8] * B[16] * B[19] * B[29] * B[30] -
               B[3] * B[8] * B[16] * B[23] * B[24] * B[31] + B[3] * B[8] * B[16] * B[23] * B[25] * B[30] +
               B[3] * B[8] * B[17] * B[18] * B[25] * B[34] - B[3] * B[8] * B[17] * B[18] * B[28] * B[31] -
               B[3] * B[8] * B[17] * B[19] * B[24] * B[34] + B[3] * B[8] * B[17] * B[19] * B[28] * B[30] +
               B[3] * B[8] * B[17] * B[22] * B[24] * B[31] - B[3] * B[8] * B[17] * B[22] * B[25] * B[30] +
               B[3] * B[10] * B[12] * B[19] * B[26] * B[35] - B[3] * B[10] * B[12] * B[19] * B[29] * B[32] -
               B[3] * B[10] * B[12] * B[20] * B[25] * B[35] + B[3] * B[10] * B[12] * B[20] * B[29] * B[31] +
               B[3] * B[10] * B[12] * B[23] * B[25] * B[32] - B[3] * B[10] * B[12] * B[23] * B[26] * B[31] -
               B[3] * B[10] * B[13] * B[18] * B[26] * B[35] + B[3] * B[10] * B[13] * B[18] * B[29] * B[32] +
               B[3] * B[10] * B[13] * B[20] * B[24] * B[35] - B[3] * B[10] * B[13] * B[20] * B[29] * B[30] -
               B[3] * B[10] * B[13] * B[23] * B[24] * B[32] + B[3] * B[10] * B[13] * B[23] * B[26] * B[30] +
               B[3] * B[10] * B[14] * B[18] * B[25] * B[35] - B[3] * B[10] * B[14] * B[18] * B[29] * B[31] -
               B[3] * B[10] * B[14] * B[19] * B[24] * B[35] + B[3] * B[10] * B[14] * B[19] * B[29] * B[30] +
               B[3] * B[10] * B[14] * B[23] * B[24] * B[31] - B[3] * B[10] * B[14] * B[23] * B[25] * B[30] -
               B[3] * B[10] * B[17] * B[18] * B[25] * B[32] + B[3] * B[10] * B[17] * B[18] * B[26] * B[31] +
               B[3] * B[10] * B[17] * B[19] * B[24] * B[32] - B[3] * B[10] * B[17] * B[19] * B[26] * B[30] -
               B[3] * B[10] * B[17] * B[20] * B[24] * B[31] + B[3] * B[10] * B[17] * B[20] * B[25] * B[30] -
               B[3] * B[11] * B[12] * B[19] * B[26] * B[34] + B[3] * B[11] * B[12] * B[19] * B[28] * B[32] +
               B[3] * B[11] * B[12] * B[20] * B[25] * B[34] - B[3] * B[11] * B[12] * B[20] * B[28] * B[31] -
               B[3] * B[11] * B[12] * B[22] * B[25] * B[32] + B[3] * B[11] * B[12] * B[22] * B[26] * B[31] +
               B[3] * B[11] * B[13] * B[18] * B[26] * B[34] - B[3] * B[11] * B[13] * B[18] * B[28] * B[32] -
               B[3] * B[11] * B[13] * B[20] * B[24] * B[34] + B[3] * B[11] * B[13] * B[20] * B[28] * B[30] +
               B[3] * B[11] * B[13] * B[22] * B[24] * B[32] - B[3] * B[11] * B[13] * B[22] * B[26] * B[30] -
               B[3] * B[11] * B[14] * B[18] * B[25] * B[34] + B[3] * B[11] * B[14] * B[18] * B[28] * B[31] +
               B[3] * B[11] * B[14] * B[19] * B[24] * B[34] - B[3] * B[11] * B[14] * B[19] * B[28] * B[30] -
               B[3] * B[11] * B[14] * B[22] * B[24] * B[31] + B[3] * B[11] * B[14] * B[22] * B[25] * B[30] +
               B[3] * B[11] * B[16] * B[18] * B[25] * B[32] - B[3] * B[11] * B[16] * B[18] * B[26] * B[31] -
               B[3] * B[11] * B[16] * B[19] * B[24] * B[32] + B[3] * B[11] * B[16] * B[19] * B[26] * B[30] +
               B[3] * B[11] * B[16] * B[20] * B[24] * B[31] - B[3] * B[11] * B[16] * B[20] * B[25] * B[30] +
               B[4] * B[6] * B[13] * B[20] * B[27] * B[35] - B[4] * B[6] * B[13] * B[20] * B[29] * B[33] -
               B[4] * B[6] * B[13] * B[21] * B[26] * B[35] + B[4] * B[6] * B[13] * B[21] * B[29] * B[32] +
               B[4] * B[6] * B[13] * B[23] * B[26] * B[33] - B[4] * B[6] * B[13] * B[23] * B[27] * B[32] -
               B[4] * B[6] * B[14] * B[19] * B[27] * B[35] + B[4] * B[6] * B[14] * B[19] * B[29] * B[33] +
               B[4] * B[6] * B[14] * B[21] * B[25] * B[35] - B[4] * B[6] * B[14] * B[21] * B[29] * B[31] -
               B[4] * B[6] * B[14] * B[23] * B[25] * B[33] + B[4] * B[6] * B[14] * B[23] * B[27] * B[31] +
               B[4] * B[6] * B[15] * B[19] * B[26] * B[35] - B[4] * B[6] * B[15] * B[19] * B[29] * B[32] -
               B[4] * B[6] * B[15] * B[20] * B[25] * B[35] + B[4] * B[6] * B[15] * B[20] * B[29] * B[31] +
               B[4] * B[6] * B[15] * B[23] * B[25] * B[32] - B[4] * B[6] * B[15] * B[23] * B[26] * B[31] -
               B[4] * B[6] * B[17] * B[19] * B[26] * B[33] + B[4] * B[6] * B[17] * B[19] * B[27] * B[32] +
               B[4] * B[6] * B[17] * B[20] * B[25] * B[33] - B[4] * B[6] * B[17] * B[20] * B[27] * B[31] -
               B[4] * B[6] * B[17] * B[21] * B[25] * B[32] + B[4] * B[6] * B[17] * B[21] * B[26] * B[31] -
               B[4] * B[7] * B[12] * B[20] * B[27] * B[35] + B[4] * B[7] * B[12] * B[20] * B[29] * B[33] +
               B[4] * B[7] * B[12] * B[21] * B[26] * B[35] - B[4] * B[7] * B[12] * B[21] * B[29] * B[32] -
               B[4] * B[7] * B[12] * B[23] * B[26] * B[33] + B[4] * B[7] * B[12] * B[23] * B[27] * B[32] +
               B[4] * B[7] * B[14] * B[18] * B[27] * B[35] - B[4] * B[7] * B[14] * B[18] * B[29] * B[33] -
               B[4] * B[7] * B[14] * B[21] * B[24] * B[35] + B[4] * B[7] * B[14] * B[21] * B[29] * B[30] +
               B[4] * B[7] * B[14] * B[23] * B[24] * B[33] - B[4] * B[7] * B[14] * B[23] * B[27] * B[30] -
               B[4] * B[7] * B[15] * B[18] * B[26] * B[35] + B[4] * B[7] * B[15] * B[18] * B[29] * B[32] +
               B[4] * B[7] * B[15] * B[20] * B[24] * B[35] - B[4] * B[7] * B[15] * B[20] * B[29] * B[30] -
               B[4] * B[7] * B[15] * B[23] * B[24] * B[32] + B[4] * B[7] * B[15] * B[23] * B[26] * B[30] +
               B[4] * B[7] * B[17] * B[18] * B[26] * B[33] - B[4] * B[7] * B[17] * B[18] * B[27] * B[32] -
               B[4] * B[7] * B[17] * B[20] * B[24] * B[33] + B[4] * B[7] * B[17] * B[20] * B[27] * B[30] +
               B[4] * B[7] * B[17] * B[21] * B[24] * B[32] - B[4] * B[7] * B[17] * B[21] * B[26] * B[30] +
               B[4] * B[8] * B[12] * B[19] * B[27] * B[35] - B[4] * B[8] * B[12] * B[19] * B[29] * B[33] -
               B[4] * B[8] * B[12] * B[21] * B[25] * B[35] + B[4] * B[8] * B[12] * B[21] * B[29] * B[31] +
               B[4] * B[8] * B[12] * B[23] * B[25] * B[33] - B[4] * B[8] * B[12] * B[23] * B[27] * B[31] -
               B[4] * B[8] * B[13] * B[18] * B[27] * B[35] + B[4] * B[8] * B[13] * B[18] * B[29] * B[33] +
               B[4] * B[8] * B[13] * B[21] * B[24] * B[35] - B[4] * B[8] * B[13] * B[21] * B[29] * B[30] -
               B[4] * B[8] * B[13] * B[23] * B[24] * B[33] + B[4] * B[8] * B[13] * B[23] * B[27] * B[30] +
               B[4] * B[8] * B[15] * B[18] * B[25] * B[35] - B[4] * B[8] * B[15] * B[18] * B[29] * B[31] -
               B[4] * B[8] * B[15] * B[19] * B[24] * B[35] + B[4] * B[8] * B[15] * B[19] * B[29] * B[30] +
               B[4] * B[8] * B[15] * B[23] * B[24] * B[31] - B[4] * B[8] * B[15] * B[23] * B[25] * B[30] -
               B[4] * B[8] * B[17] * B[18] * B[25] * B[33] + B[4] * B[8] * B[17] * B[18] * B[27] * B[31] +
               B[4] * B[8] * B[17] * B[19] * B[24] * B[33] - B[4] * B[8] * B[17] * B[19] * B[27] * B[30] -
               B[4] * B[8] * B[17] * B[21] * B[24] * B[31] + B[4] * B[8] * B[17] * B[21] * B[25] * B[30] -
               B[4] * B[9] * B[12] * B[19] * B[26] * B[35] + B[4] * B[9] * B[12] * B[19] * B[29] * B[32] +
               B[4] * B[9] * B[12] * B[20] * B[25] * B[35] - B[4] * B[9] * B[12] * B[20] * B[29] * B[31] -
               B[4] * B[9] * B[12] * B[23] * B[25] * B[32] + B[4] * B[9] * B[12] * B[23] * B[26] * B[31] +
               B[4] * B[9] * B[13] * B[18] * B[26] * B[35] - B[4] * B[9] * B[13] * B[18] * B[29] * B[32] -
               B[4] * B[9] * B[13] * B[20] * B[24] * B[35] + B[4] * B[9] * B[13] * B[20] * B[29] * B[30] +
               B[4] * B[9] * B[13] * B[23] * B[24] * B[32] - B[4] * B[9] * B[13] * B[23] * B[26] * B[30] -
               B[4] * B[9] * B[14] * B[18] * B[25] * B[35] + B[4] * B[9] * B[14] * B[18] * B[29] * B[31] +
               B[4] * B[9] * B[14] * B[19] * B[24] * B[35] - B[4] * B[9] * B[14] * B[19] * B[29] * B[30] -
               B[4] * B[9] * B[14] * B[23] * B[24] * B[31] + B[4] * B[9] * B[14] * B[23] * B[25] * B[30] +
               B[4] * B[9] * B[17] * B[18] * B[25] * B[32] - B[4] * B[9] * B[17] * B[18] * B[26] * B[31] -
               B[4] * B[9] * B[17] * B[19] * B[24] * B[32] + B[4] * B[9] * B[17] * B[19] * B[26] * B[30] +
               B[4] * B[9] * B[17] * B[20] * B[24] * B[31] - B[4] * B[9] * B[17] * B[20] * B[25] * B[30] +
               B[4] * B[11] * B[12] * B[19] * B[26] * B[33] - B[4] * B[11] * B[12] * B[19] * B[27] * B[32] -
               B[4] * B[11] * B[12] * B[20] * B[25] * B[33] + B[4] * B[11] * B[12] * B[20] * B[27] * B[31] +
               B[4] * B[11] * B[12] * B[21] * B[25] * B[32] - B[4] * B[11] * B[12] * B[21] * B[26] * B[31] -
               B[4] * B[11] * B[13] * B[18] * B[26] * B[33] + B[4] * B[11] * B[13] * B[18] * B[27] * B[32] +
               B[4] * B[11] * B[13] * B[20] * B[24] * B[33] - B[4] * B[11] * B[13] * B[20] * B[27] * B[30] -
               B[4] * B[11] * B[13] * B[21] * B[24] * B[32] + B[4] * B[11] * B[13] * B[21] * B[26] * B[30] +
               B[4] * B[11] * B[14] * B[18] * B[25] * B[33] - B[4] * B[11] * B[14] * B[18] * B[27] * B[31] -
               B[4] * B[11] * B[14] * B[19] * B[24] * B[33] + B[4] * B[11] * B[14] * B[19] * B[27] * B[30] +
               B[4] * B[11] * B[14] * B[21] * B[24] * B[31] - B[4] * B[11] * B[14] * B[21] * B[25] * B[30] -
               B[4] * B[11] * B[15] * B[18] * B[25] * B[32] + B[4] * B[11] * B[15] * B[18] * B[26] * B[31] +
               B[4] * B[11] * B[15] * B[19] * B[24] * B[32] - B[4] * B[11] * B[15] * B[19] * B[26] * B[30] -
               B[4] * B[11] * B[15] * B[20] * B[24] * B[31] + B[4] * B[11] * B[15] * B[20] * B[25] * B[30] -
               B[5] * B[6] * B[13] * B[20] * B[27] * B[34] + B[5] * B[6] * B[13] * B[20] * B[28] * B[33] +
               B[5] * B[6] * B[13] * B[21] * B[26] * B[34] - B[5] * B[6] * B[13] * B[21] * B[28] * B[32] -
               B[5] * B[6] * B[13] * B[22] * B[26] * B[33] + B[5] * B[6] * B[13] * B[22] * B[27] * B[32] +
               B[5] * B[6] * B[14] * B[19] * B[27] * B[34] - B[5] * B[6] * B[14] * B[19] * B[28] * B[33] -
               B[5] * B[6] * B[14] * B[21] * B[25] * B[34] + B[5] * B[6] * B[14] * B[21] * B[28] * B[31] +
               B[5] * B[6] * B[14] * B[22] * B[25] * B[33] - B[5] * B[6] * B[14] * B[22] * B[27] * B[31] -
               B[5] * B[6] * B[15] * B[19] * B[26] * B[34] + B[5] * B[6] * B[15] * B[19] * B[28] * B[32] +
               B[5] * B[6] * B[15] * B[20] * B[25] * B[34] - B[5] * B[6] * B[15] * B[20] * B[28] * B[31] -
               B[5] * B[6] * B[15] * B[22] * B[25] * B[32] + B[5] * B[6] * B[15] * B[22] * B[26] * B[31] +
               B[5] * B[6] * B[16] * B[19] * B[26] * B[33] - B[5] * B[6] * B[16] * B[19] * B[27] * B[32] -
               B[5] * B[6] * B[16] * B[20] * B[25] * B[33] + B[5] * B[6] * B[16] * B[20] * B[27] * B[31] +
               B[5] * B[6] * B[16] * B[21] * B[25] * B[32] - B[5] * B[6] * B[16] * B[21] * B[26] * B[31] +
               B[5] * B[7] * B[12] * B[20] * B[27] * B[34] - B[5] * B[7] * B[12] * B[20] * B[28] * B[33] -
               B[5] * B[7] * B[12] * B[21] * B[26] * B[34] + B[5] * B[7] * B[12] * B[21] * B[28] * B[32] +
               B[5] * B[7] * B[12] * B[22] * B[26] * B[33] - B[5] * B[7] * B[12] * B[22] * B[27] * B[32] -
               B[5] * B[7] * B[14] * B[18] * B[27] * B[34] + B[5] * B[7] * B[14] * B[18] * B[28] * B[33] +
               B[5] * B[7] * B[14] * B[21] * B[24] * B[34] - B[5] * B[7] * B[14] * B[21] * B[28] * B[30] -
               B[5] * B[7] * B[14] * B[22] * B[24] * B[33] + B[5] * B[7] * B[14] * B[22] * B[27] * B[30] +
               B[5] * B[7] * B[15] * B[18] * B[26] * B[34] - B[5] * B[7] * B[15] * B[18] * B[28] * B[32] -
               B[5] * B[7] * B[15] * B[20] * B[24] * B[34] + B[5] * B[7] * B[15] * B[20] * B[28] * B[30] +
               B[5] * B[7] * B[15] * B[22] * B[24] * B[32] - B[5] * B[7] * B[15] * B[22] * B[26] * B[30] -
               B[5] * B[7] * B[16] * B[18] * B[26] * B[33] + B[5] * B[7] * B[16] * B[18] * B[27] * B[32] +
               B[5] * B[7] * B[16] * B[20] * B[24] * B[33] - B[5] * B[7] * B[16] * B[20] * B[27] * B[30] -
               B[5] * B[7] * B[16] * B[21] * B[24] * B[32] + B[5] * B[7] * B[16] * B[21] * B[26] * B[30] -
               B[5] * B[8] * B[12] * B[19] * B[27] * B[34] + B[5] * B[8] * B[12] * B[19] * B[28] * B[33] +
               B[5] * B[8] * B[12] * B[21] * B[25] * B[34] - B[5] * B[8] * B[12] * B[21] * B[28] * B[31] -
               B[5] * B[8] * B[12] * B[22] * B[25] * B[33] + B[5] * B[8] * B[12] * B[22] * B[27] * B[31] +
               B[5] * B[8] * B[13] * B[18] * B[27] * B[34] - B[5] * B[8] * B[13] * B[18] * B[28] * B[33] -
               B[5] * B[8] * B[13] * B[21] * B[24] * B[34] + B[5] * B[8] * B[13] * B[21] * B[28] * B[30] +
               B[5] * B[8] * B[13] * B[22] * B[24] * B[33] - B[5] * B[8] * B[13] * B[22] * B[27] * B[30] -
               B[5] * B[8] * B[15] * B[18] * B[25] * B[34] + B[5] * B[8] * B[15] * B[18] * B[28] * B[31] +
               B[5] * B[8] * B[15] * B[19] * B[24] * B[34] - B[5] * B[8] * B[15] * B[19] * B[28] * B[30] -
               B[5] * B[8] * B[15] * B[22] * B[24] * B[31] + B[5] * B[8] * B[15] * B[22] * B[25] * B[30] +
               B[5] * B[8] * B[16] * B[18] * B[25] * B[33] - B[5] * B[8] * B[16] * B[18] * B[27] * B[31] -
               B[5] * B[8] * B[16] * B[19] * B[24] * B[33] + B[5] * B[8] * B[16] * B[19] * B[27] * B[30] +
               B[5] * B[8] * B[16] * B[21] * B[24] * B[31] - B[5] * B[8] * B[16] * B[21] * B[25] * B[30] +
               B[5] * B[9] * B[12] * B[19] * B[26] * B[34] - B[5] * B[9] * B[12] * B[19] * B[28] * B[32] -
               B[5] * B[9] * B[12] * B[20] * B[25] * B[34] + B[5] * B[9] * B[12] * B[20] * B[28] * B[31] +
               B[5] * B[9] * B[12] * B[22] * B[25] * B[32] - B[5] * B[9] * B[12] * B[22] * B[26] * B[31] -
               B[5] * B[9] * B[13] * B[18] * B[26] * B[34] + B[5] * B[9] * B[13] * B[18] * B[28] * B[32] +
               B[5] * B[9] * B[13] * B[20] * B[24] * B[34] - B[5] * B[9] * B[13] * B[20] * B[28] * B[30] -
               B[5] * B[9] * B[13] * B[22] * B[24] * B[32] + B[5] * B[9] * B[13] * B[22] * B[26] * B[30] +
               B[5] * B[9] * B[14] * B[18] * B[25] * B[34] - B[5] * B[9] * B[14] * B[18] * B[28] * B[31] -
               B[5] * B[9] * B[14] * B[19] * B[24] * B[34] + B[5] * B[9] * B[14] * B[19] * B[28] * B[30] +
               B[5] * B[9] * B[14] * B[22] * B[24] * B[31] - B[5] * B[9] * B[14] * B[22] * B[25] * B[30] -
               B[5] * B[9] * B[16] * B[18] * B[25] * B[32] + B[5] * B[9] * B[16] * B[18] * B[26] * B[31] +
               B[5] * B[9] * B[16] * B[19] * B[24] * B[32] - B[5] * B[9] * B[16] * B[19] * B[26] * B[30] -
               B[5] * B[9] * B[16] * B[20] * B[24] * B[31] + B[5] * B[9] * B[16] * B[20] * B[25] * B[30] -
               B[5] * B[10] * B[12] * B[19] * B[26] * B[33] + B[5] * B[10] * B[12] * B[19] * B[27] * B[32] +
               B[5] * B[10] * B[12] * B[20] * B[25] * B[33] - B[5] * B[10] * B[12] * B[20] * B[27] * B[31] -
               B[5] * B[10] * B[12] * B[21] * B[25] * B[32] + B[5] * B[10] * B[12] * B[21] * B[26] * B[31] +
               B[5] * B[10] * B[13] * B[18] * B[26] * B[33] - B[5] * B[10] * B[13] * B[18] * B[27] * B[32] -
               B[5] * B[10] * B[13] * B[20] * B[24] * B[33] + B[5] * B[10] * B[13] * B[20] * B[27] * B[30] +
               B[5] * B[10] * B[13] * B[21] * B[24] * B[32] - B[5] * B[10] * B[13] * B[21] * B[26] * B[30] -
               B[5] * B[10] * B[14] * B[18] * B[25] * B[33] + B[5] * B[10] * B[14] * B[18] * B[27] * B[31] +
               B[5] * B[10] * B[14] * B[19] * B[24] * B[33] - B[5] * B[10] * B[14] * B[19] * B[27] * B[30] -
               B[5] * B[10] * B[14] * B[21] * B[24] * B[31] + B[5] * B[10] * B[14] * B[21] * B[25] * B[30] +
               B[5] * B[10] * B[15] * B[18] * B[25] * B[32] - B[5] * B[10] * B[15] * B[18] * B[26] * B[31] -
               B[5] * B[10] * B[15] * B[19] * B[24] * B[32] + B[5] * B[10] * B[15] * B[19] * B[26] * B[30] +
               B[5] * B[10] * B[15] * B[20] * B[24] * B[31] - B[5] * B[10] * B[15] * B[20] * B[25] * B[30];

    L[0] = (B[1] * B[8] * B[15] * B[22] * B[29] - B[1] * B[8] * B[15] * B[23] * B[28] -
            B[1] * B[8] * B[16] * B[21] * B[29] + B[1] * B[8] * B[16] * B[23] * B[27] +
            B[1] * B[8] * B[17] * B[21] * B[28] - B[1] * B[8] * B[17] * B[22] * B[27] -
            B[1] * B[9] * B[14] * B[22] * B[29] + B[1] * B[9] * B[14] * B[23] * B[28] +
            B[1] * B[9] * B[16] * B[20] * B[29] - B[1] * B[9] * B[16] * B[23] * B[26] -
            B[1] * B[9] * B[17] * B[20] * B[28] + B[1] * B[9] * B[17] * B[22] * B[26] +
            B[1] * B[10] * B[14] * B[21] * B[29] - B[1] * B[10] * B[14] * B[23] * B[27] -
            B[1] * B[10] * B[15] * B[20] * B[29] + B[1] * B[10] * B[15] * B[23] * B[26] +
            B[1] * B[10] * B[17] * B[20] * B[27] - B[1] * B[10] * B[17] * B[21] * B[26] -
            B[1] * B[11] * B[14] * B[21] * B[28] + B[1] * B[11] * B[14] * B[22] * B[27] +
            B[1] * B[11] * B[15] * B[20] * B[28] - B[1] * B[11] * B[15] * B[22] * B[26] -
            B[1] * B[11] * B[16] * B[20] * B[27] + B[1] * B[11] * B[16] * B[21] * B[26] -
            B[2] * B[7] * B[15] * B[22] * B[29] + B[2] * B[7] * B[15] * B[23] * B[28] +
            B[2] * B[7] * B[16] * B[21] * B[29] - B[2] * B[7] * B[16] * B[23] * B[27] -
            B[2] * B[7] * B[17] * B[21] * B[28] + B[2] * B[7] * B[17] * B[22] * B[27] +
            B[2] * B[9] * B[13] * B[22] * B[29] - B[2] * B[9] * B[13] * B[23] * B[28] -
            B[2] * B[9] * B[16] * B[19] * B[29] + B[2] * B[9] * B[16] * B[23] * B[25] +
            B[2] * B[9] * B[17] * B[19] * B[28] - B[2] * B[9] * B[17] * B[22] * B[25] -
            B[2] * B[10] * B[13] * B[21] * B[29] + B[2] * B[10] * B[13] * B[23] * B[27] +
            B[2] * B[10] * B[15] * B[19] * B[29] - B[2] * B[10] * B[15] * B[23] * B[25] -
            B[2] * B[10] * B[17] * B[19] * B[27] + B[2] * B[10] * B[17] * B[21] * B[25] +
            B[2] * B[11] * B[13] * B[21] * B[28] - B[2] * B[11] * B[13] * B[22] * B[27] -
            B[2] * B[11] * B[15] * B[19] * B[28] + B[2] * B[11] * B[15] * B[22] * B[25] +
            B[2] * B[11] * B[16] * B[19] * B[27] - B[2] * B[11] * B[16] * B[21] * B[25] +
            B[3] * B[7] * B[14] * B[22] * B[29] - B[3] * B[7] * B[14] * B[23] * B[28] -
            B[3] * B[7] * B[16] * B[20] * B[29] + B[3] * B[7] * B[16] * B[23] * B[26] +
            B[3] * B[7] * B[17] * B[20] * B[28] - B[3] * B[7] * B[17] * B[22] * B[26] -
            B[3] * B[8] * B[13] * B[22] * B[29] + B[3] * B[8] * B[13] * B[23] * B[28] +
            B[3] * B[8] * B[16] * B[19] * B[29] - B[3] * B[8] * B[16] * B[23] * B[25] -
            B[3] * B[8] * B[17] * B[19] * B[28] + B[3] * B[8] * B[17] * B[22] * B[25] +
            B[3] * B[10] * B[13] * B[20] * B[29] - B[3] * B[10] * B[13] * B[23] * B[26] -
            B[3] * B[10] * B[14] * B[19] * B[29] + B[3] * B[10] * B[14] * B[23] * B[25] +
            B[3] * B[10] * B[17] * B[19] * B[26] - B[3] * B[10] * B[17] * B[20] * B[25] -
            B[3] * B[11] * B[13] * B[20] * B[28] + B[3] * B[11] * B[13] * B[22] * B[26] +
            B[3] * B[11] * B[14] * B[19] * B[28] - B[3] * B[11] * B[14] * B[22] * B[25] -
            B[3] * B[11] * B[16] * B[19] * B[26] + B[3] * B[11] * B[16] * B[20] * B[25] -
            B[4] * B[7] * B[14] * B[21] * B[29] + B[4] * B[7] * B[14] * B[23] * B[27] +
            B[4] * B[7] * B[15] * B[20] * B[29] - B[4] * B[7] * B[15] * B[23] * B[26] -
            B[4] * B[7] * B[17] * B[20] * B[27] + B[4] * B[7] * B[17] * B[21] * B[26] +
            B[4] * B[8] * B[13] * B[21] * B[29] - B[4] * B[8] * B[13] * B[23] * B[27] -
            B[4] * B[8] * B[15] * B[19] * B[29] + B[4] * B[8] * B[15] * B[23] * B[25] +
            B[4] * B[8] * B[17] * B[19] * B[27] - B[4] * B[8] * B[17] * B[21] * B[25] -
            B[4] * B[9] * B[13] * B[20] * B[29] + B[4] * B[9] * B[13] * B[23] * B[26] +
            B[4] * B[9] * B[14] * B[19] * B[29] - B[4] * B[9] * B[14] * B[23] * B[25] -
            B[4] * B[9] * B[17] * B[19] * B[26] + B[4] * B[9] * B[17] * B[20] * B[25] +
            B[4] * B[11] * B[13] * B[20] * B[27] - B[4] * B[11] * B[13] * B[21] * B[26] -
            B[4] * B[11] * B[14] * B[19] * B[27] + B[4] * B[11] * B[14] * B[21] * B[25] +
            B[4] * B[11] * B[15] * B[19] * B[26] - B[4] * B[11] * B[15] * B[20] * B[25] +
            B[5] * B[7] * B[14] * B[21] * B[28] - B[5] * B[7] * B[14] * B[22] * B[27] -
            B[5] * B[7] * B[15] * B[20] * B[28] + B[5] * B[7] * B[15] * B[22] * B[26] +
            B[5] * B[7] * B[16] * B[20] * B[27] - B[5] * B[7] * B[16] * B[21] * B[26] -
            B[5] * B[8] * B[13] * B[21] * B[28] + B[5] * B[8] * B[13] * B[22] * B[27] +
            B[5] * B[8] * B[15] * B[19] * B[28] - B[5] * B[8] * B[15] * B[22] * B[25] -
            B[5] * B[8] * B[16] * B[19] * B[27] + B[5] * B[8] * B[16] * B[21] * B[25] +
            B[5] * B[9] * B[13] * B[20] * B[28] - B[5] * B[9] * B[13] * B[22] * B[26] -
            B[5] * B[9] * B[14] * B[19] * B[28] + B[5] * B[9] * B[14] * B[22] * B[25] +
            B[5] * B[9] * B[16] * B[19] * B[26] - B[5] * B[9] * B[16] * B[20] * B[25] -
            B[5] * B[10] * B[13] * B[20] * B[27] + B[5] * B[10] * B[13] * B[21] * B[26] +
            B[5] * B[10] * B[14] * B[19] * B[27] - B[5] * B[10] * B[14] * B[21] * B[25] -
            B[5] * B[10] * B[15] * B[19] * B[26] + B[5] * B[10] * B[15] * B[20] * B[25] +
            B[1] * B[8] * B[15] * B[28] * B[35] - B[1] * B[8] * B[15] * B[29] * B[34] -
            B[1] * B[8] * B[16] * B[27] * B[35] + B[1] * B[8] * B[16] * B[29] * B[33] +
            B[1] * B[8] * B[17] * B[27] * B[34] - B[1] * B[8] * B[17] * B[28] * B[33] -
            B[1] * B[9] * B[14] * B[28] * B[35] + B[1] * B[9] * B[14] * B[29] * B[34] +
            B[1] * B[9] * B[16] * B[26] * B[35] - B[1] * B[9] * B[16] * B[29] * B[32] -
            B[1] * B[9] * B[17] * B[26] * B[34] + B[1] * B[9] * B[17] * B[28] * B[32] +
            B[1] * B[10] * B[14] * B[27] * B[35] - B[1] * B[10] * B[14] * B[29] * B[33] -
            B[1] * B[10] * B[15] * B[26] * B[35] + B[1] * B[10] * B[15] * B[29] * B[32] +
            B[1] * B[10] * B[17] * B[26] * B[33] - B[1] * B[10] * B[17] * B[27] * B[32] -
            B[1] * B[11] * B[14] * B[27] * B[34] + B[1] * B[11] * B[14] * B[28] * B[33] +
            B[1] * B[11] * B[15] * B[26] * B[34] - B[1] * B[11] * B[15] * B[28] * B[32] -
            B[1] * B[11] * B[16] * B[26] * B[33] + B[1] * B[11] * B[16] * B[27] * B[32] -
            B[2] * B[7] * B[15] * B[28] * B[35] + B[2] * B[7] * B[15] * B[29] * B[34] +
            B[2] * B[7] * B[16] * B[27] * B[35] - B[2] * B[7] * B[16] * B[29] * B[33] -
            B[2] * B[7] * B[17] * B[27] * B[34] + B[2] * B[7] * B[17] * B[28] * B[33] +
            B[2] * B[9] * B[13] * B[28] * B[35] - B[2] * B[9] * B[13] * B[29] * B[34] -
            B[2] * B[9] * B[16] * B[25] * B[35] + B[2] * B[9] * B[16] * B[29] * B[31] +
            B[2] * B[9] * B[17] * B[25] * B[34] - B[2] * B[9] * B[17] * B[28] * B[31] -
            B[2] * B[10] * B[13] * B[27] * B[35] + B[2] * B[10] * B[13] * B[29] * B[33] +
            B[2] * B[10] * B[15] * B[25] * B[35] - B[2] * B[10] * B[15] * B[29] * B[31] -
            B[2] * B[10] * B[17] * B[25] * B[33] + B[2] * B[10] * B[17] * B[27] * B[31] +
            B[2] * B[11] * B[13] * B[27] * B[34] - B[2] * B[11] * B[13] * B[28] * B[33] -
            B[2] * B[11] * B[15] * B[25] * B[34] + B[2] * B[11] * B[15] * B[28] * B[31] +
            B[2] * B[11] * B[16] * B[25] * B[33] - B[2] * B[11] * B[16] * B[27] * B[31] +
            B[3] * B[7] * B[14] * B[28] * B[35] - B[3] * B[7] * B[14] * B[29] * B[34] -
            B[3] * B[7] * B[16] * B[26] * B[35] + B[3] * B[7] * B[16] * B[29] * B[32] +
            B[3] * B[7] * B[17] * B[26] * B[34] - B[3] * B[7] * B[17] * B[28] * B[32] -
            B[3] * B[8] * B[13] * B[28] * B[35] + B[3] * B[8] * B[13] * B[29] * B[34] +
            B[3] * B[8] * B[16] * B[25] * B[35] - B[3] * B[8] * B[16] * B[29] * B[31] -
            B[3] * B[8] * B[17] * B[25] * B[34] + B[3] * B[8] * B[17] * B[28] * B[31] +
            B[3] * B[10] * B[13] * B[26] * B[35] - B[3] * B[10] * B[13] * B[29] * B[32] -
            B[3] * B[10] * B[14] * B[25] * B[35] + B[3] * B[10] * B[14] * B[29] * B[31] +
            B[3] * B[10] * B[17] * B[25] * B[32] - B[3] * B[10] * B[17] * B[26] * B[31] -
            B[3] * B[11] * B[13] * B[26] * B[34] + B[3] * B[11] * B[13] * B[28] * B[32] +
            B[3] * B[11] * B[14] * B[25] * B[34] - B[3] * B[11] * B[14] * B[28] * B[31] -
            B[3] * B[11] * B[16] * B[25] * B[32] + B[3] * B[11] * B[16] * B[26] * B[31] -
            B[4] * B[7] * B[14] * B[27] * B[35] + B[4] * B[7] * B[14] * B[29] * B[33] +
            B[4] * B[7] * B[15] * B[26] * B[35] - B[4] * B[7] * B[15] * B[29] * B[32] -
            B[4] * B[7] * B[17] * B[26] * B[33] + B[4] * B[7] * B[17] * B[27] * B[32] +
            B[4] * B[8] * B[13] * B[27] * B[35] - B[4] * B[8] * B[13] * B[29] * B[33] -
            B[4] * B[8] * B[15] * B[25] * B[35] + B[4] * B[8] * B[15] * B[29] * B[31] +
            B[4] * B[8] * B[17] * B[25] * B[33] - B[4] * B[8] * B[17] * B[27] * B[31] -
            B[4] * B[9] * B[13] * B[26] * B[35] + B[4] * B[9] * B[13] * B[29] * B[32] +
            B[4] * B[9] * B[14] * B[25] * B[35] - B[4] * B[9] * B[14] * B[29] * B[31] -
            B[4] * B[9] * B[17] * B[25] * B[32] + B[4] * B[9] * B[17] * B[26] * B[31] +
            B[4] * B[11] * B[13] * B[26] * B[33] - B[4] * B[11] * B[13] * B[27] * B[32] -
            B[4] * B[11] * B[14] * B[25] * B[33] + B[4] * B[11] * B[14] * B[27] * B[31] +
            B[4] * B[11] * B[15] * B[25] * B[32] - B[4] * B[11] * B[15] * B[26] * B[31] +
            B[5] * B[7] * B[14] * B[27] * B[34] - B[5] * B[7] * B[14] * B[28] * B[33] -
            B[5] * B[7] * B[15] * B[26] * B[34] + B[5] * B[7] * B[15] * B[28] * B[32] +
            B[5] * B[7] * B[16] * B[26] * B[33] - B[5] * B[7] * B[16] * B[27] * B[32] -
            B[5] * B[8] * B[13] * B[27] * B[34] + B[5] * B[8] * B[13] * B[28] * B[33] +
            B[5] * B[8] * B[15] * B[25] * B[34] - B[5] * B[8] * B[15] * B[28] * B[31] -
            B[5] * B[8] * B[16] * B[25] * B[33] + B[5] * B[8] * B[16] * B[27] * B[31] +
            B[5] * B[9] * B[13] * B[26] * B[34] - B[5] * B[9] * B[13] * B[28] * B[32] -
            B[5] * B[9] * B[14] * B[25] * B[34] + B[5] * B[9] * B[14] * B[28] * B[31] +
            B[5] * B[9] * B[16] * B[25] * B[32] - B[5] * B[9] * B[16] * B[26] * B[31] -
            B[5] * B[10] * B[13] * B[26] * B[33] + B[5] * B[10] * B[13] * B[27] * B[32] +
            B[5] * B[10] * B[14] * B[25] * B[33] - B[5] * B[10] * B[14] * B[27] * B[31] -
            B[5] * B[10] * B[15] * B[25] * B[32] + B[5] * B[10] * B[15] * B[26] * B[31] -
            B[7] * B[14] * B[21] * B[28] * B[35] + B[7] * B[14] * B[21] * B[29] * B[34] +
            B[7] * B[14] * B[22] * B[27] * B[35] - B[7] * B[14] * B[22] * B[29] * B[33] -
            B[7] * B[14] * B[23] * B[27] * B[34] + B[7] * B[14] * B[23] * B[28] * B[33] +
            B[7] * B[15] * B[20] * B[28] * B[35] - B[7] * B[15] * B[20] * B[29] * B[34] -
            B[7] * B[15] * B[22] * B[26] * B[35] + B[7] * B[15] * B[22] * B[29] * B[32] +
            B[7] * B[15] * B[23] * B[26] * B[34] - B[7] * B[15] * B[23] * B[28] * B[32] -
            B[7] * B[16] * B[20] * B[27] * B[35] + B[7] * B[16] * B[20] * B[29] * B[33] +
            B[7] * B[16] * B[21] * B[26] * B[35] - B[7] * B[16] * B[21] * B[29] * B[32] -
            B[7] * B[16] * B[23] * B[26] * B[33] + B[7] * B[16] * B[23] * B[27] * B[32] +
            B[7] * B[17] * B[20] * B[27] * B[34] - B[7] * B[17] * B[20] * B[28] * B[33] -
            B[7] * B[17] * B[21] * B[26] * B[34] + B[7] * B[17] * B[21] * B[28] * B[32] +
            B[7] * B[17] * B[22] * B[26] * B[33] - B[7] * B[17] * B[22] * B[27] * B[32] +
            B[8] * B[13] * B[21] * B[28] * B[35] - B[8] * B[13] * B[21] * B[29] * B[34] -
            B[8] * B[13] * B[22] * B[27] * B[35] + B[8] * B[13] * B[22] * B[29] * B[33] +
            B[8] * B[13] * B[23] * B[27] * B[34] - B[8] * B[13] * B[23] * B[28] * B[33] -
            B[8] * B[15] * B[19] * B[28] * B[35] + B[8] * B[15] * B[19] * B[29] * B[34] +
            B[8] * B[15] * B[22] * B[25] * B[35] - B[8] * B[15] * B[22] * B[29] * B[31] -
            B[8] * B[15] * B[23] * B[25] * B[34] + B[8] * B[15] * B[23] * B[28] * B[31] +
            B[8] * B[16] * B[19] * B[27] * B[35] - B[8] * B[16] * B[19] * B[29] * B[33] -
            B[8] * B[16] * B[21] * B[25] * B[35] + B[8] * B[16] * B[21] * B[29] * B[31] +
            B[8] * B[16] * B[23] * B[25] * B[33] - B[8] * B[16] * B[23] * B[27] * B[31] -
            B[8] * B[17] * B[19] * B[27] * B[34] + B[8] * B[17] * B[19] * B[28] * B[33] +
            B[8] * B[17] * B[21] * B[25] * B[34] - B[8] * B[17] * B[21] * B[28] * B[31] -
            B[8] * B[17] * B[22] * B[25] * B[33] + B[8] * B[17] * B[22] * B[27] * B[31] -
            B[9] * B[13] * B[20] * B[28] * B[35] + B[9] * B[13] * B[20] * B[29] * B[34] +
            B[9] * B[13] * B[22] * B[26] * B[35] - B[9] * B[13] * B[22] * B[29] * B[32] -
            B[9] * B[13] * B[23] * B[26] * B[34] + B[9] * B[13] * B[23] * B[28] * B[32] +
            B[9] * B[14] * B[19] * B[28] * B[35] - B[9] * B[14] * B[19] * B[29] * B[34] -
            B[9] * B[14] * B[22] * B[25] * B[35] + B[9] * B[14] * B[22] * B[29] * B[31] +
            B[9] * B[14] * B[23] * B[25] * B[34] - B[9] * B[14] * B[23] * B[28] * B[31] -
            B[9] * B[16] * B[19] * B[26] * B[35] + B[9] * B[16] * B[19] * B[29] * B[32] +
            B[9] * B[16] * B[20] * B[25] * B[35] - B[9] * B[16] * B[20] * B[29] * B[31] -
            B[9] * B[16] * B[23] * B[25] * B[32] + B[9] * B[16] * B[23] * B[26] * B[31] +
            B[9] * B[17] * B[19] * B[26] * B[34] - B[9] * B[17] * B[19] * B[28] * B[32] -
            B[9] * B[17] * B[20] * B[25] * B[34] + B[9] * B[17] * B[20] * B[28] * B[31] +
            B[9] * B[17] * B[22] * B[25] * B[32] - B[9] * B[17] * B[22] * B[26] * B[31] +
            B[10] * B[13] * B[20] * B[27] * B[35] - B[10] * B[13] * B[20] * B[29] * B[33] -
            B[10] * B[13] * B[21] * B[26] * B[35] + B[10] * B[13] * B[21] * B[29] * B[32] +
            B[10] * B[13] * B[23] * B[26] * B[33] - B[10] * B[13] * B[23] * B[27] * B[32] -
            B[10] * B[14] * B[19] * B[27] * B[35] + B[10] * B[14] * B[19] * B[29] * B[33] +
            B[10] * B[14] * B[21] * B[25] * B[35] - B[10] * B[14] * B[21] * B[29] * B[31] -
            B[10] * B[14] * B[23] * B[25] * B[33] + B[10] * B[14] * B[23] * B[27] * B[31] +
            B[10] * B[15] * B[19] * B[26] * B[35] - B[10] * B[15] * B[19] * B[29] * B[32] -
            B[10] * B[15] * B[20] * B[25] * B[35] + B[10] * B[15] * B[20] * B[29] * B[31] +
            B[10] * B[15] * B[23] * B[25] * B[32] - B[10] * B[15] * B[23] * B[26] * B[31] -
            B[10] * B[17] * B[19] * B[26] * B[33] + B[10] * B[17] * B[19] * B[27] * B[32] +
            B[10] * B[17] * B[20] * B[25] * B[33] - B[10] * B[17] * B[20] * B[27] * B[31] -
            B[10] * B[17] * B[21] * B[25] * B[32] + B[10] * B[17] * B[21] * B[26] * B[31] -
            B[11] * B[13] * B[20] * B[27] * B[34] + B[11] * B[13] * B[20] * B[28] * B[33] +
            B[11] * B[13] * B[21] * B[26] * B[34] - B[11] * B[13] * B[21] * B[28] * B[32] -
            B[11] * B[13] * B[22] * B[26] * B[33] + B[11] * B[13] * B[22] * B[27] * B[32] +
            B[11] * B[14] * B[19] * B[27] * B[34] - B[11] * B[14] * B[19] * B[28] * B[33] -
            B[11] * B[14] * B[21] * B[25] * B[34] + B[11] * B[14] * B[21] * B[28] * B[31] +
            B[11] * B[14] * B[22] * B[25] * B[33] - B[11] * B[14] * B[22] * B[27] * B[31] -
            B[11] * B[15] * B[19] * B[26] * B[34] + B[11] * B[15] * B[19] * B[28] * B[32] +
            B[11] * B[15] * B[20] * B[25] * B[34] - B[11] * B[15] * B[20] * B[28] * B[31] -
            B[11] * B[15] * B[22] * B[25] * B[32] + B[11] * B[15] * B[22] * B[26] * B[31] +
            B[11] * B[16] * B[19] * B[26] * B[33] - B[11] * B[16] * B[19] * B[27] * B[32] -
            B[11] * B[16] * B[20] * B[25] * B[33] + B[11] * B[16] * B[20] * B[27] * B[31] +
            B[11] * B[16] * B[21] * B[25] * B[32] - B[11] * B[16] * B[21] * B[26] * B[31]) /
           (DET);

    L[1] = -(B[0] * B[8] * B[15] * B[22] * B[29] - B[0] * B[8] * B[15] * B[23] * B[28] -
             B[0] * B[8] * B[16] * B[21] * B[29] + B[0] * B[8] * B[16] * B[23] * B[27] +
             B[0] * B[8] * B[17] * B[21] * B[28] - B[0] * B[8] * B[17] * B[22] * B[27] -
             B[0] * B[9] * B[14] * B[22] * B[29] + B[0] * B[9] * B[14] * B[23] * B[28] +
             B[0] * B[9] * B[16] * B[20] * B[29] - B[0] * B[9] * B[16] * B[23] * B[26] -
             B[0] * B[9] * B[17] * B[20] * B[28] + B[0] * B[9] * B[17] * B[22] * B[26] +
             B[0] * B[10] * B[14] * B[21] * B[29] - B[0] * B[10] * B[14] * B[23] * B[27] -
             B[0] * B[10] * B[15] * B[20] * B[29] + B[0] * B[10] * B[15] * B[23] * B[26] +
             B[0] * B[10] * B[17] * B[20] * B[27] - B[0] * B[10] * B[17] * B[21] * B[26] -
             B[0] * B[11] * B[14] * B[21] * B[28] + B[0] * B[11] * B[14] * B[22] * B[27] +
             B[0] * B[11] * B[15] * B[20] * B[28] - B[0] * B[11] * B[15] * B[22] * B[26] -
             B[0] * B[11] * B[16] * B[20] * B[27] + B[0] * B[11] * B[16] * B[21] * B[26] -
             B[2] * B[6] * B[15] * B[22] * B[29] + B[2] * B[6] * B[15] * B[23] * B[28] +
             B[2] * B[6] * B[16] * B[21] * B[29] - B[2] * B[6] * B[16] * B[23] * B[27] -
             B[2] * B[6] * B[17] * B[21] * B[28] + B[2] * B[6] * B[17] * B[22] * B[27] +
             B[2] * B[9] * B[12] * B[22] * B[29] - B[2] * B[9] * B[12] * B[23] * B[28] -
             B[2] * B[9] * B[16] * B[18] * B[29] + B[2] * B[9] * B[16] * B[23] * B[24] +
             B[2] * B[9] * B[17] * B[18] * B[28] - B[2] * B[9] * B[17] * B[22] * B[24] -
             B[2] * B[10] * B[12] * B[21] * B[29] + B[2] * B[10] * B[12] * B[23] * B[27] +
             B[2] * B[10] * B[15] * B[18] * B[29] - B[2] * B[10] * B[15] * B[23] * B[24] -
             B[2] * B[10] * B[17] * B[18] * B[27] + B[2] * B[10] * B[17] * B[21] * B[24] +
             B[2] * B[11] * B[12] * B[21] * B[28] - B[2] * B[11] * B[12] * B[22] * B[27] -
             B[2] * B[11] * B[15] * B[18] * B[28] + B[2] * B[11] * B[15] * B[22] * B[24] +
             B[2] * B[11] * B[16] * B[18] * B[27] - B[2] * B[11] * B[16] * B[21] * B[24] +
             B[3] * B[6] * B[14] * B[22] * B[29] - B[3] * B[6] * B[14] * B[23] * B[28] -
             B[3] * B[6] * B[16] * B[20] * B[29] + B[3] * B[6] * B[16] * B[23] * B[26] +
             B[3] * B[6] * B[17] * B[20] * B[28] - B[3] * B[6] * B[17] * B[22] * B[26] -
             B[3] * B[8] * B[12] * B[22] * B[29] + B[3] * B[8] * B[12] * B[23] * B[28] +
             B[3] * B[8] * B[16] * B[18] * B[29] - B[3] * B[8] * B[16] * B[23] * B[24] -
             B[3] * B[8] * B[17] * B[18] * B[28] + B[3] * B[8] * B[17] * B[22] * B[24] +
             B[3] * B[10] * B[12] * B[20] * B[29] - B[3] * B[10] * B[12] * B[23] * B[26] -
             B[3] * B[10] * B[14] * B[18] * B[29] + B[3] * B[10] * B[14] * B[23] * B[24] +
             B[3] * B[10] * B[17] * B[18] * B[26] - B[3] * B[10] * B[17] * B[20] * B[24] -
             B[3] * B[11] * B[12] * B[20] * B[28] + B[3] * B[11] * B[12] * B[22] * B[26] +
             B[3] * B[11] * B[14] * B[18] * B[28] - B[3] * B[11] * B[14] * B[22] * B[24] -
             B[3] * B[11] * B[16] * B[18] * B[26] + B[3] * B[11] * B[16] * B[20] * B[24] -
             B[4] * B[6] * B[14] * B[21] * B[29] + B[4] * B[6] * B[14] * B[23] * B[27] +
             B[4] * B[6] * B[15] * B[20] * B[29] - B[4] * B[6] * B[15] * B[23] * B[26] -
             B[4] * B[6] * B[17] * B[20] * B[27] + B[4] * B[6] * B[17] * B[21] * B[26] +
             B[4] * B[8] * B[12] * B[21] * B[29] - B[4] * B[8] * B[12] * B[23] * B[27] -
             B[4] * B[8] * B[15] * B[18] * B[29] + B[4] * B[8] * B[15] * B[23] * B[24] +
             B[4] * B[8] * B[17] * B[18] * B[27] - B[4] * B[8] * B[17] * B[21] * B[24] -
             B[4] * B[9] * B[12] * B[20] * B[29] + B[4] * B[9] * B[12] * B[23] * B[26] +
             B[4] * B[9] * B[14] * B[18] * B[29] - B[4] * B[9] * B[14] * B[23] * B[24] -
             B[4] * B[9] * B[17] * B[18] * B[26] + B[4] * B[9] * B[17] * B[20] * B[24] +
             B[4] * B[11] * B[12] * B[20] * B[27] - B[4] * B[11] * B[12] * B[21] * B[26] -
             B[4] * B[11] * B[14] * B[18] * B[27] + B[4] * B[11] * B[14] * B[21] * B[24] +
             B[4] * B[11] * B[15] * B[18] * B[26] - B[4] * B[11] * B[15] * B[20] * B[24] +
             B[5] * B[6] * B[14] * B[21] * B[28] - B[5] * B[6] * B[14] * B[22] * B[27] -
             B[5] * B[6] * B[15] * B[20] * B[28] + B[5] * B[6] * B[15] * B[22] * B[26] +
             B[5] * B[6] * B[16] * B[20] * B[27] - B[5] * B[6] * B[16] * B[21] * B[26] -
             B[5] * B[8] * B[12] * B[21] * B[28] + B[5] * B[8] * B[12] * B[22] * B[27] +
             B[5] * B[8] * B[15] * B[18] * B[28] - B[5] * B[8] * B[15] * B[22] * B[24] -
             B[5] * B[8] * B[16] * B[18] * B[27] + B[5] * B[8] * B[16] * B[21] * B[24] +
             B[5] * B[9] * B[12] * B[20] * B[28] - B[5] * B[9] * B[12] * B[22] * B[26] -
             B[5] * B[9] * B[14] * B[18] * B[28] + B[5] * B[9] * B[14] * B[22] * B[24] +
             B[5] * B[9] * B[16] * B[18] * B[26] - B[5] * B[9] * B[16] * B[20] * B[24] -
             B[5] * B[10] * B[12] * B[20] * B[27] + B[5] * B[10] * B[12] * B[21] * B[26] +
             B[5] * B[10] * B[14] * B[18] * B[27] - B[5] * B[10] * B[14] * B[21] * B[24] -
             B[5] * B[10] * B[15] * B[18] * B[26] + B[5] * B[10] * B[15] * B[20] * B[24] +
             B[0] * B[8] * B[15] * B[28] * B[35] - B[0] * B[8] * B[15] * B[29] * B[34] -
             B[0] * B[8] * B[16] * B[27] * B[35] + B[0] * B[8] * B[16] * B[29] * B[33] +
             B[0] * B[8] * B[17] * B[27] * B[34] - B[0] * B[8] * B[17] * B[28] * B[33] -
             B[0] * B[9] * B[14] * B[28] * B[35] + B[0] * B[9] * B[14] * B[29] * B[34] +
             B[0] * B[9] * B[16] * B[26] * B[35] - B[0] * B[9] * B[16] * B[29] * B[32] -
             B[0] * B[9] * B[17] * B[26] * B[34] + B[0] * B[9] * B[17] * B[28] * B[32] +
             B[0] * B[10] * B[14] * B[27] * B[35] - B[0] * B[10] * B[14] * B[29] * B[33] -
             B[0] * B[10] * B[15] * B[26] * B[35] + B[0] * B[10] * B[15] * B[29] * B[32] +
             B[0] * B[10] * B[17] * B[26] * B[33] - B[0] * B[10] * B[17] * B[27] * B[32] -
             B[0] * B[11] * B[14] * B[27] * B[34] + B[0] * B[11] * B[14] * B[28] * B[33] +
             B[0] * B[11] * B[15] * B[26] * B[34] - B[0] * B[11] * B[15] * B[28] * B[32] -
             B[0] * B[11] * B[16] * B[26] * B[33] + B[0] * B[11] * B[16] * B[27] * B[32] -
             B[2] * B[6] * B[15] * B[28] * B[35] + B[2] * B[6] * B[15] * B[29] * B[34] +
             B[2] * B[6] * B[16] * B[27] * B[35] - B[2] * B[6] * B[16] * B[29] * B[33] -
             B[2] * B[6] * B[17] * B[27] * B[34] + B[2] * B[6] * B[17] * B[28] * B[33] +
             B[2] * B[9] * B[12] * B[28] * B[35] - B[2] * B[9] * B[12] * B[29] * B[34] -
             B[2] * B[9] * B[16] * B[24] * B[35] + B[2] * B[9] * B[16] * B[29] * B[30] +
             B[2] * B[9] * B[17] * B[24] * B[34] - B[2] * B[9] * B[17] * B[28] * B[30] -
             B[2] * B[10] * B[12] * B[27] * B[35] + B[2] * B[10] * B[12] * B[29] * B[33] +
             B[2] * B[10] * B[15] * B[24] * B[35] - B[2] * B[10] * B[15] * B[29] * B[30] -
             B[2] * B[10] * B[17] * B[24] * B[33] + B[2] * B[10] * B[17] * B[27] * B[30] +
             B[2] * B[11] * B[12] * B[27] * B[34] - B[2] * B[11] * B[12] * B[28] * B[33] -
             B[2] * B[11] * B[15] * B[24] * B[34] + B[2] * B[11] * B[15] * B[28] * B[30] +
             B[2] * B[11] * B[16] * B[24] * B[33] - B[2] * B[11] * B[16] * B[27] * B[30] +
             B[3] * B[6] * B[14] * B[28] * B[35] - B[3] * B[6] * B[14] * B[29] * B[34] -
             B[3] * B[6] * B[16] * B[26] * B[35] + B[3] * B[6] * B[16] * B[29] * B[32] +
             B[3] * B[6] * B[17] * B[26] * B[34] - B[3] * B[6] * B[17] * B[28] * B[32] -
             B[3] * B[8] * B[12] * B[28] * B[35] + B[3] * B[8] * B[12] * B[29] * B[34] +
             B[3] * B[8] * B[16] * B[24] * B[35] - B[3] * B[8] * B[16] * B[29] * B[30] -
             B[3] * B[8] * B[17] * B[24] * B[34] + B[3] * B[8] * B[17] * B[28] * B[30] +
             B[3] * B[10] * B[12] * B[26] * B[35] - B[3] * B[10] * B[12] * B[29] * B[32] -
             B[3] * B[10] * B[14] * B[24] * B[35] + B[3] * B[10] * B[14] * B[29] * B[30] +
             B[3] * B[10] * B[17] * B[24] * B[32] - B[3] * B[10] * B[17] * B[26] * B[30] -
             B[3] * B[11] * B[12] * B[26] * B[34] + B[3] * B[11] * B[12] * B[28] * B[32] +
             B[3] * B[11] * B[14] * B[24] * B[34] - B[3] * B[11] * B[14] * B[28] * B[30] -
             B[3] * B[11] * B[16] * B[24] * B[32] + B[3] * B[11] * B[16] * B[26] * B[30] -
             B[4] * B[6] * B[14] * B[27] * B[35] + B[4] * B[6] * B[14] * B[29] * B[33] +
             B[4] * B[6] * B[15] * B[26] * B[35] - B[4] * B[6] * B[15] * B[29] * B[32] -
             B[4] * B[6] * B[17] * B[26] * B[33] + B[4] * B[6] * B[17] * B[27] * B[32] +
             B[4] * B[8] * B[12] * B[27] * B[35] - B[4] * B[8] * B[12] * B[29] * B[33] -
             B[4] * B[8] * B[15] * B[24] * B[35] + B[4] * B[8] * B[15] * B[29] * B[30] +
             B[4] * B[8] * B[17] * B[24] * B[33] - B[4] * B[8] * B[17] * B[27] * B[30] -
             B[4] * B[9] * B[12] * B[26] * B[35] + B[4] * B[9] * B[12] * B[29] * B[32] +
             B[4] * B[9] * B[14] * B[24] * B[35] - B[4] * B[9] * B[14] * B[29] * B[30] -
             B[4] * B[9] * B[17] * B[24] * B[32] + B[4] * B[9] * B[17] * B[26] * B[30] +
             B[4] * B[11] * B[12] * B[26] * B[33] - B[4] * B[11] * B[12] * B[27] * B[32] -
             B[4] * B[11] * B[14] * B[24] * B[33] + B[4] * B[11] * B[14] * B[27] * B[30] +
             B[4] * B[11] * B[15] * B[24] * B[32] - B[4] * B[11] * B[15] * B[26] * B[30] +
             B[5] * B[6] * B[14] * B[27] * B[34] - B[5] * B[6] * B[14] * B[28] * B[33] -
             B[5] * B[6] * B[15] * B[26] * B[34] + B[5] * B[6] * B[15] * B[28] * B[32] +
             B[5] * B[6] * B[16] * B[26] * B[33] - B[5] * B[6] * B[16] * B[27] * B[32] -
             B[5] * B[8] * B[12] * B[27] * B[34] + B[5] * B[8] * B[12] * B[28] * B[33] +
             B[5] * B[8] * B[15] * B[24] * B[34] - B[5] * B[8] * B[15] * B[28] * B[30] -
             B[5] * B[8] * B[16] * B[24] * B[33] + B[5] * B[8] * B[16] * B[27] * B[30] +
             B[5] * B[9] * B[12] * B[26] * B[34] - B[5] * B[9] * B[12] * B[28] * B[32] -
             B[5] * B[9] * B[14] * B[24] * B[34] + B[5] * B[9] * B[14] * B[28] * B[30] +
             B[5] * B[9] * B[16] * B[24] * B[32] - B[5] * B[9] * B[16] * B[26] * B[30] -
             B[5] * B[10] * B[12] * B[26] * B[33] + B[5] * B[10] * B[12] * B[27] * B[32] +
             B[5] * B[10] * B[14] * B[24] * B[33] - B[5] * B[10] * B[14] * B[27] * B[30] -
             B[5] * B[10] * B[15] * B[24] * B[32] + B[5] * B[10] * B[15] * B[26] * B[30] -
             B[6] * B[14] * B[21] * B[28] * B[35] + B[6] * B[14] * B[21] * B[29] * B[34] +
             B[6] * B[14] * B[22] * B[27] * B[35] - B[6] * B[14] * B[22] * B[29] * B[33] -
             B[6] * B[14] * B[23] * B[27] * B[34] + B[6] * B[14] * B[23] * B[28] * B[33] +
             B[6] * B[15] * B[20] * B[28] * B[35] - B[6] * B[15] * B[20] * B[29] * B[34] -
             B[6] * B[15] * B[22] * B[26] * B[35] + B[6] * B[15] * B[22] * B[29] * B[32] +
             B[6] * B[15] * B[23] * B[26] * B[34] - B[6] * B[15] * B[23] * B[28] * B[32] -
             B[6] * B[16] * B[20] * B[27] * B[35] + B[6] * B[16] * B[20] * B[29] * B[33] +
             B[6] * B[16] * B[21] * B[26] * B[35] - B[6] * B[16] * B[21] * B[29] * B[32] -
             B[6] * B[16] * B[23] * B[26] * B[33] + B[6] * B[16] * B[23] * B[27] * B[32] +
             B[6] * B[17] * B[20] * B[27] * B[34] - B[6] * B[17] * B[20] * B[28] * B[33] -
             B[6] * B[17] * B[21] * B[26] * B[34] + B[6] * B[17] * B[21] * B[28] * B[32] +
             B[6] * B[17] * B[22] * B[26] * B[33] - B[6] * B[17] * B[22] * B[27] * B[32] +
             B[8] * B[12] * B[21] * B[28] * B[35] - B[8] * B[12] * B[21] * B[29] * B[34] -
             B[8] * B[12] * B[22] * B[27] * B[35] + B[8] * B[12] * B[22] * B[29] * B[33] +
             B[8] * B[12] * B[23] * B[27] * B[34] - B[8] * B[12] * B[23] * B[28] * B[33] -
             B[8] * B[15] * B[18] * B[28] * B[35] + B[8] * B[15] * B[18] * B[29] * B[34] +
             B[8] * B[15] * B[22] * B[24] * B[35] - B[8] * B[15] * B[22] * B[29] * B[30] -
             B[8] * B[15] * B[23] * B[24] * B[34] + B[8] * B[15] * B[23] * B[28] * B[30] +
             B[8] * B[16] * B[18] * B[27] * B[35] - B[8] * B[16] * B[18] * B[29] * B[33] -
             B[8] * B[16] * B[21] * B[24] * B[35] + B[8] * B[16] * B[21] * B[29] * B[30] +
             B[8] * B[16] * B[23] * B[24] * B[33] - B[8] * B[16] * B[23] * B[27] * B[30] -
             B[8] * B[17] * B[18] * B[27] * B[34] + B[8] * B[17] * B[18] * B[28] * B[33] +
             B[8] * B[17] * B[21] * B[24] * B[34] - B[8] * B[17] * B[21] * B[28] * B[30] -
             B[8] * B[17] * B[22] * B[24] * B[33] + B[8] * B[17] * B[22] * B[27] * B[30] -
             B[9] * B[12] * B[20] * B[28] * B[35] + B[9] * B[12] * B[20] * B[29] * B[34] +
             B[9] * B[12] * B[22] * B[26] * B[35] - B[9] * B[12] * B[22] * B[29] * B[32] -
             B[9] * B[12] * B[23] * B[26] * B[34] + B[9] * B[12] * B[23] * B[28] * B[32] +
             B[9] * B[14] * B[18] * B[28] * B[35] - B[9] * B[14] * B[18] * B[29] * B[34] -
             B[9] * B[14] * B[22] * B[24] * B[35] + B[9] * B[14] * B[22] * B[29] * B[30] +
             B[9] * B[14] * B[23] * B[24] * B[34] - B[9] * B[14] * B[23] * B[28] * B[30] -
             B[9] * B[16] * B[18] * B[26] * B[35] + B[9] * B[16] * B[18] * B[29] * B[32] +
             B[9] * B[16] * B[20] * B[24] * B[35] - B[9] * B[16] * B[20] * B[29] * B[30] -
             B[9] * B[16] * B[23] * B[24] * B[32] + B[9] * B[16] * B[23] * B[26] * B[30] +
             B[9] * B[17] * B[18] * B[26] * B[34] - B[9] * B[17] * B[18] * B[28] * B[32] -
             B[9] * B[17] * B[20] * B[24] * B[34] + B[9] * B[17] * B[20] * B[28] * B[30] +
             B[9] * B[17] * B[22] * B[24] * B[32] - B[9] * B[17] * B[22] * B[26] * B[30] +
             B[10] * B[12] * B[20] * B[27] * B[35] - B[10] * B[12] * B[20] * B[29] * B[33] -
             B[10] * B[12] * B[21] * B[26] * B[35] + B[10] * B[12] * B[21] * B[29] * B[32] +
             B[10] * B[12] * B[23] * B[26] * B[33] - B[10] * B[12] * B[23] * B[27] * B[32] -
             B[10] * B[14] * B[18] * B[27] * B[35] + B[10] * B[14] * B[18] * B[29] * B[33] +
             B[10] * B[14] * B[21] * B[24] * B[35] - B[10] * B[14] * B[21] * B[29] * B[30] -
             B[10] * B[14] * B[23] * B[24] * B[33] + B[10] * B[14] * B[23] * B[27] * B[30] +
             B[10] * B[15] * B[18] * B[26] * B[35] - B[10] * B[15] * B[18] * B[29] * B[32] -
             B[10] * B[15] * B[20] * B[24] * B[35] + B[10] * B[15] * B[20] * B[29] * B[30] +
             B[10] * B[15] * B[23] * B[24] * B[32] - B[10] * B[15] * B[23] * B[26] * B[30] -
             B[10] * B[17] * B[18] * B[26] * B[33] + B[10] * B[17] * B[18] * B[27] * B[32] +
             B[10] * B[17] * B[20] * B[24] * B[33] - B[10] * B[17] * B[20] * B[27] * B[30] -
             B[10] * B[17] * B[21] * B[24] * B[32] + B[10] * B[17] * B[21] * B[26] * B[30] -
             B[11] * B[12] * B[20] * B[27] * B[34] + B[11] * B[12] * B[20] * B[28] * B[33] +
             B[11] * B[12] * B[21] * B[26] * B[34] - B[11] * B[12] * B[21] * B[28] * B[32] -
             B[11] * B[12] * B[22] * B[26] * B[33] + B[11] * B[12] * B[22] * B[27] * B[32] +
             B[11] * B[14] * B[18] * B[27] * B[34] - B[11] * B[14] * B[18] * B[28] * B[33] -
             B[11] * B[14] * B[21] * B[24] * B[34] + B[11] * B[14] * B[21] * B[28] * B[30] +
             B[11] * B[14] * B[22] * B[24] * B[33] - B[11] * B[14] * B[22] * B[27] * B[30] -
             B[11] * B[15] * B[18] * B[26] * B[34] + B[11] * B[15] * B[18] * B[28] * B[32] +
             B[11] * B[15] * B[20] * B[24] * B[34] - B[11] * B[15] * B[20] * B[28] * B[30] -
             B[11] * B[15] * B[22] * B[24] * B[32] + B[11] * B[15] * B[22] * B[26] * B[30] +
             B[11] * B[16] * B[18] * B[26] * B[33] - B[11] * B[16] * B[18] * B[27] * B[32] -
             B[11] * B[16] * B[20] * B[24] * B[33] + B[11] * B[16] * B[20] * B[27] * B[30] +
             B[11] * B[16] * B[21] * B[24] * B[32] - B[11] * B[16] * B[21] * B[26] * B[30]) /
           (DET);

    L[2] = (B[0] * B[7] * B[15] * B[22] * B[29] - B[0] * B[7] * B[15] * B[23] * B[28] -
            B[0] * B[7] * B[16] * B[21] * B[29] + B[0] * B[7] * B[16] * B[23] * B[27] +
            B[0] * B[7] * B[17] * B[21] * B[28] - B[0] * B[7] * B[17] * B[22] * B[27] -
            B[0] * B[9] * B[13] * B[22] * B[29] + B[0] * B[9] * B[13] * B[23] * B[28] +
            B[0] * B[9] * B[16] * B[19] * B[29] - B[0] * B[9] * B[16] * B[23] * B[25] -
            B[0] * B[9] * B[17] * B[19] * B[28] + B[0] * B[9] * B[17] * B[22] * B[25] +
            B[0] * B[10] * B[13] * B[21] * B[29] - B[0] * B[10] * B[13] * B[23] * B[27] -
            B[0] * B[10] * B[15] * B[19] * B[29] + B[0] * B[10] * B[15] * B[23] * B[25] +
            B[0] * B[10] * B[17] * B[19] * B[27] - B[0] * B[10] * B[17] * B[21] * B[25] -
            B[0] * B[11] * B[13] * B[21] * B[28] + B[0] * B[11] * B[13] * B[22] * B[27] +
            B[0] * B[11] * B[15] * B[19] * B[28] - B[0] * B[11] * B[15] * B[22] * B[25] -
            B[0] * B[11] * B[16] * B[19] * B[27] + B[0] * B[11] * B[16] * B[21] * B[25] -
            B[1] * B[6] * B[15] * B[22] * B[29] + B[1] * B[6] * B[15] * B[23] * B[28] +
            B[1] * B[6] * B[16] * B[21] * B[29] - B[1] * B[6] * B[16] * B[23] * B[27] -
            B[1] * B[6] * B[17] * B[21] * B[28] + B[1] * B[6] * B[17] * B[22] * B[27] +
            B[1] * B[9] * B[12] * B[22] * B[29] - B[1] * B[9] * B[12] * B[23] * B[28] -
            B[1] * B[9] * B[16] * B[18] * B[29] + B[1] * B[9] * B[16] * B[23] * B[24] +
            B[1] * B[9] * B[17] * B[18] * B[28] - B[1] * B[9] * B[17] * B[22] * B[24] -
            B[1] * B[10] * B[12] * B[21] * B[29] + B[1] * B[10] * B[12] * B[23] * B[27] +
            B[1] * B[10] * B[15] * B[18] * B[29] - B[1] * B[10] * B[15] * B[23] * B[24] -
            B[1] * B[10] * B[17] * B[18] * B[27] + B[1] * B[10] * B[17] * B[21] * B[24] +
            B[1] * B[11] * B[12] * B[21] * B[28] - B[1] * B[11] * B[12] * B[22] * B[27] -
            B[1] * B[11] * B[15] * B[18] * B[28] + B[1] * B[11] * B[15] * B[22] * B[24] +
            B[1] * B[11] * B[16] * B[18] * B[27] - B[1] * B[11] * B[16] * B[21] * B[24] +
            B[3] * B[6] * B[13] * B[22] * B[29] - B[3] * B[6] * B[13] * B[23] * B[28] -
            B[3] * B[6] * B[16] * B[19] * B[29] + B[3] * B[6] * B[16] * B[23] * B[25] +
            B[3] * B[6] * B[17] * B[19] * B[28] - B[3] * B[6] * B[17] * B[22] * B[25] -
            B[3] * B[7] * B[12] * B[22] * B[29] + B[3] * B[7] * B[12] * B[23] * B[28] +
            B[3] * B[7] * B[16] * B[18] * B[29] - B[3] * B[7] * B[16] * B[23] * B[24] -
            B[3] * B[7] * B[17] * B[18] * B[28] + B[3] * B[7] * B[17] * B[22] * B[24] +
            B[3] * B[10] * B[12] * B[19] * B[29] - B[3] * B[10] * B[12] * B[23] * B[25] -
            B[3] * B[10] * B[13] * B[18] * B[29] + B[3] * B[10] * B[13] * B[23] * B[24] +
            B[3] * B[10] * B[17] * B[18] * B[25] - B[3] * B[10] * B[17] * B[19] * B[24] -
            B[3] * B[11] * B[12] * B[19] * B[28] + B[3] * B[11] * B[12] * B[22] * B[25] +
            B[3] * B[11] * B[13] * B[18] * B[28] - B[3] * B[11] * B[13] * B[22] * B[24] -
            B[3] * B[11] * B[16] * B[18] * B[25] + B[3] * B[11] * B[16] * B[19] * B[24] -
            B[4] * B[6] * B[13] * B[21] * B[29] + B[4] * B[6] * B[13] * B[23] * B[27] +
            B[4] * B[6] * B[15] * B[19] * B[29] - B[4] * B[6] * B[15] * B[23] * B[25] -
            B[4] * B[6] * B[17] * B[19] * B[27] + B[4] * B[6] * B[17] * B[21] * B[25] +
            B[4] * B[7] * B[12] * B[21] * B[29] - B[4] * B[7] * B[12] * B[23] * B[27] -
            B[4] * B[7] * B[15] * B[18] * B[29] + B[4] * B[7] * B[15] * B[23] * B[24] +
            B[4] * B[7] * B[17] * B[18] * B[27] - B[4] * B[7] * B[17] * B[21] * B[24] -
            B[4] * B[9] * B[12] * B[19] * B[29] + B[4] * B[9] * B[12] * B[23] * B[25] +
            B[4] * B[9] * B[13] * B[18] * B[29] - B[4] * B[9] * B[13] * B[23] * B[24] -
            B[4] * B[9] * B[17] * B[18] * B[25] + B[4] * B[9] * B[17] * B[19] * B[24] +
            B[4] * B[11] * B[12] * B[19] * B[27] - B[4] * B[11] * B[12] * B[21] * B[25] -
            B[4] * B[11] * B[13] * B[18] * B[27] + B[4] * B[11] * B[13] * B[21] * B[24] +
            B[4] * B[11] * B[15] * B[18] * B[25] - B[4] * B[11] * B[15] * B[19] * B[24] +
            B[5] * B[6] * B[13] * B[21] * B[28] - B[5] * B[6] * B[13] * B[22] * B[27] -
            B[5] * B[6] * B[15] * B[19] * B[28] + B[5] * B[6] * B[15] * B[22] * B[25] +
            B[5] * B[6] * B[16] * B[19] * B[27] - B[5] * B[6] * B[16] * B[21] * B[25] -
            B[5] * B[7] * B[12] * B[21] * B[28] + B[5] * B[7] * B[12] * B[22] * B[27] +
            B[5] * B[7] * B[15] * B[18] * B[28] - B[5] * B[7] * B[15] * B[22] * B[24] -
            B[5] * B[7] * B[16] * B[18] * B[27] + B[5] * B[7] * B[16] * B[21] * B[24] +
            B[5] * B[9] * B[12] * B[19] * B[28] - B[5] * B[9] * B[12] * B[22] * B[25] -
            B[5] * B[9] * B[13] * B[18] * B[28] + B[5] * B[9] * B[13] * B[22] * B[24] +
            B[5] * B[9] * B[16] * B[18] * B[25] - B[5] * B[9] * B[16] * B[19] * B[24] -
            B[5] * B[10] * B[12] * B[19] * B[27] + B[5] * B[10] * B[12] * B[21] * B[25] +
            B[5] * B[10] * B[13] * B[18] * B[27] - B[5] * B[10] * B[13] * B[21] * B[24] -
            B[5] * B[10] * B[15] * B[18] * B[25] + B[5] * B[10] * B[15] * B[19] * B[24] +
            B[0] * B[7] * B[15] * B[28] * B[35] - B[0] * B[7] * B[15] * B[29] * B[34] -
            B[0] * B[7] * B[16] * B[27] * B[35] + B[0] * B[7] * B[16] * B[29] * B[33] +
            B[0] * B[7] * B[17] * B[27] * B[34] - B[0] * B[7] * B[17] * B[28] * B[33] -
            B[0] * B[9] * B[13] * B[28] * B[35] + B[0] * B[9] * B[13] * B[29] * B[34] +
            B[0] * B[9] * B[16] * B[25] * B[35] - B[0] * B[9] * B[16] * B[29] * B[31] -
            B[0] * B[9] * B[17] * B[25] * B[34] + B[0] * B[9] * B[17] * B[28] * B[31] +
            B[0] * B[10] * B[13] * B[27] * B[35] - B[0] * B[10] * B[13] * B[29] * B[33] -
            B[0] * B[10] * B[15] * B[25] * B[35] + B[0] * B[10] * B[15] * B[29] * B[31] +
            B[0] * B[10] * B[17] * B[25] * B[33] - B[0] * B[10] * B[17] * B[27] * B[31] -
            B[0] * B[11] * B[13] * B[27] * B[34] + B[0] * B[11] * B[13] * B[28] * B[33] +
            B[0] * B[11] * B[15] * B[25] * B[34] - B[0] * B[11] * B[15] * B[28] * B[31] -
            B[0] * B[11] * B[16] * B[25] * B[33] + B[0] * B[11] * B[16] * B[27] * B[31] -
            B[1] * B[6] * B[15] * B[28] * B[35] + B[1] * B[6] * B[15] * B[29] * B[34] +
            B[1] * B[6] * B[16] * B[27] * B[35] - B[1] * B[6] * B[16] * B[29] * B[33] -
            B[1] * B[6] * B[17] * B[27] * B[34] + B[1] * B[6] * B[17] * B[28] * B[33] +
            B[1] * B[9] * B[12] * B[28] * B[35] - B[1] * B[9] * B[12] * B[29] * B[34] -
            B[1] * B[9] * B[16] * B[24] * B[35] + B[1] * B[9] * B[16] * B[29] * B[30] +
            B[1] * B[9] * B[17] * B[24] * B[34] - B[1] * B[9] * B[17] * B[28] * B[30] -
            B[1] * B[10] * B[12] * B[27] * B[35] + B[1] * B[10] * B[12] * B[29] * B[33] +
            B[1] * B[10] * B[15] * B[24] * B[35] - B[1] * B[10] * B[15] * B[29] * B[30] -
            B[1] * B[10] * B[17] * B[24] * B[33] + B[1] * B[10] * B[17] * B[27] * B[30] +
            B[1] * B[11] * B[12] * B[27] * B[34] - B[1] * B[11] * B[12] * B[28] * B[33] -
            B[1] * B[11] * B[15] * B[24] * B[34] + B[1] * B[11] * B[15] * B[28] * B[30] +
            B[1] * B[11] * B[16] * B[24] * B[33] - B[1] * B[11] * B[16] * B[27] * B[30] +
            B[3] * B[6] * B[13] * B[28] * B[35] - B[3] * B[6] * B[13] * B[29] * B[34] -
            B[3] * B[6] * B[16] * B[25] * B[35] + B[3] * B[6] * B[16] * B[29] * B[31] +
            B[3] * B[6] * B[17] * B[25] * B[34] - B[3] * B[6] * B[17] * B[28] * B[31] -
            B[3] * B[7] * B[12] * B[28] * B[35] + B[3] * B[7] * B[12] * B[29] * B[34] +
            B[3] * B[7] * B[16] * B[24] * B[35] - B[3] * B[7] * B[16] * B[29] * B[30] -
            B[3] * B[7] * B[17] * B[24] * B[34] + B[3] * B[7] * B[17] * B[28] * B[30] +
            B[3] * B[10] * B[12] * B[25] * B[35] - B[3] * B[10] * B[12] * B[29] * B[31] -
            B[3] * B[10] * B[13] * B[24] * B[35] + B[3] * B[10] * B[13] * B[29] * B[30] +
            B[3] * B[10] * B[17] * B[24] * B[31] - B[3] * B[10] * B[17] * B[25] * B[30] -
            B[3] * B[11] * B[12] * B[25] * B[34] + B[3] * B[11] * B[12] * B[28] * B[31] +
            B[3] * B[11] * B[13] * B[24] * B[34] - B[3] * B[11] * B[13] * B[28] * B[30] -
            B[3] * B[11] * B[16] * B[24] * B[31] + B[3] * B[11] * B[16] * B[25] * B[30] -
            B[4] * B[6] * B[13] * B[27] * B[35] + B[4] * B[6] * B[13] * B[29] * B[33] +
            B[4] * B[6] * B[15] * B[25] * B[35] - B[4] * B[6] * B[15] * B[29] * B[31] -
            B[4] * B[6] * B[17] * B[25] * B[33] + B[4] * B[6] * B[17] * B[27] * B[31] +
            B[4] * B[7] * B[12] * B[27] * B[35] - B[4] * B[7] * B[12] * B[29] * B[33] -
            B[4] * B[7] * B[15] * B[24] * B[35] + B[4] * B[7] * B[15] * B[29] * B[30] +
            B[4] * B[7] * B[17] * B[24] * B[33] - B[4] * B[7] * B[17] * B[27] * B[30] -
            B[4] * B[9] * B[12] * B[25] * B[35] + B[4] * B[9] * B[12] * B[29] * B[31] +
            B[4] * B[9] * B[13] * B[24] * B[35] - B[4] * B[9] * B[13] * B[29] * B[30] -
            B[4] * B[9] * B[17] * B[24] * B[31] + B[4] * B[9] * B[17] * B[25] * B[30] +
            B[4] * B[11] * B[12] * B[25] * B[33] - B[4] * B[11] * B[12] * B[27] * B[31] -
            B[4] * B[11] * B[13] * B[24] * B[33] + B[4] * B[11] * B[13] * B[27] * B[30] +
            B[4] * B[11] * B[15] * B[24] * B[31] - B[4] * B[11] * B[15] * B[25] * B[30] +
            B[5] * B[6] * B[13] * B[27] * B[34] - B[5] * B[6] * B[13] * B[28] * B[33] -
            B[5] * B[6] * B[15] * B[25] * B[34] + B[5] * B[6] * B[15] * B[28] * B[31] +
            B[5] * B[6] * B[16] * B[25] * B[33] - B[5] * B[6] * B[16] * B[27] * B[31] -
            B[5] * B[7] * B[12] * B[27] * B[34] + B[5] * B[7] * B[12] * B[28] * B[33] +
            B[5] * B[7] * B[15] * B[24] * B[34] - B[5] * B[7] * B[15] * B[28] * B[30] -
            B[5] * B[7] * B[16] * B[24] * B[33] + B[5] * B[7] * B[16] * B[27] * B[30] +
            B[5] * B[9] * B[12] * B[25] * B[34] - B[5] * B[9] * B[12] * B[28] * B[31] -
            B[5] * B[9] * B[13] * B[24] * B[34] + B[5] * B[9] * B[13] * B[28] * B[30] +
            B[5] * B[9] * B[16] * B[24] * B[31] - B[5] * B[9] * B[16] * B[25] * B[30] -
            B[5] * B[10] * B[12] * B[25] * B[33] + B[5] * B[10] * B[12] * B[27] * B[31] +
            B[5] * B[10] * B[13] * B[24] * B[33] - B[5] * B[10] * B[13] * B[27] * B[30] -
            B[5] * B[10] * B[15] * B[24] * B[31] + B[5] * B[10] * B[15] * B[25] * B[30] -
            B[6] * B[13] * B[21] * B[28] * B[35] + B[6] * B[13] * B[21] * B[29] * B[34] +
            B[6] * B[13] * B[22] * B[27] * B[35] - B[6] * B[13] * B[22] * B[29] * B[33] -
            B[6] * B[13] * B[23] * B[27] * B[34] + B[6] * B[13] * B[23] * B[28] * B[33] +
            B[6] * B[15] * B[19] * B[28] * B[35] - B[6] * B[15] * B[19] * B[29] * B[34] -
            B[6] * B[15] * B[22] * B[25] * B[35] + B[6] * B[15] * B[22] * B[29] * B[31] +
            B[6] * B[15] * B[23] * B[25] * B[34] - B[6] * B[15] * B[23] * B[28] * B[31] -
            B[6] * B[16] * B[19] * B[27] * B[35] + B[6] * B[16] * B[19] * B[29] * B[33] +
            B[6] * B[16] * B[21] * B[25] * B[35] - B[6] * B[16] * B[21] * B[29] * B[31] -
            B[6] * B[16] * B[23] * B[25] * B[33] + B[6] * B[16] * B[23] * B[27] * B[31] +
            B[6] * B[17] * B[19] * B[27] * B[34] - B[6] * B[17] * B[19] * B[28] * B[33] -
            B[6] * B[17] * B[21] * B[25] * B[34] + B[6] * B[17] * B[21] * B[28] * B[31] +
            B[6] * B[17] * B[22] * B[25] * B[33] - B[6] * B[17] * B[22] * B[27] * B[31] +
            B[7] * B[12] * B[21] * B[28] * B[35] - B[7] * B[12] * B[21] * B[29] * B[34] -
            B[7] * B[12] * B[22] * B[27] * B[35] + B[7] * B[12] * B[22] * B[29] * B[33] +
            B[7] * B[12] * B[23] * B[27] * B[34] - B[7] * B[12] * B[23] * B[28] * B[33] -
            B[7] * B[15] * B[18] * B[28] * B[35] + B[7] * B[15] * B[18] * B[29] * B[34] +
            B[7] * B[15] * B[22] * B[24] * B[35] - B[7] * B[15] * B[22] * B[29] * B[30] -
            B[7] * B[15] * B[23] * B[24] * B[34] + B[7] * B[15] * B[23] * B[28] * B[30] +
            B[7] * B[16] * B[18] * B[27] * B[35] - B[7] * B[16] * B[18] * B[29] * B[33] -
            B[7] * B[16] * B[21] * B[24] * B[35] + B[7] * B[16] * B[21] * B[29] * B[30] +
            B[7] * B[16] * B[23] * B[24] * B[33] - B[7] * B[16] * B[23] * B[27] * B[30] -
            B[7] * B[17] * B[18] * B[27] * B[34] + B[7] * B[17] * B[18] * B[28] * B[33] +
            B[7] * B[17] * B[21] * B[24] * B[34] - B[7] * B[17] * B[21] * B[28] * B[30] -
            B[7] * B[17] * B[22] * B[24] * B[33] + B[7] * B[17] * B[22] * B[27] * B[30] -
            B[9] * B[12] * B[19] * B[28] * B[35] + B[9] * B[12] * B[19] * B[29] * B[34] +
            B[9] * B[12] * B[22] * B[25] * B[35] - B[9] * B[12] * B[22] * B[29] * B[31] -
            B[9] * B[12] * B[23] * B[25] * B[34] + B[9] * B[12] * B[23] * B[28] * B[31] +
            B[9] * B[13] * B[18] * B[28] * B[35] - B[9] * B[13] * B[18] * B[29] * B[34] -
            B[9] * B[13] * B[22] * B[24] * B[35] + B[9] * B[13] * B[22] * B[29] * B[30] +
            B[9] * B[13] * B[23] * B[24] * B[34] - B[9] * B[13] * B[23] * B[28] * B[30] -
            B[9] * B[16] * B[18] * B[25] * B[35] + B[9] * B[16] * B[18] * B[29] * B[31] +
            B[9] * B[16] * B[19] * B[24] * B[35] - B[9] * B[16] * B[19] * B[29] * B[30] -
            B[9] * B[16] * B[23] * B[24] * B[31] + B[9] * B[16] * B[23] * B[25] * B[30] +
            B[9] * B[17] * B[18] * B[25] * B[34] - B[9] * B[17] * B[18] * B[28] * B[31] -
            B[9] * B[17] * B[19] * B[24] * B[34] + B[9] * B[17] * B[19] * B[28] * B[30] +
            B[9] * B[17] * B[22] * B[24] * B[31] - B[9] * B[17] * B[22] * B[25] * B[30] +
            B[10] * B[12] * B[19] * B[27] * B[35] - B[10] * B[12] * B[19] * B[29] * B[33] -
            B[10] * B[12] * B[21] * B[25] * B[35] + B[10] * B[12] * B[21] * B[29] * B[31] +
            B[10] * B[12] * B[23] * B[25] * B[33] - B[10] * B[12] * B[23] * B[27] * B[31] -
            B[10] * B[13] * B[18] * B[27] * B[35] + B[10] * B[13] * B[18] * B[29] * B[33] +
            B[10] * B[13] * B[21] * B[24] * B[35] - B[10] * B[13] * B[21] * B[29] * B[30] -
            B[10] * B[13] * B[23] * B[24] * B[33] + B[10] * B[13] * B[23] * B[27] * B[30] +
            B[10] * B[15] * B[18] * B[25] * B[35] - B[10] * B[15] * B[18] * B[29] * B[31] -
            B[10] * B[15] * B[19] * B[24] * B[35] + B[10] * B[15] * B[19] * B[29] * B[30] +
            B[10] * B[15] * B[23] * B[24] * B[31] - B[10] * B[15] * B[23] * B[25] * B[30] -
            B[10] * B[17] * B[18] * B[25] * B[33] + B[10] * B[17] * B[18] * B[27] * B[31] +
            B[10] * B[17] * B[19] * B[24] * B[33] - B[10] * B[17] * B[19] * B[27] * B[30] -
            B[10] * B[17] * B[21] * B[24] * B[31] + B[10] * B[17] * B[21] * B[25] * B[30] -
            B[11] * B[12] * B[19] * B[27] * B[34] + B[11] * B[12] * B[19] * B[28] * B[33] +
            B[11] * B[12] * B[21] * B[25] * B[34] - B[11] * B[12] * B[21] * B[28] * B[31] -
            B[11] * B[12] * B[22] * B[25] * B[33] + B[11] * B[12] * B[22] * B[27] * B[31] +
            B[11] * B[13] * B[18] * B[27] * B[34] - B[11] * B[13] * B[18] * B[28] * B[33] -
            B[11] * B[13] * B[21] * B[24] * B[34] + B[11] * B[13] * B[21] * B[28] * B[30] +
            B[11] * B[13] * B[22] * B[24] * B[33] - B[11] * B[13] * B[22] * B[27] * B[30] -
            B[11] * B[15] * B[18] * B[25] * B[34] + B[11] * B[15] * B[18] * B[28] * B[31] +
            B[11] * B[15] * B[19] * B[24] * B[34] - B[11] * B[15] * B[19] * B[28] * B[30] -
            B[11] * B[15] * B[22] * B[24] * B[31] + B[11] * B[15] * B[22] * B[25] * B[30] +
            B[11] * B[16] * B[18] * B[25] * B[33] - B[11] * B[16] * B[18] * B[27] * B[31] -
            B[11] * B[16] * B[19] * B[24] * B[33] + B[11] * B[16] * B[19] * B[27] * B[30] +
            B[11] * B[16] * B[21] * B[24] * B[31] - B[11] * B[16] * B[21] * B[25] * B[30]) /
           (DET);

    L[3] = -(B[0] * B[7] * B[14] * B[22] * B[29] - B[0] * B[7] * B[14] * B[23] * B[28] -
             B[0] * B[7] * B[16] * B[20] * B[29] + B[0] * B[7] * B[16] * B[23] * B[26] +
             B[0] * B[7] * B[17] * B[20] * B[28] - B[0] * B[7] * B[17] * B[22] * B[26] -
             B[0] * B[8] * B[13] * B[22] * B[29] + B[0] * B[8] * B[13] * B[23] * B[28] +
             B[0] * B[8] * B[16] * B[19] * B[29] - B[0] * B[8] * B[16] * B[23] * B[25] -
             B[0] * B[8] * B[17] * B[19] * B[28] + B[0] * B[8] * B[17] * B[22] * B[25] +
             B[0] * B[10] * B[13] * B[20] * B[29] - B[0] * B[10] * B[13] * B[23] * B[26] -
             B[0] * B[10] * B[14] * B[19] * B[29] + B[0] * B[10] * B[14] * B[23] * B[25] +
             B[0] * B[10] * B[17] * B[19] * B[26] - B[0] * B[10] * B[17] * B[20] * B[25] -
             B[0] * B[11] * B[13] * B[20] * B[28] + B[0] * B[11] * B[13] * B[22] * B[26] +
             B[0] * B[11] * B[14] * B[19] * B[28] - B[0] * B[11] * B[14] * B[22] * B[25] -
             B[0] * B[11] * B[16] * B[19] * B[26] + B[0] * B[11] * B[16] * B[20] * B[25] -
             B[1] * B[6] * B[14] * B[22] * B[29] + B[1] * B[6] * B[14] * B[23] * B[28] +
             B[1] * B[6] * B[16] * B[20] * B[29] - B[1] * B[6] * B[16] * B[23] * B[26] -
             B[1] * B[6] * B[17] * B[20] * B[28] + B[1] * B[6] * B[17] * B[22] * B[26] +
             B[1] * B[8] * B[12] * B[22] * B[29] - B[1] * B[8] * B[12] * B[23] * B[28] -
             B[1] * B[8] * B[16] * B[18] * B[29] + B[1] * B[8] * B[16] * B[23] * B[24] +
             B[1] * B[8] * B[17] * B[18] * B[28] - B[1] * B[8] * B[17] * B[22] * B[24] -
             B[1] * B[10] * B[12] * B[20] * B[29] + B[1] * B[10] * B[12] * B[23] * B[26] +
             B[1] * B[10] * B[14] * B[18] * B[29] - B[1] * B[10] * B[14] * B[23] * B[24] -
             B[1] * B[10] * B[17] * B[18] * B[26] + B[1] * B[10] * B[17] * B[20] * B[24] +
             B[1] * B[11] * B[12] * B[20] * B[28] - B[1] * B[11] * B[12] * B[22] * B[26] -
             B[1] * B[11] * B[14] * B[18] * B[28] + B[1] * B[11] * B[14] * B[22] * B[24] +
             B[1] * B[11] * B[16] * B[18] * B[26] - B[1] * B[11] * B[16] * B[20] * B[24] +
             B[2] * B[6] * B[13] * B[22] * B[29] - B[2] * B[6] * B[13] * B[23] * B[28] -
             B[2] * B[6] * B[16] * B[19] * B[29] + B[2] * B[6] * B[16] * B[23] * B[25] +
             B[2] * B[6] * B[17] * B[19] * B[28] - B[2] * B[6] * B[17] * B[22] * B[25] -
             B[2] * B[7] * B[12] * B[22] * B[29] + B[2] * B[7] * B[12] * B[23] * B[28] +
             B[2] * B[7] * B[16] * B[18] * B[29] - B[2] * B[7] * B[16] * B[23] * B[24] -
             B[2] * B[7] * B[17] * B[18] * B[28] + B[2] * B[7] * B[17] * B[22] * B[24] +
             B[2] * B[10] * B[12] * B[19] * B[29] - B[2] * B[10] * B[12] * B[23] * B[25] -
             B[2] * B[10] * B[13] * B[18] * B[29] + B[2] * B[10] * B[13] * B[23] * B[24] +
             B[2] * B[10] * B[17] * B[18] * B[25] - B[2] * B[10] * B[17] * B[19] * B[24] -
             B[2] * B[11] * B[12] * B[19] * B[28] + B[2] * B[11] * B[12] * B[22] * B[25] +
             B[2] * B[11] * B[13] * B[18] * B[28] - B[2] * B[11] * B[13] * B[22] * B[24] -
             B[2] * B[11] * B[16] * B[18] * B[25] + B[2] * B[11] * B[16] * B[19] * B[24] -
             B[4] * B[6] * B[13] * B[20] * B[29] + B[4] * B[6] * B[13] * B[23] * B[26] +
             B[4] * B[6] * B[14] * B[19] * B[29] - B[4] * B[6] * B[14] * B[23] * B[25] -
             B[4] * B[6] * B[17] * B[19] * B[26] + B[4] * B[6] * B[17] * B[20] * B[25] +
             B[4] * B[7] * B[12] * B[20] * B[29] - B[4] * B[7] * B[12] * B[23] * B[26] -
             B[4] * B[7] * B[14] * B[18] * B[29] + B[4] * B[7] * B[14] * B[23] * B[24] +
             B[4] * B[7] * B[17] * B[18] * B[26] - B[4] * B[7] * B[17] * B[20] * B[24] -
             B[4] * B[8] * B[12] * B[19] * B[29] + B[4] * B[8] * B[12] * B[23] * B[25] +
             B[4] * B[8] * B[13] * B[18] * B[29] - B[4] * B[8] * B[13] * B[23] * B[24] -
             B[4] * B[8] * B[17] * B[18] * B[25] + B[4] * B[8] * B[17] * B[19] * B[24] +
             B[4] * B[11] * B[12] * B[19] * B[26] - B[4] * B[11] * B[12] * B[20] * B[25] -
             B[4] * B[11] * B[13] * B[18] * B[26] + B[4] * B[11] * B[13] * B[20] * B[24] +
             B[4] * B[11] * B[14] * B[18] * B[25] - B[4] * B[11] * B[14] * B[19] * B[24] +
             B[5] * B[6] * B[13] * B[20] * B[28] - B[5] * B[6] * B[13] * B[22] * B[26] -
             B[5] * B[6] * B[14] * B[19] * B[28] + B[5] * B[6] * B[14] * B[22] * B[25] +
             B[5] * B[6] * B[16] * B[19] * B[26] - B[5] * B[6] * B[16] * B[20] * B[25] -
             B[5] * B[7] * B[12] * B[20] * B[28] + B[5] * B[7] * B[12] * B[22] * B[26] +
             B[5] * B[7] * B[14] * B[18] * B[28] - B[5] * B[7] * B[14] * B[22] * B[24] -
             B[5] * B[7] * B[16] * B[18] * B[26] + B[5] * B[7] * B[16] * B[20] * B[24] +
             B[5] * B[8] * B[12] * B[19] * B[28] - B[5] * B[8] * B[12] * B[22] * B[25] -
             B[5] * B[8] * B[13] * B[18] * B[28] + B[5] * B[8] * B[13] * B[22] * B[24] +
             B[5] * B[8] * B[16] * B[18] * B[25] - B[5] * B[8] * B[16] * B[19] * B[24] -
             B[5] * B[10] * B[12] * B[19] * B[26] + B[5] * B[10] * B[12] * B[20] * B[25] +
             B[5] * B[10] * B[13] * B[18] * B[26] - B[5] * B[10] * B[13] * B[20] * B[24] -
             B[5] * B[10] * B[14] * B[18] * B[25] + B[5] * B[10] * B[14] * B[19] * B[24] +
             B[0] * B[7] * B[14] * B[28] * B[35] - B[0] * B[7] * B[14] * B[29] * B[34] -
             B[0] * B[7] * B[16] * B[26] * B[35] + B[0] * B[7] * B[16] * B[29] * B[32] +
             B[0] * B[7] * B[17] * B[26] * B[34] - B[0] * B[7] * B[17] * B[28] * B[32] -
             B[0] * B[8] * B[13] * B[28] * B[35] + B[0] * B[8] * B[13] * B[29] * B[34] +
             B[0] * B[8] * B[16] * B[25] * B[35] - B[0] * B[8] * B[16] * B[29] * B[31] -
             B[0] * B[8] * B[17] * B[25] * B[34] + B[0] * B[8] * B[17] * B[28] * B[31] +
             B[0] * B[10] * B[13] * B[26] * B[35] - B[0] * B[10] * B[13] * B[29] * B[32] -
             B[0] * B[10] * B[14] * B[25] * B[35] + B[0] * B[10] * B[14] * B[29] * B[31] +
             B[0] * B[10] * B[17] * B[25] * B[32] - B[0] * B[10] * B[17] * B[26] * B[31] -
             B[0] * B[11] * B[13] * B[26] * B[34] + B[0] * B[11] * B[13] * B[28] * B[32] +
             B[0] * B[11] * B[14] * B[25] * B[34] - B[0] * B[11] * B[14] * B[28] * B[31] -
             B[0] * B[11] * B[16] * B[25] * B[32] + B[0] * B[11] * B[16] * B[26] * B[31] -
             B[1] * B[6] * B[14] * B[28] * B[35] + B[1] * B[6] * B[14] * B[29] * B[34] +
             B[1] * B[6] * B[16] * B[26] * B[35] - B[1] * B[6] * B[16] * B[29] * B[32] -
             B[1] * B[6] * B[17] * B[26] * B[34] + B[1] * B[6] * B[17] * B[28] * B[32] +
             B[1] * B[8] * B[12] * B[28] * B[35] - B[1] * B[8] * B[12] * B[29] * B[34] -
             B[1] * B[8] * B[16] * B[24] * B[35] + B[1] * B[8] * B[16] * B[29] * B[30] +
             B[1] * B[8] * B[17] * B[24] * B[34] - B[1] * B[8] * B[17] * B[28] * B[30] -
             B[1] * B[10] * B[12] * B[26] * B[35] + B[1] * B[10] * B[12] * B[29] * B[32] +
             B[1] * B[10] * B[14] * B[24] * B[35] - B[1] * B[10] * B[14] * B[29] * B[30] -
             B[1] * B[10] * B[17] * B[24] * B[32] + B[1] * B[10] * B[17] * B[26] * B[30] +
             B[1] * B[11] * B[12] * B[26] * B[34] - B[1] * B[11] * B[12] * B[28] * B[32] -
             B[1] * B[11] * B[14] * B[24] * B[34] + B[1] * B[11] * B[14] * B[28] * B[30] +
             B[1] * B[11] * B[16] * B[24] * B[32] - B[1] * B[11] * B[16] * B[26] * B[30] +
             B[2] * B[6] * B[13] * B[28] * B[35] - B[2] * B[6] * B[13] * B[29] * B[34] -
             B[2] * B[6] * B[16] * B[25] * B[35] + B[2] * B[6] * B[16] * B[29] * B[31] +
             B[2] * B[6] * B[17] * B[25] * B[34] - B[2] * B[6] * B[17] * B[28] * B[31] -
             B[2] * B[7] * B[12] * B[28] * B[35] + B[2] * B[7] * B[12] * B[29] * B[34] +
             B[2] * B[7] * B[16] * B[24] * B[35] - B[2] * B[7] * B[16] * B[29] * B[30] -
             B[2] * B[7] * B[17] * B[24] * B[34] + B[2] * B[7] * B[17] * B[28] * B[30] +
             B[2] * B[10] * B[12] * B[25] * B[35] - B[2] * B[10] * B[12] * B[29] * B[31] -
             B[2] * B[10] * B[13] * B[24] * B[35] + B[2] * B[10] * B[13] * B[29] * B[30] +
             B[2] * B[10] * B[17] * B[24] * B[31] - B[2] * B[10] * B[17] * B[25] * B[30] -
             B[2] * B[11] * B[12] * B[25] * B[34] + B[2] * B[11] * B[12] * B[28] * B[31] +
             B[2] * B[11] * B[13] * B[24] * B[34] - B[2] * B[11] * B[13] * B[28] * B[30] -
             B[2] * B[11] * B[16] * B[24] * B[31] + B[2] * B[11] * B[16] * B[25] * B[30] -
             B[4] * B[6] * B[13] * B[26] * B[35] + B[4] * B[6] * B[13] * B[29] * B[32] +
             B[4] * B[6] * B[14] * B[25] * B[35] - B[4] * B[6] * B[14] * B[29] * B[31] -
             B[4] * B[6] * B[17] * B[25] * B[32] + B[4] * B[6] * B[17] * B[26] * B[31] +
             B[4] * B[7] * B[12] * B[26] * B[35] - B[4] * B[7] * B[12] * B[29] * B[32] -
             B[4] * B[7] * B[14] * B[24] * B[35] + B[4] * B[7] * B[14] * B[29] * B[30] +
             B[4] * B[7] * B[17] * B[24] * B[32] - B[4] * B[7] * B[17] * B[26] * B[30] -
             B[4] * B[8] * B[12] * B[25] * B[35] + B[4] * B[8] * B[12] * B[29] * B[31] +
             B[4] * B[8] * B[13] * B[24] * B[35] - B[4] * B[8] * B[13] * B[29] * B[30] -
             B[4] * B[8] * B[17] * B[24] * B[31] + B[4] * B[8] * B[17] * B[25] * B[30] +
             B[4] * B[11] * B[12] * B[25] * B[32] - B[4] * B[11] * B[12] * B[26] * B[31] -
             B[4] * B[11] * B[13] * B[24] * B[32] + B[4] * B[11] * B[13] * B[26] * B[30] +
             B[4] * B[11] * B[14] * B[24] * B[31] - B[4] * B[11] * B[14] * B[25] * B[30] +
             B[5] * B[6] * B[13] * B[26] * B[34] - B[5] * B[6] * B[13] * B[28] * B[32] -
             B[5] * B[6] * B[14] * B[25] * B[34] + B[5] * B[6] * B[14] * B[28] * B[31] +
             B[5] * B[6] * B[16] * B[25] * B[32] - B[5] * B[6] * B[16] * B[26] * B[31] -
             B[5] * B[7] * B[12] * B[26] * B[34] + B[5] * B[7] * B[12] * B[28] * B[32] +
             B[5] * B[7] * B[14] * B[24] * B[34] - B[5] * B[7] * B[14] * B[28] * B[30] -
             B[5] * B[7] * B[16] * B[24] * B[32] + B[5] * B[7] * B[16] * B[26] * B[30] +
             B[5] * B[8] * B[12] * B[25] * B[34] - B[5] * B[8] * B[12] * B[28] * B[31] -
             B[5] * B[8] * B[13] * B[24] * B[34] + B[5] * B[8] * B[13] * B[28] * B[30] +
             B[5] * B[8] * B[16] * B[24] * B[31] - B[5] * B[8] * B[16] * B[25] * B[30] -
             B[5] * B[10] * B[12] * B[25] * B[32] + B[5] * B[10] * B[12] * B[26] * B[31] +
             B[5] * B[10] * B[13] * B[24] * B[32] - B[5] * B[10] * B[13] * B[26] * B[30] -
             B[5] * B[10] * B[14] * B[24] * B[31] + B[5] * B[10] * B[14] * B[25] * B[30] -
             B[6] * B[13] * B[20] * B[28] * B[35] + B[6] * B[13] * B[20] * B[29] * B[34] +
             B[6] * B[13] * B[22] * B[26] * B[35] - B[6] * B[13] * B[22] * B[29] * B[32] -
             B[6] * B[13] * B[23] * B[26] * B[34] + B[6] * B[13] * B[23] * B[28] * B[32] +
             B[6] * B[14] * B[19] * B[28] * B[35] - B[6] * B[14] * B[19] * B[29] * B[34] -
             B[6] * B[14] * B[22] * B[25] * B[35] + B[6] * B[14] * B[22] * B[29] * B[31] +
             B[6] * B[14] * B[23] * B[25] * B[34] - B[6] * B[14] * B[23] * B[28] * B[31] -
             B[6] * B[16] * B[19] * B[26] * B[35] + B[6] * B[16] * B[19] * B[29] * B[32] +
             B[6] * B[16] * B[20] * B[25] * B[35] - B[6] * B[16] * B[20] * B[29] * B[31] -
             B[6] * B[16] * B[23] * B[25] * B[32] + B[6] * B[16] * B[23] * B[26] * B[31] +
             B[6] * B[17] * B[19] * B[26] * B[34] - B[6] * B[17] * B[19] * B[28] * B[32] -
             B[6] * B[17] * B[20] * B[25] * B[34] + B[6] * B[17] * B[20] * B[28] * B[31] +
             B[6] * B[17] * B[22] * B[25] * B[32] - B[6] * B[17] * B[22] * B[26] * B[31] +
             B[7] * B[12] * B[20] * B[28] * B[35] - B[7] * B[12] * B[20] * B[29] * B[34] -
             B[7] * B[12] * B[22] * B[26] * B[35] + B[7] * B[12] * B[22] * B[29] * B[32] +
             B[7] * B[12] * B[23] * B[26] * B[34] - B[7] * B[12] * B[23] * B[28] * B[32] -
             B[7] * B[14] * B[18] * B[28] * B[35] + B[7] * B[14] * B[18] * B[29] * B[34] +
             B[7] * B[14] * B[22] * B[24] * B[35] - B[7] * B[14] * B[22] * B[29] * B[30] -
             B[7] * B[14] * B[23] * B[24] * B[34] + B[7] * B[14] * B[23] * B[28] * B[30] +
             B[7] * B[16] * B[18] * B[26] * B[35] - B[7] * B[16] * B[18] * B[29] * B[32] -
             B[7] * B[16] * B[20] * B[24] * B[35] + B[7] * B[16] * B[20] * B[29] * B[30] +
             B[7] * B[16] * B[23] * B[24] * B[32] - B[7] * B[16] * B[23] * B[26] * B[30] -
             B[7] * B[17] * B[18] * B[26] * B[34] + B[7] * B[17] * B[18] * B[28] * B[32] +
             B[7] * B[17] * B[20] * B[24] * B[34] - B[7] * B[17] * B[20] * B[28] * B[30] -
             B[7] * B[17] * B[22] * B[24] * B[32] + B[7] * B[17] * B[22] * B[26] * B[30] -
             B[8] * B[12] * B[19] * B[28] * B[35] + B[8] * B[12] * B[19] * B[29] * B[34] +
             B[8] * B[12] * B[22] * B[25] * B[35] - B[8] * B[12] * B[22] * B[29] * B[31] -
             B[8] * B[12] * B[23] * B[25] * B[34] + B[8] * B[12] * B[23] * B[28] * B[31] +
             B[8] * B[13] * B[18] * B[28] * B[35] - B[8] * B[13] * B[18] * B[29] * B[34] -
             B[8] * B[13] * B[22] * B[24] * B[35] + B[8] * B[13] * B[22] * B[29] * B[30] +
             B[8] * B[13] * B[23] * B[24] * B[34] - B[8] * B[13] * B[23] * B[28] * B[30] -
             B[8] * B[16] * B[18] * B[25] * B[35] + B[8] * B[16] * B[18] * B[29] * B[31] +
             B[8] * B[16] * B[19] * B[24] * B[35] - B[8] * B[16] * B[19] * B[29] * B[30] -
             B[8] * B[16] * B[23] * B[24] * B[31] + B[8] * B[16] * B[23] * B[25] * B[30] +
             B[8] * B[17] * B[18] * B[25] * B[34] - B[8] * B[17] * B[18] * B[28] * B[31] -
             B[8] * B[17] * B[19] * B[24] * B[34] + B[8] * B[17] * B[19] * B[28] * B[30] +
             B[8] * B[17] * B[22] * B[24] * B[31] - B[8] * B[17] * B[22] * B[25] * B[30] +
             B[10] * B[12] * B[19] * B[26] * B[35] - B[10] * B[12] * B[19] * B[29] * B[32] -
             B[10] * B[12] * B[20] * B[25] * B[35] + B[10] * B[12] * B[20] * B[29] * B[31] +
             B[10] * B[12] * B[23] * B[25] * B[32] - B[10] * B[12] * B[23] * B[26] * B[31] -
             B[10] * B[13] * B[18] * B[26] * B[35] + B[10] * B[13] * B[18] * B[29] * B[32] +
             B[10] * B[13] * B[20] * B[24] * B[35] - B[10] * B[13] * B[20] * B[29] * B[30] -
             B[10] * B[13] * B[23] * B[24] * B[32] + B[10] * B[13] * B[23] * B[26] * B[30] +
             B[10] * B[14] * B[18] * B[25] * B[35] - B[10] * B[14] * B[18] * B[29] * B[31] -
             B[10] * B[14] * B[19] * B[24] * B[35] + B[10] * B[14] * B[19] * B[29] * B[30] +
             B[10] * B[14] * B[23] * B[24] * B[31] - B[10] * B[14] * B[23] * B[25] * B[30] -
             B[10] * B[17] * B[18] * B[25] * B[32] + B[10] * B[17] * B[18] * B[26] * B[31] +
             B[10] * B[17] * B[19] * B[24] * B[32] - B[10] * B[17] * B[19] * B[26] * B[30] -
             B[10] * B[17] * B[20] * B[24] * B[31] + B[10] * B[17] * B[20] * B[25] * B[30] -
             B[11] * B[12] * B[19] * B[26] * B[34] + B[11] * B[12] * B[19] * B[28] * B[32] +
             B[11] * B[12] * B[20] * B[25] * B[34] - B[11] * B[12] * B[20] * B[28] * B[31] -
             B[11] * B[12] * B[22] * B[25] * B[32] + B[11] * B[12] * B[22] * B[26] * B[31] +
             B[11] * B[13] * B[18] * B[26] * B[34] - B[11] * B[13] * B[18] * B[28] * B[32] -
             B[11] * B[13] * B[20] * B[24] * B[34] + B[11] * B[13] * B[20] * B[28] * B[30] +
             B[11] * B[13] * B[22] * B[24] * B[32] - B[11] * B[13] * B[22] * B[26] * B[30] -
             B[11] * B[14] * B[18] * B[25] * B[34] + B[11] * B[14] * B[18] * B[28] * B[31] +
             B[11] * B[14] * B[19] * B[24] * B[34] - B[11] * B[14] * B[19] * B[28] * B[30] -
             B[11] * B[14] * B[22] * B[24] * B[31] + B[11] * B[14] * B[22] * B[25] * B[30] +
             B[11] * B[16] * B[18] * B[25] * B[32] - B[11] * B[16] * B[18] * B[26] * B[31] -
             B[11] * B[16] * B[19] * B[24] * B[32] + B[11] * B[16] * B[19] * B[26] * B[30] +
             B[11] * B[16] * B[20] * B[24] * B[31] - B[11] * B[16] * B[20] * B[25] * B[30]) /
           (DET);

    L[4] = (B[0] * B[7] * B[14] * B[21] * B[29] - B[0] * B[7] * B[14] * B[23] * B[27] -
            B[0] * B[7] * B[15] * B[20] * B[29] + B[0] * B[7] * B[15] * B[23] * B[26] +
            B[0] * B[7] * B[17] * B[20] * B[27] - B[0] * B[7] * B[17] * B[21] * B[26] -
            B[0] * B[8] * B[13] * B[21] * B[29] + B[0] * B[8] * B[13] * B[23] * B[27] +
            B[0] * B[8] * B[15] * B[19] * B[29] - B[0] * B[8] * B[15] * B[23] * B[25] -
            B[0] * B[8] * B[17] * B[19] * B[27] + B[0] * B[8] * B[17] * B[21] * B[25] +
            B[0] * B[9] * B[13] * B[20] * B[29] - B[0] * B[9] * B[13] * B[23] * B[26] -
            B[0] * B[9] * B[14] * B[19] * B[29] + B[0] * B[9] * B[14] * B[23] * B[25] +
            B[0] * B[9] * B[17] * B[19] * B[26] - B[0] * B[9] * B[17] * B[20] * B[25] -
            B[0] * B[11] * B[13] * B[20] * B[27] + B[0] * B[11] * B[13] * B[21] * B[26] +
            B[0] * B[11] * B[14] * B[19] * B[27] - B[0] * B[11] * B[14] * B[21] * B[25] -
            B[0] * B[11] * B[15] * B[19] * B[26] + B[0] * B[11] * B[15] * B[20] * B[25] -
            B[1] * B[6] * B[14] * B[21] * B[29] + B[1] * B[6] * B[14] * B[23] * B[27] +
            B[1] * B[6] * B[15] * B[20] * B[29] - B[1] * B[6] * B[15] * B[23] * B[26] -
            B[1] * B[6] * B[17] * B[20] * B[27] + B[1] * B[6] * B[17] * B[21] * B[26] +
            B[1] * B[8] * B[12] * B[21] * B[29] - B[1] * B[8] * B[12] * B[23] * B[27] -
            B[1] * B[8] * B[15] * B[18] * B[29] + B[1] * B[8] * B[15] * B[23] * B[24] +
            B[1] * B[8] * B[17] * B[18] * B[27] - B[1] * B[8] * B[17] * B[21] * B[24] -
            B[1] * B[9] * B[12] * B[20] * B[29] + B[1] * B[9] * B[12] * B[23] * B[26] +
            B[1] * B[9] * B[14] * B[18] * B[29] - B[1] * B[9] * B[14] * B[23] * B[24] -
            B[1] * B[9] * B[17] * B[18] * B[26] + B[1] * B[9] * B[17] * B[20] * B[24] +
            B[1] * B[11] * B[12] * B[20] * B[27] - B[1] * B[11] * B[12] * B[21] * B[26] -
            B[1] * B[11] * B[14] * B[18] * B[27] + B[1] * B[11] * B[14] * B[21] * B[24] +
            B[1] * B[11] * B[15] * B[18] * B[26] - B[1] * B[11] * B[15] * B[20] * B[24] +
            B[2] * B[6] * B[13] * B[21] * B[29] - B[2] * B[6] * B[13] * B[23] * B[27] -
            B[2] * B[6] * B[15] * B[19] * B[29] + B[2] * B[6] * B[15] * B[23] * B[25] +
            B[2] * B[6] * B[17] * B[19] * B[27] - B[2] * B[6] * B[17] * B[21] * B[25] -
            B[2] * B[7] * B[12] * B[21] * B[29] + B[2] * B[7] * B[12] * B[23] * B[27] +
            B[2] * B[7] * B[15] * B[18] * B[29] - B[2] * B[7] * B[15] * B[23] * B[24] -
            B[2] * B[7] * B[17] * B[18] * B[27] + B[2] * B[7] * B[17] * B[21] * B[24] +
            B[2] * B[9] * B[12] * B[19] * B[29] - B[2] * B[9] * B[12] * B[23] * B[25] -
            B[2] * B[9] * B[13] * B[18] * B[29] + B[2] * B[9] * B[13] * B[23] * B[24] +
            B[2] * B[9] * B[17] * B[18] * B[25] - B[2] * B[9] * B[17] * B[19] * B[24] -
            B[2] * B[11] * B[12] * B[19] * B[27] + B[2] * B[11] * B[12] * B[21] * B[25] +
            B[2] * B[11] * B[13] * B[18] * B[27] - B[2] * B[11] * B[13] * B[21] * B[24] -
            B[2] * B[11] * B[15] * B[18] * B[25] + B[2] * B[11] * B[15] * B[19] * B[24] -
            B[3] * B[6] * B[13] * B[20] * B[29] + B[3] * B[6] * B[13] * B[23] * B[26] +
            B[3] * B[6] * B[14] * B[19] * B[29] - B[3] * B[6] * B[14] * B[23] * B[25] -
            B[3] * B[6] * B[17] * B[19] * B[26] + B[3] * B[6] * B[17] * B[20] * B[25] +
            B[3] * B[7] * B[12] * B[20] * B[29] - B[3] * B[7] * B[12] * B[23] * B[26] -
            B[3] * B[7] * B[14] * B[18] * B[29] + B[3] * B[7] * B[14] * B[23] * B[24] +
            B[3] * B[7] * B[17] * B[18] * B[26] - B[3] * B[7] * B[17] * B[20] * B[24] -
            B[3] * B[8] * B[12] * B[19] * B[29] + B[3] * B[8] * B[12] * B[23] * B[25] +
            B[3] * B[8] * B[13] * B[18] * B[29] - B[3] * B[8] * B[13] * B[23] * B[24] -
            B[3] * B[8] * B[17] * B[18] * B[25] + B[3] * B[8] * B[17] * B[19] * B[24] +
            B[3] * B[11] * B[12] * B[19] * B[26] - B[3] * B[11] * B[12] * B[20] * B[25] -
            B[3] * B[11] * B[13] * B[18] * B[26] + B[3] * B[11] * B[13] * B[20] * B[24] +
            B[3] * B[11] * B[14] * B[18] * B[25] - B[3] * B[11] * B[14] * B[19] * B[24] +
            B[5] * B[6] * B[13] * B[20] * B[27] - B[5] * B[6] * B[13] * B[21] * B[26] -
            B[5] * B[6] * B[14] * B[19] * B[27] + B[5] * B[6] * B[14] * B[21] * B[25] +
            B[5] * B[6] * B[15] * B[19] * B[26] - B[5] * B[6] * B[15] * B[20] * B[25] -
            B[5] * B[7] * B[12] * B[20] * B[27] + B[5] * B[7] * B[12] * B[21] * B[26] +
            B[5] * B[7] * B[14] * B[18] * B[27] - B[5] * B[7] * B[14] * B[21] * B[24] -
            B[5] * B[7] * B[15] * B[18] * B[26] + B[5] * B[7] * B[15] * B[20] * B[24] +
            B[5] * B[8] * B[12] * B[19] * B[27] - B[5] * B[8] * B[12] * B[21] * B[25] -
            B[5] * B[8] * B[13] * B[18] * B[27] + B[5] * B[8] * B[13] * B[21] * B[24] +
            B[5] * B[8] * B[15] * B[18] * B[25] - B[5] * B[8] * B[15] * B[19] * B[24] -
            B[5] * B[9] * B[12] * B[19] * B[26] + B[5] * B[9] * B[12] * B[20] * B[25] +
            B[5] * B[9] * B[13] * B[18] * B[26] - B[5] * B[9] * B[13] * B[20] * B[24] -
            B[5] * B[9] * B[14] * B[18] * B[25] + B[5] * B[9] * B[14] * B[19] * B[24] +
            B[0] * B[7] * B[14] * B[27] * B[35] - B[0] * B[7] * B[14] * B[29] * B[33] -
            B[0] * B[7] * B[15] * B[26] * B[35] + B[0] * B[7] * B[15] * B[29] * B[32] +
            B[0] * B[7] * B[17] * B[26] * B[33] - B[0] * B[7] * B[17] * B[27] * B[32] -
            B[0] * B[8] * B[13] * B[27] * B[35] + B[0] * B[8] * B[13] * B[29] * B[33] +
            B[0] * B[8] * B[15] * B[25] * B[35] - B[0] * B[8] * B[15] * B[29] * B[31] -
            B[0] * B[8] * B[17] * B[25] * B[33] + B[0] * B[8] * B[17] * B[27] * B[31] +
            B[0] * B[9] * B[13] * B[26] * B[35] - B[0] * B[9] * B[13] * B[29] * B[32] -
            B[0] * B[9] * B[14] * B[25] * B[35] + B[0] * B[9] * B[14] * B[29] * B[31] +
            B[0] * B[9] * B[17] * B[25] * B[32] - B[0] * B[9] * B[17] * B[26] * B[31] -
            B[0] * B[11] * B[13] * B[26] * B[33] + B[0] * B[11] * B[13] * B[27] * B[32] +
            B[0] * B[11] * B[14] * B[25] * B[33] - B[0] * B[11] * B[14] * B[27] * B[31] -
            B[0] * B[11] * B[15] * B[25] * B[32] + B[0] * B[11] * B[15] * B[26] * B[31] -
            B[1] * B[6] * B[14] * B[27] * B[35] + B[1] * B[6] * B[14] * B[29] * B[33] +
            B[1] * B[6] * B[15] * B[26] * B[35] - B[1] * B[6] * B[15] * B[29] * B[32] -
            B[1] * B[6] * B[17] * B[26] * B[33] + B[1] * B[6] * B[17] * B[27] * B[32] +
            B[1] * B[8] * B[12] * B[27] * B[35] - B[1] * B[8] * B[12] * B[29] * B[33] -
            B[1] * B[8] * B[15] * B[24] * B[35] + B[1] * B[8] * B[15] * B[29] * B[30] +
            B[1] * B[8] * B[17] * B[24] * B[33] - B[1] * B[8] * B[17] * B[27] * B[30] -
            B[1] * B[9] * B[12] * B[26] * B[35] + B[1] * B[9] * B[12] * B[29] * B[32] +
            B[1] * B[9] * B[14] * B[24] * B[35] - B[1] * B[9] * B[14] * B[29] * B[30] -
            B[1] * B[9] * B[17] * B[24] * B[32] + B[1] * B[9] * B[17] * B[26] * B[30] +
            B[1] * B[11] * B[12] * B[26] * B[33] - B[1] * B[11] * B[12] * B[27] * B[32] -
            B[1] * B[11] * B[14] * B[24] * B[33] + B[1] * B[11] * B[14] * B[27] * B[30] +
            B[1] * B[11] * B[15] * B[24] * B[32] - B[1] * B[11] * B[15] * B[26] * B[30] +
            B[2] * B[6] * B[13] * B[27] * B[35] - B[2] * B[6] * B[13] * B[29] * B[33] -
            B[2] * B[6] * B[15] * B[25] * B[35] + B[2] * B[6] * B[15] * B[29] * B[31] +
            B[2] * B[6] * B[17] * B[25] * B[33] - B[2] * B[6] * B[17] * B[27] * B[31] -
            B[2] * B[7] * B[12] * B[27] * B[35] + B[2] * B[7] * B[12] * B[29] * B[33] +
            B[2] * B[7] * B[15] * B[24] * B[35] - B[2] * B[7] * B[15] * B[29] * B[30] -
            B[2] * B[7] * B[17] * B[24] * B[33] + B[2] * B[7] * B[17] * B[27] * B[30] +
            B[2] * B[9] * B[12] * B[25] * B[35] - B[2] * B[9] * B[12] * B[29] * B[31] -
            B[2] * B[9] * B[13] * B[24] * B[35] + B[2] * B[9] * B[13] * B[29] * B[30] +
            B[2] * B[9] * B[17] * B[24] * B[31] - B[2] * B[9] * B[17] * B[25] * B[30] -
            B[2] * B[11] * B[12] * B[25] * B[33] + B[2] * B[11] * B[12] * B[27] * B[31] +
            B[2] * B[11] * B[13] * B[24] * B[33] - B[2] * B[11] * B[13] * B[27] * B[30] -
            B[2] * B[11] * B[15] * B[24] * B[31] + B[2] * B[11] * B[15] * B[25] * B[30] -
            B[3] * B[6] * B[13] * B[26] * B[35] + B[3] * B[6] * B[13] * B[29] * B[32] +
            B[3] * B[6] * B[14] * B[25] * B[35] - B[3] * B[6] * B[14] * B[29] * B[31] -
            B[3] * B[6] * B[17] * B[25] * B[32] + B[3] * B[6] * B[17] * B[26] * B[31] +
            B[3] * B[7] * B[12] * B[26] * B[35] - B[3] * B[7] * B[12] * B[29] * B[32] -
            B[3] * B[7] * B[14] * B[24] * B[35] + B[3] * B[7] * B[14] * B[29] * B[30] +
            B[3] * B[7] * B[17] * B[24] * B[32] - B[3] * B[7] * B[17] * B[26] * B[30] -
            B[3] * B[8] * B[12] * B[25] * B[35] + B[3] * B[8] * B[12] * B[29] * B[31] +
            B[3] * B[8] * B[13] * B[24] * B[35] - B[3] * B[8] * B[13] * B[29] * B[30] -
            B[3] * B[8] * B[17] * B[24] * B[31] + B[3] * B[8] * B[17] * B[25] * B[30] +
            B[3] * B[11] * B[12] * B[25] * B[32] - B[3] * B[11] * B[12] * B[26] * B[31] -
            B[3] * B[11] * B[13] * B[24] * B[32] + B[3] * B[11] * B[13] * B[26] * B[30] +
            B[3] * B[11] * B[14] * B[24] * B[31] - B[3] * B[11] * B[14] * B[25] * B[30] +
            B[5] * B[6] * B[13] * B[26] * B[33] - B[5] * B[6] * B[13] * B[27] * B[32] -
            B[5] * B[6] * B[14] * B[25] * B[33] + B[5] * B[6] * B[14] * B[27] * B[31] +
            B[5] * B[6] * B[15] * B[25] * B[32] - B[5] * B[6] * B[15] * B[26] * B[31] -
            B[5] * B[7] * B[12] * B[26] * B[33] + B[5] * B[7] * B[12] * B[27] * B[32] +
            B[5] * B[7] * B[14] * B[24] * B[33] - B[5] * B[7] * B[14] * B[27] * B[30] -
            B[5] * B[7] * B[15] * B[24] * B[32] + B[5] * B[7] * B[15] * B[26] * B[30] +
            B[5] * B[8] * B[12] * B[25] * B[33] - B[5] * B[8] * B[12] * B[27] * B[31] -
            B[5] * B[8] * B[13] * B[24] * B[33] + B[5] * B[8] * B[13] * B[27] * B[30] +
            B[5] * B[8] * B[15] * B[24] * B[31] - B[5] * B[8] * B[15] * B[25] * B[30] -
            B[5] * B[9] * B[12] * B[25] * B[32] + B[5] * B[9] * B[12] * B[26] * B[31] +
            B[5] * B[9] * B[13] * B[24] * B[32] - B[5] * B[9] * B[13] * B[26] * B[30] -
            B[5] * B[9] * B[14] * B[24] * B[31] + B[5] * B[9] * B[14] * B[25] * B[30] -
            B[6] * B[13] * B[20] * B[27] * B[35] + B[6] * B[13] * B[20] * B[29] * B[33] +
            B[6] * B[13] * B[21] * B[26] * B[35] - B[6] * B[13] * B[21] * B[29] * B[32] -
            B[6] * B[13] * B[23] * B[26] * B[33] + B[6] * B[13] * B[23] * B[27] * B[32] +
            B[6] * B[14] * B[19] * B[27] * B[35] - B[6] * B[14] * B[19] * B[29] * B[33] -
            B[6] * B[14] * B[21] * B[25] * B[35] + B[6] * B[14] * B[21] * B[29] * B[31] +
            B[6] * B[14] * B[23] * B[25] * B[33] - B[6] * B[14] * B[23] * B[27] * B[31] -
            B[6] * B[15] * B[19] * B[26] * B[35] + B[6] * B[15] * B[19] * B[29] * B[32] +
            B[6] * B[15] * B[20] * B[25] * B[35] - B[6] * B[15] * B[20] * B[29] * B[31] -
            B[6] * B[15] * B[23] * B[25] * B[32] + B[6] * B[15] * B[23] * B[26] * B[31] +
            B[6] * B[17] * B[19] * B[26] * B[33] - B[6] * B[17] * B[19] * B[27] * B[32] -
            B[6] * B[17] * B[20] * B[25] * B[33] + B[6] * B[17] * B[20] * B[27] * B[31] +
            B[6] * B[17] * B[21] * B[25] * B[32] - B[6] * B[17] * B[21] * B[26] * B[31] +
            B[7] * B[12] * B[20] * B[27] * B[35] - B[7] * B[12] * B[20] * B[29] * B[33] -
            B[7] * B[12] * B[21] * B[26] * B[35] + B[7] * B[12] * B[21] * B[29] * B[32] +
            B[7] * B[12] * B[23] * B[26] * B[33] - B[7] * B[12] * B[23] * B[27] * B[32] -
            B[7] * B[14] * B[18] * B[27] * B[35] + B[7] * B[14] * B[18] * B[29] * B[33] +
            B[7] * B[14] * B[21] * B[24] * B[35] - B[7] * B[14] * B[21] * B[29] * B[30] -
            B[7] * B[14] * B[23] * B[24] * B[33] + B[7] * B[14] * B[23] * B[27] * B[30] +
            B[7] * B[15] * B[18] * B[26] * B[35] - B[7] * B[15] * B[18] * B[29] * B[32] -
            B[7] * B[15] * B[20] * B[24] * B[35] + B[7] * B[15] * B[20] * B[29] * B[30] +
            B[7] * B[15] * B[23] * B[24] * B[32] - B[7] * B[15] * B[23] * B[26] * B[30] -
            B[7] * B[17] * B[18] * B[26] * B[33] + B[7] * B[17] * B[18] * B[27] * B[32] +
            B[7] * B[17] * B[20] * B[24] * B[33] - B[7] * B[17] * B[20] * B[27] * B[30] -
            B[7] * B[17] * B[21] * B[24] * B[32] + B[7] * B[17] * B[21] * B[26] * B[30] -
            B[8] * B[12] * B[19] * B[27] * B[35] + B[8] * B[12] * B[19] * B[29] * B[33] +
            B[8] * B[12] * B[21] * B[25] * B[35] - B[8] * B[12] * B[21] * B[29] * B[31] -
            B[8] * B[12] * B[23] * B[25] * B[33] + B[8] * B[12] * B[23] * B[27] * B[31] +
            B[8] * B[13] * B[18] * B[27] * B[35] - B[8] * B[13] * B[18] * B[29] * B[33] -
            B[8] * B[13] * B[21] * B[24] * B[35] + B[8] * B[13] * B[21] * B[29] * B[30] +
            B[8] * B[13] * B[23] * B[24] * B[33] - B[8] * B[13] * B[23] * B[27] * B[30] -
            B[8] * B[15] * B[18] * B[25] * B[35] + B[8] * B[15] * B[18] * B[29] * B[31] +
            B[8] * B[15] * B[19] * B[24] * B[35] - B[8] * B[15] * B[19] * B[29] * B[30] -
            B[8] * B[15] * B[23] * B[24] * B[31] + B[8] * B[15] * B[23] * B[25] * B[30] +
            B[8] * B[17] * B[18] * B[25] * B[33] - B[8] * B[17] * B[18] * B[27] * B[31] -
            B[8] * B[17] * B[19] * B[24] * B[33] + B[8] * B[17] * B[19] * B[27] * B[30] +
            B[8] * B[17] * B[21] * B[24] * B[31] - B[8] * B[17] * B[21] * B[25] * B[30] +
            B[9] * B[12] * B[19] * B[26] * B[35] - B[9] * B[12] * B[19] * B[29] * B[32] -
            B[9] * B[12] * B[20] * B[25] * B[35] + B[9] * B[12] * B[20] * B[29] * B[31] +
            B[9] * B[12] * B[23] * B[25] * B[32] - B[9] * B[12] * B[23] * B[26] * B[31] -
            B[9] * B[13] * B[18] * B[26] * B[35] + B[9] * B[13] * B[18] * B[29] * B[32] +
            B[9] * B[13] * B[20] * B[24] * B[35] - B[9] * B[13] * B[20] * B[29] * B[30] -
            B[9] * B[13] * B[23] * B[24] * B[32] + B[9] * B[13] * B[23] * B[26] * B[30] +
            B[9] * B[14] * B[18] * B[25] * B[35] - B[9] * B[14] * B[18] * B[29] * B[31] -
            B[9] * B[14] * B[19] * B[24] * B[35] + B[9] * B[14] * B[19] * B[29] * B[30] +
            B[9] * B[14] * B[23] * B[24] * B[31] - B[9] * B[14] * B[23] * B[25] * B[30] -
            B[9] * B[17] * B[18] * B[25] * B[32] + B[9] * B[17] * B[18] * B[26] * B[31] +
            B[9] * B[17] * B[19] * B[24] * B[32] - B[9] * B[17] * B[19] * B[26] * B[30] -
            B[9] * B[17] * B[20] * B[24] * B[31] + B[9] * B[17] * B[20] * B[25] * B[30] -
            B[11] * B[12] * B[19] * B[26] * B[33] + B[11] * B[12] * B[19] * B[27] * B[32] +
            B[11] * B[12] * B[20] * B[25] * B[33] - B[11] * B[12] * B[20] * B[27] * B[31] -
            B[11] * B[12] * B[21] * B[25] * B[32] + B[11] * B[12] * B[21] * B[26] * B[31] +
            B[11] * B[13] * B[18] * B[26] * B[33] - B[11] * B[13] * B[18] * B[27] * B[32] -
            B[11] * B[13] * B[20] * B[24] * B[33] + B[11] * B[13] * B[20] * B[27] * B[30] +
            B[11] * B[13] * B[21] * B[24] * B[32] - B[11] * B[13] * B[21] * B[26] * B[30] -
            B[11] * B[14] * B[18] * B[25] * B[33] + B[11] * B[14] * B[18] * B[27] * B[31] +
            B[11] * B[14] * B[19] * B[24] * B[33] - B[11] * B[14] * B[19] * B[27] * B[30] -
            B[11] * B[14] * B[21] * B[24] * B[31] + B[11] * B[14] * B[21] * B[25] * B[30] +
            B[11] * B[15] * B[18] * B[25] * B[32] - B[11] * B[15] * B[18] * B[26] * B[31] -
            B[11] * B[15] * B[19] * B[24] * B[32] + B[11] * B[15] * B[19] * B[26] * B[30] +
            B[11] * B[15] * B[20] * B[24] * B[31] - B[11] * B[15] * B[20] * B[25] * B[30]) /
           (DET);

    L[5] = -(B[0] * B[7] * B[14] * B[21] * B[28] - B[0] * B[7] * B[14] * B[22] * B[27] -
             B[0] * B[7] * B[15] * B[20] * B[28] + B[0] * B[7] * B[15] * B[22] * B[26] +
             B[0] * B[7] * B[16] * B[20] * B[27] - B[0] * B[7] * B[16] * B[21] * B[26] -
             B[0] * B[8] * B[13] * B[21] * B[28] + B[0] * B[8] * B[13] * B[22] * B[27] +
             B[0] * B[8] * B[15] * B[19] * B[28] - B[0] * B[8] * B[15] * B[22] * B[25] -
             B[0] * B[8] * B[16] * B[19] * B[27] + B[0] * B[8] * B[16] * B[21] * B[25] +
             B[0] * B[9] * B[13] * B[20] * B[28] - B[0] * B[9] * B[13] * B[22] * B[26] -
             B[0] * B[9] * B[14] * B[19] * B[28] + B[0] * B[9] * B[14] * B[22] * B[25] +
             B[0] * B[9] * B[16] * B[19] * B[26] - B[0] * B[9] * B[16] * B[20] * B[25] -
             B[0] * B[10] * B[13] * B[20] * B[27] + B[0] * B[10] * B[13] * B[21] * B[26] +
             B[0] * B[10] * B[14] * B[19] * B[27] - B[0] * B[10] * B[14] * B[21] * B[25] -
             B[0] * B[10] * B[15] * B[19] * B[26] + B[0] * B[10] * B[15] * B[20] * B[25] -
             B[1] * B[6] * B[14] * B[21] * B[28] + B[1] * B[6] * B[14] * B[22] * B[27] +
             B[1] * B[6] * B[15] * B[20] * B[28] - B[1] * B[6] * B[15] * B[22] * B[26] -
             B[1] * B[6] * B[16] * B[20] * B[27] + B[1] * B[6] * B[16] * B[21] * B[26] +
             B[1] * B[8] * B[12] * B[21] * B[28] - B[1] * B[8] * B[12] * B[22] * B[27] -
             B[1] * B[8] * B[15] * B[18] * B[28] + B[1] * B[8] * B[15] * B[22] * B[24] +
             B[1] * B[8] * B[16] * B[18] * B[27] - B[1] * B[8] * B[16] * B[21] * B[24] -
             B[1] * B[9] * B[12] * B[20] * B[28] + B[1] * B[9] * B[12] * B[22] * B[26] +
             B[1] * B[9] * B[14] * B[18] * B[28] - B[1] * B[9] * B[14] * B[22] * B[24] -
             B[1] * B[9] * B[16] * B[18] * B[26] + B[1] * B[9] * B[16] * B[20] * B[24] +
             B[1] * B[10] * B[12] * B[20] * B[27] - B[1] * B[10] * B[12] * B[21] * B[26] -
             B[1] * B[10] * B[14] * B[18] * B[27] + B[1] * B[10] * B[14] * B[21] * B[24] +
             B[1] * B[10] * B[15] * B[18] * B[26] - B[1] * B[10] * B[15] * B[20] * B[24] +
             B[2] * B[6] * B[13] * B[21] * B[28] - B[2] * B[6] * B[13] * B[22] * B[27] -
             B[2] * B[6] * B[15] * B[19] * B[28] + B[2] * B[6] * B[15] * B[22] * B[25] +
             B[2] * B[6] * B[16] * B[19] * B[27] - B[2] * B[6] * B[16] * B[21] * B[25] -
             B[2] * B[7] * B[12] * B[21] * B[28] + B[2] * B[7] * B[12] * B[22] * B[27] +
             B[2] * B[7] * B[15] * B[18] * B[28] - B[2] * B[7] * B[15] * B[22] * B[24] -
             B[2] * B[7] * B[16] * B[18] * B[27] + B[2] * B[7] * B[16] * B[21] * B[24] +
             B[2] * B[9] * B[12] * B[19] * B[28] - B[2] * B[9] * B[12] * B[22] * B[25] -
             B[2] * B[9] * B[13] * B[18] * B[28] + B[2] * B[9] * B[13] * B[22] * B[24] +
             B[2] * B[9] * B[16] * B[18] * B[25] - B[2] * B[9] * B[16] * B[19] * B[24] -
             B[2] * B[10] * B[12] * B[19] * B[27] + B[2] * B[10] * B[12] * B[21] * B[25] +
             B[2] * B[10] * B[13] * B[18] * B[27] - B[2] * B[10] * B[13] * B[21] * B[24] -
             B[2] * B[10] * B[15] * B[18] * B[25] + B[2] * B[10] * B[15] * B[19] * B[24] -
             B[3] * B[6] * B[13] * B[20] * B[28] + B[3] * B[6] * B[13] * B[22] * B[26] +
             B[3] * B[6] * B[14] * B[19] * B[28] - B[3] * B[6] * B[14] * B[22] * B[25] -
             B[3] * B[6] * B[16] * B[19] * B[26] + B[3] * B[6] * B[16] * B[20] * B[25] +
             B[3] * B[7] * B[12] * B[20] * B[28] - B[3] * B[7] * B[12] * B[22] * B[26] -
             B[3] * B[7] * B[14] * B[18] * B[28] + B[3] * B[7] * B[14] * B[22] * B[24] +
             B[3] * B[7] * B[16] * B[18] * B[26] - B[3] * B[7] * B[16] * B[20] * B[24] -
             B[3] * B[8] * B[12] * B[19] * B[28] + B[3] * B[8] * B[12] * B[22] * B[25] +
             B[3] * B[8] * B[13] * B[18] * B[28] - B[3] * B[8] * B[13] * B[22] * B[24] -
             B[3] * B[8] * B[16] * B[18] * B[25] + B[3] * B[8] * B[16] * B[19] * B[24] +
             B[3] * B[10] * B[12] * B[19] * B[26] - B[3] * B[10] * B[12] * B[20] * B[25] -
             B[3] * B[10] * B[13] * B[18] * B[26] + B[3] * B[10] * B[13] * B[20] * B[24] +
             B[3] * B[10] * B[14] * B[18] * B[25] - B[3] * B[10] * B[14] * B[19] * B[24] +
             B[4] * B[6] * B[13] * B[20] * B[27] - B[4] * B[6] * B[13] * B[21] * B[26] -
             B[4] * B[6] * B[14] * B[19] * B[27] + B[4] * B[6] * B[14] * B[21] * B[25] +
             B[4] * B[6] * B[15] * B[19] * B[26] - B[4] * B[6] * B[15] * B[20] * B[25] -
             B[4] * B[7] * B[12] * B[20] * B[27] + B[4] * B[7] * B[12] * B[21] * B[26] +
             B[4] * B[7] * B[14] * B[18] * B[27] - B[4] * B[7] * B[14] * B[21] * B[24] -
             B[4] * B[7] * B[15] * B[18] * B[26] + B[4] * B[7] * B[15] * B[20] * B[24] +
             B[4] * B[8] * B[12] * B[19] * B[27] - B[4] * B[8] * B[12] * B[21] * B[25] -
             B[4] * B[8] * B[13] * B[18] * B[27] + B[4] * B[8] * B[13] * B[21] * B[24] +
             B[4] * B[8] * B[15] * B[18] * B[25] - B[4] * B[8] * B[15] * B[19] * B[24] -
             B[4] * B[9] * B[12] * B[19] * B[26] + B[4] * B[9] * B[12] * B[20] * B[25] +
             B[4] * B[9] * B[13] * B[18] * B[26] - B[4] * B[9] * B[13] * B[20] * B[24] -
             B[4] * B[9] * B[14] * B[18] * B[25] + B[4] * B[9] * B[14] * B[19] * B[24] +
             B[0] * B[7] * B[14] * B[27] * B[34] - B[0] * B[7] * B[14] * B[28] * B[33] -
             B[0] * B[7] * B[15] * B[26] * B[34] + B[0] * B[7] * B[15] * B[28] * B[32] +
             B[0] * B[7] * B[16] * B[26] * B[33] - B[0] * B[7] * B[16] * B[27] * B[32] -
             B[0] * B[8] * B[13] * B[27] * B[34] + B[0] * B[8] * B[13] * B[28] * B[33] +
             B[0] * B[8] * B[15] * B[25] * B[34] - B[0] * B[8] * B[15] * B[28] * B[31] -
             B[0] * B[8] * B[16] * B[25] * B[33] + B[0] * B[8] * B[16] * B[27] * B[31] +
             B[0] * B[9] * B[13] * B[26] * B[34] - B[0] * B[9] * B[13] * B[28] * B[32] -
             B[0] * B[9] * B[14] * B[25] * B[34] + B[0] * B[9] * B[14] * B[28] * B[31] +
             B[0] * B[9] * B[16] * B[25] * B[32] - B[0] * B[9] * B[16] * B[26] * B[31] -
             B[0] * B[10] * B[13] * B[26] * B[33] + B[0] * B[10] * B[13] * B[27] * B[32] +
             B[0] * B[10] * B[14] * B[25] * B[33] - B[0] * B[10] * B[14] * B[27] * B[31] -
             B[0] * B[10] * B[15] * B[25] * B[32] + B[0] * B[10] * B[15] * B[26] * B[31] -
             B[1] * B[6] * B[14] * B[27] * B[34] + B[1] * B[6] * B[14] * B[28] * B[33] +
             B[1] * B[6] * B[15] * B[26] * B[34] - B[1] * B[6] * B[15] * B[28] * B[32] -
             B[1] * B[6] * B[16] * B[26] * B[33] + B[1] * B[6] * B[16] * B[27] * B[32] +
             B[1] * B[8] * B[12] * B[27] * B[34] - B[1] * B[8] * B[12] * B[28] * B[33] -
             B[1] * B[8] * B[15] * B[24] * B[34] + B[1] * B[8] * B[15] * B[28] * B[30] +
             B[1] * B[8] * B[16] * B[24] * B[33] - B[1] * B[8] * B[16] * B[27] * B[30] -
             B[1] * B[9] * B[12] * B[26] * B[34] + B[1] * B[9] * B[12] * B[28] * B[32] +
             B[1] * B[9] * B[14] * B[24] * B[34] - B[1] * B[9] * B[14] * B[28] * B[30] -
             B[1] * B[9] * B[16] * B[24] * B[32] + B[1] * B[9] * B[16] * B[26] * B[30] +
             B[1] * B[10] * B[12] * B[26] * B[33] - B[1] * B[10] * B[12] * B[27] * B[32] -
             B[1] * B[10] * B[14] * B[24] * B[33] + B[1] * B[10] * B[14] * B[27] * B[30] +
             B[1] * B[10] * B[15] * B[24] * B[32] - B[1] * B[10] * B[15] * B[26] * B[30] +
             B[2] * B[6] * B[13] * B[27] * B[34] - B[2] * B[6] * B[13] * B[28] * B[33] -
             B[2] * B[6] * B[15] * B[25] * B[34] + B[2] * B[6] * B[15] * B[28] * B[31] +
             B[2] * B[6] * B[16] * B[25] * B[33] - B[2] * B[6] * B[16] * B[27] * B[31] -
             B[2] * B[7] * B[12] * B[27] * B[34] + B[2] * B[7] * B[12] * B[28] * B[33] +
             B[2] * B[7] * B[15] * B[24] * B[34] - B[2] * B[7] * B[15] * B[28] * B[30] -
             B[2] * B[7] * B[16] * B[24] * B[33] + B[2] * B[7] * B[16] * B[27] * B[30] +
             B[2] * B[9] * B[12] * B[25] * B[34] - B[2] * B[9] * B[12] * B[28] * B[31] -
             B[2] * B[9] * B[13] * B[24] * B[34] + B[2] * B[9] * B[13] * B[28] * B[30] +
             B[2] * B[9] * B[16] * B[24] * B[31] - B[2] * B[9] * B[16] * B[25] * B[30] -
             B[2] * B[10] * B[12] * B[25] * B[33] + B[2] * B[10] * B[12] * B[27] * B[31] +
             B[2] * B[10] * B[13] * B[24] * B[33] - B[2] * B[10] * B[13] * B[27] * B[30] -
             B[2] * B[10] * B[15] * B[24] * B[31] + B[2] * B[10] * B[15] * B[25] * B[30] -
             B[3] * B[6] * B[13] * B[26] * B[34] + B[3] * B[6] * B[13] * B[28] * B[32] +
             B[3] * B[6] * B[14] * B[25] * B[34] - B[3] * B[6] * B[14] * B[28] * B[31] -
             B[3] * B[6] * B[16] * B[25] * B[32] + B[3] * B[6] * B[16] * B[26] * B[31] +
             B[3] * B[7] * B[12] * B[26] * B[34] - B[3] * B[7] * B[12] * B[28] * B[32] -
             B[3] * B[7] * B[14] * B[24] * B[34] + B[3] * B[7] * B[14] * B[28] * B[30] +
             B[3] * B[7] * B[16] * B[24] * B[32] - B[3] * B[7] * B[16] * B[26] * B[30] -
             B[3] * B[8] * B[12] * B[25] * B[34] + B[3] * B[8] * B[12] * B[28] * B[31] +
             B[3] * B[8] * B[13] * B[24] * B[34] - B[3] * B[8] * B[13] * B[28] * B[30] -
             B[3] * B[8] * B[16] * B[24] * B[31] + B[3] * B[8] * B[16] * B[25] * B[30] +
             B[3] * B[10] * B[12] * B[25] * B[32] - B[3] * B[10] * B[12] * B[26] * B[31] -
             B[3] * B[10] * B[13] * B[24] * B[32] + B[3] * B[10] * B[13] * B[26] * B[30] +
             B[3] * B[10] * B[14] * B[24] * B[31] - B[3] * B[10] * B[14] * B[25] * B[30] +
             B[4] * B[6] * B[13] * B[26] * B[33] - B[4] * B[6] * B[13] * B[27] * B[32] -
             B[4] * B[6] * B[14] * B[25] * B[33] + B[4] * B[6] * B[14] * B[27] * B[31] +
             B[4] * B[6] * B[15] * B[25] * B[32] - B[4] * B[6] * B[15] * B[26] * B[31] -
             B[4] * B[7] * B[12] * B[26] * B[33] + B[4] * B[7] * B[12] * B[27] * B[32] +
             B[4] * B[7] * B[14] * B[24] * B[33] - B[4] * B[7] * B[14] * B[27] * B[30] -
             B[4] * B[7] * B[15] * B[24] * B[32] + B[4] * B[7] * B[15] * B[26] * B[30] +
             B[4] * B[8] * B[12] * B[25] * B[33] - B[4] * B[8] * B[12] * B[27] * B[31] -
             B[4] * B[8] * B[13] * B[24] * B[33] + B[4] * B[8] * B[13] * B[27] * B[30] +
             B[4] * B[8] * B[15] * B[24] * B[31] - B[4] * B[8] * B[15] * B[25] * B[30] -
             B[4] * B[9] * B[12] * B[25] * B[32] + B[4] * B[9] * B[12] * B[26] * B[31] +
             B[4] * B[9] * B[13] * B[24] * B[32] - B[4] * B[9] * B[13] * B[26] * B[30] -
             B[4] * B[9] * B[14] * B[24] * B[31] + B[4] * B[9] * B[14] * B[25] * B[30] -
             B[6] * B[13] * B[20] * B[27] * B[34] + B[6] * B[13] * B[20] * B[28] * B[33] +
             B[6] * B[13] * B[21] * B[26] * B[34] - B[6] * B[13] * B[21] * B[28] * B[32] -
             B[6] * B[13] * B[22] * B[26] * B[33] + B[6] * B[13] * B[22] * B[27] * B[32] +
             B[6] * B[14] * B[19] * B[27] * B[34] - B[6] * B[14] * B[19] * B[28] * B[33] -
             B[6] * B[14] * B[21] * B[25] * B[34] + B[6] * B[14] * B[21] * B[28] * B[31] +
             B[6] * B[14] * B[22] * B[25] * B[33] - B[6] * B[14] * B[22] * B[27] * B[31] -
             B[6] * B[15] * B[19] * B[26] * B[34] + B[6] * B[15] * B[19] * B[28] * B[32] +
             B[6] * B[15] * B[20] * B[25] * B[34] - B[6] * B[15] * B[20] * B[28] * B[31] -
             B[6] * B[15] * B[22] * B[25] * B[32] + B[6] * B[15] * B[22] * B[26] * B[31] +
             B[6] * B[16] * B[19] * B[26] * B[33] - B[6] * B[16] * B[19] * B[27] * B[32] -
             B[6] * B[16] * B[20] * B[25] * B[33] + B[6] * B[16] * B[20] * B[27] * B[31] +
             B[6] * B[16] * B[21] * B[25] * B[32] - B[6] * B[16] * B[21] * B[26] * B[31] +
             B[7] * B[12] * B[20] * B[27] * B[34] - B[7] * B[12] * B[20] * B[28] * B[33] -
             B[7] * B[12] * B[21] * B[26] * B[34] + B[7] * B[12] * B[21] * B[28] * B[32] +
             B[7] * B[12] * B[22] * B[26] * B[33] - B[7] * B[12] * B[22] * B[27] * B[32] -
             B[7] * B[14] * B[18] * B[27] * B[34] + B[7] * B[14] * B[18] * B[28] * B[33] +
             B[7] * B[14] * B[21] * B[24] * B[34] - B[7] * B[14] * B[21] * B[28] * B[30] -
             B[7] * B[14] * B[22] * B[24] * B[33] + B[7] * B[14] * B[22] * B[27] * B[30] +
             B[7] * B[15] * B[18] * B[26] * B[34] - B[7] * B[15] * B[18] * B[28] * B[32] -
             B[7] * B[15] * B[20] * B[24] * B[34] + B[7] * B[15] * B[20] * B[28] * B[30] +
             B[7] * B[15] * B[22] * B[24] * B[32] - B[7] * B[15] * B[22] * B[26] * B[30] -
             B[7] * B[16] * B[18] * B[26] * B[33] + B[7] * B[16] * B[18] * B[27] * B[32] +
             B[7] * B[16] * B[20] * B[24] * B[33] - B[7] * B[16] * B[20] * B[27] * B[30] -
             B[7] * B[16] * B[21] * B[24] * B[32] + B[7] * B[16] * B[21] * B[26] * B[30] -
             B[8] * B[12] * B[19] * B[27] * B[34] + B[8] * B[12] * B[19] * B[28] * B[33] +
             B[8] * B[12] * B[21] * B[25] * B[34] - B[8] * B[12] * B[21] * B[28] * B[31] -
             B[8] * B[12] * B[22] * B[25] * B[33] + B[8] * B[12] * B[22] * B[27] * B[31] +
             B[8] * B[13] * B[18] * B[27] * B[34] - B[8] * B[13] * B[18] * B[28] * B[33] -
             B[8] * B[13] * B[21] * B[24] * B[34] + B[8] * B[13] * B[21] * B[28] * B[30] +
             B[8] * B[13] * B[22] * B[24] * B[33] - B[8] * B[13] * B[22] * B[27] * B[30] -
             B[8] * B[15] * B[18] * B[25] * B[34] + B[8] * B[15] * B[18] * B[28] * B[31] +
             B[8] * B[15] * B[19] * B[24] * B[34] - B[8] * B[15] * B[19] * B[28] * B[30] -
             B[8] * B[15] * B[22] * B[24] * B[31] + B[8] * B[15] * B[22] * B[25] * B[30] +
             B[8] * B[16] * B[18] * B[25] * B[33] - B[8] * B[16] * B[18] * B[27] * B[31] -
             B[8] * B[16] * B[19] * B[24] * B[33] + B[8] * B[16] * B[19] * B[27] * B[30] +
             B[8] * B[16] * B[21] * B[24] * B[31] - B[8] * B[16] * B[21] * B[25] * B[30] +
             B[9] * B[12] * B[19] * B[26] * B[34] - B[9] * B[12] * B[19] * B[28] * B[32] -
             B[9] * B[12] * B[20] * B[25] * B[34] + B[9] * B[12] * B[20] * B[28] * B[31] +
             B[9] * B[12] * B[22] * B[25] * B[32] - B[9] * B[12] * B[22] * B[26] * B[31] -
             B[9] * B[13] * B[18] * B[26] * B[34] + B[9] * B[13] * B[18] * B[28] * B[32] +
             B[9] * B[13] * B[20] * B[24] * B[34] - B[9] * B[13] * B[20] * B[28] * B[30] -
             B[9] * B[13] * B[22] * B[24] * B[32] + B[9] * B[13] * B[22] * B[26] * B[30] +
             B[9] * B[14] * B[18] * B[25] * B[34] - B[9] * B[14] * B[18] * B[28] * B[31] -
             B[9] * B[14] * B[19] * B[24] * B[34] + B[9] * B[14] * B[19] * B[28] * B[30] +
             B[9] * B[14] * B[22] * B[24] * B[31] - B[9] * B[14] * B[22] * B[25] * B[30] -
             B[9] * B[16] * B[18] * B[25] * B[32] + B[9] * B[16] * B[18] * B[26] * B[31] +
             B[9] * B[16] * B[19] * B[24] * B[32] - B[9] * B[16] * B[19] * B[26] * B[30] -
             B[9] * B[16] * B[20] * B[24] * B[31] + B[9] * B[16] * B[20] * B[25] * B[30] -
             B[10] * B[12] * B[19] * B[26] * B[33] + B[10] * B[12] * B[19] * B[27] * B[32] +
             B[10] * B[12] * B[20] * B[25] * B[33] - B[10] * B[12] * B[20] * B[27] * B[31] -
             B[10] * B[12] * B[21] * B[25] * B[32] + B[10] * B[12] * B[21] * B[26] * B[31] +
             B[10] * B[13] * B[18] * B[26] * B[33] - B[10] * B[13] * B[18] * B[27] * B[32] -
             B[10] * B[13] * B[20] * B[24] * B[33] + B[10] * B[13] * B[20] * B[27] * B[30] +
             B[10] * B[13] * B[21] * B[24] * B[32] - B[10] * B[13] * B[21] * B[26] * B[30] -
             B[10] * B[14] * B[18] * B[25] * B[33] + B[10] * B[14] * B[18] * B[27] * B[31] +
             B[10] * B[14] * B[19] * B[24] * B[33] - B[10] * B[14] * B[19] * B[27] * B[30] -
             B[10] * B[14] * B[21] * B[24] * B[31] + B[10] * B[14] * B[21] * B[25] * B[30] +
             B[10] * B[15] * B[18] * B[25] * B[32] - B[10] * B[15] * B[18] * B[26] * B[31] -
             B[10] * B[15] * B[19] * B[24] * B[32] + B[10] * B[15] * B[19] * B[26] * B[30] +
             B[10] * B[15] * B[20] * B[24] * B[31] - B[10] * B[15] * B[20] * B[25] * B[30]) /
           (DET);

    Real norm = 0;
    for (int i = 0; i < 6; i++)
        norm += abs(L[i]);

    if (norm > 100) {
        //        printf("Li, ");
        L[0] = 1.0;
        L[1] = 0.0;
        L[2] = 0.0;
        L[3] = 1.0;
        L[4] = 0.0;
        L[5] = 1.0;
    }
}

/// @} fsi_solver

}  // namespace fsi
}  // namespace chrono

#endif
