///////////////////////////////////////////////////////////////////////////////
//                                                         
// CommandExecuter.cc
// ------------------
// Executing commands via process forking 
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2007-2017 Bjoern Lemke                        
//
// IMPLEMENTATION MODULE
//
// Class: CommandExecuter
// 
// Description: Execution of a forked sub program or a stream pipe 
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// SYSTEM INCLUDES
#include <unistd.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>

#ifndef HAVE_MINGW
#include <sys/wait.h>
#endif

// BASE INCLUDES
#include "Exception.h"

#include "CommandExecuter.h"
#include "Tokenizer.h"
#include "File.h"

// DEFINES
#define SHELLPATH "/bin/bash"
#define SHELL "bash"

CommandExecuter::CommandExecuter(const Chain& shellPath) : SigHandler()
{
    _shellPath = shellPath;
}

CommandExecuter::~CommandExecuter()  
{
}

int CommandExecuter::execute(const Chain& command, int timeout)
{

#ifdef HAVE_MINGW

    // not implemented
    return 0;

#else

    File checkFile(_shellPath);
    if (checkFile.exists() == false )
    {
	Chain msg = "Invalid execution shell path <" +  _shellPath + ">";
	throw Exception(EXLOC, msg);	
    }
    
    Tokenizer argTok(_shellPath, Chain("/"));
    Chain shellProg;
    while ( argTok.nextToken(shellProg) );
    
    alarm(0);
    alarm(timeout);

    install(SIGALRM);
    
    int pid = fork();

    if ( pid > 0)
    {
	_childPid = pid;

	int status;
	waitpid(pid, &status, WUNTRACED);

	if ( WIFSIGNALED(status) )
	{	   	    
	    Chain msg = "Timeout occured for command <" +  command + ">";
	    throw Exception(EXLOC, msg);	    
	}
	
	alarm(0);
	return WEXITSTATUS(status);

    }
    else if (pid == 0) 
    { 
	char* argArray[4]; 
	argArray[0]=(char*)shellProg;
	argArray[1]=(char*)"-c";
	argArray[2]=(char*)command;
	argArray[3]=0;

	int result;
	result = execv(_shellPath, argArray);

	exit (result);
    }
    else
    {
	Chain msg = "Cannot execute command <" +  command + ">";
	throw Exception(EXLOC, msg);
    }
#endif

}

Chain CommandExecuter::executeStream(const Chain& command)
{

    Chain streamResult;
    
    FILE *outStream;

    char buff[512];

    if (! ( outStream = popen((char*)command, "r")))
    {	
	Chain msg = Chain("Cannot execute command <") + command + Chain("> : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
 
    while( fgets(buff, sizeof(buff), outStream ) != NULL )
    {
	streamResult += Chain(buff);
    }

    int status = pclose(outStream);

#ifdef HAVE_MINGW

    if ( status == -1 )
    {
	Chain msg = Chain("Stream command failed with errno ") + Chain(errno);
	throw Exception(EXLOC, msg);	
    }
    int errorCode = status;
#else
    int errorCode = WEXITSTATUS(status);
#endif
    
    if ( errorCode != 0 )
    {
	Chain msg = Chain("Stream command failed with status ") + Chain(errorCode);
	throw Exception(EXLOC, msg);	
    }

    return streamResult;

}


void CommandExecuter::sigCatch(int sig)
{

#ifdef HAVE_MINGW
    // not implemented
#else
    kill(_childPid, SIGTERM);
    _childPid = 0;
#endif

}
