// Copyright 2021 gRPC authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <grpc/support/port_platform.h>

#ifndef GRPC_NO_BINDER

#include <grpcpp/create_channel_binder.h>

// The interface is only defined if GPR_ANDROID is defined, because some
// arguments requires JNI.
// Furthermore, the interface is non-phony only when
// GPR_SUPPORT_BINDER_TRANSPORT is true because actual implementation of binder
// transport requires newer version of NDK API

#ifdef GPR_ANDROID

#include <grpc/grpc.h>
#include <grpc/grpc_posix.h>

#include "src/core/lib/gprpp/crash.h"

#ifdef GPR_SUPPORT_BINDER_TRANSPORT

#include "absl/log/check.h"
#include "absl/log/log.h"
#include "absl/memory/memory.h"
#include "absl/strings/substitute.h"
#include "absl/time/clock.h"
#include "absl/time/time.h"

#include <grpc/support/port_platform.h>
#include <grpcpp/impl/grpc_library.h>

#include "src/core/client_channel/client_channel_filter.h"
#include "src/core/ext/transport/binder/client/channel_create_impl.h"
#include "src/core/ext/transport/binder/client/connection_id_generator.h"
#include "src/core/ext/transport/binder/client/endpoint_binder_pool.h"
#include "src/core/ext/transport/binder/client/jni_utils.h"
#include "src/core/ext/transport/binder/client/security_policy_setting.h"
#include "src/core/ext/transport/binder/transport/binder_transport.h"
#include "src/core/ext/transport/binder/wire_format/binder.h"
#include "src/core/ext/transport/binder/wire_format/binder_android.h"
#include "src/core/lib/channel/channel_args.h"
#include "src/core/lib/config/core_configuration.h"
#include "src/core/lib/surface/channel.h"
#include "src/core/lib/transport/transport.h"
#include "src/cpp/client/create_channel_internal.h"

namespace {
// grpc.io.action.BIND is the standard action name for binding to binder
// transport server.
const char* kStandardActionName = "grpc.io.action.BIND";
}  // namespace

namespace grpc {
namespace experimental {

std::shared_ptr<grpc::Channel> CreateBinderChannel(
    void* jni_env_void, jobject context, absl::string_view package_name,
    absl::string_view class_name,
    std::shared_ptr<grpc::experimental::binder::SecurityPolicy>
        security_policy) {
  return CreateCustomBinderChannel(jni_env_void, context, package_name,
                                   class_name, security_policy,
                                   ChannelArguments());
}

std::shared_ptr<grpc::Channel> CreateCustomBinderChannel(
    void* jni_env_void, jobject context, absl::string_view package_name,
    absl::string_view class_name,
    std::shared_ptr<grpc::experimental::binder::SecurityPolicy> security_policy,
    const ChannelArguments& args) {
  return CreateCustomBinderChannel(
      jni_env_void, context,
      absl::Substitute("android-app://$0#Intent;action=$1;component=$0/$2;end",
                       package_name, kStandardActionName, class_name),
      security_policy, args);
}

std::shared_ptr<grpc::Channel> CreateBinderChannel(
    void* jni_env_void, jobject context, absl::string_view uri,
    std::shared_ptr<grpc::experimental::binder::SecurityPolicy>
        security_policy) {
  return CreateCustomBinderChannel(jni_env_void, context, uri, security_policy,
                                   ChannelArguments());
}

std::shared_ptr<grpc::Channel> CreateCustomBinderChannel(
    void* jni_env_void, jobject context, absl::string_view uri,
    std::shared_ptr<grpc::experimental::binder::SecurityPolicy> security_policy,
    const ChannelArguments& args) {
  grpc_init();

  CHECK_NE(jni_env_void, nullptr);
  CHECK_NE(security_policy, nullptr);

  // Generate an unique connection ID that identifies this connection (Useful
  // for mapping connection between Java and C++ code).
  std::string connection_id =
      grpc_binder::GetConnectionIdGenerator()->Generate(uri);

  LOG(ERROR) << "connection id is " << connection_id;

  // After invoking this Java method, Java code will put endpoint binder into
  // `EndpointBinderPool` after the connection succeeds
  // TODO(mingcl): Consider if we want to delay the connection establishment
  // until SubchannelConnector start establishing connection. For now we don't
  // see any benifits doing that.
  grpc_binder::TryEstablishConnectionWithUri(static_cast<JNIEnv*>(jni_env_void),
                                             context, uri, connection_id);

  grpc_binder::GetSecurityPolicySetting()->Set(connection_id, security_policy);

  // Set server URI to a URI that contains connection id. The URI will be used
  // by subchannel connector to obtain correct endpoint binder from
  // `EndpointBinderPool`.
  string server_uri = "binder:" + connection_id;

  grpc_channel_args channel_args;
  args.SetChannelArgs(&channel_args);

  return CreateChannelInternal(
      "",
      grpc::internal::CreateClientBinderChannelImpl(server_uri, &channel_args),
      std::vector<
          std::unique_ptr<experimental::ClientInterceptorFactoryInterface>>());
}

bool InitializeBinderChannelJavaClass(void* jni_env_void) {
  return grpc_binder::FindNativeConnectionHelper(
             static_cast<JNIEnv*>(jni_env_void)) != nullptr;
}

bool InitializeBinderChannelJavaClass(
    void* jni_env_void, std::function<void*(std::string)> class_finder) {
  return grpc_binder::FindNativeConnectionHelper(
             static_cast<JNIEnv*>(jni_env_void), class_finder) != nullptr;
}

}  // namespace experimental
}  // namespace grpc

#else  // !GPR_SUPPORT_BINDER_TRANSPORT

namespace grpc {
namespace experimental {

std::shared_ptr<grpc::Channel> CreateBinderChannel(
    void*, jobject, absl::string_view, absl::string_view,
    std::shared_ptr<grpc::experimental::binder::SecurityPolicy>) {
  LOG(ERROR) << "This APK is compiled with Android API level = "
             << __ANDROID_API__
             << ", which is not supported. See port_platform.h for supported "
                "versions.";
  CHECK(0);
  return {};
}

std::shared_ptr<grpc::Channel> CreateCustomBinderChannel(
    void*, jobject, absl::string_view, absl::string_view,
    std::shared_ptr<grpc::experimental::binder::SecurityPolicy>,
    const ChannelArguments&) {
  LOG(ERROR) << "This APK is compiled with Android API level = "
             << __ANDROID_API__
             << ", which is not supported. See port_platform.h for supported "
                "versions.";
  CHECK(0);
  return {};
}

std::shared_ptr<grpc::Channel> CreateBinderChannel(
    void*, jobject, absl::string_view,
    std::shared_ptr<grpc::experimental::binder::SecurityPolicy>) {
  LOG(ERROR) << "This APK is compiled with Android API level = "
             << __ANDROID_API__
             << ", which is not supported. See port_platform.h for supported "
                "versions.";
  CHECK(0);
  return {};
}

std::shared_ptr<grpc::Channel> CreateCustomBinderChannel(
    void*, jobject, absl::string_view,
    std::shared_ptr<grpc::experimental::binder::SecurityPolicy>,
    const ChannelArguments&) {
  LOG(ERROR) << "This APK is compiled with Android API level = "
             << __ANDROID_API__
             << ", which is not supported. See port_platform.h for supported "
                "versions.";
  CHECK(0);
  return {};
}

bool InitializeBinderChannelJavaClass(void* jni_env_void) {
  LOG(ERROR) << "This APK is compiled with Android API level = "
             << __ANDROID_API__
             << ", which is not supported. See port_platform.h for supported "
                "versions.";
  CHECK(0);
  return {};
}

bool InitializeBinderChannelJavaClass(
    void* jni_env_void, std::function<void*(std::string)> class_finder) {
  LOG(ERROR) << "This APK is compiled with Android API level = "
             << __ANDROID_API__
             << ", which is not supported. See port_platform.h for supported "
                "versions.";
  CHECK(0);
  return {};
}

}  // namespace experimental
}  // namespace grpc

#endif  // GPR_SUPPORT_BINDER_TRANSPORT

#endif  // GPR_ANDROID

#endif  // GRPC_NO_BINDER
