--
--  Copyright (C) 2024, AdaCore
--
--  SPDX-License-Identifier: Apache-2.0 WITH LLVM-Exception
--

with Ada.Containers.Indefinite_Ordered_Maps;

limited with GPR2.Build.Actions.Compile.Ada;
limited with GPR2.Build.Actions.Post_Bind;
with GPR2.Build.Artifacts.Files;
with GPR2.Build.Compilation_Unit.Maps;
with GPR2.Path_Name; use GPR2.Path_Name;

with GNATCOLL.OS.Process;

package GPR2.Build.Actions.Ada_Bind is

   type Ada_Bind_Id (<>) is new Actions.Action_Id with private;

   type Object is new Actions.Object with private;

   Undefined : constant Object;

   function Is_Defined (Self : Object) return Boolean;

   overriding function UID (Self : Object) return Actions.Action_Id'Class;

   procedure Initialize
     (Self           : in out Object;
      Basename       : Simple_Name;
      Context        : GPR2.Project.View.Object;
      Main_Unit      : GPR2.Build.Compilation_Unit.Object;
      SAL_In_Closure : Boolean;
      Skip           : Boolean := False);
   --  Basename:  will produce b__<basename>.ad[bs]
   --  Context:   the view responsible for the bind action
   --  Main_Unit: the main unit when binding an executable. May be undefined
   --             if the entry point is not in Ada or when binding a
   --             standalone library.
   --  Skip:      when set, this will skip the bind/post-bind stages
   --             and will consider the action valid (e.g. the signature
   --             is not checked). Used with --no-sal-binding option.

   procedure Add_Root_Unit
     (Self : in out Object;
      Root : GPR2.Build.Compilation_Unit.Object);

   overriding function On_Tree_Insertion
     (Self : Object;
      Db   : in out GPR2.Build.Tree_Db.Object) return Boolean;

   overriding procedure Compute_Command
     (Self           : in out Object;
      Slot           : Positive;
      Cmd_Line       : in out GPR2.Build.Command_Line.Object;
      Signature_Only : Boolean);

   overriding procedure Compute_Response_Files
     (Self           : in out Object;
      Cmd_Line       : in out GPR2.Build.Command_Line.Object);

   overriding function Working_Directory
     (Self : Object) return Path_Name.Object;

   function Linker_Options
     (Self : Object) return GNATCOLL.OS.Process.Argument_List;
   --  Get the linker options generated by the binder

   function Generated_Spec (Self : Object) return Artifacts.Files.Object;
   function Generated_Body (Self : Object) return Artifacts.Files.Object;
   function Post_Bind (Self : Object) return Actions.Post_Bind.Object;

   overriding function On_Tree_Propagation
     (Self : in out Object) return Boolean;
   --  Used to perform an initial population of the bind dependencies from
   --  explicitly withed units of the known entry points

   function On_Ali_Parsed
     (Self : in out Object;
      Comp : GPR2.Build.Actions.Compile.Ada.Object) return Boolean;
   --  Propagated the Ada imports to the tree. In case this actually comes from
   --  an actual up-to-date ALI file, Sound_Input is set, else it must not, so
   --  that further updates are taken into account.

   function Extended_Interface
     (Self : Object) return Compilation_Unit.Maps.Map;

private

   No_Binder_Found : exception;

   type Ada_Bind_Id (Name_Len : Natural) is new Actions.Action_Id
     with record
      Ctxt : GPR2.Project.View.Object;
      BN   : Filename_Type (1 .. Name_Len);
   end record;

   overriding function View (Self : Ada_Bind_Id) return Project.View.Object is
     (Self.Ctxt);

   overriding function Action_Class (Self : Ada_Bind_Id) return Value_Type is
     ("Bind");

   overriding function Language (Self : Ada_Bind_Id) return Language_Id is
     (Ada_Language);

   overriding function Action_Parameter (Self : Ada_Bind_Id) return Value_Type
   is (Value_Type (Self.BN));

   function Less (L, R : Compilation_Unit.Object) return Boolean is
     (L.Name < R.Name);

   package Extended_Interface_Map is new Ada.Containers.Indefinite_Ordered_Maps
     (Compilation_Unit.Object, Name_Type, Less);

   type Object is new Actions.Object with record
      Basename    : Unbounded_String;
      --  Basename used to generate b__<bn>,ads and b__<bn>.adb
      Output_Spec : Artifacts.Files.Object;
      --  The generated spec, as artifact
      Output_Body : Artifacts.Files.Object;
      --  The generated body, as artifact
      Ctxt        : GPR2.Project.View.Object;
      --  View referenced by the generated compilation unit
      Has_Main    : Boolean := True;
      --  Whether the binder generates a main entry point
      SAL_Closure : Boolean := False;
      --  Whether the main depends on standalone libraries, in which case the
      --  binder needs to handle potentially duplicated elaboration
      Linker_Opts : GNATCOLL.OS.Process.Argument_List;
      --  Linker options generated by the binder in the generated body
      Analyzed    : GPR2.Containers.Name_Set;
      --  List of units that are part of the closure of the gnatbind
      --  analysis, and for which a valid ALI file has been analyzed.
      Pre_Analyzed : GPR2.Containers.Name_Set;
      --  Same as Analyzed but filled when just the Ada parser is available.
      --  This allows proper pruning of the added dependencies to lower the
      --  algorithm cost
      Itf_Analyzed : GPR2.Containers.Name_Set;
      --  Same as Analyzed but filled when checking for interface extension.
      Roots       : GPR2.Build.Compilation_Unit.Maps.Map;
      --  The root units as explicit inputs to the binder
      Extra_Intf  : Extended_Interface_Map.Map;
      Skip        : Boolean := False;
   end record;

   overriding function Valid_Signature (Self : Object) return Boolean is
     (Self.Skip or else GPR2.Build.Actions.Object (Self).Valid_Signature);

   overriding procedure Compute_Signature
     (Self      : in out Object;
      Load_Mode : Boolean);

   overriding function Post_Command
     (Self   : in out Object;
      Status : Execution_Status;
      Stdout : Unbounded_String := Null_Unbounded_String;
      Stderr : Unbounded_String := Null_Unbounded_String) return Boolean;

   overriding function View (Self : Object) return GPR2.Project.View.Object is
     (Self.Ctxt);

   overriding function Extended (Self : Object) return Object is
     (raise Internal_Error with "This action is not extending");

   function Generated_Spec (Self : Object) return Artifacts.Files.Object is
      (Self.Output_Spec);

   function Generated_Body (Self : Object) return Artifacts.Files.Object is
      (Self.Output_Body);

   Undefined : constant Object := (others => <>);

   function Is_Defined (Self : Object) return Boolean is
     (Self /= Undefined);

   function Linker_Options
     (Self : Object) return GNATCOLL.OS.Process.Argument_List is
     (Self.Linker_Opts);

   overriding function Working_Directory
     (Self : Object) return Path_Name.Object is
     (Self.Ctxt.Object_Directory);

   function Hash
     (CU : Compilation_Unit.Object)
      return Ada.Containers.Hash_Type is
      (GPR2.Hash (CU.Name));

end GPR2.Build.Actions.Ada_Bind;
