! This file is part of mstore.
! SPDX-Identifier: Apache-2.0
!
! Licensed under the Apache License, Version 2.0 (the "License");
! you may not use this file except in compliance with the License.
! You may obtain a copy of the License at
!
!     http://www.apache.org/licenses/LICENSE-2.0
!
! Unless required by applicable law or agreed to in writing, software
! distributed under the License is distributed on an "AS IS" BASIS,
! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
! See the License for the specific language governing permissions and
! limitations under the License.

module mstore_amino20x4
   use mctc_env_accuracy, only : wp
   use mctc_io_structure, only : structure_type, new
   use mstore_data_record, only : record_type, new_record, select_record
   implicit none
   private

   public :: get_amino20x4_records

contains

subroutine get_amino20x4_records(records)
   type(record_type), allocatable, intent(out) :: records(:)

   records = [ &
      new_record("ALA_xab", ALA_xab), &
      new_record("ALA_xac", ALA_xac), &
      new_record("ALA_xag", ALA_xag), &
      new_record("ALA_xai", ALA_xai), &
      new_record("ALA_xak", ALA_xak), &
      new_record("ARG_xak", ARG_xak), &
      new_record("ARG_xbv", ARG_xbv), &
      new_record("ARG_xbx", ARG_xbx), &
      new_record("ARG_xby", ARG_xby), &
      new_record("ARG_xci", ARG_xci), &
      new_record("ASN_xab", ASN_xab), &
      new_record("ASN_xae", ASN_xae), &
      new_record("ASN_xaf", ASN_xaf), &
      new_record("ASN_xah", ASN_xah), &
      new_record("ASN_xaj", ASN_xaj), &
      new_record("ASP_xad", ASP_xad), &
      new_record("ASP_xau", ASP_xau), &
      new_record("ASP_xay", ASP_xay), &
      new_record("ASP_xaz", ASP_xaz), &
      new_record("ASP_xbc", ASP_xbc), &
      new_record("CYS_xag", CYS_xag), &
      new_record("CYS_xah", CYS_xah), &
      new_record("CYS_xai", CYS_xai), &
      new_record("CYS_xal", CYS_xal), &
      new_record("CYS_xao", CYS_xao), &
      new_record("GLN_xai", GLN_xai), &
      new_record("GLN_xal", GLN_xal), &
      new_record("GLN_xan", GLN_xan), &
      new_record("GLN_xap", GLN_xap), &
      new_record("GLN_xat", GLN_xat), &
      new_record("GLU_xad", GLU_xad), &
      new_record("GLU_xal", GLU_xal), &
      new_record("GLU_xar", GLU_xar), &
      new_record("GLU_xav", GLU_xav), &
      new_record("GLU_xbi", GLU_xbi), &
      new_record("GLY_xab", GLY_xab), &
      new_record("GLY_xac", GLY_xac), &
      new_record("GLY_xad", GLY_xad), &
      new_record("GLY_xae", GLY_xae), &
      new_record("GLY_xag", GLY_xag), &
      new_record("HIS_xah", HIS_xah), &
      new_record("HIS_xam", HIS_xam), &
      new_record("HIS_xaq", HIS_xaq), &
      new_record("HIS_xau", HIS_xau), &
      new_record("HIS_xav", HIS_xav), &
      new_record("ILE_xae", ILE_xae), &
      new_record("ILE_xag", ILE_xag), &
      new_record("ILE_xaj", ILE_xaj), &
      new_record("ILE_xak", ILE_xak), &
      new_record("ILE_xaq", ILE_xaq), &
      new_record("LEU_xad", LEU_xad), &
      new_record("LEU_xae", LEU_xae), &
      new_record("LEU_xap", LEU_xap), &
      new_record("LEU_xaq", LEU_xaq), &
      new_record("LEU_xbb", LEU_xbb), &
      new_record("LYS_xan", LYS_xan), &
      new_record("LYS_xao", LYS_xao), &
      new_record("LYS_xap", LYS_xap), &
      new_record("LYS_xas", LYS_xas), &
      new_record("LYS_xat", LYS_xat), &
      new_record("MET_xag", MET_xag), &
      new_record("MET_xav", MET_xav), &
      new_record("MET_xbf", MET_xbf), &
      new_record("MET_xbm", MET_xbm), &
      new_record("MET_xbo", MET_xbo), &
      new_record("PHE_xab", PHE_xab), &
      new_record("PHE_xal", PHE_xal), &
      new_record("PHE_xan", PHE_xan), &
      new_record("PHE_xar", PHE_xar), &
      new_record("PHE_xaw", PHE_xaw), &
      new_record("PRO_xab", PRO_xab), &
      new_record("PRO_xac", PRO_xac), &
      new_record("PRO_xad", PRO_xad), &
      new_record("PRO_xae", PRO_xae), &
      new_record("PRO_xaf", PRO_xaf), &
      new_record("SER_xad", SER_xad), &
      new_record("SER_xaf", SER_xaf), &
      new_record("SER_xah", SER_xah), &
      new_record("SER_xak", SER_xak), &
      new_record("SER_xar", SER_xar), &
      new_record("THR_xab", THR_xab), &
      new_record("THR_xag", THR_xag), &
      new_record("THR_xah", THR_xah), &
      new_record("THR_xal", THR_xal), &
      new_record("THR_xaq", THR_xaq), &
      new_record("TRP_xac", TRP_xac), &
      new_record("TRP_xaf", TRP_xaf), &
      new_record("TRP_xag", TRP_xag), &
      new_record("TRP_xah", TRP_xah), &
      new_record("TRP_xao", TRP_xao), &
      new_record("TYR_xab", TYR_xab), &
      new_record("TYR_xag", TYR_xag), &
      new_record("TYR_xah", TYR_xah), &
      new_record("TYR_xan", TYR_xan), &
      new_record("TYR_xar", TYR_xar), &
      new_record("VAL_xad", VAL_xad), &
      new_record("VAL_xaf", VAL_xaf), &
      new_record("VAL_xah", VAL_xah), &
      new_record("VAL_xaj", VAL_xaj), &
      new_record("VAL_xak", VAL_xak) &
      ]

end subroutine get_amino20x4_records

subroutine ALA_xab(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 22
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "H", "O", "C", "C", "H", "H", &
      & "H", "O", "N", "H", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.65893135608838_wp, -2.39249423371715_wp, -3.66065400053935_wp, &
      &  3.49612941769371_wp, -0.88484673975624_wp, -2.85194146578362_wp, &
      & -0.06354076626069_wp, -2.63180732150005_wp, -3.28819116275323_wp, &
      & -1.07444177498884_wp, -1.92306930149582_wp, -4.93716401361053_wp, &
      & -0.83329925447427_wp, -5.37320588052218_wp, -2.81379718546920_wp, &
      & -0.90691285352090_wp, -1.04371377845950_wp, -1.04918016247507_wp, &
      & -2.86418317801214_wp, -5.46484901686185_wp, -2.49961410229771_wp, &
      & -0.34235262692151_wp, -6.52310417728877_wp, -4.43935278498325_wp, &
      &  0.13208660968384_wp, -6.10946566962768_wp, -1.15032982743173_wp, &
      & -2.96060093623907_wp,  0.01043357425890_wp, -0.99937552379387_wp, &
      &  3.76519127865000_wp, -3.27106236675729_wp, -5.83678272799149_wp, &
      &  6.47957316843231_wp, -2.46911747464509_wp, -6.21176914665408_wp, &
      &  7.32688324906998_wp, -1.67889171278096_wp, -4.51496113512671_wp, &
      &  6.54881843238363_wp, -1.06760660462911_wp, -7.71597456720663_wp, &
      &  7.56369260941896_wp, -4.10015651865148_wp, -6.82588105651977_wp, &
      &  2.64916867837331_wp, -4.60764575400925_wp, -7.35167957128511_wp, &
      &  0.77231592220237_wp, -0.92788783332000_wp,  0.90692539619101_wp, &
      &  2.18437036702702_wp, -2.20200039553542_wp,  0.92105755612696_wp, &
      &  0.01367202674183_wp,  0.22095199845428_wp,  3.27728206652909_wp, &
      &  1.67849497305706_wp,  0.53855308534857_wp,  4.43416031916610_wp, &
      & -0.89254709011762_wp,  2.01704896333243_wp,  2.87780123699499_wp, &
      & -1.32658751691561_wp, -0.95404596601807_wp,  4.30967630773603_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ALA_xab

subroutine ALA_xac(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 22
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "H", "O", "C", "C", "H", "H", &
      & "H", "O", "N", "H", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.89812994422336_wp, -3.49785066920537_wp, -4.04176475935607_wp, &
      &  0.22096411844816_wp, -4.86653879303702_wp, -5.16965309356488_wp, &
      &  1.50858972981055_wp, -4.22297129418255_wp, -1.44296740345170_wp, &
      &  1.26381263207546_wp, -6.26173130725955_wp, -1.36929929792062_wp, &
      &  4.22584479532188_wp, -3.58094126128602_wp, -0.72765628187693_wp, &
      & -0.36535595783691_wp, -3.23105046121391_wp,  0.53224473210756_wp, &
      &  4.59700287054853_wp, -4.26587358691560_wp,  1.17638157393339_wp, &
      &  5.53245639246579_wp, -4.49962127256050_wp, -2.02509339376622_wp, &
      &  4.54878065243139_wp, -1.55600067058432_wp, -0.78659473595735_wp, &
      & -1.40338188617737_wp, -4.70093312894944_wp,  1.99692732352687_wp, &
      &  1.19184541389054_wp, -1.17965860262351_wp, -5.08564964981169_wp, &
      &  0.66646223547991_wp, -1.02784609220985_wp, -7.88469520699739_wp, &
      &  2.46126071643208_wp, -0.84993899853549_wp, -8.87377461956087_wp, &
      & -0.42509529147581_wp,  0.67053350705954_wp, -8.25313731918547_wp, &
      & -0.33603421332938_wp, -2.66534515804911_wp, -8.61567815458241_wp, &
      &  1.87087395689416_wp,  0.69377329947433_wp, -3.88238849685894_wp, &
      & -0.66896549096126_wp, -0.70041907517856_wp,  0.57158466492815_wp, &
      &  0.24390855421877_wp,  0.32329261951026_wp, -0.76688726802653_wp, &
      & -2.34873581523733_wp,  0.49815011834652_wp,  2.36364902826355_wp, &
      & -2.51669513442034_wp, -0.71675917183750_wp,  4.00880640863199_wp, &
      & -1.57466603805426_wp,  2.31572781563081_wp,  2.92500014923521_wp, &
      & -4.23036142386086_wp,  0.78917833057040_wp,  1.57683879578790_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ALA_xac

subroutine ALA_xag(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 22
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "H", "O", "C", "C", "H", "H", &
      & "H", "O", "N", "H", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.98293112719749_wp, -0.68900404398734_wp,  0.37628864997560_wp, &
      & -1.54035272113532_wp, -2.33144052962209_wp,  1.23491020893064_wp, &
      &  0.09692928120200_wp,  0.99422262169598_wp, -0.11989378229633_wp, &
      & -0.09901909722504_wp,  1.72588002674889_wp, -2.03875555767431_wp, &
      &  0.14141827166397_wp,  3.20957080884314_wp,  1.73528063332959_wp, &
      &  2.50745780392154_wp, -0.55594193008926_wp,  0.12427542626420_wp, &
      &  1.69093368632231_wp,  4.49899142526548_wp,  1.31794453504710_wp, &
      & -1.63338894984043_wp,  4.23428690562497_wp,  1.59126592199206_wp, &
      &  0.36779723029067_wp,  2.50163220083067_wp,  3.65420914638345_wp, &
      &  2.56229717302803_wp, -2.53711155537531_wp,  1.32871903135090_wp, &
      & -4.36159407476856_wp, -0.12962702640436_wp, -0.39562772950648_wp, &
      & -6.34100574546165_wp, -2.09075933777830_wp,  0.21330645338918_wp, &
      & -7.23610050442420_wp, -2.68187533006900_wp, -1.53914540310063_wp, &
      & -7.78623541548118_wp, -1.21956781217005_wp,  1.38592307383011_wp, &
      & -5.57984815064465_wp, -3.73504338108486_wp,  1.18089285147183_wp, &
      & -4.86717200167912_wp,  1.83918213313466_wp, -1.51162714188889_wp, &
      &  4.55965923505744_wp,  0.48183329726772_wp, -0.99463516518498_wp, &
      &  4.30889971335617_wp,  2.04332235450700_wp, -2.04508383499641_wp, &
      &  7.03944933546151_wp, -0.66526542357055_wp, -0.85849489668778_wp, &
      &  8.44432563879401_wp,  0.71932844719313_wp, -0.28192138996626_wp, &
      &  7.59387580143664_wp, -1.47860319740057_wp, -2.66533359746007_wp, &
      &  6.94508476368302_wp, -2.16228363458999_wp,  0.53955258865532_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ALA_xag

subroutine ALA_xai(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 22
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "H", "O", "C", "C", "H", "H", &
      & "H", "O", "N", "H", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.95551647678988_wp, -1.75376552896675_wp, -0.39423746098570_wp, &
      & -1.91644320331392_wp, -3.09789464304491_wp, -1.74463271907330_wp, &
      &  0.45987400323567_wp, -1.12410177663264_wp,  0.73636846819641_wp, &
      &  0.05339140292548_wp, -0.00334180768460_wp,  2.40517517642917_wp, &
      &  1.86642693142553_wp, -3.52603763992019_wp,  1.44755505447877_wp, &
      &  2.03926411711777_wp,  0.35098371923426_wp, -1.17874686426642_wp, &
      &  3.66116478607346_wp, -3.06959857262461_wp,  2.34437902240689_wp, &
      &  0.73935075343686_wp, -4.65155804911349_wp,  2.74764693445837_wp, &
      &  2.26310832839085_wp, -4.62712666492313_wp, -0.24541134770668_wp, &
      &  2.38101211490903_wp, -0.52315843705605_wp, -3.29921789587556_wp, &
      & -3.72638021028698_wp,  0.11052911150071_wp, -0.65293969404915_wp, &
      & -5.93797949775141_wp, -0.53553417569933_wp, -2.32894426606195_wp, &
      & -7.64146756627280_wp,  0.27160175573181_wp, -1.51947545511119_wp, &
      & -6.17329766428632_wp, -2.56113998377180_wp, -2.57713222098262_wp, &
      & -5.63321138595267_wp,  0.33076050133696_wp, -4.17092545951266_wp, &
      & -3.53123737527919_wp,  2.15433348848326_wp,  0.41984792339724_wp, &
      &  3.07873399695339_wp,  2.55677235807315_wp, -0.40292454308350_wp, &
      &  4.07814835229996_wp,  3.44512133141085_wp, -1.75891184333584_wp, &
      &  2.45088801986036_wp,  3.93925785039746_wp,  1.88649543668059_wp, &
      &  3.28474803943292_wp,  5.80918247735094_wp,  1.74573821617921_wp, &
      &  0.41495662313163_wp,  4.14191143879495_wp,  2.08893392641995_wp, &
      &  3.22811383808006_wp,  3.03170960638074_wp,  3.56251180698608_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ALA_xai

subroutine ALA_xak(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 22
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "H", "O", "C", "C", "H", "H", &
      & "H", "O", "N", "H", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.03344209594157_wp,  0.94874045088261_wp, -1.07451287292339_wp, &
      &  2.46605761260186_wp,  1.74973560369532_wp, -2.75139739626457_wp, &
      &  0.02325475159729_wp,  2.05983554639044_wp,  0.41387445859801_wp, &
      &  0.67246150427191_wp,  2.43563604725592_wp,  2.33188161891808_wp, &
      & -0.80002930567209_wp,  4.54354433743522_wp, -0.78677610015840_wp, &
      & -2.31473952232019_wp,  0.38057757938484_wp,  0.70526673749029_wp, &
      & -2.34334723686566_wp,  5.35056089290866_wp,  0.29978395439111_wp, &
      &  0.76681816951430_wp,  5.87515996291786_wp, -0.83371775175150_wp, &
      & -1.45696746224491_wp,  4.21826136488519_wp, -2.71271613381049_wp, &
      & -3.92810866361723_wp,  0.90564788680233_wp,  2.28032225692816_wp, &
      &  3.82053085336039_wp, -0.74039954698721_wp, -0.23187424033844_wp, &
      &  3.44268840727487_wp, -1.73084458827947_wp,  2.41359232971359_wp, &
      &  3.84180719123463_wp, -0.24792474720280_wp,  3.78592455409350_wp, &
      &  4.75264169686898_wp, -3.27842143418951_wp,  2.70756523069823_wp, &
      &  1.51338793250931_wp, -2.37187473518562_wp,  2.72387488318668_wp, &
      &  5.58403473443618_wp, -1.37889165823908_wp, -1.57705433714131_wp, &
      & -2.45054416487403_wp, -1.59281740584842_wp, -0.91212416478343_wp, &
      & -0.98491627112049_wp, -1.83668497628644_wp, -2.09834204260762_wp, &
      & -4.57555431581973_wp, -3.31098801739238_wp, -0.93701145636929_wp, &
      & -3.94364614751308_wp, -5.25519727712234_wp, -0.72043135706702_wp, &
      & -5.78598251191709_wp, -2.80871211428581_wp,  0.64153742645889_wp, &
      & -5.65031917116663_wp, -3.14504039910972_wp, -2.68342063820897_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ALA_xak

subroutine ARG_xak(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 35
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "H", "H", "C", "H", &
      & "H", "N", "H", "C", "N", "N", "H", "H", "H", "N", "C", "H", "H", "H", &
      & "H", "C", "C", "O", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  4.67785833080939_wp,  0.31634028410585_wp,  0.80133285105224_wp, &
      &  5.27440218546968_wp,  2.00268561080735_wp,  1.47309895648034_wp, &
      &  2.36244051553551_wp,  0.43560131062116_wp, -0.63776088765558_wp, &
      &  2.80741614095648_wp,  0.12329965464254_wp, -2.63647737861516_wp, &
      &  0.44395394320743_wp, -1.58033312844574_wp,  0.17267877696471_wp, &
      &  1.39884893257802_wp,  3.14982805421985_wp, -0.36963517016416_wp, &
      &  1.48874012948782_wp, -3.15287861133957_wp,  0.99191770565525_wp, &
      & -0.78217019431279_wp, -0.80331309691274_wp,  1.63890807489788_wp, &
      & -1.09200346456729_wp, -2.52962658422687_wp, -2.06862321890780_wp, &
      &  2.81700587757621_wp,  4.85169574098129_wp,  0.34908567208630_wp, &
      &  0.15523570669850_wp, -3.64195718059480_wp, -3.27952743483402_wp, &
      & -1.78084219476241_wp, -0.96195660257043_wp, -3.20456706410493_wp, &
      & -3.34076914579272_wp, -4.18270785875701_wp, -1.32847816412898_wp, &
      & -2.68526098195640_wp, -6.05473956745358_wp, -0.77494626844965_wp, &
      & -4.57672858022238_wp, -4.41370645023403_wp, -2.95951312143008_wp, &
      & -4.86338189210036_wp, -3.27027441544487_wp,  0.77230696910501_wp, &
      & -4.34631637843670_wp, -3.85190498994461_wp,  2.50714732479184_wp, &
      & -6.04187503485480_wp, -0.97301399397251_wp,  0.71085903682113_wp, &
      & -5.85501852912762_wp,  0.49802628552170_wp, -1.22868921976204_wp, &
      & -7.33297526724018_wp, -0.45160627616224_wp,  2.94699818534936_wp, &
      & -7.07379415966069_wp,  1.97032782778335_wp, -1.02961918365714_wp, &
      & -8.44470427441131_wp,  1.08915170851111_wp,  2.84726246744810_wp, &
      & -8.25516799085994_wp, -1.94949119017541_wp,  3.67946900193538_wp, &
      & -1.02669655742152_wp,  3.57394709224334_wp, -1.00724514824551_wp, &
      & -2.05045456702119_wp,  6.10185040840134_wp, -0.78712648377824_wp, &
      & -2.26918859158819_wp,  2.14366599537278_wp, -1.30229812005974_wp, &
      & -0.80983112667169_wp,  7.44355792020126_wp, -1.72435677194051_wp, &
      & -2.23673108785656_wp,  6.68037345695427_wp,  1.18132924406105_wp, &
      & -3.89489781007265_wp,  6.13514324867668_wp, -1.68701846730634_wp, &
      &  6.34251748343694_wp, -1.59210938293046_wp,  0.38908360980326_wp, &
      &  8.86232216695947_wp, -1.32650050892525_wp,  1.70046406131034_wp, &
      &  5.82464523259722_wp, -3.42420511904025_wp, -0.93871447592074_wp, &
      &  8.93312719725664_wp,  0.27575041643516_wp,  2.98370684019120_wp, &
      &  9.25068265454150_wp, -3.05608201635353_wp,  2.73788270576952_wp, &
      & 10.32405072748436_wp, -1.11650496709526_wp,  0.26956110336282_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ARG_xak

subroutine ARG_xbv(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 35
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "H", "H", "C", "H", &
      & "H", "N", "H", "C", "N", "N", "H", "H", "H", "N", "C", "H", "H", "H", &
      & "H", "C", "C", "O", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  4.75363399573858_wp, -0.44616980191452_wp,  0.80360897118234_wp, &
      &  5.56843944315359_wp,  0.84084379626982_wp,  1.95660054332154_wp, &
      &  2.37757650057151_wp,  0.33222027993645_wp, -0.27539179174895_wp, &
      &  2.63332686572101_wp,  0.60960885900706_wp, -2.31094818221742_wp, &
      &  0.27746178214888_wp, -1.64318865449477_wp,  0.10689093450910_wp, &
      &  1.75901703108944_wp,  2.90101952816397_wp,  0.90932775413428_wp, &
      &  1.16482162169608_wp, -3.44051226819710_wp,  0.57071367180712_wp, &
      & -0.88951452811006_wp, -1.08043290181278_wp,  1.71136775599760_wp, &
      & -1.31764188777826_wp, -1.96244325570692_wp, -2.27114048375581_wp, &
      &  3.33866494066299_wp,  4.04660214596171_wp,  2.17752193454236_wp, &
      & -0.20605213368749_wp, -2.99211531157435_wp, -3.67040377701585_wp, &
      & -1.70336383944564_wp, -0.11576700186448_wp, -3.09904785232323_wp, &
      & -3.80159875613321_wp, -3.37676152465223_wp, -1.82953170162827_wp, &
      & -3.42321090176585_wp, -5.37038881029678_wp, -1.48589749065209_wp, &
      & -4.97783402623313_wp, -3.28699486684533_wp, -3.52080095385243_wp, &
      & -5.26698344842186_wp, -2.48716789425457_wp,  0.33254645781463_wp, &
      & -4.74559236570610_wp, -3.29030382395835_wp,  1.97933767198946_wp, &
      & -5.80512302816535_wp,  0.05037336377029_wp,  0.57487126961459_wp, &
      & -5.86112199997502_wp,  1.73083961491058_wp, -1.18820428060445_wp, &
      & -6.23226920434863_wp,  0.79758347655108_wp,  3.05673689932252_wp, &
      & -5.46722333920413_wp,  0.92469760396003_wp, -2.88264844034456_wp, &
      & -7.13014084404128_wp, -0.49816082352756_wp,  4.12679142033351_wp, &
      & -7.10462595886192_wp,  2.49132157311537_wp,  3.11508774841924_wp, &
      & -0.56407059824300_wp,  3.83074326018609_wp,  0.45664813843558_wp, &
      & -1.33583370765807_wp,  6.20578630212034_wp,  1.57992284494703_wp, &
      & -1.93145238505377_wp,  2.78684817747547_wp, -0.37224146578306_wp, &
      & -1.35219181525958_wp,  6.08489793139719_wp,  3.63487611352742_wp, &
      & -3.22191083863002_wp,  6.63910915249255_wp,  0.89869525732363_wp, &
      & -0.04030465416311_wp,  7.70597732563175_wp,  1.03891815774414_wp, &
      &  6.09944117111300_wp, -2.33021833628209_wp, -0.29267105295554_wp, &
      &  8.70477487635957_wp, -2.76954229271466_wp,  0.78059536259640_wp, &
      &  5.24161645431839_wp, -3.60548536925093_wp, -2.03436821962049_wp, &
      & 10.09430419788398_wp, -2.27481145133243_wp, -0.65226758093036_wp, &
      &  9.06527817927830_wp, -1.67298958072279_wp,  2.47957384012291_wp, &
      &  8.91943439978551_wp, -4.76993550575649_wp,  1.19267081904664_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ARG_xbv

subroutine ARG_xbx(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 35
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "H", "H", "C", "H", &
      & "H", "N", "H", "C", "N", "N", "H", "H", "H", "N", "C", "H", "H", "H", &
      & "H", "C", "C", "O", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  4.84948422615137_wp, -0.50442054677881_wp,  0.90505618479626_wp, &
      &  5.54058153635980_wp,  0.68156914890200_wp,  2.23231828748076_wp, &
      &  2.44752775287185_wp,  0.21978571967831_wp, -0.15065490166900_wp, &
      &  2.70411079159939_wp,  0.62665988263516_wp, -2.16301945784872_wp, &
      &  0.45081840327216_wp, -1.88148180682339_wp,  0.10010679645794_wp, &
      &  1.67582971009609_wp,  2.65772505017049_wp,  1.20337707123805_wp, &
      &  1.44258982498166_wp, -3.66592949655018_wp,  0.34637626011224_wp, &
      & -0.68818966696681_wp, -1.54959228583443_wp,  1.78522832678615_wp, &
      & -1.23822350548843_wp, -2.05912875005441_wp, -2.23011674351576_wp, &
      &  3.08987669666499_wp,  3.68018312335633_wp,  2.74364898432547_wp, &
      & -0.25219051854030_wp, -3.12628844243484_wp, -3.69266702303217_wp, &
      & -1.55706649360367_wp, -0.18536936278730_wp, -3.02315826070715_wp, &
      & -3.78247885088563_wp, -3.31957111745312_wp, -1.71099956298129_wp, &
      & -3.47489234454339_wp, -5.27393569345533_wp, -1.14234629906514_wp, &
      & -4.91089123144112_wp, -3.36527207502034_wp, -3.44446875020300_wp, &
      & -5.15146274614030_wp, -2.11427633848385_wp,  0.33403068131090_wp, &
      & -5.98771234298082_wp, -3.21303182942779_wp,  1.63756528192655_wp, &
      & -5.90921079359766_wp,  0.35761924651370_wp,  0.25205550364317_wp, &
      & -5.76743469650082_wp,  1.89365447610033_wp, -1.63912745639023_wp, &
      & -6.89828726948542_wp,  1.19292291561413_wp,  2.53469390335101_wp, &
      & -5.36274629284895_wp,  0.92377826558286_wp, -3.24549935234570_wp, &
      & -7.29848003152813_wp,  3.05428833304807_wp,  2.47090364294397_wp, &
      & -5.86807960527482_wp,  0.71283499951806_wp,  4.06376733306730_wp, &
      & -0.61320720637900_wp,  3.60716897031544_wp,  0.62846751639692_wp, &
      & -1.50108471685347_wp,  5.89693476493258_wp,  1.83791733209180_wp, &
      & -1.83271826000691_wp,  2.71953782432903_wp, -0.54117094398446_wp, &
      & -0.15701904703184_wp,  7.42621951461426_wp,  1.55707831412566_wp, &
      & -1.73443069767184_wp,  5.62637558731530_wp,  3.86576717810926_wp, &
      & -3.30261956566337_wp,  6.39361508042545_wp,  0.99078156886290_wp, &
      &  6.32119506642687_wp, -2.21664616679375_wp, -0.30361314478907_wp, &
      &  8.90859696313147_wp, -2.63288269863376_wp,  0.82102362948921_wp, &
      &  5.58234301331341_wp, -3.37097060516684_wp, -2.17869910417725_wp, &
      & 10.32067028227628_wp, -2.11891416146787_wp, -0.58148792060249_wp, &
      &  9.22400729481644_wp, -1.54661680373970_wp,  2.53551019732375_wp, &
      &  9.13518327418153_wp, -4.63393636272951_wp,  1.22736385426752_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ARG_xbx

subroutine ARG_xby(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 35
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "H", "H", "C", "H", &
      & "H", "N", "H", "C", "N", "N", "H", "H", "H", "N", "C", "H", "H", "H", &
      & "H", "C", "C", "O", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -4.45105392510503_wp, -1.59739508618354_wp, -1.03654796853476_wp, &
      & -4.45271946810651_wp, -2.81168604931023_wp, -2.50941199146143_wp, &
      & -2.30913783859582_wp, -1.65712431681358_wp,  0.66465483176143_wp, &
      & -2.85132079581481_wp, -2.28929957559123_wp,  2.55219870282891_wp, &
      & -0.35565642045585_wp, -3.50333046180867_wp, -0.42234450518454_wp, &
      & -1.06408362413164_wp,  0.92080529924330_wp,  1.01419655867710_wp, &
      & -1.30549144051692_wp, -5.31679511870411_wp, -0.67514453364353_wp, &
      &  0.15857868749740_wp, -2.80845537919445_wp, -2.29785540580247_wp, &
      &  2.03209745259328_wp, -3.81190574950132_wp,  1.19287178172225_wp, &
      &  0.07722699832818_wp,  1.42380638230635_wp,  2.98470782187375_wp, &
      &  2.17483535997977_wp, -5.74585573712424_wp,  1.89071143765542_wp, &
      &  1.92601938872902_wp, -2.59437156618478_wp,  2.84606046420392_wp, &
      &  4.43312371017632_wp, -3.20425155672192_wp, -0.28506482253174_wp, &
      &  4.62950152053949_wp, -4.56048614121386_wp, -1.82523900355595_wp, &
      &  6.07607561725505_wp, -3.41104270887958_wp,  0.94155178122328_wp, &
      &  4.39059950654216_wp, -0.68094784938697_wp, -1.42243131806604_wp, &
      &  4.77648964438246_wp, -0.67137416437189_wp, -3.29090169048692_wp, &
      &  5.73393614038473_wp,  1.29479292948407_wp, -0.32360256391291_wp, &
      &  6.94296026045843_wp,  2.81810923535716_wp, -1.78229251425996_wp, &
      &  5.59179317420557_wp,  1.32426384339337_wp,  2.29626933950876_wp, &
      &  7.62825970605880_wp,  4.29202083681860_wp, -0.75820940951458_wp, &
      &  3.78909402601231_wp,  1.15022280423202_wp,  2.93679743652841_wp, &
      &  6.43028448919416_wp,  2.86411059302247_wp,  3.04247141052621_wp, &
      & -1.25299852978523_wp,  2.50055085558888_wp, -0.96141454134152_wp, &
      &  0.14883709652623_wp,  4.84855470446851_wp, -1.09036481632704_wp, &
      & -2.20401690329424_wp,  1.86153250068434_wp, -2.48047162709028_wp, &
      &  0.42147457590947_wp,  5.53265625203333_wp,  0.82542184069092_wp, &
      & -0.92546408160760_wp,  6.23260620283280_wp, -2.16084586739295_wp, &
      &  1.99329056761263_wp,  4.57278792511889_wp, -1.96620044592356_wp, &
      & -6.80941462789126_wp, -0.62922318791625_wp, -0.49882592837094_wp, &
      & -7.01378622653753_wp,  0.88541927839498_wp,  1.90463222072706_wp, &
      & -8.58345881983099_wp, -0.97311547703451_wp, -1.93304714010506_wp, &
      & -8.94881557060575_wp,  1.53262781784113_wp,  2.08757031958259_wp, &
      & -6.51222613401539_wp, -0.25406191624083_wp,  3.54364810466502_wp, &
      & -5.74159233957036_wp,  2.50196346830902_wp,  1.86140079235464_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ARG_xby

subroutine ARG_xci(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 35
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "H", "H", "C", "H", &
      & "H", "N", "H", "C", "N", "N", "H", "H", "H", "N", "C", "H", "H", "H", &
      & "H", "C", "C", "O", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.64995234040493_wp, -0.21548420313173_wp,  0.20577796173015_wp, &
      &  5.03472511849470_wp,  0.56373383269137_wp,  1.26087844797787_wp, &
      &  1.10791020329174_wp,  0.39283831992809_wp,  0.99631494935956_wp, &
      & -0.06413422098639_wp,  0.37014531951662_wp, -0.68936338627915_wp, &
      &  0.11312415303209_wp, -1.53147836653875_wp,  2.93220827249075_wp, &
      &  1.11399848545454_wp,  3.02211986789374_wp,  2.14195726932333_wp, &
      &  1.68159709302190_wp, -2.01850330596251_wp,  4.17522913879635_wp, &
      & -1.31621629301132_wp, -0.63123020573211_wp,  4.12247145409774_wp, &
      & -0.99608800379033_wp, -3.88493210576964_wp,  1.65944927122987_wp, &
      &  2.90110945996681_wp,  3.78047491279985_wp,  3.42641696329796_wp, &
      & -0.49808776006072_wp, -5.59118532110555_wp,  2.70592088410173_wp, &
      & -0.20905041978140_wp, -4.07797831136997_wp, -0.23617010044503_wp, &
      & -3.88420462446956_wp, -3.78085428820841_wp,  1.51808481950111_wp, &
      & -4.66850455497203_wp, -3.99121842134176_wp,  3.41176607268388_wp, &
      & -4.59391843891644_wp, -5.37433623041809_wp,  0.42210719703345_wp, &
      & -4.94791567391595_wp, -1.43701894814876_wp,  0.52482097052354_wp, &
      & -4.99774115127727_wp,  0.01256375439327_wp,  1.76252139193142_wp, &
      & -4.40018875888481_wp, -0.56339230183663_wp, -1.86312105575365_wp, &
      & -4.12387544628015_wp,  1.84100965908400_wp, -2.21004875310815_wp, &
      & -4.12671841340409_wp, -2.42718057952085_wp, -3.64227137414643_wp, &
      & -3.95932863032575_wp,  2.23764159831742_wp, -4.08034473952088_wp, &
      & -5.18439471370454_wp, -3.99294199593816_wp, -3.46456412329680_wp, &
      & -3.75739912012224_wp, -1.87325267768897_wp, -5.41828255095184_wp, &
      & -1.02666649623783_wp,  4.33282585858492_wp,  1.72187966309664_wp, &
      & -1.31737415662602_wp,  6.85574782409077_wp,  2.73762481114905_wp, &
      & -2.23920513616011_wp,  3.68416446578410_wp,  0.35922279532065_wp, &
      & -0.79233566078356_wp,  6.86847517242492_wp,  4.72357751702048_wp, &
      & -3.27963640062133_wp,  7.42550854782272_wp,  2.54395366781581_wp, &
      & -0.11891446180879_wp,  8.21590652577064_wp,  1.75906085236325_wp, &
      &  4.17297983011164_wp, -1.10721339305136_wp, -2.14423665205732_wp, &
      &  6.94929938350184_wp, -1.25253209841068_wp, -2.78998807689081_wp, &
      &  2.52918268416750_wp, -1.76385194860837_wp, -3.64519483422730_wp, &
      &  8.17017571236949_wp, -0.94549116432872_wp, -1.16663828671301_wp, &
      &  7.33783274756123_wp, -3.09540601293544_wp, -3.60722640620001_wp, &
      &  7.36128123581153_wp,  0.16723154595415_wp, -4.21985177329706_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ARG_xci

subroutine ASN_xab(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "O", "N", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.33149858813656_wp,  1.55171022435229_wp, -0.86101968131624_wp, &
      & -0.05724741866183_wp, -0.90102751247194_wp,  0.35847535533220_wp, &
      &  2.53619591827489_wp, -2.09163922487050_wp, -0.21352040144980_wp, &
      &  2.74302319091280_wp, -4.01700274957004_wp, -1.48932068697783_wp, &
      & -2.12336409011337_wp, -2.68414782830148_wp, -0.57796206800600_wp, &
      & -4.70505781964112_wp, -1.50483334639947_wp, -0.31364396150344_wp, &
      & -5.40299786612074_wp,  0.22772816671205_wp, -1.69271467729768_wp, &
      & -6.16703939542048_wp, -2.41550281772431_wp,  1.58365491444967_wp, &
      & -1.82662626962509_wp,  1.70234298866016_wp, -2.04856882919658_wp, &
      & -0.20218403017530_wp, -0.61695981504775_wp,  2.40145277085697_wp, &
      & -1.95965161944533_wp, -4.46595687400444_wp,  0.43224537662203_wp, &
      & -1.78677762472026_wp, -3.07770512459122_wp, -2.56806171187370_wp, &
      & -5.64727168096117_wp, -3.92636225090330_wp,  2.59830855029708_wp, &
      & -7.91445795481494_wp, -1.70408631154430_wp,  1.79942431259560_wp, &
      &  4.47684383499267_wp, -0.82182128834456_wp,  0.83980330174048_wp, &
      &  4.04699864669780_wp,  0.83506611268802_wp,  1.70649633364738_wp, &
      &  7.07409253857227_wp, -1.53881917698635_wp,  0.38611947435304_wp, &
      &  8.15228382667595_wp, -1.43132025445822_wp,  2.13189781844888_wp, &
      &  7.08760174531843_wp, -3.46901332514531_wp, -0.30808209989526_wp, &
      &  7.96297234551719_wp, -0.32556801273727_wp, -1.02097168167435_wp, &
      &  0.48941557890438_wp,  3.69991340526872_wp,  0.27263653944724_wp, &
      &  1.91223538764818_wp,  3.68130805568355_wp,  2.11708446006864_wp, &
      & -0.45792591640180_wp,  6.10873647137776_wp, -0.91741751264983_wp, &
      & -2.51073530603124_wp,  6.19595994564867_wp, -0.79838129628923_wp, &
      &  0.36248542143320_wp,  7.70724556848198_wp,  0.06625361636932_wp, &
      &  0.06282832535819_wp,  6.17642423649854_wp, -2.90605652108631_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ASN_xab

subroutine ASN_xae(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "O", "N", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.18799173957967_wp, -1.80862873381051_wp, -1.03186601632518_wp, &
      &  0.19831681780521_wp, -0.90023697936129_wp, -0.00210475220731_wp, &
      &  0.45525958474199_wp,  1.94127368279888_wp, -0.53963738541777_wp, &
      &  1.86840382027212_wp,  2.77936615028494_wp, -2.19342155782426_wp, &
      &  2.36936562194622_wp, -2.39820652710910_wp, -1.14581735413852_wp, &
      &  4.79891750818278_wp, -2.01278412582593_wp,  0.34027507158241_wp, &
      &  5.39411854957629_wp, -3.43177099722892_wp,  2.06391118720463_wp, &
      &  6.21246346468173_wp, -0.01003601717406_wp, -0.42199822570955_wp, &
      & -2.18518084299406_wp, -2.59273171778993_wp, -2.76367988501784_wp, &
      &  0.11641903077927_wp, -1.19995015752150_wp,  2.03595314014446_wp, &
      &  2.60967442783473_wp, -1.84245201641155_wp, -3.11412987732120_wp, &
      &  1.92089008470602_wp, -4.39997124200725_wp, -1.02405276552923_wp, &
      &  7.68045437200640_wp,  0.47140857256021_wp,  0.68453435124425_wp, &
      &  5.36651400958842_wp,  1.32640000729482_wp, -1.48858032104353_wp, &
      & -1.02784064634886_wp,  3.38780555034563_wp,  0.92746830938651_wp, &
      & -2.30617994420582_wp,  2.46972580548158_wp,  2.02242070599381_wp, &
      & -1.19716278805007_wp,  6.09164357176878_wp,  0.54969219613062_wp, &
      & -2.56531452458845_wp,  6.57540494592865_wp, -0.91211659230745_wp, &
      & -1.74948845761819_wp,  6.98750261851080_wp,  2.31124420253245_wp, &
      &  0.64166961111693_wp,  6.79135095450491_wp, -0.03142671622444_wp, &
      & -4.40275277382029_wp, -1.56628730755940_wp,  0.23373583118976_wp, &
      & -4.54274750122038_wp, -0.48346127245316_wp,  2.28830215994099_wp, &
      & -6.68459453406861_wp, -2.71809251983235_wp, -1.01839561315743_wp, &
      & -8.07577129253608_wp, -1.23995053349202_wp, -1.33765875674536_wp, &
      & -6.25293876590705_wp, -3.64621976599249_wp, -2.79904632106185_wp, &
      & -7.51051968755222_wp, -4.08274537755230_wp,  0.27657866665283_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ASN_xae

subroutine ASN_xaf(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "O", "N", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.13603388993819_wp, -1.90354788237015_wp,  0.12899432596055_wp, &
      &  0.19671645502625_wp, -0.08747275977542_wp,  1.70447886452388_wp, &
      &  0.35794732200074_wp,  2.56728894009847_wp,  0.52322762679447_wp, &
      &  2.39733885223755_wp,  3.65564149006792_wp,  0.21212128012760_wp, &
      &  2.83650052663305_wp, -1.10596265900137_wp,  2.28206132792513_wp, &
      &  4.15998810834337_wp, -1.95354658679884_wp, -0.11575577146562_wp, &
      &  3.66672041450448_wp, -4.01513795278188_wp, -1.07711460609595_wp, &
      &  5.87209285926388_wp, -0.31699920516412_wp, -1.07854344701764_wp, &
      & -0.05368975570851_wp, -3.25157495120453_wp, -0.69672052858298_wp, &
      & -0.86853405708859_wp,  0.10874902040441_wp,  3.46362934150801_wp, &
      &  2.65708191227087_wp, -2.76423072285100_wp,  3.48350294167749_wp, &
      &  3.90601641466616_wp,  0.34500863784939_wp,  3.26274479344581_wp, &
      &  5.60730914914504_wp,  1.52015107141429_wp, -0.63090890794453_wp, &
      &  6.52813348126810_wp, -0.72927347473996_wp, -2.81638485522495_wp, &
      & -1.90373824181854_wp,  3.56433723121246_wp, -0.04429426623377_wp, &
      & -3.44759032727190_wp,  2.50737390917139_wp,  0.39880938216878_wp, &
      & -2.16839724277707_wp,  6.04702985760623_wp, -1.15735842274794_wp, &
      & -0.28878847698368_wp,  6.77005598608782_wp, -1.54443870435898_wp, &
      & -3.23528101326935_wp,  5.94212322586126_wp, -2.91184141172765_wp, &
      & -3.12935938969529_wp,  7.33083705277748_wp,  0.13121451899426_wp, &
      & -3.67937224225776_wp, -2.08597028454785_wp,  0.07268546134949_wp, &
      & -5.07245365579565_wp, -0.53508282864409_wp,  1.12258814319970_wp, &
      & -4.72665280574370_wp, -4.30764974942388_wp, -1.36478867864384_wp, &
      & -3.27265156817180_wp, -5.41920075448055_wp, -2.29595697542834_wp, &
      & -5.75764735361144_wp, -5.50252676995051_wp, -0.04731865592800_wp, &
      & -6.06791724900582_wp, -3.61620215295104_wp, -2.75838366526890_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ASN_xaf

subroutine ASN_xah(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "O", "N", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.22475950984104_wp,  1.72025947663879_wp, -0.63635775560324_wp, &
      & -0.18671839219936_wp, -0.74117445737591_wp,  0.56593444575827_wp, &
      &  2.38246895913443_wp, -2.05826180038407_wp,  0.52342082967376_wp, &
      &  2.73596019270725_wp, -3.92387127414306_wp,  1.84915118933072_wp, &
      & -2.06779857175035_wp, -2.49806494492513_wp, -0.76284237191558_wp, &
      & -4.69448030202581_wp, -1.38653518653312_wp, -0.69868464591299_wp, &
      & -5.26626650930229_wp,  0.45463056689208_wp, -1.99378221495410_wp, &
      & -6.33586645866972_wp, -2.49289974336893_wp,  0.92173433924708_wp, &
      & -1.73682643236780_wp,  2.04269204041555_wp, -1.76966722847942_wp, &
      & -0.66974051846081_wp, -0.56399030214048_wp,  2.56528835726761_wp, &
      & -1.96020042380822_wp, -4.33799779994140_wp,  0.14666272024418_wp, &
      & -1.51640239439075_wp, -2.69748557715417_wp, -2.73758699339419_wp, &
      & -8.07344405733273_wp, -1.75007080539064_wp,  1.10746742924403_wp, &
      & -5.83055613716478_wp, -3.96932699197819_wp,  1.99385771933481_wp, &
      &  4.06266987385349_wp, -1.11878467890406_wp, -1.14804293356946_wp, &
      &  3.60770227863375_wp,  0.54110621676406_wp, -1.95546325957870_wp, &
      &  6.62085623688869_wp, -2.06875449921041_wp, -1.28500904803106_wp, &
      &  7.26947155757994_wp, -2.07147972142471_wp, -3.23402389190726_wp, &
      &  7.90896935597096_wp, -0.93992401533273_wp, -0.14170340340865_wp, &
      &  6.61487076700045_wp, -3.98845162826834_wp, -0.56237513695754_wp, &
      &  0.73980635861326_wp,  3.74571905550395_wp,  0.67071425213185_wp, &
      &  2.08206377071630_wp,  3.46732713725542_wp,  2.53016952227780_wp, &
      &  0.03035923363398_wp,  6.30154948765319_wp, -0.37320985627281_wp, &
      & -1.52162368943961_wp,  7.05659729084790_wp,  0.74814763133977_wp, &
      & -0.56302608369772_wp,  6.21090088131946_wp, -2.33794050217366_wp, &
      &  1.63003888582773_wp,  7.57123664589285_wp, -0.17778875303600_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ASN_xah

subroutine ASN_xaj(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "O", "N", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.33224020541844_wp, -2.00657635051395_wp,  0.16913100907757_wp, &
      & -0.10480799058680_wp, -0.49501173573060_wp, -0.19919729892557_wp, &
      &  2.17144253724508_wp, -2.27696749016666_wp, -0.20684970440762_wp, &
      &  1.87649006777431_wp, -4.58276336240037_wp, -0.18591045584309_wp, &
      &  0.13105859015010_wp,  1.61837442026403_wp,  1.80273829608567_wp, &
      &  1.05459638656353_wp,  3.96933204905096_wp,  0.48096991002409_wp, &
      &  3.31777662745887_wp,  4.39532802521638_wp,  0.14657285799908_wp, &
      & -0.82277578110447_wp,  5.45644638868082_wp, -0.40931192480058_wp, &
      & -2.03272501598816_wp, -3.88941677143067_wp,  0.28986590226051_wp, &
      & -0.21673194694664_wp,  0.39081557572829_wp, -2.06540620400325_wp, &
      &  1.46258723928317_wp,  1.09157436623699_wp,  3.27565973192863_wp, &
      & -1.71777853879680_wp,  1.94030373148256_wp,  2.63022526624164_wp, &
      & -0.34108859152246_wp,  6.84526453001881_wp, -1.61441484818640_wp, &
      & -2.57981260591420_wp,  4.71259893504905_wp, -0.50608601792130_wp, &
      &  4.41683139433813_wp, -1.08703807245848_wp, -0.26222174702437_wp, &
      &  4.44015196310794_wp,  0.83238894815005_wp, -0.17070590796523_wp, &
      &  6.78203830934882_wp, -2.46235408286885_wp, -0.28790393846040_wp, &
      &  7.90495917842411_wp, -1.93728284588881_wp, -1.92794138781347_wp, &
      &  7.86951310468394_wp, -2.08240187894268_wp,  1.41604640154780_wp, &
      &  6.34538272065707_wp, -4.46393560668098_wp, -0.37674031561534_wp, &
      & -4.63927552850191_wp, -0.98682574596354_wp, -0.22379323395837_wp, &
      & -4.91843356097490_wp,  1.28071191446203_wp, -0.69595918482079_wp, &
      & -6.84986724892511_wp, -2.76789414682762_wp, -0.02253257827969_wp, &
      & -8.15896775033162_wp, -2.03644491643629_wp,  1.38161606518223_wp, &
      & -7.82790176957181_wp, -2.80069871468442_wp, -1.82988943232913_wp, &
      & -6.29362874996653_wp, -4.67755400556065_wp,  0.48751675869346_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ASN_xaj

subroutine ASP_xad(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 25
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "O", "O", "N", "C", &
      & "H", "H", "H", "H", "C", "C", "O", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.32631148980870_wp,  1.09102844888502_wp,  0.17429150204109_wp, &
      & -3.77579270425515_wp,  2.81839999076862_wp, -0.49861211213562_wp, &
      & -0.66644291233756_wp,  0.52908358594809_wp,  0.25559942147472_wp, &
      & -0.16182562015151_wp, -0.16652396405748_wp,  2.12937651528264_wp, &
      & -0.01726798036523_wp, -1.52474094832930_wp, -1.67771015706780_wp, &
      &  0.70324067781222_wp,  3.01571000399313_wp, -0.29946176472286_wp, &
      & -0.05869709383738_wp, -0.78146858809819_wp, -3.59968287380105_wp, &
      & -1.46150148623293_wp, -2.98958949644540_wp, -1.55207585587179_wp, &
      &  2.47108487989728_wp, -2.80500927428096_wp, -1.22771740977071_wp, &
      & -0.43506882272150_wp,  4.82149620305978_wp, -1.22002764765907_wp, &
      &  2.90515293928870_wp, -4.61307713678030_wp, -2.96214646378342_wp, &
      &  3.94777122596628_wp, -2.36135544530784_wp,  0.47473283686023_wp, &
      &  3.18148481736578_wp,  3.01239347272468_wp,  0.27168539472525_wp, &
      &  4.80272377250370_wp,  5.15162257655927_wp, -0.25193238447482_wp, &
      &  3.91734484729627_wp,  1.37807222236325_wp,  0.92907662151705_wp, &
      &  3.62788486187583_wp,  6.65627871373020_wp, -1.00058204897044_wp, &
      &  5.72234558804372_wp,  5.80233579008944_wp,  1.46767185960156_wp, &
      &  6.24226895459884_wp,  4.66400795773928_wp, -1.63834964241809_wp, &
      & -5.06162432018853_wp, -0.54790097846627_wp,  1.10208514010165_wp, &
      & -7.76841415956434_wp,  0.34062874345042_wp,  1.00866987227892_wp, &
      & -4.45901739834904_wp, -2.61702841135309_wp,  1.96472230252142_wp, &
      & -8.46268283767971_wp,  0.51720409362791_wp,  2.93543960781302_wp, &
      & -8.89888742985948_wp, -1.09304948111223_wp,  0.06788741620158_wp, &
      & -7.99240529516597_wp,  2.13856088688451_wp,  0.04102148966017_wp, &
      &  4.51192232299974_wp, -5.36759665725960_wp, -2.50973635754521_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ASP_xad

subroutine ASP_xau(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 25
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "O", "O", "N", "C", &
      & "H", "H", "H", "H", "C", "C", "O", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.44180425380034_wp, -2.18505415077051_wp, -0.13650734503127_wp, &
      &  0.65633914038548_wp, -3.71288729138640_wp, -0.97388953168577_wp, &
      & -0.39490011757029_wp, -0.80609849932065_wp,  1.39019275058962_wp, &
      & -0.55921163284091_wp, -1.76085354417308_wp,  3.21899122211695_wp, &
      &  0.23932382853615_wp,  1.95037574196574_wp,  1.97032739567163_wp, &
      & -2.93081217261504_wp, -1.31963898536927_wp,  0.06754016123544_wp, &
      &  1.92258071602293_wp,  2.00395355700467_wp,  3.14107482108569_wp, &
      & -1.33387886019989_wp,  2.71148071901250_wp,  3.05465807892008_wp, &
      &  0.58305436571966_wp,  3.61716292882430_wp, -0.33536712365967_wp, &
      & -3.40710385854449_wp, -3.50719018002527_wp, -0.56421047775728_wp, &
      &  2.90456807504770_wp,  3.75333012075234_wp, -1.27660682718178_wp, &
      & -1.17084657238019_wp,  4.77692646016465_wp, -1.26933835025060_wp, &
      & -4.52296209606057_wp,  0.63210917859965_wp, -0.24708664243285_wp, &
      & -6.90349876490753_wp,  0.20553293445623_wp, -1.54303383232564_wp, &
      & -3.78392311615049_wp,  2.40258503100367_wp, -0.27877508229059_wp, &
      & -8.09314480554107_wp,  1.85725844733294_wp, -1.29279109733466_wp, &
      & -6.61898694873863_wp, -0.13679531963903_wp, -3.55316236705209_wp, &
      & -7.82583019234370_wp, -1.43944951162811_wp, -0.73580875488378_wp, &
      &  3.97134129037817_wp, -1.93451960294383_wp, -0.18291477045877_wp, &
      &  5.36744346600842_wp, -3.91342319764233_wp, -1.67668343470569_wp, &
      &  5.13356258423760_wp, -0.19063039599014_wp,  0.86385682199144_wp, &
      &  6.69022082774468_wp, -4.85335304152571_wp, -0.41515004766290_wp, &
      &  4.13188780447203_wp, -5.30896684046203_wp, -2.53683037454189_wp, &
      &  6.46382855064435_wp, -2.97035896049618_wp, -3.13621029022931_wp, &
      &  3.98107263781750_wp,  2.54023541633432_wp, -0.35125441522096_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ASP_xau

subroutine ASP_xay(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 25
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "O", "O", "N", "C", &
      & "H", "H", "H", "H", "C", "C", "O", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.10222010373344_wp, -1.82890761872876_wp, -1.00167357788522_wp, &
      & -1.99499904325156_wp, -2.72334772538139_wp, -2.67589439666605_wp, &
      &  0.21979148836415_wp, -0.81961548524134_wp,  0.08321931258833_wp, &
      &  0.10958115044244_wp, -1.12205225208072_wp,  2.12114019242673_wp, &
      &  2.44948386079244_wp, -2.25655509484506_wp, -1.04051975604037_wp, &
      &  0.39471809074309_wp,  2.03317894644175_wp, -0.42651036228340_wp, &
      &  2.65808254876903_wp, -1.71840000864700_wp, -3.02037272514267_wp, &
      &  2.06157460154265_wp, -4.26899874452501_wp, -0.90844466058799_wp, &
      &  4.92870435255888_wp, -1.81593370473756_wp,  0.35020746822669_wp, &
      &  2.01266815848470_wp,  2.93561767204907_wp, -1.85876363187520_wp, &
      &  5.96762719879544_wp,  0.46998343403475_wp,  0.04887638157954_wp, &
      &  5.90053806254784_wp, -3.42955405687089_wp,  1.64041661492383_wp, &
      & -1.34237484899283_wp,  3.40593434369637_wp,  0.78802016394959_wp, &
      & -1.53887686842993_wp,  6.11734359109869_wp,  0.46673833253758_wp, &
      & -2.65558015719959_wp,  2.43276176629435_wp,  1.79663573461961_wp, &
      & -2.98349093745464_wp,  6.60702683550058_wp, -0.91502359477330_wp, &
      & -2.00628638256833_wp,  6.98732379900534_wp,  2.26614535957152_wp, &
      &  0.27085336434472_wp,  6.82203473898343_wp, -0.19139940236165_wp, &
      & -4.35736348391656_wp, -1.70832216875331_wp,  0.20928988708096_wp, &
      & -6.52135204710841_wp, -3.06264466767510_wp, -1.04559959221845_wp, &
      & -4.61673617722131_wp, -0.56823489513135_wp,  2.22129680838213_wp, &
      & -7.18484102103641_wp, -4.53411880068385_wp,  0.22665231094266_wp, &
      & -6.01659828332818_wp, -3.90509782968261_wp, -2.84932335860281_wp, &
      & -8.06155449483811_wp, -1.73109538885319_wp, -1.31705498103450_wp, &
      &  4.79491887681528_wp,  1.51237684295412_wp, -0.94940834481046_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ASP_xay

subroutine ASP_xaz(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 25
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "O", "O", "N", "C", &
      & "H", "H", "H", "H", "C", "C", "O", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.16952132161618_wp, -2.12154219323301_wp,  0.13405248947687_wp, &
      & -1.81587541706702_wp, -3.99243713507253_wp,  0.28781152957826_wp, &
      &  0.00146728635539_wp, -0.52444193840933_wp, -0.20337127669522_wp, &
      & -0.11746256758567_wp,  0.34106473142033_wp, -2.07793754214212_wp, &
      &  0.13936185135996_wp,  1.60137615946930_wp,  1.80115473309693_wp, &
      &  2.35504834173198_wp, -2.19259336459355_wp, -0.18231575559234_wp, &
      &  1.63703914022462_wp,  1.24796555649411_wp,  3.15884375571242_wp, &
      & -1.65949051619553_wp,  1.70778871957001_wp,  2.78393821866927_wp, &
      &  0.66371878027235_wp,  4.04030401948901_wp,  0.42026666249419_wp, &
      &  2.18373940017606_wp, -4.50479526100467_wp, -0.08056563613812_wp, &
      & -1.33194397883329_wp,  5.02390238736553_wp, -0.77326123888143_wp, &
      &  2.76145734268421_wp,  4.95409022951481_wp,  0.25571722557961_wp, &
      &  4.53677598144161_wp, -0.88647510742250_wp, -0.31387964242462_wp, &
      &  6.97782630578185_wp, -2.12650786772738_wp, -0.34179190274019_wp, &
      &  4.46437495529401_wp,  1.02435609289951_wp, -0.29723632486373_wp, &
      &  6.64817211010172_wp, -4.14974317709413_wp, -0.38222847283201_wp, &
      &  8.04584455355734_wp, -1.57677738892158_wp, -2.00932764781227_wp, &
      &  8.06244264916872_wp, -1.65129687090219_wp,  1.33927263010944_wp, &
      & -4.50157718655301_wp, -1.15648142745338_wp, -0.18408175947121_wp, &
      & -6.67485356137539_wp, -2.97435407971280_wp,  0.00707948363876_wp, &
      & -4.84271380225306_wp,  1.12201643547764_wp, -0.60132886618364_wp, &
      & -7.64136584458151_wp, -3.02707671532324_wp, -1.80628130006197_wp, &
      & -8.00570667403189_wp, -2.26697022522844_wp,  1.40249212512684_wp, &
      & -6.08023309487483_wp, -4.87159874928598_wp,  0.51884622502704_wp, &
      & -2.74742559519496_wp,  3.82221992533511_wp, -0.64378264740682_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ASP_xaz

subroutine ASP_xbc(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 25
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "O", "O", "N", "C", &
      & "H", "H", "H", "H", "C", "C", "O", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.35056998142984_wp,  1.78830375373518_wp, -1.65642116364797_wp, &
      &  1.30806814583970_wp,  2.25151934909860_wp, -3.23097698932147_wp, &
      & -0.17898797606613_wp, -0.90460457520374_wp, -1.36051007249484_wp, &
      &  0.34867606044537_wp, -1.76808169143789_wp, -3.14652109647652_wp, &
      &  1.40458905889623_wp, -2.10345148201367_wp,  0.72536475184745_wp, &
      & -2.97050097003861_wp, -1.60813329250713_wp, -1.03288359609048_wp, &
      &  0.88534803965233_wp, -4.08809965047159_wp,  0.93371273400562_wp, &
      &  1.06921039778545_wp, -1.18895119596290_wp,  2.53762189360690_wp, &
      &  4.17715835552101_wp, -1.97490134292218_wp,  0.12096084555662_wp, &
      & -3.85707099946449_wp, -3.34332430346013_wp, -2.29257608589709_wp, &
      &  5.58594451571577_wp, -2.91277005435471_wp,  2.02960676558260_wp, &
      &  5.09089868891353_wp, -1.16064928563266_wp, -1.81330556129849_wp, &
      & -4.24047302618444_wp, -0.29929401072900_wp,  0.74320735345822_wp, &
      & -6.86594255126641_wp, -0.81983420847893_wp,  1.29830491260746_wp, &
      & -3.27001716879749_wp,  1.04258463966482_wp,  1.71133857532521_wp, &
      & -8.10204897151009_wp,  0.63723899907012_wp,  0.53184582734973_wp, &
      & -7.35238423454512_wp, -2.61823942260797_wp,  0.43938735143569_wp, &
      & -7.14931853561382_wp, -0.93032309800807_wp,  3.33071850839752_wp, &
      &  0.32258808247938_wp,  3.53636507955090_wp,  0.21001745267221_wp, &
      &  1.46811543899164_wp,  6.05752328673939_wp, -0.45720254953997_wp, &
      & -0.56208978283225_wp,  3.12993267877888_wp,  2.32864615880244_wp, &
      &  0.25293901860115_wp,  7.54315087954189_wp,  0.26922618673631_wp, &
      &  3.28888046065397_wp,  6.21139005583130_wp,  0.48821275720039_wp, &
      &  1.74211817972206_wp,  6.31076383534115_wp, -2.47676914489538_wp, &
      &  7.32583056487302_wp, -2.78155397675573_wp,  1.47287331169872_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ASP_xbc

subroutine CYS_xag(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 23
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "S", "H", "H", "H", "H", "C", "O", "N", "C", &
      & "H", "H", "H", "H", "C", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.88081600192595_wp, -0.45523912889591_wp, -0.49426311236501_wp, &
      & -0.02027605138398_wp,  0.34231758439842_wp,  1.29749904923794_wp, &
      & -2.61787962829337_wp, -0.81177268219733_wp,  0.77256620809539_wp, &
      & -4.54789974734622_wp,  0.20051165620105_wp,  1.56562221988579_wp, &
      & -0.23377292016504_wp,  3.20976111585809_wp,  1.43017107017687_wp, &
      & -1.03684917983945_wp,  4.60987430942333_wp, -1.60942994974511_wp, &
      &  1.79448408389374_wp,  0.38008716747177_wp, -2.20856135888611_wp, &
      &  0.59988121125262_wp, -0.36946079435628_wp,  3.13399853738402_wp, &
      & -1.64461207859542_wp,  3.75067926131034_wp,  2.81718613893677_wp, &
      &  1.58427608017843_wp,  4.01204454537032_wp,  1.96187993957650_wp, &
      &  4.18324644695437_wp, -1.35294105803256_wp,  0.31125359597188_wp, &
      &  4.58982729779963_wp, -1.95136561750861_wp,  2.50208546628905_wp, &
      & -2.57525814266416_wp, -2.99311724513997_wp, -0.54838283462959_wp, &
      & -4.84209564339707_wp, -4.42407589607929_wp, -1.07676906530652_wp, &
      & -5.05153406635957_wp, -4.74396602708424_wp, -3.09693134908080_wp, &
      & -4.81813155515602_wp, -6.23817224203853_wp, -0.10701983244369_wp, &
      & -6.43264344240733_wp, -3.32214504785989_wp, -0.39571697067795_wp, &
      & -0.87174606205064_wp, -3.62548008625957_wp, -1.10514938367686_wp, &
      &  6.15817466526303_wp, -1.57575977210918_wp, -1.73377025577078_wp, &
      &  7.55784909677773_wp, -0.09480061469363_wp, -1.45222622839324_wp, &
      &  5.37688153949153_wp, -1.40603665157185_wp, -3.62629340833896_wp, &
      &  7.11169458463753_wp, -3.38246725943627_wp, -1.53534031278322_wp, &
      & -3.42540382366603_wp,  3.77818506010286_wp, -1.60501295962982_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine CYS_xag

subroutine CYS_xah(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 23
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "S", "H", "H", "H", "H", "C", "O", "N", "C", &
      & "H", "H", "H", "H", "C", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.91992733263756_wp, -1.83849379432400_wp,  0.84224119504275_wp, &
      & -0.47801420732044_wp, -1.02155397642576_wp, -0.23822625487488_wp, &
      & -0.99567512187054_wp,  1.72568225145018_wp,  0.52548222109561_wp, &
      & -2.46810936610854_wp,  2.27134208648126_wp,  2.23738653017811_wp, &
      & -2.57401251073919_wp, -2.75270795616136_wp,  0.66268500058392_wp, &
      & -5.57305925490875_wp, -2.10214139347652_wp, -0.86590304739409_wp, &
      &  1.92054759432741_wp, -2.58030448384979_wp,  2.59336560601530_wp, &
      & -0.28293849560395_wp, -1.13977356665407_wp, -2.28400887540447_wp, &
      & -2.80513243425876_wp, -2.58984684750024_wp,  2.70016253479108_wp, &
      & -2.10400017881165_wp, -4.69994348918384_wp,  0.19154692402621_wp, &
      &  4.16053467204936_wp, -1.31367225250059_wp, -0.29404851212982_wp, &
      &  4.28881439878645_wp, -0.13904771225484_wp, -2.29553874705634_wp, &
      &  0.35195396119806_wp,  3.41755342025168_wp, -0.82126379719511_wp, &
      &  0.30795489759171_wp,  6.08373085470676_wp, -0.21460067310073_wp, &
      &  1.82534444824371_wp,  6.60059801155634_wp,  1.07923133050844_wp, &
      &  0.48877783079977_wp,  7.18791689185642_wp, -1.93597144070938_wp, &
      & -1.48636641229333_wp,  6.50382535883827_wp,  0.68587216739188_wp, &
      &  1.70937124149079_wp,  2.70832701302039_wp, -1.96870663441257_wp, &
      &  6.48323621400928_wp, -2.29378962384082_wp,  1.02985416578067_wp, &
      &  7.45964016783530_wp, -3.57866185726289_wp, -0.24234464672913_wp, &
      &  6.06626591071595_wp, -3.26978151384711_wp,  2.78836104456549_wp, &
      &  7.74017218803501_wp, -0.71260487269362_wp,  1.40496109116790_wp, &
      & -5.93302702867262_wp,  0.09849225322546_wp,  0.32449226598317_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine CYS_xah

subroutine CYS_xai(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 23
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "S", "H", "H", "H", "H", "C", "O", "N", "C", &
      & "H", "H", "H", "H", "C", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.55403842124459_wp, -1.27685465349123_wp,  0.37937966525844_wp, &
      & -0.23437190851079_wp, -0.05470098220828_wp, -0.33354124512356_wp, &
      &  1.85881230796409_wp, -2.02591272490102_wp, -0.04831456543481_wp, &
      &  1.42636865360774_wp, -4.12902857267703_wp,  0.84130859791635_wp, &
      &  0.20604830126620_wp,  2.28509537980293_wp,  1.32982010211021_wp, &
      &  2.47505454408393_wp,  4.53367089082696_wp,  0.05391087585609_wp, &
      & -2.36854928850557_wp, -3.05812979924244_wp,  1.04061438516301_wp, &
      & -0.36504701256809_wp,  0.56846454122625_wp, -2.29738278493240_wp, &
      & -1.58982166071823_wp,  3.24819261946156_wp,  1.56968896221686_wp, &
      &  0.91777211778075_wp,  1.73210135945332_wp,  3.18128649922200_wp, &
      & -4.81167948773508_wp, -0.28883052887113_wp, -0.33605625339409_wp, &
      & -4.96347018507185_wp,  1.75989400660516_wp, -1.41492949478779_wp, &
      &  4.16597124722727_wp, -1.29051276902528_wp, -0.82759793858553_wp, &
      &  6.33200511761392_wp, -2.93435163898845_wp, -0.50571353060085_wp, &
      &  7.90179352729999_wp, -2.17829988476813_wp, -1.58818440964720_wp, &
      &  6.88208288681186_wp, -3.09126341275257_wp,  1.47197658049678_wp, &
      &  5.87151590775953_wp, -4.81354122828552_wp, -1.19149271796432_wp, &
      &  4.40440312693804_wp,  0.54025170963973_wp, -1.30641511093489_wp, &
      & -7.10188351920578_wp, -1.86292835555582_wp,  0.29397566444068_wp, &
      & -8.43786452872124_wp, -0.69500067602549_wp,  1.32826832610336_wp, &
      & -6.64883883247388_wp, -3.53286962615322_wp,  1.40033576823827_wp, &
      & -8.00403931301156_wp, -2.43465007806150_wp, -1.46254139008525_wp, &
      &  1.01101402517916_wp,  5.28507826323503_wp, -1.86632977694685_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine CYS_xai

subroutine CYS_xal(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 23
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "S", "H", "H", "H", "H", "C", "O", "N", "C", &
      & "H", "H", "H", "H", "C", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.97098658338820_wp, -1.27330359726610_wp, -0.62359793899190_wp, &
      & -0.08049781628230_wp, -0.67344334363074_wp,  1.04030316271116_wp, &
      & -2.52812352560703_wp, -1.19984315922862_wp, -0.37324346569410_wp, &
      & -2.60379823787070_wp, -1.25780202093089_wp, -2.68991422416729_wp, &
      & -0.02493426146513_wp,  2.08540444189520_wp,  1.95569441702608_wp, &
      &  0.06429475609833_wp,  4.41233867833066_wp, -0.55784380720842_wp, &
      &  1.56489288710404_wp, -1.26140747838930_wp, -2.48479446577798_wp, &
      &  0.05329174928870_wp, -1.86543382798611_wp,  2.72167505975940_wp, &
      & -1.61635294477044_wp,  2.44454033685132_wp,  3.21047010754757_wp, &
      &  1.71718642402033_wp,  2.36218665270793_wp,  3.01407423070297_wp, &
      &  4.38530285162641_wp, -1.35905877365005_wp,  0.24055528416557_wp, &
      &  4.89250474301042_wp, -1.13463018390317_wp,  2.49285198689088_wp, &
      & -4.58658992322822_wp, -1.45180725514541_wp,  1.12566210960034_wp, &
      & -7.09443513952584_wp, -1.86814145794771_wp,  0.11036695446876_wp, &
      & -8.43370008357430_wp, -0.54304553582133_wp,  0.93080847884725_wp, &
      & -6.99057542099958_wp, -1.56703520723232_wp, -1.91530249062940_wp, &
      & -7.73725356177990_wp, -3.79000095823237_wp,  0.46460603338123_wp, &
      & -4.31698592523973_wp, -1.54171006339497_wp,  3.00338915623237_wp, &
      &  6.38287328290249_wp, -1.73547576465536_wp, -1.75615395850798_wp, &
      &  7.32525439433009_wp,  0.06343508823183_wp, -2.08427340363255_wp, &
      &  5.61263013794713_wp, -2.41666997615333_wp, -3.53454729812841_wp, &
      &  7.77822036180375_wp, -3.06292078235912_wp, -1.04714526899663_wp, &
      & -2.04741863153924_wp,  3.66560694687422_wp, -1.73475292399990_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine CYS_xal

subroutine CYS_xao(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 23
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "S", "H", "H", "H", "H", "C", "O", "N", "C", &
      & "H", "H", "H", "H", "C", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.94112103805850_wp,  1.35180766045427_wp,  0.90371952108256_wp, &
      &  0.37612515301125_wp, -0.97717864102884_wp, -0.43774842913593_wp, &
      & -2.30807922851827_wp, -1.89082002504276_wp,  0.20980172096419_wp, &
      & -2.68630762519216_wp, -3.71873666717009_wp,  1.58608889003611_wp, &
      &  2.24912736998987_wp, -3.03648174063932_wp,  0.26018015756036_wp, &
      &  5.54340779992854_wp, -2.11502466466635_wp, -0.03289566749770_wp, &
      &  2.34273526695163_wp,  1.29200960602338_wp,  2.19055392154596_wp, &
      &  0.46787573195774_wp, -0.56982494721595_wp, -2.46092197411534_wp, &
      &  1.86854945736115_wp, -4.70964926471415_wp, -0.86604765878636_wp, &
      &  1.98257483605122_wp, -3.57181987857477_wp,  2.22911946524483_wp, &
      &  0.26640080489471_wp,  3.63692643834480_wp, -0.03775389575268_wp, &
      & -1.14872303553483_wp,  3.85720478026807_wp, -1.87321415319328_wp, &
      & -4.14680029873735_wp, -0.50439140113402_wp, -0.87801007555192_wp, &
      & -6.77438864485307_wp, -0.97745179803814_wp, -0.28441157022671_wp, &
      & -7.95304503250921_wp, -0.08572305456040_wp, -1.70767008956040_wp, &
      & -7.11110084151548_wp, -3.00214390759714_wp, -0.28654376832668_wp, &
      & -7.28604113548125_wp, -0.24676716357005_wp,  1.57280419935856_wp, &
      & -3.60647522490453_wp,  1.13055393041662_wp, -1.71882392410252_wp, &
      &  1.35001332613479_wp,  5.90059206133020_wp,  1.30491175227646_wp, &
      &  2.58846579613214_wp,  6.89619531841450_wp,  0.00034786199678_wp, &
      &  2.40352489751717_wp,  5.40764235611588_wp,  2.99746979061924_wp, &
      & -0.18712358347789_wp,  7.16984899630573_wp,  1.79856719509230_wp, &
      &  5.43463928740996_wp, -1.42618485497875_wp, -2.46345236781880_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine CYS_xao

subroutine GLN_xai(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "O", "H", "H", "H", "H", "H", "H", &
      & "C", "O", "N", "H", "C", "H", "H", "H", "C", "H", "H", "H", "N", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.18499983665937_wp, -0.99391093637633_wp, -0.02260128975613_wp, &
      & -0.93260873525957_wp,  0.46720149632686_wp, -0.48021077397059_wp, &
      & -1.72554339296064_wp,  3.19040854959770_wp, -0.98038766808638_wp, &
      & -3.93850867174243_wp,  3.77196685966293_wp, -1.38005710566481_wp, &
      &  0.69996708726748_wp, -0.55983029543238_wp, -2.66220753783808_wp, &
      &  2.74746837011503_wp, -2.33475500441766_wp, -1.68759150351661_wp, &
      &  4.27583637258617_wp, -0.93175030478113_wp,  0.28341732195986_wp, &
      &  4.98037466649259_wp,  1.26022521399961_wp, -0.04261123327536_wp, &
      & -4.81513096162772_wp, -0.12053621869373_wp, -0.50126616973255_wp, &
      &  0.17968862266247_wp,  0.44928909445909_wp,  1.25667288576891_wp, &
      & -0.52977640178007_wp, -1.51560224175516_wp, -4.00800251147916_wp, &
      &  1.61010242321908_wp,  1.00619604635244_wp, -3.63636135216360_wp, &
      &  4.02452062448901_wp, -2.87190154837294_wp, -3.21436208785766_wp, &
      &  1.92666580724230_wp, -4.03807564514348_wp, -0.88639026873158_wp, &
      & -3.16830372029965_wp, -3.00702074612748_wp,  1.55521928894058_wp, &
      & -1.20709519577409_wp, -3.87231898793686_wp,  2.46389576774564_wp, &
      &  0.23454819277314_wp,  4.81919904571274_wp, -0.94117564861697_wp, &
      &  1.98149632449536_wp,  4.09717977168587_wp, -0.64232272105160_wp, &
      & -0.06732548101495_wp,  7.48231604656865_wp, -1.48067342363361_wp, &
      &  0.76982530526829_wp,  8.61783580145762_wp,  0.01391918463387_wp, &
      &  0.80598573696481_wp,  7.98659744537939_wp, -3.27457310400319_wp, &
      & -2.07845908264974_wp,  7.87162308688167_wp, -1.58368499176385_wp, &
      & -5.71528171580550_wp, -4.15722325681035_wp,  2.11308511841124_wp, &
      & -7.26157036524731_wp, -3.18806647330242_wp,  1.17136088934032_wp, &
      & -6.01668643389474_wp, -4.11535899174608_wp,  4.14547883822413_wp, &
      & -5.68799700608695_wp, -6.12695149357069_wp,  1.52872037577910_wp, &
      &  4.79957625700896_wp, -2.26042283500725_wp,  2.40977406420545_wp, &
      &  5.59818416766158_wp, -1.29794777331204_wp,  3.84288011100064_wp, &
      &  3.65677379315861_wp, -3.71560727729109_wp,  2.84969955957394_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLN_xai

subroutine GLN_xal(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "O", "H", "H", "H", "H", "H", "H", &
      & "C", "O", "N", "H", "C", "H", "H", "H", "C", "H", "H", "H", "N", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.47709399123037_wp, -1.84588020667621_wp,  1.29999843478707_wp, &
      & -0.00113536320104_wp, -0.79539503588446_wp,  0.72095445807299_wp, &
      &  0.14416832433690_wp,  2.09520537088583_wp,  0.57561683646646_wp, &
      & -0.82610028747336_wp,  3.49737861039532_wp,  2.15942931646409_wp, &
      & -1.88018278609413_wp, -1.73501606485592_wp,  2.68438253674428_wp, &
      & -4.56971451074782_wp, -0.72190570938982_wp,  2.33325099710979_wp, &
      & -5.32490020667887_wp, -0.87320144869641_wp, -0.41939402502594_wp, &
      & -5.73142864574883_wp, -2.88047901697355_wp, -1.48948235932761_wp, &
      &  2.85829141460083_wp, -2.29315014093103_wp,  3.10796328530127_wp, &
      & -0.51687498334804_wp, -1.50719875126591_wp, -1.14602857878252_wp, &
      & -1.90448490953655_wp, -3.79126760827721_wp,  2.56270168626402_wp, &
      & -1.21711086015094_wp, -1.20090275872079_wp,  4.56351038803810_wp, &
      & -5.87480714954620_wp, -1.89984043030265_wp,  3.39996997416169_wp, &
      & -4.67691023870176_wp,  1.20379717802772_wp,  3.03130954994809_wp, &
      &  4.26355605281171_wp, -2.25574308767711_wp, -0.48712855189429_wp, &
      &  3.96346108497233_wp, -1.63776042467585_wp, -2.71094771156471_wp, &
      &  1.42133229346373_wp,  2.94199733515517_wp, -1.45671432488632_wp, &
      &  2.31844546781731_wp,  1.63243820746127_wp, -2.53179988051879_wp, &
      &  1.90494794403995_wp,  5.61238870440864_wp, -1.78899000232252_wp, &
      &  2.57362367311508_wp,  5.93341705407260_wp, -3.70175700425897_wp, &
      &  0.17227034313927_wp,  6.67264252176082_wp, -1.48427994066663_wp, &
      &  3.31168822096547_wp,  6.30662021872916_wp, -0.45418723357422_wp, &
      &  6.64526414026559_wp, -3.53635033403051_wp,  0.40447736271247_wp, &
      &  6.83410921418268_wp, -5.31859637721222_wp, -0.60149701658431_wp, &
      &  6.66827330627820_wp, -3.89586224346222_wp,  2.42667872624238_wp, &
      &  8.25223581704490_wp, -2.36097287139426_wp, -0.10119905895520_wp, &
      & -5.43852944137471_wp,  1.40170051596864_wp, -1.61814657744663_wp, &
      & -5.71835764438347_wp,  1.41494722152728_wp, -3.49665428065981_wp, &
      & -4.85081443586089_wp,  2.98123118522288_wp, -0.74708660218025_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLN_xal

subroutine GLN_xan(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "O", "H", "H", "H", "H", "H", "H", &
      & "C", "O", "N", "H", "C", "H", "H", "H", "C", "H", "H", "H", "N", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.30070575656536_wp,  0.76272855429613_wp,  0.18793374054458_wp, &
      &  0.59464529583889_wp,  0.47556685471908_wp,  0.17757961582283_wp, &
      & -0.55401902818358_wp,  3.11550827331260_wp,  0.09642738292228_wp, &
      &  0.67483266603247_wp,  4.96319823560382_wp,  0.79467688208335_wp, &
      & -0.27575598892809_wp, -1.00180300728305_wp,  2.52095526578754_wp, &
      & -2.99941350998510_wp, -1.96357769558202_wp,  2.33493318768217_wp, &
      & -3.52148813076969_wp, -3.02181910022407_wp, -0.26969402737276_wp, &
      & -5.00103710648269_wp, -1.99179692918638_wp, -1.73903691115134_wp, &
      &  3.95701579000672_wp,  2.51340919997658_wp,  0.56444626803329_wp, &
      &  0.06801918917318_wp, -0.57647427289983_wp, -1.51597591647548_wp, &
      &  1.01697747879698_wp, -2.59167610189308_wp,  2.71718126373877_wp, &
      & -0.05105897301147_wp,  0.18922889340730_wp,  4.18807371324584_wp, &
      & -3.31693364482807_wp, -3.41776305001432_wp,  3.75875965175063_wp, &
      & -4.35186881658053_wp, -0.45153320615705_wp,  2.64843485900945_wp, &
      &  4.82098060575041_wp, -1.22888469695403_wp, -0.31229609161537_wp, &
      &  3.97020948644656_wp, -3.34889721240407_wp, -0.76521196747968_wp, &
      & -2.96605658536997_wp,  3.20488359716218_wp, -0.71250546845383_wp, &
      & -3.81452546184665_wp,  1.58912704500077_wp, -1.30076379940715_wp, &
      & -4.37226225269250_wp,  5.55164572793360_wp, -0.78903931065007_wp, &
      & -6.22884486120172_wp,  5.29024999210035_wp,  0.05219227171960_wp, &
      & -3.32310478433693_wp,  6.95465195707325_wp,  0.27726759643333_wp, &
      & -4.60306967717504_wp,  6.22009824353397_wp, -2.72151952181155_wp, &
      &  7.61886218508956_wp, -0.69267455068239_wp, -0.29103317038135_wp, &
      &  8.52167552398839_wp, -1.96407822584021_wp,  1.04676684671502_wp, &
      &  8.06127489346701_wp,  1.25080737922587_wp,  0.20426261782770_wp, &
      &  8.37739274076469_wp, -1.10943968094510_wp, -2.15519661559612_wp, &
      & -2.21412159268437_wp, -5.14001518352903_wp, -0.89398765196593_wp, &
      & -2.27151141867945_wp, -5.63198574018826_wp, -2.73149155383179_wp, &
      & -0.58370616169460_wp, -5.51771325632651_wp,  0.01344237159239_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLN_xan

subroutine GLN_xap(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "O", "H", "H", "H", "H", "H", "H", &
      & "C", "O", "N", "H", "C", "H", "H", "H", "C", "H", "H", "H", "N", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.41941017156016_wp, -1.87158697725520_wp, -1.14415489363786_wp, &
      & -0.10988220863155_wp, -0.56870347519163_wp, -0.43122049324227_wp, &
      & -0.56308094067482_wp,  2.20213321282691_wp,  0.22255226621488_wp, &
      &  0.94092080874308_wp,  3.37614570888394_wp,  1.55656386122534_wp, &
      &  1.77988387508148_wp, -0.79503411120460_wp, -2.61099231917860_wp, &
      &  4.36094516048575_wp,  0.37469698427962_wp, -2.05555330653932_wp, &
      &  5.60594063133213_wp, -0.98362800891821_wp,  0.14373027386576_wp, &
      &  6.63749629454163_wp, -3.03674048241464_wp, -0.13282287232438_wp, &
      & -2.34785870978144_wp, -2.98734991529823_wp, -2.68566017014079_wp, &
      &  0.66463155415007_wp, -1.42157753760479_wp,  1.27825236641241_wp, &
      &  2.05294720615330_wp, -2.80185208928586_wp, -3.01185747256566_wp, &
      &  0.96027407753670_wp,  0.08661421160901_wp, -4.28822508360671_wp, &
      &  5.57295419232228_wp,  0.10363159678309_wp, -3.69384333788496_wp, &
      &  4.17957310655912_wp,  2.38778170563512_wp, -1.68411430587999_wp, &
      & -4.15695167439246_wp, -2.47032802273126_wp,  0.69322427670193_wp, &
      & -4.04784908580208_wp, -1.53293360181208_wp,  2.79838356578244_wp, &
      & -2.58937256479555_wp,  3.27515584201859_wp, -0.88108480079178_wp, &
      & -3.82855123483957_wp,  2.08750844778466_wp, -1.69890468078929_wp, &
      & -3.32407994292427_wp,  5.84939199700719_wp, -0.31913341459631_wp, &
      & -4.25855851615858_wp,  6.67353805482409_wp, -1.95080829074531_wp, &
      & -4.58408539669025_wp,  5.93618493537953_wp,  1.30713486160578_wp, &
      & -1.62763013908405_wp,  6.91771834185305_wp,  0.11216279701745_wp, &
      & -6.17906093714980_wp, -4.32112105335958_wp, -0.08848700426790_wp, &
      & -5.70891063821744_wp, -6.16883418745678_wp,  0.68562208226661_wp, &
      & -6.36606766360928_wp, -4.48428721653709_wp, -2.12844096633837_wp, &
      & -7.96681086937106_wp, -3.73231801534644_wp,  0.72814317852135_wp, &
      &  5.44565568448659_wp,  0.25023990478008_wp,  2.38826621434664_wp, &
      &  6.00721775309913_wp, -0.70559294220569_wp,  3.93259509918545_wp, &
      &  4.14192690068600_wp,  1.63900328209821_wp,  2.57111257856976_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLN_xap

subroutine GLN_xat(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "O", "H", "H", "H", "H", "H", "H", &
      & "C", "O", "N", "H", "C", "H", "H", "H", "C", "H", "H", "H", "N", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.06641704176257_wp, -2.10998555462759_wp, -0.08885260180470_wp, &
      & -0.66209329282025_wp, -0.33018170058632_wp,  1.47288663227020_wp, &
      & -0.70981859924564_wp,  2.36341216624898_wp,  0.37072733718441_wp, &
      &  1.23615524417289_wp,  3.49456980655745_wp, -0.23769056094310_wp, &
      &  2.03605000487402_wp, -1.24505840532026_wp,  1.81860619633373_wp, &
      &  3.40385249353396_wp, -1.76442548046510_wp, -0.69956707817077_wp, &
      &  6.16393073416333_wp, -0.99517591942466_wp, -0.55769364712131_wp, &
      &  7.89041601962047_wp, -2.52452028238872_wp, -0.42569100137664_wp, &
      & -1.07936630765538_wp, -3.24765094616746_wp, -1.24825742130885_wp, &
      & -1.63093590177640_wp, -0.25474165136565_wp,  3.29407663788442_wp, &
      &  3.04437619162539_wp,  0.20651077581034_wp,  2.86934310837514_wp, &
      &  2.02710059215920_wp, -2.94780117464497_wp,  2.97605220013969_wp, &
      &  2.51350459125143_wp, -0.68994896852727_wp, -2.21525593124222_wp, &
      &  3.36671928146988_wp, -3.76355239195138_wp, -1.17698780936870_wp, &
      & -4.60424228530095_wp, -2.39751215731422_wp,  0.05584740077866_wp, &
      & -5.93678613356199_wp, -1.02046363838821_wp,  1.37975969485071_wp, &
      & -3.04602562501353_wp,  3.34174534130926_wp,  0.19128369576534_wp, &
      & -4.49557693300445_wp,  2.22766845821103_wp,  0.77632914443367_wp, &
      & -3.51892913797675_wp,  5.83153492873747_wp, -0.83576518098247_wp, &
      & -4.73330282070743_wp,  5.71908470287670_wp, -2.49178586589677_wp, &
      & -4.41420180277145_wp,  7.03953047364717_wp,  0.56708471306000_wp, &
      & -1.71439233400216_wp,  6.64473286719400_wp, -1.37190490521456_wp, &
      & -5.70673464292452_wp, -4.52034296307629_wp, -1.48701167760913_wp, &
      & -4.33238967755119_wp, -5.43379315178531_wp, -2.71012395221777_wp, &
      & -7.24921959402577_wp, -3.77650840958797_wp, -2.62027397273996_wp, &
      & -6.49409537671209_wp, -5.91560764597358_wp, -0.19856080145479_wp, &
      &  6.49292100727788_wp,  1.54928477751606_wp, -0.56890840201576_wp, &
      &  8.24275722140420_wp,  2.22860832821311_wp, -0.29294021925259_wp, &
      &  4.97282538976573_wp,  2.70117239161772_wp, -0.50360749444468_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLN_xat

subroutine GLU_xad(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "H", "H", "C", "O", &
      & "O", "C", "C", "O", "H", "H", "H", "N", "C", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.22356935615370_wp, -1.74476254479465_wp, -0.59680924732116_wp, &
      &  0.47004798804160_wp, -1.91531687882412_wp, -2.33472620290962_wp, &
      & -0.48921567761402_wp, -1.64005110511351_wp,  1.54893255479104_wp, &
      & -0.23349214536065_wp, -3.32111110461090_wp,  2.71124462406269_wp, &
      &  0.05169804455144_wp,  0.66150965413437_wp,  3.22853867352046_wp, &
      & -3.24153248209951_wp, -1.65183765276674_wp,  0.70958462499215_wp, &
      &  2.02806559027233_wp,  0.57150864535732_wp,  3.78841922397771_wp, &
      & -1.10124373245228_wp,  0.48330665897615_wp,  4.92632601293786_wp, &
      & -0.52786794970829_wp,  3.22223646117971_wp,  2.00258972613375_wp, &
      & -4.87059870276948_wp, -2.64048888471442_wp,  2.02206616911598_wp, &
      & -2.53140443274504_wp,  3.40567592949706_wp,  1.58354670823651_wp, &
      &  0.00143287521730_wp,  4.70988098783030_wp,  3.32299236940956_wp, &
      &  0.90675308066812_wp,  3.64962982063407_wp, -0.40878703667063_wp, &
      &  3.35927384288795_wp,  4.16100610393452_wp,  0.02214333442603_wp, &
      &  0.02236391032738_wp,  3.57206050961119_wp, -2.53103765608797_wp, &
      &  3.69844408994149_wp, -2.40672762569886_wp, -0.27429866937921_wp, &
      &  5.17547122083896_wp, -2.75130514636524_wp, -2.69169934735102_wp, &
      &  4.64508252694585_wp, -2.68626686506623_wp,  1.81609356368397_wp, &
      &  5.47953487516255_wp, -4.76219002841309_wp, -3.00129429298742_wp, &
      &  7.01499394070835_wp, -1.86914498836805_wp, -2.46325791972690_wp, &
      &  4.22254359440533_wp, -1.95675075469971_wp, -4.33054751850242_wp, &
      & -3.75837482235087_wp, -0.50415982213952_wp, -1.53636922005621_wp, &
      & -6.38337389290734_wp, -0.25213098211009_wp, -2.29559083666525_wp, &
      & -2.52852934912924_wp,  0.81329902045670_wp, -2.18629019847432_wp, &
      & -7.43749585806301_wp,  0.96726678878305_wp, -1.01226270005906_wp, &
      & -6.44775415283866_wp,  0.52820328780120_wp, -4.19192855273665_wp, &
      & -7.27664665450196_wp, -2.09948301760170_wp, -2.28379409456424_wp, &
      &  4.13108708260294_wp,  4.36341695565464_wp, -1.62699014781103_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLU_xad

subroutine GLU_xal(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "H", "H", "C", "O", &
      & "O", "C", "C", "O", "H", "H", "H", "N", "C", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.43785716715568_wp, -2.20825184594770_wp, -0.68957488497598_wp, &
      & -2.03582116310203_wp, -3.07828071688461_wp, -2.33177989239973_wp, &
      & -0.56954163777708_wp, -0.48196448292040_wp,  0.36099214721598_wp, &
      & -0.77877136637753_wp, -0.57543238870482_wp,  2.41096009663977_wp, &
      &  2.05511551788015_wp, -1.30962247005537_wp, -0.43407196334644_wp, &
      & -1.15174859763384_wp,  2.22010077106008_wp, -0.52149315692124_wp, &
      &  2.22906844658612_wp, -1.13738392574955_wp, -2.47918715442981_wp, &
      &  2.30671067580862_wp, -3.29367525415328_wp,  0.05709363998321_wp, &
      &  4.09764690082888_wp,  0.26854728064911_wp,  0.81502596097501_wp, &
      &  0.05915480375217_wp,  3.26080293090676_wp, -2.20552492166369_wp, &
      &  3.92738201500418_wp,  0.23788551167727_wp,  2.87115851749839_wp, &
      &  3.95477691696006_wp,  2.23805590158397_wp,  0.23254720740162_wp, &
      &  6.69821780382163_wp, -0.65574174458480_wp,  0.15264345431663_wp, &
      &  8.50148116919631_wp,  0.77370150668204_wp,  1.27131999737028_wp, &
      &  7.18975349444016_wp, -2.43503332760019_wp, -1.19446826885994_wp, &
      & -4.72871137492742_wp, -2.56727519109729_wp,  0.39685482761609_wp, &
      & -6.41355101898496_wp, -4.48920578918100_wp, -0.85827923953672_wp, &
      & -5.38827746447549_wp, -1.41896398778666_wp,  2.31061011271194_wp, &
      & -6.75720292574288_wp, -6.01741588768463_wp,  0.47243113026606_wp, &
      & -5.61618383789920_wp, -5.26415070873514_wp, -2.58532907307001_wp, &
      & -8.22019280516314_wp, -3.60467493854636_wp, -1.27377760954882_wp, &
      & -3.11322104422207_wp,  3.28775734016457_wp,  0.71340421578580_wp, &
      & -4.09532899768621_wp,  5.72365311165986_wp, -0.05045733115985_wp, &
      & -4.15751977350919_wp,  2.12680275241829_wp,  1.82192787228382_wp, &
      & -5.00900784017853_wp,  6.62285087134924_wp,  1.55247427723796_wp, &
      & -2.53024289181534_wp,  6.88558691032537_wp, -0.69023689639377_wp, &
      & -5.44996966704575_wp,  5.55896111168834_wp, -1.59436123750615_wp, &
      & 10.09710122328953_wp,  0.05343512917042_wp,  0.73594301446009_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLU_xal

subroutine GLU_xar(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "H", "H", "C", "O", &
      & "O", "C", "C", "O", "H", "H", "H", "N", "C", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.37004565040616_wp, -2.25279485153864_wp,  0.65747598106268_wp, &
      &  1.92402191239695_wp, -3.07287297267810_wp,  2.31404607509638_wp, &
      &  0.56558260478372_wp, -0.50140876950654_wp, -0.45920722353629_wp, &
      &  0.86432827744100_wp, -0.58254728266017_wp, -2.49773865460971_wp, &
      & -2.10334237783444_wp, -1.30704202140009_wp,  0.20597855279259_wp, &
      &  1.13319657774802_wp,  2.18528643071693_wp,  0.47515431914191_wp, &
      & -2.37334906897986_wp, -1.15063402613501_wp,  2.24065192426907_wp, &
      & -2.33498126669350_wp, -3.28368250451551_wp, -0.33487400644053_wp, &
      & -4.06153828068548_wp,  0.32727034669914_wp, -1.13829775848569_wp, &
      & -0.10795978460808_wp,  3.20000081007523_wp,  2.15249196605471_wp, &
      & -3.74136517353957_wp,  0.36481383532195_wp, -3.16871427278914_wp, &
      & -3.92900423912735_wp,  2.25110849897599_wp, -0.41199299997207_wp, &
      & -6.72226549070189_wp, -0.57223454220472_wp, -0.73476828495459_wp, &
      & -7.20401441320190_wp, -1.01993726353308_wp,  1.74225224244316_wp, &
      & -8.29760365594541_wp, -0.86462559253416_wp, -2.36336262230426_wp, &
      &  4.70665865580460_wp, -2.61047850814827_wp, -0.33349862736068_wp, &
      &  6.34386626891495_wp, -4.51983046100742_wp,  1.00112654979000_wp, &
      &  5.43825571821778_wp, -1.47260027835737_wp, -2.22568113376247_wp, &
      &  8.11331663358350_wp, -3.61559334693105_wp,  1.52195401565223_wp, &
      &  6.78104611715986_wp, -6.03347650419573_wp, -0.31845178292353_wp, &
      &  5.46246445083428_wp, -5.31690347252194_wp,  2.67646249693061_wp, &
      &  3.12391999353403_wp,  3.26494595931778_wp, -0.70187901620359_wp, &
      &  4.10112708052578_wp,  5.68331151821121_wp,  0.12241326169604_wp, &
      &  4.19090465743254_wp,  2.11570065726787_wp, -1.79982639855457_wp, &
      &  5.43675930293905_wp,  5.48490181097584_wp,  1.67877405008893_wp, &
      &  5.03518071853987_wp,  6.61357302827138_wp, -1.45073377166278_wp, &
      &  2.52946773613508_wp,  6.83386786735626_wp,  0.76623280731315_wp, &
      & -8.95711694176251_wp, -1.54449747614717_wp,  1.80419441242372_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLU_xar

subroutine GLU_xav(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "H", "H", "C", "O", &
      & "O", "C", "C", "O", "H", "H", "H", "N", "C", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.45078299068315_wp, -2.01680540870854_wp,  1.26210467833896_wp, &
      &  2.69788100463380_wp, -2.58607916846835_wp,  3.05906961739115_wp, &
      &  0.19379325733311_wp, -0.56777194392502_wp,  0.64751671354127_wp, &
      & -0.28395171836885_wp, -1.03321228196707_wp, -1.30024736791907_wp, &
      & -1.93962215141351_wp, -1.32249771558706_wp,  2.41063715211165_wp, &
      &  0.77757690286546_wp,  2.26585490061875_wp,  0.85022126163990_wp, &
      & -1.44004241920031_wp, -0.75263487529118_wp,  4.32636391899705_wp, &
      & -2.13566740683340_wp, -3.37150890014813_wp,  2.34700431947772_wp, &
      & -4.43744928479755_wp, -0.08882158984403_wp,  1.71631001378912_wp, &
      &  0.08838594059371_wp,  3.55039796174235_wp,  2.65734217941572_wp, &
      & -5.85796619780401_wp, -0.46008338485691_wp,  3.16401322822213_wp, &
      & -4.25299857520183_wp,  1.95828910467576_wp,  1.61651968625291_wp, &
      & -5.49426729755606_wp, -1.05246371319990_wp, -0.73713656749438_wp, &
      & -7.57861427485333_wp,  0.25672205493983_wp, -1.43011798820266_wp, &
      & -4.66092848710862_wp, -2.78638807748151_wp, -1.97505997314623_wp, &
      &  4.28597354387891_wp, -2.48427822985667_wp, -0.46254803246008_wp, &
      &  6.44537488523083_wp, -4.11145821757769_wp,  0.43249073829198_wp, &
      &  4.18962903996542_wp, -1.66158984574221_wp, -2.63598940833164_wp, &
      &  6.47681742882348_wp, -5.82401641021595_wp, -0.70346391593596_wp, &
      &  6.31203074228033_wp, -4.62080060580479_wp,  2.41797013219335_wp, &
      &  8.20675528371179_wp, -3.10888387646351_wp,  0.09823170394076_wp, &
      &  2.12463979751700_wp,  3.15384158981045_wp, -1.12756362162924_wp, &
      &  3.07860678629444_wp,  5.71542280633203_wp, -1.16551130719146_wp, &
      &  2.82316741688331_wp,  1.84085822318898_wp, -2.33512065144491_wp, &
      &  3.04285649140750_wp,  6.44415531078937_wp, -3.08513597314696_wp, &
      &  1.86818905326725_wp,  6.86387306460809_wp,  0.02756628092498_wp, &
      &  5.00479267026136_wp,  5.82759513249900_wp, -0.44281139207528_wp, &
      & -8.13439868832390_wp, -0.50999291043981_wp, -2.99749816064226_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLU_xav

subroutine GLU_xbi(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "H", "H", "C", "O", &
      & "O", "C", "C", "O", "H", "H", "H", "N", "C", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.46534623025001_wp,  0.80125666765034_wp,  0.30423927213405_wp, &
      &  4.05608831283786_wp,  2.46393978564518_wp,  1.02423805655804_wp, &
      &  0.76890205085439_wp,  0.47683103504108_wp,  0.10954474753238_wp, &
      &  0.35249755204895_wp, -0.41863334403391_wp, -1.69838571279667_wp, &
      & -0.18410412791342_wp, -1.21707493813125_wp,  2.26596552427063_wp, &
      & -0.40045773562791_wp,  3.10615614527338_wp,  0.23479081894525_wp, &
      &  0.23265008810522_wp, -0.27989316951378_wp,  4.05354155130880_wp, &
      &  0.90813228349892_wp, -2.96049523914231_wp,  2.18006566059350_wp, &
      & -3.01470144027425_wp, -1.84229099947122_wp,  2.16600465701718_wp, &
      &  0.67512867265735_wp,  4.82808561452301_wp,  1.36895239813306_wp, &
      & -3.47460588688097_wp, -3.14452625388501_wp,  3.68749483654010_wp, &
      & -4.14173938256074_wp, -0.13720183662916_wp,  2.37237659403014_wp, &
      & -3.66494191479612_wp, -3.03460072229446_wp, -0.32180424043088_wp, &
      & -3.38387134987122_wp, -5.55657808379534_wp, -0.28553039091058_wp, &
      & -4.29119975071491_wp, -1.89062192524082_wp, -2.21749689963120_wp, &
      &  5.07810097058409_wp, -1.02674266948690_wp, -0.48285821241875_wp, &
      &  7.85106600644101_wp, -0.42177618996928_wp, -0.20908237060409_wp, &
      &  4.32674888743530_wp, -3.03817198833558_wp, -1.36331520511564_wp, &
      &  8.72417507083035_wp, -1.86469774944696_wp,  0.96459664452946_wp, &
      &  8.19665010464420_wp,  1.42877715563185_wp,  0.61276848889854_wp, &
      &  8.72537446705545_wp, -0.51685079209789_wp, -2.06627518539309_wp, &
      & -2.65696312665386_wp,  3.32673428589447_wp, -0.92647733721290_wp, &
      & -4.08715588783999_wp,  5.65809284496164_wp, -0.89046117003338_wp, &
      & -3.38661641279170_wp,  1.77181918740863_wp, -1.77051878192952_wp, &
      & -4.36004933465642_wp,  6.37800398945763_wp, -2.79730940756335_wp, &
      & -5.92699559626414_wp,  5.37699163807372_wp, -0.01410458169130_wp, &
      & -3.01890220472528_wp,  7.02767452947023_wp,  0.20001082994291_wp, &
      & -3.68946574509230_wp, -6.12942451664528_wp, -1.99949175807846_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLU_xbi

subroutine GLY_xab(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 19
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "C", "H", "H", "N", "H", "C", "O", "C", "O", "N", "H", "C", "H", "H", &
      & "H", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.53697621126446_wp, -0.65099063952992_wp, -0.06842344609593_wp, &
      &  0.66606396606498_wp, -1.84908473893675_wp, -1.74602107389109_wp, &
      &  0.73714640987883_wp, -1.90906095563423_wp,  1.55622907233919_wp, &
      & -1.82574358459953_wp,  0.66879384664154_wp,  0.00965367833541_wp, &
      & -1.71834834695807_wp,  2.57045597217434_wp, -0.02030245466366_wp, &
      &  2.64192914615766_wp,  1.29567045597847_wp, -0.08762236151957_wp, &
      &  2.20028663370963_wp,  3.57093697557199_wp, -0.10443720360120_wp, &
      & -4.04126347449978_wp, -0.61495325709523_wp,  0.04737821493846_wp, &
      & -4.11379567812652_wp, -2.93168655342912_wp,  0.04575123508834_wp, &
      &  5.01943785067081_wp,  0.34728720702742_wp, -0.11361070522132_wp, &
      &  6.39200998259351_wp,  1.66224813444128_wp, -0.02959593192953_wp, &
      &  5.68336946115410_wp, -2.29762769258562_wp,  0.12605662883987_wp, &
      &  7.68411873730242_wp, -2.50423992766703_wp, -0.27752624236656_wp, &
      &  5.32426816075263_wp, -3.03031164730635_wp,  2.01780667307248_wp, &
      &  4.64936297450760_wp, -3.44235191807082_wp, -1.23175634873567_wp, &
      & -6.39563168014953_wp,  0.99431371108626_wp,  0.07965894381020_wp, &
      & -7.45072804824264_wp,  0.64611117266087_wp, -1.64962372038080_wp, &
      & -5.99727064538394_wp,  3.00348619958780_wp,  0.23616069466895_wp, &
      & -7.57092922518358_wp,  0.40340972802370_wp,  1.65690269279488_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLY_xab

subroutine GLY_xac(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 19
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "C", "H", "H", "N", "H", "C", "O", "C", "O", "N", "H", "C", "H", "H", &
      & "H", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.76908667200988_wp, -5.79264131728894_wp,  2.10346298589694_wp, &
      &  4.66042167961057_wp, -5.05866998786113_wp,  2.45062394259666_wp, &
      &  2.36152703138849_wp, -7.31027937505492_wp,  3.42102761660140_wp, &
      &  2.71498657258732_wp, -6.79190694879211_wp, -0.45191495119351_wp, &
      &  1.61113808544757_wp, -8.28913146813204_wp, -0.82813758520983_wp, &
      &  0.79979565983286_wp, -3.72569775869823_wp,  2.52216091624816_wp, &
      & -1.07532315322252_wp, -4.07856593782436_wp,  3.83722633967961_wp, &
      &  3.91810294229316_wp, -5.61670974258406_wp, -2.38224498810532_wp, &
      &  5.11212701357647_wp, -3.64380320146376_wp, -2.07762970155825_wp, &
      &  1.34429714348730_wp, -1.54295947342177_wp,  1.30815755087036_wp, &
      &  2.86890251673388_wp, -1.55543295863960_wp,  0.15308046983471_wp, &
      & -0.37538232912086_wp,  0.58036824594651_wp,  1.33038998208632_wp, &
      &  0.66477614934253_wp,  2.33110178934906_wp,  1.60220676189920_wp, &
      & -1.45344916070190_wp,  0.70766584130167_wp, -0.42004429212957_wp, &
      & -1.68206192696093_wp,  0.31327095078976_wp,  2.88896872649332_wp, &
      &  3.73451359910018_wp, -6.88572609221333_wp, -4.92458488942418_wp, &
      &  2.60785173688799_wp, -8.60300802875712_wp, -4.89895472541549_wp, &
      &  5.63560886546990_wp, -7.33201085248455_wp, -5.56400432230945_wp, &
      &  2.92290337340454_wp, -5.55560349017693_wp, -6.26356187918549_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLY_xac

subroutine GLY_xad(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 19
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "C", "H", "H", "N", "H", "C", "O", "C", "O", "N", "H", "C", "H", "H", &
      & "H", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.20326539155309_wp, -2.36558724751024_wp, -0.63495194135872_wp, &
      &  0.47793409277708_wp, -3.84054145930777_wp, -2.03917530352805_wp, &
      & -0.33313585356539_wp, -3.25534164144004_wp,  1.14130330495024_wp, &
      & -1.83160600596073_wp, -0.70528740325999_wp, -1.38595359647827_wp, &
      & -2.17271600989916_wp, -0.44676549309125_wp, -3.23516281685821_wp, &
      &  2.76551620060111_wp, -1.12292012885014_wp, -0.20067734012286_wp, &
      &  4.65007969705563_wp, -2.43255796341582_wp,  0.08010638722000_wp, &
      & -3.59979564847825_wp,  0.11701496350567_wp,  0.31696113630858_wp, &
      & -3.39848140530655_wp, -0.26991338518467_wp,  2.58331248954612_wp, &
      &  2.75233642207676_wp,  1.43301426276031_wp, -0.14733010096095_wp, &
      &  1.06040060799106_wp,  2.28987489409000_wp, -0.24027472166221_wp, &
      &  4.99066360880505_wp,  2.86804088996451_wp,  0.48446158145191_wp, &
      &  5.21782553359459_wp,  4.45152905211779_wp, -0.80533747958686_wp, &
      &  6.59743573908754_wp,  1.60540231477343_wp,  0.30535448885380_wp, &
      &  4.91535326810565_wp,  3.57427854369579_wp,  2.41652350480412_wp, &
      & -5.80675306201883_wp,  1.52777577322431_wp, -0.81424188277139_wp, &
      & -5.47197406324459_wp,  2.13935007356211_wp, -2.74760782520766_wp, &
      & -7.45277386049152_wp,  0.29461140225852_wp, -0.78334508077450_wp, &
      & -6.22137997227954_wp,  3.15696728844009_wp,  0.36256999378213_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLY_xad

subroutine GLY_xae(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 19
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "C", "H", "H", "N", "H", "C", "O", "C", "O", "N", "H", "C", "H", "H", &
      & "H", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  5.25721959349274_wp, -1.94285812878400_wp, -0.46772350830604_wp, &
      &  5.86768557398737_wp, -1.10357003287924_wp, -2.25191933833981_wp, &
      &  6.96674336701165_wp, -2.33608967714845_wp,  0.61983173080175_wp, &
      &  3.83179279962980_wp, -4.20628011088961_wp, -0.89753399183467_wp, &
      &  2.04212810386865_wp, -4.19908357116913_wp, -0.24564681520542_wp, &
      &  3.59279501279957_wp, -0.09052320609638_wp,  0.96508127447659_wp, &
      &  1.40268409834591_wp, -0.59309781749461_wp,  1.52906591646642_wp, &
      &  4.82256803438495_wp, -6.14461474076049_wp, -2.24659315584308_wp, &
      &  6.97554361283315_wp, -6.06715842975971_wp, -3.10068465336403_wp, &
      &  4.73772316390950_wp,  2.12487839049720_wp,  1.53286756420870_wp, &
      &  6.55521999880749_wp,  2.34608214671874_wp,  1.02949960110558_wp, &
      &  3.47133785427091_wp,  4.12360904630987_wp,  2.90598597158576_wp, &
      &  4.29512134317192_wp,  4.38879420135075_wp,  4.77219726052631_wp, &
      &  3.56730605997107_wp,  5.89405035159564_wp,  1.86636881985675_wp, &
      &  1.50861856479080_wp,  3.56903483641907_wp,  3.11672446990435_wp, &
      &  3.11053234572430_wp, -8.39838570598497_wp, -2.60083646410921_wp, &
      &  2.96833939266405_wp, -8.80237992698345_wp, -4.60976686446968_wp, &
      &  1.22997450531077_wp, -8.11408305941850_wp, -1.82451216970939_wp, &
      &  3.97682767344838_wp,-10.02789008726395_wp, -1.69640329961806_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLY_xae

subroutine GLY_xag(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 19
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "C", "H", "H", "N", "H", "C", "O", "C", "O", "N", "H", "C", "H", "H", &
      & "H", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.40090381281468_wp, -3.73458503857643_wp, -3.79318735556084_wp, &
      &  0.07573905254333_wp, -2.35294315197221_wp, -5.28207155190357_wp, &
      & -0.34304145893837_wp, -5.52156709756223_wp, -4.48812193626667_wp, &
      &  3.07964501875151_wp, -3.88406168937640_wp, -3.29639991514089_wp, &
      &  4.22901849221458_wp, -2.53695402445116_wp, -3.99789645073915_wp, &
      & -1.26840623046409_wp, -2.96036536022323_wp, -1.56839531811284_wp, &
      & -3.57061851851258_wp, -3.02894203972890_wp, -1.77388741645917_wp, &
      &  4.34118924494342_wp, -5.94213819648810_wp, -2.33529429869630_wp, &
      &  6.64162698292798_wp, -5.95073835624741_wp, -2.15507701158651_wp, &
      & -0.01140876522361_wp, -2.22561452520780_wp,  0.53094967067308_wp, &
      &  1.88720199963144_wp, -2.20412798510163_wp,  0.43973831090062_wp, &
      & -1.31377467559354_wp, -1.38189276290808_wp,  2.78077779826958_wp, &
      & -1.12695425303065_wp,  0.65077696980410_wp,  3.04127818258842_wp, &
      & -3.30007662890388_wp, -1.83957382420038_wp,  2.54837055328836_wp, &
      & -0.57915161868359_wp, -2.34259475439700_wp,  4.44210719601334_wp, &
      &  2.70550408627520_wp, -8.12943713122838_wp, -1.51937171350866_wp, &
      &  1.15015400556981_wp, -7.51272326860108_wp, -0.32292788721370_wp, &
      &  3.88572962124111_wp, -9.46026579239744_wp, -0.50328304399290_wp, &
      &  1.90249298863180_wp, -9.07046867758468_wp, -3.16561537494664_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine GLY_xag

subroutine HIS_xah(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "N", "C", "H", "C", &
      & "N", "H", "H", "C", "C", "O", "H", "H", "H", "N", "C", "H", "H", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.16616723377026_wp, -1.90856486593797_wp,  0.99942384043339_wp, &
      &  3.23179063518608_wp, -2.63902079167480_wp,  2.75386053304163_wp, &
      &  0.78134038619985_wp, -0.86422644044582_wp,  0.10639894072651_wp, &
      &  0.70763999736620_wp, -1.18920435714193_wp, -1.92744658530206_wp, &
      & -1.40355707259994_wp, -2.21940569533356_wp,  1.40581545708851_wp, &
      &  0.72907206511532_wp,  1.99288943003244_wp,  0.60596728053010_wp, &
      & -1.12941162111535_wp, -4.24654558729373_wp,  1.15833551084408_wp, &
      & -1.32587721578431_wp, -1.80371580165763_wp,  3.42697531933185_wp, &
      & -3.92380535966480_wp, -1.54807195266363_wp,  0.35952055039144_wp, &
      & -0.53956075852380_wp,  2.93589354816268_wp,  2.32168590885520_wp, &
      & -5.06107315328229_wp,  0.74933366007350_wp,  0.73682618192162_wp, &
      & -5.62941493200341_wp, -2.89828271231739_wp, -1.08526613754116_wp, &
      & -4.20807068462024_wp,  2.18711869969908_wp,  1.66554174936017_wp, &
      & -7.32865994454816_wp,  0.70655904357230_wp, -0.46689172109133_wp, &
      & -7.74244281881295_wp, -1.49025376175098_wp, -1.59439435217979_wp, &
      & -5.43199475550924_wp, -4.80968657969887_wp, -1.75754680616975_wp, &
      & -8.59479388599313_wp,  2.29825099384506_wp, -0.46799645081574_wp, &
      &  5.30614795712740_wp, -1.83517001547315_wp, -0.40878405593333_wp, &
      &  7.58782949597019_wp, -3.10631295033388_wp,  0.72321720658223_wp, &
      &  5.38007563740853_wp, -0.80793244060326_wp, -2.49506022513032_wp, &
      &  9.10204557354434_wp, -1.72558473201386_wp,  0.87117655715712_wp, &
      &  8.21135473301014_wp, -4.58067117135337_wp, -0.56509850109335_wp, &
      &  7.22816129933375_wp, -3.92823501427881_wp,  2.57090128407495_wp, &
      &  2.21989022478915_wp,  3.31570066031946_wp, -0.96629234258615_wp, &
      &  2.59739622142885_wp,  6.01055463282397_wp, -0.68524997293860_wp, &
      &  3.34607832180912_wp,  2.29357583761739_wp, -2.13407751513418_wp, &
      &  2.62220379319309_wp,  6.90658612801099_wp, -2.53274324860585_wp, &
      &  1.04217543754550_wp,  6.75931292167807_wp,  0.42183841159632_wp, &
      &  4.36286800078608_wp,  6.42738139910717_wp,  0.28862977117266_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine HIS_xah

subroutine HIS_xam(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "N", "C", "H", "C", &
      & "N", "H", "H", "C", "C", "O", "H", "H", "H", "N", "C", "H", "H", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.75675658421712_wp, -0.81676847541084_wp,  0.17785387910062_wp, &
      &  5.29462330508551_wp,  0.28268202701730_wp,  0.43672939351505_wp, &
      &  1.37417639043907_wp,  0.45808608277985_wp, -0.06431391536125_wp, &
      &  0.49579448327011_wp, -0.14468004058304_wp, -1.83140383781792_wp, &
      & -0.48093073105108_wp, -0.19077123708424_wp,  2.12112129955568_wp, &
      &  1.87434811124771_wp,  3.28489836219003_wp, -0.19942624743857_wp, &
      & -0.60205997428014_wp,  1.37177788007300_wp,  3.45612141434965_wp, &
      &  0.28766360549075_wp, -1.81419035196351_wp,  3.12763101703097_wp, &
      & -3.02448066772645_wp, -0.79490432850752_wp,  1.11292695629532_wp, &
      &  3.98881796186726_wp,  4.17278574009415_wp,  0.14777555508927_wp, &
      & -3.36996579602833_wp, -2.87616202545325_wp, -0.39174170125136_wp, &
      & -5.35711710855440_wp,  0.37830576044568_wp,  1.17477671105204_wp, &
      & -1.94765320667073_wp, -4.08281549367780_wp, -0.84147279438904_wp, &
      & -5.82434767008464_wp, -2.88938362895840_wp, -1.15112793623253_wp, &
      & -7.09480611403792_wp, -0.93806742509946_wp, -0.23249373426670_wp, &
      & -5.86680866696090_wp,  2.06286698411708_wp,  2.19986901407811_wp, &
      & -6.59077392401978_wp, -4.34351411875168_wp, -2.34892754847757_wp, &
      &  3.87716123747222_wp, -3.32828798137712_wp, -0.25014541163423_wp, &
      &  6.45521710477463_wp, -4.51810255442530_wp, -0.08542373740588_wp, &
      &  1.96696947280456_wp, -4.57715533528005_wp, -0.74036998781454_wp, &
      &  6.90581754558333_wp, -5.35694076139873_wp, -1.90668449562445_wp, &
      &  6.39299233591516_wp, -6.03375803009226_wp,  1.30006261267799_wp, &
      &  7.92193420482446_wp, -3.17690743110776_wp,  0.43052760988591_wp, &
      & -0.22134858129860_wp,  4.65996288340469_wp, -0.68334678461145_wp, &
      & -0.19598320772313_wp,  7.39495497362994_wp, -0.77952889140920_wp, &
      & -1.87039246765427_wp,  3.72477098632684_wp, -0.89207832783487_wp, &
      &  1.73832957425238_wp,  8.00278344471757_wp, -0.47280891619000_wp, &
      & -0.83325196306080_wp,  8.06789019044172_wp, -2.61441031923544_wp, &
      & -1.39661069849837_wp,  8.19556592808195_wp,  0.68556644912370_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine HIS_xam

subroutine HIS_xaq(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "N", "C", "H", "C", &
      & "N", "H", "H", "C", "C", "O", "H", "H", "H", "N", "C", "H", "H", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.34795042620554_wp,  2.10325297439630_wp,  0.15744885630818_wp, &
      & -1.99160474771297_wp,  1.44854121696748_wp, -0.56387071451795_wp, &
      &  0.90884039772917_wp,  0.49974427773044_wp,  1.99486881797532_wp, &
      &  1.71752176301358_wp,  1.73877071084191_wp,  3.43414836866157_wp, &
      & -1.05135101723299_wp, -1.25774464592285_wp,  3.19868673388691_wp, &
      &  3.10735375843204_wp, -0.98151675610544_wp,  0.80252587377890_wp, &
      & -2.23850246856613_wp, -0.13751039335874_wp,  4.45443638221311_wp, &
      & -0.05856704035346_wp, -2.65487494444875_wp,  4.33887476297500_wp, &
      & -2.71924836152659_wp, -2.49354933333447_wp,  1.30179866356426_wp, &
      &  3.11832986192458_wp, -3.31119163685840_wp,  0.64618912598562_wp, &
      & -1.89199326231927_wp, -4.30007102493352_wp, -0.35751391526165_wp, &
      & -5.22588151665085_wp, -2.10701420367430_wp,  0.65960802362271_wp, &
      & -0.05659818365718_wp, -4.85022898897844_wp, -0.42173338933393_wp, &
      & -3.85824016379673_wp, -4.92027929708192_wp, -1.88634450909977_wp, &
      & -5.92690003505652_wp, -3.62650072807509_wp, -1.31974459872994_wp, &
      & -6.54939212316529_wp, -0.85111157867919_wp,  1.56503842813574_wp, &
      & -3.69796784983430_wp, -6.30847145964841_wp, -3.36425221511761_wp, &
      &  0.51140616578029_wp,  4.41972625732598_wp, -0.49411109245921_wp, &
      & -1.18228104688255_wp,  5.92607251501026_wp, -2.21475367409404_wp, &
      &  2.54167741295999_wp,  5.26075974969036_wp,  0.28560240861051_wp, &
      & -2.74467005042478_wp,  4.82545779949852_wp, -2.96770225781949_wp, &
      & -0.05484615302003_wp,  6.67624059814294_wp, -3.75794475073407_wp, &
      & -1.93216076055436_wp,  7.52036422636095_wp, -1.15409215056030_wp, &
      &  4.99566692081428_wp,  0.51084201669952_wp,  0.00383914026922_wp, &
      &  7.21429878312197_wp, -0.47962368044981_wp, -1.25178414300235_wp, &
      &  4.72617694624270_wp,  2.40546306614547_wp,  0.17708798774767_wp, &
      &  7.22447308105490_wp,  0.02426215112699_wp, -3.24694774999337_wp, &
      &  8.91813583507021_wp,  0.25561415442779_wp, -0.36920231250255_wp, &
      &  7.18254423767951_wp, -2.52237483167589_wp, -1.07510409101089_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine HIS_xaq

subroutine HIS_xau(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "N", "C", "H", "C", &
      & "N", "H", "H", "C", "C", "O", "H", "H", "H", "N", "C", "H", "H", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.12075618293491_wp,  0.19864201363805_wp, -0.57873189806485_wp, &
      & -2.85978584903121_wp,  0.68519160363588_wp, -2.41208565811085_wp, &
      & -1.38696789893829_wp,  1.43437610416932_wp,  1.13065428380078_wp, &
      & -2.44225435836602_wp,  2.68335533244520_wp,  2.39324326739686_wp, &
      &  0.06589858963482_wp, -0.50635060357084_wp,  2.74152629612049_wp, &
      &  0.41393993678389_wp,  2.97297229630845_wp, -0.52003872229931_wp, &
      &  1.02877012619968_wp,  0.44271703096881_wp,  4.29870602909081_wp, &
      & -1.33672377294619_wp, -1.75295915991967_wp,  3.59316585319374_wp, &
      &  1.98437833484425_wp, -1.90481873224606_wp,  1.24454285888696_wp, &
      &  0.24195699605032_wp,  2.96701898582813_wp, -2.83430456182629_wp, &
      &  1.39318908899300_wp, -3.39187778392576_wp, -0.79676374924084_wp, &
      &  4.58943991579258_wp, -2.00269473383442_wp,  1.37986502479797_wp, &
      & -0.36518139109975_wp, -3.64250010307440_wp, -1.47810375497825_wp, &
      &  3.59096722096215_wp, -4.29137701783278_wp, -1.78652925904057_wp, &
      &  5.57698253781662_wp, -3.48919823466438_wp, -0.49822671305839_wp, &
      &  5.78130122511549_wp, -1.09214287162981_wp,  2.75709573605824_wp, &
      &  3.63795468531762_wp, -5.50321739490262_wp, -3.41910804239193_wp, &
      & -5.43698443352558_wp, -0.56930627742314_wp,  0.26681297028407_wp, &
      & -7.26178716849146_wp, -1.40096052029868_wp, -1.76022853405199_wp, &
      & -5.96947416694222_wp, -0.61041307165882_wp,  2.51809264129533_wp, &
      & -8.84264177547991_wp, -0.08644213818596_wp, -1.79086462051916_wp, &
      & -8.00287730632925_wp, -3.25092587526699_wp, -1.26369072439638_wp, &
      & -6.40997428193898_wp, -1.47440700091841_wp, -3.62810362370021_wp, &
      &  2.17460157859929_wp,  4.31832857425198_wp,  0.74088638143336_wp, &
      &  4.22203104660957_wp,  5.53612475606706_wp, -0.61656154515421_wp, &
      &  2.37122742503036_wp,  4.01156698573238_wp,  2.60378877663271_wp, &
      &  5.13793266637517_wp,  6.88762819712322_wp,  0.62573521205179_wp, &
      &  5.60325650403779_wp,  4.15958094922166_wp, -1.27898157239760_wp, &
      &  3.44931028249104_wp,  6.51959786569785_wp, -2.24193767696774_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine HIS_xau

subroutine HIS_xav(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "C", "H", "H", "C", "O", "N", "C", "H", "C", &
      & "N", "H", "H", "C", "C", "O", "H", "H", "H", "N", "C", "H", "H", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.93747190862467_wp, -1.85921706889780_wp, -1.57927750620701_wp, &
      &  2.95733903945398_wp, -3.04690965413812_wp, -3.06555359737369_wp, &
      &  0.60849874919040_wp, -0.46698806763134_wp, -1.10746053645381_wp, &
      & -0.32395411055851_wp, -0.26006184465289_wp, -2.93740951904926_wp, &
      & -1.20984878268663_wp, -1.85335211533291_wp,  0.68969674886143_wp, &
      &  1.18102241113012_wp,  2.22106943247060_wp, -0.18677034296310_wp, &
      & -0.99532751565241_wp, -3.87411208322091_wp,  0.34637907725109_wp, &
      & -0.63353437045855_wp, -1.48624400350188_wp,  2.63134787409944_wp, &
      & -3.91517203323517_wp, -1.22008654650982_wp,  0.26624420052093_wp, &
      & -0.25292193615649_wp,  3.36836085913936_wp,  1.23714208184396_wp, &
      & -4.92338179060821_wp,  1.12231685293627_wp,  0.70806703707560_wp, &
      & -5.92453428056288_wp, -2.65025374824683_wp, -0.59525469431104_wp, &
      & -3.84400492817402_wp,  2.59939191296192_wp,  1.27398693833643_wp, &
      & -7.41825965030897_wp,  1.02779785496927_wp,  0.11155963221339_wp, &
      & -8.10035225650524_wp, -1.24681605230188_wp, -0.68532318732121_wp, &
      & -5.90025487774379_wp, -4.61646549027404_wp, -1.12357247014713_wp, &
      & -8.64559647090919_wp,  2.64035340341820_wp,  0.28477717885367_wp, &
      &  4.55802312635983_wp, -2.37992284852213_wp,  0.38257672991337_wp, &
      &  6.58702547479841_wp, -4.30042794471973_wp, -0.18392991187434_wp, &
      &  4.36271267959900_wp, -1.31959905118625_wp,  2.42346152960274_wp, &
      &  6.04204307557178_wp, -6.09571753313615_wp,  0.66160665363421_wp, &
      &  8.33555014621495_wp, -3.69315594960718_wp,  0.70123112283822_wp, &
      &  6.88632836608295_wp, -4.57887586957283_wp, -2.19775099842251_wp, &
      &  3.26806530728614_wp,  3.28616657837479_wp, -1.17971850378892_wp, &
      &  4.11336701908950_wp,  5.74256093585860_wp, -0.31639461872048_wp, &
      &  4.48930840216378_wp,  2.12418688312899_wp, -2.05523927448712_wp, &
      &  4.93524722673763_wp,  5.63404399675744_wp,  1.56851434258060_wp, &
      &  5.50857103530064_wp,  6.47120963315441_wp, -1.63250962246530_wp, &
      &  2.50722058392985_wp,  7.01684993509582_wp, -0.25051786097413_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine HIS_xav

subroutine ILE_xae(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 31
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", &
      & "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.20567553286199_wp,  1.88737351577296_wp, -1.11727673762583_wp, &
      & -0.48474453780906_wp, -0.60068853171041_wp,  0.02307456024999_wp, &
      &  1.94971430424278_wp, -2.09317916383979_wp, -0.46770069647194_wp, &
      &  2.17036255207312_wp, -3.59480740705717_wp, -2.22021716528553_wp, &
      & -2.81334944485480_wp, -1.96305174972987_wp, -0.97601963326623_wp, &
      & -5.18375896812984_wp, -0.30553178105327_wp, -0.82249025499422_wp, &
      & -3.14912542537653_wp, -4.44922793240013_wp,  0.44695886994094_wp, &
      & -5.76211788108837_wp,  0.72296597339145_wp,  1.80726490571433_wp, &
      & -0.63851302660223_wp,  2.09478175017944_wp, -2.95750523889051_wp, &
      & -0.66602926888384_wp, -0.27794582990041_wp,  2.05034127006218_wp, &
      & -2.45121623158631_wp, -2.41939804894948_wp, -2.95820584582738_wp, &
      & -5.01286217679413_wp,  1.27772656201988_wp, -2.13285698218976_wp, &
      & -6.77929325720576_wp, -1.43156319743650_wp, -1.48930021970474_wp, &
      & -1.54281965606490_wp, -5.69112004423038_wp,  0.12881558432666_wp, &
      & -3.33272722282436_wp, -4.12942718061893_wp,  2.47395734439571_wp, &
      & -4.84627340270962_wp, -5.42073358609275_wp, -0.19773884114086_wp, &
      & -5.97744134463125_wp, -0.79087274765922_wp,  3.18403392311989_wp, &
      & -7.51656496547542_wp,  1.79809673485998_wp,  1.79140072414144_wp, &
      & -4.27164723508443_wp,  1.97956210831323_wp,  2.47017775648719_wp, &
      &  3.82119905880779_wp, -1.52292284553738_wp,  1.18011364680856_wp, &
      &  3.54106461592451_wp,  0.01611784587259_wp,  2.28070543679425_wp, &
      &  6.35198484787910_wp, -2.49192679854230_wp,  0.80708707462236_wp, &
      &  7.41397617232831_wp, -1.37535149989905_wp, -0.56157164916240_wp, &
      &  7.35216596092773_wp, -2.48977262188349_wp,  2.59936970402206_wp, &
      &  6.21684156989398_wp, -4.41282187382739_wp,  0.09941155364093_wp, &
      &  0.96800327884376_wp,  3.80638380994310_wp,  0.11269629447304_wp, &
      &  1.80911540580040_wp,  3.56991344677424_wp,  2.26896139387893_wp, &
      &  1.13972464711536_wp,  6.26984587817686_wp, -1.30683936820214_wp, &
      &  3.11543245879378_wp,  6.82058272552165_wp, -1.42895804693528_wp, &
      &  0.15517350060191_wp,  7.71258051068637_wp, -0.22393240854481_wp, &
      &  0.33688089732351_wp,  6.17555278275869_wp, -3.19545759144591_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ILE_xae

subroutine ILE_xag(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 31
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", &
      & "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.28133319222194_wp, -0.67403783217144_wp, -0.85797198470987_wp, &
      &  0.11320951852765_wp, -0.47725607815091_wp,  0.78268788041350_wp, &
      & -1.96244418379251_wp, -2.12996426912348_wp, -0.34155127523764_wp, &
      & -1.87025889903417_wp, -2.89587806288952_wp, -2.52970309846312_wp, &
      & -0.81053158998928_wp,  2.27055602123666_wp,  1.09751831155015_wp, &
      & -1.73788426278045_wp,  3.35036046457669_wp, -1.41408378321664_wp, &
      &  1.27353947200069_wp,  3.87252038098767_wp,  2.27569204144700_wp, &
      & -2.81823140545181_wp,  6.01066883852298_wp, -1.18854116528265_wp, &
      &  1.87802055558808_wp, -1.30445908985715_wp, -2.61481901963878_wp, &
      &  0.67224482164203_wp, -1.19625093526069_wp,  2.63649551497556_wp, &
      & -2.41644428565060_wp,  2.21554349354523_wp,  2.40347253274474_wp, &
      & -3.17943892993183_wp,  2.10409847126096_wp, -2.20564366009940_wp, &
      & -0.16217754913775_wp,  3.34550678038751_wp, -2.74846158886491_wp, &
      &  2.01447077294233_wp,  2.99068604807846_wp,  3.97969529141525_wp, &
      &  2.84143126925330_wp,  4.08180414572318_wp,  0.95592498844974_wp, &
      &  0.58038824225688_wp,  5.74808222540876_wp,  2.75549294227010_wp, &
      & -4.31501285765704_wp,  6.07646209348243_wp,  0.22715103840482_wp, &
      & -3.62133213663596_wp,  6.62743430043299_wp, -2.97961232961042_wp, &
      & -1.37931417086077_wp,  7.37669467383049_wp, -0.64958388457004_wp, &
      & -3.93484614054508_wp, -2.57811011053827_wp,  1.22406823637300_wp, &
      & -3.78268369494944_wp, -2.02356720101279_wp,  3.03373975050733_wp, &
      & -6.09700016044291_wp, -4.06795487322055_wp,  0.45376023441053_wp, &
      & -5.98396773866323_wp, -5.99782622130986_wp,  1.16101418628856_wp, &
      & -7.83133121106005_wp, -3.20029617115134_wp,  1.13126700228541_wp, &
      & -6.11386389444150_wp, -4.12114920105465_wp, -1.59610055337038_wp, &
      &  4.64103180653941_wp, -1.07300965903423_wp,  0.10277425091820_wp, &
      &  5.12400683391828_wp, -0.89809065918422_wp,  2.35990141807532_wp, &
      &  6.63881958314944_wp, -1.70320305767950_wp, -1.83414993834606_wp, &
      &  7.34919518053494_wp, -3.59335613082280_wp, -1.44803954772694_wp, &
      &  8.20393472284326_wp, -0.39197349487920_wp, -1.60969586387184_wp, &
      &  5.94512615022772_wp, -1.61683806179804_wp, -3.76554262617263_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ILE_xag

subroutine ILE_xaj(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 31
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", &
      & "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.35346655715600_wp,  2.09805094714897_wp,  0.75185144229675_wp, &
      &  0.37594818603958_wp, -0.30133824881456_wp, -0.38268873106418_wp, &
      & -1.51072980458822_wp, -2.30182739749198_wp,  0.52710670548584_wp, &
      & -1.11074127098702_wp, -3.64818751398475_wp,  2.37204676397615_wp, &
      &  3.09714815624698_wp, -0.99866155935508_wp,  0.25132150152083_wp, &
      &  4.90633999835512_wp,  1.12278893522341_wp, -0.50783460517415_wp, &
      &  3.76367206622877_wp, -3.47427475679073_wp, -1.06758444446671_wp, &
      &  7.63635002731961_wp,  0.63584817068831_wp,  0.27130119053223_wp, &
      &  0.26749465988631_wp,  2.48983922197094_wp,  2.50665961515916_wp, &
      &  0.17932648157168_wp, -0.05208573169882_wp, -2.42129108401832_wp, &
      &  3.20424246388463_wp, -1.29805582554813_wp,  2.29531567422379_wp, &
      &  4.27802986085093_wp,  2.89774320462731_wp,  0.32988811273434_wp, &
      &  4.79598818154745_wp,  1.38111334501936_wp, -2.55623078634483_wp, &
      &  2.45457125879016_wp, -4.96725234027001_wp, -0.54007317955753_wp, &
      &  3.70219542780668_wp, -3.22550324090901_wp, -3.11539176208617_wp, &
      &  5.65592241272074_wp, -4.09816124279783_wp, -0.56286484600249_wp, &
      &  7.75996217763882_wp,  0.25870847485129_wp,  2.29333198636543_wp, &
      &  8.81816419365813_wp,  2.26933235284558_wp, -0.13892473528902_wp, &
      &  8.44063073772064_wp, -0.97554815306100_wp, -0.71986881293394_wp, &
      & -3.66289992184786_wp, -2.36181746390332_wp, -0.85404915337089_wp, &
      & -3.94938174864267_wp, -0.88103758841114_wp, -2.03007561580378_wp, &
      & -5.78821265930840_wp, -3.88212995826512_wp, -0.04271062031820_wp, &
      & -7.02405750408961_wp, -4.22865361207623_wp, -1.64411418625566_wp, &
      & -5.07977130440577_wp, -5.66847798483668_wp,  0.67564125526376_wp, &
      & -6.86461013615554_wp, -2.96826983880090_wp,  1.45866251983655_wp, &
      & -2.17937736707027_wp,  3.57368270931167_wp, -0.28088446539016_wp, &
      & -3.25554774743059_wp,  3.01610178401591_wp, -2.26588015572999_wp, &
      & -2.78042107308267_wp,  5.97275055419907_wp,  1.13275561971608_wp, &
      & -4.78501711718516_wp,  5.98429317728495_wp,  1.58348671095030_wp, &
      & -2.41072313185373_wp,  7.56728987496013_wp, -0.10975744056920_wp, &
      & -1.68900703981687_wp,  6.19300642098842_wp,  2.85918991300524_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ILE_xaj

subroutine ILE_xak(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 31
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", &
      & "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.77807096374319_wp, -1.05865259024732_wp,  0.41877537094431_wp, &
      & -0.34004930638133_wp, -0.37069746208589_wp, -0.57935619731324_wp, &
      &  1.47302974645084_wp, -2.47039729032925_wp,  0.18376910063556_wp, &
      &  1.04314438742045_wp, -3.83075172075315_wp,  2.01531283743590_wp, &
      &  0.50471347475642_wp,  2.24531633410729_wp,  0.40663325309397_wp, &
      &  2.72900131435766_wp,  3.31382854913464_wp, -1.09203059484161_wp, &
      &  1.02671579167646_wp,  2.17669503036053_wp,  3.23671339605153_wp, &
      &  3.14870996361675_wp,  6.12428813141918_wp, -0.61605914439886_wp, &
      & -2.78285164988460_wp, -2.41682705991338_wp,  1.75559712509848_wp, &
      & -0.49061615658257_wp, -0.25185896690551_wp, -2.63567503795455_wp, &
      & -1.12522893090487_wp,  3.46167569178002_wp,  0.04377675023780_wp, &
      &  2.37383967740936_wp,  3.03024964678957_wp, -3.10978521867402_wp, &
      &  4.44956317057066_wp,  2.27439340335521_wp, -0.62034651770387_wp, &
      & -0.53626746222264_wp,  1.30277200515702_wp,  4.25039016370851_wp, &
      &  2.73174813757251_wp,  1.09185901061480_wp,  3.64752749856965_wp, &
      &  1.27937189088376_wp,  4.07475656372928_wp,  3.98688890390792_wp, &
      &  1.44598617061963_wp,  7.19383419027888_wp, -1.06020275293761_wp, &
      &  4.67850760320344_wp,  6.85325441463730_wp, -1.78321278114961_wp, &
      &  3.63561486306016_wp,  6.49081007712198_wp,  1.34808775712812_wp, &
      &  3.57675881883496_wp, -2.66472103012852_wp, -1.25682185540800_wp, &
      &  3.73143703037381_wp, -1.52082429601062_wp, -2.76397699516426_wp, &
      &  5.55441739433336_wp, -4.48088032068636_wp, -0.73219947864579_wp, &
      &  5.70925347429810_wp, -5.86729088135364_wp, -2.24374409637783_wp, &
      &  7.36681948535109_wp, -3.53976333424209_wp, -0.49103418482236_wp, &
      &  5.05205800406541_wp, -5.44241301187459_wp,  1.00796905004926_wp, &
      & -4.93941053845376_wp, -0.11155519840655_wp, -0.58317523060242_wp, &
      & -4.92841712763509_wp,  1.43065856308787_wp, -2.31591878671499_wp, &
      & -7.35705117361481_wp, -1.07757400675203_wp,  0.58252025671478_wp, &
      & -8.47280998789707_wp, -1.98534889316706_wp, -0.88530351493780_wp, &
      & -8.43368830344812_wp,  0.52784126795853_wp,  1.27922140814577_wp, &
      & -7.04030095678171_wp, -2.40457452489694_wp,  2.11839961475330_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ILE_xak

subroutine ILE_xaq(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 31
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", &
      & "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.77356300120155_wp, -1.95601139252937_wp, -0.60270778410803_wp, &
      &  0.25487485029745_wp, -0.02483438456578_wp,  1.06630333620323_wp, &
      & -0.60815254139485_wp,  2.63749992627159_wp,  0.27474438597040_wp, &
      &  0.87854301789540_wp,  4.28424579738403_wp, -0.40430027339094_wp, &
      &  3.12833850929390_wp, -0.23178184231434_wp,  1.19688545551024_wp, &
      &  4.35779720715122_wp, -0.25060453472931_wp, -1.42722765297703_wp, &
      &  3.89222567774440_wp, -2.54619481422013_wp,  2.73423994096040_wp, &
      &  7.09713932487869_wp,  0.63267054431143_wp, -1.34618876077949_wp, &
      &  0.28510491466002_wp, -2.51605623701070_wp, -2.07799727957384_wp, &
      & -0.55578385936536_wp, -0.39659382285754_wp,  2.92921719092715_wp, &
      &  3.76711037027489_wp,  1.47590787645141_wp,  2.16033129314257_wp, &
      &  3.30419629620699_wp,  0.99052573928947_wp, -2.68842823051526_wp, &
      &  4.28907740867909_wp, -2.16700759825128_wp, -2.20713531442386_wp, &
      &  3.20674168832113_wp, -2.41577800252726_wp,  4.67176439920819_wp, &
      &  3.10887844196984_wp, -4.26023043412288_wp,  1.89830005647957_wp, &
      &  5.93813457555290_wp, -2.75501184316624_wp,  2.80059600849668_wp, &
      &  7.17978415073679_wp,  2.58866642513462_wp, -0.71148713970336_wp, &
      &  7.97271161623658_wp,  0.52764765257425_wp, -3.20509073656235_wp, &
      &  8.22497357472198_wp, -0.50879458144305_wp, -0.05684396049269_wp, &
      & -3.13721919265754_wp,  2.94912708720770_wp,  0.44459932556516_wp, &
      & -4.16419882316187_wp,  1.44293896708413_wp,  1.03304971731946_wp, &
      & -4.36481587204237_wp,  5.28295112811646_wp, -0.27072025025123_wp, &
      & -5.59727354747922_wp,  5.00959296768834_wp, -1.89630751257860_wp, &
      & -5.47857864953670_wp,  6.02828207753903_wp,  1.28922959947616_wp, &
      & -2.89957110510098_wp,  6.63319499146603_wp, -0.75856900120234_wp, &
      & -3.10973223701798_wp, -2.95543464829091_wp, -0.31150580438865_wp, &
      & -4.58409369806412_wp, -2.22733921963767_wp,  1.33718820894327_wp, &
      & -3.81100628931047_wp, -5.02442527698202_wp, -2.13986439923616_wp, &
      & -4.39232472339504_wp, -6.67229623735338_wp, -1.05954187803181_wp, &
      & -2.27917470022153_wp, -5.54506940230676_wp, -3.40581204484746_wp, &
      & -5.42796486088362_wp, -4.40289018109290_wp, -3.24620634125500_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine ILE_xaq

subroutine LEU_xad(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 31
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", &
      & "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.25130767164541_wp,  1.69573068572740_wp, -1.04588306727819_wp, &
      & -0.16064482286832_wp, -0.70464018097290_wp,  0.23545257436538_wp, &
      &  2.05823938005174_wp, -2.48876862705978_wp, -0.30995657709335_wp, &
      &  1.91109038944602_wp, -4.24469193012707_wp, -1.81492680270801_wp, &
      & -2.65606970018828_wp, -1.85832596310725_wp, -0.57546209147048_wp, &
      & -4.95533503615861_wp, -0.16753795619405_wp, -0.14155901250673_wp, &
      & -7.33878869955968_wp, -1.56202896305000_wp, -0.96379744158369_wp, &
      & -5.17329342432998_wp,  0.67007789177319_wp,  2.60655794531092_wp, &
      & -0.38863251232082_wp,  1.88942129326354_wp, -2.82625110541075_wp, &
      & -0.16322666786523_wp, -0.27835727431329_wp,  2.24988936057035_wp, &
      & -2.90050318338728_wp, -3.62394143191357_wp,  0.46663955890623_wp, &
      & -2.51627671280810_wp, -2.39718722358297_wp, -2.56311356168375_wp, &
      & -4.76393413502936_wp,  1.52392455250969_wp, -1.31355033117285_wp, &
      & -7.62685328269034_wp, -3.23339853582801_wp,  0.20921062569873_wp, &
      & -9.00340091692895_wp, -0.36443508547508_wp, -0.77791601636664_wp, &
      & -7.20657932575218_wp, -2.18222199120626_wp, -2.92331720724815_wp, &
      & -5.22325844096826_wp, -0.97403047773921_wp,  3.85266710914457_wp, &
      & -6.90877364956651_wp,  1.73785339547875_wp,  2.90530417132684_wp, &
      & -3.59757292690422_wp,  1.86273994311880_wp,  3.18359301312197_wp, &
      &  4.16246159121422_wp, -1.89531330938409_wp,  1.01700153589228_wp, &
      &  4.13451772814959_wp, -0.23972956839810_wp,  1.97577463652524_wp, &
      &  6.52653175327318_wp, -3.19543107310430_wp,  0.57502475454891_wp, &
      &  7.63585399680203_wp, -3.21259336629706_wp,  2.30254771637927_wp, &
      &  6.10654538879367_wp, -5.12257228319690_wp,  0.01077143405202_wp, &
      &  7.62873673875368_wp, -2.30740991174170_wp, -0.92290397415842_wp, &
      &  1.67455330336019_wp,  3.56332700392606_wp, -0.02233060939239_wp, &
      &  2.70858515584765_wp,  3.34344859251859_wp,  2.05095699406443_wp, &
      &  1.88742735583598_wp,  5.95066499228300_wp, -1.56119083296537_wp, &
      &  3.87722883768627_wp,  6.34760023481281_wp, -1.88405954173847_wp, &
      &  1.11230345190937_wp,  7.50290509534117_wp, -0.45959696246968_wp, &
      &  0.90827612609041_wp,  5.84542007211738_wp, -3.36409309673151_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine LEU_xad

subroutine LEU_xae(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 31
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", &
      & "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.71070257568859_wp, -0.07565482470483_wp, -0.80696488208727_wp, &
      & -0.04009447634881_wp, -0.57962964810232_wp, -0.35425276817293_wp, &
      &  1.40928264720763_wp,  1.90389804989421_wp, -0.72970923826958_wp, &
      &  2.43050150879309_wp,  2.46783512476130_wp, -2.73271370786005_wp, &
      &  0.90138438160160_wp, -2.65317081132583_wp, -2.10649797536304_wp, &
      &  3.22021456233019_wp, -4.08078063683009_wp, -1.11576379699174_wp, &
      &  5.42191518547256_wp, -2.33293420492917_wp, -0.47501294900918_wp, &
      &  2.55149499612923_wp, -5.73780286015763_wp,  1.14766548030458_wp, &
      & -3.37876650986999_wp, -0.25846005298489_wp, -2.57793967452734_wp, &
      &  0.09403237042210_wp, -1.14684627781575_wp,  1.62081136149460_wp, &
      &  1.33207155580111_wp, -1.78967445626646_wp, -3.92938896732214_wp, &
      & -0.63081995401693_wp, -4.01040608165842_wp, -2.37761120777904_wp, &
      &  3.82654200166767_wp, -5.32747400765832_wp, -2.64704977817208_wp, &
      &  4.96015623281413_wp, -1.12138943462117_wp,  1.12955933820576_wp, &
      &  7.07780453329421_wp, -3.44017433260281_wp,  0.04879616344932_wp, &
      &  5.91076101148810_wp, -1.11680074726502_wp, -2.05948829069018_wp, &
      &  1.97364541043994_wp, -4.60004848690908_wp,  2.76615180283099_wp, &
      &  4.17982091010506_wp, -6.85319142799668_wp,  1.73529012403397_wp, &
      &  1.01489091414511_wp, -7.03569502771803_wp,  0.70198770165471_wp, &
      &  1.38511057943880_wp,  3.41034695645130_wp,  1.33621366656952_wp, &
      &  0.10903111743279_wp,  2.95995385134312_wp,  2.68927360701001_wp, &
      &  2.37855042674743_wp,  5.95629799880465_wp,  1.19240792100458_wp, &
      &  1.11799245601074_wp,  7.22024875208591_wp,  0.16241512104579_wp, &
      &  2.65494970236335_wp,  6.67675176544850_wp,  3.09376972429377_wp, &
      &  4.17907374530673_wp,  5.91341855880698_wp,  0.20875736505549_wp, &
      & -4.22058060405730_wp,  0.91609439954159_wp,  1.00983859783896_wp, &
      & -3.43744010027238_wp,  1.41094832010059_wp,  3.14460217005575_wp, &
      & -6.93710320333966_wp,  1.34868494919973_wp,  0.26983613783381_wp, &
      & -7.36211551594269_wp,  3.34820101779801_wp,  0.48102326549419_wp, &
      & -8.13829439300394_wp,  0.31382662830929_wp,  1.57644333508855_wp, &
      & -7.35890958015195_wp,  0.76160668776085_wp, -1.65292757093335_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine LEU_xae

subroutine LEU_xap(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 31
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", &
      & "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.94868549741146_wp, -2.11209808486697_wp, -0.98029500837367_wp, &
      &  0.16153334488158_wp, -0.66769625928889_wp,  0.03915782300710_wp, &
      & -0.21273727253653_wp,  2.12587041608050_wp, -0.62495167166559_wp, &
      &  0.80685536401769_wp,  3.11976129776480_wp, -2.45581467455375_wp, &
      &  2.63350040555812_wp, -1.68481700799557_wp, -0.99518402141237_wp, &
      &  4.98691457942580_wp, -0.37606333467373_wp,  0.04885854068557_wp, &
      &  5.07860524908487_wp, -0.45648623348382_wp,  2.92868560507811_wp, &
      &  7.33662867938341_wp, -1.61579794648712_wp, -1.06388386455652_wp, &
      & -1.79551015253363_wp, -2.81576625239636_wp, -2.74014259127915_wp, &
      &  0.06481983942016_wp, -0.89723661515875_wp,  2.08224424333184_wp, &
      &  2.61622136872956_wp, -1.47599833573575_wp, -3.04944029319437_wp, &
      &  2.72865093293097_wp, -3.70339483465871_wp, -0.55686091523086_wp, &
      &  4.93187837371554_wp,  1.59145741778146_wp, -0.56394213758020_wp, &
      &  4.94532894191621_wp, -2.40195845398677_wp,  3.60550464011944_wp, &
      &  6.85556429852250_wp,  0.32667194547565_wp,  3.61424419339855_wp, &
      &  3.55397706409669_wp,  0.62773973888498_wp,  3.78775242079454_wp, &
      &  7.46443764843404_wp, -3.58556844842956_wp, -0.46352147484873_wp, &
      &  9.04690073827624_wp, -0.65372459250545_wp, -0.44063043511643_wp, &
      &  7.29577444212388_wp, -1.57778935506012_wp, -3.12263109650142_wp, &
      & -1.79743536786073_wp,  3.34277555081409_wp,  0.97114542436764_wp, &
      & -2.84426346015123_wp,  2.23178427523441_wp,  2.12541078229011_wp, &
      & -2.60766755388308_wp,  5.89984393281368_wp,  0.43192141830913_wp, &
      & -3.32974897021107_wp,  6.75634774424987_wp,  2.15131869641234_wp, &
      & -0.99502004604051_wp,  6.97344103869851_wp, -0.24386574356001_wp, &
      & -4.06918775008562_wp,  5.95784886724689_wp, -1.02007268022981_wp, &
      & -4.18656349349802_wp, -2.28670264722946_wp,  0.25501551679873_wp, &
      & -4.55646342552030_wp, -1.25748135617669_wp,  2.30872891010317_wp, &
      & -6.19593461443681_wp, -3.84768714622874_wp, -1.02560586808836_wp, &
      & -6.76994841838099_wp, -5.34568992806728_wp,  0.25808667356163_wp, &
      & -5.57456037758729_wp, -4.67425538983172_wp, -2.80045367222634_wp, &
      & -7.83442068350144_wp, -2.65484151285747_wp, -1.36506828483991_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine LEU_xap

subroutine LEU_xaq(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 31
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", &
      & "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.57476145225391_wp, -1.52359271944836_wp, -1.73812706385267_wp, &
      & -0.00180892261315_wp, -0.52980573327180_wp, -1.55909846212022_wp, &
      & -0.11947540703616_wp,  2.32348565928105_wp, -1.08630245872549_wp, &
      & -1.23698590690225_wp,  3.72554959050547_wp, -2.56500879353284_wp, &
      & -1.60251773591567_wp, -2.00825403224099_wp,  0.33990548534790_wp, &
      & -4.30214357977962_wp, -1.01746592851642_wp,  0.60021281085357_wp, &
      & -5.71164075173741_wp, -1.02644123198781_wp, -1.91375607852195_wp, &
      & -5.70669572164450_wp, -2.59972796563736_wp,  2.55557410492472_wp, &
      &  3.16195118427392_wp, -2.05719499740619_wp, -3.46306299187323_wp, &
      & -0.80197963806843_wp, -0.75743901653920_wp, -3.43814596770767_wp, &
      & -0.67145305228774_wp, -1.97139775601682_wp,  2.17547575509367_wp, &
      & -1.66090436179208_wp, -3.98067076038285_wp, -0.28087455353490_wp, &
      & -4.20806817386602_wp,  0.92970397836085_wp,  1.28632512532180_wp, &
      & -5.68300762803164_wp, -2.91653652577460_wp, -2.74421768475643_wp, &
      & -7.68229819091142_wp, -0.49661431899550_wp, -1.63577438465933_wp, &
      & -4.89242644912205_wp,  0.30259429562905_wp, -3.25289997959092_wp, &
      & -5.86891208546301_wp, -4.55328543929918_wp,  1.91323677192887_wp, &
      & -7.61022654156452_wp, -1.87713986643587_wp,  2.86237710996700_wp, &
      & -4.72091446337037_wp, -2.60624128660210_wp,  4.36395578013170_wp, &
      &  0.97092337319221_wp,  3.13035311159736_wp,  1.06684230661723_wp, &
      &  1.97727369552273_wp,  1.85995307744085_wp,  2.09121527280076_wp, &
      &  0.96554162301518_wp,  5.79205206556269_wp,  1.69203283727823_wp, &
      &  1.75041648385821_wp,  6.02916937099128_wp,  3.57272429264089_wp, &
      &  2.07931441812147_wp,  6.88364820279336_wp,  0.34767391216481_wp, &
      & -0.95509516305270_wp,  6.52331235916759_wp,  1.65603026311738_wp, &
      &  4.24088021143273_wp, -1.81332300015387_wp,  0.18290971315853_wp, &
      &  3.82573290351182_wp, -1.12426236630291_wp,  2.36816977452855_wp, &
      &  6.70168305513898_wp, -3.07548025523663_wp, -0.51377476651701_wp, &
      &  8.24796503551922_wp, -1.87222489438878_wp,  0.10235275960788_wp, &
      &  6.89095026111537_wp, -3.46001982102564_wp, -2.52259528075914_wp, &
      &  6.83490994129623_wp, -4.84345823866535_wp,  0.52656797538133_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine LEU_xaq

subroutine LEU_xbb(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 31
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", &
      & "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.29309910497892_wp,  1.65570864806829_wp,  1.05241861600091_wp, &
      &  0.13828784837653_wp, -0.72223847808865_wp, -0.26917268454196_wp, &
      & -2.09745790270988_wp, -2.51086860805036_wp,  0.22151994563219_wp, &
      & -1.92340578173735_wp, -4.38479775020807_wp,  1.57240542404854_wp, &
      &  2.63643528520766_wp, -1.89738911763696_wp,  0.52710980764840_wp, &
      &  4.94521909811099_wp, -0.92544291007973_wp, -0.90818699007471_wp, &
      &  5.30455441731558_wp,  1.92684614153830_wp, -0.70591128725040_wp, &
      &  7.30186611097614_wp, -2.29359070517805_wp,  0.02639124691699_wp, &
      &  0.46973966790507_wp,  1.86484372815144_wp,  2.78075302535714_wp, &
      &  0.16497963803472_wp, -0.26593526902936_wp, -2.28313272365614_wp, &
      &  2.47583862056241_wp, -3.93396185755996_wp,  0.27148053240172_wp, &
      &  2.90306425120676_wp, -1.62299659216897_wp,  2.56140424046393_wp, &
      &  4.67172197650607_wp, -1.39512476862801_wp, -2.90369309933615_wp, &
      &  5.48446338502774_wp,  2.49501334278368_wp,  1.27040289851953_wp, &
      &  7.02708759500681_wp,  2.50143109359921_wp, -1.67852798784202_wp, &
      &  3.73453070561780_wp,  2.96903716334914_wp, -1.53054512570836_wp, &
      &  7.65196065482808_wp, -1.84018417592065_wp,  2.00786191981647_wp, &
      &  8.96400435645108_wp, -1.73282428894674_wp, -1.05165426942742_wp, &
      &  7.09276599464321_wp, -4.33623490015215_wp, -0.12659446450880_wp, &
      & -4.23461785069537_wp, -1.77383289391687_wp, -0.97129861552330_wp, &
      & -4.18474912430074_wp, -0.08359149813331_wp, -1.86886838458756_wp, &
      & -6.61114742211254_wp, -3.06443464850630_wp, -0.57793468129474_wp, &
      & -7.78711121675884_wp, -2.09947529105532_wp,  0.81150425922679_wp, &
      & -7.64344105377717_wp, -3.19305261777576_wp, -2.34927029210519_wp, &
      & -6.19976625277207_wp, -4.95187978810431_wp,  0.11205277516931_wp, &
      & -1.65276379788885_wp,  3.57380999930185_wp,  0.04607514448488_wp, &
      & -2.72075546576602_wp,  3.39365574929193_wp, -2.01503926463490_wp, &
      & -1.72446791370301_wp,  5.97469174225074_wp,  1.57620579845355_wp, &
      & -0.56685839265382_wp,  7.38464296450563_wp,  0.62572818329341_wp, &
      & -1.03342757130360_wp,  5.72859234531356_wp,  3.49530419434438_wp, &
      & -3.65433735387101_wp,  6.67256878393711_wp,  1.62112217323613_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine LEU_xbb

subroutine LYS_xan(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 33
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "H", "H", "H", "H", "H", "N", "H", "C", &
      & "H", "H", "H", "C", "O", "C", "H", "H", "H", "H", "C", "H", "H", "C", &
      & "H", "H", "N", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.07334916903130_wp, -2.19574968486908_wp, -0.77026753710657_wp, &
      & -1.06166612657699_wp, -0.60043865272590_wp,  0.22556273495023_wp, &
      & -1.55588454385165_wp,  2.14301062365209_wp, -0.57288894256972_wp, &
      & -0.46296312037716_wp,  3.13348588595147_wp, -2.36232288792062_wp, &
      &  1.48708896348865_wp, -1.52203171239620_wp, -0.70306963318666_wp, &
      &  3.67948403362485_wp, -0.15750104283050_wp,  0.56633265328739_wp, &
      & -2.82871196510198_wp, -2.98793460862456_wp, -2.48137889825513_wp, &
      & -1.17821577078039_wp, -0.73567024854244_wp,  2.27953888751721_wp, &
      &  1.61780488460523_wp, -3.54920477258153_wp, -0.33532979163913_wp, &
      &  3.53808330127676_wp, -0.39581832259754_wp,  2.61468759846634_wp, &
      &  3.53189966479582_wp,  1.86042745783656_wp,  0.17177658518422_wp, &
      & -3.32540023802572_wp,  3.30930842652724_wp,  0.85453904701771_wp, &
      & -4.35251000204289_wp,  2.18736388633117_wp,  2.01625799711235_wp, &
      & -4.24259364046361_wp,  5.79596353940397_wp,  0.17969307903463_wp, &
      & -5.67511788380064_wp,  5.71703246984640_wp, -1.29991039698857_wp, &
      & -2.66594127086281_wp,  6.90920168201332_wp, -0.51624806258646_wp, &
      & -5.04065190636687_wp,  6.69958130247631_wp,  1.84071100187397_wp, &
      & -5.32764283347130_wp, -2.44715133088087_wp,  0.41863421455961_wp, &
      & -5.81515974072021_wp, -1.34246846422850_wp,  2.40779150258883_wp, &
      & -7.20007943760237_wp, -4.19786755129512_wp, -0.82337250258749_wp, &
      & -7.63723425538094_wp, -5.71698754171430_wp,  0.48978933845776_wp, &
      & -6.51636800061883_wp, -5.00204772729362_wp, -2.58551980203181_wp, &
      & -8.93679165412314_wp, -3.15794005379796_wp, -1.17402685827713_wp, &
      &  1.59310124424983_wp, -1.22626001603908_wp, -2.74250391040043_wp, &
      &  6.23132996815291_wp, -1.12523449540248_wp, -0.34206843071034_wp, &
      &  6.38571055720480_wp, -0.90475270548590_wp, -2.38645375134235_wp, &
      &  6.38311618484168_wp, -3.14958623501257_wp,  0.05434388141319_wp, &
      &  8.43100509086916_wp,  0.24143348861281_wp,  0.89409402517469_wp, &
      &  8.24409504311959_wp,  0.09237201768246_wp,  2.95395469394312_wp, &
      &  8.32103186364023_wp,  2.24610047721702_wp,  0.42613218876341_wp, &
      & 10.82851278461951_wp, -0.72928938861216_wp, -0.08489902843119_wp, &
      & 12.28947932163187_wp,  0.21595189894382_wp,  0.71029609039009_wp, &
      & 11.03505760314329_wp, -2.55518927591615_wp,  0.45214828939886_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine LYS_xan

subroutine LYS_xao(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 33
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "H", "H", "H", "H", "H", "N", "H", "C", &
      & "H", "H", "H", "C", "O", "C", "H", "H", "H", "H", "C", "H", "H", "C", &
      & "H", "H", "N", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.08629288118877_wp, -2.19561127309795_wp, -0.77217893055321_wp, &
      & -1.06879606686991_wp, -0.61608658924481_wp,  0.23634288443356_wp, &
      & -1.53102984802615_wp,  2.13141574530499_wp, -0.56939167294673_wp, &
      & -0.40983742362732_wp,  3.10958800397462_wp, -2.34867178991298_wp, &
      &  1.47971484502131_wp, -1.55589129675000_wp, -0.67464620457756_wp, &
      &  3.67080881782305_wp, -0.22160454696147_wp,  0.62879195584726_wp, &
      & -2.83661151752225_wp, -2.99270493895025_wp, -2.48021099590057_wp, &
      & -1.20083247698939_wp, -0.74777619088370_wp,  2.28964206874229_wp, &
      &  1.58937150127564_wp, -3.58727643425421_wp, -0.32481723884977_wp, &
      &  3.54409665680710_wp, -0.53132058661176_wp,  2.66860726830112_wp, &
      &  3.50974930263916_wp,  1.80833214591191_wp,  0.30677605183608_wp, &
      & -3.30361007773525_wp,  3.31485339209112_wp,  0.83839008978212_wp, &
      & -4.34880376214547_wp,  2.20488174175011_wp,  1.99552260527961_wp, &
      & -4.18868179160437_wp,  5.81154501452241_wp,  0.15747382156164_wp, &
      & -5.61569230213706_wp,  5.74761437268114_wp, -1.32808061177308_wp, &
      & -2.59583748873156_wp,  6.90533962601390_wp, -0.53230790566849_wp, &
      & -4.98328471638590_wp,  6.72600829931603_wp,  1.81417357766629_wp, &
      & -5.35207705391596_wp, -2.42418436069169_wp,  0.40031295493727_wp, &
      & -5.84409781413938_wp, -1.31026328950066_wp,  2.38293392048313_wp, &
      & -7.23017418928331_wp, -4.16191765723655_wp, -0.85113016451490_wp, &
      & -7.68462505102300_wp, -5.67776148707435_wp,  0.46000324484997_wp, &
      & -6.54293162706832_wp, -4.97137068473344_wp, -2.60950586255879_wp, &
      & -8.95780193409875_wp, -3.11008592668001_wp, -1.21083256960746_wp, &
      &  1.60633375878005_wp, -1.24066001627663_wp, -2.71011415264373_wp, &
      &  6.22896524570903_wp, -1.13514352758590_wp, -0.31993549624624_wp, &
      &  6.36077259797159_wp, -0.80721133290823_wp, -2.35609502036389_wp, &
      &  6.39494201117646_wp, -3.17620769071747_wp, -0.03322800857001_wp, &
      &  8.43206186995541_wp,  0.18178876655507_wp,  0.98995969125296_wp, &
      &  8.26394746388151_wp,  2.21871263547301_wp,  0.71486715223124_wp, &
      &  8.30590057110490_wp, -0.13981983839829_wp,  3.02371506696762_wp, &
      & 10.95680155885510_wp, -0.57943795611462_wp,  0.16732903266954_wp, &
      & 11.15655125465616_wp, -2.46919075399082_wp,  0.40005961272418_wp, &
      & 11.12834218297359_wp, -0.27087148385594_wp, -1.71428160151084_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine LYS_xao

subroutine LYS_xap(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 33
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "H", "H", "H", "H", "H", "N", "H", "C", &
      & "H", "H", "H", "C", "O", "C", "H", "H", "H", "H", "C", "H", "H", "C", &
      & "H", "H", "N", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.08883048558477_wp, -2.19102660903053_wp, -0.76979713078107_wp, &
      & -1.06919374729060_wp, -0.61549565873415_wp,  0.24123050718566_wp, &
      & -1.52551360147593_wp,  2.13301490569785_wp, -0.56481932333436_wp, &
      & -0.39573297493041_wp,  3.11359120727677_wp, -2.33730603406593_wp, &
      &  1.47772487606313_wp, -1.56212821462897_wp, -0.66651350755786_wp, &
      &  3.67090320175083_wp, -0.23544868508198_wp,  0.64096747604507_wp, &
      & -2.82753541527553_wp, -3.01047048134432_wp, -2.46540794498288_wp, &
      & -1.20366109255265_wp, -0.74585395016978_wp,  2.29457937340513_wp, &
      &  1.58231094541439_wp, -3.59371366634294_wp, -0.31745929039612_wp, &
      &  3.53804382600349_wp, -0.54602352645102_wp,  2.68018458789943_wp, &
      &  3.51693865545930_wp,  1.79458214078809_wp,  0.31770099613536_wp, &
      & -3.30392500301699_wp,  3.31656342655757_wp,  0.83590378861577_wp, &
      & -4.35713259247048_wp,  2.20527905666152_wp,  1.98486581083101_wp, &
      & -4.18487719158946_wp,  5.81315241619520_wp,  0.14995097108012_wp, &
      & -5.59068629102358_wp,  5.75075395546938_wp, -1.35583081152951_wp, &
      & -2.58564974706739_wp,  6.91353642235019_wp, -0.51432018261127_wp, &
      & -5.00495246814471_wp,  6.72033423637702_wp,  1.79822149483643_wp, &
      & -5.35568731962444_wp, -2.41926399120647_wp,  0.39907577984044_wp, &
      & -5.85527720825561_wp, -1.29575340322300_wp,  2.37477990860100_wp, &
      & -7.22679756724420_wp, -4.16881727028054_wp, -0.84645032970330_wp, &
      & -8.96323624623573_wp, -3.12892228996392_wp, -1.19774130815910_wp, &
      & -7.66469393447992_wp, -5.68951448417014_wp,  0.46481856380671_wp, &
      & -6.54102066194256_wp, -4.97169151175718_wp, -2.60841412141977_wp, &
      &  1.60830688104038_wp, -1.24638887168378_wp, -2.70167933363271_wp, &
      &  6.22165376278531_wp, -1.17033770042330_wp, -0.30343808177997_wp, &
      &  6.34572636729513_wp, -0.84939258346293_wp, -2.34195695471131_wp, &
      &  6.40047277114233_wp, -3.20396239321993_wp, -0.00690163714845_wp, &
      &  8.42332774062421_wp,  0.14042126769442_wp,  0.98807513034232_wp, &
      &  8.22533214732114_wp,  2.18804078190636_wp,  0.73452044586067_wp, &
      &  8.32995370659272_wp, -0.22515495417113_wp,  3.01541623759018_wp, &
      & 10.81910448072659_wp, -0.87858947237144_wp,  0.05313876445727_wp, &
      & 11.00045878404653_wp, -0.44736827538223_wp, -1.80347707737429_wp, &
      & 12.28225495118548_wp, -0.01363872987115_wp,  0.93173321549591_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine LYS_xap

subroutine LYS_xas(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 33
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "H", "H", "H", "H", "H", "N", "H", "C", &
      & "H", "H", "H", "C", "O", "C", "H", "H", "H", "H", "C", "H", "H", "C", &
      & "H", "H", "N", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.65224297270960_wp, -2.28069773882032_wp, -0.62621191034950_wp, &
      & -0.94981571529677_wp, -0.54321843950040_wp,  0.66460832895965_wp, &
      & -1.36615703798161_wp,  2.13155598869344_wp, -0.37772124085332_wp, &
      &  0.04230760944328_wp,  3.08415815217748_wp, -1.95551870161436_wp, &
      &  1.76915661916862_wp, -1.37731348313426_wp,  0.32294716218836_wp, &
      &  3.57773386875602_wp,  0.13208820753412_wp,  1.98058241507396_wp, &
      & -2.02058974410119_wp, -3.16949581478165_wp, -2.18373926895484_wp, &
      & -1.48081529010951_wp, -0.58133452148367_wp,  2.65654721338977_wp, &
      &  1.90656500522650_wp, -3.38277639984846_wp,  0.79452329091455_wp, &
      &  3.01023762097571_wp, -0.09104286818988_wp,  3.95306866728385_wp, &
      &  3.40064001562029_wp,  2.13390907912221_wp,  1.51851608224613_wp, &
      & -3.43636802857681_wp,  3.28108590001463_wp,  0.57993292369427_wp, &
      & -4.64306722937612_wp,  2.16931503608106_wp,  1.56601404437593_wp, &
      & -4.28057230647653_wp,  5.68531096950507_wp, -0.41356498961085_wp, &
      & -5.35858402549646_wp,  5.46644022567106_wp, -2.15639336282811_wp, &
      & -2.63502850232141_wp,  6.84099746693737_wp, -0.82106365233187_wp, &
      & -5.44991449619890_wp,  6.62559318199001_wp,  0.98668210225305_wp, &
      & -5.09338183534597_wp, -2.57911714502317_wp,  0.08717872810986_wp, &
      & -6.02719417045697_wp, -1.38836447741460_wp,  1.85496019113218_wp, &
      & -6.59743679507662_wp, -4.49098642926855_wp, -1.39462204899360_wp, &
      & -8.26130903716987_wp, -3.56294793429480_wp, -2.16323368581427_wp, &
      & -7.24029572188284_wp, -5.95316337069799_wp, -0.10159009998944_wp, &
      & -5.53120555212272_wp, -5.35829077613365_wp, -2.92112351553030_wp, &
      &  2.26552435292835_wp, -1.14061979053743_wp, -1.66327844149676_wp, &
      &  6.32640981223547_wp, -0.68232872952031_wp,  1.68423450359251_wp, &
      &  6.52987188564269_wp, -2.69928882859611_wp,  2.06883652720262_wp, &
      &  7.47434370653195_wp,  0.30468325895284_wp,  3.08999737770069_wp, &
      &  7.40691418969692_wp, -0.12226124499396_wp, -0.91830131684597_wp, &
      &  7.07287267237465_wp,  1.87466883479420_wp, -1.36233252838719_wp, &
      &  6.40956219514348_wp, -1.22331589508054_wp, -2.34440449972147_wp, &
      & 10.06875467972303_wp, -0.87119140968072_wp, -1.00249376449817_wp, &
      & 10.78968432935877_wp, -0.51770811634841_wp, -2.73913432162432_wp, &
      & 11.06800461959559_wp,  0.22487107971482_wp,  0.20828277905496_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine LYS_xas

subroutine LYS_xat(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 33
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "H", "H", "H", "H", "H", "N", "H", "C", &
      & "H", "H", "H", "C", "O", "C", "H", "H", "H", "H", "C", "H", "H", "C", &
      & "H", "H", "N", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.22062705667571_wp, -2.00363040040262_wp,  1.11014154529121_wp, &
      &  0.75900078297263_wp, -0.82699376879624_wp,  0.75420216484373_wp, &
      &  1.07438072097192_wp,  2.04934494821199_wp,  0.83487923498234_wp, &
      &  0.70795915773117_wp,  3.29118830953362_wp,  2.75885783733559_wp, &
      & -1.07503426133255_wp, -1.75760840105353_wp,  2.75322975361041_wp, &
      & -3.72576818105593_wp, -0.65264288407837_wp,  2.47034497879209_wp, &
      &  3.78578784647540_wp, -2.40018285914990_wp,  2.88194431480904_wp, &
      &  0.13602863405970_wp, -1.38398933294473_wp, -1.12700153315522_wp, &
      & -1.15204590724108_wp, -3.81669176666400_wp,  2.62558394567420_wp, &
      & -3.66647339113817_wp,  1.35349477983694_wp,  2.93460431155240_wp, &
      & -4.95404341315007_wp, -1.55164358457659_wp,  3.86496512514014_wp, &
      &  1.82979013750043_wp,  3.04903433333618_wp, -1.39591209730037_wp, &
      &  2.49813762037056_wp,  1.80434695123178_wp, -2.68713709741531_wp, &
      &  2.53474217811972_wp,  5.68846703904667_wp, -1.55438559917942_wp, &
      &  4.40235796025429_wp,  6.03933706148131_wp, -0.75654790825573_wp, &
      &  1.17224058993570_wp,  6.80079709404117_wp, -0.49781679365393_wp, &
      &  2.51960831445224_wp,  6.27399894910540_wp, -3.52111453406977_wp, &
      &  4.86914508444004_wp, -2.28935396559546_wp, -0.83095734803428_wp, &
      &  4.36913820827907_wp, -1.60150111345227_wp, -2.99669264832527_wp, &
      &  7.34986106064423_wp, -3.53043595823553_wp, -0.18426533639325_wp, &
      &  8.87188468211486_wp, -2.22304762618297_wp, -0.62849261083658_wp, &
      &  7.57940874172271_wp, -5.18398665622915_wp, -1.38183615288740_wp, &
      &  7.48758857562271_wp, -4.09320535457289_wp,  1.78638696582019_wp, &
      & -0.32889503682496_wp, -1.25890526091949_wp,  4.61109618705032_wp, &
      & -4.86653949589634_wp, -1.00737757800494_wp, -0.15116142165625_wp, &
      & -4.72524333753361_wp, -2.99083377470664_wp, -0.72428868581398_wp, &
      & -3.81068897675856_wp,  0.09330483612348_wp, -1.54496619710366_wp, &
      & -7.62911759111974_wp, -0.19147397042860_wp, -0.25349786158871_wp, &
      & -8.71940697553044_wp, -1.31075554508099_wp,  1.09329625800574_wp, &
      & -7.77827522948696_wp,  1.76930154107873_wp,  0.36888472974790_wp, &
      & -8.85886396333461_wp, -0.39567964373711_wp, -2.71771778304537_wp, &
      & -7.89920906202725_wp,  0.65998027446252_wp, -3.99424815223876_wp, &
      & -8.73379061867493_wp, -2.20112147307204_wp, -3.34215438177381_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine LYS_xat

subroutine MET_xag(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "H", "H", "C", "H", "H", "S", "C", "H", "H", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "H", "H", "C", "H", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.45578838529099_wp,  1.58926865881272_wp,  0.22739398964214_wp, &
      & -2.02302594601558_wp,  0.69720144172319_wp, -0.41503074454726_wp, &
      &  1.08436044925995_wp,  0.27489235598702_wp,  2.08986318360246_wp, &
      &  1.72564220039796_wp,  1.70239362419894_wp,  3.43755322778506_wp, &
      & -0.45171740787356_wp, -1.72254673462198_wp,  3.48029714351319_wp, &
      & -1.86770849922830_wp, -0.78999267552758_wp,  4.65598338389725_wp, &
      &  0.84477158355493_wp, -2.70785189083993_wp,  4.73784022443857_wp, &
      & -1.71760880480359_wp, -3.69533486170741_wp,  1.78826910650659_wp, &
      & -0.50162162224480_wp, -4.15401915304158_wp,  0.19485557394584_wp, &
      & -2.05239143214679_wp, -5.42384211421478_wp,  2.85521464257023_wp, &
      & -4.75902854723851_wp, -2.57890784597224_wp,  0.62206251038382_wp, &
      & -5.36427456778641_wp, -5.06763085187603_wp, -1.63248281064648_wp, &
      & -7.17689275160376_wp, -4.67780012531301_wp, -2.51622478336922_wp, &
      & -5.46111526549819_wp, -6.88453772473109_wp, -0.67593316617306_wp, &
      & -3.89951899886126_wp, -5.11519807809382_wp, -3.07312554783166_wp, &
      &  3.46773224701937_wp, -0.88429374463485_wp,  0.88779450498305_wp, &
      &  3.87789547906652_wp, -3.17085436129397_wp,  0.89849793186162_wp, &
      &  5.02603374343230_wp,  0.86336519046815_wp, -0.11836149182604_wp, &
      &  4.46187018445129_wp,  2.69415174264866_wp, -0.01084922025139_wp, &
      & -0.00093836629715_wp,  3.99839432962530_wp, -0.48291044259165_wp, &
      &  1.86898187625819_wp,  5.19310576290576_wp,  0.23619955642971_wp, &
      & -1.92905998793513_wp,  5.14718330017453_wp, -2.23626923874210_wp, &
      & -2.60959358692860_wp,  6.89393358006298_wp, -1.39494828104343_wp, &
      & -3.51842388387039_wp,  3.90055720268680_wp, -2.60969702843445_wp, &
      & -1.00411790198553_wp,  5.62961738898537_wp, -4.00748134553430_wp, &
      &  7.33604963602204_wp,  0.17818320411175_wp, -1.40605028871652_wp, &
      &  8.94166221056998_wp,  1.17095655808730_wp, -0.59075217076881_wp, &
      &  7.60396611286902_wp, -1.84282227982917_wp, -1.17839100297275_wp, &
      &  7.23121618353314_wp,  0.62101651755009_wp, -3.41354044727008_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine MET_xag

subroutine MET_xav(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "H", "H", "C", "H", "H", "S", "C", "H", "H", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "H", "H", "C", "H", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.97579823750657_wp,  0.24948531688068_wp,  0.25320033926645_wp, &
      &  4.71013995193895_wp,  1.82439000378639_wp,  1.03513259412887_wp, &
      &  1.26644869376885_wp,  0.20669512954560_wp, -0.01572098611533_wp, &
      &  0.80512035070488_wp, -0.61657854915200_wp, -1.84756117444565_wp, &
      &  0.06871763111704_wp, -1.40823063429709_wp,  2.08025623064625_wp, &
      &  0.70933275318568_wp, -0.66067230721825_wp,  3.89350247409687_wp, &
      &  0.82709232133376_wp, -3.31181519480863_wp,  1.86871253143816_wp, &
      & -2.80758569206009_wp, -1.48273567949844_wp,  2.08163557416085_wp, &
      & -3.46704231699419_wp, -2.78577367730183_wp,  3.53458147875382_wp, &
      & -3.59208019608362_wp,  0.38217961963761_wp,  2.48112964061860_wp, &
      & -4.02662955628140_wp, -2.54081617888057_wp, -0.94992860109677_wp, &
      & -7.35269579413052_wp, -2.27535487842617_wp, -0.22872125811413_wp, &
      & -8.39629011638453_wp, -2.82973812929880_wp, -1.90870488041385_wp, &
      & -7.84210779392552_wp, -0.33878419223905_wp,  0.26114624912049_wp, &
      & -7.85495185639915_wp, -3.53251366400549_wp,  1.31810936296035_wp, &
      &  0.37102330059504_wp,  2.93679850629925_wp,  0.13162977169600_wp, &
      &  1.56572551866212_wp,  4.52856313617350_wp,  1.33177383829052_wp, &
      & -1.80364335140574_wp,  3.41690304169113_wp, -1.11191150104145_wp, &
      & -2.70893882940793_wp,  1.92969066815053_wp, -1.90205051552286_wp, &
      &  5.43153357674191_wp, -1.70366859602505_wp, -0.54425408963891_wp, &
      &  4.52569784402088_wp, -3.60783061259067_wp, -1.50858966005581_wp, &
      &  8.23861712660035_wp, -1.37841315456983_wp, -0.15375181065033_wp, &
      &  9.18413967429579_wp, -1.66804226483270_wp, -1.95413177860597_wp, &
      &  8.89343855883572_wp, -2.83739400431609_wp,  1.13741966670625_wp, &
      &  8.74243336106888_wp,  0.46672734152526_wp,  0.59580531774211_wp, &
      & -3.02141435009052_wp,  5.86294573371246_wp, -0.99562460822434_wp, &
      & -1.60098608448518_wp,  7.25052914844936_wp, -0.48175303737816_wp, &
      & -3.81215833062991_wp,  6.33836334012841_wp, -2.83039173070912_wp, &
      & -4.52147197980959_wp,  5.90608212816708_wp,  0.41632711236204_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine MET_xav

subroutine MET_xbf(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "H", "H", "C", "H", "H", "S", "C", "H", "H", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "H", "H", "C", "H", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.39594249648204_wp,  1.04252550603481_wp,  0.31103332197442_wp, &
      &  3.94514128074333_wp,  2.75481964865002_wp,  0.94217520345301_wp, &
      &  0.70882796401256_wp,  0.63477785400529_wp,  0.12496508464409_wp, &
      &  0.31961890220271_wp, -0.39386983171616_wp, -1.61558798469405_wp, &
      & -0.23602310445654_wp, -0.92728726476801_wp,  2.37587646257912_wp, &
      &  0.14581492479753_wp,  0.13655829833045_wp,  4.10357748334943_wp, &
      &  0.91726884017065_wp, -2.63258515324068_wp,  2.43342150361238_wp, &
      & -3.02830621042191_wp, -1.63208270309764_wp,  2.31561497049823_wp, &
      & -3.43640149266344_wp, -2.96892864249222_wp,  3.82938103400165_wp, &
      & -4.21983372455825_wp,  0.02166973645629_wp,  2.61311542965674_wp, &
      & -4.12332786701106_wp, -2.96339889830827_wp, -0.65506220342198_wp, &
      & -1.81048766125946_wp, -5.44576735422469_wp, -1.07895331241960_wp, &
      & -2.35933223706081_wp, -6.46545673200751_wp, -2.77621264733912_wp, &
      & -1.86789503187262_wp, -6.74027971892330_wp,  0.51808793367748_wp, &
      &  0.08973217141152_wp, -4.69650834232211_wp, -1.31861306672688_wp, &
      & -0.53798755281597_wp,  3.22286843926891_wp,  0.05677395773879_wp, &
      &  0.44337132028587_wp,  5.05471273021490_wp,  1.09586543625051_wp, &
      & -2.77603850255943_wp,  3.27833878091458_wp, -1.16470259265196_wp, &
      & -3.39074542146233_wp,  1.64108369823335_wp, -1.93206704896756_wp, &
      &  5.05785498103795_wp, -0.79780045858401_wp, -0.31483615396005_wp, &
      &  4.36567177031558_wp, -2.89190418002757_wp, -1.04888247216154_wp, &
      &  7.81305440197006_wp, -0.10380725556571_wp, -0.08485044655882_wp, &
      &  8.64271588344821_wp, -0.09109894015783_wp, -1.96590088712046_wp, &
      &  8.76706075939521_wp, -1.55711078272588_wp,  1.00835314528543_wp, &
      &  8.11008707704080_wp,  1.72812237796936_wp,  0.79571529534263_wp, &
      & -4.31569154998968_wp,  5.53413249481170_wp, -1.30584305099879_wp, &
      & -4.57669360519324_wp,  6.12464865761675_wp, -3.25822082479654_wp, &
      & -6.15916924233506_wp,  5.22153051724555_wp, -0.44827531676128_wp, &
      & -3.33413265743904_wp,  7.01513743272354_wp, -0.28099806313661_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine MET_xbf

subroutine MET_xbm(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "H", "H", "C", "H", "H", "S", "C", "H", "H", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "H", "H", "C", "H", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.58646648841266_wp, -2.23150308081548_wp,  0.74522854517507_wp, &
      &  2.07548710377867_wp, -3.11599974535411_wp,  2.34871111124707_wp, &
      &  0.70556416184564_wp, -0.70547891525080_wp, -0.56164055244250_wp, &
      &  1.15806190805393_wp, -0.81430860447580_wp, -2.57116942005703_wp, &
      & -1.92356788095287_wp, -1.72603085568402_wp, -0.04317896178133_wp, &
      & -2.37111628347707_wp, -1.37744756234123_wp,  1.93721618309295_wp, &
      & -1.94627603617336_wp, -3.76017740783321_wp, -0.37057880907499_wp, &
      & -3.89340511134476_wp, -0.46377220195359_wp, -1.71650675693642_wp, &
      & -3.66411689912692_wp, -1.06823855277140_wp, -3.67322921998325_wp, &
      & -3.69744838742997_wp,  1.58521958821216_wp, -1.65321807320619_wp, &
      & -7.10928337345393_wp, -1.25138148286981_wp, -0.85045496420483_wp, &
      & -7.37405190906627_wp,  0.72439398825839_wp,  1.92339304590796_wp, &
      & -9.27218115214997_wp,  0.46306722978868_wp,  2.66527825261808_wp, &
      & -7.10388819980064_wp,  2.70016271546150_wp,  1.42751117696333_wp, &
      & -6.01310429583212_wp,  0.19039974455915_wp,  3.36549451505511_wp, &
      &  0.92852654483135_wp,  2.05357285908788_wp,  0.31296779576115_wp, &
      & -0.54844438826806_wp,  2.99051430769527_wp,  1.83879956512352_wp, &
      &  2.89719121395795_wp,  3.29389090392450_wp, -0.73317678674416_wp, &
      &  4.16580936795398_wp,  2.22344151501738_wp, -1.68814281958195_wp, &
      &  5.01499260323688_wp, -2.38172248227224_wp, -0.05469621644761_wp, &
      &  5.78724839214402_wp, -1.20063505527416_wp, -1.90470904523437_wp, &
      &  6.71129197001910_wp, -4.11601309150014_wp,  1.43632636110924_wp, &
      &  8.36394082007904_wp, -3.05616837177287_wp,  2.03998379837455_wp, &
      &  7.35426633658673_wp, -5.62061817399321_wp,  0.19252935450366_wp, &
      &  5.79028299706006_wp, -4.94230166037896_wp,  3.07585261716902_wp, &
      &  3.55541594434709_wp,  5.82133048227358_wp,  0.08014300399889_wp, &
      &  4.57399732369103_wp,  6.77615375777225_wp, -1.42417481184321_wp, &
      &  1.82657940973072_wp,  6.84448668330331_wp,  0.49625139053901_wp, &
      &  4.72015530177284_wp,  5.80551110459039_wp,  1.77983771492966_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine MET_xbm

subroutine MET_xbo(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 29
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "H", "C", "H", "C", "H", "H", "C", "H", "H", "S", "C", "H", "H", &
      & "H", "C", "O", "N", "H", "C", "O", "C", "H", "H", "H", "C", "H", "H", &
      & "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.91280566232239_wp, -1.91174663638767_wp,  1.13180150821852_wp, &
      &  3.28038045735285_wp, -2.43387412859600_wp,  2.92285336500815_wp, &
      &  0.50968312218163_wp, -0.68634008327208_wp,  0.59121239447164_wp, &
      &  0.05934414069696_wp, -1.11864047131794_wp, -1.37105529168332_wp, &
      & -1.50942836612069_wp, -1.71285265353618_wp,  2.35250232976048_wp, &
      & -1.00655958515868_wp, -1.21189031736150_wp,  4.28961638046251_wp, &
      & -1.53243101494393_wp, -3.76877182884213_wp,  2.19010628388532_wp, &
      & -4.12965816730165_wp, -0.65634652549377_wp,  1.81270198831471_wp, &
      & -5.48506546250945_wp, -1.45133323122226_wp,  3.14536805198406_wp, &
      & -4.11806576926376_wp,  1.38981960945647_wp,  2.02760497992790_wp, &
      & -5.12154284060615_wp, -1.46365590884482_wp, -1.37147011885210_wp, &
      & -8.13006075867010_wp,  0.14469896091242_wp, -1.34131108641277_wp, &
      & -9.00170154389355_wp, -0.14435651931942_wp, -3.17871045892396_wp, &
      & -7.86843542617420_wp,  2.16006587672792_wp, -1.03176234712656_wp, &
      & -9.35868795399267_wp, -0.63209135869933_wp,  0.11286055073346_wp, &
      &  0.81006855770855_wp,  2.18213347169949_wp,  0.88044484511642_wp, &
      &  0.10760003353394_wp,  3.31733796055759_wp,  2.78028841046705_wp, &
      &  1.95049492682606_wp,  3.27977949524965_wp, -1.12009134927209_wp, &
      &  2.71850143663220_wp,  2.09905745992102_wp, -2.41559630781016_wp, &
      &  4.75800416508395_wp, -2.12030005566539_wp, -0.63520524925907_wp, &
      &  4.51239240881002_wp, -1.29025764059502_wp, -2.79352561460379_wp, &
      &  7.12602175253804_wp, -3.46825508350473_wp,  0.19529891956339_wp, &
      &  8.72881273346106_wp, -2.21692844720937_wp, -0.09806688317593_wp, &
      &  7.38980559588280_wp, -5.11354340328956_wp, -1.00728135774165_wp, &
      &  7.07572226842790_wp, -4.06666251711685_wp,  2.15973022568005_wp, &
      &  2.63230439042745_wp,  5.92697088596353_wp, -1.08904766046650_wp, &
      &  4.53129922377389_wp,  6.22432065314661_wp, -0.34642090082200_wp, &
      &  2.53016883983416_wp,  6.68773494884554_wp, -2.99329375255730_wp, &
      &  1.30242657413295_wp,  6.91717237559674_wp,  0.11884362617542_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine MET_xbo

subroutine PHE_xab(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 32
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "C", "C", "C", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", &
      & "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.00834360805732_wp,  1.44375119780842_wp, -0.43126537115732_wp, &
      &  1.53896990476084_wp, -1.20300217558436_wp,  0.09268889487261_wp, &
      &  4.29372065158310_wp, -1.83188227446188_wp, -0.56763727729304_wp, &
      &  4.86453876536490_wp, -3.09302778209035_wp, -2.42693361574807_wp, &
      & -0.27261829902841_wp, -2.87623814149095_wp, -1.39732394258550_wp, &
      & -2.97396304803075_wp, -2.13468357501846_wp, -0.94733291929354_wp, &
      & -4.37691654407873_wp, -0.92638276932977_wp, -2.82990793137556_wp, &
      & -4.11193630680401_wp, -2.54765114354599_wp,  1.40216659762594_wp, &
      & -6.85172523330848_wp, -0.13834818737213_wp, -2.38334796647225_wp, &
      & -6.58177809416295_wp, -1.76825273858681_wp,  1.85978495529253_wp, &
      & -7.95714779049464_wp, -0.55343210132549_wp, -0.03209658335125_wp, &
      & -0.08068732411006_wp,  1.85676573600241_wp, -1.93538892078958_wp, &
      &  1.26657577945165_wp, -1.46999649952492_wp,  2.12118706117556_wp, &
      &  0.06848239276727_wp, -4.82988606072540_wp, -0.84076056766262_wp, &
      &  0.20836562141201_wp, -2.74810200259549_wp, -3.39638294040403_wp, &
      & -3.53286116276348_wp, -0.63447594400490_wp, -4.67350869973907_wp, &
      & -3.05596031572404_wp, -3.49350202113978_wp,  2.87951868317820_wp, &
      & -7.91124900107569_wp,  0.78519814013704_wp, -3.86739958664723_wp, &
      & -7.43439495894127_wp, -2.10954720897983_wp,  3.68573838651385_wp, &
      & -9.87649763415445_wp,  0.05118333717366_wp,  0.32193075773795_wp, &
      &  5.96862917864771_wp, -0.86035435627991_wp,  1.09594550535559_wp, &
      &  5.27288678027899_wp,  0.37419217594146_wp,  2.38575425269732_wp, &
      &  8.66101877746401_wp, -1.05469526044325_wp,  0.66555253315004_wp, &
      &  9.64468249437090_wp, -1.09792161408121_wp,  2.46723926808868_wp, &
      &  9.02886562227854_wp, -2.79375684243924_wp, -0.35872171597463_wp, &
      &  9.37921622495422_wp,  0.52022169354820_wp, -0.45281069341420_wp, &
      &  1.48379891527072_wp,  3.28560185140938_wp,  1.28123043374964_wp, &
      &  2.65854389958518_wp,  2.87861013723543_wp,  3.24961000461197_wp, &
      &  0.48145877610694_wp,  5.87076931378210_wp,  0.63421696808660_wp, &
      & -0.89123745059200_wp,  6.40706637311568_wp,  2.06726064160255_wp, &
      & -0.40679726278710_wp,  5.95403903080959_wp, -1.21613983787555_wp, &
      &  2.02979727252603_wp,  7.21827895874421_wp,  0.71386455692691_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine PHE_xab

subroutine PHE_xal(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 32
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "C", "C", "C", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", &
      & "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.00387724218387_wp, -0.82476854560629_wp, -0.79328706050534_wp, &
      & -2.32509150697366_wp,  1.08159559415233_wp,  1.01533901226853_wp, &
      & -1.43136715963278_wp,  3.36020359910219_wp, -0.50301656066274_wp, &
      & -0.96105436569669_wp,  3.22427532888890_wp, -2.76848337817413_wp, &
      & -0.24698999390653_wp,  0.17255407877584_wp,  2.86371018734893_wp, &
      &  2.02021940431673_wp, -0.81105002501525_wp,  1.48244346629880_wp, &
      &  3.91622295559178_wp,  0.81816935314529_wp,  0.62611638579780_wp, &
      &  2.21512393857384_wp, -3.38756370481636_wp,  0.93007770212752_wp, &
      &  5.95864499373503_wp, -0.10310380516742_wp, -0.75434102499870_wp, &
      &  4.25544345848440_wp, -4.31737599693420_wp, -0.45000347132807_wp, &
      &  6.12905677870838_wp, -2.67402084250750_wp, -1.30158610405485_wp, &
      & -2.37487101458458_wp, -0.52232862236541_wp, -2.56880112514124_wp, &
      & -3.98590984514668_wp,  1.59065654970197_wp,  2.13352754284808_wp, &
      & -1.08435261595154_wp, -1.30907895102694_wp,  4.01926546711298_wp, &
      &  0.27284303925040_wp,  1.74003908717357_wp,  4.09900379754209_wp, &
      &  3.79309969990144_wp,  2.81698805671692_wp,  1.05220208057963_wp, &
      &  0.76726150876582_wp, -4.66254857994254_wp,  1.61580951659294_wp, &
      &  7.41294273559831_wp,  1.17969168171593_wp, -1.40011363919179_wp, &
      &  4.38630124056713_wp, -6.31729800422195_wp, -0.85242025910831_wp, &
      &  7.71571903578212_wp, -3.39168597823583_wp, -2.37097983656024_wp, &
      & -1.10184309508985_wp,  5.50579393352098_wp,  0.85105621497528_wp, &
      & -1.74708308680802_wp,  5.56537822851572_wp,  2.63535791332798_wp, &
      & -0.27095686783459_wp,  7.81614939969622_wp, -0.36153442177526_wp, &
      &  0.46507967056511_wp,  9.09622291317392_wp,  1.06345411491553_wp, &
      &  1.21292668392431_wp,  7.35908615189133_wp, -1.70361579394298_wp, &
      & -1.80072941004710_wp,  8.73956096501633_wp, -1.38460512321563_wp, &
      & -4.13264718960517_wp, -3.01842506559544_wp, -0.10971816306468_wp, &
      & -4.78416378222923_wp, -3.45414220474512_wp,  2.07531124599825_wp, &
      & -4.48258386004562_wp, -4.90521893897248_wp, -2.21931372334641_wp, &
      & -6.29396967763169_wp, -5.83854588452481_wp, -1.97718472919818_wp, &
      & -3.00436777442800_wp, -6.33021073892981_wp, -2.07272099761535_wp, &
      & -4.38215626678392_wp, -4.04334575745232_wp, -4.08174023289139_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine PHE_xal

subroutine PHE_xan(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 32
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "C", "C", "C", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", &
      & "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.09951526224621_wp, -1.79338749239852_wp, -0.11549420035477_wp, &
      & -1.83933368233589_wp,  0.06344557550636_wp,  1.87451332654534_wp, &
      & -2.06151491313775_wp,  2.79759557693949_wp,  0.95788415933121_wp, &
      & -1.60197050525127_wp,  4.55749802361245_wp,  2.39668386771580_wp, &
      &  0.67025802894079_wp, -0.28304474318347_wp,  3.29526121028647_wp, &
      &  2.86671611228730_wp, -0.25713852681266_wp,  1.49711427386353_wp, &
      &  3.88883691050694_wp, -2.52213647678148_wp,  0.59055652349506_wp, &
      &  3.84597152607891_wp,  2.02692182786413_wp,  0.58527409266538_wp, &
      &  5.83291379172083_wp, -2.51767069811258_wp, -1.18968346122599_wp, &
      &  5.78928794950143_wp,  2.03597005576194_wp, -1.18915682930694_wp, &
      &  6.78161415722780_wp, -0.23359335610237_wp, -2.08957414082203_wp, &
      & -0.51275841962965_wp, -2.26364249104300_wp, -1.06421702843185_wp, &
      & -3.41820232159464_wp, -0.24044657476997_wp,  3.16496496660884_wp, &
      &  0.60541407998248_wp, -2.07401265869253_wp,  4.31151144563785_wp, &
      &  0.81632644052175_wp,  1.24840216663423_wp,  4.66076982221594_wp, &
      &  3.17932527661257_wp, -4.30039433828380_wp,  1.31924874139191_wp, &
      &  3.07887349649121_wp,  3.78860778533011_wp,  1.28688444131462_wp, &
      &  6.60967501428985_wp, -4.28729315872538_wp, -1.85580617153147_wp, &
      &  6.53508106537151_wp,  3.81552777778602_wp, -1.86395706040438_wp, &
      &  8.29380231038077_wp, -0.22056228472631_wp, -3.46389121012161_wp, &
      & -2.78616065696345_wp,  3.08059975125848_wp, -1.47063481865533_wp, &
      & -3.18269642568236_wp,  1.49218213944668_wp, -2.43517027127096_wp, &
      & -3.20906756932604_wp,  5.53858278423090_wp, -2.58425806024397_wp, &
      & -2.51705538475848_wp,  6.93946077616394_wp, -1.25464673730662_wp, &
      & -2.18537185031964_wp,  5.71497495712645_wp, -4.35829729126109_wp, &
      & -5.21107936834637_wp,  5.87377317051063_wp, -2.92348990002022_wp, &
      & -4.09856928548609_wp, -3.44431137367425_wp, -0.18949004545971_wp, &
      & -5.95349155736564_wp, -3.22235645901717_wp,  1.16549285566961_wp, &
      & -3.84527786274700_wp, -5.54447735343690_wp, -2.10542894194159_wp, &
      & -3.73808538039176_wp, -7.33347233465930_wp, -1.09753570181689_wp, &
      & -2.19202764843440_wp, -5.34801500633614_wp, -3.31042789630108_wp, &
      & -5.54006648209511_wp, -5.58690480970015_wp, -3.26485128401155_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine PHE_xan

subroutine PHE_xar(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 32
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "C", "C", "C", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", &
      & "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.09189752508508_wp,  1.17222735786226_wp,  1.97445341024775_wp, &
      &  1.50222877291938_wp, -1.38156051080881_wp,  1.01393432015643_wp, &
      &  4.27340365238080_wp, -2.09736759258720_wp,  0.58447020176149_wp, &
      &  5.09458003963014_wp, -4.09205788634857_wp,  1.43882754874366_wp, &
      & -0.07954517763722_wp, -1.95027273903295_wp, -1.36120381542132_wp, &
      & -2.81192085052302_wp, -1.33606844165603_wp, -0.91899517431509_wp, &
      & -3.86808201308455_wp,  0.88471751546852_wp, -1.88320432560092_wp, &
      & -4.31984310590095_wp, -2.91784321902946_wp,  0.56701294458535_wp, &
      & -6.37465296801913_wp,  1.51400527201264_wp, -1.37053910332204_wp, &
      & -6.82407646025377_wp, -2.29593540770438_wp,  1.08472245974553_wp, &
      & -7.85642500031889_wp, -0.07131928044868_wp,  0.12099426672719_wp, &
      &  0.29477856992866_wp,  1.29984296139992_wp,  3.69422297395036_wp, &
      &  0.87716613352050_wp, -2.64270529072523_wp,  2.51281700513462_wp, &
      &  0.16911761609270_wp, -3.95567738433278_wp, -1.77432348493057_wp, &
      &  0.64866458184194_wp, -0.87134042903301_wp, -2.95220636266634_wp, &
      & -2.71427561193317_wp,  2.11276451543958_wp, -3.04514135923969_wp, &
      & -3.52922477298905_wp, -4.65710156603408_wp,  1.30609298819297_wp, &
      & -7.17020220051403_wp,  3.23187614148203_wp, -2.14267144495243_wp, &
      & -7.96973451466601_wp, -3.54673476999931_wp,  2.22551252648859_wp, &
      & -9.80223022651919_wp,  0.41192893776633_wp,  0.51629069887556_wp, &
      &  5.60773196792920_wp, -0.45673905369984_wp, -0.83893478781663_wp, &
      &  4.68750493400198_wp,  1.09563797673577_wp, -1.48684573519756_wp, &
      &  8.22492305385079_wp, -0.91022788062173_wp, -1.48101708753947_wp, &
      &  8.50401753021888_wp, -0.75591885467684_wp, -3.51228473204878_wp, &
      &  9.47236982860097_wp,  0.42651661255854_wp, -0.53523529801777_wp, &
      &  8.69991612272749_wp, -2.81062396210154_wp, -0.87130876816259_wp, &
      &  1.16560954629017_wp,  3.32350053089553_wp,  0.59199516192174_wp, &
      &  2.02004369178836_wp,  3.41437592109993_wp, -1.57427362157073_wp, &
      &  0.06650324190562_wp,  5.62062186122497_wp,  1.86715331450001_wp, &
      &  0.18104462696213_wp,  5.51733198505737_wp,  3.91681235183857_wp, &
      & -1.91618060365257_wp,  5.76684809586055_wp,  1.32824919486737_wp, &
      &  1.05068465886880_wp,  7.29029241968894_wp,  1.19732099656131_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine PHE_xar

subroutine PHE_xaw(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 32
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "C", "C", "C", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", &
      & "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -4.43772936272773_wp, -1.51220272890704_wp, -1.47756225777864_wp, &
      & -2.07940509334210_wp, -0.62270294295864_wp, -0.46476458878990_wp, &
      & -2.05646887207407_wp,  2.23870441667033_wp, -0.72805235448194_wp, &
      & -3.26369232187527_wp,  3.29777890385918_wp, -2.40607063911046_wp, &
      &  0.18986688241370_wp, -1.77633191940437_wp, -1.89722907928477_wp, &
      &  2.65015375555222_wp, -0.98189081606479_wp, -0.74255309294161_wp, &
      &  3.39263864724048_wp, -1.97877085186886_wp,  1.59242461271076_wp, &
      &  4.15318644435323_wp,  0.87891291142551_wp, -1.86266595757104_wp, &
      &  5.57383306509951_wp, -1.11641889663684_wp,  2.79033597387851_wp, &
      &  6.34174275652197_wp,  1.73856490725001_wp, -0.67748766354435_wp, &
      &  7.04985907053995_wp,  0.75159660142775_wp,  1.66029843714685_wp, &
      & -5.36532555762532_wp, -0.31173857018907_wp, -2.63076559166571_wp, &
      & -1.94775226533201_wp, -1.18561012925944_wp,  1.51440541682608_wp, &
      &  0.07387342913345_wp, -1.18211332629194_wp, -3.86809293969712_wp, &
      & -0.04043084710656_wp, -3.82131210715664_wp, -1.80771173752109_wp, &
      &  2.25191873670719_wp, -3.44134898184370_wp,  2.46180089230449_wp, &
      &  3.59138286557448_wp,  1.66197702325228_wp, -3.66899330678062_wp, &
      &  6.12624804601236_wp, -1.90790947841309_wp,  4.59215757486998_wp, &
      &  7.48942590364090_wp,  3.17320128641829_wp, -1.57379560252512_wp, &
      &  8.74539287314953_wp,  1.41844690991167_wp,  2.58540006938406_wp, &
      & -0.60021535626433_wp,  3.46957418420145_wp,  0.97206829203691_wp, &
      &  0.54213091875174_wp,  2.40815102401483_wp,  2.06445613895310_wp, &
      & -0.02941922331769_wp,  6.13103887877688_wp,  0.68502644165086_wp, &
      &  0.29601182090619_wp,  6.96834545102821_wp,  2.53076211688858_wp, &
      &  1.63450409411422_wp,  6.42356403841007_wp, -0.49597552748964_wp, &
      & -1.63927497119741_wp,  7.03670728749871_wp, -0.20522587990384_wp, &
      & -5.31470722127527_wp, -3.85525383926526_wp, -0.91873347126618_wp, &
      & -4.15484710606230_wp, -5.30382865510773_wp,  0.47370651249803_wp, &
      & -7.80504994158879_wp, -4.55540674924786_wp, -2.11801320426633_wp, &
      & -7.51783375907359_wp, -6.20966884108041_wp, -3.30308543695536_wp, &
      & -8.61301694880619_wp, -3.04129212036409_wp, -3.24662716943255_wp, &
      & -9.12208085973760_wp, -5.07636729893054_wp, -0.62971882555636_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine PHE_xaw

subroutine PRO_xab(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.26447810872580_wp,  0.23797066353381_wp, -0.42843097345004_wp, &
      &  0.21912492244883_wp,  0.78390753054649_wp, -1.42560199877158_wp, &
      &  2.32652044338437_wp, -0.09152010015889_wp,  0.34640633485968_wp, &
      &  2.04481186776956_wp, -0.18934114606999_wp,  2.63852392011971_wp, &
      &  0.20270172379411_wp,  3.68565198374920_wp, -1.56661310201134_wp, &
      & -1.38658595002225_wp,  4.41961990809734_wp,  0.73164640264828_wp, &
      & -3.47341077344463_wp,  2.42954419835858_wp,  0.74604464878739_wp, &
      &  0.47906536710283_wp, -0.05693999930578_wp, -3.29219544889444_wp, &
      & -0.73949754019178_wp,  4.26190119989731_wp, -3.30640709608740_wp, &
      &  2.10051510245743_wp,  4.47945595042193_wp, -1.55261028847306_wp, &
      & -2.12343927551595_wp,  6.33383916561332_wp,  0.60575891763821_wp, &
      & -0.25672915726771_wp,  4.25004813734561_wp,  2.44052843394925_wp, &
      & -5.11518751581803_wp,  3.01110667786799_wp, -0.35968740532000_wp, &
      & -4.12139024258679_wp,  1.93375377065549_wp,  2.63213384096284_wp, &
      &  4.52968487708332_wp, -0.64769230352856_wp, -0.85075889213702_wp, &
      &  4.57129347134053_wp, -0.55392178383033_wp, -2.74643811249199_wp, &
      &  6.79939464582760_wp, -1.33866628855950_wp,  0.51131260767483_wp, &
      &  6.30525414031798_wp, -1.42120283886887_wp,  2.50049969707037_wp, &
      &  7.48669221503641_wp, -3.18071452399202_wp, -0.09059435170874_wp, &
      &  8.28685376863555_wp,  0.05714550942514_wp,  0.24390536564138_wp, &
      & -3.29739703694261_wp, -2.10734908882048_wp, -0.15563101354875_wp, &
      & -5.35833515943644_wp, -2.40615387921832_wp,  0.85799231478982_wp, &
      & -1.75860165755764_wp, -4.27936602205924_wp, -1.18961962708669_wp, &
      &  0.02375965682511_wp, -4.45346268101255_wp, -0.17377971430922_wp, &
      & -2.84721058029445_wp, -5.99840717873244_wp, -0.94511095733517_wp, &
      & -1.33813627300938_wp, -4.01896658674527_wp, -3.18831229353687_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine PRO_xab

subroutine PRO_xac(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.85348562928722_wp,  0.22959982785031_wp, -0.13719628089565_wp, &
      & -0.14475765687371_wp,  0.86858833756085_wp,  1.64884071003628_wp, &
      & -2.77577782022113_wp, -0.06645524739541_wp,  0.89637304511822_wp, &
      & -4.58670468518547_wp,  0.31754254849294_wp,  2.28647871128085_wp, &
      & -0.09037983079501_wp,  3.76444176639348_wp,  1.66053688705637_wp, &
      &  0.63390433191714_wp,  4.38027710650824_wp, -1.06594057502062_wp, &
      &  2.65378375061979_wp,  2.40909785332788_wp, -1.65429069051451_wp, &
      &  0.23090197420209_wp,  0.10320818687689_wp,  3.52089218337821_wp, &
      & -1.89154763333627_wp,  4.54398565525227_wp,  2.26418388164129_wp, &
      &  1.38375039438699_wp,  4.41042510447179_wp,  2.94691934860684_wp, &
      & -0.99532688543885_wp,  4.12219845818028_wp, -2.30009453665256_wp, &
      &  1.32528422802407_wp,  6.29997610579658_wp, -1.30651580330719_wp, &
      &  2.76898753028312_wp,  1.90366540105769_wp, -3.64432839284905_wp, &
      &  4.53365176356374_wp,  3.01572352583721_wp, -1.05863913241559_wp, &
      & -2.90155721970647_wp, -1.27289588241633_wp, -1.35447483837376_wp, &
      & -1.27170055764746_wp, -1.46574292897781_wp, -2.31583030654429_wp, &
      & -5.24739616883575_wp, -2.26317950245269_wp, -2.34759758827651_wp, &
      & -5.09307694330069_wp, -4.27953834189070_wp, -2.71952853553178_wp, &
      & -6.69233031832740_wp, -1.94312887822713_wp, -0.92622736114764_wp, &
      & -5.78631400368658_wp, -1.29973460517622_wp, -4.08348469411866_wp, &
      &  3.36416944101009_wp, -1.86853014954706_wp, -0.03659646803709_wp, &
      &  5.23125024515924_wp, -2.06034959755694_wp, -1.39344853252343_wp, &
      &  2.57789065594353_wp, -3.90014322504399_wp,  1.80768015041895_wp, &
      &  3.61017319088873_wp, -5.60989127634991_wp,  1.34597528777588_wp, &
      &  0.55431105079430_wp, -4.26209145117366_wp,  1.74905377051532_wp, &
      &  3.06663681766593_wp, -3.33442689377920_wp,  3.72745676317501_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine PRO_xac

subroutine PRO_xad(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  4.69973216693074_wp,  2.16920021429453_wp,  5.48978942245055_wp, &
      &  7.05549586002302_wp,  3.63673686487836_wp,  5.28714680365749_wp, &
      &  8.27837961853874_wp,  3.91549885321623_wp,  7.90764466264031_wp, &
      &  8.12179077982362_wp,  5.86258833639557_wp,  9.15822641259811_wp, &
      &  6.21234579016813_wp,  6.18494942557381_wp,  4.20675723115774_wp, &
      &  3.54742820379652_wp,  5.71762882154813_wp,  3.18960286843237_wp, &
      &  2.50675338577603_wp,  3.78156392870278_wp,  5.04990662479238_wp, &
      &  8.33657299426572_wp,  2.60306590980844_wp,  4.04702186028522_wp, &
      &  7.51329510870605_wp,  6.86926601587140_wp,  2.77212556284927_wp, &
      &  6.17646603167825_wp,  7.54191873031119_wp,  5.75269829077634_wp, &
      &  3.61465589850709_wp,  4.87873156654991_wp,  1.30898235582839_wp, &
      &  2.41172255263542_wp,  7.42903249968986_wp,  3.10521918401129_wp, &
      &  0.93105647741793_wp,  2.69608118825551_wp,  4.31086317616302_wp, &
      &  1.92293659108564_wp,  4.69396491166368_wp,  6.80812548661077_wp, &
      &  9.50326433186445_wp,  1.81429148032959_wp,  8.67509194721199_wp, &
      &  9.16731301007563_wp,  0.22710885904923_wp,  7.65395288032887_wp, &
      & 10.54039136737323_wp,  1.64897200334083_wp, 11.19907585540978_wp, &
      & 12.01818453863771_wp,  0.22489334411204_wp, 11.23320129368240_wp, &
      & 11.33291192596081_wp,  3.47202963323874_wp, 11.70705412411583_wp, &
      &  9.10491852709076_wp,  1.16203054452567_wp, 12.59567813419614_wp, &
      &  4.75578056279291_wp, -0.39002113395244_wp,  5.49979410069723_wp, &
      &  6.76780685520453_wp, -1.57472050274266_wp,  5.57094365859872_wp, &
      &  2.24438128420914_wp, -1.74044908136608_wp,  5.42535265873732_wp, &
      &  2.59667111440037_wp, -3.73223079325503_wp,  5.75312456171220_wp, &
      &  0.95133213503025_wp, -1.01942116993291_wp,  6.85192791760970_wp, &
      &  1.35272627339801_wp, -1.51617243546908_wp,  3.58286796628113_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine PRO_xad

subroutine PRO_xae(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.85404409730758_wp, -0.05226243957689_wp,  0.09990484165060_wp, &
      & -0.18023580030735_wp,  1.10443825475375_wp,  1.60023282056908_wp, &
      & -2.74206889638658_wp,  0.81119402774824_wp,  0.26416012996506_wp, &
      & -3.86955738710595_wp,  2.61805923701853_wp, -0.65823379610921_wp, &
      &  0.62997767757356_wp,  3.85880032220108_wp,  1.75487038035785_wp, &
      &  1.93399022474350_wp,  4.28025787121151_wp, -0.78728673236964_wp, &
      &  3.43008484906576_wp,  1.84371698974828_wp, -1.16168403912688_wp, &
      & -0.26292962072930_wp,  0.15694005505540_wp,  3.42820919840041_wp, &
      &  1.96889808306402_wp,  4.09888824020517_wp,  3.30248307452118_wp, &
      & -0.98054850135026_wp,  5.09791132378366_wp,  2.03511389177692_wp, &
      &  3.14567987753737_wp,  5.93967868903169_wp, -0.82193719249325_wp, &
      &  0.50143387836043_wp,  4.46710521921216_wp, -2.25048860078577_wp, &
      &  5.28085325800517_wp,  1.96269091166862_wp, -0.25196630655319_wp, &
      &  3.70875044919540_wp,  1.37574630404632_wp, -3.14554480360531_wp, &
      & -3.58055689462722_wp, -1.59652845993458_wp,  0.22873931857565_wp, &
      & -2.37449509784336_wp, -2.93106526346288_wp,  0.89433977583143_wp, &
      & -5.87118205426446_wp, -2.30257061889019_wp, -1.08267845339794_wp, &
      & -6.96183243981867_wp, -3.61946085273094_wp,  0.05670793975175_wp, &
      & -6.96991224364747_wp, -0.59916219542580_wp, -1.39766222230200_wp, &
      & -5.46726799723431_wp, -3.17471158918371_wp, -2.90463806205653_wp, &
      &  2.38181485137710_wp, -2.54851377525339_wp,  0.27235449813697_wp, &
      &  1.03188938552322_wp, -4.02142355101386_wp,  1.48430157405026_wp, &
      &  4.70559065689755_wp, -3.43593208020416_wp, -1.11697312818230_wp, &
      &  4.93935049222557_wp, -5.43987078132684_wp, -0.76041493543007_wp, &
      &  4.50587443073182_wp, -3.11975611099160_wp, -3.14121247310361_wp, &
      &  6.38181186134194_wp, -2.42009253052674_wp, -0.48920891841704_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine PRO_xae

subroutine PRO_xaf(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.81476769625463_wp, -0.08931587233540_wp, -0.31641583574778_wp, &
      & -0.02797611389507_wp,  1.15266673473121_wp,  1.32155762651231_wp, &
      & -2.77179718228637_wp,  0.41673082212513_wp,  0.87182613179259_wp, &
      & -4.38215738015736_wp,  0.86876349757246_wp,  2.47088827471992_wp, &
      &  0.32852653520401_wp,  3.97267830668765_wp,  0.70334452374600_wp, &
      &  3.08339628378522_wp,  4.12940045306759_wp, -0.13358885139506_wp, &
      &  3.39171665318244_wp,  1.72891210635593_wp, -1.69110636278291_wp, &
      &  0.36366696960190_wp,  0.80048188072797_wp,  3.31485199713977_wp, &
      & -0.90847430189242_wp,  4.48319463953428_wp, -0.86595686250160_wp, &
      & -0.14801874516211_wp,  5.15945153468612_wp,  2.30919909194003_wp, &
      &  4.33142449383685_wp,  4.05602741689955_wp,  1.50420968719081_wp, &
      &  3.50351665333065_wp,  5.83501882729231_wp, -1.20121181390948_wp, &
      &  2.69755760977361_wp,  1.99373334442264_wp, -3.61770168722612_wp, &
      &  5.30998062425935_wp,  1.00439203375072_wp, -1.79573642835671_wp, &
      & -3.24417652644249_wp, -0.60756284769237_wp, -1.42521739787743_wp, &
      & -1.73816275531437_wp, -0.96861377138639_wp, -2.52984929237304_wp, &
      & -5.76392917176014_wp, -1.30373151209600_wp, -2.22213110759822_wp, &
      & -5.87981470542014_wp, -3.31565149955356_wp, -2.63359476746714_wp, &
      & -7.03543749191627_wp, -0.85682633362547_wp, -0.67500325448493_wp, &
      & -6.33358286949981_wp, -0.24372422020899_wp, -3.89029156911155_wp, &
      &  2.63706434494895_wp, -2.53376246039117_wp, -0.01045142546973_wp, &
      &  4.40861323233509_wp, -3.37569579326797_wp, -1.23985129531665_wp, &
      &  1.21784284238446_wp, -4.09590196896628_wp,  1.90917923677405_wp, &
      & -0.80424316128019_wp, -4.11022635564801_wp,  1.53406861822439_wp, &
      &  1.95066932397567_wp, -6.00806479242428_wp,  1.83375116138501_wp, &
      &  1.49273817416557_wp, -3.33975263029909_wp,  3.80493519285806_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine PRO_xaf

subroutine SER_xad(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 23
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "O", "H", "H", "H", "H", "H", "N", "H", "C", &
      & "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.00902007682636_wp,  1.09148690102623_wp, -0.15166402804466_wp, &
      & -0.27787670706623_wp,  1.28069752328163_wp,  1.35055276357891_wp, &
      & -2.39065153081095_wp, -0.31724367075187_wp,  0.17773128841524_wp, &
      & -4.19108824523285_wp,  0.66002327911403_wp, -0.93853735764153_wp, &
      & -1.06282222658281_wp,  4.04105783302588_wp,  1.44707359866906_wp, &
      & -1.32338329412544_wp,  5.02909544171743_wp, -1.02808968068295_wp, &
      &  2.23627667054604_wp,  2.41674018294770_wp, -1.50134413058746_wp, &
      &  0.14579815850270_wp,  0.56785059036159_wp,  3.23800935643633_wp, &
      & -2.81306118882961_wp,  4.21409853553126_wp,  2.52414981248063_wp, &
      &  0.40336231817521_wp,  5.13864300370467_wp,  2.38322146253689_wp, &
      & -2.70746286884532_wp,  4.05405090480913_wp, -1.73249224546283_wp, &
      & -2.06516579007619_wp, -2.82042935436854_wp,  0.47200147813628_wp, &
      & -0.38903880381293_wp, -3.37300659530107_wp,  1.22244146336378_wp, &
      & -3.74543354250454_wp, -4.64439927981829_wp, -0.68202795895711_wp, &
      & -2.93276194553950_wp, -5.43738183651010_wp, -2.39956945869801_wp, &
      & -4.13974722619042_wp, -6.16853396924265_wp,  0.63695577937867_wp, &
      & -5.49274494296809_wp, -3.68307059511597_wp, -1.15993869777858_wp, &
      &  3.63482186897538_wp, -0.86681589667476_wp,  0.06539459198046_wp, &
      &  3.25778022840982_wp, -2.63933384087293_wp,  1.53107443668471_wp, &
      &  5.96488804063154_wp, -0.73310625249835_wp, -1.56427712459957_wp, &
      &  7.60767860467566_wp, -0.58643931964619_wp, -0.33731512068279_wp, &
      &  5.94918059695797_wp,  0.85740094836721_wp, -2.86335452414599_wp, &
      &  6.13006550047522_wp, -2.48653834953869_wp, -2.62088763134388_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine SER_xad

subroutine SER_xaf(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 23
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "O", "H", "H", "H", "H", "H", "N", "H", "C", &
      & "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.97401442529541_wp, -1.53663797674849_wp, -0.96404844410074_wp, &
      &  0.51551948914996_wp, -1.60524831391057_wp,  0.19221490891819_wp, &
      &  2.17210096056972_wp,  0.59724637522747_wp, -0.70916573390697_wp, &
      &  4.08923886618656_wp,  0.24108237190890_wp, -1.98752156809471_wp, &
      &  1.76570317687931_wp, -4.11759975531098_wp, -0.44679220456557_wp, &
      &  4.19958436234711_wp, -4.30168597468846_wp,  0.60150873004831_wp, &
      & -2.26713053316303_wp, -2.58701507989182_wp, -2.52189414145288_wp, &
      &  0.26438864132988_wp, -1.44513461802136_wp,  2.23132100858573_wp, &
      &  0.63398066494729_wp, -5.64576014595614_wp,  0.33923773233735_wp, &
      &  1.80970810609912_wp, -4.33127073191786_wp, -2.50697965016646_wp, &
      &  5.15060975559885_wp, -2.98991213422085_wp, -0.24802414750269_wp, &
      &  1.33660706239557_wp,  2.87960000384031_wp,  0.03101465221902_wp, &
      & -0.30983995784043_wp,  2.91539302943321_wp,  1.01191075172800_wp, &
      &  2.59435333097942_wp,  5.19820611340683_wp, -0.68831173300965_wp, &
      &  1.48308873744316_wp,  6.25848761212443_wp, -2.05823897701269_wp, &
      &  2.91684925268930_wp,  6.37572828263293_wp,  0.96465665995623_wp, &
      &  4.39574510879649_wp,  4.70036963847640_wp, -1.53297679344686_wp, &
      & -3.88373542751531_wp, -0.07954385776297_wp, -0.07478749078557_wp, &
      & -3.59748683014043_wp,  1.38333609823243_wp,  1.71123470314302_wp, &
      & -6.38990219859491_wp, -0.36937644350831_wp, -1.39403579888641_wp, &
      & -7.77040566232197_wp, -1.02793902830601_wp, -0.02140002845590_wp, &
      & -6.33061266589768_wp, -1.67996171134074_wp, -2.97449240552962_wp, &
      & -7.00130793522704_wp,  1.47637781216631_wp, -2.05657216209248_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine SER_xaf

subroutine SER_xah(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 23
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "O", "H", "H", "H", "H", "H", "N", "H", "C", &
      & "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.97952692121746_wp, -1.44061155028990_wp,  0.67274574185257_wp, &
      & -0.48049347607733_wp, -1.47456825805101_wp, -0.54781497573537_wp, &
      & -2.08814055606108_wp,  0.80351660403918_wp,  0.23942159124728_wp, &
      & -4.04016596202855_wp,  0.56395318879892_wp,  1.49375120484682_wp, &
      & -1.82386414111201_wp, -3.92452337226183_wp,  0.14283484459837_wp, &
      & -4.24118308704672_wp, -4.06366062205663_wp, -0.94995942039141_wp, &
      &  2.19745140366228_wp, -2.43936722064941_wp,  2.27627795063085_wp, &
      & -0.17153512899731_wp, -1.39046354906876_wp, -2.58330414853871_wp, &
      & -0.72693978356083_wp, -5.51280571766518_wp, -0.56974685898048_wp, &
      & -1.91628714104682_wp, -4.06985175078056_wp,  2.20749487068252_wp, &
      & -5.16633503710111_wp, -2.69730753016065_wp, -0.15975858834522_wp, &
      & -1.19557005309746_wp,  3.03981829199036_wp, -0.55957825152180_wp, &
      &  0.48839964475557_wp,  3.03550876075906_wp, -1.47272416351874_wp, &
      & -2.45152694703482_wp,  5.37639599588128_wp,  0.10859642913111_wp, &
      & -1.75724173702866_wp,  6.87437907119508_wp, -1.10860158073759_wp, &
      & -4.47728487299977_wp,  5.16391634351618_wp, -0.15151702264725_wp, &
      & -2.10978454806184_wp,  5.88731061941420_wp,  2.07373610032327_wp, &
      &  3.94504185960410_wp, -0.04494147552401_wp, -0.19348187649496_wp, &
      &  3.74459897877379_wp,  1.37513998891258_wp, -2.02504057044852_wp, &
      &  6.40531827399644_wp, -0.35647335049731_wp,  1.20492482837850_wp, &
      &  7.06339116357056_wp,  1.49656373605376_wp,  1.79817006913336_wp, &
      &  7.79644208709546_wp, -1.12432586687042_wp, -0.09887199337141_wp, &
      &  6.26130288423874_wp, -1.59060030778676_wp,  2.84056129456930_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine SER_xah

subroutine SER_xak(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 23
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "O", "H", "H", "H", "H", "H", "N", "H", "C", &
      & "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.20862683576754_wp, -1.00520868255573_wp,  0.46656506317632_wp, &
      & -0.04471617938892_wp,  0.43315987955973_wp, -0.33036129988136_wp, &
      &  2.25473717636100_wp, -1.29981769167987_wp, -0.11610551781982_wp, &
      &  2.03254846943545_wp, -3.56059141766289_wp,  0.37440264781074_wp, &
      &  0.23273094613160_wp,  2.83771853529681_wp,  1.25624577014973_wp, &
      &  2.02414420969152_wp,  4.54627254451932_wp,  0.17941140451948_wp, &
      & -1.82338062091500_wp, -2.81252253424800_wp,  0.95170537243319_wp, &
      & -0.28440980103920_wp,  1.01251970438839_wp, -2.30192885848699_wp, &
      & -1.61923772861451_wp,  3.71552009189707_wp,  1.42295391895172_wp, &
      &  0.92560276698345_wp,  2.36552956249867_wp,  3.13476580394331_wp, &
      &  1.18659089115804_wp,  5.40035071211662_wp, -1.19083344086741_wp, &
      &  4.49193687923530_wp, -0.15535839165144_wp, -0.51135885231083_wp, &
      &  4.48637027265355_wp,  1.74896323574225_wp, -0.63979812590931_wp, &
      &  6.83519027920091_wp, -1.54526345283740_wp, -0.25674470505644_wp, &
      &  8.34774787254571_wp, -0.46495676411219_wp, -1.12489422193577_wp, &
      &  7.30204946389114_wp, -1.90102132368824_wp,  1.71690058463946_wp, &
      &  6.65700177641530_wp, -3.35588945206581_wp, -1.20685380104263_wp, &
      & -4.56576503159677_wp, -0.29940690353121_wp, -0.28000362840500_wp, &
      & -4.96089515007017_wp,  1.71693360741445_wp, -1.35511489840351_wp, &
      & -6.64847472206076_wp, -2.15019574653299_wp,  0.31852364359506_wp, &
      & -8.18957354739154_wp, -1.13720943740252_wp,  1.22110045831082_wp, &
      & -6.02798134363004_wp, -3.68583378965615_wp,  1.53288237163001_wp, &
      & -7.35990831647469_wp, -2.92778436369519_wp, -1.44714685694022_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine SER_xak

subroutine SER_xar(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 23
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "O", "H", "H", "H", "H", "H", "N", "H", "C", &
      & "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.76195282273126_wp, -1.28530691890781_wp,  0.12953352935164_wp, &
      &  0.23357608608633_wp,  0.49946565131706_wp, -0.40579652163385_wp, &
      &  2.70035341146913_wp, -0.92810111937306_wp,  0.05500052017879_wp, &
      &  2.73761179352673_wp, -2.98593331513531_wp,  1.12553399464469_wp, &
      &  0.08264828626065_wp,  2.92437589823366_wp,  1.19706744527637_wp, &
      & -1.49971342240242_wp,  4.76357431726440_wp,  0.12153254812442_wp, &
      & -1.15247034578895_wp, -2.95855879205461_wp,  0.82052852124056_wp, &
      &  0.14465229858591_wp,  1.08471467942121_wp, -2.38669868359846_wp, &
      & -0.50774315733472_wp,  2.37389917644830_wp,  3.10449513475365_wp, &
      &  1.95694739609322_wp,  3.76570314270570_wp,  1.33960772437044_wp, &
      & -3.03537343294867_wp,  3.87051768532566_wp, -0.37050084389456_wp, &
      &  4.79100086084766_wp,  0.27591267850013_wp, -0.77972279324686_wp, &
      &  4.56879269736835_wp,  1.92010008946056_wp, -1.70487879484540_wp, &
      &  7.30442764354374_wp, -0.76350470213185_wp, -0.47277350815131_wp, &
      &  8.08302423868767_wp, -1.38310407281685_wp, -2.27316895758456_wp, &
      &  8.56112992887975_wp,  0.62988350836060_wp,  0.36442174500248_wp, &
      &  7.15202151941704_wp, -2.38098273511207_wp,  0.77767893911143_wp, &
      & -4.23408998065542_wp, -0.83965610854975_wp, -0.29112013180118_wp, &
      & -5.04433911194975_wp,  1.20555005825484_wp, -1.06512324268998_wp, &
      & -5.98254911094213_wp, -3.03089857749176_wp,  0.20886588164712_wp, &
      & -6.74776210321938_wp, -3.67371038895268_wp, -1.58815562298354_wp, &
      & -7.55160182460074_wp, -2.36857381541028_wp,  1.35523615848389_wp, &
      & -5.05311853637285_wp, -4.60401265026394_wp,  1.14584679468527_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine SER_xar

subroutine THR_xab(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "O", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.50610814260308_wp, -0.80007359762784_wp,  0.65819609015323_wp, &
      &  0.12711962561789_wp,  0.80973659354347_wp, -0.82637548552336_wp, &
      &  2.92826668164225_wp,  0.10034396158528_wp, -0.69791428311140_wp, &
      &  4.52385381371854_wp,  1.48928038373611_wp, -1.63986323453608_wp, &
      & -0.19828098540864_wp,  3.55358745666088_wp, -0.01893450661442_wp, &
      &  0.16288392253625_wp,  3.51795632041019_wp,  2.66552256356706_wp, &
      & -2.77222238552608_wp,  4.59957604447233_wp, -0.70917369596224_wp, &
      & -1.72403046379420_wp, -0.29136606033224_wp,  2.48197343337460_wp, &
      & -0.43993584749766_wp,  0.59439369208314_wp, -2.79595428478552_wp, &
      &  1.30632196731767_wp,  4.62805528691711_wp, -0.92873870251429_wp, &
      &  0.12059940875620_wp,  5.23588584402879_wp,  3.25619464829255_wp, &
      & -2.97355640586295_wp,  6.52950466402134_wp, -0.01835063995478_wp, &
      & -4.26539437210598_wp,  3.45636658247060_wp,  0.12651866037752_wp, &
      & -3.02069068134675_wp,  4.62258444657137_wp, -2.75217756920879_wp, &
      &  3.43837307042612_wp, -2.12185491644953_wp,  0.44552540879586_wp, &
      &  1.95377533964891_wp, -3.10793113202034_wp,  1.10233293296537_wp, &
      &  5.97400190330973_wp, -3.12111177830375_wp,  0.61547076264494_wp, &
      &  6.49010525458674_wp, -3.48315074840121_wp,  2.57288954089637_wp, &
      &  6.16226772058690_wp, -4.86219236211019_wp, -0.46456679275596_wp, &
      &  7.23820808723311_wp, -1.70730129968398_wp, -0.16675931520660_wp, &
      & -3.33973742541401_wp, -2.25843865993760_wp, -0.45136151891458_wp, &
      & -3.44992834850690_wp, -2.60499077366759_wp, -2.73077768290011_wp, &
      & -5.20135102360664_wp, -3.42986261018801_wp,  1.36687031084645_wp, &
      & -5.43275322612790_wp, -5.41242035379000_wp,  0.88681229201771_wp, &
      & -7.02391040197635_wp, -2.51005033166527_wp,  1.11509207491468_wp, &
      & -4.62828619680824_wp, -3.25588249279735_wp,  3.33240430589997_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine THR_xab

subroutine THR_xag(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "O", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.99253479463345_wp, -0.74258531464601_wp,  0.84463279512754_wp, &
      & -0.10689088339546_wp,  0.60569371420861_wp, -0.60228302338302_wp, &
      &  2.46922399218099_wp, -0.70530962220296_wp, -0.54534590898213_wp, &
      &  4.39467610766018_wp,  0.41596247122471_wp, -1.22695899462033_wp, &
      &  0.14635958394470_wp,  3.39489364740600_wp,  0.19581381901527_wp, &
      &  1.59330526181819_wp,  4.73620810443912_wp, -1.60175956246277_wp, &
      &  1.19379196024318_wp,  3.69010332777203_wp,  2.85641786419264_wp, &
      & -1.77369087530815_wp, -0.75606130331498_wp,  2.74359082710176_wp, &
      & -0.68134472343891_wp,  0.61607283690751_wp, -2.57832562120290_wp, &
      & -1.75375152048938_wp,  4.19626308321388_wp,  0.12595258660675_wp, &
      &  3.13944671354314_wp,  3.75851864501349_wp, -1.76847569111465_wp, &
      &  1.28744659168616_wp,  5.68909223945211_wp,  3.32260562178621_wp, &
      &  3.09597041022664_wp,  2.91224737908949_wp,  2.97345947974482_wp, &
      & -0.00000737586190_wp,  2.76855599292398_wp,  4.26123088987998_wp, &
      &  2.45355720136942_wp, -3.11635447684739_wp,  0.26562243926544_wp, &
      &  0.75459182811027_wp, -3.85946956101360_wp,  0.69227710436735_wp, &
      &  4.71087687352992_wp, -4.66079762416958_wp,  0.30953324851930_wp, &
      &  4.94649061819053_wp, -5.53557433781871_wp,  2.15351949478223_wp, &
      &  4.63368101875584_wp, -6.12877617570244_wp, -1.12915732923795_wp, &
      &  6.30766420433890_wp, -3.43260936230502_wp, -0.07382677913612_wp, &
      & -4.48400245051472_wp, -1.10742442363476_wp,  0.15495176092340_wp, &
      & -6.04486309582487_wp, -1.77223070368460_wp,  1.71595296316582_wp, &
      & -5.12195919971834_wp, -0.71639368417671_wp, -2.59527510104460_wp, &
      & -3.90742217172105_wp, -1.83956717095137_wp, -3.81841555083874_wp, &
      & -4.90343256262750_wp,  1.26031494854241_wp, -3.12665896712925_wp, &
      & -7.07435803291993_wp, -1.27409753908914_wp, -2.86617480460224_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine THR_xag

subroutine THR_xah(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "O", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.88949762166867_wp, -0.39491686380126_wp, -0.82913279911430_wp, &
      &  0.23844420499918_wp,  0.89643661093050_wp,  0.90765678826423_wp, &
      & -2.49746962392326_wp, -0.05851125246123_wp,  0.95504857925193_wp, &
      & -3.99898274115606_wp,  0.83422596692261_wp,  2.47651217243375_wp, &
      &  0.20613353433539_wp,  3.71210344852318_wp,  0.28006013950639_wp, &
      & -0.46755812811984_wp,  3.82178602539799_wp, -2.33560065694389_wp, &
      &  2.72731037995279_wp,  4.97835920556644_wp,  0.77068852721076_wp, &
      &  2.06238364984609_wp,  0.35634838863471_wp, -2.57636827949794_wp, &
      &  0.92974854840167_wp,  0.69939361762026_wp,  2.83695981459206_wp, &
      & -1.27379119748896_wp,  4.55914410707703_wp,  1.44008257768984_wp, &
      & -0.51147132413897_wp,  5.57108013584227_wp, -2.82767375909974_wp, &
      &  2.64891860073593_wp,  6.96188291842252_wp,  0.22065363461951_wp, &
      &  4.22456724561978_wp,  4.06283040636574_wp, -0.30101057098630_wp, &
      &  3.19630809697219_wp,  4.90043740923969_wp,  2.77379474494690_wp, &
      & -3.05073268981448_wp, -1.90741011878229_wp, -0.71176696740557_wp, &
      & -1.71302349332822_wp, -2.34987759854456_wp, -1.98764347020343_wp, &
      & -5.59100776112713_wp, -2.88175446809734_wp, -0.97643598605047_wp, &
      & -5.53705009391544_wp, -4.90707007016143_wp, -1.31243789278158_wp, &
      & -6.59144535820169_wp, -2.49993223710576_wp,  0.77325349885704_wp, &
      & -6.59664360384784_wp, -1.96498512434976_wp, -2.52167155289756_wp, &
      &  3.21840708851438_wp, -2.57571092447808_wp, -0.40251263015919_wp, &
      &  4.50625341176907_wp, -3.53871348312686_wp, -2.06183970372306_wp, &
      &  2.97984063952152_wp, -3.68862704438505_wp,  2.21304327465323_wp, &
      &  1.00750062747367_wp, -3.98076251640596_wp,  2.72367336459902_wp, &
      &  3.81958769213991_wp, -2.43610301729507_wp,  3.61521871993030_wp, &
      &  3.97311521599207_wp, -5.48025953991400_wp,  2.23344149062616_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine THR_xah

subroutine THR_xal(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "O", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.32337289074009_wp, -0.52443088807223_wp,  0.76221779262340_wp, &
      &  0.15241174877138_wp, -0.18300824568791_wp, -0.31990341778584_wp, &
      &  2.01940653580983_wp, -1.34634466481553_wp,  1.53707103689535_wp, &
      &  1.52187373603454_wp, -1.37371087805405_wp,  3.81020838130530_wp, &
      &  0.65517926688285_wp,  2.64511218633230_wp, -0.75912173172639_wp, &
      &  2.94920072896734_wp,  2.80189872958134_wp, -2.16840151743142_wp, &
      &  0.81092329125002_wp,  4.16339894646920_wp,  1.66328101822147_wp, &
      & -2.36688485189821_wp, -0.88331360946650_wp,  2.63324174175432_wp, &
      &  0.19954496231912_wp, -1.10616719020696_wp, -2.15923075184768_wp, &
      & -0.93020373673545_wp,  3.30024111991312_wp, -1.91076969275855_wp, &
      &  3.29783008960685_wp,  4.56478359232966_wp, -2.44398305312741_wp, &
      &  1.02385926678659_wp,  6.16439880422169_wp,  1.21929405061460_wp, &
      &  2.41586610492742_wp,  3.56218775963042_wp,  2.79904597589857_wp, &
      & -0.90427001986065_wp,  3.94189023770743_wp,  2.77601270433896_wp, &
      &  4.20536761189874_wp, -2.28098790330109_wp,  0.61378289677221_wp, &
      &  5.30206701188149_wp, -3.06874641292727_wp,  1.95808943062292_wp, &
      &  4.86611929922403_wp, -2.69646936695167_wp, -2.01759347366140_wp, &
      &  4.41743344828329_wp, -1.03801557471238_wp, -3.13086208111475_wp, &
      &  6.89310445245312_wp, -3.00400862244193_wp, -2.12337948465401_wp, &
      &  3.91566336743662_wp, -4.34888406879415_wp, -2.80008901826533_wp, &
      & -4.42744672996989_wp, -0.60779048120851_wp, -0.70513681509603_wp, &
      & -4.35250516158907_wp, -0.28774755862323_wp, -2.99914251873481_wp, &
      & -6.86400249239842_wp, -1.12174534847192_wp,  0.68635795374460_wp, &
      & -7.65926881236983_wp, -2.88035151243000_wp, -0.02038938785058_wp, &
      & -8.19473906016134_wp,  0.38182113643567_wp,  0.25247838358586_wp, &
      & -6.62079689887026_wp, -1.25538170099738_wp,  2.72171777274231_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine THR_xal

subroutine THR_xaq(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "O", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.05817655108548_wp, -1.16073964230573_wp, -0.36062612256348_wp, &
      &  0.18488367123848_wp,  0.39028786096366_wp, -0.62668828905803_wp, &
      &  2.39269444951894_wp, -1.38081906067269_wp, -0.12644927574363_wp, &
      &  2.20819273778174_wp, -3.10551669006216_wp,  1.41660018878749_wp, &
      &  0.17435375413739_wp,  2.65148665674963_wp,  1.21766378161504_wp, &
      & -1.44939832265934_wp,  4.57137066368301_wp,  0.32256557717459_wp, &
      &  2.74521069323249_wp,  3.86064577005741_wp,  1.58363560163117_wp, &
      & -1.78983766653343_wp, -2.78781072245143_wp,  0.59894064755122_wp, &
      &  0.26029818239484_wp,  1.15451682019040_wp, -2.54400015428539_wp, &
      & -0.48464792676394_wp,  1.90175994542057_wp,  3.03488811660930_wp, &
      & -2.81095868243051_wp,  3.77061681514792_wp, -0.62656238706883_wp, &
      &  2.50191443228882_wp,  5.53337757508339_wp,  2.75316412160516_wp, &
      &  3.52061140615141_wp,  4.47380169105735_wp, -0.22361749368536_wp, &
      &  4.07333521452721_wp,  2.59215245381300_wp,  2.50914554147773_wp, &
      &  4.50665895545156_wp, -0.90523467368599_wp, -1.47451783376083_wp, &
      &  4.50705410964372_wp,  0.60150288215365_wp, -2.62940830573417_wp, &
      &  6.84318740052098_wp, -2.25745975737889_wp, -1.02715668155118_wp, &
      &  6.37184506914276_wp, -4.02776605758655_wp, -0.10636673228189_wp, &
      &  7.78432322435960_wp, -2.64205652544075_wp, -2.81097591937350_wp, &
      &  8.11425054571574_wp, -1.19288657216088_wp,  0.19395120491391_wp, &
      & -4.35523385635683_wp, -0.51920621530224_wp, -1.26145880126749_wp, &
      & -4.77477874583937_wp,  1.52927444443023_wp, -2.29235649882623_wp, &
      & -6.39004198112188_wp, -2.48809514421552_wp, -0.97008081772765_wp, &
      & -8.03697856944244_wp, -1.60691392706133_wp, -0.11735010442563_wp, &
      & -5.79926154554915_wp, -4.09623405414328_wp,  0.16172064821258_wp, &
      & -6.93116528008261_wp, -3.14220844381375_wp, -2.84265018702793_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine THR_xaq

subroutine TRP_xac(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "H", "H", "H", "H", "C", "O", "N", "H", "C", &
      & "H", "H", "H", "C", "H", "H", "H", "C", "C", "C", "H", "C", "C", "H", &
      & "C", "H", "N", "C", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.28184066498160_wp,  2.38748899988803_wp, -0.43677173080904_wp, &
      & -2.95066403824966_wp,  0.30790427134652_wp, -2.07169776531025_wp, &
      & -3.89704486768427_wp, -2.05423856112176_wp, -0.69039292748394_wp, &
      & -4.60125364332795_wp, -3.90813322094938_wp, -1.89419596211560_wp, &
      & -0.66031267886246_wp, -0.48649736618110_wp, -3.67257956171789_wp, &
      & -0.46198437845280_wp,  2.53961890547129_wp,  0.13460090316200_wp, &
      & -4.46593657513628_wp,  0.88176924466867_wp, -3.34338409943025_wp, &
      & -1.29422963231305_wp, -1.97627589901829_wp, -4.94183335841129_wp, &
      & -0.06060269812434_wp,  1.12260160778694_wp, -4.81308885295923_wp, &
      & -3.88533815590503_wp,  4.16693802759755_wp,  0.55189577291625_wp, &
      & -3.12270684334215_wp,  5.78397170482555_wp,  2.01717347366797_wp, &
      & -3.86816810557225_wp, -1.92990074704527_wp,  1.85768555291836_wp, &
      & -3.06603008525562_wp, -0.39462283376861_wp,  2.64224441337380_wp, &
      & -4.49670579943068_wp, -4.09930857039799_wp,  3.39575661459498_wp, &
      & -2.81926791544719_wp, -5.19248535396068_wp,  3.88366435791822_wp, &
      & -5.42816402012612_wp, -3.48880383211492_wp,  5.12088728383401_wp, &
      & -5.76404558963196_wp, -5.29545103043403_wp,  2.31309702676821_wp, &
      & -6.60963127075650_wp,  4.00008882358652_wp, -0.27861776405360_wp, &
      & -7.67278013325563_wp,  5.41338824421813_wp,  0.75583678345179_wp, &
      & -7.39184639865435_wp,  2.13264166048397_wp,  0.09138854594752_wp, &
      & -6.77905733927816_wp,  4.36822019631524_wp, -2.29718977177012_wp, &
      &  1.45632161952334_wp, -1.40588884378959_wp, -2.05123698550723_wp, &
      &  1.77401598373513_wp, -3.84716104211677_wp, -1.18729396257971_wp, &
      &  3.42871597199685_wp,  0.06243932123846_wp, -0.92760844008783_wp, &
      &  0.66411008618760_wp, -5.50649001864835_wp, -1.59470730533117_wp, &
      &  4.88862296414278_wp, -1.59696697234995_wp,  0.60280991673030_wp, &
      &  4.12859206398290_wp,  2.62449278428290_wp, -1.06932082035175_wp, &
      &  4.49084685452849_wp, -5.53020965612664_wp,  1.21985805593236_wp, &
      &  6.20596783130251_wp,  3.44313410934755_wp,  0.29798213597060_wp, &
      &  3.07436208280563_wp,  3.93761323661044_wp, -2.23303691614660_wp, &
      &  3.82461320643843_wp, -3.95143959050895_wp,  0.40163795921211_wp, &
      &  6.98798097479348_wp, -0.77389276711733_wp,  1.97418510413480_wp, &
      &  7.62010427820534_wp,  1.76355358143245_wp,  1.80440435888099_wp, &
      &  8.08250957519875_wp, -2.05949957712178_wp,  3.12837245780790_wp, &
      &  9.23112763404616_wp,  2.46674130800377_wp,  2.84615674122419_wp, &
      &  6.75892184367677_wp,  5.40822628652736_wp,  0.21009845490880_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine TRP_xac

subroutine TRP_xaf(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "H", "H", "H", "H", "C", "O", "N", "H", "C", &
      & "H", "H", "H", "C", "H", "H", "H", "C", "C", "C", "H", "C", "C", "H", &
      & "C", "H", "N", "C", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.11849723892225_wp,  1.26723580798323_wp,  0.26828674013957_wp, &
      & -2.89174948387648_wp, -1.30314092698646_wp, -0.30021766483433_wp, &
      & -5.43564051118517_wp, -2.00970211778138_wp,  0.85542377224423_wp, &
      & -6.09797925779686_wp, -4.22445346160962_wp,  1.00189787195776_wp, &
      & -0.85686977733761_wp, -3.24191084457665_wp,  0.44685930785952_wp, &
      & -1.40097086433040_wp,  1.55207079024000_wp,  2.01379092989909_wp, &
      & -3.12122750739564_wp, -1.35384554539634_wp, -2.35005086159973_wp, &
      & -0.75206567777266_wp, -3.35641277768572_wp,  2.50644286354345_wp, &
      & -1.46305199251542_wp, -5.09242073514157_wp, -0.21441455023597_wp, &
      & -1.01124672139522_wp,  2.72359805492801_wp, -1.60172105812975_wp, &
      & -1.23904352170178_wp,  2.24426877559041_wp, -3.84588283453043_wp, &
      & -6.85049877549191_wp, -0.03114816399514_wp,  1.65226737639064_wp, &
      & -6.15777082035441_wp,  1.70331820159526_wp,  1.29632643395321_wp, &
      & -9.39430097288992_wp, -0.36249000837854_wp,  2.59120661448571_wp, &
      & -9.64641738922020_wp, -2.35912598309061_wp,  2.98962788669082_wp, &
      & -9.67369151831458_wp,  0.71929777058196_wp,  4.31630240889085_wp, &
      &-10.79701762728874_wp,  0.21704359600056_wp,  1.19996748386852_wp, &
      &  0.46570900961940_wp,  4.96265349713617_wp, -0.64018522483527_wp, &
      &  2.44543852446221_wp,  4.41106570320168_wp, -0.47363419623909_wp, &
      & -0.18319634433493_wp,  5.59903082026107_wp,  1.20411815992153_wp, &
      &  0.31983459702752_wp,  6.49039655355519_wp, -2.00200251823876_wp, &
      &  1.63149624013075_wp, -2.47624656508786_wp, -0.62558164054968_wp, &
      &  2.47617271471412_wp, -2.87817002417493_wp, -3.05704058760091_wp, &
      &  3.42062524399931_wp, -0.81867768124288_wp,  0.52796619842960_wp, &
      &  1.63618567361021_wp, -3.97199575679961_wp, -4.55560623873002_wp, &
      &  5.30257202412713_wp, -0.25589459108081_wp, -1.30811738100338_wp, &
      &  3.61911954856778_wp,  0.27431706932796_wp,  2.94158634056748_wp, &
      &  5.63037066752720_wp, -1.50118202479424_wp, -5.09482034668238_wp, &
      &  5.60918899678733_wp,  1.90310033483542_wp,  3.43739854099740_wp, &
      &  2.25765133627232_wp, -0.16884002908280_wp,  4.40624741804060_wp, &
      &  4.67938428428902_wp, -1.56349581251951_wp, -3.45264658687790_wp, &
      &  7.31234462118128_wp,  1.38219029174404_wp, -0.81143291696534_wp, &
      &  7.42956461287616_wp,  2.46362624578920_wp,  1.57454770883890_wp, &
      &  8.72412241866887_wp,  1.80399922945984_wp, -2.23015043565866_wp, &
      &  8.95734170834459_wp,  3.74588074778232_wp,  2.01953240833212_wp, &
      &  5.78348132626363_wp,  2.75581544561569_wp,  5.28696312137437_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine TRP_xaf

subroutine TRP_xag(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "H", "H", "H", "H", "C", "O", "N", "H", "C", &
      & "H", "H", "H", "C", "H", "H", "H", "C", "C", "C", "H", "C", "C", "H", &
      & "C", "H", "N", "C", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.77895463012031_wp,  1.27155266271372_wp, -0.26800099182896_wp, &
      &  2.69131648885045_wp, -1.19254340640759_wp,  0.55186445848477_wp, &
      &  5.17982237227110_wp, -1.84461119888567_wp, -0.79187314629336_wp, &
      &  5.30576825990939_wp, -3.46388519948070_wp, -2.44716545670443_wp, &
      &  0.69362560770611_wp, -3.20996051019545_wp,  0.02846107103399_wp, &
      &  0.19408619356463_wp,  1.32983799350696_wp, -1.32347360157018_wp, &
      &  3.04688600611600_wp, -1.05012973415177_wp,  2.58117666849608_wp, &
      &  0.64151932743209_wp, -3.56143542248793_wp, -2.00094992502442_wp, &
      &  1.32431126138429_wp, -4.96893694623115_wp,  0.89318072175182_wp, &
      &  2.55487717692842_wp,  3.44226966847505_wp,  0.83904981613031_wp, &
      &  4.31897702038708_wp,  3.51391586757882_wp,  2.35965240426410_wp, &
      &  7.16140975353154_wp, -0.46953361853862_wp,  0.04222421819709_wp, &
      &  6.75266464608678_wp,  0.99223267261082_wp,  1.21351908909171_wp, &
      &  9.61262491096841_wp, -0.65443172768328_wp, -1.15299269338887_wp, &
      &  9.80949629685853_wp,  0.67201993073209_wp, -2.71788009441775_wp, &
      & 11.08312668689243_wp, -0.27901485716354_wp,  0.23010955533534_wp, &
      &  9.83145096827094_wp, -2.55711336107626_wp, -1.88849381004926_wp, &
      &  1.11989911865991_wp,  5.78476852957421_wp,  0.08513004398986_wp, &
      &  2.44237800025087_wp,  7.14790509372523_wp, -0.69935387679783_wp, &
      &  0.30093078347018_wp,  6.61159792398309_wp,  1.77941996656584_wp, &
      & -0.37362607628521_wp,  5.41061984331705_wp, -1.27593462268462_wp, &
      & -1.84677480542116_wp, -2.44675658099597_wp,  0.98195544172308_wp, &
      & -2.79628438629888_wp, -2.73457287708718_wp,  3.39154380568855_wp, &
      & -3.71532346337473_wp, -1.06897123284281_wp, -0.40033312265719_wp, &
      & -1.97493594423590_wp, -3.66702260100113_wp,  5.00578465252469_wp, &
      & -5.75368689470877_wp, -0.56087510249254_wp,  1.27437256511581_wp, &
      & -3.86689093681904_wp, -0.20452776912926_wp, -2.91042198508642_wp, &
      & -6.22947239589291_wp, -1.59583338455393_wp,  5.10926777966566_wp, &
      & -5.98809188874096_wp,  1.14022918111436_wp, -3.65699880931646_wp, &
      & -2.37837685180502_wp, -0.61964340878232_wp, -4.25555342866013_wp, &
      & -5.13211929362152_wp, -1.60749434044429_wp,  3.55995465153391_wp, &
      & -7.89584025799772_wp,  0.78346281262504_wp,  0.52041856897180_wp, &
      & -7.97851134741028_wp,  1.63271629272975_wp, -1.95833884475229_wp, &
      & -9.43601608520481_wp,  1.15651039819209_wp,  1.81333748633012_wp, &
      & -9.60889536673386_wp,  2.68211522028287_wp, -2.60326579492377_wp, &
      & -6.13428911996988_wp,  1.80876810785010_wp, -5.58276336031421_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine TRP_xag

subroutine TRP_xah(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "H", "H", "H", "H", "C", "O", "N", "H", "C", &
      & "H", "H", "H", "C", "H", "H", "H", "C", "C", "C", "H", "C", "C", "H", &
      & "C", "H", "N", "C", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.82279860098090_wp, -1.78950310560160_wp,  0.01595758018687_wp, &
      & -3.07467734423175_wp, -0.10282828335417_wp,  2.01819285728018_wp, &
      & -2.25332433990775_wp,  2.53201417618451_wp,  1.15715189480181_wp, &
      & -1.92504886605506_wp,  4.20947235727606_wp,  2.72778445678488_wp, &
      & -0.89483419613601_wp, -1.31418030964292_wp,  3.52683579286721_wp, &
      & -2.51346807242738_wp, -3.02805339133752_wp, -0.62548973960466_wp, &
      & -4.64239687706101_wp,  0.20254512769202_wp,  3.31870252038942_wp, &
      & -0.30875218230300_wp,  0.04581828661220_wp,  4.95808257126078_wp, &
      & -1.63820165555012_wp, -2.97817927124570_wp,  4.49113361955431_wp, &
      & -6.09038948738286_wp, -1.83265589046700_wp, -1.23464279300697_wp, &
      & -6.46089990876444_wp, -3.29429353311235_wp, -2.98717389081611_wp, &
      & -1.91063746562750_wp,  2.81800780609743_wp, -1.35246406176979_wp, &
      & -2.07725723044198_wp,  1.25898995222369_wp, -2.42892461106653_wp, &
      & -0.89416542350704_wp,  5.12074215771364_wp, -2.41962483803037_wp, &
      & -1.84072861975862_wp,  5.55368376125163_wp, -4.19084683777148_wp, &
      & -1.22804994666089_wp,  6.63333438963177_wp, -1.07387619253164_wp, &
      &  1.13373925751242_wp,  4.95621355444465_wp, -2.74396827781517_wp, &
      & -8.05957298243070_wp,  0.00637954144493_wp, -0.30287728640211_wp, &
      & -7.33246539205454_wp,  1.93150161717680_wp, -0.28084982929427_wp, &
      & -8.64000325536511_wp, -0.47113445784880_wp,  1.61414251565278_wp, &
      & -9.68300568468128_wp, -0.10947014232720_wp, -1.54747102856112_wp, &
      &  1.27113082166963_wp, -2.05021947157904_wp,  1.87975936687500_wp, &
      &  1.72306136835909_wp, -4.43817232952472_wp,  0.92052398859556_wp, &
      &  3.21331757759842_wp, -0.45282414245878_wp,  0.87848691672296_wp, &
      &  0.72069855078448_wp, -6.17943981457799_wp,  1.26499093038541_wp, &
      &  4.78157114276111_wp, -1.98301317356894_wp, -0.67472445447282_wp, &
      &  3.81341504562594_wp,  2.11930731240518_wp,  1.17674382214455_wp, &
      &  4.55464525686613_wp, -5.89628117506865_wp, -1.50151414616441_wp, &
      &  5.90392945452670_wp,  3.06982604424245_wp, -0.08169636443072_wp, &
      &  2.65992476163630_wp,  3.33100347298667_wp,  2.35356214098591_wp, &
      &  3.81921067110297_wp, -4.39124747666108_wp, -0.60746891805910_wp, &
      &  6.88342671429603_wp, -1.02137034432468_wp, -1.94951383640345_wp, &
      &  7.41907827816572_wp,  1.52361960878393_wp, -1.63374406257424_wp, &
      &  8.05537822381601_wp, -2.21121253321841_wp, -3.12994245160775_wp, &
      &  9.03509347402971_wp,  2.33343060125134_wp, -2.58728022491708_wp, &
      &  6.39426992054518_wp,  5.04252092320254_wp,  0.13598489981294_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine TRP_xah

subroutine TRP_xao(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "H", "H", "H", "H", "C", "O", "N", "H", "C", &
      & "H", "H", "H", "C", "H", "H", "H", "C", "C", "C", "H", "C", "C", "H", &
      & "C", "H", "N", "C", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  5.36178562021548_wp, -0.86048871890065_wp,  0.24155874701671_wp, &
      &  2.78966205618807_wp, -0.01857319695014_wp, -0.00147688083250_wp, &
      &  2.84104063182052_wp,  2.74042888211389_wp, -0.78465064604050_wp, &
      &  4.60541970814365_wp,  4.13057422570706_wp, -0.19038757356460_wp, &
      &  1.33549211238043_wp, -0.31092052441334_wp,  2.51394746903429_wp, &
      &  6.67970883875203_wp,  0.49567588459453_wp,  0.47406551066963_wp, &
      &  1.85526836035514_wp, -1.16409080206517_wp, -1.43553551701080_wp, &
      &  1.53855635580278_wp, -2.28165704805242_wp,  3.08585205133863_wp, &
      &  2.25312255860401_wp,  0.87468814254255_wp,  3.93092442797753_wp, &
      &  5.93779681412877_wp, -3.35550990499241_wp,  0.18059304600424_wp, &
      &  4.31242952996756_wp, -4.99089353003860_wp, -0.09222256562040_wp, &
      &  0.80495393670656_wp,  3.50799599070577_wp, -2.12295802701400_wp, &
      & -0.65324812283945_wp,  2.28729164547247_wp, -2.24901195237222_wp, &
      &  0.35439559555871_wp,  6.15827607642616_wp, -2.62700605984220_wp, &
      & -0.68598672820270_wp,  6.36001168219921_wp, -4.38529905603229_wp, &
      & -0.70771784523920_wp,  7.05121230044431_wp, -1.10203207032776_wp, &
      &  2.17165689338667_wp,  7.09392898350949_wp, -2.79467911553577_wp, &
      &  8.70325572107707_wp, -3.99117164598731_wp,  0.46294177287978_wp, &
      &  8.93431131578095_wp, -5.18734453574402_wp,  2.11780232265859_wp, &
      &  9.89872748314451_wp, -2.33100985966900_wp,  0.64907981009734_wp, &
      &  9.29334111193918_wp, -5.07654382631895_wp, -1.17882425414283_wp, &
      & -1.37328653302652_wp,  0.38285660041822_wp,  2.22230891525923_wp, &
      & -2.60517236613291_wp,  2.44722666787026_wp,  3.22404512288238_wp, &
      & -3.15190335520321_wp, -0.85051257178068_wp,  0.60138880392625_wp, &
      & -1.91414152304361_wp,  3.86314557632254_wp,  4.51557115097497_wp, &
      & -5.43795150764749_wp,  0.55702165228641_wp,  0.69754344909581_wp, &
      & -3.02562042258436_wp, -3.02525978399114_wp, -0.92398590899136_wp, &
      & -6.34998198296018_wp,  3.81766133749838_wp,  2.83538872366304_wp, &
      & -5.15186144782377_wp, -3.71618705681880_wp, -2.28779399452654_wp, &
      & -1.31486555432538_wp, -4.14839809047111_wp, -1.00105796241923_wp, &
      & -5.03639894617785_wp,  2.55332729715239_wp,  2.30533364404180_wp, &
      & -7.58239747731925_wp, -0.14837530941998_wp, -0.67152917529297_wp, &
      & -7.40214725385649_wp, -2.29522106376775_wp, -2.16486716557214_wp, &
      & -9.31882247447703_wp,  0.92809868240080_wp, -0.57610453196310_wp, &
      & -9.02527628632429_wp, -2.89560706352924_wp, -3.25177471191203_wp, &
      & -5.09473532025296_wp, -5.38546037960255_wp, -3.46518450927112_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine TRP_xao

subroutine TYR_xab(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 33
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "C", "C", "C", "O", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", &
      & "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.66289111453127_wp,  1.32985956703856_wp, -0.96622838402346_wp, &
      &  2.09201196487016_wp, -1.04806743919723_wp,  0.34692014869124_wp, &
      &  4.78560128454679_wp, -2.00336033417540_wp, -0.15005668517722_wp, &
      &  5.22868903543263_wp, -3.81551580607968_wp, -1.52707059706282_wp, &
      &  0.14585534812600_wp, -2.99506141845851_wp, -0.50350662568840_wp, &
      & -2.49893832652159_wp, -1.99796144704559_wp, -0.25989022717500_wp, &
      & -3.92293507650161_wp, -1.37231399901071_wp, -2.39504027232629_wp, &
      & -3.57458544044415_wp, -1.56632677936236_wp,  2.11289863947693_wp, &
      & -6.33516847444843_wp, -0.34321325404017_wp, -2.18879896311598_wp, &
      & -5.98429758327274_wp, -0.54198881689337_wp,  2.34858552421431_wp, &
      & -7.37003486264875_wp,  0.08089395185972_wp,  0.19316010855949_wp, &
      & -9.74091771500962_wp,  1.09645978876892_wp,  0.30039455768535_wp, &
      &  0.49931664595812_wp,  1.31655767253442_wp, -2.47173544005810_wp, &
      &  1.89180897356978_wp, -0.65726287602813_wp,  2.36403484190725_wp, &
      &  0.55773912834828_wp, -3.52394153581808_wp, -2.45161935517749_wp, &
      &  0.41548152238812_wp, -4.69573451740712_wp,  0.62692948416957_wp, &
      & -3.14729910011055_wp, -1.72965265151201_wp, -4.25681826863174_wp, &
      & -2.52384573955173_wp, -2.04111358994364_wp,  3.80454455122264_wp, &
      & -7.43512545333359_wp,  0.12658898031300_wp, -3.84474910728116_wp, &
      & -6.78919936289522_wp, -0.22883087631456_wp,  4.20541276414295_wp, &
      &-10.21178298637114_wp,  1.25832036375991_wp,  2.05034460536992_wp, &
      &  6.57636938693571_wp, -0.65817430819707_wp,  1.07111287255184_wp, &
      &  6.00732484838854_wp,  0.96377729921065_wp,  1.91710592475900_wp, &
      &  9.22872788466294_wp, -1.15518105556421_wp,  0.63253681194464_wp, &
      &  9.86810868709415_wp, -0.38904256264020_wp, -1.17047702622942_wp, &
      & 10.32876824266482_wp, -0.31587231313850_wp,  2.14801869126056_wp, &
      &  9.53764381552637_wp, -3.18425931781666_wp,  0.60612125627655_wp, &
      &  2.29383844192370_wp,  3.58341906896155_wp,  0.06916607452121_wp, &
      &  3.54409668467644_wp,  3.74533008659775_wp,  2.02720461030738_wp, &
      &  1.36367360684994_wp,  5.88987075526440_wp, -1.31688568103947_wp, &
      &  0.03195903722374_wp,  6.87878987642808_wp, -0.10208998769546_wp, &
      &  0.44597288144718_wp,  5.44497068504998_wp, -3.10006070203835_wp, &
      &  2.95422707153955_wp,  7.14425326970044_wp, -1.65550157529068_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine TYR_xab

subroutine TYR_xag(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 33
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "C", "C", "C", "O", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", &
      & "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  5.23115965604457_wp, -0.27890753638359_wp,  0.24597784911967_wp, &
      &  2.57414984207536_wp,  0.02434078618284_wp, -0.23010793331735_wp, &
      &  2.02466836849966_wp,  2.84494770276726_wp, -0.26078333274025_wp, &
      &  3.36727684928512_wp,  4.34231750735038_wp,  0.90123244302968_wp, &
      &  0.97331501980564_wp, -1.29115869938207_wp,  1.82841550125503_wp, &
      & -1.79163734701974_wp, -1.19028892813479_wp,  1.22636770992832_wp, &
      & -2.77906195719775_wp, -2.68019514984526_wp, -0.71959364648747_wp, &
      & -3.41656982088666_wp,  0.49400995141498_wp,  2.45323758937911_wp, &
      & -5.30638519043062_wp, -2.48333874069372_wp, -1.43084346374973_wp, &
      & -5.94520223293742_wp,  0.70081609681500_wp,  1.76777420492423_wp, &
      & -6.89612933536707_wp, -0.78672162004452_wp, -0.18852788180765_wp, &
      & -9.38983552572081_wp, -0.50411993640646_wp, -0.79225637467640_wp, &
      &  6.13324679719916_wp,  1.23207719559946_wp,  0.97750079274851_wp, &
      &  2.11845665740064_wp, -0.83674284267005_wp, -2.04750461362565_wp, &
      &  1.63376334617461_wp, -3.24001763023955_wp,  1.92198305707176_wp, &
      &  1.36795056432908_wp, -0.37638283847796_wp,  3.63374572526750_wp, &
      & -1.55873678391524_wp, -4.01443404691659_wp, -1.68160983121953_wp, &
      & -2.68724561479418_wp,  1.66639738838334_wp,  3.96467156659768_wp, &
      & -6.04510795684144_wp, -3.65900980714134_wp, -2.93629859484081_wp, &
      & -7.20238781119867_wp,  1.99383775332356_wp,  2.72762210417257_wp, &
      & -9.78892985751892_wp, -1.67488389534534_wp, -2.12686284680734_wp, &
      & -0.03512683606609_wp,  3.52731309289954_wp, -1.60961558305245_wp, &
      & -1.16538986990553_wp,  2.12523346428492_wp, -2.22727558626018_wp, &
      & -1.05981874825708_wp,  6.06087517656298_wp, -1.45749474126616_wp, &
      &  0.48839888768217_wp,  7.34970884360700_wp, -1.07516417140652_wp, &
      & -1.94263283622337_wp,  6.55868459730977_wp, -3.24285056359581_wp, &
      & -2.45062449989243_wp,  6.22355977612186_wp,  0.05481705640472_wp, &
      &  6.42397196948537_wp, -2.48331282462886_wp, -0.28784083763520_wp, &
      &  5.30249150092336_wp, -4.30360246342357_wp, -1.18991153657943_wp, &
      &  9.21351669450925_wp, -2.54742378366602_wp,  0.29944579879575_wp, &
      &  9.55939191720878_wp, -4.04144014462394_wp,  1.66748826260447_wp, &
      &  9.92492544918466_wp, -0.76832984404314_wp,  1.03963993679347_wp, &
      & 10.23109255451489_wp, -3.02854983764200_wp, -1.41934985932501_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine TYR_xag

subroutine TYR_xah(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 33
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "C", "C", "C", "O", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", &
      & "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.65384489573616_wp, -1.82581709391886_wp,  0.16209005858145_wp, &
      &  2.64375083207776_wp,  0.04879673248038_wp, -1.82825190040322_wp, &
      &  2.80550622819937_wp,  2.77091171351029_wp, -0.86520497121252_wp, &
      &  2.51552165951579_wp,  4.55207674647145_wp, -2.32219936203003_wp, &
      &  0.29764795697382_wp, -0.24577979530299_wp, -3.51382231166457_wp, &
      & -2.08329626456848_wp, -0.18393102273101_wp, -1.96988487733470_wp, &
      & -3.24721174964477_wp, -2.42418985349794_wp, -1.17834144325469_wp, &
      & -3.12639622506120_wp,  2.10887734881721_wp, -1.16432763950322_wp, &
      & -5.37414462414758_wp, -2.39832969927557_wp,  0.37069126590764_wp, &
      & -5.25427996863231_wp,  2.15545387089351_wp,  0.38022311181796_wp, &
      & -6.37906197520119_wp, -0.09771298028361_wp,  1.16103168906060_wp, &
      & -8.46657721800195_wp, -0.16776039194982_wp,  2.67845993286183_wp, &
      &  0.96287807246614_wp, -2.27200239016044_wp,  0.92452388586262_wp, &
      &  4.35000774101992_wp, -0.27195717705555_wp, -2.94030724495517_wp, &
      &  0.32942195216337_wp,  1.28956069242816_wp, -4.88235454187099_wp, &
      &  0.44015925553129_wp, -2.03457174622113_wp, -4.52617834673383_wp, &
      & -2.50226356078619_wp, -4.22128892293241_wp, -1.81969295947416_wp, &
      & -2.26644614081024_wp,  3.86544229051881_wp, -1.76303550895748_wp, &
      & -6.28011293174404_wp, -4.13160059242955_wp,  0.96093355127440_wp, &
      & -6.04648902008921_wp,  3.94864614053865_wp,  0.97339187234143_wp, &
      & -8.97037461412021_wp,  1.54477630302625_wp,  3.03063340256668_wp, &
      &  3.28850586151602_wp,  3.02048486621097_wp,  1.62670348571816_wp, &
      &  3.56130460133098_wp,  1.41662212386472_wp,  2.60862748397690_wp, &
      &  3.63462690409669_wp,  5.46053529983798_wp,  2.80346119129760_wp, &
      &  3.11631642748891_wp,  6.88498322365435_wp,  1.42083629776953_wp, &
      &  2.43025349517635_wp,  5.64461610845561_wp,  4.45957160126897_wp, &
      &  5.59445285986250_wp,  5.75329104527281_wp,  3.35922202258174_wp, &
      &  4.60086969892078_wp, -3.51678182769017_wp,  0.43714798356923_wp, &
      &  6.59772320298782_wp, -3.32007492686549_wp, -0.70333046232867_wp, &
      &  4.09775788972837_wp, -5.62847853650429_wp,  2.29008201184565_wp, &
      &  4.05022400500195_wp, -7.40406675317940_wp,  1.25414020528900_wp, &
      &  2.33360091714391_wp, -5.40045546524789_wp,  3.31937780047263_wp, &
      &  5.66156753141610_wp, -5.72705988810016_wp,  3.61780951532150_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine TYR_xah

subroutine TYR_xan(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 33
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "C", "C", "C", "O", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", &
      & "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.75161272413616_wp,  1.15731754585141_wp,  1.96326854598955_wp, &
      &  2.16792260935546_wp, -1.39582847924770_wp,  1.00388060970957_wp, &
      &  4.94216504201068_wp, -2.11897689149105_wp,  0.61167372745597_wp, &
      &  5.74517301658264_wp, -4.11901912210060_wp,  1.47170724899817_wp, &
      &  0.61609763631899_wp, -1.95562267492899_wp, -1.39336060531747_wp, &
      & -2.12232514925251_wp, -1.34423107931737_wp, -0.99246156509236_wp, &
      & -3.16672955248999_wp,  0.89523578228445_wp, -1.92956709549194_wp, &
      & -3.66949209229559_wp, -2.93931295914717_wp,  0.43239544133659_wp, &
      & -5.67441640321747_wp,  1.53164022053197_wp, -1.46094899319033_wp, &
      & -6.18152999326719_wp, -2.32280695124816_wp,  0.91476646621823_wp, &
      & -7.18885097241470_wp, -0.07716942931631_wp, -0.02792374330027_wp, &
      & -9.63760626751047_wp,  0.63816792775363_wp,  0.37883525652522_wp, &
      &  0.92651312154328_wp,  1.28406454602709_wp,  3.66977183474840_wp, &
      &  1.51825410499887_wp, -2.65783749214089_wp,  2.49158571007912_wp, &
      &  1.36676437816025_wp, -0.87065791511956_wp, -2.96992944835848_wp, &
      &  0.87404465706133_wp, -3.95847549373105_wp, -1.81327859721945_wp, &
      & -1.99942063353804_wp,  2.14788113379088_wp, -3.05082539517273_wp, &
      & -2.91328074782044_wp, -4.69728300496017_wp,  1.16214143117868_wp, &
      & -6.48735436786021_wp,  3.25386804876813_wp, -2.20125984766974_wp, &
      & -7.35937612711215_wp, -3.59059216963545_wp,  2.01022150298969_wp, &
      &-10.45368064670863_wp, -0.66566275669907_wp,  1.35067286233262_wp, &
      &  6.30153109610149_wp, -0.47955967841497_wp, -0.78914310052213_wp, &
      &  5.39496159714428_wp,  1.07703312302566_wp, -1.44646198518224_wp, &
      &  8.92551852341387_wp, -0.94040611024871_wp, -1.39690953684964_wp, &
      &  9.38786038262414_wp, -2.84089126340499_wp, -0.77783675475078_wp, &
      &  9.23140236240910_wp, -0.79052162133939_wp, -3.42469490701965_wp, &
      & 10.16368033794106_wp,  0.39519766791971_wp, -0.43744335970246_wp, &
      &  1.84996338034404_wp,  3.31119164434385_wp,  0.58638737330456_wp, &
      &  2.73854944519104_wp,  3.40519082132545_wp, -1.56610618356805_wp, &
      &  0.73578951597287_wp,  5.60775396978376_wp,  1.84945753399166_wp, &
      &  0.82337501616154_wp,  5.50149709733747_wp,  3.90034731648343_wp, &
      & -1.23960509458874_wp,  5.75685851061086_wp,  1.28517419243681_wp, &
      &  1.73032612353342_wp,  7.27741017728146_wp,  1.19513023643785_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine TYR_xan

subroutine TYR_xar(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 33
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "C", "C", "C", "C", "O", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "N", "H", "C", "H", "H", "H", "C", &
      & "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -5.14881142503109_wp, -1.54134862225749_wp, -1.44667480638212_wp, &
      & -2.75730780412959_wp, -0.66977807377502_wp, -0.49748215666943_wp, &
      & -2.71959981351362_wp,  2.19091073571111_wp, -0.76131640358841_wp, &
      & -3.96996568820332_wp,  3.26305584806441_wp, -2.39930273649702_wp, &
      & -0.53587498341067_wp, -1.84152026454143_wp, -1.98849134154818_wp, &
      &  1.96456316545982_wp, -1.05413019166580_wp, -0.91942273715322_wp, &
      &  2.78125807973564_wp, -2.01477436195663_wp,  1.40713040193189_wp, &
      &  3.45206665344952_wp,  0.77684494232163_wp, -2.10080117617416_wp, &
      &  4.99949507365612_wp, -1.16364027276867_wp,  2.53209225549454_wp, &
      &  5.68358321326656_wp,  1.63618257884881_wp, -0.99990970734014_wp, &
      &  6.45770979877027_wp,  0.67282088455576_wp,  1.32775376960050_wp, &
      &  8.61790209896111_wp,  1.43841545254197_wp,  2.51479518224549_wp, &
      & -6.10292130716799_wp, -0.32990791931321_wp, -2.56621111071053_wp, &
      & -2.57728002451866_wp, -1.23205108869313_wp,  1.47804399344025_wp, &
      & -0.77548126732098_wp, -3.88488252803499_wp, -1.88539769066281_wp, &
      & -0.70890250368961_wp, -1.25732701908741_wp, -3.95837676086365_wp, &
      &  1.66609401152777_wp, -3.45591195628473_wp,  2.34242813539979_wp, &
      &  2.85362472260304_wp,  1.55065518972711_wp, -3.89894174368350_wp, &
      &  5.63534258975931_wp, -1.91374848444956_wp,  4.32274038178625_wp, &
      &  6.82162967410182_wp,  3.04985173012062_wp, -1.94905638314027_wp, &
      &  9.45079438841707_wp,  2.65879756916660_wp,  1.45280464832530_wp, &
      & -1.20300087737416_wp,  3.41081199146132_wp,  0.89365445055705_wp, &
      & -0.02435271116615_wp,  2.34063761376008_wp,  1.93826025684545_wp, &
      & -0.62718412112838_wp,  6.06849727907837_wp,  0.58419651759402_wp, &
      & -2.28726257608012_wp,  7.00130170270924_wp, -0.17673052581165_wp, &
      & -0.15654036984303_wp,  6.88394882284894_wp,  2.40809028987453_wp, &
      &  0.94681580585589_wp,  6.35578423778402_wp, -0.71576060890304_wp, &
      & -6.03743486012378_wp, -3.87088102578027_wp, -0.85210373540755_wp, &
      & -4.85552426883671_wp, -5.32908610485325_wp,  0.51139364016136_wp, &
      & -8.57102192419426_wp, -4.54504310593206_wp, -1.97385213365128_wp, &
      & -8.34570079820420_wp, -6.22752598161204_wp, -3.13190997521587_wp, &
      & -9.37925526278777_wp, -3.03763829505742_wp, -3.11131953818480_wp, &
      & -9.85907984356600_wp, -5.00991093747797_wp, -0.44208739963882_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine TYR_xar

subroutine VAL_xad(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.33155476149775_wp, -1.70545921741178_wp,  0.81346087433291_wp, &
      & -0.91233683513150_wp, -0.58604513278175_wp, -0.32147737137044_wp, &
      & -1.03805735086385_wp,  2.18603962383204_wp,  0.50876020096835_wp, &
      & -2.29634213670049_wp,  2.88949291896202_wp,  2.32507295140936_wp, &
      & -3.28870929114782_wp, -2.04429106045432_wp,  0.38729490459953_wp, &
      & -5.56509268203769_wp, -0.88717210795507_wp, -0.94798375249529_wp, &
      & -3.03584114401925_wp, -4.82974671485025_wp, -0.30431341155242_wp, &
      &  1.19724886913994_wp, -2.39925975650434_wp,  2.57937652596024_wp, &
      & -0.62548681025080_wp, -0.67637202225223_wp, -2.36279477265950_wp, &
      & -3.55839535788595_wp, -1.84202228990621_wp,  2.42504497728325_wp, &
      & -5.82954092195203_wp,  1.07811588723988_wp, -0.41111311299780_wp, &
      & -7.27860236908855_wp, -1.92114153421388_wp, -0.46687004712755_wp, &
      & -5.32075140582479_wp, -0.99390583114794_wp, -2.99374852936323_wp, &
      & -1.46593679395468_wp, -5.74174790768299_wp,  0.66172296267437_wp, &
      & -4.75950456064034_wp, -5.84457416461019_wp,  0.18182389721999_wp, &
      & -2.73154246662919_wp, -5.03680106391558_wp, -2.33376469613119_wp, &
      &  0.42385674523511_wp,  3.73178845112376_wp, -0.91044105743700_wp, &
      &  1.68654750204192_wp,  2.87395494626506_wp, -2.06303615711132_wp, &
      &  0.80052844293010_wp,  6.34168886344909_wp, -0.18120594059688_wp, &
      &  2.20939084305156_wp,  6.53231036889370_wp,  1.31095012922442_wp, &
      &  1.40171709554486_wp,  7.42328127676065_wp, -1.81849453353671_wp, &
      & -0.97719835526554_wp,  7.08881972686019_wp,  0.51910343755631_wp, &
      &  3.65038759434496_wp, -1.44849781393099_wp, -0.25077699446835_wp, &
      &  3.97948849825316_wp, -0.32598022504964_wp, -2.26111278444344_wp, &
      &  5.80438995912876_wp, -2.66575216703170_wp,  1.16104460664208_wp, &
      &  6.67568041288949_wp, -4.06479643860202_wp, -0.06631996318693_wp, &
      &  5.22059647020434_wp, -3.57026117787926_wp,  2.91066780168636_wp, &
      &  7.21535545661069_wp, -1.22936906112351_wp,  1.57040400450565_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine VAL_xad

subroutine VAL_xaf(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.52403205817081_wp, -1.82588874673075_wp, -0.44433318961569_wp, &
      &  0.66836142816610_wp, -0.73506388180940_wp,  0.81157058340563_wp, &
      &  1.21245259493930_wp,  1.88668285614364_wp, -0.30935518193437_wp, &
      &  2.92304671953406_wp,  2.32361621258778_wp, -1.81569611550793_wp, &
      &  2.90181071430293_wp, -2.56680477768741_wp,  0.76066438781545_wp, &
      &  3.70133858423468_wp, -3.39106386942862_wp, -1.88721395735108_wp, &
      &  5.13463224968511_wp, -1.44268661907675_wp,  2.19711023906420_wp, &
      & -1.24582348617234_wp, -2.87477267739567_wp, -2.00334843450639_wp, &
      &  0.09809738097817_wp, -0.46027474280734_wp,  2.77479411441420_wp, &
      &  2.25090153710847_wp, -4.24018668799158_wp,  1.78546851071253_wp, &
      &  2.19414674783479_wp, -4.36538346704961_wp, -2.90350180867288_wp, &
      &  5.26505454396097_wp, -4.72362576718146_wp, -1.74679641618310_wp, &
      &  4.30428337714621_wp, -1.77110733615533_wp, -2.99458683959747_wp, &
      &  4.57347970747791_wp, -0.81861064674671_wp,  4.07923683785385_wp, &
      &  6.62231945642433_wp, -2.84860976208726_wp,  2.41368319576845_wp, &
      &  5.90529385373371_wp,  0.16640355666566_wp,  1.17544291559392_wp, &
      & -0.42414736706153_wp,  3.66354069429079_wp,  0.52766607849595_wp, &
      & -1.94159459646144_wp,  3.04382900014169_wp,  1.51680048407933_wp, &
      & -0.39226290939119_wp,  6.19287665124214_wp, -0.51091019737394_wp, &
      & -1.23995842650424_wp,  7.48921182561637_wp,  0.83569698758861_wp, &
      &  1.55588728545404_wp,  6.73693000898669_wp, -0.85533985545165_wp, &
      & -1.41641391039435_wp,  6.30872549173706_wp, -2.29561587161654_wp, &
      & -3.92108853964237_wp, -1.27967922671641_wp,  0.27519265160206_wp, &
      & -4.40012489011625_wp,  0.18703377919325_wp,  2.01733196122871_wp, &
      & -5.98251645216909_wp, -2.61167178761536_wp, -1.17107353827408_wp, &
      & -7.04560630166264_wp, -3.78061318097651_wp,  0.14322822234830_wp, &
      & -5.27399582924757_wp, -3.78713852817013_wp, -2.69957658965779_wp, &
      & -7.26719801551892_wp, -1.20012086484465_wp, -1.93086717746676_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine VAL_xaf

subroutine VAL_xah(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.94286186007705_wp, -0.80107999153426_wp,  0.79094856396162_wp, &
      &  0.12846168266956_wp,  0.21743478957165_wp, -0.65771485020823_wp, &
      &  2.52945983210346_wp, -1.09040573136701_wp,  0.24932373668230_wp, &
      &  2.61238613220292_wp, -2.23192724245921_wp,  2.26766622355507_wp, &
      &  0.37688394859742_wp,  3.10807361175921_wp, -0.37560805980312_wp, &
      & -2.00972222739401_wp,  4.41456466116849_wp, -1.31500195400464_wp, &
      &  0.95356345038258_wp,  3.85072369351295_wp,  2.34557769286647_wp, &
      & -1.41819420713476_wp, -1.65913176492086_wp,  2.41407709190753_wp, &
      & -0.22588242582872_wp, -0.22116555322302_wp, -2.64429945112323_wp, &
      &  1.96121719922571_wp,  3.68826151803574_wp, -1.57045357134414_wp, &
      & -2.45714613089452_wp,  3.88167656270563_wp, -3.25049273285621_wp, &
      & -1.78450768057662_wp,  6.45924900474131_wp, -1.21817234253324_wp, &
      & -3.61607266171733_wp,  3.89157005324518_wp, -0.13812461431324_wp, &
      &  2.71189750783345_wp,  3.02198857748640_wp,  3.02395909445107_wp, &
      &  1.10073647220503_wp,  5.89744941248432_wp,  2.50965231348037_wp, &
      & -0.56818813782834_wp,  3.22477262519527_wp,  3.58539821264051_wp, &
      &  4.55482756302267_wp, -0.78868952630666_wp, -1.28323990176737_wp, &
      &  4.28156940306037_wp,  0.03577683956710_wp, -2.97170521373079_wp, &
      &  7.00778597005275_wp, -1.84411781203415_wp, -0.68543100502992_wp, &
      &  8.48597528929860_wp, -0.47536757886041_wp, -1.08772149918549_wp, &
      &  7.01628298357361_wp, -2.29184412385416_wp,  1.31572306630335_wp, &
      &  7.37330070501013_wp, -3.56845409822307_wp, -1.74837173269827_wp, &
      & -4.13936853197059_wp, -1.51532051421004_wp, -0.35550441221018_wp, &
      & -4.63333270492068_wp, -0.99244476172841_wp, -2.55585024524801_wp, &
      & -5.94841324012219_wp, -2.96851868511636_wp,  1.30462125610968_wp, &
      & -5.32289107512728_wp, -3.10036808085930_wp,  3.25670222239822_wp, &
      & -6.16808953834020_wp, -4.86054580466918_wp,  0.53056285012182_wp, &
      & -7.78374214491602_wp, -2.04921220648569_wp,  1.23336151366432_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine VAL_xah

subroutine VAL_xaj(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.94676816796255_wp,  1.64032063322157_wp,  0.68946258933075_wp, &
      &  0.74453628984122_wp, -0.64494348600364_wp, -0.82769438343822_wp, &
      & -1.60432722313136_wp, -2.19346375807200_wp, -0.09437896978801_wp, &
      & -1.47340378483393_wp, -4.32982338961723_wp,  0.79907243906515_wp, &
      &  3.17827830513127_wp, -2.19110013560537_wp, -0.68319460040886_wp, &
      &  5.28631324768353_wp, -0.87827797092283_wp, -2.14283438133501_wp, &
      &  4.00192248059422_wp, -2.74648869323590_wp,  2.02585886081752_wp, &
      &  1.95440953394313_wp,  1.56222426368521_wp,  2.29794420236049_wp, &
      &  0.46080847431842_wp, -0.00197106331626_wp, -2.76931011825575_wp, &
      &  2.74297899991048_wp, -3.99322262690535_wp, -1.58255710069702_wp, &
      &  4.78214965426612_wp, -0.61670732461371_wp, -4.12217939184135_wp, &
      &  7.02190935001069_wp, -1.98278623043789_wp, -2.06280419721417_wp, &
      &  5.68314710809320_wp,  0.97751724930690_wp, -1.33814896388556_wp, &
      &  2.46177143734685_wp, -3.54949176712023_wp,  3.12240295086472_wp, &
      &  5.55861036084050_wp, -4.09321204406879_wp,  2.01784238549341_wp, &
      &  4.71181863175987_wp, -1.05001429822093_wp,  2.96564916085728_wp, &
      & -3.78766478813798_wp, -0.95565393569553_wp, -0.57168124637690_wp, &
      & -3.65035586245999_wp,  0.80802132724782_wp, -1.30535504881992_wp, &
      & -6.22661624855872_wp, -2.02506062427493_wp,  0.04001553077594_wp, &
      & -7.45333705516149_wp, -2.01975563603356_wp, -1.61005163760224_wp, &
      & -5.92339551436037_wp, -3.95956763791655_wp,  0.65143181740436_wp, &
      & -7.14530859057892_wp, -0.97844268693911_wp,  1.55599981295444_wp, &
      & -0.28406342781207_wp,  3.80865857086425_wp,  0.11506008267347_wp, &
      & -1.73664042865140_wp,  3.97942054339036_wp, -1.69636041109198_wp, &
      &  0.24176774203491_wp,  6.02448714783408_wp,  1.82636762506261_wp, &
      &  1.01464144652835_wp,  7.55114786854609_wp,  0.68834086500188_wp, &
      &  1.55023034384685_wp,  5.58526612809118_wp,  3.34803481310261_wp, &
      & -1.53728774847075_wp,  6.67772159119001_wp,  2.62007257902923_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine VAL_xaj

subroutine VAL_xak(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "N", "H", "C", "H", "H", "H", "C", "O", "C", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.16550986507110_wp, -0.28862016669173_wp,  1.64799098017106_wp, &
      & -0.20927399213306_wp,  1.09031755814856_wp,  1.32086475634073_wp, &
      & -2.49435129612966_wp, -0.67224435018703_wp,  1.09058644278622_wp, &
      & -4.21914331577617_wp, -0.60467603939292_wp,  2.64159341541046_wp, &
      & -0.11166397284520_wp,  3.06128738090373_wp, -0.80348556777435_wp, &
      & -2.61630070704739_wp,  4.48352002927149_wp, -0.89056834756338_wp, &
      &  2.07799172675864_wp,  4.88227441175868_wp, -0.38896025954669_wp, &
      &  2.93394395995268_wp, -0.27416203672429_wp,  3.38393319578012_wp, &
      & -0.51831094849774_wp,  2.09347086830864_wp,  3.09413447148002_wp, &
      &  0.16787064381551_wp,  2.07984514952649_wp, -2.59144573426340_wp, &
      & -4.20008573714480_wp,  3.22459724586796_wp, -1.26368487858754_wp, &
      & -2.56892802806470_wp,  5.91094294307183_wp, -2.37276044034758_wp, &
      & -2.97535346038192_wp,  5.43613910819199_wp,  0.90294621689096_wp, &
      &  3.89135311861508_wp,  3.91094217123995_wp, -0.38030857642833_wp, &
      &  2.12447353537213_wp,  6.29774763576288_wp, -1.88285451329413_wp, &
      &  1.86453817524921_wp,  5.87102455464120_wp,  1.41022607503894_wp, &
      & -2.48459240511383_wp, -2.19492515154250_wp, -0.95480223671483_wp, &
      & -0.88329112919351_wp, -2.26120495381005_wp, -2.00519237821100_wp, &
      & -4.51069362129654_wp, -3.98738195919085_wp, -1.34901223121118_wp, &
      & -6.31619824091419_wp, -3.06071873051924_wp, -1.03412738230562_wp, &
      & -4.42296755466235_wp, -4.67825155814166_wp, -3.27930735016650_wp, &
      & -4.38564531849930_wp, -5.58494625392956_wp, -0.05391016262128_wp, &
      &  3.42855958461703_wp, -1.57172448742632_wp, -0.16941583723595_wp, &
      &  2.64256565170380_wp, -1.82147739327692_wp, -2.34675137361443_wp, &
      &  5.92030724907409_wp, -2.70176258818258_wp,  0.63568934701270_wp, &
      &  7.39634039190781_wp, -1.93210521264107_wp, -0.56942959900230_wp, &
      &  5.85046688742009_wp, -4.73014335233649_wp,  0.31435248538312_wp, &
      &  6.39179007445964_wp, -2.33930360931913_wp,  2.60172285978818_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine VAL_xak

end module mstore_amino20x4
