use std::collections::BTreeMap;

pub enum CallingConvention {
    Stdcall(usize),
    Cdecl,
}

pub fn default_metadata() -> Vec<metadata::File> {
    vec![
        metadata::File::new(
            std::include_bytes!("../../../libs/bindgen/default/Windows.winmd").to_vec(),
        )
        .expect("invalid winmd"),
        metadata::File::new(
            std::include_bytes!("../../../libs/bindgen/default/Windows.Win32.winmd").to_vec(),
        )
        .expect("invalid winmd"),
        metadata::File::new(
            std::include_bytes!("../../../libs/bindgen/default/Windows.Wdk.winmd").to_vec(),
        )
        .expect("invalid winmd"),
    ]
}

/// Returns the libraries and their function and stack sizes used by the gnu and msvc tools to build the umbrella libs.
pub fn libraries() -> BTreeMap<String, BTreeMap<String, CallingConvention>> {
    let mut libraries = BTreeMap::<String, BTreeMap<String, CallingConvention>>::new();

    let files = default_metadata();
    let reader = metadata::Reader::new(files);
    combine_libraries(reader, &mut libraries);

    // StgConvertPropertyToVariant was removed https://github.com/microsoft/win32metadata/issues/1566
    // It is very unlikely that anybody is calling that function, but this just ensures that the libs
    // are stable and we don't break the `windows-targets` crate compatibility until the next major
    // release of that crate.

    let compat =
        vec![
            metadata::File::new(std::include_bytes!("../Windows.Win32.49.winmd").to_vec())
                .expect("invalid winmd"),
        ];

    let reader = metadata::Reader::new(compat);
    combine_libraries(reader, &mut libraries);

    libraries
}

fn combine_libraries(
    reader: &metadata::Reader,
    libraries: &mut BTreeMap<String, BTreeMap<String, CallingConvention>>,
) {
    for item in reader.items() {
        let metadata::Item::Fn(method, _) = item else {
            continue;
        };

        let library = method.module_name();
        let impl_map = method.impl_map().expect("ImplMap not found");
        let flags = impl_map.flags();
        let name = impl_map.import_name().to_string();

        // TODO: don't include these in metadata to begin with
        if name.starts_with('#') || library == "forceinline" {
            continue;
        }

        if flags.contains(metadata::PInvokeAttributes::CallConvPlatformapi) {
            let params = method.signature(&[]).size();
            libraries
                .entry(library)
                .or_default()
                .insert(name, CallingConvention::Stdcall(params));
        } else if flags.contains(metadata::PInvokeAttributes::CallConvCdecl) {
            libraries
                .entry(library)
                .or_default()
                .insert(name, CallingConvention::Cdecl);
        } else {
            unimplemented!();
        }
    }
}
