///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoQueryCache.cc
// -----------------
// Cego query cache implementation
//      
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2019 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoQueryCache
// 
// Description: Query Cache Management
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#include <lfcbase/ThreadLock.h>
#include "CegoQueryCache.h"
#include "CegoXMLdef.h"

#include <stdlib.h>
#include <string.h>

static ThreadLock cacheLock("QUERYCACHE");
extern bool __lockStatOn;

CegoQueryCache::QueryCacheEntry::QueryCacheEntry()
{
    _pCacheArray = 0;
    _numHit=0;
    _numUsed=0;
}

CegoQueryCache::QueryCacheEntry::QueryCacheEntry(const Chain& queryId)
{
    _queryId = queryId;
    _pCacheArray = 0;
    _numHit=0;
    _numUsed=0;
}

CegoQueryCache::QueryCacheEntry::QueryCacheEntry(const Chain& queryId, const SetT<CegoObject>& objList, ListT< ListT<CegoFieldValue> >* pCacheList, const ListT<CegoField>& cacheSchema)
{
    _queryId = queryId;    
    _objList = objList;

    // the cache entry has to be converted from list to array to be thread safe
    
    _pCacheArray = new CegoFieldValue**[ pCacheList->Size() ];
    ListT<CegoFieldValue>* pFVL = pCacheList->First();
    _numRow = 0;
    _numCol = 0;
    while ( pFVL )
    {
	CegoFieldValue** pCFVL = new CegoFieldValue*[pFVL->Size()];
	CegoFieldValue* pFV = pFVL->First();
	int i = 0;
	while ( pFV )
	{
	    CegoFieldValue* pCFV = new CegoFieldValue(pFV->getLocalCopy());
	    pCFVL[i] = pCFV;
	    pFV = pFVL->Next();
	    i++;
	}
	_numCol = i;
       
	_pCacheArray[_numRow] = pCFVL;
	_numRow++;
	pFVL = pCacheList->Next();	
    }
    
    _cacheSchema = cacheSchema;
    _numHit=1;
    _numUsed=0;
}

CegoQueryCache::QueryCacheEntry::~QueryCacheEntry()
{
}

bool CegoQueryCache::QueryCacheEntry::cleanCache()
{
    if ( _numUsed > 0 )
	return false;
    
    for ( int i=0; i<_numRow; i++ )
    {
	for ( int j=0; j<_numCol; j++ )
	    delete _pCacheArray[i][j];
	delete _pCacheArray[i];
    }
    delete _pCacheArray;
    _pCacheArray = 0;
    return true;
}

int CegoQueryCache::QueryCacheEntry::getSize() const
{    
    int s =  _queryId.length();
    CegoObject *pO = _objList.First();
    while ( pO )
    {
	s += pO->size();
	pO = _objList.Next();
    }

    CegoField *pF = _cacheSchema.First();
    while ( pF )
    {
	s += pF->size();
	pF = _cacheSchema.Next();
    }

    for ( int i=0; i<_numRow; i++ )
    {
	for ( int j=0; j<_numCol; j++ )
	{
	    CegoFieldValue *pF = _pCacheArray[i][j];
	    s += pF->size();
	}
    }

    return s;
}

int CegoQueryCache::QueryCacheEntry::getHashPos(int hashSize) const
{
    return _queryId.getHashPos(hashSize);    
}

const Chain CegoQueryCache::QueryCacheEntry::getQueryId() const
{
    return _queryId;
}

int CegoQueryCache::QueryCacheEntry::getNumRows() const
{
    return _numRow;
}

const SetT<CegoObject>& CegoQueryCache::QueryCacheEntry::getObjectList() const
{
    return _objList; 
}

CegoFieldValue*** CegoQueryCache::QueryCacheEntry::claimCache()
{
    _numUsed++;
    return _pCacheArray;
}

void CegoQueryCache::QueryCacheEntry::releaseCache()
{
    _numUsed--;
}

const ListT<CegoField>& CegoQueryCache::QueryCacheEntry::getSchema() const
{
    return _cacheSchema;
}

unsigned long CegoQueryCache::QueryCacheEntry::getHit() const
{
    return _numHit;
}

void CegoQueryCache::QueryCacheEntry::incHit()
{
    _numHit++;
}

CegoQueryCache::QueryCacheEntry& CegoQueryCache::QueryCacheEntry::operator = ( const CegoQueryCache::QueryCacheEntry& qce)
{
    _queryId = qce._queryId;
    _objList = qce._objList;
    _pCacheArray = qce._pCacheArray;
    _cacheSchema = qce._cacheSchema;
    _numHit = qce._numHit;
    _numRow = qce._numRow;
    _numCol = qce._numCol;
    return (*this);
}

bool CegoQueryCache::QueryCacheEntry::operator == ( const CegoQueryCache::QueryCacheEntry& qce)
{
    if ( _queryId == qce._queryId )
	return true;
    return false;
}


/////////////////////////
// Query Cache Methods //
/////////////////////////

#define HASHRANGE 20

CegoQueryCache::CegoQueryCache(int maxEntry, int maxSize)
{
    cacheLock.init(LCKMNG_LOCKWAITDELAY, __lockStatOn);  
    _maxEntry = maxEntry;
    _maxSize = maxSize;

    _pQueryCache = new HashT<QueryCacheEntry>(_maxEntry, HASHRANGE);
    _usedSize = 0;
}

CegoQueryCache::~CegoQueryCache()
{
    clean();
    delete _pQueryCache;
}

void CegoQueryCache::PR()
{
    cacheLock.readLock(DBM_LOCKTIMEOUT);
}

void CegoQueryCache::PW()
{
    cacheLock.writeLock(DBM_LOCKTIMEOUT);
}

void CegoQueryCache::V()
{
    cacheLock.unlock();
}

Element* CegoQueryCache::getCacheInfo()
{
    Element* pCacheInfo = new Element(XML_CACHEINFO_ELEMENT);

    Element *pN = new Element(XML_CACHE_ELEMENT);
    pN->setAttribute(XML_ATTRNAME_ATTR, Chain("MaxEntry"));
    pN->setAttribute(XML_VALUE_ATTR, Chain(_maxEntry));
    pCacheInfo->addContent(pN);

    pN = new Element(XML_CACHE_ELEMENT);
    pN->setAttribute(XML_ATTRNAME_ATTR, Chain("MaxSize"));
    pN->setAttribute(XML_VALUE_ATTR, Chain(_maxSize));
    pCacheInfo->addContent(pN);

    pN = new Element(XML_CACHE_ELEMENT);
    pN->setAttribute(XML_ATTRNAME_ATTR, Chain("UsedSize"));
    pN->setAttribute(XML_VALUE_ATTR, Chain(_usedSize));
    pCacheInfo->addContent(pN);

    return pCacheInfo;
}

Element* CegoQueryCache::getCacheList()
{
    Element* pCacheInfo = new Element(XML_CACHEINFO_ELEMENT);
    PR();

    QueryCacheEntry *pCE = _pQueryCache->First();	
    while ( pCE )
    {
	Element *pN = new Element(XML_CACHE_ELEMENT);
	pN->setAttribute(XML_ID_ATTR, pCE->getQueryId());
	pN->setAttribute(XML_NUMROWS_ATTR, Chain(pCE->getNumRows()));
	pN->setAttribute(XML_NUMHITS_ATTR, Chain(pCE->getHit()));
	pN->setAttribute(XML_SIZE_ATTR, Chain(pCE->getSize()));
	pCacheInfo->addContent(pN);
	pCE = _pQueryCache->Next();
    }
    V();
    return pCacheInfo;
}

int CegoQueryCache::getMaxEntry() const
{
    return _maxEntry;
}

void CegoQueryCache::setMaxEntry(int maxEntry)
{
    _maxEntry = maxEntry;
}

int CegoQueryCache::getMaxSize() const
{
    return _maxSize;
}

void CegoQueryCache::setMaxSize(int maxSize)
{
    _maxSize = maxSize;
}

int CegoQueryCache::getNumQueryCache() const
{
    return _pQueryCache->numEntry();
}

int CegoQueryCache::getUsedSize() const
{
    return _usedSize;
}

void CegoQueryCache::invalidate(const CegoObject& obj)
{
    bool isClean=false;

    while ( isClean == false )
    {
	PW();

	// reset to clean
	isClean=true;
	
	QueryCacheEntry *pQCE = _pQueryCache->First();
	while ( pQCE )
	{
	    if ( pQCE->getObjectList().Find(obj) )
	    {
		int s = pQCE->getSize();
		
		if ( pQCE->cleanCache() )
		{
		    // cout << "Cache invalition, decreased size from " << _usedSize << " to " << _usedSize - s  << "(" << s << ")" << endl; 
		    _usedSize = _usedSize - s;
		    _pQueryCache->Remove(*pQCE);
		    pQCE = _pQueryCache->First();
		}
		else
		{
		    // was not able to cleanup, cache still in use
		    // cout << "Cache still in use .." << endl;
		    isClean=false;
		    // we skip this entry and go to next
		    pQCE = _pQueryCache->Next();
		}
	    }
	    else
	    {
		// cache entry not relevant for object, go to next
		pQCE = _pQueryCache->Next();
	    }
	}
    
	V();
    }
}

CegoFieldValue*** CegoQueryCache::claimEntry(const Chain& queryId, ListT<CegoField>& cacheSchema, int& cacheRows)
{
    CegoFieldValue*** pCE = 0;

    // cout << _pQueryCache->numEntry() << " entries available" << endl;
    // cout << "Claiming entry " << queryId << endl;
    
    PR();
    QueryCacheEntry *pQCE = _pQueryCache->Find( QueryCacheEntry(queryId) );
    if ( pQCE )
    {
	pQCE->incHit();
	pCE = pQCE->claimCache();
	cacheSchema = pQCE->getSchema();
	cacheRows = pQCE->getNumRows();
    }
    V();
    
    return pCE;
}

void CegoQueryCache::releaseEntry(const Chain& queryId)
{
    PR();
    QueryCacheEntry *pQCE = _pQueryCache->Find( QueryCacheEntry(queryId) );
    if ( pQCE )
    {
	pQCE->releaseCache();
    }
    V();
}

void CegoQueryCache::addEntry(const Chain& queryId, const SetT<CegoObject>& objectList, ListT< ListT<CegoFieldValue> >* pCacheList, const ListT<CegoField>& cacheSchema )
{    
    bool isFreed = false;
    
    while ( isFreed == false )
    {	
	PW();

	// at default, we assume, we don't have to free an entry
	isFreed=true;
	
	QueryCacheEntry *pQCE = _pQueryCache->Find( QueryCacheEntry(queryId));
	if ( pQCE )
	{
	    // cache entry already exists, we can return 
	    V();
	    return;
	}
	if ( _pQueryCache->numEntry() > _maxEntry )
	{
	    QueryCacheEntry *pRE = 0;
	    
	    unsigned long minHit = 0;
	    QueryCacheEntry *pQCE = _pQueryCache->First();
	    while ( pQCE )
	    {
		if ( minHit == 0 || pQCE->getHit() < minHit )
		{
		    pRE = pQCE;
		    minHit = pQCE->getHit();
		}
		
		pQCE = _pQueryCache->Next();
	    }
	    if ( pRE )
	    {
		int s = pRE->getSize();
		if ( pRE->cleanCache() )
		{
		    _usedSize = _usedSize - s;
		    _pQueryCache->Remove(*pRE);
		}
		else
		{
		    // cache entry is still occupied, we have to try again
		    isFreed=false;
		}
	    }
	}

	// if we could free an entry, we can insert new one now
	if ( isFreed )
	{
	    QueryCacheEntry qce(queryId, objectList, pCacheList, cacheSchema);

	    _pQueryCache->Insert(qce);
	    
	    _usedSize = _usedSize + qce.getSize(); 
	}
	
	V();
    }
}

void CegoQueryCache::clean()
{
    bool isClean=false;
    
    while ( isClean == false )
    {
	isClean = true;
	
	PW();

	QueryCacheEntry *pQCE = _pQueryCache->First();
	while ( pQCE )
	{	    	    
	    // perhaps we have to wait for cache users,
	    // so we repeat until cache is clean
	    if ( pQCE->cleanCache() )
	    {
		_pQueryCache->Remove(*pQCE);
		pQCE = _pQueryCache->First();		
	    }
	    else
	    {		
		isClean=false;
		pQCE = _pQueryCache->Next();
	    }	    
	}
	
	V();	
    }

    _usedSize = 0;
}
