///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoFunction.cc
// ---------------
// Cego internal sql function implementation
//     
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2019 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoFunction
// 
// Description: All internal SQL database functions
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// LFC INCLUDES
#include <lfcbase/Datetime.h>
#include <lfcbase/Tokenizer.h>

// CEGO INCLUDES
#include "CegoFunction.h"
#include "CegoExpr.h"
#include "CegoDistManager.h"
#include "CegoXMLdef.h"
#include "CegoDatabaseFormater.h"
#include "CegoProcedure.h"
#include "CegoAction.h"
#include "CegoTypeConverter.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

// improved bsd random functions still not supported for mingw
#ifdef HAVE_MINGW
#define random rand
#endif

CegoFunction::CegoFunction(CegoFunction::FunctionType type)
{
    _pTabMng = 0;
    _type = type;
}

CegoFunction::CegoFunction(CegoFunction::FunctionType type, ListT<CegoExpr*>& exprList)
{
    _pTabMng = 0;
    _exprList = exprList;
    _type = type;
}

CegoFunction::CegoFunction(char* buf, CegoDistManager *pTabMng, int tabSetId)
{
    _pTabMng = pTabMng;
    _tabSetId = tabSetId;
    decode(buf, pTabMng, tabSetId);
}

CegoFunction::CegoFunction(CegoDistManager* pTabMng, int tabSetId, CegoFunction::FunctionType type)
{
    _pTabMng = pTabMng;
    _tabSetId = tabSetId;
    _type = type;
}

CegoFunction::CegoFunction(CegoDistManager* pTabMng, int tabSetId, const Chain& funcName, ListT<CegoExpr*>& exprList)
{
    _pTabMng = pTabMng;
    _exprList = exprList;
    _funcName = funcName;
    _tabSetId = tabSetId;
    _type = CegoFunction::USERDEFINED;
}

CegoFunction::CegoFunction(Element* pFunctionElement, CegoDistManager *pGTM)
{
    fromElement(pFunctionElement, pGTM);
}

CegoFunction::~CegoFunction()
{
}

void CegoFunction::cleanUp()
{
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->cleanUp();
	pExpr = _exprList.Next();
    }
}

void CegoFunction::setTabSetId(int tabSetId)
{
    _tabSetId = tabSetId;
    
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->setTabSetId(tabSetId);
	pExpr = _exprList.Next();
    }
}

int CegoFunction::getTabSetId() const
{
    return _tabSetId;
}

Chain CegoFunction::getName() const
{
    Chain funcString;

    switch (_type)
    {

    case CegoFunction::INT2ASC:
    {
	funcString = Chain("int2asc");
	break;	
    }
    case CegoFunction::ASC2INT:
    {
	funcString = Chain("asc2int");
	break;	
    }
    case CegoFunction::TRIM:
    {
	funcString = Chain("trim");
	break;	
    }
    case CegoFunction::RTRIM:
    {
	funcString = Chain("rtrim");
	break;	
    }
    case CegoFunction::LTRIM:
    {
	funcString = Chain("ltrim");
	break;	
    }
    case CegoFunction::ROUND:
    {
	funcString = Chain("round");
	break;	
    }
    case CegoFunction::TRUNC:
    {
	funcString = Chain("trunc");
	break;	
    }
    case CegoFunction::DATE2STR:
    {
	funcString = Chain("date2str");
	break;	
    }
    case CegoFunction::DATE2LONG:
    {
	funcString = Chain("date2long");
	break;	
    }
    case CegoFunction::LONG2DATE:
    {
	funcString = Chain("long2date");
	break;	
    }
    case CegoFunction::NEWDATE:
    {
	funcString = Chain("newdate");
	break;	
    }
    case CegoFunction::LOWER:
    {
	funcString = Chain("lower");
	break;	
    }
    case CegoFunction::UPPER:
    {
	funcString = Chain("upper");
	break;	
    }
    case CegoFunction::LEFT:
    {
	funcString = Chain("left");
	break;	
    }
    case CegoFunction::RIGHT:
    {
	funcString = Chain("right");
	break;	
    }
    case CegoFunction::SUBSTR:
    {
	funcString = Chain("substr");
	break;	
    }
    case CegoFunction::GETPOS:
    {
	funcString = Chain("getpos");
	break;		
    }
    case CegoFunction::STR2INT:
    {
	funcString = Chain("str2int");
	break;		
    }
    case CegoFunction::STR2LONG:
    {
	funcString = Chain("str2long");
	break;		
    }
    case CegoFunction::STR2DATE:
    {
	funcString = Chain("str2date");
	break;		
    }
    case CegoFunction::RANDSTR:
    {
	funcString = Chain("randstr");
	break;		
    }
    case CegoFunction::RANDINT:
    {
	funcString = Chain("randint");
	break;		
    }
    case CegoFunction::MOD:
    {
	funcString = Chain("mod");
	break;		
    }
    case CegoFunction::LMOD:
    {
	funcString = Chain("lmod");
	break;		
    }
    case CegoFunction::DIV:
    {
	funcString = Chain("div");
	break;		
    }
    case CegoFunction::LDIV:
    {
	funcString = Chain("ldiv");
	break;		
    }
    case CegoFunction::POWER:
    {
	funcString = Chain("power");
	break;		
    }
    case CegoFunction::BITAND:
    {
	funcString = Chain("bitand");
	break;		
    }
    case CegoFunction::BITOR:
    {
	funcString = Chain("bitor");
	break;		
    }
    case CegoFunction::BITXOR:
    {
	funcString = Chain("bitxor");
	break;		
    }
    case CegoFunction::BLOBSIZE:
    {
	funcString = Chain("blobsize");
	break;		
    }
    case CegoFunction::BLOBREF:
    {
	funcString = Chain("blobref");
	break;		
    }
    case CegoFunction::CLOBSIZE:
    {
	funcString = Chain("clobsize");
	break;		
    }
    case CegoFunction::CLOBREF:
    {
	funcString = Chain("clobref");
	break;		
    }
    case CegoFunction::CLOB2STR:
    {
	funcString = Chain("clob2str");
	break;		
    }
    case CegoFunction::REPLACE:
    {
	funcString = Chain("replace");
	break;	
    }
    case CegoFunction::LENGTH:
    {
	funcString = Chain("length");
	break;	
    }
    case CegoFunction::USERDEFINED:
    {
	funcString = _funcName;
	break;	
    }
    case CegoFunction::NEXTCOUNT:
    {
	funcString = Chain("nextcount");
	break;	
    }
    case CegoFunction::SETCOUNT:
    {
	funcString = Chain("setcount");
	break;	
    }
    }
    return funcString;
}

CegoDataType CegoFunction::getReturnType() const
{
    switch ( _type )
    {
    case INT2ASC:
    case TRIM:
    case LTRIM:
    case RTRIM:
    case DATE2STR:
    case LOWER:
    case UPPER:
    case LEFT:
    case RIGHT:
    case SUBSTR:
    case REPLACE:
    case RANDSTR:
    case CLOB2STR:
	return VARCHAR_TYPE;
    case ASC2INT:
    case ROUND:
    case GETPOS:
    case LENGTH:
    case TRUNC:
    case STR2INT:
    case RANDINT:
    case MOD:
    case DIV:
    case POWER:
    case BITAND:
    case BITOR:
    case BITXOR:
	return INT_TYPE;
    case LONG2DATE:
    case NEWDATE:
    case STR2DATE:
	return DATETIME_TYPE;
    case STR2LONG:
    case BLOBSIZE:
    case BLOBREF:
    case CLOBSIZE:
    case CLOBREF:
    case NEXTCOUNT:
    case SETCOUNT:
    case LMOD:
    case LDIV:
    case DATE2LONG:
	return LONG_TYPE;
    case USERDEFINED:
    {
	if ( _pTabMng == 0 )
	    throw Exception(EXLOC, "No valid table manager set up");
	
	CegoDataType returnType;
		
	try 
	{	    
	    _pTabMng->getDBMng()->useObject(_tabSetId, _funcName, CegoObject::PROCEDURE, CegoDatabaseManager::SHARED, _pTabMng->getThreadId());    
	    CegoProcedure* pProc = _pTabMng->getProcedure(_tabSetId, _funcName);	    
	    returnType = pProc->getReturnType();	  	    
	}
	catch ( Exception e )
	{
	    _pTabMng->getDBMng()->unuseObject(_tabSetId, _funcName, CegoObject::PROCEDURE);
	    throw e;
	}

	_pTabMng->getDBMng()->unuseObject(_tabSetId, _funcName, CegoObject::PROCEDURE);

	return returnType;	
    }
    }
}

int CegoFunction::getReturnTypeLen(const ListT<CegoField>& fl) const
{
    switch ( _type )
    {
    case INT2ASC:
    {
	return 1;
    }
    case ASC2INT:
    {
	return sizeof(int);
    }
    case RANDSTR:
    {
	CegoExpr** pExpr = _exprList.First();
	if ( pExpr )
	{
	    if ( (*pExpr)->checkConst() )
		return (*pExpr)->evalFieldValue().asInteger();
	    return RETVAL_LEN;
	}
	return 0;
    }		
    case DATE2STR:
    {
	return 20;
    }
    case LEFT:
    case RIGHT:
    {
	CegoExpr** pSourceExpr = _exprList.First();
	CegoExpr** pLengthExpr = _exprList.Next();
	if ( pLengthExpr )
	{
	    if ( (*pLengthExpr)->checkConst() )	    
		return (*pLengthExpr)->evalFieldValue().asInteger();
	    else
		return (*pSourceExpr)->evalField(fl).getLength();	    
	}
	return 0;
    }
    case SUBSTR:
    {
	CegoExpr** pSourceExpr = _exprList.First();
	CegoExpr** pOffsetExpr = _exprList.Next(); // not used
	CegoExpr** pLengthExpr = _exprList.Next();
	if ( pLengthExpr )
	{	    
	    if ( (*pLengthExpr)->checkConst() )	    
		return (*pLengthExpr)->evalFieldValue().asInteger();
	    else
		return (*pSourceExpr)->evalField(fl).getLength();
	}
	
	return 0;
    }
    case CLOB2STR:
    {
	return CLOB_FORMAT_LEN;
    }
    case LTRIM:
    case RTRIM:
    case TRIM:
    case LOWER:
    case UPPER:
    {
	CegoExpr** pExpr = _exprList.First();
	if ( pExpr )
	{
	    return (*pExpr)->evalField(fl).getLength();
	}
	return 0;
    }
    case REPLACE:
    {
	CegoExpr** pExpr = _exprList.First();
	return (*pExpr)->evalField(fl).getLength();
    }
    case ROUND:
    case GETPOS:
    case LENGTH:
    case TRUNC:
    case STR2INT:
    case RANDINT:
    case MOD:
    case DIV:
    case POWER:
    case BITAND:
    case BITOR:
    case BITXOR:
	return sizeof(int);
    case STR2DATE:
    case NEWDATE:
	return sizeof(int);
    case STR2LONG:
    case BLOBSIZE:
    case BLOBREF:
    case CLOBSIZE:
    case CLOBREF:
    case NEXTCOUNT:
    case SETCOUNT:
    case LMOD:
    case LDIV:
    case DATE2LONG:
    case LONG2DATE:
	return sizeof(long long);
    case USERDEFINED:
    {
	if ( _pTabMng == 0 )
	    throw Exception(EXLOC, "No valid table manager set up");
	
	int returnTypeLen;
		
	try 
	{	    
	    _pTabMng->getDBMng()->useObject(_tabSetId, _funcName, CegoObject::PROCEDURE, CegoDatabaseManager::SHARED, _pTabMng->getThreadId());    
	    CegoProcedure* pProc = _pTabMng->getProcedure(_tabSetId, _funcName);	    
	    returnTypeLen = pProc->getReturnTypeLen();	  	    
	}
	catch ( Exception e )
	{
	    _pTabMng->getDBMng()->unuseObject(_tabSetId, _funcName, CegoObject::PROCEDURE); 
	    throw e;
	}

	_pTabMng->getDBMng()->unuseObject(_tabSetId, _funcName, CegoObject::PROCEDURE);

	return returnTypeLen;	
    }
    }
}
    
const CegoFunction::FunctionType CegoFunction::getType() const
{
    return _type;
}

void CegoFunction::setExprList(ListT<CegoExpr*>& exprList)
{    
    if ( _type == INT2ASC && exprList.Size() != 1)
	throw Exception(EXLOC, Chain("Invalid parameter count for int2asc function"));
    if ( _type == ASC2INT && exprList.Size() != 1)
	throw Exception(EXLOC, Chain("Invalid parameter count for asc2int function"));
    if ( _type == TRIM && exprList.Size() != 2)
	throw Exception(EXLOC, Chain("Invalid parameter count for trim function"));
    if ( _type == LTRIM && exprList.Size() != 2)
	throw Exception(EXLOC, Chain("Invalid parameter count for ltrim function"));
    if ( _type == RTRIM && exprList.Size() != 2)
	throw Exception(EXLOC, Chain("Invalid parameter count for rtrim function"));
    if ( _type == ROUND && ( exprList.Size() < 1 || exprList.Size() > 2) )
	throw Exception(EXLOC, Chain("Invalid parameter count for round function"));
    if ( _type == DATE2STR && exprList.Size() != 2)
	throw Exception(EXLOC, Chain("Invalid parameter count for date2str function"));
    if ( _type == DATE2LONG && exprList.Size() != 1)
	throw Exception(EXLOC, Chain("Invalid parameter count for date2long function"));
    if ( _type == LONG2DATE && exprList.Size() != 1)
	throw Exception(EXLOC, Chain("Invalid parameter count for long2date function"));
    if ( _type == NEWDATE && exprList.Size() > 6 )
	throw Exception(EXLOC, Chain("Invalid parameter count for newdate function"));
    if ( _type == LOWER && exprList.Size() != 1)
	throw Exception(EXLOC, Chain("Invalid parameter count for lower function"));
    if ( _type == UPPER && exprList.Size() != 1)
	throw Exception(EXLOC, Chain("Invalid parameter count for upper function"));
    if ( _type == LEFT && exprList.Size() != 2)
	throw Exception(EXLOC, Chain("Invalid parameter count for left function"));
    if ( _type == RIGHT && exprList.Size() != 2)
	throw Exception(EXLOC, Chain("Invalid parameter count for right function"));
    if ( _type == GETPOS && ( exprList.Size() < 2 || exprList.Size() > 4 ) )
	throw Exception(EXLOC, Chain("Invalid parameter count for getpos function"));
    if ( _type == SUBSTR && ( exprList.Size() < 2 || exprList.Size() > 3 ) )
	throw Exception(EXLOC, Chain("Invalid parameter count for substr function"));
    if ( _type == REPLACE && exprList.Size() != 3)
	throw Exception(EXLOC, Chain("Invalid parameter count for replace function"));
    if ( _type == LENGTH && exprList.Size() != 1)
	throw Exception(EXLOC, Chain("Invalid parameter count for length function"));
    if ( _type == TRUNC && exprList.Size() != 1)
	throw Exception(EXLOC, Chain("Invalid parameter count for trunc function"));
    if ( _type == STR2INT && exprList.Size() != 1)
	throw Exception(EXLOC, Chain("Invalid parameter count for str2int function"));
    if ( _type == STR2LONG && exprList.Size() != 1)
	throw Exception(EXLOC, "Invalid parameter count for str2long function");
    if ( _type == STR2DATE && exprList.Size() != 2)
	throw Exception(EXLOC, "Invalid parameter count for str2date function");
    if ( _type == RANDSTR && exprList.Size() != 1)
	throw Exception(EXLOC, "Invalid parameter count for randstr function");
    if ( _type == RANDINT && exprList.Size() != 1)
	throw Exception(EXLOC, "Invalid parameter count for randint function");
    if ( _type == MOD && exprList.Size() != 2)
	throw Exception(EXLOC, "Invalid parameter count for mod function");
    if ( _type == DIV && exprList.Size() != 2)
	throw Exception(EXLOC, "Invalid parameter count for div function");
    if ( _type == LMOD && exprList.Size() != 2)
	throw Exception(EXLOC, "Invalid parameter count for lmod function");
    if ( _type == LDIV && exprList.Size() != 2)
	throw Exception(EXLOC, "Invalid parameter count for ldiv function");
    if ( _type == POWER && exprList.Size() != 2)
	throw Exception(EXLOC, "Invalid parameter count for power function");
    if ( _type == BITAND && exprList.Size() != 2)
	throw Exception(EXLOC, "Invalid parameter count for bitand function");
    if ( _type == BITOR && exprList.Size() != 2)
	throw Exception(EXLOC, "Invalid parameter count for bitor function");
    if ( _type == BITXOR && exprList.Size() != 2)
	throw Exception(EXLOC, "Invalid parameter count for bitxor function");
    if ( _type == BLOBSIZE && exprList.Size() != 1)
	throw Exception(EXLOC, "Invalid parameter count for blobsize function");
    if ( _type == BLOBREF && exprList.Size() != 1)
	throw Exception(EXLOC, "Invalid parameter count for blobref function");
    if ( _type == CLOBSIZE && exprList.Size() != 1)
	throw Exception(EXLOC, "Invalid parameter count for clobsize function");
    if ( _type == CLOBREF && exprList.Size() != 1)
	throw Exception(EXLOC, "Invalid parameter count for clobref function");
    if ( _type == CLOB2STR && exprList.Size() != 1)
	throw Exception(EXLOC, "Invalid parameter count for clob2str function");

    _exprList = exprList;
}

ListT<CegoExpr*>& CegoFunction::getExprList()
{
    return _exprList;
}

void CegoFunction::getFieldList(ListT<CegoField>& fl) const
{
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->getFieldList(fl);
	pExpr = _exprList.Next();
    }
}

void CegoFunction::setFieldListArray(ListT<CegoField> **pFLA)
{
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->setFieldListArray(pFLA);
	pExpr = _exprList.Next();
    }
}

void CegoFunction::setBlock(CegoProcBlock *pBlock)
{
    _pBlock = pBlock;
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->setBlock(pBlock);	    
	pExpr = _exprList.Next();
    }
}

void CegoFunction::clearAttrCache()
{
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->clearAttrCache();	    
	pExpr = _exprList.Next();
    }
}

void CegoFunction::setCounterId(const Chain& counterId)
{
    _counterId = counterId;
}

void CegoFunction::setCounterExpr(CegoExpr *pExpr)
{
    _exprList.Insert(pExpr);
}

ListT<CegoAttrDesc*> CegoFunction::getAttrRefList() const
{
    ListT<CegoAttrDesc*> attrList;
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	attrList += (*pExpr)->getAttrRefList();
	pExpr = _exprList.Next();
    }
    return attrList;
}

int CegoFunction::evalReferences(CegoContentObject *pCO, const ListT<CegoField>& fl)
{
    int refCount = 0;

    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	refCount += (*pExpr)->evalReferences(pCO, fl);
	pExpr = _exprList.Next();
    }
    return refCount;
}

ListT<CegoAggregation*> CegoFunction::getAggregationList()
{
    ListT<CegoAggregation*> aggList;	

    CegoExpr **pExpr = _exprList.First();
    while ( pExpr ) 
    {
	aggList = aggList + (*pExpr)->getAggregationList();		    
	pExpr = _exprList.Next();

    }
    return aggList;
}

CegoFieldValue CegoFunction::evalFieldValue() const
{
    switch ( _type ) 
    {
    case CegoFunction::INT2ASC:
    {
	CegoExpr** pExpr1 = _exprList.First();
	
	CegoFieldValue fv1;
	
	fv1 = (*pExpr1)->evalFieldValue();

	if ( fv1.getType() == NULL_TYPE )
	    return CegoFieldValue();

	if ( fv1.castTo(INT_TYPE) == false )
	{
	    Chain msg = Chain("Invalid type for ascii conversion"); 
	    throw Exception(EXLOC, msg);	    
	}

	int val = 0; 
	if ( fv1.getValue() != 0 )
	    memcpy(&val, fv1.getValue(), sizeof(int));

	if ( val > 127 || val < 0 )
	{
	    Chain msg = Chain("Integer value out of range for ascii conversion"); 
	    throw Exception(EXLOC, msg);	    	    
	}
	char c = (char)val;
	
	CegoFieldValue fv = CegoFieldValue(VARCHAR_TYPE, Chain(c));	
		
	return fv;
    }
    case CegoFunction::ASC2INT:
    {
	CegoExpr** pExpr1 = _exprList.First();
	
	CegoFieldValue fv1;
	
	fv1 = (*pExpr1)->evalFieldValue();
	
	if ( fv1.getType() == NULL_TYPE )
	    return CegoFieldValue();

	if ( fv1.castTo(VARCHAR_TYPE) == false )
	{
	    Chain msg = Chain("Invalid type for ascii conversion"); 
	    throw Exception(EXLOC, msg);	    
	}

	if ( fv1.getLength() != 2 )
	{
	    Chain msg = Chain("Invalid length for ascii conversion"); 
	    throw Exception(EXLOC, msg);	    
	}

	char c;
	memcpy(&c, fv1.getValue(), sizeof(char));
	int i = c;

	CegoFieldValue fv = CegoFieldValue(INT_TYPE, &i, sizeof(int), false);	
		
	return fv;      	
    }
    case CegoFunction::ROUND:
    {
	
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	int precision = 0;

	if ( pExpr2 != 0 )
	{
	    CegoFieldValue pv = (*pExpr2)->evalFieldValue();
	    
	    if ( pv.getType() != INT_TYPE )
	    {
		Chain msg = Chain("Invalid precision in function round"); 
		throw Exception(EXLOC, msg);
	    }
	    if ( pv.getValue() != 0 )
		memcpy(&precision, pv.getValue(), sizeof(int));
	    // precision = *((int*)pv.getValue());
	}
	
	CegoFieldValue fv;
	
	fv = (*pExpr1)->evalFieldValue();
	
	Tokenizer tok(fv.valAsChain(), Chain("."));

	Chain lpart;
	tok.nextToken(lpart);
	Chain rpart;
	tok.nextToken(rpart);
	
	CegoFieldValue retVal;
	if ( precision > 0 )
	{	    
	    retVal = CegoFieldValue(VARCHAR_TYPE, lpart + Chain(".") + rpart.subChain(1, precision));
	}
	else
	{
	    retVal = CegoFieldValue(INT_TYPE, lpart);
	}
	return retVal;
    }
    case CegoFunction::DATE2STR:
    {		
	CegoFieldValue fv;
	
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv1.getType() == NULL_TYPE )
	    return fv;

	if ( fv1.castTo(DATETIME_TYPE) == false )
	{
	    Chain msg = Chain("Cannot cast to datetime"); 
	    throw Exception(EXLOC, msg);	    
	}

	int val = 0; 
	if ( fv1.getValue() != 0 )
	    memcpy(&val, fv1.getValue(), sizeof(int));

	Datetime dt;

	if ( val != 0 )
	{
	    dt = Datetime(val);
	}

	fv = CegoFieldValue(VARCHAR_TYPE, dt.asChain(fv2.valAsChain()));
	
	return fv;	
    }
    case CegoFunction::DATE2LONG:
    {		

	CegoExpr** pExpr1 = _exprList.First();
	
	CegoFieldValue fv1;
	
	fv1 = (*pExpr1)->evalFieldValue();

	if ( fv1.getType() == NULL_TYPE )
	    return CegoFieldValue();

	if ( fv1.castTo(DATETIME_TYPE) == false )
	{
	    Chain msg = Chain("Cannot cast to datetime"); 
	    throw Exception(EXLOC, msg);	    
	}

	unsigned long long dateVal = 0; 
	if ( fv1.getValue() != 0 )
	    memcpy(&dateVal, fv1.getValue(), sizeof(unsigned long long));
	if ( dateVal == 0 )
	{	   
	    Datetime dt;
	    dateVal = dt.asLong();
	}
	CegoFieldValue fv = CegoFieldValue(LONG_TYPE, Chain(dateVal));
	return fv;	
    }
    case CegoFunction::LONG2DATE:
    {			
	CegoExpr** pExpr1 = _exprList.First();
	
	CegoFieldValue fv1;	
	fv1 = (*pExpr1)->evalFieldValue();

	if ( fv1.getType() == NULL_TYPE )
	    return CegoFieldValue();

	if ( fv1.castTo(LONG_TYPE) == false )
	{
	    Chain msg = Chain("Cannot cast to long"); 
	    throw Exception(EXLOC, msg);	    	    
	}
	unsigned long long *pVal = new unsigned long long;
	if ( fv1.getValue() != 0 )
	    memcpy(pVal, fv1.getValue(), sizeof(unsigned long long));
	else
	    *pVal = 0;

	CegoFieldValue fv = CegoFieldValue(DATETIME_TYPE, pVal, sizeof(unsigned long long), true);
	return fv;
    }
    case CegoFunction::NEWDATE:
    {
	int year = 0;
	int month = 1;
	int day = 1;
	int hour = 0;
	int minute = 0;
	int second = 0;
	
	CegoExpr** pYearExpr = _exprList.First();
	if ( pYearExpr )
	{
	    CegoFieldValue yearVal;	
	    yearVal = (*pYearExpr)->evalFieldValue();
	    
	    if ( yearVal.castTo(INT_TYPE) == false )
	    {
		Chain msg = Chain("Cannot cast year to int"); 
		throw Exception(EXLOC, msg);	    	    
	    }
	    	
	    memcpy(&year, yearVal.getValue(), sizeof(int));
	
	    
	    CegoExpr** pMonthExpr = _exprList.Next();

	    if ( pMonthExpr )
	    {		
		CegoFieldValue monthVal;	
		monthVal = (*pMonthExpr)->evalFieldValue();
		
		if ( monthVal.castTo(INT_TYPE) == false )
		{
		    Chain msg = Chain("Cannot cast month to int"); 
		    throw Exception(EXLOC, msg);	    	    
		}
		
		memcpy(&month, monthVal.getValue(), sizeof(int));
		
		CegoExpr** pDayExpr = _exprList.Next();

		if ( pDayExpr )
		{
		    CegoFieldValue dayVal;	
		    dayVal = (*pDayExpr)->evalFieldValue();
		    
		    if ( dayVal.castTo(INT_TYPE) == false )
		    {
			Chain msg = Chain("Cannot cast day to int"); 
			throw Exception(EXLOC, msg);	    	    
		    }
		    
		    memcpy(&day, dayVal.getValue(), sizeof(int));
		    		    
		    CegoExpr** pHourExpr = _exprList.Next();

		    if ( pHourExpr )
		    {
			CegoFieldValue hourVal;	
			hourVal = (*pHourExpr)->evalFieldValue();
			
			if ( hourVal.castTo(INT_TYPE) == false )
			{
			    Chain msg = Chain("Cannot cast hour to int"); 
			    throw Exception(EXLOC, msg);	    	    
			}
			
			
			memcpy(&hour, hourVal.getValue(), sizeof(int));
			
			
			CegoExpr** pMinuteExpr = _exprList.Next();

			if ( pMinuteExpr )
			{
			    CegoFieldValue minuteVal;	
			    minuteVal = (*pMinuteExpr)->evalFieldValue();
			    
			    if ( minuteVal.castTo(INT_TYPE) == false )
			    {
				Chain msg = Chain("Cannot cast minute to int"); 
				throw Exception(EXLOC, msg);	    	    
			    }
			    
			    memcpy(&minute, minuteVal.getValue(), sizeof(int));
			    
			    CegoExpr** pSecondExpr = _exprList.Next();

			    if ( pSecondExpr )
			    {
				CegoFieldValue secondVal;	
				secondVal = (*pSecondExpr)->evalFieldValue();
				
				if ( secondVal.castTo(INT_TYPE) == false )
				{
				    Chain msg = Chain("Cannot cast second to int"); 
				    throw Exception(EXLOC, msg);	    	    
				}
				
				memcpy(&second, secondVal.getValue(), sizeof(int));
				
			    }
			}
		    }
		}
	    }
	}
	
	unsigned long long *pDateTimeValue = new unsigned long long;
	
	Datetime dt(year, month, day, hour, minute, second);
	*pDateTimeValue = dt.asLong();
	
	CegoFieldValue fv = CegoFieldValue(DATETIME_TYPE, pDateTimeValue, sizeof(unsigned long long), true);
	return fv;
    }    
    case CegoFunction::TRUNC:
    {
	
	CegoExpr** pExpr = _exprList.First();
	
	CegoFieldValue fv1;

	fv1 = (*pExpr)->evalFieldValue();

	if ( fv1.getType() == NULL_TYPE )	
	    return CegoFieldValue();	
	
	Tokenizer t(fv1.valAsChain(), ".");
	Chain truncVal;
	t.nextToken(truncVal);
	
	CegoFieldValue fv(INT_TYPE, truncVal);
	return fv;
    }
    case CegoFunction::LOWER:
    {
	CegoExpr** pExpr = _exprList.First();
	
	CegoFieldValue fv1;
	
	fv1 = (*pExpr)->evalFieldValue();

	if ( fv1.getType() == NULL_TYPE )	
	    return CegoFieldValue();	
	
	CegoFieldValue fv(VARCHAR_TYPE, fv1.valAsChain().toLower());
	return fv;
    }
    case CegoFunction::UPPER:
    {	
	CegoExpr** pExpr = _exprList.First();
	
	CegoFieldValue fv1;
	
	fv1 = (*pExpr)->evalFieldValue();

	if ( fv1.getType() == NULL_TYPE )	
	    return CegoFieldValue();	

	CegoFieldValue fv(VARCHAR_TYPE, fv1.valAsChain().toUpper());
	return fv;
    }
    case CegoFunction::RIGHT:
    {	
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv2.getType() != INT_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function right"); 
	    throw Exception(EXLOC, msg);
	}
	int len = 0; 
	if ( fv2.getValue() != 0 )
	    memcpy(&len, fv2.getValue(), sizeof(int));
	// len = *((int*)fv2.getValue());
			
	Chain right;
	if ( len < fv1.valAsChain().length() )
	    right = fv1.valAsChain().subChain(fv1.valAsChain().length() - len, fv1.valAsChain().length());
	else
	    right = fv1.valAsChain();
	
	CegoFieldValue fv(VARCHAR_TYPE, right);
	return fv;
    }
    case CegoFunction::LEFT:
    {
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv2.getType() != INT_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function right"); 
	    throw Exception(EXLOC, msg);
	}

	int len = 0; 
	if ( fv2.getValue() != 0 )
	    memcpy(&len, fv2.getValue(), sizeof(int));
	// len = *((int*)fv2.getValue());
	
	Chain left;
	if ( len < fv1.valAsChain().length() )
	    left = fv1.valAsChain().subChain(1, len);
	else
	    left = fv1.valAsChain();
	
	CegoFieldValue fv(VARCHAR_TYPE, left);
	return fv;
    }
    case CegoFunction::GETPOS:
    {    
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	CegoExpr** pExpr3 = _exprList.Next();
	CegoExpr** pExpr4 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	int start = 0;
	int occ = 1;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( pExpr3 )
	{
	    CegoFieldValue fv = (*pExpr3)->evalFieldValue();
	    start = fv.valAsChain().asInteger();    
	}
	if ( pExpr4 )
	{
	    CegoFieldValue fv = (*pExpr4)->evalFieldValue();
	    occ = fv.valAsChain().asInteger();    
	}

	Chain s = fv1.valAsChain();
			
	int pos;
	if ( s.posStr(fv2.valAsChain(), pos, start, occ) )
	{		
	    CegoFieldValue fv(INT_TYPE, pos);
	    return fv;
	}
	
	CegoFieldValue nullField;	
	return nullField;
    }
    case CegoFunction::SUBSTR:
    {
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	CegoExpr** pExpr3 = _exprList.Next();
	
	CegoFieldValue fv1, fv2, fv3;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();
	
	Chain s = fv1.valAsChain();
	int start = fv2.valAsChain().asInteger();
		
	int len=0;
	
	if ( pExpr3 )
	{
	    fv3 = (*pExpr3)->evalFieldValue();
	    len=fv3.valAsChain().asInteger();
	}


	// we start at min at position 1
	if ( start < 1 )
	    start = 1;

	// we start at max s.length()
	if ( start > s.length() )
	{
	    start = s.length();
	    // throw Exception(EXLOC, "Start postion after end");
	}
		
	if ( start+len-1 > s.length())
	{
	    len = s.length() - start;
	    // throw Exception(EXLOC, "End postion after end");
	}
	
	Chain sub;
	if ( len > 0 )
	{
	    sub = s.subChain(start, start+len-1);
	}
	else
	{
	    sub = s.subChain(start, s.length());
	}
	
	CegoFieldValue fv(VARCHAR_TYPE, sub);
	
	return fv;
    }
    case CegoFunction::REPLACE:
    {    
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	CegoExpr** pExpr3 = _exprList.Next();
	
	CegoFieldValue fv1, fv2, fv3;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();
	fv3 = (*pExpr3)->evalFieldValue();
	
	Chain src = fv1.valAsChain();
	Chain search = fv2.valAsChain();
	Chain replace = fv3.valAsChain();
	
	Chain res;
	src.replaceAll(search, replace, res);
	
	CegoFieldValue fv(VARCHAR_TYPE, res);
	return fv;	

    }
    case CegoFunction::LENGTH:
    {
	CegoExpr** pExpr1 = _exprList.First();
	
	CegoFieldValue fv1;
	fv1 = (*pExpr1)->evalFieldValue();

	if ( fv1.getType() == NULL_TYPE )	
	    return CegoFieldValue();	

	Chain s = fv1.valAsChain();
	
	CegoFieldValue fv(INT_TYPE, s.length() - 1);

	return fv;
    }
    case CegoFunction::TRIM:
    {	
	CegoExpr** pExpr1 = _exprList.First();

	CegoFieldValue fv1;
	fv1 = (*pExpr1)->evalFieldValue();
	
	CegoExpr** pExpr2 = _exprList.Next();

	CegoFieldValue fv2;
	fv2 = (*pExpr2)->evalFieldValue();
	
	Chain s = fv1.valAsChain();
		
	Chain t = s.cutTrailing(fv2.valAsChain());
				    
	CegoFieldValue fv(VARCHAR_TYPE, t);
	return fv;
    }
    case CegoFunction::RTRIM:
    {	
	CegoExpr** pExpr1 = _exprList.First();

	CegoFieldValue fv1;
	fv1 = (*pExpr1)->evalFieldValue();
	
	CegoExpr** pExpr2 = _exprList.Next();

	CegoFieldValue fv2;
	fv2 = (*pExpr2)->evalFieldValue();
	
	Chain s = fv1.valAsChain();
		
	Chain t = s.truncRight(fv2.valAsChain());
				    
	CegoFieldValue fv(VARCHAR_TYPE, t);
	return fv;
    }
    case CegoFunction::LTRIM:
    {	
	CegoExpr** pExpr1 = _exprList.First();

	CegoFieldValue fv1;
	fv1 = (*pExpr1)->evalFieldValue();
	
	CegoExpr** pExpr2 = _exprList.Next();

	CegoFieldValue fv2;
	fv2 = (*pExpr2)->evalFieldValue();
	
	Chain s = fv1.valAsChain();
		
	Chain t = s.truncLeft(fv2.valAsChain());
				    
	CegoFieldValue fv(VARCHAR_TYPE, t);
	return fv;
    }
    case CegoFunction::STR2INT:
    {	
	CegoExpr** pExpr = _exprList.First();
	
	CegoFieldValue fv1;

	fv1 = (*pExpr)->evalFieldValue();

	if ( fv1.getType() == NULL_TYPE )	
	    return CegoFieldValue();	
	
	CegoFieldValue fv(INT_TYPE, fv1.valAsChain());
	return fv;
    }
    case CegoFunction::STR2LONG:
    {	
	CegoExpr** pExpr = _exprList.First();
	
	CegoFieldValue fv;

	fv = (*pExpr)->evalFieldValue();
	
	CegoFieldValue retVal(LONG_TYPE, fv.valAsChain());
	return retVal;
    }
    case CegoFunction::STR2DATE:
    {
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();
	
	unsigned long long *pDV = new (unsigned long long);
	Datetime dt(fv1.valAsChain(), fv2.valAsChain());
	*pDV = dt.asLong();
	
	CegoFieldValue retVal(DATETIME_TYPE, pDV, sizeof(unsigned long long), true);
	return retVal;
    }
    case CegoFunction::RANDSTR:
    {	
	CegoExpr** pExpr = _exprList.First();
	
	CegoFieldValue fv;

	fv = (*pExpr)->evalFieldValue();

	int l = (int)random() % fv.valAsChain().asInteger();
	if ( l == 0 )
	    l = fv.valAsChain().asInteger();
	
	Chain rs;
	for ( int j=0; j<l; j++ ) 
	{ 
	    rs += Chain((char)(65 + (random() % 26))); 
	}
	
	CegoFieldValue retVal(VARCHAR_TYPE, rs);
	return retVal;
    }
    case CegoFunction::RANDINT:
    {	
	CegoExpr** pExpr = _exprList.First();
	
	CegoFieldValue fv;

	fv = (*pExpr)->evalFieldValue();
	
	int l = fv.valAsChain().asInteger();

	CegoFieldValue retVal(INT_TYPE, random() % l);  
	return retVal;
    }
    case CegoFunction::MOD:
    {	
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv1.castTo(INT_TYPE) == false )	    
	{
	    Chain msg = Chain("Invalid datatype in function mod"); 
	    throw Exception(EXLOC, msg);
	}

	if ( fv2.castTo(INT_TYPE) == false )	    
	{
	    Chain msg = Chain("Invalid datatype in function mod"); 
	    throw Exception(EXLOC, msg);
	}

	int op1 = 0;
	if ( fv1.getValue() != 0 )
	    memcpy(&op1, fv1.getValue(), sizeof(int));
	// cast to long

	int op2 = 0;
	if ( fv2.getValue() != 0 )
	    memcpy(&op2, fv2.getValue(), sizeof(int));

	if ( op2 == 0 )
	    throw Exception(EXLOC, "Division by zero");
	
	int modval = op1 % op2;
	CegoFieldValue fv(INT_TYPE, modval);
	return fv;	
    }
    case CegoFunction::LMOD:
    {
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv1.castTo(LONG_TYPE) == false )	    
	{
	    Chain msg = Chain("Invalid datatype in function lmod"); 
	    throw Exception(EXLOC, msg);
	}

	if ( fv2.castTo(LONG_TYPE) == false )	    
	{
	    Chain msg = Chain("Invalid datatype in function lmod"); 
	    throw Exception(EXLOC, msg);
	}
	
	long long op1 = 0;
	if ( fv1.getValue() != 0 )
	    memcpy(&op1, fv1.getValue(), sizeof(long long));

	long long op2 = 0;
	if ( fv2.getValue() != 0 )
	    memcpy(&op2, fv2.getValue(), sizeof(long long));

	if ( op2 == 0 )
	    throw Exception(EXLOC, "Division by zero");
	
	long long modval = op1 % op2;
	CegoFieldValue fv(LONG_TYPE, modval);
	return fv;	
    }
    case CegoFunction::DIV:
    {
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv1.castTo(INT_TYPE) == false )	    
	{
	    Chain msg = Chain("Invalid datatype in function div"); 
	    throw Exception(EXLOC, msg);
	}

	if ( fv2.castTo(INT_TYPE) == false )	    
	{
	    Chain msg = Chain("Invalid datatype in function div"); 
	    throw Exception(EXLOC, msg);
	}

	int op1 = 0;
	if ( fv1.getValue() != 0 )
	    memcpy(&op1, fv1.getValue(), sizeof(int));

	int op2 = 0;
	if ( fv2.getValue() != 0 )
	    memcpy(&op2, fv2.getValue(), sizeof(int));

	if ( op2 == 0 )
	    throw Exception(EXLOC, "Division by zero");
	
	int divval = op1 / op2;
	CegoFieldValue fv(INT_TYPE, divval);
	return fv;		
    }
    case CegoFunction::LDIV:
    {
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv1.castTo(LONG_TYPE) == false )	    
	{
	    Chain msg = Chain("Invalid datatype in function ldiv"); 
	    throw Exception(EXLOC, msg);
	}

	if ( fv2.castTo(LONG_TYPE) == false )	    
	{
	    Chain msg = Chain("Invalid datatype in function ldiv"); 
	    throw Exception(EXLOC, msg);
	}
	
	long long op1 = 0;
	if ( fv1.getValue() != 0 )
	    memcpy(&op1, fv1.getValue(), sizeof(long long));

	long long op2 = 0;
	if ( fv2.getValue() != 0 )
	    memcpy(&op2, fv2.getValue(), sizeof(long long));

	if ( op2 == 0 )
	    throw Exception(EXLOC, "Division by zero");
	
	long long divval = op1 / op2;
	CegoFieldValue fv(LONG_TYPE, divval);
	return fv;	
    }
    case CegoFunction::POWER:
    {
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv1.getType() != INT_TYPE || fv2.getType() != INT_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function power"); 
	    throw Exception(EXLOC, msg);
	}
	
	int op1 = 0;
	if ( fv1.getValue() != 0 )
	    memcpy(&op1, fv1.getValue(), sizeof(int));
	
	int op2 = 0;
	if ( fv2.getValue() != 0 )
	    memcpy(&op2, fv2.getValue(), sizeof(int));
	
	int i,p;
	p = 1;
	for (i = 1; i <= op2; ++i)
	    p *= op1;
	
	CegoFieldValue fv(INT_TYPE, p);
	return fv;	
    }
    case CegoFunction::BITAND:
    {
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv1.getType() != INT_TYPE || fv2.getType() != INT_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function bitand"); 
	    throw Exception(EXLOC, msg);
	}
	
	int op1 = 0;
	if ( fv1.getValue() != 0 )
	    memcpy(&op1, fv1.getValue(), sizeof(int));
	
	int op2 = 0;
	if ( fv2.getValue() != 0 )
	    memcpy(&op2, fv2.getValue(), sizeof(int));
	
	int a = op1 & op2;

	CegoFieldValue fv(INT_TYPE, a);
	return fv;	
    }
    case CegoFunction::BITOR:
    {
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv1.getType() != INT_TYPE || fv2.getType() != INT_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function bitor"); 
	    throw Exception(EXLOC, msg);
	}
	
	int op1 = 0;
	if ( fv1.getValue() != 0 )
	    memcpy(&op1, fv1.getValue(), sizeof(int));
	
	int op2 = 0;
	if ( fv2.getValue() != 0 )
	    memcpy(&op2, fv2.getValue(), sizeof(int));
	
	int o = op1 | op2;

	CegoFieldValue fv(INT_TYPE, o);
	return fv;
    }
    case CegoFunction::BITXOR:
    {
	CegoExpr** pExpr1 = _exprList.First();
	CegoExpr** pExpr2 = _exprList.Next();
	
	CegoFieldValue fv1, fv2;
	
	fv1 = (*pExpr1)->evalFieldValue();
	fv2 = (*pExpr2)->evalFieldValue();

	if ( fv1.getType() != INT_TYPE || fv2.getType() != INT_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function bitxcor"); 
	    throw Exception(EXLOC, msg);
	}
	
	int op1 = 0;
	if ( fv1.getValue() != 0 )
	    memcpy(&op1, fv1.getValue(), sizeof(int));
	
	int op2 = 0;
	if ( fv2.getValue() != 0 )
	    memcpy(&op2, fv2.getValue(), sizeof(int));
	
	int xo = op1 ^ op2;

	CegoFieldValue fv(INT_TYPE, xo);
	return fv;	
    }
    case CegoFunction::BLOBSIZE:
    {
	CegoExpr** pExpr = _exprList.First();

	unsigned long long blobSize = 0;	
	CegoFieldValue fv;
	
	fv = (*pExpr)->evalFieldValue();

	if ( fv.getType() == NULL_TYPE )
	{
	    CegoFieldValue rfv(LONG_TYPE, blobSize);
	    return rfv;	    
	}
	
	if ( fv.getType() != BLOB_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function blobsize"); 
	    throw Exception(EXLOC, msg);
	}
	
	Tokenizer tok(fv.valAsChain(), Chain(LOBSEP));

	Chain pstr;
	PageIdType pageId = 0;
	if ( tok.nextToken(pstr) )
	{
	    pageId = pstr.asUnsignedLongLong();
	}
	else
	{
	    Chain msg = Chain("Cannot get pageid from blob value"); 
	    throw Exception(EXLOC, msg);	   
	}	

	CegoDatabaseManager* pDBMng = _pTabMng->getDBMng();
	CegoLockHandler* pLockHandle = _pTabMng->getLockHandler();

	CegoBufferPage bp;
	try
	{
	    pDBMng->bufferFix(bp, _tabSetId, pageId, CegoBufferPool::SYNC, pLockHandle);	    
	    memcpy (&blobSize, bp.getChunkEntry() + sizeof(unsigned long long), sizeof(unsigned long long));
	    pDBMng->bufferUnfix(bp, true, pLockHandle);
	}
	catch ( Exception e )
	{
	    if ( bp.isFixed() )
		pDBMng->bufferUnfix(bp, true, pLockHandle);
	}

	CegoFieldValue rfv(LONG_TYPE, blobSize);
	return rfv;	
    }
    case CegoFunction::BLOBREF:
    {
	CegoExpr** pExpr = _exprList.First();

	unsigned long long blobRef = 0;	
	CegoFieldValue fv;
	
	fv = (*pExpr)->evalFieldValue();

	if ( fv.getType() == NULL_TYPE )
	{
	    CegoFieldValue rfv(LONG_TYPE, blobRef);
	    return rfv;	    
	}
	
	if ( fv.getType() != BLOB_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function blobref"); 
	    throw Exception(EXLOC, msg);
	}
	
	Tokenizer tok(fv.valAsChain(), Chain(LOBSEP));

	Chain pstr;
	PageIdType pageId = 0;
	if ( tok.nextToken(pstr) )
	{
	    pageId = pstr.asUnsignedLongLong();
	}
	else
	{
	    Chain msg = Chain("Cannot get pageid from blob value"); 
	    throw Exception(EXLOC, msg);	   
	}	

	CegoDatabaseManager* pDBMng = _pTabMng->getDBMng();
	CegoLockHandler* pLockHandle = _pTabMng->getLockHandler();

	CegoBufferPage bp;
	try
	{
	    pDBMng->bufferFix(bp, _tabSetId, pageId, CegoBufferPool::SYNC, pLockHandle);	    
	    memcpy (&blobRef, bp.getChunkEntry(), sizeof(unsigned long long));
	    pDBMng->bufferUnfix(bp, true, pLockHandle);
	}
	catch ( Exception e )
	{
	    if ( bp.isFixed() )
		pDBMng->bufferUnfix(bp, true, pLockHandle);
	}

	CegoFieldValue rfv(LONG_TYPE, blobRef);
	return rfv;	
    }
    case CegoFunction::CLOBSIZE:
    {
	CegoExpr** pExpr = _exprList.First();

	unsigned long long clobSize = 0;
	CegoFieldValue fv;
	
	fv = (*pExpr)->evalFieldValue();

	if ( fv.getType() == NULL_TYPE )
	{
	    CegoFieldValue rfv(LONG_TYPE, clobSize);
	    return rfv;	    
	}
	
	if ( fv.getType() != CLOB_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function clobsize"); 
	    throw Exception(EXLOC, msg);
	}
	
	Tokenizer tok(fv.valAsChain(), Chain(LOBSEP));
	Chain pstr;
	PageIdType pageId = 0;
	if ( tok.nextToken(pstr) )
	{
	    pageId = pstr.asUnsignedLongLong();
	}
	else
	{
	    Chain msg = Chain("Cannot get pageid from clob value"); 
	    throw Exception(EXLOC, msg);	   
	}	

	CegoDatabaseManager* pDBMng = _pTabMng->getDBMng();
	CegoLockHandler* pLockHandle = _pTabMng->getLockHandler();

	CegoBufferPage bp;
	try
	{
	    pDBMng->bufferFix(bp, _tabSetId, pageId, CegoBufferPool::SYNC, pLockHandle);	    
	    memcpy (&clobSize, bp.getChunkEntry() + sizeof(unsigned long long), sizeof(unsigned long long));
	    pDBMng->bufferUnfix(bp, true, pLockHandle);
	}
	catch ( Exception e )
	{
	    if ( bp.isFixed() )
		pDBMng->bufferUnfix(bp, true, pLockHandle);
	}

	CegoFieldValue rfv(LONG_TYPE, clobSize);
	return rfv;	
    }
    case CegoFunction::CLOBREF:
    {
	CegoExpr** pExpr = _exprList.First();

	unsigned long long clobRef = 0;
	CegoFieldValue fv;
	
	fv = (*pExpr)->evalFieldValue();

	if ( fv.getType() == NULL_TYPE )
	{
	    CegoFieldValue rfv(LONG_TYPE, clobRef);
	    return rfv;	    
	}
	
	if ( fv.getType() != CLOB_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function clobref"); 
	    throw Exception(EXLOC, msg);
	}
	
	Tokenizer tok(fv.valAsChain(), Chain(LOBSEP));
	Chain pstr;
	PageIdType pageId = 0;
	if ( tok.nextToken(pstr) )
	{
	    pageId = pstr.asUnsignedLongLong();
	}
	else
	{
	    Chain msg = Chain("Cannot get pageid from clob value"); 
	    throw Exception(EXLOC, msg);	   
	}	

	CegoDatabaseManager* pDBMng = _pTabMng->getDBMng();
	CegoLockHandler* pLockHandle = _pTabMng->getLockHandler();

	CegoBufferPage bp;
	try
	{
	    pDBMng->bufferFix(bp, _tabSetId, pageId, CegoBufferPool::SYNC, pLockHandle);	    
	    memcpy (&clobRef, bp.getChunkEntry(), sizeof(unsigned long long));
	    pDBMng->bufferUnfix(bp, true, pLockHandle);
	}
	catch ( Exception e )
	{
	    if ( bp.isFixed() )
		pDBMng->bufferUnfix(bp, true, pLockHandle);
	}

	CegoFieldValue rfv(LONG_TYPE, clobRef);
	return rfv;	
    }
    case CegoFunction::CLOB2STR:
    {
	CegoExpr** pExpr1 = _exprList.First();
	
	CegoFieldValue fv;
	
	fv = (*pExpr1)->evalFieldValue();

	if ( fv.getType() == NULL_TYPE )
	{
	    CegoFieldValue zs(VARCHAR_TYPE, Chain(""));
	    return zs;	    
	}

	if ( fv.getType() != CLOB_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function clob2str"); 
	    throw Exception(EXLOC, msg);
	}
		
	Tokenizer tok(fv.valAsChain(), Chain(LOBSEP));
	Chain pstr;
	PageIdType pageId = 0;

	if ( tok.nextToken(pstr) )
	{
	    pageId = pstr.asUnsignedLong();
	}
	else
	{
	    Chain msg = Chain("Cannot get pageid from clob value"); 
	    throw Exception(EXLOC, msg);	   
	}

	unsigned long long clobSize;

	char* pC = _pTabMng->getClobData(_tabSetId, pageId, clobSize);
	
	// memory is allocated by getClobData, so we build CegoFieldValue with localCopy = true
	bool isLocalCopy = true;

	// we construct field value with zero termination character
	CegoFieldValue rfv(VARCHAR_TYPE, pC, clobSize + 1, isLocalCopy);
	
	return rfv;
    }   
    case CegoFunction::NEXTCOUNT:
    {       		
	long v = _pTabMng->getDBMng()->getCounterValue(_tabSetId, _counterId, 1);
	CegoFieldValue retVal(LONG_TYPE, Chain(v));
	return retVal;
    }
    case CegoFunction::SETCOUNT:
    {
	CegoFieldValue fv;
	CegoExpr** pCounterExpr = _exprList.First();		
	fv = (*pCounterExpr)->evalFieldValue();	
	fv.castTo(LONG_TYPE);

	if ( fv.getType() != LONG_TYPE )
	{
	    Chain msg = Chain("Invalid datatype in function setcount"); 
	    throw Exception(EXLOC, msg);
	}

	long l = fv.valAsChain().asLong();

	// long v = _pTabMng->getDBMng()->getCounterValue(_tabSetId, _counterId, 0);

	long v = _pTabMng->getDBMng()->setCounterValue(_tabSetId, _counterId, l);

	CegoFieldValue retVal(LONG_TYPE, Chain(v));
	return retVal;
    }
    case CegoFunction::USERDEFINED:
    {
	CegoFieldValue retVal;
	
	_pTabMng->getDBMng()->useObject(_tabSetId, _funcName, CegoObject::PROCEDURE, CegoDatabaseManager::SHARED, _pTabMng->getThreadId());

	ListT<CegoFieldValue> clobRefList;
		
	try 
	{
	    // to allow recursive usage of user defined functions and procedures, we have to allocate
	    // a dedicated procedure instance. The most easy way is to parse the object from scratch

	    CegoProcedure* pProc = _pTabMng->getProcedure(_tabSetId, _funcName);
	    
	    bool doDelete=false;
	    
	    if ( pProc->getMasterBlock() != 0 ) // procedure is already in use by this thread ( recursive call ) 
	    {
		pProc = loadProcedure(_tabSetId, _funcName);
		doDelete = true;
	    }
	    
	    if ( pProc )
	    {		
		pProc->setMasterBlock(_pBlock);
		
		ListT<CegoProcVar> argList;
		pProc->getArgList(argList);

		ListT<CegoFieldValue> fvl;
			
		CegoProcVar *pVar = argList.First();
		CegoExpr **pExpr = _exprList.First();
		int pos=1;
		while ( pVar && pExpr ) 
		{		    
		    (*pExpr)->setBlock(_pBlock);
		    
		    if ( pVar->getVarType() == CegoProcVar::OUTVAR )
		    {
			Chain outVar;
			(*pExpr)->checkVar(outVar);
			
			CegoProcVar *pCheckVar = _pBlock->getVarList().Find(CegoProcVar(outVar));
			if ( pCheckVar == 0 )
			{
			    CegoFieldValue nullVal;
			    _pBlock->getVarList().Insert(CegoProcVar(outVar, CegoProcVar::BLOCKVAR, NULL_TYPE, 0, 0, nullVal));
			}
			// we use fv value to store variable name
			fvl.Insert( CegoFieldValue(VARCHAR_TYPE, outVar) );
		    }
		    else // INVAR
		    {
			CegoFieldValue fv = (*pExpr)->evalFieldValue();
			
			if ( fv.getType() == NULL_TYPE )
			{
			    // nothing to to
			}
			else if ( pVar->getType() != fv.getType() )		
			{
			    
			    bool clobCastDone=false;
			    if ( pVar->getType() == CLOB_TYPE )
			    {
				if ( CegoQueryHelper::string2Clob(fv, _pTabMng, _tabSetId) )
				{
				    clobCastDone=true;
				    clobRefList.Insert(fv);
				}
			    }
			    if ( clobCastDone == false )
			    {
				if ( fv.castTo(pVar->getType(), pVar->getDim()) == false )
				{
				    throw Exception(EXLOC, Chain("Mismatched datatype <") 
						    + CEGO_TYPE_MAP[(int)fv.getType()] 
						    + Chain("> in value list for argument ") + Chain(pos) 
						    + " ( expected " 
						    + CEGO_TYPE_MAP[(int)pVar->getType()] + " )");
				}
			    }
			}
			
			fvl.Insert(fv);
		    }
		    
		    pExpr = _exprList.Next();
		    pVar = argList.Next();
		    pos++;
		}

		if ( pVar || pExpr )
		{	
		    Chain msg = Chain("Mismatched parameter count for function ") + _funcName;
		    throw Exception(EXLOC, msg);
		}
		
		pProc->execute(fvl);

		retVal = pProc->getRetVal();

		if ( doDelete )
		    delete pProc;
		else
		    pProc->setMasterBlock(0); // indicate proc as not in use
	    }
	    else
	    {
		Chain msg= Chain("Function ") + _funcName + Chain(" not loaded");
		throw Exception(EXLOC, msg); 
	    }
	}
	catch ( Exception e )
	{

	    CegoFieldValue *pClobRef = clobRefList.First();
	    while ( pClobRef )
	    {
		
		if ( pClobRef->getType() == CLOB_TYPE && pClobRef->getValue() != 0 )
		{
		    PageIdType pageId;
		    memcpy(&pageId, pClobRef->getValue(), sizeof(PageIdType));
		    
		    _pTabMng->decreaseClobRef(_tabSetId, pageId);
		}
		pClobRef = clobRefList.Next();
	    }
	    
	    _pTabMng->getDBMng()->unuseObject(_tabSetId, _funcName, CegoObject::PROCEDURE);
	    throw e;
	}
	
	_pTabMng->getDBMng()->unuseObject(_tabSetId, _funcName, CegoObject::PROCEDURE);

	return retVal;	
    }
    }
}

CegoProcedure* CegoFunction::loadProcedure(int tabSetId, const Chain& procName) const
{
    CegoProcObject po;
    _pTabMng->getObject(tabSetId, procName, CegoObject::PROCEDURE, po);
    
    Chain loadString = Chain("load ") + po.getProcText();
	
    // cout << "Loading procedure " << loadString << endl;

    CegoProcedure* pProc = 0;

    try
    {
	CegoAction* pPA = _pTabMng->getParser();
	
	pPA->cleanUp();
	pPA->setTableSet(_pTabMng->getDBMng()->getTabSetName(tabSetId));
	pPA->setCommandChain(loadString);

	pPA->parse();

	pPA->execute();

	pProc = pPA->getProcedure();	
    }
    catch ( Exception e ) 
    {
	throw e;
    }
    
    return pProc;
}
    
CegoFunction* CegoFunction::clone(bool isAttrRef)
{
    if ( _exprList.isEmpty() )
	return ( new CegoFunction(_type ) );

    ListT<CegoExpr*> cloneList;
    
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	cloneList.Insert( (*pExpr)->clone(isAttrRef));	    
	pExpr = _exprList.Next();
    }
    
    if ( _pTabMng == 0 )
	return ( new CegoFunction(_type, cloneList) );
    else 
    {
	if ( _type == CegoFunction::USERDEFINED )
	{	
	    return ( new CegoFunction(_pTabMng, _tabSetId, _funcName, cloneList) );
	}
	else
	{
	    return ( new CegoFunction(_pTabMng, _tabSetId, _type) );
	}
    }
}

Chain CegoFunction::getId() const
{
    if ( _type ==  CegoFunction::NEXTCOUNT
	 || _type ==  CegoFunction::SETCOUNT
	 || _type ==  CegoFunction::RANDINT
	 || _type ==  CegoFunction::RANDSTR
	 || _type ==  CegoFunction::BLOBSIZE
	 || _type ==  CegoFunction::BLOBREF
	 || _type ==  CegoFunction::BLOBSIZE
	 || _type ==  CegoFunction::CLOBREF
	 || _type ==  CegoFunction::CLOBSIZE
	 || _type ==  CegoFunction::CLOB2STR
	 || _type ==  CegoFunction::USERDEFINED )
    {	 
	// we throw an exception here which is catched by CegoSelect::getQueryId, since this is a modyfying query
	throw Exception(EXLOC, Chain(MOD_QUERY_ID));
    }

    Chain argString;

    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	argString += (*pExpr)->getId();	    
	pExpr = _exprList.Next();
    }
    
    Chain funcString = getName();

    Chain s = funcString + Chain("(") + argString + Chain(")");
    return s;   
}

Chain CegoFunction::toChain(const Chain& indent) const
{
    Chain argString;

    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	argString += (*pExpr)->toChain();	    
	pExpr = _exprList.Next();
	if ( pExpr )
	    argString += Chain(",") + indent;
    }


    if ( _type == CegoFunction::NEXTCOUNT )
    {
	argString = _counterId;
    }
    else if ( _type == CegoFunction::SETCOUNT )
    {
	argString = _counterId + Chain(",") + argString;
    }
    
    Chain funcString = getName();
    Chain s = indent + funcString + Chain("(") + argString + Chain(")");

    return s;
}

Chain CegoFunction::dbFormat(CegoDatabaseFormater *pForm) const
{
    return pForm->formatFunction(_type, _exprList, _funcName, _counterId);
}

Element* CegoFunction::toElement() const
{
    Element* pFunctionElement = new Element(XML_FUNCTION_ELEMENT);

    switch (_type)
    {
    case CegoFunction::INT2ASC:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_INT2ASCFUNC_VALUE );
	break;	
    }
    case CegoFunction::ASC2INT:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_ASC2INTFUNC_VALUE );
	break;	
    }
    case CegoFunction::TRIM:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_TRIMFUNC_VALUE );
	break;	
    }
    case CegoFunction::RTRIM:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_RTRIMFUNC_VALUE );
	break;	
    }
    case CegoFunction::LTRIM:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_LTRIMFUNC_VALUE );
	break;	
    }
    case CegoFunction::ROUND:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_ROUNDFUNC_VALUE );
	break;	
    }
    case CegoFunction::TRUNC:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_TRUNCFUNC_VALUE);
	break;
    }
    case CegoFunction::DATE2STR:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_DATE2STRFUNC_VALUE);
	break;
    }
    case CegoFunction::DATE2LONG:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_DATE2LONGFUNC_VALUE);
	break;
    }
    case CegoFunction::LONG2DATE:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_LONG2DATEFUNC_VALUE);
	break;
    }
    case CegoFunction::NEWDATE:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_NEWDATEFUNC_VALUE);
	break;
    }
    case CegoFunction::LOWER:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_LOWERFUNC_VALUE);
	break;
    }
    case CegoFunction::UPPER:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_UPPERFUNC_VALUE);
	break;
    }
    case CegoFunction::LEFT:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_LEFTFUNC_VALUE);
	break;
    }
    case CegoFunction::RIGHT:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_RIGHTFUNC_VALUE);
	break;
    }
    case CegoFunction::SUBSTR:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_SUBSTRFUNC_VALUE);
	break;
    }
    case CegoFunction::GETPOS:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_GETPOSFUNC_VALUE);
	break;
    }
    case CegoFunction::STR2INT:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_STR2INTFUNC_VALUE);
	break;
    }
    case CegoFunction::STR2LONG:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_STR2LONGFUNC_VALUE);
	break;
    }
    case CegoFunction::STR2DATE:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_STR2DATEFUNC_VALUE);
	break;
    }
    case CegoFunction::RANDSTR:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_RANDSTRFUNC_VALUE);
	break;
    }
    case CegoFunction::RANDINT:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_RANDINTFUNC_VALUE);
	break;
    }
    case CegoFunction::REPLACE:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_REPLACEFUNC_VALUE);
	break;
    }
    case CegoFunction::MOD:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_MODFUNC_VALUE);
	break;
    }
    case CegoFunction::DIV:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_DIVFUNC_VALUE);
	break;
    }
    case CegoFunction::LMOD:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_LMODFUNC_VALUE);
	break;
    }
    case CegoFunction::LDIV:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_LDIVFUNC_VALUE);
	break;
    }
    case CegoFunction::POWER:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_POWERFUNC_VALUE);
	break;
    }
    case CegoFunction::BITAND:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_BITANDFUNC_VALUE);
	break;
    }
    case CegoFunction::BITOR:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_BITORFUNC_VALUE);
	break;
    }
    case CegoFunction::BITXOR:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_BITXORFUNC_VALUE);
	break;
    }
    case CegoFunction::BLOBSIZE:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_BLOBSIZEFUNC_VALUE);
	break;
    }
    case CegoFunction::BLOBREF:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_BLOBREFFUNC_VALUE);
	break;
    }
    case CegoFunction::CLOBSIZE:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_CLOBSIZEFUNC_VALUE);
	break;
    }
    case CegoFunction::CLOBREF:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_CLOBREFFUNC_VALUE);
	break;
    }
    case CegoFunction::CLOB2STR:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_CLOB2STRFUNC_VALUE);
	break;
    }
    case CegoFunction::LENGTH:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_LENGTHFUNC_VALUE);
	break;
    }
    case CegoFunction::NEXTCOUNT:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_NEXTCOUNTFUNC_VALUE);
	break;
    }
    case CegoFunction::SETCOUNT:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_SETCOUNTFUNC_VALUE);
	break;
    }
    case CegoFunction::USERDEFINED:
    {
	pFunctionElement->setAttribute( XML_FUNCTYPE_ATTR, XML_USERDEFINEDFUNC_VALUE);
	break;
    }
    }

    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	pFunctionElement->addContent((*pExpr)->toElement());	    
	pExpr = _exprList.Next();
    }

    return pFunctionElement;
}

void CegoFunction::fromElement(Element *pFunctionElement, CegoDistManager *pGTM)
{
    Chain functionTypeString = pFunctionElement->getAttributeValue( XML_FUNCTYPE_ATTR );

    if ( functionTypeString == Chain(XML_INT2ASCFUNC_VALUE) )
    {
	_type = CegoFunction::INT2ASC;
    }
    else if ( functionTypeString == Chain(XML_ASC2INTFUNC_VALUE) )
    {
	_type = CegoFunction::ASC2INT;
    }
    else if ( functionTypeString == Chain(XML_TRIMFUNC_VALUE) )
    {
	_type = CegoFunction::TRIM;
    }
    else if ( functionTypeString == Chain(XML_RTRIMFUNC_VALUE) )
    {
	_type = CegoFunction::RTRIM;
    }
    else if ( functionTypeString == Chain(XML_LTRIMFUNC_VALUE) )
    {
	_type = CegoFunction::LTRIM;
    }
    else if ( functionTypeString == Chain(XML_ROUNDFUNC_VALUE) )
    {
	_type = CegoFunction::ROUND;
    }
    else if ( functionTypeString == Chain(XML_TRUNCFUNC_VALUE) )
    {
	_type = CegoFunction::TRUNC;
    }
    else if ( functionTypeString == Chain(XML_DATE2STRFUNC_VALUE) )
    {
	_type = CegoFunction::DATE2STR;
    }
    else if ( functionTypeString == Chain(XML_DATE2LONGFUNC_VALUE) )
    {
	_type = CegoFunction::DATE2LONG;
    }
    else if ( functionTypeString == Chain(XML_LONG2DATEFUNC_VALUE) )
    {
	_type = CegoFunction::LONG2DATE;
    }
    else if ( functionTypeString == Chain(XML_NEWDATEFUNC_VALUE) )
    {
	_type = CegoFunction::NEWDATE;
    }
    else if ( functionTypeString == Chain(XML_LOWERFUNC_VALUE) )
    {
	_type = CegoFunction::LOWER;
    }
    else if ( functionTypeString == Chain(XML_UPPERFUNC_VALUE) )
    {
	_type = CegoFunction::UPPER;
    }
    else if ( functionTypeString == Chain(XML_LEFTFUNC_VALUE) )
    {
	_type = CegoFunction::LEFT;
    }
    else if ( functionTypeString == Chain(XML_RIGHTFUNC_VALUE) )
    {
	_type = CegoFunction::RIGHT;
    }
    else if ( functionTypeString == Chain(XML_SUBSTRFUNC_VALUE) )
    {
	_type = CegoFunction::SUBSTR;
    }
    else if ( functionTypeString == Chain(XML_GETPOSFUNC_VALUE) )
    {
	_type = CegoFunction::GETPOS;
    }
    else if ( functionTypeString == Chain(XML_STR2INTFUNC_VALUE) )
    {
	_type = CegoFunction::STR2INT;
    }
    else if ( functionTypeString == Chain(XML_STR2LONGFUNC_VALUE) )
    {
	_type = CegoFunction::STR2LONG;
    }
    else if ( functionTypeString == Chain(XML_RANDSTRFUNC_VALUE) )
    {
	_type = CegoFunction::RANDSTR;
    }
    else if ( functionTypeString == Chain(XML_RANDINTFUNC_VALUE) )
    {
	_type = CegoFunction::RANDINT;
    }
    else if ( functionTypeString == Chain(XML_REPLACEFUNC_VALUE) )
    {
	_type = CegoFunction::REPLACE;
    }
    else if ( functionTypeString == Chain(XML_MODFUNC_VALUE) )
    {
	_type = CegoFunction::MOD;
    }
    else if ( functionTypeString == Chain(XML_DIVFUNC_VALUE) )
    {
	_type = CegoFunction::DIV;
    }
    else if ( functionTypeString == Chain(XML_LMODFUNC_VALUE) )
    {
	_type = CegoFunction::LMOD;
    }
    else if ( functionTypeString == Chain(XML_LDIVFUNC_VALUE) )
    {
	_type = CegoFunction::LDIV;
    }
    else if ( functionTypeString == Chain(XML_POWERFUNC_VALUE) )
    {
	_type = CegoFunction::POWER;
    }
    else if ( functionTypeString == Chain(XML_BITANDFUNC_VALUE) )
    {
	_type = CegoFunction::BITAND;
    }
    else if ( functionTypeString == Chain(XML_BITORFUNC_VALUE) )
    {
	_type = CegoFunction::BITOR;
    }
    else if ( functionTypeString == Chain(XML_BITXORFUNC_VALUE) )
    {
	_type = CegoFunction::BITXOR;
    }
    else if ( functionTypeString == Chain(XML_BLOBSIZEFUNC_VALUE) )
    {
	_type = CegoFunction::BLOBSIZE;
    }
    else if ( functionTypeString == Chain(XML_LENGTHFUNC_VALUE) )
    {
	_type = CegoFunction::LENGTH;
    }
    else if ( functionTypeString == Chain(XML_NEXTCOUNTFUNC_VALUE) )
    {
	_type = CegoFunction::NEXTCOUNT;
    }
    else if ( functionTypeString == Chain(XML_SETCOUNTFUNC_VALUE) )
    {
	_type = CegoFunction::SETCOUNT;
    }
    else if ( functionTypeString == Chain(XML_USERDEFINEDFUNC_VALUE) )
    {
	_type = CegoFunction::USERDEFINED;
    }
    
    ListT<Element*> el = pFunctionElement->getChildren(XML_EXPR_ELEMENT);
    Element **pEE = el.First();
    while ( pEE )
    {
	CegoExpr *pExpr = new CegoExpr(*pEE, pGTM);
	_exprList.Insert (pExpr);
	pEE = el.Next();
    }
}

void CegoFunction::encode(char *buf)
{
    char* pE = (char*)buf;

    memcpy( pE, &_type, sizeof(CegoFunction::FunctionType));
    pE = pE + sizeof(CegoFunction::FunctionType);

    if ( _type == CegoFunction::USERDEFINED )
    {
	int len = _funcName.length() - 1;
	memcpy(pE, &len, sizeof(int));
	pE = pE + sizeof(int);
	memcpy(pE, (char*)_funcName, len);
	pE = pE + len; 
    }

    int numExpr = _exprList.Size();

    memcpy( pE, &numExpr, sizeof(int));
    pE = pE + sizeof(int);
        
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->encode(pE);
	pE = pE + (*pExpr)->getEncodingLength();
	pExpr = _exprList.Next();
    }
}

void CegoFunction::decode(char *buf, CegoDistManager* pGTM, int tabSetId)
{
    char* pE = (char*)buf;

    memcpy( &_type, pE, sizeof(CegoFunction::FunctionType));
    pE = pE + sizeof(CegoFunction::FunctionType);

    if ( _type == CegoFunction::USERDEFINED )
    {
	int len;
	memcpy( &len, pE, sizeof(int));
	pE = pE + sizeof(int);
	_funcName = Chain((char*)pE, len);
	pE = pE + len;
    }

    int numExpr;
    memcpy( &numExpr, pE, sizeof(int));
    pE = pE + sizeof(int);

    int i=0;
    while ( i < numExpr )
    {
	CegoExpr *pExpr = new CegoExpr(pE, pGTM, tabSetId);
	pE = pE + pExpr->getEncodingLength();
	_exprList.Insert(pExpr);
	i++;
    }    
}

int CegoFunction::getEncodingLength() const
{
    int len = 0;
    
    len += sizeof(CegoFunction::FunctionType);
    if ( _type == CegoFunction::USERDEFINED )
    {
	len += sizeof(int); // size of user funcname
	len += _funcName.length() - 1;
    }
    len += sizeof(int); // numExpr
    
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	len += (*pExpr)->getEncodingLength();
	pExpr = _exprList.Next();
    }
    return len;
}

CegoFunction& CegoFunction::operator = ( const CegoFunction& f)
{
    _type = f._type;
    _exprList = f._exprList;
    _tabSetId = f._tabSetId;
    _pTabMng = f._pTabMng;
    _funcName = f._funcName;
    return (*this);
}

ostream& operator << (ostream& s, const CegoFunction& f)
{
    s << f.toChain();
    return s;
}
