///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoAdmMon.cc  
// -------------
// Cego admin monitor
//
// Design and Implementation by Bjoern Lemke
//     
// (C)opyright 2000-2019 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoAdmMon
// 
// Description: Cursor based administration database administration frontend
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <lfcbase/Tokenizer.h>

#include "CegoAdmMon.h"
#include "CegoXMLdef.h"

#define INPUT_TYPE_STRING "S"
#define INPUT_TYPE_ID "I"
#define INPUT_TYPE_NUMBER "N"
#define INPUT_TYPE_PWD "P"
#define INPUT_TYPE_MENU "M"

#define MAXNAMELEN "20"
#define MAXSIZELEN "20"
#define MAXPATHLEN "80"
#define MAXPASSWDLEN "20"

#define DEFREFINT 3000

// action defines
#define TS_CREATE_ACTION 1
#define TS_START_ACTION 2
#define TS_STOP_ACTION 3
#define TS_DROP_ACTION 4
#define TS_RESTORE_ACTION 5
#define TS_RECOVER_ACTION 6
#define TS_REMOVE_ACTION 7
#define TS_ADD_FILE 8
#define TS_LIST_FILE 9
#define TS_ADD_ARCHLOG 10
#define TS_LIST_ARCHLOG 11
#define TS_REMOVE_ARCHLOG 12
#define TS_TOGGLE_ARCHLOG 13
#define TS_RESTORE 14
#define TS_BACKUP_ACTION 15
#define TS_LIST_BACKUP 16

// ts status definition
#define TS_DEFINED_STATUS "DEFINED"
#define TS_ONLINE_STATUS "ONLINE"
#define TS_OFFLINE_STATUS "OFFLINE"

// user action defines
#define USER_TOGGLETRACE_ACTION 1
#define USER_DELETE_ACTION 2
#define USER_ASSIGNROLE_ACTION 3
#define USER_REMOVEROLE_ACTION 4
#define USER_CHANGEPWD_ACTION 5

// role action defines
#define ROLE_PERM_LIST_ACTION 1
#define ROLE_PERM_SET_ACTION 2
#define ROLE_PERM_REMOVE_ACTION 3
#define ROLE_DROP_ACTION 4

#define ERROR_TITLE "ERROR"
#define INFO_TITLE "INFORMATION"

#define MSGBOX_WIDTH 60

#define HELPMASK_WIDTH 50
#define HELPMASK_HEIGHT 30

#define VAL_MAX_LEN 30
#define VAL_MAX_NUM 20

#define INFO_ACTION 100
#define ADD_ACTION 200
#define DELETE_ACTION 300
#define NEW_ACTION 400
#define FILEINFO_ACTION 500
#define RESTORE_ACTION 600

#define ATTRSEPTOKEN Chain("#")
#define LISTSEPTOKEN Chain("&")
#define VALSEPTOKEN Chain("=")

CegoAdmMon::CegoAdmMon(CegoAdminHandler *pAH) : Monitor(ATTRSEPTOKEN, LISTSEPTOKEN, VALSEPTOKEN)
{
    _pAH = pAH;
    _refInterval = DEFREFINT;
}

CegoAdmMon::~CegoAdmMon()  
{    
}

void CegoAdmMon::showMonitor()
{
    regMenu(Chain("BufferPool"), 1); 
    regMenu(Chain("DBThread"), 2); 
    regMenu(Chain("AdmThread"), 3); 
    regMenu(Chain("LogThread"), 4);
    regMenu(Chain("TableSet"), 5);
    regMenu(Chain("Role"), 6);
    regMenu(Chain("User"), 7);
    regMenu(Chain("Refresh"), 8);
    regMenu(Chain("Quit"), 9);

    regShortCut('i', INFO_ACTION);
    regShortCut('a', ADD_ACTION);
    regShortCut('n', NEW_ACTION);
    regShortCut('d', DELETE_ACTION);
    regShortCut('f', FILEINFO_ACTION); 
    regShortCut('r', RESTORE_ACTION);
    
    int midx = 0; 
    while ( midx != 9 ) 
    { 
	midx = showHeader(); 

	if ( midx == 1 ) 
	{ 
	    showPool();
	}
	else if ( midx == 2 )
	{
	    showDBThread();
	}
	else if ( midx == 3 )
	{
	    showAdmThread();
	}
	else if ( midx == 4 )
	{
	    showLogThread();
	}
	else if ( midx == 5 )
	{
	    showTableSet();
	}
	else if ( midx == 6 )
	{
	    showRole();
	}
	else if ( midx == 7 )
	{
	    showUser();
	}
	else if ( midx == 8 )
	{
	    showRefreshMenu();
	}	
    }    
}

void CegoAdmMon::showPool()
{    
    int ret = 1;
    bool doInit = true;
    
    while ( ret != MON_LEAVE )
    {
	ListT<ListT<CegoFieldValue> > poolinfo1;
	ListT<ListT<CegoFieldValue> > poolinfo2;
	
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;
	
	res = _pAH->reqPoolInfo();
	
	_pAH->getPoolInfo(oe, poolinfo1, poolinfo2);	    	    
		    

	ListT< ListT<Chain> > poolInfoAll;

	ListT<Chain> poolInfoA;
	ListT<CegoFieldValue> *pFVL = poolinfo1.First();
	while ( pFVL )
	{
	    Chain key, value;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		key = pFV->valAsChain();
	    }
	    
	    pFV = pFVL->Next();
	    if ( pFV )
	    {
		value = pFV->valAsChain();
	    }
	    
	    poolInfoA.Insert(key + VALSEPTOKEN + value);	    	    
	    pFVL = poolinfo1.Next();
	}

	poolInfoAll.Insert(poolInfoA);

	ListT<Chain> poolInfoB;
	pFVL = poolinfo2.First();
	while ( pFVL )
	{
	    Chain key, value;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		key = pFV->valAsChain();
	    }
	    
	    pFV = pFVL->Next();
	    if ( pFV )
	    {
		value = pFV->valAsChain();
	    }
	    
	    poolInfoB.Insert(key + VALSEPTOKEN + value);	    	    
	    pFVL = poolinfo2.Next();
	}

	poolInfoAll.Insert(poolInfoB);

	ret = showAttributeBox(20, 30, poolInfoAll, _refInterval);
	doInit = false;
    }
}

void CegoAdmMon::showDBThread()
{
    bool doInit = true;
    int ret = 1;
    while ( ret != MON_LEAVE )
    {
	ListT<ListT<CegoFieldValue> > dbthreadinfo;
	
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;
	Chain format;    
	
	res = _pAH->reqDbThreadInfo();
	_pAH->getDbThreadInfo(oe, dbthreadinfo, format);

	ListT<Chain> threadSchema;
	threadSchema.Insert(Chain("ThreadId") + VALSEPTOKEN + Chain("10"));
	threadSchema.Insert(Chain("NumConReq") + VALSEPTOKEN + Chain("10"));
	threadSchema.Insert(Chain("NumQueryReq") + VALSEPTOKEN + Chain("20"));
	threadSchema.Insert(Chain("Load") + VALSEPTOKEN + Chain("10"));
	threadSchema.Insert(Chain("SortAlloc") + VALSEPTOKEN + Chain("15"));
	threadSchema.Insert(Chain("Status") + VALSEPTOKEN + Chain("10"));
		
	ListT<ListT<Chain> > threadTable;
		   
	ListT<CegoFieldValue> *pFVL = dbthreadinfo.First();
	while ( pFVL )
	{
	    ListT<Chain> threadRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    while ( pFV )
	    {
		threadRow.Insert(pFV->valAsChain());
		pFV = pFVL->Next();
	    }

	    threadTable.Insert(threadRow);
	    pFVL = dbthreadinfo.Next();
	}

	ListT<Monitor::TableColor> colorMap;		   
	colorMap.Insert(Monitor::TableColor(Chain("BUSY"), MON_RED));
	colorMap.Insert(Monitor::TableColor(Chain("READY"), MON_GREEN));
	colorMap.Insert(Monitor::TableColor(Chain("CONNECTED"), MON_YELLOW));

	ret = showTableBox(Chain("DB Threads"), threadSchema, threadTable, _refInterval, colorMap, doInit);
	doInit = false;
	if ( ret == INFO_ACTION )
	{
	    int selectedThread = getSelectedRow() - 1;
	    showLastThreadAction(selectedThread);
	}
    }
}

void CegoAdmMon::showLastThreadAction(int threadId)
{
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > info;    
    Chain format;
    
    _pAH->getDbThreadLastQuery(threadId, oe, info, format);
    
    ListT<CegoFieldValue> *pFVL = info.First();
    if ( pFVL )
    {
	CegoFieldValue *pFV = pFVL->First();
	if ( pFV )
	{
	    showInfoBox("Last action", pFV->valAsChain(), MSGBOX_WIDTH);
	}
    }
}

void CegoAdmMon::showAdmThread()
{
    bool doInit = true;
    int ret = 1;
    while ( ret != MON_LEAVE )
    {

	ListT<ListT<CegoFieldValue> > admthreadinfo;
	
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;
	Chain format;    
	
	res = _pAH->reqAdmThreadInfo();
	_pAH->getAdmThreadInfo(oe, admthreadinfo, format);


	ListT<Chain> threadSchema;
	threadSchema.Insert(Chain("ThreadId") + VALSEPTOKEN + Chain("10"));
	threadSchema.Insert(Chain("NumRequest") + VALSEPTOKEN + Chain("10"));
	threadSchema.Insert(Chain("Load") + VALSEPTOKEN + Chain("10"));
	threadSchema.Insert(Chain("Status") + VALSEPTOKEN + Chain("10"));
		
	ListT<ListT<Chain> > threadTable;
		   
	ListT<CegoFieldValue> *pFVL = admthreadinfo.First();
	while ( pFVL )
	{
	    ListT<Chain> threadRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    while ( pFV )
	    {
		threadRow.Insert(pFV->valAsChain());
		pFV = pFVL->Next();
	    }

	    threadTable.Insert(threadRow);
	    pFVL = admthreadinfo.Next();
	}
	
	ListT<Monitor::TableColor> colorMap;		   
	// colorMap.Insert(Monitor::TableColor(Chain("Alpha"), MON_RED));

	ret = showTableBox(Chain("Admin Threads"), threadSchema, threadTable, _refInterval, colorMap, doInit);
	doInit = false;

	// int selectedRow = getSelectedRow();
    }
}

void CegoAdmMon::showLogThread()
{
    bool doInit = true;
    int ret = 1;
    while ( ret != MON_LEAVE )
    {
	ListT<ListT<CegoFieldValue> > logthreadinfo;
	
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;
	Chain format;    
    
	res = _pAH->reqLogThreadInfo();
	_pAH->getLogThreadInfo(oe, logthreadinfo, format);

	ListT<Chain> threadSchema;
	threadSchema.Insert(Chain("ThreadId") + VALSEPTOKEN + Chain("10"));
	threadSchema.Insert(Chain("NumRequest") + VALSEPTOKEN + Chain("10"));
	threadSchema.Insert(Chain("Load") + VALSEPTOKEN + Chain("10"));
	threadSchema.Insert(Chain("Status") + VALSEPTOKEN + Chain("10"));
		
	ListT<ListT<Chain> > threadTable;
	
	ListT<CegoFieldValue> *pFVL = logthreadinfo.First();
	while ( pFVL )
	{
	    ListT<Chain> threadRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    while ( pFV )
	    {
		threadRow.Insert(pFV->valAsChain());
		pFV = pFVL->Next();
	    }

	    threadTable.Insert(threadRow);
	    pFVL = logthreadinfo.Next();
	}

	ListT<Monitor::TableColor> colorMap;		   
	// colorMap.Insert(Monitor::TableColor(Chain("Alpha"), MON_RED));

	ret = showTableBox(Chain("Log Threads"), threadSchema, threadTable, _refInterval, colorMap, doInit);
	doInit = false;

	// int selectedRow = getSelectedRow();
    }
}

void CegoAdmMon::showTableSet()
{    
    bool doInit = true;
    int ret = MON_TIMEOUT;
    while ( ret != MON_LEAVE )
    {
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;	

	ListT<ListT<CegoFieldValue> > tsList;
    
	res = _pAH->medGetTableSetList(true);
	_pAH->getTableSetList(oe, tsList, true);

	ListT<Chain> tsSchema;
	tsSchema.Insert(Chain("Name") + VALSEPTOKEN + Chain("20"));
	tsSchema.Insert(Chain("RunState") + VALSEPTOKEN + Chain("10"));
	tsSchema.Insert(Chain("SyncState") + VALSEPTOKEN + Chain("10"));
	tsSchema.Insert(Chain("TempUsage") + VALSEPTOKEN + Chain("10"));
	tsSchema.Insert(Chain("AppUsage") + VALSEPTOKEN + Chain("10"));
		
	ListT<ListT<Chain> > tsTable;
		   
	ListT<CegoFieldValue> *pFVL = tsList.First();
	while ( pFVL )
	{
	    ListT<Chain> tsRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    while ( pFV )
	    {
		tsRow.Insert(pFV->valAsChain());
		pFV = pFVL->Next();
	    }

	    tsTable.Insert(tsRow);
	    pFVL = tsList.Next();
	}

	ListT<Monitor::TableColor> colorMap;		   
	// colorMap.Insert(Monitor::TableColor(Chain("Alpha"), MON_RED));

	ret = showTableBox(Chain("Tablesets"), tsSchema, tsTable, _refInterval, colorMap, doInit);
	doInit = false;

	if ( ret == MON_SELECT)
	{
	    int tsIndex = getSelectedRow() - 1;
	    Chain tableSet = tsTable[tsIndex][0];
	    Chain tsStatus = tsTable[tsIndex][1];
	    int tsAction = showTableSetActionMenu(tsStatus);
	    if ( tsAction > 0 )
	    {
		if ( tsAction == TS_ADD_FILE )
		{
		    showTableSetAddFile(tableSet);
		}
		else if ( tsAction == TS_LIST_FILE )
		{
		    showTableSetFileInfo(tableSet);
		}
		else if ( tsAction == TS_LIST_BACKUP )
		{
		    showTableSetBackupInfo(tableSet);
		}
		else if ( tsAction == TS_ADD_ARCHLOG )
		{
		    showTableSetAddArchlog(tableSet);
		}
		else if ( tsAction == TS_LIST_ARCHLOG )
		{
		    showTableSetListArchlog(tableSet);
		}		
		else if ( tsAction == TS_REMOVE_ARCHLOG )
		{
		    showTableSetRemoveArchlog(tableSet);
		}		
		else if ( tsAction == TS_TOGGLE_ARCHLOG )
		{
		    showTableSetToggleArchlog(tableSet);
		}		
		else if ( tsAction == TS_RESTORE )
		{
		    showTableSetRestore(tableSet);
		}		
		else
		{			
		    performTableSetAction(tableSet, tsAction);
		    if ( tsAction == TS_REMOVE_ACTION )
			setSelectedRow(1);
		}
	    }
	}
	else if ( ret == INFO_ACTION)
	{
	    int tsIndex = getSelectedRow() - 1;
	    Chain tableSet = tsTable[tsIndex][0];
	    showTableSetInfo(tableSet);
	}
	else if ( ret == NEW_ACTION)
	{
	    showTableSetDefineMask();
	}
	else if ( ret == ADD_ACTION)
	{
	    int tsIndex = getSelectedRow() - 1;

	    Chain tableSet = tsTable[tsIndex][0];
	    showTableSetAddFile(tableSet);
	}
	else if ( ret == FILEINFO_ACTION)
	{
	    int tsIndex = getSelectedRow() - 1;

	    Chain tableSet = tsTable[tsIndex][0];
	    showTableSetFileInfo(tableSet);
	}
    }
}

void CegoAdmMon::showTableSetInfo(const Chain& tableSet)
{
    int ret = 1;
    bool doInit = true;
    
    while ( ret != MON_LEAVE )
    {
	ListT< ListT<CegoFieldValue> > tsInfo;
	
	CegoAdminHandler::ResultType res;
	CegoTableObject oe;	
	
	res = _pAH->medGetDetailedTableSetInfo(tableSet);	
	
	while ( res == CegoAdminHandler::ADM_INFO )
	{
	    res = _pAH->nextInfo();	    
	}
	
	_pAH->getDetailedTableSetInfo(oe, tsInfo);
	

	ListT<Chain> tsMonInfo;
	ListT<CegoFieldValue> *pFVL = tsInfo.First();
	while ( pFVL )
	{
	    Chain key, value;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {		       
		key = pFV->valAsChain();
	    }
	    
	    pFV = pFVL->Next();
	    if ( pFV )
	    {
		value = pFV->valAsChain();
	    }
	    
	    tsMonInfo.Insert(key + VALSEPTOKEN + value);	    	    
	    pFVL = tsInfo.Next();
	}

	ListT< ListT<Chain> > tsMonInfoAll;
	tsMonInfoAll.Insert(tsMonInfo);

	ret = showAttributeBox(20, 60, tsMonInfoAll, _refInterval);
	doInit = false;
    }
    return;
}

void CegoAdmMon::showTableSetFileInfo(const Chain& tableSet)
{
    int ret = MON_TIMEOUT;
    while ( ret != MON_LEAVE )
    {
	CegoAdminHandler::ResultType res;	
	res = _pAH->medGetDetailedTableSetInfo(tableSet);	
	
	while ( res == CegoAdminHandler::ADM_INFO )
	{
	    res = _pAH->nextInfo();	    
	}
	
	ListT<ListT<CegoFieldValue> > fileInfo;
	CegoTableObject oe;
	Chain msg;
	Chain format;
	_pAH->getMsg(msg);
	_pAH->getDataFileInfo(oe, fileInfo, format);
	
	int maxPathLen=0;
	ListT<CegoFieldValue>* pFVL = fileInfo.First();	
	while ( pFVL )
	{	    
	    CegoFieldValue *pFV = pFVL->First();
	    if (pFV )
	    {		
		Chain fileName=(char*)pFV->valAsChain();
		if ( fileName.length() > maxPathLen )
		    maxPathLen = fileName.length();		
	    }
	    pFVL = fileInfo.Next();	
	}
	       
	// int width = maxPathLen+30;
	// int height = fileInfo.Size() +3;
		
	ListT<Chain> fiSchema;
	fiSchema.Insert(Chain("Name") + VALSEPTOKEN + Chain("50"));
	fiSchema.Insert(Chain("Type") + VALSEPTOKEN + Chain("10"));
	fiSchema.Insert(Chain("Size") + VALSEPTOKEN + Chain("10"));
	fiSchema.Insert(Chain("Used") + VALSEPTOKEN + Chain("10"));
	
	ListT< ListT<Chain> > fiTable;
	
	pFVL = fileInfo.First();
    	
	while ( pFVL )
	{
	    ListT<Chain> fiRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    while (pFV )
	    {
		fiRow.Insert(pFV->valAsChain());
		pFV = pFVL->Next();
	    }

	    fiTable.Insert(fiRow);
	    
	    pFVL = fileInfo.Next();
	}

	ListT<Monitor::TableColor> colorMap;		   
	// colorMap.Insert(Monitor::TableColor(Chain("Alpha"), MON_RED));

	ret = showTableBox(tableSet + Chain(" files"), fiSchema, fiTable, _refInterval, colorMap, false, false);
    }
}

void CegoAdmMon::showTableSetBackupInfo(const Chain& tableSet)
{
    int ret = MON_TIMEOUT;
    while ( ret != MON_LEAVE )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->medListBackup(tableSet);

	Chain msg;
	_pAH->getMsg(msg);
	
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > backupInfo;
    
	_pAH->getBackupInfo(oe, backupInfo);
	
	ListT<Chain> infoSchema;
	infoSchema.Insert(Chain("BackupId") + VALSEPTOKEN + Chain("50"));
		
	ListT<ListT<Chain> > infoTable;
		   
	ListT<CegoFieldValue> *pFVL = backupInfo.First();
	while ( pFVL )
	{
	    ListT<Chain> backupRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    while ( pFV )
	    {
		backupRow.Insert(pFV->valAsChain());
		pFV = pFVL->Next();
	    }

	    infoTable.Insert(backupRow);
	    pFVL = backupInfo.Next();
	}

	ListT<Monitor::TableColor> colorMap;		   

	ret = showTableBox(Chain("Backup List"), infoSchema, infoTable, _refInterval, colorMap, false, false);
    }
}

void CegoAdmMon::showTableSetDefineMask()
{    
    ListT<Chain> attrList; 
    ListT<Chain> valList; 
    attrList.Insert( Chain("Name") + ATTRSEPTOKEN + INPUT_TYPE_ID + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + Chain("TabNetName"));
    attrList.Insert( Chain("Root Path") + ATTRSEPTOKEN + INPUT_TYPE_STRING + ATTRSEPTOKEN + MAXPATHLEN + ATTRSEPTOKEN + Chain("./RootPath"));
    attrList.Insert( Chain("Primary") + ATTRSEPTOKEN + INPUT_TYPE_ID + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN);
    attrList.Insert( Chain("Secondary") + ATTRSEPTOKEN + INPUT_TYPE_ID + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN);
    attrList.Insert( Chain("SystemSpace") + ATTRSEPTOKEN + INPUT_TYPE_NUMBER + ATTRSEPTOKEN + MAXSIZELEN + ATTRSEPTOKEN + Chain("100"));
    attrList.Insert( Chain("TempSpace") + ATTRSEPTOKEN + INPUT_TYPE_NUMBER + ATTRSEPTOKEN + MAXSIZELEN + ATTRSEPTOKEN + Chain("100"));
    attrList.Insert( Chain("AppSpace") + ATTRSEPTOKEN + INPUT_TYPE_NUMBER + ATTRSEPTOKEN + MAXSIZELEN + ATTRSEPTOKEN + Chain("500"));
    attrList.Insert( Chain("LogSize") + ATTRSEPTOKEN + INPUT_TYPE_NUMBER + ATTRSEPTOKEN + MAXSIZELEN + ATTRSEPTOKEN + Chain("1000000"));
    attrList.Insert( Chain("LogNum") + ATTRSEPTOKEN + INPUT_TYPE_NUMBER + ATTRSEPTOKEN + MAXSIZELEN + ATTRSEPTOKEN + Chain("3"));
    attrList.Insert( Chain("Sort Area Size") + ATTRSEPTOKEN + INPUT_TYPE_NUMBER + ATTRSEPTOKEN + MAXSIZELEN + ATTRSEPTOKEN + Chain("1000000"));
    
    int r = showFormBox("Tableset Definition", attrList, valList);
    
    if ( r )
    {	
	Chain tableSet = valList[0];
	Chain tsRoot = valList[1];
	Chain primary = valList[2];
	Chain secondary = valList[3];
	int sysSize = valList[4].asInteger();
	int tmpSize = valList[5].asInteger();
	int appSize = valList[6].asInteger();
	int logSize = valList[7].asInteger();
	int logNum = valList[8].asInteger();
	unsigned long long sortAreaSize = valList[9].asUnsignedLongLong();

	CegoAdminHandler::ResultType res;
	res = _pAH->medDefineTableSet(tableSet,
				      tsRoot,
				      primary,
				      secondary,
				      sysSize,
				      tmpSize,
				      appSize,
				      logSize,
				      logNum, 
				      sortAreaSize);
	
	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

void CegoAdmMon::showTableSetAddFile(const Chain&  tableSet)
{
    ListT<Chain> attrList; 
    ListT<Chain> valList;

    Chain typeMenuString = Chain("app") + VALSEPTOKEN + Chain("1")
	+ LISTSEPTOKEN + Chain("temp") + VALSEPTOKEN + Chain("2")
	+ LISTSEPTOKEN + Chain("sys") + VALSEPTOKEN + Chain("3");
    
    attrList.Insert( Chain("Type") + ATTRSEPTOKEN + INPUT_TYPE_MENU + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + typeMenuString);
    attrList.Insert( Chain("Path") + ATTRSEPTOKEN + INPUT_TYPE_STRING + ATTRSEPTOKEN + MAXPATHLEN + ATTRSEPTOKEN + Chain("./FilePath"));
    attrList.Insert( Chain("NumPages") + ATTRSEPTOKEN + INPUT_TYPE_NUMBER + ATTRSEPTOKEN + MAXSIZELEN + ATTRSEPTOKEN + Chain("1000"));

    int r = showFormBox(Chain("Add datafile to ") + tableSet, attrList, valList);
        
    if ( r )
    {	
	Chain fileType = valList[0];
	Chain dataFile = valList[1];
	Chain numPages = valList[2];
	
	bool typeCheck = true;
	if ( fileType == Chain("1" ) )
	{
	    fileType = Chain(XML_APPFILE_VALUE);
	}
	else if ( fileType == Chain("2" ) )
	{
	    fileType = Chain(XML_TEMPFILE_VALUE);
	}
	else if ( fileType == Chain("3" ) )
	{
	    fileType = Chain(XML_SYSFILE_VALUE);
	}
	else
	{
	    typeCheck=false;
	    showInfoBox(ERROR_TITLE, "Invalid file type, must be either app, temp or sys", MSGBOX_WIDTH);
	}
	
	if ( typeCheck )
	{
	    CegoAdminHandler::ResultType res;
	
	    res = _pAH->medAddDataFile(tableSet, fileType, dataFile, numPages.asInteger());
	    
	    Chain msg;
	    _pAH->getMsg(msg);
	    
	    if ( res == CegoAdminHandler::ADM_ERROR )
	    {
		showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	    }
	    else
	    {
		showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	    }
	}
    }
}

void CegoAdmMon::showTableSetListArchlog(const Chain& tableSet)
{
    int ret = MON_TIMEOUT;
    while ( ret != MON_LEAVE )
    {
	CegoAdminHandler::ResultType res;	
	res = _pAH->medGetDetailedTableSetInfo(tableSet);	

	while ( res == CegoAdminHandler::ADM_INFO )
	{
	    res = _pAH->nextInfo();	    
	}

	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > archLogInfo;	
	_pAH->getArchLogInfo(oe, archLogInfo);


	int maxPathLen=0;
	ListT<CegoFieldValue>* pFVL = archLogInfo.First();	
	while ( pFVL )
	{	    
	    CegoFieldValue *pFV = pFVL->First();
	    pFV = pFVL->Next();
	    if (pFV )
	    {		
		Chain archLogPath=(char*)pFV->valAsChain();
		
		if ( archLogPath.length() > maxPathLen )
		    maxPathLen = archLogPath.length();		
	    }
	    pFVL = archLogInfo.Next();	
	}
	
	
	// int width = maxPathLen+30;
	// int height = archLogInfo.Size() +3;
	
	
	ListT<Chain> alSchema;
	alSchema.Insert(Chain("ArchId") + VALSEPTOKEN + Chain("20"));
	alSchema.Insert(Chain("Path") + VALSEPTOKEN + Chain("50"));
	
	ListT< ListT<Chain> > alTable;
	
	pFVL = archLogInfo.First();
    	
	while ( pFVL )
	{
	    ListT<Chain> alRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    while (pFV )
	    {
		alRow.Insert(pFV->valAsChain());
		pFV = pFVL->Next();
	    }

	    alTable.Insert(alRow);
	    
	    pFVL = archLogInfo.Next();
	}

	ListT<Monitor::TableColor> colorMap;		   
	// colorMap.Insert(Monitor::TableColor(Chain("Alpha"), MON_RED));

	ret = showTableBox(tableSet + Chain(" archive destinations"), alSchema, alTable, _refInterval, colorMap, false, false);
    }
}

void CegoAdmMon::showTableSetAddArchlog(const Chain& tableSet)
{
    ListT<Chain> attrList; 
    ListT<Chain> valList;

    attrList.Insert( Chain("ArchId") + ATTRSEPTOKEN + INPUT_TYPE_STRING + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + Chain("ALID"));
    attrList.Insert( Chain("Path") + ATTRSEPTOKEN + INPUT_TYPE_STRING + ATTRSEPTOKEN + MAXPATHLEN + ATTRSEPTOKEN + Chain("./FilePath"));

    int r = showFormBox(Chain("Add archive destination to ") + tableSet, attrList, valList);
        
    if ( r )
    {	
	Chain archId = valList[0];
	Chain archPath = valList[1];

	CegoAdminHandler::ResultType res;    
	res = _pAH->medAddArchLog(tableSet, archId, archPath);
	    
	Chain msg;
	_pAH->getMsg(msg);
	
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}      
    }
}

void CegoAdmMon::showTableSetRemoveArchlog(const Chain& tableSet)
{
    CegoAdminHandler::ResultType res;	
    res = _pAH->medGetDetailedTableSetInfo(tableSet);	
    
    while ( res == CegoAdminHandler::ADM_INFO )
    {
	res = _pAH->nextInfo();	    
    }
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > archLogInfo;	
    _pAH->getArchLogInfo(oe, archLogInfo);
    
    Chain archIdMenuString; 
    
    ListT<CegoFieldValue> *pFVL = archLogInfo.First(); 
    while ( pFVL ) 
    {	
	CegoFieldValue *pFV = pFVL->First(); 
	if ( pFV ) 
	{		
	    archIdMenuString += pFV->valAsChain() + VALSEPTOKEN + pFV->valAsChain(); 
	}
	pFVL = archLogInfo.Next(); 
	
	if ( pFVL) 
	    archIdMenuString += LISTSEPTOKEN; 
    } 
    
    ListT<Chain> attrList; 
    ListT<Chain> valList; 
    
    attrList.Insert(Chain("ArchId") + ATTRSEPTOKEN + INPUT_TYPE_MENU + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + archIdMenuString); 
    
    int r = showFormBox(Chain("Remove archive distination from ") + tableSet, attrList, valList);
    
    if ( r )
    {
	Chain archId = valList[0]; 
	
	CegoAdminHandler::ResultType res;   
	res = _pAH->medRemoveArchLog(tableSet, archId);
	
	
	Chain msg;
	_pAH->getMsg(msg);
	
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

void CegoAdmMon::showTableSetToggleArchlog(const Chain& tableSet)
{
    ListT<Chain> attrList; 
    ListT<Chain> valList;

    Chain modeMenuString = Chain("Enabled") + VALSEPTOKEN + Chain("E")
	+ LISTSEPTOKEN + Chain("Disabled") + VALSEPTOKEN + Chain("D");

    attrList.Insert( Chain("Status") + ATTRSEPTOKEN + INPUT_TYPE_MENU + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + modeMenuString);
    
    int r = showFormBox(Chain("Toggle archlog for ") + tableSet, attrList, valList);
        
    if ( r )
    {	
	Chain mode = valList[0];

	CegoAdminHandler::ResultType res;   
	if ( mode == Chain("E"))
	{
	    res = _pAH->medEnableArchLog(tableSet);
	}
	else
	{
	    res = _pAH->medDisableArchLog(tableSet);
	}
	Chain msg;
	_pAH->getMsg(msg);
	
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

void CegoAdmMon::showTableSetRestore(const Chain& tableSet)
{
    bool doInit = true;
    int ret = 1;
    while ( ret != MON_LEAVE )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->medListBackup(tableSet);

	Chain msg;
	_pAH->getMsg(msg);
	
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > backupInfo;
    
	_pAH->getBackupInfo(oe, backupInfo);
	
	ListT<Chain> infoSchema;
	infoSchema.Insert(Chain("BackupId") + VALSEPTOKEN + Chain("50"));
		
	ListT<ListT<Chain> > infoTable;
		   
	ListT<CegoFieldValue> *pFVL = backupInfo.First();
	while ( pFVL )
	{
	    ListT<Chain> backupRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    while ( pFV )
	    {
		backupRow.Insert(pFV->valAsChain());
		pFV = pFVL->Next();
	    }

	    infoTable.Insert(backupRow);
	    pFVL = backupInfo.Next();
	}

	ListT<Monitor::TableColor> colorMap;		   

	ret = showTableBox(Chain("Backup List"), infoSchema, infoTable, _refInterval, colorMap, doInit);
	doInit = false;
	if ( ret == RESTORE_ACTION )
	{
	    int selectedBackup = getSelectedRow() - 1;

	    // Chain msg = Chain("Triggering backup for ") +  infoTable[selectedBackup][0];

	    Chain backupId = infoTable[selectedBackup][0];
	    CegoAdminHandler::ResultType res;
	    res = _pAH->medRestore(tableSet, backupId);
	    
	    Chain msg;
	    
	    while ( res == CegoAdminHandler::ADM_INFO )
	    {	
		_pAH->getMsg(msg);
		res = _pAH->nextInfo();
	    }
	    	   	 
	    if ( res == CegoAdminHandler::ADM_ERROR )
	    {
		showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	    }
	    else
	    {
		showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	    }	   
	}
    }
}

void CegoAdmMon::performTableSetAction(const Chain& tableSet, int action)
{
    if ( action == TS_START_ACTION )
    {
	bool doCleanup=false;
	bool doForceload=false;
	bool cpDump=false;
	bool noInit=false;
	
	CegoAdminHandler::ResultType res;
	res = _pAH->medStartTableSet(tableSet, doCleanup, doForceload, cpDump, noInit);
		
	Chain msg;
	_pAH->getMsg(msg);

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}	
    }
    else if ( action == TS_STOP_ACTION )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->medStopTableSet(tableSet);
		
	Chain msg;
	_pAH->getMsg(msg);

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
    else if ( action == TS_CREATE_ACTION )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->medCreateTableSet(tableSet);
		
	Chain msg;
	_pAH->getMsg(msg);

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
    else if ( action == TS_DROP_ACTION )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->medDropTableSet(tableSet);
		
	Chain msg;
	_pAH->getMsg(msg);

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
    else if ( action == TS_RECOVER_ACTION )
    {
	CegoAdminHandler::ResultType res;

	Chain pit;
	res = _pAH->medRecover(tableSet, pit);

	Chain msg;
	_pAH->getMsg(msg);

	while ( res == CegoAdminHandler::ADM_INFO )
	{	
	    _pAH->getMsg(msg);
	    res = _pAH->nextInfo();
	}

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }    
    else if ( action == TS_REMOVE_ACTION )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->medRemoveTableSet(tableSet);
		
	Chain msg;
	_pAH->getMsg(msg);

	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
    else if ( action == TS_BACKUP_ACTION )
    {
	CegoAdminHandler::ResultType res;
	Chain msg;

	Chain buMsg("Backup via CegoAdmMon");
	
	res = _pAH->medBeginBackup(tableSet, buMsg);	
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    _pAH->getMsg(msg);
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}

	res = _pAH->medExecuteBackup(tableSet, buMsg);
	
	while ( res == CegoAdminHandler::ADM_INFO )
	{
	    res = _pAH->nextInfo();	    
	}
	
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    _pAH->getMsg(msg);
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}

	res = _pAH->medEndBackup(tableSet, buMsg, false);


	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    _pAH->getMsg(msg);
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    msg = Chain("Backup finished succesful");
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

void CegoAdmMon::showRefreshMenu()
{
    ListT<Chain> refreshList;
    refreshList.Insert(Chain("1 sec") + VALSEPTOKEN + Chain("1000"));
    refreshList.Insert(Chain("3 sec") + VALSEPTOKEN + Chain("3000"));
    refreshList.Insert(Chain("10 sec") + VALSEPTOKEN + Chain("10000"));	
        
    _refInterval = showSelectBox("Select refresh interval", refreshList);
}

int CegoAdmMon::showTableSetActionMenu(const Chain& tsStatus)
{
    ListT<Chain> tsActionList;	       	     	           
    if ( tsStatus == Chain(TS_DEFINED_STATUS) )
    {
	tsActionList.Insert(Chain("Create tableset") + VALSEPTOKEN + Chain(TS_CREATE_ACTION));
	tsActionList.Insert(Chain("Remove tableset") + VALSEPTOKEN + Chain(TS_REMOVE_ACTION));
    }
    else if ( tsStatus == Chain(TS_OFFLINE_STATUS) )
    {	
	tsActionList.Insert(Chain("Start tableset") + VALSEPTOKEN + Chain(TS_START_ACTION));
	tsActionList.Insert(Chain("Drop tableset") + VALSEPTOKEN + Chain(TS_DROP_ACTION));
	tsActionList.Insert(Chain("Restore tableset") + VALSEPTOKEN + Chain(TS_RESTORE));
	tsActionList.Insert(Chain("Recover tableset") + VALSEPTOKEN + Chain(TS_RECOVER_ACTION));
    }
    else if ( tsStatus == Chain(TS_ONLINE_STATUS) )
    {
	tsActionList.Insert(Chain("Stop tableset") + VALSEPTOKEN + Chain(TS_STOP_ACTION));
	tsActionList.Insert(Chain("Online backup") + VALSEPTOKEN + Chain(TS_BACKUP_ACTION));
    }

    tsActionList.Insert(Chain("List backup") + VALSEPTOKEN + Chain(TS_LIST_BACKUP));
    
    tsActionList.Insert(Chain("List datafile") + VALSEPTOKEN + Chain(TS_LIST_FILE));
    tsActionList.Insert(Chain("Add datafile") + VALSEPTOKEN + Chain(TS_ADD_FILE));

    tsActionList.Insert(Chain("List archlog") + VALSEPTOKEN + Chain(TS_LIST_ARCHLOG));
    tsActionList.Insert(Chain("Add archlog") + VALSEPTOKEN + Chain(TS_ADD_ARCHLOG));
    tsActionList.Insert(Chain("Remove archlog") + VALSEPTOKEN + Chain(TS_REMOVE_ARCHLOG));

    tsActionList.Insert(Chain("Toggle archlog") + VALSEPTOKEN + Chain(TS_TOGGLE_ARCHLOG));

    return showSelectBox("Tableset Action", tsActionList);
}

void CegoAdmMon::showTableSetHelp()
{
    /*
    Chain title("Tableset Help Menu");

    int width = 70;
    int height = 10;
 
    int highlight=1;
    int choice = 0;
    int c;
    
    int startx = (_mainx - width) / 2;
    int starty = (_mainy - height) / 2;
    
    WINDOW* helpwin = newwin(height, width, starty, startx);

    noecho();

    keypad(helpwin, TRUE);

    refresh();
    
    int x, y, i;	
    wattron(helpwin, A_BOLD);
    mvwprintw(helpwin, 1, 1, "%s", (char*)title);
    wattroff(helpwin, A_BOLD);


    int row=3;
    int sepCol=18;
    int descCol=20;
    
    box(helpwin, 0, 0);
    
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Down");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select next tableset");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Up");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select previous tableset");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"i");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Get detailed information about selected tableset");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"n");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Define new tableset");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"RET");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Open tableset submenu");

    bool showIt=true;

    while( showIt )
    {	
	c = wgetch(helpwin);
	switch(c)
	{	
	case KEY_RETURN:
	{
	    showIt = false;
	    choice = highlight;
	    break;
	}
	case KEY_ESC:
	{
	    showIt = false;
	    break;
	}
	}
    }
    delwin(helpwin);

    */
    return;
	
}

void CegoAdmMon::showUser()
{
    bool doInit = true;
    int ret = MON_TIMEOUT;
    while ( ret != MON_LEAVE )
    {
	CegoAdminHandler::ResultType res;	
	res = _pAH->reqShowUser();

	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > userList;
	
	_pAH->getUserInfo(oe, userList);
	
	ListT<Chain> userSchema;
	userSchema.Insert(Chain("Name") + VALSEPTOKEN + Chain("30"));
	userSchema.Insert(Chain("Role") + VALSEPTOKEN + Chain("40"));
	userSchema.Insert(Chain("Trace") + VALSEPTOKEN + Chain("10"));
	userSchema.Insert(Chain("NumRequest") + VALSEPTOKEN + Chain("15"));
	userSchema.Insert(Chain("NumQuery") + VALSEPTOKEN + Chain("15"));
		
	ListT< ListT<Chain> > userTable;
		   
	ListT<CegoFieldValue> *pFVL = userList.First();
	while ( pFVL )
	{
	    ListT<Chain> userRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    while ( pFV )
	    {
		userRow.Insert(pFV->valAsChain());
		pFV = pFVL->Next();
	    }

	    userTable.Insert(userRow);
	    pFVL = userList.Next();
	}

	ListT<Monitor::TableColor> colorMap;		   
	// colorMap.Insert(Monitor::TableColor(Chain("Alpha"), MON_RED));

	ret = showTableBox(Chain("Users"), userSchema, userTable, _refInterval, colorMap, doInit);
	doInit = false;

	if ( ret == MON_SELECT)
	{
	    int userIndex = getSelectedRow() - 1;
	    Chain usName = userTable[userIndex][0];
	    Chain roleListString = userTable[userIndex][1];
	    Chain usTrace = userTable[userIndex][2];
	    int usAction = showUserActionMenu();
	    if ( usAction > 0 )
	    {
		if ( usAction == USER_ASSIGNROLE_ACTION )
		{
		    ListT<Chain> roleList = roleString2List(roleListString);
		    showRoleAssignMask(usName, roleList);
		}
		else if ( usAction == USER_REMOVEROLE_ACTION )
		{
		    ListT<Chain> roleList = roleString2List(roleListString);
		    showRoleRemoveMask(usName, roleList);
		}
		else if ( usAction == USER_CHANGEPWD_ACTION )
		{
		    showUserChangePwd(usName);
		}
		else if ( usAction == USER_TOGGLETRACE_ACTION )
		{		    
		    CegoAdminHandler::ResultType res;

		    if ( usTrace == Chain("ON"))
		    {
			res = _pAH->reqUserTrace(usName, false);
		    }
		    else
		    {
			res = _pAH->reqUserTrace(usName, true);
		    }
		    
		    Chain msg;
		    _pAH->getMsg(msg);
		    
		    if ( res == CegoAdminHandler::ADM_ERROR )
		    {
			showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
		    }
		    else
		    {
			showInfoBox(INFO_TITLE, Chain("User trace enabled"), MSGBOX_WIDTH);
		    }			   
		}
		else if ( usAction == USER_DELETE_ACTION )
		{

		    CegoAdminHandler::ResultType res;
		    
		    res = _pAH->reqRemoveUser(usName);
		    
		    Chain msg;
		    _pAH->getMsg(msg);
		    if ( res == CegoAdminHandler::ADM_ERROR )
		    {
			showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
		    }
		    else
		    {
			showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
		    }

		    setSelectedRow(1);
		}
	    }		
	}
	else if ( ret == NEW_ACTION)
	{
	    showUserAddMask();
	}
    }
}

void CegoAdmMon::showUserAddMask()
{
    ListT<Chain> attrList;
    ListT<Chain> valList;
    
    attrList.Insert(Chain("Name") + ATTRSEPTOKEN + INPUT_TYPE_ID + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + Chain("Username"));
    attrList.Insert(Chain("Passwd") + ATTRSEPTOKEN + INPUT_TYPE_PWD + ATTRSEPTOKEN + MAXPASSWDLEN + ATTRSEPTOKEN + Chain());
         
    int r = showFormBox("User Definition", attrList, valList);
    
    if ( r )
    {
	
	Chain userName = valList[0];
	Chain passwd = valList[1];


	CegoAdminHandler::ResultType res;
    
	res = _pAH->reqAddUser(userName, passwd);
	
	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

int CegoAdmMon::showUserActionMenu()
{
    ListT<Chain> userActionList;	       	     	           
    
    userActionList.Insert(Chain("Toggle trace") + VALSEPTOKEN + Chain(USER_TOGGLETRACE_ACTION));
    userActionList.Insert(Chain("Assign role") + VALSEPTOKEN + Chain(USER_ASSIGNROLE_ACTION));
    userActionList.Insert(Chain("Remove role") + VALSEPTOKEN + Chain(USER_REMOVEROLE_ACTION));
    userActionList.Insert(Chain("Change Password") + VALSEPTOKEN + Chain(USER_CHANGEPWD_ACTION));       
    userActionList.Insert(Chain("Delete") + VALSEPTOKEN + Chain(USER_DELETE_ACTION));

    return showSelectBox("User Action", userActionList);
}

void CegoAdmMon::showRole()
{
    int ret = MON_TIMEOUT;
    bool doInit = true;
    while ( ret != MON_LEAVE )
    {         
	CegoAdminHandler::ResultType res;	
	res = _pAH->reqListRole();
	
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > roleList;	
	_pAH->getRoleList(oe, roleList);

	ListT<Chain> roleSchema;
	roleSchema.Insert(Chain("Name") + VALSEPTOKEN + Chain("10"));

	ListT< ListT<Chain> > roleTable;	
	ListT<CegoFieldValue> *pFVL = roleList.First();
	while ( pFVL )
	{
	    ListT<Chain> roleRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    if ( pFV )
	    {
		roleRow.Insert(pFV->valAsChain());		
	    }
	    pFVL = roleList.Next();

	    roleTable.Insert(roleRow);
	}
	
	ListT<Monitor::TableColor> colorMap;		   
	// colorMap.Insert(Monitor::TableColor(Chain("Alpha"), MON_RED));
	
	ret = showTableBox(Chain("Roles"), roleSchema, roleTable, _refInterval, colorMap, doInit);
	doInit = false;
	if ( ret == NEW_ACTION )
	{
	    showRoleAddMask();
	}       
	else if ( ret == MON_SELECT )
	{
	    int roleIdx = getSelectedRow() - 1;	    

	    Chain roleName = roleTable[roleIdx][0];	    
	    int rlAction = showRoleActionMenu();
	    if ( rlAction > 0 )
	    {
		if ( rlAction == ROLE_PERM_LIST_ACTION )
		{
		    showRolePermListMask(roleName);
		}
		else if ( rlAction == ROLE_PERM_SET_ACTION )
		{
		    showRolePermSetMask(roleName);
		}
		else if ( rlAction == ROLE_PERM_REMOVE_ACTION )
		{
		    showRolePermRemoveMask(roleName);
		}
		else if ( rlAction == ROLE_DROP_ACTION )
		{
		    CegoAdminHandler::ResultType res;
		    
		    res = _pAH->reqDropRole(roleName);
		    
		    Chain msg;
		    _pAH->getMsg(msg);
		    if ( res == CegoAdminHandler::ADM_ERROR )
		    {
			showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
		    }
		    else
		    {
			showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
		    }		    		   		    
		    setSelectedRow(1);
		}
	    }
	}
    }
}

void CegoAdmMon::showRoleAddMask()
{
    ListT<Chain> attrList;
    ListT<Chain> valList;
    
    attrList.Insert(Chain("Name") + ATTRSEPTOKEN + INPUT_TYPE_ID + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + Chain("RoleName"));
         
    int r = showFormBox("Role Definition", attrList, valList);
    
    if ( r )
    {	
	Chain roleName = valList[0];

	CegoAdminHandler::ResultType res;
    
	res = _pAH->reqCreateRole(roleName);
  	
	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

void CegoAdmMon::showRoleAssignMask(const Chain& userName, const ListT<Chain>& userRoleList) 
{ 
    ListT<Chain> attrList; 
    ListT<Chain> valList; 

    CegoAdminHandler::ResultType res; 
    res = _pAH->reqListRole(); 

    CegoTableObject oe; 
    ListT<ListT<CegoFieldValue> > roleList; 
    _pAH->getRoleList(oe, roleList); 

    ListT<CegoFieldValue> adminRoleList;
    adminRoleList.Insert(CegoFieldValue(VARCHAR_TYPE, XML_ADMIN_ROLE));
    roleList.Insert(adminRoleList);

    int ridx=0;
    Chain roleMenuString;

    ListT<CegoFieldValue> *pFVL = roleList.First(); 
    while ( pFVL ) 
    {
	bool roleAdded=false;
        CegoFieldValue *pFV = pFVL->First(); 
        if ( pFV ) 
        {
	    if ( userRoleList.Find(pFV->valAsChain()) == 0)
	    {
		roleAdded=true;
		roleMenuString += pFV->valAsChain() + VALSEPTOKEN + Chain(ridx); 
	    }
	}
	pFVL = roleList.Next(); 
        if ( pFVL && roleAdded) 
             roleMenuString += LISTSEPTOKEN; 
        ridx++; 
    } 

    attrList.Insert(Chain("Role") + ATTRSEPTOKEN + INPUT_TYPE_MENU + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + roleMenuString); 

    int r = showFormBox(Chain("Assign role to ") + userName, attrList, valList); 
    
    if ( r ) 
    {
        int ridx = valList[0].asInteger(); 
        Chain roleName = roleList[ridx][0].valAsChain(); 

	
        CegoAdminHandler::ResultType res; 

        res = _pAH->reqAssignRole(userName, roleName);
	
	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

void CegoAdmMon::showRoleRemoveMask(const Chain& userName, const ListT<Chain>& userRoleList) 
{
    ListT<Chain> attrList; 
    ListT<Chain> valList; 

    CegoAdminHandler::ResultType res; 
    res = _pAH->reqListRole(); 

    CegoTableObject oe; 
    ListT<ListT<CegoFieldValue> > roleList; 
    _pAH->getRoleList(oe, roleList); 

    ListT<CegoFieldValue> adminRoleList;
    adminRoleList.Insert(CegoFieldValue(VARCHAR_TYPE, XML_ADMIN_ROLE));
    roleList.Insert(adminRoleList);

    Chain roleMenuString; 
    int ridx=0; 

    ListT<CegoFieldValue> *pFVL = roleList.First(); 
    while ( pFVL ) 
    {
	bool roleAdded=false;
        CegoFieldValue *pFV = pFVL->First(); 
        if ( pFV ) 
        {
	    if ( userRoleList.Find(pFV->valAsChain()) != 0 )
	    {
		roleAdded=true;
		roleMenuString += pFV->valAsChain() + VALSEPTOKEN + Chain(ridx); 
	    }
	}
	pFVL = roleList.Next(); 
        if ( pFVL && roleAdded) 
             roleMenuString += LISTSEPTOKEN; 
        ridx++; 
    } 

    attrList.Insert(Chain("Role") + ATTRSEPTOKEN + INPUT_TYPE_MENU + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + roleMenuString); 
         
    int r = showFormBox(Chain("Remove role from ") + userName, attrList, valList);

    if ( r )
    {
	int ridx = valList[0].asInteger(); 
        Chain roleName = roleList[ridx][0].valAsChain();
	
	CegoAdminHandler::ResultType res;
	
	res = _pAH->reqRemoveRole(userName, roleName);

	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

void CegoAdmMon::showUserChangePwd(const Chain& userName)
{    
    ListT<Chain> attrList;
    ListT<Chain> valList;

    attrList.Insert(Chain("Password") + ATTRSEPTOKEN + INPUT_TYPE_PWD + ATTRSEPTOKEN + MAXPASSWDLEN + ATTRSEPTOKEN + Chain());
         
    int r = showFormBox(Chain("Set password for ") + userName, attrList, valList);
    
    if ( r )
    {	
	Chain pwd = valList[0];
	
	CegoAdminHandler::ResultType res;
	
	res = _pAH->reqChangePwd(userName, pwd);

	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

int CegoAdmMon::showRoleActionMenu()
{
    ListT<Chain> roleActionList;
    roleActionList.Insert(Chain("List permission") + VALSEPTOKEN + Chain(ROLE_PERM_LIST_ACTION));
    roleActionList.Insert(Chain("Set permission") + VALSEPTOKEN + Chain(ROLE_PERM_SET_ACTION));
    roleActionList.Insert(Chain("Remove permission") + VALSEPTOKEN + Chain(ROLE_PERM_REMOVE_ACTION));
    roleActionList.Insert(Chain("Drop role") + VALSEPTOKEN + Chain(ROLE_DROP_ACTION));

    return showSelectBox("Role Action", roleActionList);
}

void CegoAdmMon::showRolePermListMask(const Chain& role)
{
    bool doInit = true;
    int ret = 1;
    while ( ret != MON_LEAVE )
    {
	CegoAdminHandler::ResultType res;
	res = _pAH->reqShowRole(role);
		
	CegoTableObject oe;
	ListT<ListT<CegoFieldValue> > permInfo;
	
	_pAH->getRoleInfo(oe, permInfo);

	Chain msg;
	_pAH->getMsg(msg);
	
	ListT<Chain> permSchema;
	permSchema.Insert(Chain("PermId") + VALSEPTOKEN + Chain("20"));
	permSchema.Insert(Chain("TableSet") + VALSEPTOKEN + Chain("20"));
	permSchema.Insert(Chain("Filter") + VALSEPTOKEN + Chain("20"));
	permSchema.Insert(Chain("Permission") + VALSEPTOKEN + Chain("20"));
		
	ListT<ListT<Chain> > permTable;
		   
	ListT<CegoFieldValue> *pFVL = permInfo.First();
	while ( pFVL )
	{
	    ListT<Chain> permRow;
	    
	    CegoFieldValue *pFV = pFVL->First();
	    while ( pFV )
	    {
		permRow.Insert(pFV->valAsChain());
		pFV = pFVL->Next();
	    }

	    permTable.Insert(permRow);
	    pFVL = permInfo.Next();
	}

	ListT<Monitor::TableColor> colorMap;		   
	// colorMap.Insert(Monitor::TableColor(Chain("Alpha"), MON_RED));
	
	ret = showTableBox(role + Chain(" permissions"), permSchema, permTable, _refInterval, colorMap, doInit);
	doInit = false;
    }
}

void CegoAdmMon::showRolePermSetMask(const Chain& role)
{    
    ListT<Chain> attrList;
    ListT<Chain> valList;

    Chain rightMenuString = Chain("READ") + VALSEPTOKEN + Chain("1")
	+ LISTSEPTOKEN + Chain("WRITE") + VALSEPTOKEN + Chain("2")
	+ LISTSEPTOKEN + Chain("MODIFY") + VALSEPTOKEN + Chain("3")
	+ LISTSEPTOKEN + Chain("EXEC") + VALSEPTOKEN + Chain("4")
	+ LISTSEPTOKEN + Chain("ALL") + VALSEPTOKEN + Chain("5");
        
    attrList.Insert(Chain("PermId") + ATTRSEPTOKEN + INPUT_TYPE_ID + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + Chain("P1"));
    attrList.Insert(Chain("TableSet") + ATTRSEPTOKEN + INPUT_TYPE_ID + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + Chain("TS1"));
    attrList.Insert(Chain("Filter") + ATTRSEPTOKEN + INPUT_TYPE_STRING + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + Chain("ALL"));
    attrList.Insert( Chain("Right") + ATTRSEPTOKEN + INPUT_TYPE_MENU + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + rightMenuString);
         
    int r = showFormBox("Permission Definition", attrList, valList);
    
    if ( r )
    {	
	Chain permId = valList[0];
	Chain tableSet = valList[1];
	Chain filterPerm = valList[2];
	Chain rightId = valList[3];
	Chain rightPerm;

	if ( rightId == Chain("1" ) )
	{
	    rightPerm = Chain("READ");
	}
	else if ( rightId == Chain("2" ) )
	{
	    rightPerm = Chain("WRITE");
	}
	else if ( rightId == Chain("3" ) )
	{
	    rightPerm = Chain("MODIFY");
	}
	else if ( rightId == Chain("4" ) )
	{
	    rightPerm = Chain("EXEC");
	}
	else if ( rightId == Chain("5" ) )
	{
	    rightPerm = Chain("ALL");
	}

	CegoAdminHandler::ResultType res;
	    
	res = _pAH->reqSetPermission(role, permId, tableSet, filterPerm, rightPerm);
	
	Chain msg;
	_pAH->getMsg(msg);
	    
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

void CegoAdmMon::showRolePermRemoveMask(const Chain& role)
{
    
    CegoAdminHandler::ResultType res;
    res = _pAH->reqShowRole(role);
    
    CegoTableObject oe;
    ListT<ListT<CegoFieldValue> > permInfo;
    
    _pAH->getRoleInfo(oe, permInfo);
    
    Chain msg;
    _pAH->getMsg(msg);
    
    Chain permIdString;

    ListT<CegoFieldValue> *pFVL = permInfo.First();
    while ( pFVL )
    {
	ListT<Chain> permRow;
	
	CegoFieldValue *pFV = pFVL->First();
	if ( pFV )
	{
	    permIdString = permIdString + pFV->valAsChain() + VALSEPTOKEN + pFV->valAsChain();
	}
	pFVL = permInfo.Next();
	if ( pFVL )
	    permIdString = permIdString + LISTSEPTOKEN;
    }

    ListT<Chain> attrList;
    ListT<Chain> valList;

    attrList.Insert( Chain("PermId") + ATTRSEPTOKEN + INPUT_TYPE_MENU + ATTRSEPTOKEN + MAXNAMELEN + ATTRSEPTOKEN + permIdString);
         
    int r = showFormBox("Delete Permission", attrList, valList);
    
    if ( r )
    {	
	Chain permId = valList[0];
		
	CegoAdminHandler::ResultType res;
	res = _pAH->reqRemovePermission(role, permId);

	Chain msg;
	_pAH->getMsg(msg);
	
	if ( res == CegoAdminHandler::ADM_ERROR )
	{
	    showInfoBox(ERROR_TITLE, msg, MSGBOX_WIDTH);
	}
	else
	{
	    showInfoBox(INFO_TITLE, msg, MSGBOX_WIDTH);
	}
    }
}

ListT<Chain> CegoAdmMon::roleString2List(const Chain& roleString)
{
    Tokenizer t(roleString, Chain(","));

    Chain role;
    ListT<Chain> roleList;
    
    while ( t.nextToken(role))
	roleList.Insert(role);

    return roleList;		
}

/*
void CegoAdmMon::showHelp()
{

    int width = HELPMASK_WIDTH;
    int height = HELPMASK_HEIGHT;
 
    int c;

    int startx = (_mainx - width) / 2;
    int starty = (_mainy - height) / 2;
    
    WINDOW *helpwin = newwin(height, width, starty, startx);

    noecho();

    keypad(helpwin, TRUE);

    wattron(helpwin, A_UNDERLINE);
    mvwprintw(helpwin, 1, 1, "Available commands");
    wattroff(helpwin, A_UNDERLINE);

    box(helpwin, 0, 0);

    int row=3;
    int sepCol=18;
    int descCol=20;

    wattron(helpwin, A_BOLD);
    mvwprintw(helpwin, row, 1, "%s", (char*)"Header Menu");
    wattroff(helpwin, A_BOLD);

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Left");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select left header menu");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Right");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select right header menu");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Return");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Open selected menu");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Esc");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Go back");

    row++;
    row++;
    wattron(helpwin, A_BOLD);
    mvwprintw(helpwin, row, 1, "%s", (char*)"Sub Menu");
    wattroff(helpwin, A_BOLD);

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Return");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Open option menu");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Esc");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Go back");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"n");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"New entry");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"i");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Get information about entry");

    row++;
    row++;
    wattron(helpwin, A_BOLD);
    mvwprintw(helpwin, row, 1, "%s", (char*)"Selection Menu");
    wattroff(helpwin, A_BOLD);

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Return");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select item action");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Esc");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Go back");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Down");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select next item");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Up");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select previous item");

    row++;
    row++;
    wattron(helpwin, A_BOLD);
    mvwprintw(helpwin, row, 1, "%s", (char*)"Input Mask");
    wattroff(helpwin, A_BOLD);

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Tab,Cursor Down");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select next field");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Up");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Select previous field");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Left");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Move input cursor left");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Cursor Right");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Move input cursor right");

    row++;
    mvwprintw(helpwin, row, 1, "%s", (char*)"Return");
    mvwprintw(helpwin, row, sepCol, "%s", (char*)"-");
    mvwprintw(helpwin, row, descCol, "%s", (char*)"Confirm or Abort");

    refresh();

    c = wgetch(helpwin);
   
    delwin(helpwin);

    clear();

}

*/
