﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/OperatingSystem.h>
#include <aws/ssm/model/PatchAction.h>
#include <aws/ssm/model/PatchComplianceLevel.h>
#include <aws/ssm/model/PatchComplianceStatus.h>
#include <aws/ssm/model/PatchFilterGroup.h>
#include <aws/ssm/model/PatchRuleGroup.h>
#include <aws/ssm/model/PatchSource.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace SSM {
namespace Model {
class GetPatchBaselineResult {
 public:
  AWS_SSM_API GetPatchBaselineResult() = default;
  AWS_SSM_API GetPatchBaselineResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_SSM_API GetPatchBaselineResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>The ID of the retrieved patch baseline.</p>
   */
  inline const Aws::String& GetBaselineId() const { return m_baselineId; }
  template <typename BaselineIdT = Aws::String>
  void SetBaselineId(BaselineIdT&& value) {
    m_baselineIdHasBeenSet = true;
    m_baselineId = std::forward<BaselineIdT>(value);
  }
  template <typename BaselineIdT = Aws::String>
  GetPatchBaselineResult& WithBaselineId(BaselineIdT&& value) {
    SetBaselineId(std::forward<BaselineIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the patch baseline.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  GetPatchBaselineResult& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Returns the operating system specified for the patch baseline.</p>
   */
  inline OperatingSystem GetOperatingSystem() const { return m_operatingSystem; }
  inline void SetOperatingSystem(OperatingSystem value) {
    m_operatingSystemHasBeenSet = true;
    m_operatingSystem = value;
  }
  inline GetPatchBaselineResult& WithOperatingSystem(OperatingSystem value) {
    SetOperatingSystem(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A set of global filters used to exclude patches from the baseline.</p>
   */
  inline const PatchFilterGroup& GetGlobalFilters() const { return m_globalFilters; }
  template <typename GlobalFiltersT = PatchFilterGroup>
  void SetGlobalFilters(GlobalFiltersT&& value) {
    m_globalFiltersHasBeenSet = true;
    m_globalFilters = std::forward<GlobalFiltersT>(value);
  }
  template <typename GlobalFiltersT = PatchFilterGroup>
  GetPatchBaselineResult& WithGlobalFilters(GlobalFiltersT&& value) {
    SetGlobalFilters(std::forward<GlobalFiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A set of rules used to include patches in the baseline.</p>
   */
  inline const PatchRuleGroup& GetApprovalRules() const { return m_approvalRules; }
  template <typename ApprovalRulesT = PatchRuleGroup>
  void SetApprovalRules(ApprovalRulesT&& value) {
    m_approvalRulesHasBeenSet = true;
    m_approvalRules = std::forward<ApprovalRulesT>(value);
  }
  template <typename ApprovalRulesT = PatchRuleGroup>
  GetPatchBaselineResult& WithApprovalRules(ApprovalRulesT&& value) {
    SetApprovalRules(std::forward<ApprovalRulesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of explicitly approved patches for the baseline.</p>
   */
  inline const Aws::Vector<Aws::String>& GetApprovedPatches() const { return m_approvedPatches; }
  template <typename ApprovedPatchesT = Aws::Vector<Aws::String>>
  void SetApprovedPatches(ApprovedPatchesT&& value) {
    m_approvedPatchesHasBeenSet = true;
    m_approvedPatches = std::forward<ApprovedPatchesT>(value);
  }
  template <typename ApprovedPatchesT = Aws::Vector<Aws::String>>
  GetPatchBaselineResult& WithApprovedPatches(ApprovedPatchesT&& value) {
    SetApprovedPatches(std::forward<ApprovedPatchesT>(value));
    return *this;
  }
  template <typename ApprovedPatchesT = Aws::String>
  GetPatchBaselineResult& AddApprovedPatches(ApprovedPatchesT&& value) {
    m_approvedPatchesHasBeenSet = true;
    m_approvedPatches.emplace_back(std::forward<ApprovedPatchesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Returns the specified compliance severity level for approved patches in the
   * patch baseline.</p>
   */
  inline PatchComplianceLevel GetApprovedPatchesComplianceLevel() const { return m_approvedPatchesComplianceLevel; }
  inline void SetApprovedPatchesComplianceLevel(PatchComplianceLevel value) {
    m_approvedPatchesComplianceLevelHasBeenSet = true;
    m_approvedPatchesComplianceLevel = value;
  }
  inline GetPatchBaselineResult& WithApprovedPatchesComplianceLevel(PatchComplianceLevel value) {
    SetApprovedPatchesComplianceLevel(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the list of approved patches includes non-security updates
   * that should be applied to the managed nodes. The default value is
   * <code>false</code>. Applies to Linux managed nodes only.</p>
   */
  inline bool GetApprovedPatchesEnableNonSecurity() const { return m_approvedPatchesEnableNonSecurity; }
  inline void SetApprovedPatchesEnableNonSecurity(bool value) {
    m_approvedPatchesEnableNonSecurityHasBeenSet = true;
    m_approvedPatchesEnableNonSecurity = value;
  }
  inline GetPatchBaselineResult& WithApprovedPatchesEnableNonSecurity(bool value) {
    SetApprovedPatchesEnableNonSecurity(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of explicitly rejected patches for the baseline.</p>
   */
  inline const Aws::Vector<Aws::String>& GetRejectedPatches() const { return m_rejectedPatches; }
  template <typename RejectedPatchesT = Aws::Vector<Aws::String>>
  void SetRejectedPatches(RejectedPatchesT&& value) {
    m_rejectedPatchesHasBeenSet = true;
    m_rejectedPatches = std::forward<RejectedPatchesT>(value);
  }
  template <typename RejectedPatchesT = Aws::Vector<Aws::String>>
  GetPatchBaselineResult& WithRejectedPatches(RejectedPatchesT&& value) {
    SetRejectedPatches(std::forward<RejectedPatchesT>(value));
    return *this;
  }
  template <typename RejectedPatchesT = Aws::String>
  GetPatchBaselineResult& AddRejectedPatches(RejectedPatchesT&& value) {
    m_rejectedPatchesHasBeenSet = true;
    m_rejectedPatches.emplace_back(std::forward<RejectedPatchesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The action specified to take on patches included in the
   * <code>RejectedPatches</code> list. A patch can be allowed only if it is a
   * dependency of another package, or blocked entirely along with packages that
   * include it as a dependency.</p>
   */
  inline PatchAction GetRejectedPatchesAction() const { return m_rejectedPatchesAction; }
  inline void SetRejectedPatchesAction(PatchAction value) {
    m_rejectedPatchesActionHasBeenSet = true;
    m_rejectedPatchesAction = value;
  }
  inline GetPatchBaselineResult& WithRejectedPatchesAction(PatchAction value) {
    SetRejectedPatchesAction(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Patch groups included in the patch baseline.</p>
   */
  inline const Aws::Vector<Aws::String>& GetPatchGroups() const { return m_patchGroups; }
  template <typename PatchGroupsT = Aws::Vector<Aws::String>>
  void SetPatchGroups(PatchGroupsT&& value) {
    m_patchGroupsHasBeenSet = true;
    m_patchGroups = std::forward<PatchGroupsT>(value);
  }
  template <typename PatchGroupsT = Aws::Vector<Aws::String>>
  GetPatchBaselineResult& WithPatchGroups(PatchGroupsT&& value) {
    SetPatchGroups(std::forward<PatchGroupsT>(value));
    return *this;
  }
  template <typename PatchGroupsT = Aws::String>
  GetPatchBaselineResult& AddPatchGroups(PatchGroupsT&& value) {
    m_patchGroupsHasBeenSet = true;
    m_patchGroups.emplace_back(std::forward<PatchGroupsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date the patch baseline was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedDate() const { return m_createdDate; }
  template <typename CreatedDateT = Aws::Utils::DateTime>
  void SetCreatedDate(CreatedDateT&& value) {
    m_createdDateHasBeenSet = true;
    m_createdDate = std::forward<CreatedDateT>(value);
  }
  template <typename CreatedDateT = Aws::Utils::DateTime>
  GetPatchBaselineResult& WithCreatedDate(CreatedDateT&& value) {
    SetCreatedDate(std::forward<CreatedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date the patch baseline was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetModifiedDate() const { return m_modifiedDate; }
  template <typename ModifiedDateT = Aws::Utils::DateTime>
  void SetModifiedDate(ModifiedDateT&& value) {
    m_modifiedDateHasBeenSet = true;
    m_modifiedDate = std::forward<ModifiedDateT>(value);
  }
  template <typename ModifiedDateT = Aws::Utils::DateTime>
  GetPatchBaselineResult& WithModifiedDate(ModifiedDateT&& value) {
    SetModifiedDate(std::forward<ModifiedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the patch baseline.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  GetPatchBaselineResult& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the patches to use to update the managed nodes, including
   * target operating systems and source repositories. Applies to Linux managed nodes
   * only.</p>
   */
  inline const Aws::Vector<PatchSource>& GetSources() const { return m_sources; }
  template <typename SourcesT = Aws::Vector<PatchSource>>
  void SetSources(SourcesT&& value) {
    m_sourcesHasBeenSet = true;
    m_sources = std::forward<SourcesT>(value);
  }
  template <typename SourcesT = Aws::Vector<PatchSource>>
  GetPatchBaselineResult& WithSources(SourcesT&& value) {
    SetSources(std::forward<SourcesT>(value));
    return *this;
  }
  template <typename SourcesT = PatchSource>
  GetPatchBaselineResult& AddSources(SourcesT&& value) {
    m_sourcesHasBeenSet = true;
    m_sources.emplace_back(std::forward<SourcesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the compliance status of managed nodes for which security-related
   * patches are available but were not approved. This preference is specified when
   * the <code>CreatePatchBaseline</code> or <code>UpdatePatchBaseline</code>
   * commands are run.</p> <p>Applies to Windows Server managed nodes only.</p>
   */
  inline PatchComplianceStatus GetAvailableSecurityUpdatesComplianceStatus() const { return m_availableSecurityUpdatesComplianceStatus; }
  inline void SetAvailableSecurityUpdatesComplianceStatus(PatchComplianceStatus value) {
    m_availableSecurityUpdatesComplianceStatusHasBeenSet = true;
    m_availableSecurityUpdatesComplianceStatus = value;
  }
  inline GetPatchBaselineResult& WithAvailableSecurityUpdatesComplianceStatus(PatchComplianceStatus value) {
    SetAvailableSecurityUpdatesComplianceStatus(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  GetPatchBaselineResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_baselineId;

  Aws::String m_name;

  OperatingSystem m_operatingSystem{OperatingSystem::NOT_SET};

  PatchFilterGroup m_globalFilters;

  PatchRuleGroup m_approvalRules;

  Aws::Vector<Aws::String> m_approvedPatches;

  PatchComplianceLevel m_approvedPatchesComplianceLevel{PatchComplianceLevel::NOT_SET};

  bool m_approvedPatchesEnableNonSecurity{false};

  Aws::Vector<Aws::String> m_rejectedPatches;

  PatchAction m_rejectedPatchesAction{PatchAction::NOT_SET};

  Aws::Vector<Aws::String> m_patchGroups;

  Aws::Utils::DateTime m_createdDate{};

  Aws::Utils::DateTime m_modifiedDate{};

  Aws::String m_description;

  Aws::Vector<PatchSource> m_sources;

  PatchComplianceStatus m_availableSecurityUpdatesComplianceStatus{PatchComplianceStatus::NOT_SET};

  Aws::String m_requestId;
  bool m_baselineIdHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_operatingSystemHasBeenSet = false;
  bool m_globalFiltersHasBeenSet = false;
  bool m_approvalRulesHasBeenSet = false;
  bool m_approvedPatchesHasBeenSet = false;
  bool m_approvedPatchesComplianceLevelHasBeenSet = false;
  bool m_approvedPatchesEnableNonSecurityHasBeenSet = false;
  bool m_rejectedPatchesHasBeenSet = false;
  bool m_rejectedPatchesActionHasBeenSet = false;
  bool m_patchGroupsHasBeenSet = false;
  bool m_createdDateHasBeenSet = false;
  bool m_modifiedDateHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_sourcesHasBeenSet = false;
  bool m_availableSecurityUpdatesComplianceStatusHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
