"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SecurityClient = void 0;
var _common = require("../../common");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

class SecurityClient {
  constructor(esClient) {
    this.esClient = esClient;
  }
  async authenticate(request, credentials) {
    const authHeader = Buffer.from(`${credentials.username}:${credentials.password}`).toString('base64');
    try {
      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers: {
          authorization: `Basic ${authHeader}`
        }
      });
      return {
        username: credentials.username,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.tenants,
        selectedTenant: esResponse.user_requested_tenant,
        credentials,
        proxyCredentials: credentials,
        tenancy_configs: esResponse.tenancy_configs
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }
  async authenticateWithHeader(request, headerName, headerValue, whitelistedHeadersAndValues = {}, additionalAuthHeaders = {}) {
    try {
      const credentials = {
        headerName,
        headerValue
      };
      const headers = {};
      if (headerValue) {
        headers[headerName] = headerValue;
      }

      // cannot get config elasticsearch.requestHeadersWhitelist from kibana.yml file in new platfrom
      // meanwhile, do we really need to save all headers in cookie?
      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers
      });
      return {
        username: esResponse.user_name,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.tenants,
        selectedTenant: esResponse.user_requested_tenant,
        credentials
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }
  async authenticateWithHeaders(request, additionalAuthHeaders = {}) {
    try {
      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers: additionalAuthHeaders
      });
      return {
        username: esResponse.user_name,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.tenants,
        selectedTenant: esResponse.user_requested_tenant
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }
  async authinfo(request, headers = {}) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers
      });
    } catch (error) {
      throw new Error(error.message);
    }
  }
  async dashboardsinfo(request, headers = {}) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.dashboardsinfo', {
        headers
      });
    } catch (error) {
      throw new Error(error.message);
    }
  }

  // Multi-tenancy APIs
  async getMultitenancyInfo(request) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.multitenancyinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }
  async putMultitenancyConfigurations(request, tenancyConfigSettings) {
    const body = {
      multitenancy_enabled: tenancyConfigSettings.multitenancy_enabled,
      private_tenant_enabled: tenancyConfigSettings.private_tenant_enabled,
      default_tenant: tenancyConfigSettings.default_tenant
    };
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.tenancy_configs', {
        body
      });
    } catch (error) {
      throw new Error(error.message);
    }
  }
  async getTenantInfoWithInternalUser() {
    try {
      return this.esClient.callAsInternalUser('opensearch_security.tenantinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }
  async getTenantInfo(request) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.tenantinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }
  async getSamlHeader(request) {
    try {
      // response is expected to be an error
      await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        [_common.AUTH_TYPE_PARAM]: _common.AuthType.SAML
      });
    } catch (error) {
      // the error looks like
      // wwwAuthenticateDirective:
      //   '
      //     X-Security-IdP realm="Open Distro Security"
      //     location="https://<your-auth-domain.com>/api/saml2/v1/sso?SAMLRequest=<some-encoded-string>"
      //     requestId="<request_id>"
      //   '

      if (!error.wwwAuthenticateDirective) {
        throw error;
      }
      try {
        const locationRegExp = /location="(.*?)"/;
        const requestIdRegExp = /requestId="(.*?)"/;
        const locationExecArray = locationRegExp.exec(error.wwwAuthenticateDirective);
        const requestExecArray = requestIdRegExp.exec(error.wwwAuthenticateDirective);
        if (locationExecArray && requestExecArray) {
          return {
            location: locationExecArray[1],
            requestId: requestExecArray[1]
          };
        }
        throw Error('failed parsing SAML config');
      } catch (parsingError) {
        console.log(parsingError);
        throw new Error(parsingError);
      }
    }
    throw new Error(`Invalid SAML configuration.`);
  }
  async authToken({
    requestId,
    samlResponse,
    acsEndpoint = undefined,
    authRequestType
  }) {
    const body = {
      RequestId: requestId,
      SAMLResponse: samlResponse,
      acsEndpoint
    };
    try {
      return await this.esClient.asScoped().callAsCurrentUser('opensearch_security.authtoken', {
        body,
        [_common.AUTH_TYPE_PARAM]: authRequestType
      });
    } catch (error) {
      console.log(error);
      throw new Error('failed to get token');
    }
  }
}
exports.SecurityClient = SecurityClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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