/*
 * Decompiled with CFR 0.152.
 */
package ghidra.server.security;

import ghidra.framework.remote.GhidraPrincipal;
import ghidra.framework.remote.SignatureCallback;
import ghidra.net.ApplicationKeyManagerFactory;
import ghidra.net.ApplicationKeyManagerUtils;
import ghidra.net.SignedToken;
import ghidra.server.RepositoryManager;
import ghidra.server.UserManager;
import ghidra.server.security.AuthenticationModule;
import ghidra.server.security.TokenGenerator;
import java.io.IOException;
import java.security.Principal;
import java.security.Signature;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.login.FailedLoginException;
import javax.security.auth.login.LoginException;
import javax.security.auth.x500.X500Principal;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class PKIAuthenticationModule
implements AuthenticationModule {
    static final Logger log = LogManager.getLogger(PKIAuthenticationModule.class);
    private static final long MAX_TOKEN_TIME = 300000L;
    private static final int TOKEN_SIZE = 64;
    private X500Principal[] authorities;
    private boolean anonymousAllowed;

    public PKIAuthenticationModule(boolean anonymousAllowed) throws IOException, CertificateException {
        this.anonymousAllowed = anonymousAllowed;
        this.authorities = ApplicationKeyManagerUtils.getTrustedIssuers();
        if (this.authorities == null) {
            throw new IOException("trusted PKI Certificate Authorities have not been configured");
        }
    }

    @Override
    public boolean anonymousCallbacksAllowed() {
        return false;
    }

    @Override
    public Callback[] getAuthenticationCallbacks() {
        SignatureCallback sigCb;
        try {
            byte[] token = TokenGenerator.getNewToken(64);
            boolean usingSelfSignedCert = ApplicationKeyManagerFactory.usingGeneratedSelfSignedCertificate();
            SignedToken signedToken = ApplicationKeyManagerUtils.getSignedToken((Principal[])(usingSelfSignedCert ? null : this.authorities), (byte[])token);
            sigCb = new SignatureCallback(this.authorities, token, signedToken.signature);
        }
        catch (Throwable t) {
            throw new RuntimeException("Unable to generate signed token", t);
        }
        return new Callback[]{sigCb};
    }

    @Override
    public boolean isNameCallbackAllowed() {
        return false;
    }

    @Override
    public String authenticate(UserManager userMgr, Subject subject, Callback[] callbacks) throws LoginException {
        GhidraPrincipal user = GhidraPrincipal.getGhidraPrincipal((Subject)subject);
        if (user == null) {
            throw new FailedLoginException("GhidraPrincipal required");
        }
        String username = user.getName();
        SignatureCallback sigCb = null;
        if (callbacks != null) {
            for (int i = 0; i < callbacks.length; ++i) {
                if (!(callbacks[i] instanceof SignatureCallback)) continue;
                sigCb = (SignatureCallback)callbacks[i];
                break;
            }
        }
        if (sigCb == null) {
            throw new FailedLoginException("PKI Signature callback required");
        }
        try {
            String dnUsername;
            byte[] token = sigCb.getToken();
            if (!TokenGenerator.isRecentToken(token, 300000L)) {
                throw new FailedLoginException("Stale Signature callback");
            }
            boolean usingSelfSignedCert = ApplicationKeyManagerFactory.usingGeneratedSelfSignedCertificate();
            if (!ApplicationKeyManagerUtils.isMySignature((Principal[])(usingSelfSignedCert ? null : this.authorities), (byte[])token, (byte[])sigCb.getServerSignature())) {
                throw new FailedLoginException("Invalid Signature callback");
            }
            X509Certificate[] certChain = sigCb.getCertificateChain();
            if (certChain == null || certChain.length == 0) {
                throw new FailedLoginException("user certificate not provided");
            }
            ApplicationKeyManagerUtils.validateClient((X509Certificate[])certChain, (String)"RSA");
            byte[] sigBytes = sigCb.getSignature();
            if (sigBytes != null) {
                Signature sig = Signature.getInstance(certChain[0].getSigAlgName());
                sig.initVerify(certChain[0]);
                sig.update(token);
                if (!sig.verify(sigBytes)) {
                    throw new FailedLoginException("Incorrect signature");
                }
            }
            if ((dnUsername = userMgr.getUserByDistinguishedName(certChain[0].getSubjectX500Principal())) != null) {
                return dnUsername;
            }
            if (userMgr.isValidUser(username)) {
                X500Principal x500User = userMgr.getDistinguishedName(username);
                if (x500User == null) {
                    userMgr.logUnknownDN(username, certChain[0].getSubjectX500Principal());
                    if (!this.anonymousAllowed) {
                        throw new FailedLoginException("Distinguished name is unknown");
                    }
                    log.log(Level.WARN, "Know user's DN not found (" + username + ") ");
                    username = "-anonymous-";
                } else {
                    userMgr.logUnknownDN(username, certChain[0].getSubjectX500Principal());
                    if (!this.anonymousAllowed) {
                        throw new FailedLoginException("Expected distinguished name: " + x500User.getName());
                    }
                    username = "-anonymous-";
                }
            } else {
                if (!this.anonymousAllowed) {
                    throw new FailedLoginException("Unknown user: " + username);
                }
                username = "-anonymous-";
            }
            if ("-anonymous-".equals(username)) {
                RepositoryManager.log(null, null, "Anonymous access allowed for: " + certChain[0].getSubjectX500Principal().toString(), user.getName());
            }
        }
        catch (LoginException e) {
            throw e;
        }
        catch (Throwable t) {
            String msg = t.getMessage();
            if (msg == null) {
                msg = t.toString();
            }
            throw new FailedLoginException(msg);
        }
        return username;
    }
}

