/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.bin;

import ghidra.app.util.bin.ByteProvider;
import ghidra.app.util.bin.InvalidDataException;
import ghidra.util.BigEndianDataConverter;
import ghidra.util.DataConverter;
import ghidra.util.LittleEndianDataConverter;
import ghidra.util.NumericUtilities;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

public class BinaryReader {
    private static final int MAX_SANE_BUFFER = 0x7FFFFBFF;
    public static final int SIZEOF_BYTE = 1;
    public static final int SIZEOF_SHORT = 2;
    public static final int SIZEOF_INT = 4;
    public static final int SIZEOF_LONG = 8;
    protected final ByteProvider provider;
    protected DataConverter converter;
    protected long currentIndex;

    public BinaryReader(ByteProvider provider, boolean isLittleEndian) {
        this(provider, DataConverter.getInstance((!isLittleEndian ? 1 : 0) != 0), 0L);
    }

    public BinaryReader(ByteProvider provider, DataConverter converter, long initialIndex) {
        this.provider = provider;
        this.converter = converter;
        this.currentIndex = initialIndex;
    }

    public BinaryReader clone(long newIndex) {
        return new BinaryReader(this.provider, this.converter, newIndex);
    }

    public BinaryReader clone() {
        return this.clone(this.currentIndex);
    }

    public BinaryReader asBigEndian() {
        return new BinaryReader(this.provider, (DataConverter)BigEndianDataConverter.INSTANCE, this.currentIndex);
    }

    public BinaryReader asLittleEndian() {
        return new BinaryReader(this.provider, (DataConverter)LittleEndianDataConverter.INSTANCE, this.currentIndex);
    }

    public boolean isLittleEndian() {
        return this.converter instanceof LittleEndianDataConverter;
    }

    public boolean isBigEndian() {
        return this.converter instanceof BigEndianDataConverter;
    }

    public void setLittleEndian(boolean isLittleEndian) {
        this.converter = DataConverter.getInstance((!isLittleEndian ? 1 : 0) != 0);
    }

    public long length() throws IOException {
        return this.provider.length();
    }

    public boolean isValidIndex(int index) {
        return this.provider.isValidIndex(Integer.toUnsignedLong(index));
    }

    public boolean isValidIndex(long index) {
        return this.provider.isValidIndex(index);
    }

    public boolean isValidRange(long startIndex, int count) {
        if (count < 0) {
            return false;
        }
        if (count > 1) {
            long endIndex = startIndex + (long)(count - 1);
            if (Long.compareUnsigned(endIndex, startIndex) < 0) {
                return false;
            }
            if (!this.provider.isValidIndex(endIndex)) {
                return false;
            }
            --count;
        }
        for (int i = 0; i < count; ++i) {
            if (this.provider.isValidIndex(startIndex + (long)i)) continue;
            return false;
        }
        return true;
    }

    public boolean hasNext() {
        return this.provider.isValidIndex(this.currentIndex);
    }

    public boolean hasNext(int count) {
        return this.isValidRange(this.currentIndex, count);
    }

    public int align(int alignValue) {
        long prevIndex = this.currentIndex;
        this.currentIndex = NumericUtilities.getUnsignedAlignedValue((long)this.currentIndex, (long)alignValue);
        return (int)(this.currentIndex - prevIndex);
    }

    public long setPointerIndex(int index) {
        return this.setPointerIndex(Integer.toUnsignedLong(index));
    }

    public long setPointerIndex(long index) {
        long oldIndex = this.currentIndex;
        this.currentIndex = index;
        return oldIndex;
    }

    public long getPointerIndex() {
        return this.currentIndex;
    }

    public InputStream getInputStream() {
        return new BinaryReaderInputStream();
    }

    public byte peekNextByte() throws IOException {
        return this.readByte(this.currentIndex);
    }

    public short peekNextShort() throws IOException {
        return this.readShort(this.currentIndex);
    }

    public int peekNextInt() throws IOException {
        return this.readInt(this.currentIndex);
    }

    public long peekNextLong() throws IOException {
        return this.readLong(this.currentIndex);
    }

    public byte readNextByte() throws IOException {
        byte b = this.readByte(this.currentIndex);
        ++this.currentIndex;
        return b;
    }

    public int readNextUnsignedByte() throws IOException {
        return Byte.toUnsignedInt(this.readNextByte());
    }

    public short readNextShort() throws IOException {
        return this.readNextShort(this.converter);
    }

    public short readNextShort(DataConverter dc) throws IOException {
        short s = this.readShort(dc, this.currentIndex);
        this.currentIndex += 2L;
        return s;
    }

    public int readNextUnsignedShort() throws IOException {
        return Short.toUnsignedInt(this.readNextShort(this.converter));
    }

    public int readNextUnsignedShort(DataConverter dc) throws IOException {
        return Short.toUnsignedInt(this.readNextShort(dc));
    }

    public int readNextInt() throws IOException {
        return this.readNextInt(this.converter);
    }

    public int readNextInt(DataConverter dc) throws IOException {
        int i = this.readInt(dc, this.currentIndex);
        this.currentIndex += 4L;
        return i;
    }

    public long readNextUnsignedInt() throws IOException {
        return Integer.toUnsignedLong(this.readNextInt(this.converter));
    }

    public long readNextUnsignedInt(DataConverter dc) throws IOException {
        return Integer.toUnsignedLong(this.readNextInt(dc));
    }

    public long readNextLong() throws IOException {
        return this.readNextLong(this.converter);
    }

    public long readNextLong(DataConverter dc) throws IOException {
        long l = this.readLong(dc, this.currentIndex);
        this.currentIndex += 8L;
        return l;
    }

    public long readNextValue(int len) throws IOException {
        return this.readNextValue(this.converter, len);
    }

    public long readNextValue(DataConverter dc, int len) throws IOException {
        long result = this.readValue(dc, this.currentIndex, len);
        this.currentIndex += (long)len;
        return result;
    }

    public long readNextUnsignedValue(int len) throws IOException {
        return this.readNextUnsignedValue(this.converter, len);
    }

    public long readNextUnsignedValue(DataConverter dc, int len) throws IOException {
        long result = this.readUnsignedValue(dc, this.currentIndex, len);
        this.currentIndex += (long)len;
        return result;
    }

    public String readNextAsciiString() throws IOException {
        return this.readNextString(StandardCharsets.US_ASCII, 1);
    }

    public String readNextAsciiString(int length) throws IOException {
        return this.readNextString(length, StandardCharsets.US_ASCII, 1);
    }

    public String readNextUnicodeString() throws IOException {
        return this.readNextString(this.getUTF16Charset(), 2);
    }

    public String readNextUnicodeString(int charCount) throws IOException {
        return this.readNextString(charCount, this.getUTF16Charset(), 2);
    }

    public String readNextUtf8String() throws IOException {
        return this.readNextString(StandardCharsets.UTF_8, 1);
    }

    public String readNextUtf8String(int length) throws IOException {
        return this.readNextString(length, StandardCharsets.UTF_8, 1);
    }

    private String readNextString(Charset charset, int charLen) throws IOException {
        byte[] bytes = this.readUntilNullTerm(this.currentIndex, charLen);
        this.currentIndex += (long)(bytes.length + charLen);
        String result = new String(bytes, charset);
        return result;
    }

    private String readNextString(int charCount, Charset charset, int charLen) throws IOException {
        if (charCount < 0) {
            throw new IllegalArgumentException(String.format("Invalid charCount: %d", charCount));
        }
        byte[] bytes = this.readByteArray(this.currentIndex, charCount * charLen);
        this.currentIndex += (long)bytes.length;
        int strLen = this.getLengthWithoutTrailingNullTerms(bytes, charLen);
        String result = new String(bytes, 0, strLen, charset);
        return result;
    }

    public byte[] readNextByteArray(int nElements) throws IOException {
        byte[] b = this.readByteArray(this.currentIndex, nElements);
        this.currentIndex += (long)(1 * nElements);
        return b;
    }

    public short[] readNextShortArray(int nElements) throws IOException {
        short[] s = this.readShortArray(this.currentIndex, nElements);
        this.currentIndex += (long)(2 * nElements);
        return s;
    }

    public int[] readNextIntArray(int nElements) throws IOException {
        int[] i = this.readIntArray(this.currentIndex, nElements);
        this.currentIndex += (long)(4 * nElements);
        return i;
    }

    public long[] readNextLongArray(int nElements) throws IOException {
        long[] l = this.readLongArray(this.currentIndex, nElements);
        this.currentIndex += (long)(8 * nElements);
        return l;
    }

    public int readNextUnsignedIntExact() throws IOException, InvalidDataException {
        return this.readNextUnsignedIntExact(this.converter);
    }

    public int readNextUnsignedIntExact(DataConverter dc) throws IOException, InvalidDataException {
        long i = this.readNextUnsignedInt(dc);
        BinaryReader.ensureInt32u(i);
        return (int)i;
    }

    private byte[] readUntilNullTerm(long index, int charLen) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        long curPos = index;
        while (Long.compareUnsigned(curPos, index) >= 0) {
            if ((long)baos.size() + (long)charLen >= 0x7FFFFBFFL) {
                throw new EOFException("Run-on unterminated string at 0x%s..0x%s".formatted(Long.toUnsignedString(index, 16), Long.toUnsignedString(curPos, 16)));
            }
            try {
                byte[] bytes = this.readByteArray(curPos, charLen);
                if (this.isNullTerm(bytes, 0, charLen)) {
                    return baos.toByteArray();
                }
                baos.write(bytes);
            }
            catch (IOException e) {
                if (baos.size() != 0) break;
                throw new EOFException("Attempted to read string at 0x%s".formatted(Long.toUnsignedString(index, 16)));
            }
            curPos += (long)charLen;
        }
        throw new EOFException("Unterminated string at 0x%s..0x%s".formatted(Long.toUnsignedString(index, 16), Long.toUnsignedString(curPos, 16)));
    }

    private boolean isNullTerm(byte[] bytes, int offset, int charLen) {
        for (int i = offset; i < offset + charLen; ++i) {
            if (bytes[i] == 0) continue;
            return false;
        }
        return true;
    }

    private int getLengthWithoutTrailingNullTerms(byte[] bytes, int charLen) {
        int termPos;
        for (termPos = bytes.length - charLen; termPos >= 0 && this.isNullTerm(bytes, termPos, charLen); termPos -= charLen) {
        }
        return termPos + charLen;
    }

    private Charset getUTF16Charset() {
        return this.isBigEndian() ? StandardCharsets.UTF_16BE : StandardCharsets.UTF_16LE;
    }

    public String readAsciiString(long index) throws IOException {
        return this.readString(index, StandardCharsets.US_ASCII, 1);
    }

    public String readAsciiString(long index, int length) throws IOException {
        return this.readString(index, length, StandardCharsets.US_ASCII, 1);
    }

    public String readUnicodeString(long index) throws IOException {
        return this.readString(index, this.getUTF16Charset(), 2);
    }

    public String readUnicodeString(long index, int charCount) throws IOException {
        return this.readString(index, charCount, this.getUTF16Charset(), 2);
    }

    public String readUtf8String(long index) throws IOException {
        return this.readString(index, StandardCharsets.UTF_8, 1);
    }

    public String readUtf8String(long index, int length) throws IOException {
        return this.readString(index, length, StandardCharsets.UTF_8, 1);
    }

    private String readString(long index, int charCount, Charset charset, int charLen) throws IOException {
        if (charCount < 0) {
            throw new IllegalArgumentException(String.format("Invalid charCount: %d", charCount));
        }
        byte[] bytes = this.readByteArray(index, charCount * charLen);
        int strLen = this.getLengthWithoutTrailingNullTerms(bytes, charLen);
        String result = new String(bytes, 0, strLen, charset);
        return result;
    }

    private String readString(long index, Charset charset, int charLen) throws IOException {
        byte[] bytes = this.readUntilNullTerm(index, charLen);
        String result = new String(bytes, charset);
        return result;
    }

    public byte readByte(long index) throws IOException {
        return this.provider.readByte(index);
    }

    public int readUnsignedByte(long index) throws IOException {
        return Byte.toUnsignedInt(this.readByte(index));
    }

    public short readShort(long index) throws IOException {
        return this.readShort(this.converter, index);
    }

    public short readShort(DataConverter dc, long index) throws IOException {
        byte[] bytes = this.provider.readBytes(index, 2L);
        return dc.getShort(bytes);
    }

    public int readUnsignedShort(long index) throws IOException {
        return Short.toUnsignedInt(this.readShort(this.converter, index));
    }

    public int readUnsignedShort(DataConverter dc, long index) throws IOException {
        return Short.toUnsignedInt(this.readShort(dc, index));
    }

    public int readInt(long index) throws IOException {
        return this.readInt(this.converter, index);
    }

    public int readInt(DataConverter dc, long index) throws IOException {
        byte[] bytes = this.provider.readBytes(index, 4L);
        return dc.getInt(bytes);
    }

    public long readUnsignedInt(long index) throws IOException {
        return Integer.toUnsignedLong(this.readInt(this.converter, index));
    }

    public long readUnsignedInt(DataConverter dc, long index) throws IOException {
        return Integer.toUnsignedLong(this.readInt(dc, index));
    }

    public long readLong(long index) throws IOException {
        return this.readLong(this.converter, index);
    }

    public long readLong(DataConverter dc, long index) throws IOException {
        byte[] bytes = this.provider.readBytes(index, 8L);
        return dc.getLong(bytes);
    }

    public long readValue(long index, int len) throws IOException {
        return this.readValue(this.converter, index, len);
    }

    public long readValue(DataConverter dc, long index, int len) throws IOException {
        byte[] bytes = this.provider.readBytes(index, len);
        return dc.getSignedValue(bytes, len);
    }

    public long readUnsignedValue(long index, int len) throws IOException {
        return this.readUnsignedValue(this.converter, index, len);
    }

    public long readUnsignedValue(DataConverter dc, long index, int len) throws IOException {
        byte[] bytes = this.provider.readBytes(index, len);
        return dc.getValue(bytes, len);
    }

    public byte[] readByteArray(long index, int nElements) throws IOException {
        if (nElements < 0) {
            throw new IOException("Invalid number of elements specified: " + nElements);
        }
        return this.provider.readBytes(index, nElements);
    }

    public short[] readShortArray(long index, int nElements) throws IOException {
        if (nElements < 0) {
            throw new IOException("Invalid number of elements specified: " + nElements);
        }
        short[] arr = new short[nElements];
        for (int i = 0; i < nElements; ++i) {
            arr[i] = this.readShort(index);
            index += 2L;
        }
        return arr;
    }

    public int[] readIntArray(long index, int nElements) throws IOException {
        if (nElements < 0) {
            throw new IOException("Invalid number of elements specified: " + nElements);
        }
        int[] arr = new int[nElements];
        for (int i = 0; i < nElements; ++i) {
            arr[i] = this.readInt(index);
            index += 4L;
        }
        return arr;
    }

    public long[] readLongArray(long index, int nElements) throws IOException {
        if (nElements < 0) {
            throw new IOException("Invalid number of elements specified: " + nElements);
        }
        long[] arr = new long[nElements];
        for (int i = 0; i < nElements; ++i) {
            arr[i] = this.readLong(index);
            index += 8L;
        }
        return arr;
    }

    public ByteProvider getByteProvider() {
        return this.provider;
    }

    public int readNextVarInt(ReaderFunction<Long> func) throws IOException, InvalidDataException {
        long value = func.get(this);
        BinaryReader.ensureInt32s(value);
        return (int)value;
    }

    public int readNextVarInt(InputStreamReaderFunction<Long> func) throws IOException, InvalidDataException {
        long value = func.get(this.getInputStream());
        BinaryReader.ensureInt32s(value);
        return (int)value;
    }

    public int readNextUnsignedVarIntExact(ReaderFunction<Long> func) throws IOException, InvalidDataException {
        long value = func.get(this);
        BinaryReader.ensureInt32u(value);
        return (int)value;
    }

    public int readNextUnsignedVarIntExact(InputStreamReaderFunction<Long> func) throws IOException, InvalidDataException {
        long value = func.get(this.getInputStream());
        BinaryReader.ensureInt32u(value);
        return (int)value;
    }

    public <T> T readNext(ReaderFunction<T> func) throws IOException {
        T obj = func.get(this);
        return obj;
    }

    public <T> T readNext(InputStreamReaderFunction<T> func) throws IOException {
        T obj = func.get(this.getInputStream());
        return obj;
    }

    private static void ensureInt32u(long value) throws InvalidDataException {
        if (value < 0L || value > Integer.MAX_VALUE) {
            throw new InvalidDataException("Value out of range for positive java 32 bit unsigned int: %s".formatted(Long.toUnsignedString(value)));
        }
    }

    private static void ensureInt32s(long value) throws InvalidDataException {
        if (value < Integer.MIN_VALUE || value > Integer.MAX_VALUE) {
            throw new InvalidDataException("Value out of range for java 32 bit signed int: %d".formatted(value));
        }
    }

    private class BinaryReaderInputStream
    extends InputStream {
        private BinaryReaderInputStream() {
        }

        @Override
        public int read() throws IOException {
            if (!BinaryReader.this.hasNext()) {
                return -1;
            }
            return BinaryReader.this.readNextUnsignedByte();
        }
    }

    public static interface ReaderFunction<T> {
        public T get(BinaryReader var1) throws IOException;
    }

    public static interface InputStreamReaderFunction<T> {
        public T get(InputStream var1) throws IOException;
    }
}

