/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2005 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <config.h>

#include <string.h>
#include <libxml/HTMLparser.h>

#include "atom.h"
#include "journal.h"
#include "network.h"
#include "blog_atom.h"

typedef struct _AtomLink AtomLink;

typedef enum
{
	ATOM_LINK_POST,
	ATOM_LINK_FEED
} AtomLinkType;

struct _AtomLink
{
	gchar *uri;
	gchar *title;
	AtomLinkType type;
};

/* Return a list of AtomEntries */

static GSList*
retrieve_atom_entries (const gchar *xml)
{
	xmlDocPtr doc;
	xmlNodePtr node;
	GSList *entries = NULL;
	
	/* Parse the XML into a tree */
	doc = xmlReadMemory (xml, strlen (xml), NULL, NULL, 
			XML_PARSE_NOERROR | XML_PARSE_NOBLANKS);
	if (!doc)
		return NULL;
	
	node = xmlDocGetRootElement (doc);
	if (!node || xmlStrcasecmp (node->name, (xmlChar *)"feed"))
	{
		g_warning ("retrieve_atom_entries: XML is not a <feed>");
		xmlFreeDoc (doc);
		return NULL;
	}
	
	/* Drop inside the <feed> element and look for <entry> elements */
	node = node->xmlChildrenNode;
	while (node)
	{
		if (!xmlStrcasecmp (node->name, (xmlChar *)"entry"))
		{
			AtomEntry *entry;
			entry = atom_entry_parse (doc, node->xmlChildrenNode);
			if (entry)
				entries = g_slist_prepend (entries, entry);
		}
		
		node = node->next;
	}
	
	/* Reverse the list so that it is sorted newest->oldest */
	entries = g_slist_reverse (entries);
	
	return entries;
}

/* Return a list of AtomLinks */

static GSList*
retrieve_atom_links (xmlNodePtr node)
{
	GSList *links = NULL;
	
	/* Drop inside the <feed> or <head> elements */
	node = node->xmlChildrenNode;
	while (node)
	{
		/* Retrieve the uri and title attributes from each <link> */
		if (!xmlStrcmp (node->name, (xmlChar *)"link"))
		{
			gchar *type, *rel, *uri, *title;
			
			type = (gchar *)xmlGetProp (node, (xmlChar *)"type");
			if (type && !xmlStrcmp ((xmlChar *)type, (xmlChar *)"application/atom+xml"))
			{
				uri = (gchar *)xmlGetProp (node, (xmlChar *)"href");
				title = (gchar *)xmlGetProp (node, (xmlChar *)"title");
				rel = (gchar *)xmlGetProp (node, (xmlChar *)"rel");
				if (uri && title && rel)
				{
					AtomLink *link = g_new0 (AtomLink, 1);
					
					/* Add the link information to our list */
					link->uri = uri;
					link->title = title;
					if (!xmlStrcasecmp ((xmlChar *)rel, (xmlChar *)"server.post"))
						link->type = ATOM_LINK_POST;
					else
						link->type = ATOM_LINK_FEED;
					links = g_slist_prepend (links, link);
				}
			}
		}
		node = node->next;
	}
	
	return links;
}

/* Parse the HTML journal list and returna list of AtomLinks */

static GSList*
parse_html_journal_list (const gchar *html)
{
	htmlDocPtr doc;
	htmlNodePtr node;
	GSList *links = NULL;
	
	g_return_val_if_fail (html, NULL);
		
	doc = htmlReadMemory (html, strlen (html), NULL, NULL, HTML_PARSE_NOERROR);
	if (!doc)
		return NULL;
	
	node = xmlDocGetRootElement (doc);
	if (!node || xmlStrcasecmp (node->name, (xmlChar *)"html"))
	{
		g_warning ("parse_html_journal_list: Not an HTML document");
		xmlFreeDoc (doc);
		return NULL;
	}
	
	/* Drop inside the <html> element */
	node = node->xmlChildrenNode;
	
	/* Build a list of Atom links */
	links = retrieve_atom_links (node);
	
	xmlFreeDoc (doc);
	
	return links;
}

/* Parse the XML journal list and return a list of AtomLinks */

static GSList*
parse_xml_journal_list (const gchar *xml)
{
	xmlDocPtr doc;
	xmlNodePtr node;
	GSList *links = NULL;
	
	g_return_val_if_fail (xml, NULL);
	
	/* Parse the XML into a tree */
	doc = xmlReadMemory (xml, strlen (xml), NULL, NULL, XML_PARSE_NOERROR);
	if (!doc)
		return NULL;
	
	node = xmlDocGetRootElement (doc);
	if (!node || xmlStrcasecmp (node->name, (xmlChar *)"feed"))
	{
		g_warning ("parse_xml_journal_list: XML is not a <feed>");
		xmlFreeDoc (doc);
		return NULL;
	}
	
	/* Build a list of Atom links */
	links = retrieve_atom_links (node);
	
	xmlFreeDoc (doc);
	
	return links;
}

/* Send a GET request to the blog's index page */

DrivelRequest*
blog_atom_build_login_request (const gchar *username, const gchar *password,
		const gchar *uri)
{
	DrivelRequest *dr;
	
	g_return_val_if_fail (username, NULL);
	g_return_val_if_fail (password, NULL);
	g_return_val_if_fail (uri, NULL);
	
	debug ("blog_atom_build_login_request()\n");
	
	dr = drivel_request_new_with_items (
			REQUEST_TYPE_LOGIN,
			REQUEST_PROTOCOL_GET,
			BLOG_API_ATOM,
			uri,
			g_strdup ("http_basic"), g_strdup ("yes"),
			g_strdup ("username"), g_strdup (username),
			g_strdup ("password"), g_strdup (password),
			NULL);
	
	return dr;
}

/* Parse the response for the <link> tag holding the Atom interface URL */

void
blog_atom_parse_login_request (DrivelClient *dc, DrivelRequest *dr)
{
	DrivelRequestData *data;
	GSList *links, *current;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_atom_parse_login_request()\n");
	
	data = drivel_request_get_data (dr);
	links = parse_xml_journal_list (data->data);
	if (!links)
	{
		links = parse_html_journal_list (data->data);
		if (!links)
		{
			display_error_dialog (dc, _("Server error"), 
					_("The server did not return a valid Atom response."));
			return;
		}
	}
	
	dc->journals = 0;
	for (current = links; current; current = current->next)
	{
		GSList *journal;
		AtomLink *link;
		DrivelJournal *dj;
		gboolean exists = FALSE;
		
		link = current->data;
		for (journal = dc->journal_list; journal && !exists; journal = journal->next)
		{
			dj = journal->data;
			if (!strcmp (dj->name, link->title))
			{
				switch (link->type)
				{
					case ATOM_LINK_POST: dj->uri_post = g_strdup (link->uri); break;
					case ATOM_LINK_FEED: dj->uri_feed = g_strdup (link->uri); break;
				}
				exists = TRUE;
			}
		}
		
		if (!exists)
		{
			dj = drivel_journal_new ();
			dj->name = link->title;
			dj->uri_post = link->uri;
			dj->type = JOURNAL_TYPE_USER;
			dc->journal_list = g_slist_prepend (dc->journal_list, dj);
			dc->journals++;
		}
		else
		{
			g_free (link->title);
			g_free (link->uri);
			g_free (link);
		}
	}
	
	for (current = dc->journal_list; current; current = current->next)
	{
		DrivelRequest *dr;
		DrivelJournal *dj = current->data;
		
		/* get the recent entries for this journal */
		dr = blog_atom_build_getevents_request (dc->user->username,
				dc->user->password, dj->uri_feed, FALSE);
		net_enqueue_request (dc, dr);
	}
	
	add_account_to_list (dc);
	
	gtk_widget_hide (dc->login_window);
	journal_window_build (dc);
	
	return;
}

/* Build a request with all of the post entry information */
/* Pass link=NULL for a new post or link='entry_uri' to update an existing
 * post. */
DrivelRequest*
blog_atom_build_post_request (const gchar *username, const gchar *password,
		const gchar *uri, const gchar *subject, const gchar *content,
		const gchar *link)
{
	DrivelRequest *dr;
	AtomEntry *entry;
	gchar *packet, *server, *put;
	
	g_return_val_if_fail (username, NULL);
	g_return_val_if_fail (password, NULL);
	g_return_val_if_fail (uri, NULL);
	
	entry = atom_entry_new ();
	if (subject && strcmp (subject, ""))
		entry->title = g_strdup (subject);
	else
		entry->title = g_strdup ("Untitled");
	if (content)
		entry->content = g_strdup (content);
	else
		entry->content = g_strdup (" ");
	entry->issued = get_w3dtf_timestamp ();
	
	if (link)
	{
		server = g_strdup (link);
		put = g_strdup ("yes");
	}
	else
	{
		server = g_strdup (uri);
		put = NULL;
	}
	
	packet = atom_build_packet (entry);
	
	dr = drivel_request_new_with_items (
			REQUEST_TYPE_POSTEVENT,
			REQUEST_PROTOCOL_ATOM,
			BLOG_API_ATOM,
			server,
			g_strdup ("http_basic"), g_strdup ("yes"),
			g_strdup ("username"), g_strdup (username),
			g_strdup ("password"), g_strdup (password),
			g_strdup ("http_put"), put,
			g_strdup ("xml"), packet,
			NULL);
	
	return dr;
}

void
blog_atom_parse_post_request (DrivelClient *dc, DrivelRequest *dr)
{
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_atom_parse_post_request()");
	
	journal_finished_post (dc);
	
	return;
}

DrivelRequest*
blog_atom_build_delete_request (const gchar *username, const gchar *password,
		const gchar *uri)
{
	DrivelRequest *dr;
	
	g_return_val_if_fail (username, NULL);
	g_return_val_if_fail (password, NULL);
	g_return_val_if_fail (uri, NULL);
	
	debug ("blog_atom_build_delete_request()\n");
	
	dr = drivel_request_new_with_items (
			REQUEST_TYPE_DELETEEVENT,
			REQUEST_PROTOCOL_GET,
			BLOG_API_ATOM,
			uri,
			g_strdup ("http_basic"), g_strdup ("yes"),
			g_strdup ("username"), g_strdup (username),
			g_strdup ("password"), g_strdup (password),
			g_strdup ("http_type"), g_strdup ("delete"),
			NULL);
	
	return dr;
}

DrivelRequest*
blog_atom_build_getevents_request (const gchar *username, const gchar *password,
		const gchar *uri, gboolean last_entry)
{
	DrivelRequest *dr;
	
	g_return_val_if_fail (username, NULL);
	g_return_val_if_fail (password, NULL);
	g_return_val_if_fail (uri, NULL);
	
	debug ("blog_atom_build_getevents_request_request()\n");
	
	dr = drivel_request_new_with_items (
			REQUEST_TYPE_GETEVENTS,
			REQUEST_PROTOCOL_GET,
			BLOG_API_ATOM,
			uri,
			g_strdup ("http_basic"), g_strdup ("yes"),
			g_strdup ("username"), g_strdup (username),
			g_strdup ("password"), g_strdup (password),
			NULL);

	if (last_entry)
	{
		drivel_request_add_items (dr, 
				g_strdup ("last_entry"), g_strdup ("true"),
				NULL);
	}

	return dr;
}

void
blog_atom_parse_getevents_request (DrivelClient *dc, DrivelRequest *dr)
{
	DrivelRequestData *data;
	GSList *entries, *current;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_atom_parse_getevents_request");
	
	data = drivel_request_get_data (dr);
	entries = retrieve_atom_entries (data->data);
	if (entries)
	{
		if (drivel_request_item_lookup (dr, "last_entry"))
		{
			AtomEntry *entry = entries->data;
			journal_edit_entry (dc, entry->id, entry->content, NULL, 
					NULL, entry->title,	NULL, NULL, NULL, 
					NULL, NULL, NULL, entry->link, NULL);
		}
		else
		{
			const gchar *feed_uri;
			
			feed_uri = drivel_request_get_uri (dr);
			
			for (current = entries; current; current = current->next)
			{
				DrivelJournalEntry *jentry = journal_entry_new ();
				AtomEntry *aentry = current->data;
				
				jentry->content = g_strdup (aentry->content);
				jentry->subject = g_strdup (aentry->title);
				jentry->userid = g_strdup (feed_uri);
				jentry->postid = g_strdup (aentry->id);
				jentry->link = g_strdup (aentry->link);
				g_ptr_array_add (dc->recent_entries, jentry);
			}
			
			journal_refresh_recent_entries (dc);
		}
	}
	
	return;
}
