/* valamarkupreader.vala
 *
 * Copyright (C) 2008-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vapigen/valamarkupreader.h>
#include <gee/hashmap.h>
#include <gee/map.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>




struct _ValaMarkupReaderPrivate {
	char* _filename;
	char* _name;
	GMappedFile* mapped_file;
	gchar* begin;
	gchar* current;
	gchar* end;
	gint line;
	gint column;
	GeeMap* attributes;
	gboolean empty_element;
};

#define VALA_MARKUP_READER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_MARKUP_READER, ValaMarkupReaderPrivate))
enum  {
	VALA_MARKUP_READER_DUMMY_PROPERTY,
	VALA_MARKUP_READER_FILENAME,
	VALA_MARKUP_READER_NAME
};
static char* vala_markup_reader_read_name (ValaMarkupReader* self);
static void vala_markup_reader_space (ValaMarkupReader* self);
static void vala_markup_reader_set_filename (ValaMarkupReader* self, const char* value);
static void vala_markup_reader_set_name (ValaMarkupReader* self, const char* value);
static GObject * vala_markup_reader_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer vala_markup_reader_parent_class = NULL;
static void vala_markup_reader_finalize (GObject* obj);



ValaMarkupReader* vala_markup_reader_construct (GType object_type, const char* filename) {
	GParameter * __params;
	GParameter * __params_it;
	ValaMarkupReader * self;
	g_return_val_if_fail (filename != NULL, NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "filename";
	g_value_init (&__params_it->value, G_TYPE_STRING);
	g_value_set_string (&__params_it->value, filename);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


ValaMarkupReader* vala_markup_reader_new (const char* filename) {
	return vala_markup_reader_construct (VALA_TYPE_MARKUP_READER, filename);
}


char* vala_markup_reader_get_attribute (ValaMarkupReader* self, const char* attr) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (attr != NULL, NULL);
	return (char*) gee_map_get (self->priv->attributes, attr);
}


static char* vala_markup_reader_read_name (ValaMarkupReader* self) {
	gchar* begin;
	g_return_val_if_fail (self != NULL, NULL);
	begin = self->priv->current;
	while (self->priv->current < self->priv->end) {
		gboolean _tmp0;
		gboolean _tmp1;
		gboolean _tmp2;
		gunichar u;
		_tmp0 = FALSE;
		_tmp1 = FALSE;
		_tmp2 = FALSE;
		if (self->priv->current[0] == ' ') {
			_tmp2 = TRUE;
		} else {
			_tmp2 = self->priv->current[0] == '>';
		}
		if (_tmp2) {
			_tmp1 = TRUE;
		} else {
			_tmp1 = self->priv->current[0] == '/';
		}
		if (_tmp1) {
			_tmp0 = TRUE;
		} else {
			_tmp0 = self->priv->current[0] == '=';
		}
		if (_tmp0) {
			break;
		}
		u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
		if (u != ((gunichar) (-1))) {
			self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
		} else {
			vala_report_error (NULL, "invalid UTF-8 character");
		}
	}
	if (self->priv->current == begin) {
	}
	/* syntax error: invalid name*/
	return g_strndup ((const char*) begin, self->priv->current - begin);
}


ValaMarkupTokenType vala_markup_reader_read_token (ValaMarkupReader* self, ValaSourceLocation* token_begin, ValaSourceLocation* token_end) {
	ValaMarkupTokenType type;
	gchar* begin;
	g_return_val_if_fail (self != NULL, 0);
	gee_map_clear (self->priv->attributes);
	if (self->priv->empty_element) {
		self->priv->empty_element = FALSE;
		return VALA_MARKUP_TOKEN_TYPE_END_ELEMENT;
	}
	vala_markup_reader_space (self);
	type = VALA_MARKUP_TOKEN_TYPE_NONE;
	begin = self->priv->current;
	(*token_begin).pos = begin;
	(*token_begin).line = self->priv->line;
	(*token_begin).column = self->priv->column;
	if (self->priv->current >= self->priv->end) {
		type = VALA_MARKUP_TOKEN_TYPE_EOF;
	} else {
		if (self->priv->current[0] == '<') {
			self->priv->current++;
			if (self->priv->current >= self->priv->end) {
			} else {
				if (self->priv->current[0] == '?') {
				} else {
					if (self->priv->current[0] == '!') {
						gboolean _tmp1;
						gboolean _tmp2;
						/* comment or doctype*/
						self->priv->current++;
						_tmp1 = FALSE;
						_tmp2 = FALSE;
						if (self->priv->current < (self->priv->end - 1)) {
							_tmp2 = self->priv->current[0] == '-';
						} else {
							_tmp2 = FALSE;
						}
						if (_tmp2) {
							_tmp1 = self->priv->current[1] == '-';
						} else {
							_tmp1 = FALSE;
						}
						if (_tmp1) {
							/* comment*/
							self->priv->current = self->priv->current + 2;
							while (self->priv->current < (self->priv->end - 2)) {
								gboolean _tmp3;
								gboolean _tmp4;
								_tmp3 = FALSE;
								_tmp4 = FALSE;
								if (self->priv->current[0] == '-') {
									_tmp4 = self->priv->current[1] == '-';
								} else {
									_tmp4 = FALSE;
								}
								if (_tmp4) {
									_tmp3 = self->priv->current[2] == '>';
								} else {
									_tmp3 = FALSE;
								}
								if (_tmp3) {
									/* end of comment*/
									self->priv->current = self->priv->current + 3;
									break;
								}
								self->priv->current++;
							}
							/* ignore comment, read next token*/
							return vala_markup_reader_read_token (self, &(*token_begin), &(*token_end));
						}
					} else {
						if (self->priv->current[0] == '/') {
							char* _tmp6;
							gboolean _tmp7;
							type = VALA_MARKUP_TOKEN_TYPE_END_ELEMENT;
							self->priv->current++;
							_tmp6 = NULL;
							vala_markup_reader_set_name (self, _tmp6 = vala_markup_reader_read_name (self));
							_tmp6 = (g_free (_tmp6), NULL);
							_tmp7 = FALSE;
							if (self->priv->current >= self->priv->end) {
								_tmp7 = TRUE;
							} else {
								_tmp7 = self->priv->current[0] != '>';
							}
							if (_tmp7) {
							}
							/* error*/
							self->priv->current++;
						} else {
							char* _tmp8;
							gboolean _tmp15;
							type = VALA_MARKUP_TOKEN_TYPE_START_ELEMENT;
							_tmp8 = NULL;
							vala_markup_reader_set_name (self, _tmp8 = vala_markup_reader_read_name (self));
							_tmp8 = (g_free (_tmp8), NULL);
							vala_markup_reader_space (self);
							while (TRUE) {
								gboolean _tmp9;
								gboolean _tmp10;
								char* attr_name;
								gboolean _tmp11;
								gboolean _tmp12;
								gchar* attr_begin;
								char* attr_value;
								gboolean _tmp14;
								_tmp9 = FALSE;
								_tmp10 = FALSE;
								if (self->priv->current < self->priv->end) {
									_tmp10 = self->priv->current[0] != '>';
								} else {
									_tmp10 = FALSE;
								}
								if (_tmp10) {
									_tmp9 = self->priv->current[0] != '/';
								} else {
									_tmp9 = FALSE;
								}
								if (!_tmp9) {
									break;
								}
								attr_name = vala_markup_reader_read_name (self);
								_tmp11 = FALSE;
								if (self->priv->current >= self->priv->end) {
									_tmp11 = TRUE;
								} else {
									_tmp11 = self->priv->current[0] != '=';
								}
								if (_tmp11) {
								}
								/* error*/
								self->priv->current++;
								_tmp12 = FALSE;
								if (self->priv->current >= self->priv->end) {
									_tmp12 = TRUE;
								} else {
									_tmp12 = self->priv->current[0] != '"';
								}
								/* FIXME allow single quotes*/
								if (_tmp12) {
								}
								/* error*/
								self->priv->current++;
								attr_begin = self->priv->current;
								while (TRUE) {
									gboolean _tmp13;
									gunichar u;
									_tmp13 = FALSE;
									if (self->priv->current < self->priv->end) {
										_tmp13 = self->priv->current[0] != '"';
									} else {
										_tmp13 = FALSE;
									}
									if (!_tmp13) {
										break;
									}
									u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
									if (u != ((gunichar) (-1))) {
										self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
									} else {
										vala_report_error (NULL, "invalid UTF-8 character");
									}
								}
								/* TODO process &amp; &gt; &lt; &quot; &apos;*/
								attr_value = g_strndup ((const char*) attr_begin, self->priv->current - attr_begin);
								_tmp14 = FALSE;
								if (self->priv->current >= self->priv->end) {
									_tmp14 = TRUE;
								} else {
									_tmp14 = self->priv->current[0] != '"';
								}
								if (_tmp14) {
								}
								/* error*/
								self->priv->current++;
								gee_map_set (self->priv->attributes, attr_name, attr_value);
								vala_markup_reader_space (self);
								attr_name = (g_free (attr_name), NULL);
								attr_value = (g_free (attr_value), NULL);
							}
							if (self->priv->current[0] == '/') {
								self->priv->empty_element = TRUE;
								self->priv->current++;
								vala_markup_reader_space (self);
							} else {
								self->priv->empty_element = FALSE;
							}
							_tmp15 = FALSE;
							if (self->priv->current >= self->priv->end) {
								_tmp15 = TRUE;
							} else {
								_tmp15 = self->priv->current[0] != '>';
							}
							if (_tmp15) {
							}
							/* error*/
							self->priv->current++;
						}
					}
				}
			}
		} else {
			gchar* text_begin;
			vala_markup_reader_space (self);
			text_begin = self->priv->current;
			while (TRUE) {
				gboolean _tmp16;
				gunichar u;
				_tmp16 = FALSE;
				if (self->priv->current < self->priv->end) {
					_tmp16 = self->priv->current[0] != '<';
				} else {
					_tmp16 = FALSE;
				}
				if (!_tmp16) {
					break;
				}
				u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
				if (u != ((gunichar) (-1))) {
					self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
				} else {
					vala_report_error (NULL, "invalid UTF-8 character");
				}
			}
			if (text_begin == self->priv->current) {
				/* no text
				 read next token*/
				return vala_markup_reader_read_token (self, &(*token_begin), &(*token_end));
			}
			type = VALA_MARKUP_TOKEN_TYPE_TEXT;
		}
	}
	/* TODO process &amp; &gt; &lt; &quot; &apos;
	 string text = ((string) text_begin).ndup (current - text_begin);*/
	self->priv->column = self->priv->column + ((gint) (self->priv->current - begin));
	(*token_end).pos = self->priv->current;
	(*token_end).line = self->priv->line;
	(*token_end).column = self->priv->column - 1;
	return type;
}


static void vala_markup_reader_space (ValaMarkupReader* self) {
	g_return_if_fail (self != NULL);
	while (TRUE) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if (self->priv->current < self->priv->end) {
			_tmp0 = g_ascii_isspace (self->priv->current[0]);
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		if (self->priv->current[0] == '\n') {
			self->priv->line++;
			self->priv->column = 0;
		}
		self->priv->current++;
		self->priv->column++;
	}
}


const char* vala_markup_reader_get_filename (ValaMarkupReader* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_filename;
}


static void vala_markup_reader_set_filename (ValaMarkupReader* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_filename = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_filename = (g_free (self->priv->_filename), NULL), _tmp2);
	g_object_notify ((GObject *) self, "filename");
}


const char* vala_markup_reader_get_name (ValaMarkupReader* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_name;
}


static void vala_markup_reader_set_name (ValaMarkupReader* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_name = (g_free (self->priv->_name), NULL), _tmp2);
	g_object_notify ((GObject *) self, "name");
}


static GObject * vala_markup_reader_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	ValaMarkupReaderClass * klass;
	GObjectClass * parent_class;
	ValaMarkupReader * self;
	GError * inner_error;
	klass = VALA_MARKUP_READER_CLASS (g_type_class_peek (VALA_TYPE_MARKUP_READER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = VALA_MARKUP_READER (obj);
	inner_error = NULL;
	{
		{
			GMappedFile* _tmp0;
			_tmp0 = NULL;
			self->priv->mapped_file = (_tmp0 = g_mapped_file_new (self->priv->_filename, FALSE, &inner_error), (self->priv->mapped_file == NULL) ? NULL : (self->priv->mapped_file = (g_mapped_file_free (self->priv->mapped_file), NULL)), _tmp0);
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch3_g_file_error;
				}
				goto __finally3;
			}
			self->priv->begin = g_mapped_file_get_contents (self->priv->mapped_file);
			self->priv->end = self->priv->begin + g_mapped_file_get_length (self->priv->mapped_file);
			self->priv->current = self->priv->begin;
			self->priv->line = 1;
			self->priv->column = 1;
		}
		goto __finally3;
		__catch3_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp1;
				_tmp1 = NULL;
				vala_report_error (NULL, _tmp1 = g_strdup_printf ("Unable to map file `%s': %s", self->priv->_filename, e->message));
				_tmp1 = (g_free (_tmp1), NULL);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally3:
		if (inner_error != NULL) {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
	}
	return obj;
}


static void vala_markup_reader_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaMarkupReader * self;
	gpointer boxed;
	self = VALA_MARKUP_READER (object);
	switch (property_id) {
		case VALA_MARKUP_READER_FILENAME:
		g_value_set_string (value, vala_markup_reader_get_filename (self));
		break;
		case VALA_MARKUP_READER_NAME:
		g_value_set_string (value, vala_markup_reader_get_name (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_markup_reader_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaMarkupReader * self;
	self = VALA_MARKUP_READER (object);
	switch (property_id) {
		case VALA_MARKUP_READER_FILENAME:
		vala_markup_reader_set_filename (self, g_value_get_string (value));
		break;
		case VALA_MARKUP_READER_NAME:
		vala_markup_reader_set_name (self, g_value_get_string (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_markup_reader_class_init (ValaMarkupReaderClass * klass) {
	vala_markup_reader_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMarkupReaderPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_markup_reader_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_markup_reader_set_property;
	G_OBJECT_CLASS (klass)->constructor = vala_markup_reader_constructor;
	G_OBJECT_CLASS (klass)->finalize = vala_markup_reader_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MARKUP_READER_FILENAME, g_param_spec_string ("filename", "filename", "filename", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MARKUP_READER_NAME, g_param_spec_string ("name", "name", "name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_markup_reader_instance_init (ValaMarkupReader * self) {
	self->priv = VALA_MARKUP_READER_GET_PRIVATE (self);
	self->priv->attributes = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
}


static void vala_markup_reader_finalize (GObject* obj) {
	ValaMarkupReader * self;
	self = VALA_MARKUP_READER (obj);
	self->priv->_filename = (g_free (self->priv->_filename), NULL);
	self->priv->_name = (g_free (self->priv->_name), NULL);
	(self->priv->mapped_file == NULL) ? NULL : (self->priv->mapped_file = (g_mapped_file_free (self->priv->mapped_file), NULL));
	(self->priv->attributes == NULL) ? NULL : (self->priv->attributes = (gee_collection_object_unref (self->priv->attributes), NULL));
	G_OBJECT_CLASS (vala_markup_reader_parent_class)->finalize (obj);
}


GType vala_markup_reader_get_type (void) {
	static GType vala_markup_reader_type_id = 0;
	if (vala_markup_reader_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMarkupReaderClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_markup_reader_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMarkupReader), 0, (GInstanceInitFunc) vala_markup_reader_instance_init, NULL };
		vala_markup_reader_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaMarkupReader", &g_define_type_info, 0);
	}
	return vala_markup_reader_type_id;
}


const char* vala_markup_token_type_to_string (ValaMarkupTokenType self) {
	switch (self) {
		case VALA_MARKUP_TOKEN_TYPE_START_ELEMENT:
		{
			return "start element";
		}
		case VALA_MARKUP_TOKEN_TYPE_END_ELEMENT:
		{
			return "end element";
		}
		case VALA_MARKUP_TOKEN_TYPE_TEXT:
		{
			return "text";
		}
		case VALA_MARKUP_TOKEN_TYPE_EOF:
		{
			return "end of file";
		}
		default:
		{
			return "unknown token type";
		}
	}
}



GType vala_markup_token_type_get_type (void) {
	static GType vala_markup_token_type_type_id = 0;
	if (G_UNLIKELY (vala_markup_token_type_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_MARKUP_TOKEN_TYPE_NONE, "VALA_MARKUP_TOKEN_TYPE_NONE", "none"}, {VALA_MARKUP_TOKEN_TYPE_START_ELEMENT, "VALA_MARKUP_TOKEN_TYPE_START_ELEMENT", "start-element"}, {VALA_MARKUP_TOKEN_TYPE_END_ELEMENT, "VALA_MARKUP_TOKEN_TYPE_END_ELEMENT", "end-element"}, {VALA_MARKUP_TOKEN_TYPE_TEXT, "VALA_MARKUP_TOKEN_TYPE_TEXT", "text"}, {VALA_MARKUP_TOKEN_TYPE_EOF, "VALA_MARKUP_TOKEN_TYPE_EOF", "eof"}, {0, NULL, NULL}};
		vala_markup_token_type_type_id = g_enum_register_static ("ValaMarkupTokenType", values);
	}
	return vala_markup_token_type_type_id;
}




