/* wptGPGZip.cpp
 *        Copyright (C) 2002-2004 Timo Schulz
 *
 * This file is part of PTD.
 *
 * PTD is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PTD is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PTD; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <direct.h>
#include <time.h>

#include "wptGPGZIP.h"

static int debug = 0;

enum {
    PK_LOCAL_SIG = 0x04034b50,
    PK_FILE_SIG  = 0x02014b50,
    PK_END_SIG   = 0x06054b50
};

enum {
    PK_FLAG_ENCRYPTED = 1,
    PK_FLAG_8KDICT    = 2,
    PK_FLAG_SF        = 4
};


typedef struct {
    unsigned long   magic;
    unsigned short  ver;
    unsigned short  flags;
    unsigned short  method;
    unsigned short  ftime;
    unsigned short  fdate;
    unsigned long   crc;
    unsigned long   c_size;
    unsigned long   u_size;
    unsigned short  f_len;
    unsigned short  extra_len;
    char *	    name;
    FILE *	    fp;
} PK_local_hdr;

typedef struct {
    unsigned long   magic;
    unsigned short  ver_made;
    unsigned short  ver_ext;
    unsigned short  flags;
    unsigned short  method;
    unsigned short  ftime;
    unsigned short  fdate;
    unsigned long   crc;
    unsigned long   c_size;
    unsigned long   u_size;
    unsigned short  f_len;
    unsigned short  extra_len;
    unsigned short  comment_len;
    unsigned short  disc_nr;
    unsigned short  int_attr;
    unsigned long   ext_attr;
    unsigned long   lochdr_off;
    char *	    name;
    size_t	    size;
} PK_file_hdr;

typedef struct {
    unsigned long   magic;
    unsigned short  disc_nr;
    unsigned short  disc_nr_cd;
    unsigned short  total_dirs_disc;
    unsigned short  total_dirs;
    unsigned long   size_of_dir;
    unsigned long   off_dir;
    unsigned short  comment_len;
    char *	    comment;
} PK_end_hdr;


typedef struct {
    union {
        PK_local_hdr * local;
        PK_file_hdr  * file;
        PK_end_hdr   * end;
    } u;
    unsigned long pkttype;
} PK_packet;


struct PK_file_list {
    struct PK_file_list *   next;
    PK_local_hdr *	    hdr;
    size_t		    off;
    size_t		    len;
    char		    d[1];
};


static unsigned long crc_table[256] = {
    0x00000000L, 0x77073096L, 0xee0e612cL, 0x990951baL, 0x076dc419L,
    0x706af48fL, 0xe963a535L, 0x9e6495a3L, 0x0edb8832L, 0x79dcb8a4L,
    0xe0d5e91eL, 0x97d2d988L, 0x09b64c2bL, 0x7eb17cbdL, 0xe7b82d07L,
    0x90bf1d91L, 0x1db71064L, 0x6ab020f2L, 0xf3b97148L, 0x84be41deL,
    0x1adad47dL, 0x6ddde4ebL, 0xf4d4b551L, 0x83d385c7L, 0x136c9856L,
    0x646ba8c0L, 0xfd62f97aL, 0x8a65c9ecL, 0x14015c4fL, 0x63066cd9L,
    0xfa0f3d63L, 0x8d080df5L, 0x3b6e20c8L, 0x4c69105eL, 0xd56041e4L,
    0xa2677172L, 0x3c03e4d1L, 0x4b04d447L, 0xd20d85fdL, 0xa50ab56bL,
    0x35b5a8faL, 0x42b2986cL, 0xdbbbc9d6L, 0xacbcf940L, 0x32d86ce3L,
    0x45df5c75L, 0xdcd60dcfL, 0xabd13d59L, 0x26d930acL, 0x51de003aL,
    0xc8d75180L, 0xbfd06116L, 0x21b4f4b5L, 0x56b3c423L, 0xcfba9599L,
    0xb8bda50fL, 0x2802b89eL, 0x5f058808L, 0xc60cd9b2L, 0xb10be924L,
    0x2f6f7c87L, 0x58684c11L, 0xc1611dabL, 0xb6662d3dL, 0x76dc4190L,
    0x01db7106L, 0x98d220bcL, 0xefd5102aL, 0x71b18589L, 0x06b6b51fL,
    0x9fbfe4a5L, 0xe8b8d433L, 0x7807c9a2L, 0x0f00f934L, 0x9609a88eL,
    0xe10e9818L, 0x7f6a0dbbL, 0x086d3d2dL, 0x91646c97L, 0xe6635c01L,
    0x6b6b51f4L, 0x1c6c6162L, 0x856530d8L, 0xf262004eL, 0x6c0695edL,
    0x1b01a57bL, 0x8208f4c1L, 0xf50fc457L, 0x65b0d9c6L, 0x12b7e950L,
    0x8bbeb8eaL, 0xfcb9887cL, 0x62dd1ddfL, 0x15da2d49L, 0x8cd37cf3L,
    0xfbd44c65L, 0x4db26158L, 0x3ab551ceL, 0xa3bc0074L, 0xd4bb30e2L,
    0x4adfa541L, 0x3dd895d7L, 0xa4d1c46dL, 0xd3d6f4fbL, 0x4369e96aL,
    0x346ed9fcL, 0xad678846L, 0xda60b8d0L, 0x44042d73L, 0x33031de5L,
    0xaa0a4c5fL, 0xdd0d7cc9L, 0x5005713cL, 0x270241aaL, 0xbe0b1010L,
    0xc90c2086L, 0x5768b525L, 0x206f85b3L, 0xb966d409L, 0xce61e49fL,
    0x5edef90eL, 0x29d9c998L, 0xb0d09822L, 0xc7d7a8b4L, 0x59b33d17L,
    0x2eb40d81L, 0xb7bd5c3bL, 0xc0ba6cadL, 0xedb88320L, 0x9abfb3b6L,
    0x03b6e20cL, 0x74b1d29aL, 0xead54739L, 0x9dd277afL, 0x04db2615L,
    0x73dc1683L, 0xe3630b12L, 0x94643b84L, 0x0d6d6a3eL, 0x7a6a5aa8L,
    0xe40ecf0bL, 0x9309ff9dL, 0x0a00ae27L, 0x7d079eb1L, 0xf00f9344L,
    0x8708a3d2L, 0x1e01f268L, 0x6906c2feL, 0xf762575dL, 0x806567cbL,
    0x196c3671L, 0x6e6b06e7L, 0xfed41b76L, 0x89d32be0L, 0x10da7a5aL,
    0x67dd4accL, 0xf9b9df6fL, 0x8ebeeff9L, 0x17b7be43L, 0x60b08ed5L,
    0xd6d6a3e8L, 0xa1d1937eL, 0x38d8c2c4L, 0x4fdff252L, 0xd1bb67f1L,
    0xa6bc5767L, 0x3fb506ddL, 0x48b2364bL, 0xd80d2bdaL, 0xaf0a1b4cL,
    0x36034af6L, 0x41047a60L, 0xdf60efc3L, 0xa867df55L, 0x316e8eefL,
    0x4669be79L, 0xcb61b38cL, 0xbc66831aL, 0x256fd2a0L, 0x5268e236L,
    0xcc0c7795L, 0xbb0b4703L, 0x220216b9L, 0x5505262fL, 0xc5ba3bbeL,
    0xb2bd0b28L, 0x2bb45a92L, 0x5cb36a04L, 0xc2d7ffa7L, 0xb5d0cf31L,
    0x2cd99e8bL, 0x5bdeae1dL, 0x9b64c2b0L, 0xec63f226L, 0x756aa39cL,
    0x026d930aL, 0x9c0906a9L, 0xeb0e363fL, 0x72076785L, 0x05005713L,
    0x95bf4a82L, 0xe2b87a14L, 0x7bb12baeL, 0x0cb61b38L, 0x92d28e9bL,
    0xe5d5be0dL, 0x7cdcefb7L, 0x0bdbdf21L, 0x86d3d2d4L, 0xf1d4e242L,
    0x68ddb3f8L, 0x1fda836eL, 0x81be16cdL, 0xf6b9265bL, 0x6fb077e1L,
    0x18b74777L, 0x88085ae6L, 0xff0f6a70L, 0x66063bcaL, 0x11010b5cL,
    0x8f659effL, 0xf862ae69L, 0x616bffd3L, 0x166ccf45L, 0xa00ae278L,
    0xd70dd2eeL, 0x4e048354L, 0x3903b3c2L, 0xa7672661L, 0xd06016f7L,
    0x4969474dL, 0x3e6e77dbL, 0xaed16a4aL, 0xd9d65adcL, 0x40df0b66L,
    0x37d83bf0L, 0xa9bcae53L, 0xdebb9ec5L, 0x47b2cf7fL, 0x30b5ffe9L,
    0xbdbdf21cL, 0xcabac28aL, 0x53b39330L, 0x24b4a3a6L, 0xbad03605L,
    0xcdd70693L, 0x54de5729L, 0x23d967bfL, 0xb3667a2eL, 0xc4614ab8L,
    0x5d681b02L, 0x2a6f2b94L, 0xb40bbe37L, 0xc30c8ea1L, 0x5a05df1bL,
    0x2d02ef8dL
};


static unsigned long
update_crc( unsigned long crc, unsigned char * buf, size_t buflen )
{
    crc ^= 0xffffffffL;
    while( buflen-- )
        crc = crc_table[((int)(crc) ^ (*buf++)) & 0xff] ^ ((crc) >> 8);
    crc ^= 0xffffffffL;
    return crc;
}

        
static unsigned short
read_16( FILE * in )
{
    unsigned short u;
    u  = fgetc( in )     ;
    u |= fgetc( in ) << 8;
    return u;
}


static void
write_16( FILE * out, unsigned short a )
{
    fputc( a     , out );
    fputc( a >> 8, out );
}


static unsigned long
read_32( FILE * in )
{
    unsigned long u;
    u  = fgetc( in )      ;
    u |= fgetc( in ) <<  8;
    u |= fgetc( in ) << 16;
    u |= fgetc( in ) << 24;
    return u;
}


static void
write_32( FILE * out, unsigned long a )
{
    fputc( a      , out );
    fputc( a >> 8 , out );
    fputc( a >> 16, out );
    fputc( a >> 24, out );
}


#if 0 /* not used */
static void
date_from_dos( unsigned short t, int *day, int *mon, int *year )
{
    if( mon )
        *mon = t % 32;
    if( day )
        *day = (t>>5) % 16;
    if( year )
        *year = (t>>9) % 64;   
}
#endif

static unsigned short
date_to_dos( time_t t )
{
    struct tm * tm;
    tm = localtime( &t );
    return tm->tm_mday | (tm->tm_mon+1)<<5 | ((tm->tm_year-80)<<9);
}


static unsigned short
time_to_dos( time_t t )
{
    struct tm * tm;
    tm = localtime( &t );
    return (tm->tm_hour<<11) | (tm->tm_min<<5) | (tm->tm_sec>>1);
}


static char *
m_strdup (const char * s)
{
    char * p = new char[strlen (s)+1];
    if (p)
	strcpy (p, s);
    return p;
}


static PK_local_hdr *
create_local_header( const char * name, FILE * fp )
{
    struct stat statbuf;
    PK_local_hdr * a;

    if( fstat( fileno( fp ), &statbuf ) )
        return NULL;
    a = new PK_local_hdr;
    if( !a )
        return NULL;
    memset (a, 0, sizeof *a);
    a->magic = PK_LOCAL_SIG;
    a->ver = 0 | (10 << 8);
    a->flags = 0;
    a->method = 0;
    a->fdate = date_to_dos( statbuf.st_mtime );
    a->ftime = time_to_dos( statbuf.st_mtime );
    a->crc = 0;
    a->c_size = statbuf.st_size;
    a->u_size = statbuf.st_size;
    a->f_len = strlen( name );
    a->name = m_strdup( name );
    a->extra_len = 0;
    a->fp = fp;
    return a;
}


static int
write_local_header( FILE * out, PK_local_hdr * hdr )
{
    long crc_pos = 0, curr_pos = 0;
    
    write_32( out, hdr->magic );
    write_16( out, hdr->ver );
    write_16( out, hdr->flags );
    write_16( out, hdr->method );
    write_16( out, hdr->ftime );
    write_16( out, hdr->fdate );
    crc_pos = ftell( out );
    write_32( out, hdr->crc );
    write_32( out, hdr->c_size );
    write_32( out, hdr->u_size );
    write_16( out, hdr->f_len );
    write_16( out, hdr->extra_len );
    if( hdr->f_len )
        fwrite( hdr->name, 1, hdr->f_len, out );
    if( hdr->extra_len )
        fwrite( NULL, 1, hdr->extra_len, out ); /* xxx */
    while (!feof (hdr->fp))
    {
        unsigned char buf[1024];
        unsigned long nread;
        nread = fread( buf, 1, sizeof buf-1, hdr->fp );
        if( nread ) {
	    /*fprintf( stderr, "** nread=%d\n", nread );*/
            hdr->crc = update_crc( hdr->crc, buf, nread );
            fwrite( buf, 1, nread, out );
        }
    }
    curr_pos = ftell( out );
    fseek( out, crc_pos, SEEK_SET );
    write_32( out, hdr->crc );
    fseek( out, curr_pos, SEEK_SET );
    return 0;
}


static int
create_missing_dirs( const char * name )
{
    struct stat dirbuf;
    char *p, *dir;
    int rc;
    
    p = strrchr( name, '/' );
    if( !p )
        return PKERR_GENERAL;
    dir = new char[(p-name)+1];
    if( !dir )
	return -1;
    memset (dir, 0, (p-name)+1);
    strncpy( dir, name, (p-name) );
    if( !stat( dir, &dirbuf ) && ( dirbuf.st_mode & _S_IFDIR ) ) {
        delete dir;
        return 0;
    }
    rc = mkdir( dir );
    delete dir;
    return rc;
}


static PK_local_hdr *
read_local_header( FILE * in, unsigned long magic, int create )
{
    PK_local_hdr * hdr;
    FILE * out = NULL;
    size_t n;

    if( magic != PK_LOCAL_SIG )
        return NULL;
    hdr = new PK_local_hdr;
    if( !hdr )
	return NULL;
    memset (hdr, 0, sizeof * hdr);
    hdr->magic = magic;
    hdr->ver = read_16( in );
    hdr->flags = read_16( in );
    hdr->method = read_16( in );
    hdr->ftime = read_16( in );
    hdr->fdate = read_16( in );
    hdr->crc = read_32( in );
    hdr->c_size = read_32( in );
    hdr->u_size = read_32( in );
    hdr->f_len = read_16( in );
    hdr->extra_len = read_16( in );

    /*if( debug ) {
        printf("=====BEGIN LOCAL HEADER=====\n");
        printf( "magic %08lX version=%d\n", hdr->magic, hdr->ver );
        printf( "flags encrypted=%d 8k dict=%d shannon=%d\n",
                hdr->flags & PK_FLAG_ENCRYPTED, hdr->flags & PK_FLAG_8KDICT,
                hdr->flags & PK_FLAG_SF );
        printf( "method=" );
        switch( hdr->method ) {
        case 0: printf( "stored\n" ); break;
        default: printf( "unknown (%d)\n", hdr->method );
        }
        printf( "filetime=%d %d\n", hdr->ftime, hdr->fdate );
        printf( "CRC=%08lX\n", hdr->crc );
        printf( "size: compressed=%lu uncompressed=%lu\n",
                hdr->c_size, hdr->u_size );
        printf( "filename len=%d extra_len=%d\n", hdr->f_len, hdr->extra_len );
    }*/

    if( hdr->f_len ) {
        hdr->name = new char[hdr->f_len+1];
	if( !hdr->name )
	    return NULL;
	memset (hdr->name, 0, hdr->f_len+1);
        fread( hdr->name, 1, hdr->f_len, in );
        /*if( debug )
            printf( "filename %s\n", hdr->name );*/
    }
    /*if( debug )
        printf( "skip extra header (%d)\n", hdr->extra_len );*/
    n = hdr->extra_len;
    while( n-- )
        fgetc( in );
    /*if( debug )
        printf( "skip compressed data (%lu)\n", hdr->c_size );*/
    if( create && hdr->name ) {
        create_missing_dirs( hdr->name );
        out = fopen( hdr->name, "wb" );
        /*if( debug )
            printf( "create output `%s' (status %s)\n", hdr->name,
                    out? "success" : "failed" );*/
    }
    n = hdr->c_size;
    while( n-- ) {
        int c = fgetc( in );
        if( out )
            fputc( c, out );
    }
    if( out )
        fclose( out );
    return hdr;
}


static PK_file_hdr *
create_file_header( PK_local_hdr * loc, long loc_off )
{
    PK_file_hdr * a;
    
    a = new PK_file_hdr;
    if( !a )
        return NULL;
    memset (a, 0, sizeof *a);
    a->magic = PK_FILE_SIG;
    a->ver_made = loc->ver;
    a->ver_ext = loc->ver;
    a->flags = loc->flags;
    a->method = loc->method;
    a->ftime = loc->ftime;
    a->fdate = loc->fdate;
    a->crc = loc->crc;
    a->c_size = loc->c_size;
    a->u_size = loc->u_size;
    a->f_len = loc->f_len;
    a->extra_len = loc->extra_len;
    a->comment_len = 0;
    a->disc_nr = 0;
    a->int_attr = 0;
    a->ext_attr = 0;
    a->lochdr_off = loc_off;
    a->size = 46 + a->f_len;
    a->name = m_strdup( loc->name );
    return a;
}


static int
write_file_header( FILE * out, PK_file_hdr * hdr )
{
    write_32( out, hdr->magic );
    write_16( out, hdr->ver_made );
    write_16( out, hdr->ver_ext );
    write_16( out, hdr->flags );
    write_16( out, hdr->method );
    write_16( out, hdr->ftime );
    write_16( out, hdr->fdate );
    write_32( out, hdr->crc );
    write_32( out, hdr->c_size );
    write_32( out, hdr->u_size );
    write_16( out, hdr->f_len );
    write_16( out, hdr->extra_len );
    write_16( out, hdr->comment_len );
    write_16( out, hdr->disc_nr );
    write_16( out, hdr->int_attr );
    write_32( out, hdr->ext_attr );
    write_32( out, hdr->lochdr_off );
    if( hdr->f_len )
        fwrite( hdr->name, 1, hdr->f_len, out );
    if( hdr->extra_len )
        fwrite( NULL, 1, hdr->extra_len, out ); /* xxx */
    if( hdr->comment_len )
        fwrite( NULL, 1, hdr->comment_len, out ); /* xxx */
    return 0;
}
    

static PK_file_hdr *
read_file_header( FILE * in, unsigned long magic )
{
    PK_file_hdr * hdr;
    size_t n;

    if( magic != PK_FILE_SIG )
        return NULL;
    hdr = new PK_file_hdr;
    if( !hdr )
	return NULL;
    memset (hdr, 0, sizeof * hdr);
    hdr->magic = magic;
    hdr->ver_made = read_16( in );
    hdr->ver_ext = read_16( in );
    hdr->flags = read_16( in );
    hdr->method = read_16( in );
    hdr->ftime = read_16( in );
    hdr->fdate = read_16( in );
    hdr->crc = read_32( in );
    hdr->c_size = read_32( in );
    hdr->u_size = read_32( in );
    hdr->f_len = read_16( in );
    hdr->extra_len = read_16( in );
    hdr->comment_len = read_16( in );
    hdr->disc_nr = read_16( in );
    hdr->int_attr = read_16( in );
    hdr->ext_attr = read_32( in );
    hdr->lochdr_off = read_32( in );

    /*if( debug ) {
        printf("=====BEGIN FILE HEADER=====\n");
        printf( "magic=%08lX\n", hdr->magic );
        printf( "version made=%d needed=%d\n", hdr->ver_made, hdr->ver_ext );
        printf( "flags encrypted=%d endheader=%d extended=%d\n",
                hdr->flags & PK_FLAG_ENCRYPTED, hdr->flags & PK_FLAG_8KDICT,
                hdr->flags & PK_FLAG_SF );
        printf( "method=" );
        switch( hdr->method ) {
        case 0: printf( "stored\n" ); break;
        default: printf( "unknown (%d)\n", hdr->method );   
        }
        printf( "filetime=%d %d\n", hdr->ftime, hdr->fdate );
        printf( "CRC=%08lX\n", hdr->crc );
        printf( "size compressed=%lu uncompressed=%lu\n",
                hdr->c_size, hdr->u_size );
        printf( "extra len=%d comment len=%d\n", hdr->extra_len,
                hdr->comment_len );
        printf( "disk nummer=%d\n", hdr->disc_nr );
        printf( "attrs intern=%d extern=%lu\n", hdr->int_attr, hdr->ext_attr );
        printf( "disk offset=%lu\n", hdr->lochdr_off );
    }*/
    
    if( hdr->f_len ) {
        hdr->name = new char[hdr->f_len + 1];
	if( !hdr->name )
	    return NULL;
	memset (hdr->name, 0, hdr->f_len+1);
        fread( hdr->name, 1, hdr->f_len, in );
        /*if( debug )
            printf( "filename=%s\n", hdr->name );*/
    }
    n = hdr->extra_len;
    while( n-- )
        fgetc( in );
    n = hdr->comment_len;
    while( n-- )
        fgetc( in );
    return hdr;
}


static PK_end_hdr *
create_end_header( size_t ntotal, size_t nsize, long off_dir )
{
    PK_end_hdr * a;

    a = new PK_end_hdr;
    if( !a )
        return NULL;
    memset (a, 0, sizeof * a);
    a->magic = PK_END_SIG;
    a->disc_nr = 0;
    a->disc_nr_cd = 0;
    a->total_dirs = ntotal;
    a->total_dirs_disc = ntotal;
    a->size_of_dir = nsize;
    a->off_dir = off_dir;
    a->comment_len = 0;
    return a;
}


static int
write_end_header( FILE * out, PK_end_hdr * hdr )
{
    write_32( out, hdr->magic );
    write_16( out, hdr->disc_nr );
    write_16( out, hdr->disc_nr_cd );
    write_16( out, hdr->total_dirs );
    write_16( out, hdr->total_dirs_disc );
    write_32( out, hdr->size_of_dir );
    write_32( out, hdr->off_dir );
    write_16( out, hdr->comment_len );
    if( hdr->comment_len )
        fwrite( hdr->comment, 1, hdr->comment_len, out );
    return 0;
}


static PK_end_hdr *
read_end_header( FILE * in, unsigned long magic )
{
    PK_end_hdr * hdr;

    if( magic != PK_END_SIG )
        return NULL;
    hdr = new PK_end_hdr;
    if( !hdr )
	return NULL;
    memset (hdr, 0, sizeof *hdr);
    hdr->magic = magic;
    hdr->disc_nr = read_16( in );
    hdr->disc_nr_cd = read_16( in );
    hdr->total_dirs = read_16( in );
    hdr->total_dirs_disc = read_16( in );
    hdr->size_of_dir = read_32( in );
    hdr->off_dir = read_32( in );
    hdr->comment_len = read_16( in );

    /*if( debug ) {
        printf("=====BEGIN END HEADER=====\n");
        printf( "magic=%08lX\n", hdr->magic );
        printf( "disc number=%d in this directory=%d\n", hdr->disc_nr,
                hdr->disc_nr_cd );
        printf( "total number of disc=%d in this directory=%d\n",
                hdr->total_dirs, hdr->total_dirs_disc );
        printf( "offset of central directory=%lu\n", hdr->off_dir );
        printf( "size of directory=%lu\n", hdr->size_of_dir );
    }*/
    
    if( hdr->comment_len ) {
	hdr->comment = new char[hdr->comment_len+1];       
	if( !hdr->comment )
	    return NULL;
	memset (hdr->comment, 0, hdr->comment_len+1);
        fread( hdr->comment, 1, hdr->comment_len, in );
        /*if( debug )
            printf( "comment(%d)=%s\n", hdr->comment_len, hdr->comment );*/
    }
    return hdr;
}


static void
free_packet( PK_packet * pkt )
{
    switch( pkt->pkttype ) {
    case PK_LOCAL_SIG:
        if( pkt->u.local->name )
            delete []pkt->u.local->name;
        delete pkt->u.local;
        break;
    case PK_FILE_SIG:
        if( pkt->u.file->name )
            delete[] pkt->u.file->name;
        delete pkt->u.file;
        break;
    case PK_END_SIG:
        if( pkt->u.end->comment )
            delete [] pkt->u.end->comment;
        delete pkt->u.end;
        break;
    }
    pkt->pkttype = 0;
}


int
pk_archiv_parse( FILE * in )
{
    unsigned long magic;
    PK_packet pkt;
    
    while( !feof( in ) ) {
        magic = read_32( in );
        if( feof( in ) )
            break;
        /*printf("\n** read magic value %08lX\n\n", magic );*/
        switch( magic ) {
        case PK_LOCAL_SIG:
            pkt.u.local = read_local_header( in, magic, 1 );
            pkt.pkttype = PK_LOCAL_SIG;
            break;
            
        case PK_FILE_SIG:
            pkt.u.file = read_file_header( in, magic );
            pkt.pkttype = PK_FILE_SIG;
            break;
            
        case PK_END_SIG:
            pkt.u.end = read_end_header( in, magic );
            pkt.pkttype = PK_END_SIG;
            break;
            
        default:
            /*printf( "unknown magic value (%08lX).\n", magic );*/
            if( ftell( in ) == 4 ) {
                free_packet( &pkt );
                return PKERR_INV_FILE;
            }
            break;
        }
        free_packet( &pkt );
    }
    return 0;
}


static int
is_directory( const char * fname )
{
    struct stat statbuf;
    if( stat( fname, &statbuf ) )
	return 0;
    return statbuf.st_mode & S_IFDIR;
}


int
pk_archiv_create( struct PK_file_list * list, const char * output )
{
    struct PK_file_list * r;
    PK_file_hdr * f;
    PK_end_hdr * e;
    PK_packet pkt;
    FILE * fp, * out;
    long foff = 0, n = 0;
    size_t size = 0;

    out = fopen( output, "wb" );
    if( !out )
        return PKERR_FILE;
    
    for( r = list; r; r = r->next ) 
    {
        /*fprintf( stderr, "process file `%s'\n", r->d );*/
	if( is_directory( r->d ) )
	    continue;
        fp = fopen( r->d, "rb" );
        if( !fp )
	{
	    fclose (out);
            return PKERR_FILE;
	}
        r->off = ftell( out );
        r->hdr = create_local_header( r->d, fp );
        if (!r->hdr)
	{
            fclose (fp);
	    fclose (out);
            return PKERR_GENERAL;
        }
        if( r->off )
            r->off += r->hdr->c_size;
        write_local_header( out, r->hdr );
        n++;
        fclose( fp );
    }
    foff = ftell( out );
    for( r = list; r; r = r->next ) {
	if( is_directory( r->d ) )
	    continue;
        fp = fopen( r->d, "rb" );
        if( !fp )
	{
	    fclose (out);
            return PKERR_FILE;
	}
        f = create_file_header( r->hdr, r->off );
        if (!f)
	{
	    fclose (out);
            fclose (fp);
            return PKERR_GENERAL;
        }
        size += f->size;
        write_file_header( out, f );
        fclose (fp);
        pkt.u.file = f;
        free_packet (&pkt);
    }
    
    e = create_end_header (n, size, foff);
    if (!e)
    {
	fclose (out);
        return PKERR_GENERAL;
    }
    write_end_header (out, e);
    fclose (out);
    return 0;
}


void
pk_list_add( struct PK_file_list **list, const char * name )
{
    struct PK_file_list * l;

    l = new PK_file_list;
    if (l)
	memset (l, 0, sizeof *l);
    if( l ) {
	l->next = *list;
	strcpy( l->d, name );
	*list = l;
    }
}


void
pk_list_free( struct PK_file_list * list )
{
    struct PK_file_list * l;
    PK_packet pkt;

    while( list ) {
        l = list->next;
	if( list->hdr ) {
	    pkt.pkttype = PK_LOCAL_SIG;
	    pkt.u.local = list->hdr;
	    free_packet( &pkt );
	}
        delete list;
        list = l;
    }
}

