/*
 * gaia - opensource 3D interface to the planet
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "gefetch_internal.h"

/**
 * Internal function to clean stale metadata chunks
 *
 * Checks whether summary size of all metadata currently stored
 * in cache exceeds maximum value.
 *
 * @param extraspace reserve extra space in cache
 */
void gefetch_remove_extra_meta(gefetch *handle, size_t extraspace) {
	while (handle->metasize > handle->maxmetasize - extraspace && handle->firstmeta != handle->lastmeta) {
		gefetch_meta *last = handle->lastmeta;

		last->prev->next = 0;
		handle->lastmeta = last->prev;

		handle->metasize -= sizeof(gefetch_meta) + sizeof(gefetch_meta_entry) * last->nentries;

		free(last);
	}
}

/**
 * Internal function to recursively parse metadata chunk
 *
 * @param currentry pointer to metadata entry currently being parsed
 * @param badentry pointed to metadata entry that lies just after last metadata entry in chunk, for bound checking
 * @param x current x coordinate (relative to metadata chunk)
 * @param y current y coordinate (relative to metadata chunk)
 * @param level current level (relative to metadata chunk)
 * @param tx target x coordinate (relative to metadata chunk)
 * @param ty target y coordinate (relative to metadata chunk)
 * @param tlevel target level (relative to metadata chunk)
 * @param resentry where to place pointer to found metadata entry
 */
static gefetch_error gefetch_parse_meta_recursive(gefetch *handle, gefetch_meta_entry **currentry, gefetch_meta_entry *badentry, int x, int y, int level, int tx, int ty, int tlevel, gefetch_meta_entry **resentry) {
	gefetch_meta_entry *thisentry = *currentry;

	/* check bounds */
	if (thisentry == badentry)
		return GEFETCH_CORRUPT_DATA;

	/* check for success */
	if (x == tx && y == ty && level == tlevel) {
		*resentry = thisentry;
		return GEFETCH_OK;
	}

	/* descend */
	gefetch_error res;

	if (thisentry->flags & GEFETCH_META_FLAG_CHILD0) {
		(*currentry)++;
		if ((res = gefetch_parse_meta_recursive(handle, currentry, badentry,x*2, y*2, level+1, tx, ty, tlevel, resentry)) != GEFETCH_NOT_FOUND)
			return res;
	}

	if (thisentry->flags & GEFETCH_META_FLAG_CHILD1) {
		(*currentry)++;
		if ((res = gefetch_parse_meta_recursive(handle, currentry, badentry, x*2+1, y*2, level+1, tx, ty, tlevel, resentry)) != GEFETCH_NOT_FOUND)
			return res;
	}

	if (thisentry->flags & GEFETCH_META_FLAG_CHILD2) {
		(*currentry)++;
		if ((res = gefetch_parse_meta_recursive(handle, currentry, badentry, x*2+1, y*2+1, level+1, tx, ty, tlevel, resentry)) != GEFETCH_NOT_FOUND)
			return res;
	}

	if (thisentry->flags & GEFETCH_META_FLAG_CHILD3) {
		(*currentry)++;
		if ((res = gefetch_parse_meta_recursive(handle, currentry, badentry, x*2, y*2+1, level+1, tx, ty, tlevel, resentry)) != GEFETCH_NOT_FOUND)
			return res;
	}

	return GEFETCH_NOT_FOUND;
}

/**
 * Internal function to parse metadata chunk
 *
 * Parses chunk of metadata to find info on specific (x, y, level) object. Actually
 * it's wrapper around gefetch_parse_meta_recursive() to ease it's usage.
 *
 * @param meta pointer to gefetch_meta object to parse
 * @param x x coordinate of object for which metadata is needed
 * @param y y coordinate of object for which metadata is needed
 * @param level level of object for which metadata is needed
 * @param resentry where to place pointer to found metadata entry
 */
gefetch_error gefetch_parse_meta(gefetch *handle, gefetch_meta *meta, int x, int y, int level, gefetch_meta_entry **resentry) {
	int meta_level = level < 4 ? 0 : (level & (~0x3)) - 1;
	int meta_x = x >> (level - meta_level);
	int meta_y = y >> (level - meta_level);

	gefetch_meta_entry *current = meta->entries;

	return gefetch_parse_meta_recursive(handle, &current, meta->entries + meta->nentries, 0, 0, 0,
			x - (meta_x << (level - meta_level)),
			y - (meta_y << (level - meta_level)),
			level - meta_level, resentry);
}

/**
 * Internal function to search for metadata incatalogue
 *
 * libgefetch contains catalogue (or cache) for metadata. It is arranged as
 * double linked list in which most recently accessed elements are stored first.
 * When metadata is requested, cache is searched first. If metadata is found,
 * it's moved to beginning of cache. If not found, it's fetched by calling
 * gefetch_fetch_meta() and, again, placed to beginning of cache.
 * Cache size is limited, see gefetch_set_max_metasize().
 *
 * @param x x coordinate of object for which metadata is needed
 * @param y y coordinate of object for which metadata is needed
 * @param level level of object for which metadata is needed
 * @param resmeta where to place pointer to found metadata (in case it's found, of course)
 */
gefetch_error gefetch_find_meta(gefetch *handle, int x, int y, int level, gefetch_meta **resmeta) {
	int meta_level = level < 4 ? 0 : (level & (~0x3)) - 1;
	int meta_x = x >> (level - meta_level);
	int meta_y = y >> (level - meta_level);

	gefetch_meta *cur;
	for (cur = handle->firstmeta; cur; cur = cur->next) {
		if (cur->x == meta_x && cur->y == meta_y && cur->level == meta_level) {
			/* meta found in catalogue */
			if (cur != handle->firstmeta) {
				/* move found meta to beginning of list, to fasten search
				 * and to ensure that unpopular metas are dropped first */

				/* remove from current postition */
				cur->prev->next = cur->next;
				if (cur->next)	/* not last element */
					cur->next->prev = cur->prev;
				else		/* last element */
					handle->lastmeta = cur->prev;

				/* insert into beginning of list */
				cur->prev = 0;
				cur->next = handle->firstmeta;
				handle->firstmeta->prev = cur;
				handle->firstmeta = cur;
			}

			*resmeta = cur;
			return GEFETCH_OK;
		}
	}

	/* fetch metadata */
	gefetch_error res;
	if ((res = gefetch_fetch_meta(handle, meta_x, meta_y, meta_level)) != GEFETCH_OK)
		return res;

	/* decompress metadata */
	if ((res = gefetch_decompress_current_file(handle)) != GEFETCH_OK)
		return res;

	/* it should have at least header */
	if (gefetch_get_data_size(handle) < sizeof(gefetch_meta_header))
		return GEFETCH_CORRUPT_DATA;

	gefetch_meta_header *hdr = (gefetch_meta_header *)gefetch_get_data_ptr(handle);

	int nmetaentries = ((int)hdr->nentries[0]) |
		((int)hdr->nentries[1] << 8);

	/* it should have at least header + entries */
	if (gefetch_get_data_size(handle) < sizeof(gefetch_meta_header) + sizeof(gefetch_meta_entry)*nmetaentries)
		return GEFETCH_CORRUPT_DATA;

	/* free space for new meta */
	gefetch_remove_extra_meta(handle, sizeof(gefetch_meta) + sizeof(gefetch_meta_entry)*nmetaentries);

	/* allocate memory for new meta chunk */
	gefetch_meta *newmeta;
	if ((newmeta = (gefetch_meta *)malloc(sizeof(gefetch_meta) + sizeof(gefetch_meta_entry)*nmetaentries)) == 0)
		return GEFETCH_NOMEM;

	/* fill meta info */
	newmeta->x = meta_x;
	newmeta->y = meta_y;
	newmeta->level = meta_level;

	newmeta->nentries = nmetaentries;
	memcpy(newmeta->entries, (unsigned char *)gefetch_get_data_ptr(handle) + sizeof(gefetch_meta_header), sizeof(gefetch_meta_entry)*nmetaentries);

	/* insert into list */
	newmeta->prev = 0;
	newmeta->next = handle->firstmeta;
	if (handle->firstmeta)
		handle->firstmeta->prev = newmeta;
	handle->firstmeta = newmeta;

	if (!handle->lastmeta)
		handle->lastmeta = newmeta;

	handle->metasize += sizeof(gefetch_meta) + sizeof(gefetch_meta_entry)*nmetaentries;

	*resmeta = newmeta;

	return GEFETCH_OK;
}

/**
 * Internal function to fetch single metadata chunk
 *
 * Metadata contain precious information needed to download
 * other data types, like whether the specific image exists
 * at all and from what node it should be downloaded.
 *
 * @param x x coordinate of metadata chunk [0..(2^level-1)]
 * @param y y coordinate of metadata chunk [0..(2^level-1)]
 * @param level level of metadata chunk [0..]
 */
gefetch_error gefetch_fetch_meta(gefetch *handle, int x, int y, int level) {
	char index[32];
	gefetch_coords_to_string(x, y, level, index, sizeof(index));

	/* form full url */
	char urlbuf[1024];
	if (snprintf(urlbuf, sizeof(urlbuf), "%s/flatfile?q2-%s", handle->url, index) >= sizeof(urlbuf))
		return GEFETCH_SMALL_BUFFER;

	/* fetch */
	return gefetch_fetch(handle, urlbuf);
}
