// 
//   Copyright (C) 2005, 2006 Free Software Foundation, Inc.
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

// 
//
//

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "PropertyList.h"
#include "Property.h" 


#include "log.h"

#include "as_function.h"
#include "as_environment.h" // for enumerateValues
#include "as_value.h" // for enumerateValues

namespace gnash {

PropertyList::PropertyList(as_object& owner)
	:
	_owner(&owner)
{
}

PropertyList::PropertyList(const PropertyList& pl, as_object& new_owner)
	:
	_props(pl._props),
	_owner(&new_owner)
{
}



bool
PropertyList::getValue(const std::string& key, as_value& val) const
{
	const_iterator found = _props.find( key );
	if ( found == _props.end() )
	{
		return false;
	}

	val=found->second->getValue(_owner);

	//log_msg("Property %s found, assigning to return (%s)", key.c_str(), val.to_string());

	return true;
}

bool
PropertyList::setValue(const std::string& key, const as_value& val) 
{
	iterator found = _props.find( key );
	if ( found == _props.end() )
	{
		// create a new member
		_props[key] = new SimpleProperty(val);
		return true;
	}

	Property* prop = found->second;

	if ( prop->isReadOnly() )
	{
		log_warning("Property %s is read-only, not setting it", key.c_str());
		return false;
	}

	//log_msg("Property %s set to value %s", key.c_str(), val.to_string());
	prop->setValue(_owner, val);
	return true;
}

bool
PropertyList::setFlags(const std::string& key,
		int setFlags, int clearFlags)
{
	iterator found = _props.find( key );
	if ( found == _props.end() ) return false;

	Property* prop = found->second;

	as_prop_flags& f = prop->getFlags();
	return f.set_flags(setFlags, clearFlags);
}

std::pair<size_t,size_t>
PropertyList::setFlagsAll(int setFlags, int clearFlags)
{
	size_t success=0;
	size_t failure=0;

	for ( iterator it=_props.begin(), far=_props.end(); it != far; ++it)
	{
		Property* prop = it->second;
		as_prop_flags& f = prop->getFlags();
		if ( f.set_flags(setFlags, clearFlags) ) ++success;
		else ++failure;
	}

	return std::make_pair(success,failure);
}

std::pair<size_t,size_t>
PropertyList::setFlagsAll(const PropertyList& props,
		int flagsSet, int flagsClear)
{
	size_t success=0;
	size_t failure=0;

	for (const_iterator it = props.begin(), itEnd = props.end(); it != itEnd; ++it )
	{
		const std::string& name = it->first;

		if ( setFlags(name, flagsSet, flagsClear) ) ++success;
		else ++failure;
	}

	return std::make_pair(success,failure);

}

void
PropertyList::enumerateValues(as_environment& env) const
{
	for ( const_iterator i=begin(), ie=end(); i != ie; ++i)
	{
		const Property* prop = i->second;

		if ( prop->getFlags().get_dont_enum() ) continue;

		env.push(as_value(i->first.c_str()));
	}
}

void
PropertyList::dump() const
{
	for ( const_iterator it=begin(), itEnd=end(); it != itEnd; ++it )
	{
		log_msg("  %s: %s", it->first.c_str(),
			it->second->getValue(_owner).to_string());
	}
}

void
PropertyList::import(const PropertyList& o) 
{
	for (const_iterator it = o.begin(), itEnd = o.end(); it != itEnd; ++it)
	{
		const std::string& name = it->first;
		const Property* prop = it->second;

		// TODO: don't call get_member_value, we
		//       must copy also 'getset' members ...
		_props[name] = prop->clone();
		//setValue(name, prop.getValue());
	}
}

bool
PropertyList::addGetterSetter(const std::string& key, as_function& getter,
	as_function& setter)
{
	iterator found = _props.find( key );
	if ( found != _props.end() ) return false; // already exists !!

	_props[key] = new GetterSetterProperty(GetterSetter(getter, setter));
	return true;
}

void
PropertyList::clear()
{
	for (iterator it = begin(), itEnd = end(); it != itEnd; ++it)
		delete it->second;
	_props.clear();
}

PropertyList::~PropertyList()
{
	for (iterator it = begin(), itEnd = end(); it != itEnd; ++it)
		delete it->second;
}

} // end of gnash namespace

