/* valacodevisitor.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valacodevisitor.h>
#include <vala/valasourcefile.h>
#include <vala/valatypeparameter.h>
#include <vala/valadatatype.h>
#include <vala/valamember.h>
#include <vala/valafield.h>
#include <vala/valastringliteral.h>
#include <vala/valaexpression.h>
#include <vala/valaconstant.h>
#include <vala/valacreationmethod.h>
#include <vala/valaformalparameter.h>
#include <vala/valabooleanliteral.h>
#include <vala/valarealliteral.h>
#include <vala/valaintegerliteral.h>
#include <vala/valaenumvalue.h>
#include <vala/valamemberaccess.h>
#include <vala/valalocalvariable.h>
#include <vala/valadelegate.h>
#include <vala/valanullliteral.h>
#include <vala/valasignal.h>
#include <vala/valainterface.h>
#include <vala/valaenum.h>
#include <vala/valamethod.h>
#include <vala/valaproperty.h>
#include <vala/valaclass.h>
#include <vala/valastruct.h>
#include <vala/valaerrorcode.h>
#include <vala/valaerrordomain.h>
#include <vala/valareturnstatement.h>
#include <vala/valaassignment.h>
#include <vala/valaelementaccess.h>
#include <vala/valamethodcall.h>
#include <vala/valaobjectcreationexpression.h>
#include <vala/valaconstructor.h>
#include <vala/valablock.h>
#include <vala/valadestructor.h>
#include <vala/valausingdirective.h>
#include <vala/valapointerindirection.h>
#include <vala/valabaseaccess.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valaunaryexpression.h>
#include <vala/valabinaryexpression.h>
#include <vala/valawhilestatement.h>
#include <vala/valabreakstatement.h>
#include <vala/valaifstatement.h>
#include <vala/valadostatement.h>
#include <vala/valadeclarationstatement.h>
#include <vala/valaforstatement.h>
#include <vala/valayieldstatement.h>
#include <vala/valalambdaexpression.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valanamespace.h>
#include <vala/valaemptystatement.h>
#include <vala/valaarraycreationexpression.h>
#include <vala/valainitializerlist.h>
#include <vala/valaswitchlabel.h>
#include <vala/valaswitchsection.h>
#include <vala/valaswitchstatement.h>
#include <vala/valaforeachstatement.h>
#include <vala/valacontinuestatement.h>
#include <vala/valathrowstatement.h>
#include <vala/valacatchclause.h>
#include <vala/valatrystatement.h>
#include <vala/valalockstatement.h>
#include <vala/valadeletestatement.h>
#include <vala/valacharacterliteral.h>
#include <vala/valapostfixexpression.h>
#include <vala/valasizeofexpression.h>
#include <vala/valatypeofexpression.h>
#include <vala/valacastexpression.h>
#include <vala/valaaddressofexpression.h>
#include <vala/valareferencetransferexpression.h>
#include <vala/valatypecheck.h>
#include <vala/valaconditionalexpression.h>
#include <gobject/gvaluecollector.h>




enum  {
	VALA_CODE_VISITOR_DUMMY_PROPERTY
};
static void vala_code_visitor_real_visit_source_file (ValaCodeVisitor* self, ValaSourceFile* source_file);
static void vala_code_visitor_real_visit_namespace (ValaCodeVisitor* self, ValaNamespace* ns);
static void vala_code_visitor_real_visit_class (ValaCodeVisitor* self, ValaClass* cl);
static void vala_code_visitor_real_visit_struct (ValaCodeVisitor* self, ValaStruct* st);
static void vala_code_visitor_real_visit_interface (ValaCodeVisitor* self, ValaInterface* iface);
static void vala_code_visitor_real_visit_enum (ValaCodeVisitor* self, ValaEnum* en);
static void vala_code_visitor_real_visit_enum_value (ValaCodeVisitor* self, ValaEnumValue* ev);
static void vala_code_visitor_real_visit_error_domain (ValaCodeVisitor* self, ValaErrorDomain* edomain);
static void vala_code_visitor_real_visit_error_code (ValaCodeVisitor* self, ValaErrorCode* ecode);
static void vala_code_visitor_real_visit_delegate (ValaCodeVisitor* self, ValaDelegate* d);
static void vala_code_visitor_real_visit_member (ValaCodeVisitor* self, ValaMember* m);
static void vala_code_visitor_real_visit_constant (ValaCodeVisitor* self, ValaConstant* c);
static void vala_code_visitor_real_visit_field (ValaCodeVisitor* self, ValaField* f);
static void vala_code_visitor_real_visit_method (ValaCodeVisitor* self, ValaMethod* m);
static void vala_code_visitor_real_visit_creation_method (ValaCodeVisitor* self, ValaCreationMethod* m);
static void vala_code_visitor_real_visit_formal_parameter (ValaCodeVisitor* self, ValaFormalParameter* p);
static void vala_code_visitor_real_visit_property (ValaCodeVisitor* self, ValaProperty* prop);
static void vala_code_visitor_real_visit_property_accessor (ValaCodeVisitor* self, ValaPropertyAccessor* acc);
static void vala_code_visitor_real_visit_signal (ValaCodeVisitor* self, ValaSignal* sig);
static void vala_code_visitor_real_visit_constructor (ValaCodeVisitor* self, ValaConstructor* c);
static void vala_code_visitor_real_visit_destructor (ValaCodeVisitor* self, ValaDestructor* d);
static void vala_code_visitor_real_visit_type_parameter (ValaCodeVisitor* self, ValaTypeParameter* p);
static void vala_code_visitor_real_visit_using_directive (ValaCodeVisitor* self, ValaUsingDirective* ns);
static void vala_code_visitor_real_visit_data_type (ValaCodeVisitor* self, ValaDataType* type);
static void vala_code_visitor_real_visit_block (ValaCodeVisitor* self, ValaBlock* b);
static void vala_code_visitor_real_visit_empty_statement (ValaCodeVisitor* self, ValaEmptyStatement* stmt);
static void vala_code_visitor_real_visit_declaration_statement (ValaCodeVisitor* self, ValaDeclarationStatement* stmt);
static void vala_code_visitor_real_visit_local_variable (ValaCodeVisitor* self, ValaLocalVariable* local);
static void vala_code_visitor_real_visit_initializer_list (ValaCodeVisitor* self, ValaInitializerList* list);
static void vala_code_visitor_real_visit_expression_statement (ValaCodeVisitor* self, ValaExpressionStatement* stmt);
static void vala_code_visitor_real_visit_if_statement (ValaCodeVisitor* self, ValaIfStatement* stmt);
static void vala_code_visitor_real_visit_switch_statement (ValaCodeVisitor* self, ValaSwitchStatement* stmt);
static void vala_code_visitor_real_visit_switch_section (ValaCodeVisitor* self, ValaSwitchSection* section);
static void vala_code_visitor_real_visit_switch_label (ValaCodeVisitor* self, ValaSwitchLabel* label);
static void vala_code_visitor_real_visit_while_statement (ValaCodeVisitor* self, ValaWhileStatement* stmt);
static void vala_code_visitor_real_visit_do_statement (ValaCodeVisitor* self, ValaDoStatement* stmt);
static void vala_code_visitor_real_visit_for_statement (ValaCodeVisitor* self, ValaForStatement* stmt);
static void vala_code_visitor_real_visit_foreach_statement (ValaCodeVisitor* self, ValaForeachStatement* stmt);
static void vala_code_visitor_real_visit_break_statement (ValaCodeVisitor* self, ValaBreakStatement* stmt);
static void vala_code_visitor_real_visit_continue_statement (ValaCodeVisitor* self, ValaContinueStatement* stmt);
static void vala_code_visitor_real_visit_return_statement (ValaCodeVisitor* self, ValaReturnStatement* stmt);
static void vala_code_visitor_real_visit_yield_statement (ValaCodeVisitor* self, ValaYieldStatement* y);
static void vala_code_visitor_real_visit_throw_statement (ValaCodeVisitor* self, ValaThrowStatement* stmt);
static void vala_code_visitor_real_visit_try_statement (ValaCodeVisitor* self, ValaTryStatement* stmt);
static void vala_code_visitor_real_visit_catch_clause (ValaCodeVisitor* self, ValaCatchClause* clause);
static void vala_code_visitor_real_visit_lock_statement (ValaCodeVisitor* self, ValaLockStatement* stmt);
static void vala_code_visitor_real_visit_delete_statement (ValaCodeVisitor* self, ValaDeleteStatement* stmt);
static void vala_code_visitor_real_visit_expression (ValaCodeVisitor* self, ValaExpression* expr);
static void vala_code_visitor_real_visit_array_creation_expression (ValaCodeVisitor* self, ValaArrayCreationExpression* expr);
static void vala_code_visitor_real_visit_boolean_literal (ValaCodeVisitor* self, ValaBooleanLiteral* lit);
static void vala_code_visitor_real_visit_character_literal (ValaCodeVisitor* self, ValaCharacterLiteral* lit);
static void vala_code_visitor_real_visit_integer_literal (ValaCodeVisitor* self, ValaIntegerLiteral* lit);
static void vala_code_visitor_real_visit_real_literal (ValaCodeVisitor* self, ValaRealLiteral* lit);
static void vala_code_visitor_real_visit_string_literal (ValaCodeVisitor* self, ValaStringLiteral* lit);
static void vala_code_visitor_real_visit_null_literal (ValaCodeVisitor* self, ValaNullLiteral* lit);
static void vala_code_visitor_real_visit_member_access (ValaCodeVisitor* self, ValaMemberAccess* expr);
static void vala_code_visitor_real_visit_method_call (ValaCodeVisitor* self, ValaMethodCall* expr);
static void vala_code_visitor_real_visit_element_access (ValaCodeVisitor* self, ValaElementAccess* expr);
static void vala_code_visitor_real_visit_base_access (ValaCodeVisitor* self, ValaBaseAccess* expr);
static void vala_code_visitor_real_visit_postfix_expression (ValaCodeVisitor* self, ValaPostfixExpression* expr);
static void vala_code_visitor_real_visit_object_creation_expression (ValaCodeVisitor* self, ValaObjectCreationExpression* expr);
static void vala_code_visitor_real_visit_sizeof_expression (ValaCodeVisitor* self, ValaSizeofExpression* expr);
static void vala_code_visitor_real_visit_typeof_expression (ValaCodeVisitor* self, ValaTypeofExpression* expr);
static void vala_code_visitor_real_visit_unary_expression (ValaCodeVisitor* self, ValaUnaryExpression* expr);
static void vala_code_visitor_real_visit_cast_expression (ValaCodeVisitor* self, ValaCastExpression* expr);
static void vala_code_visitor_real_visit_pointer_indirection (ValaCodeVisitor* self, ValaPointerIndirection* expr);
static void vala_code_visitor_real_visit_addressof_expression (ValaCodeVisitor* self, ValaAddressofExpression* expr);
static void vala_code_visitor_real_visit_reference_transfer_expression (ValaCodeVisitor* self, ValaReferenceTransferExpression* expr);
static void vala_code_visitor_real_visit_binary_expression (ValaCodeVisitor* self, ValaBinaryExpression* expr);
static void vala_code_visitor_real_visit_type_check (ValaCodeVisitor* self, ValaTypeCheck* expr);
static void vala_code_visitor_real_visit_conditional_expression (ValaCodeVisitor* self, ValaConditionalExpression* expr);
static void vala_code_visitor_real_visit_lambda_expression (ValaCodeVisitor* self, ValaLambdaExpression* expr);
static void vala_code_visitor_real_visit_assignment (ValaCodeVisitor* self, ValaAssignment* a);
static void vala_code_visitor_real_visit_end_full_expression (ValaCodeVisitor* self, ValaExpression* expr);
static gpointer vala_code_visitor_parent_class = NULL;
static void vala_code_visitor_finalize (ValaCodeVisitor* obj);



/**
 * Visit operation called for source files.
 *
 * @param source_file a source file
 */
static void vala_code_visitor_real_visit_source_file (ValaCodeVisitor* self, ValaSourceFile* source_file) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (source_file != NULL);
}


void vala_code_visitor_visit_source_file (ValaCodeVisitor* self, ValaSourceFile* source_file) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_source_file (self, source_file);
}


/**
 * Visit operation called for namespaces.
 *
 * @param ns a namespace
 */
static void vala_code_visitor_real_visit_namespace (ValaCodeVisitor* self, ValaNamespace* ns) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (ns != NULL);
}


void vala_code_visitor_visit_namespace (ValaCodeVisitor* self, ValaNamespace* ns) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_namespace (self, ns);
}


/**
 * Visit operation called for classes.
 *
 * @param cl a class
 */
static void vala_code_visitor_real_visit_class (ValaCodeVisitor* self, ValaClass* cl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
}


void vala_code_visitor_visit_class (ValaCodeVisitor* self, ValaClass* cl) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_class (self, cl);
}


/**
 * Visit operation called for structs.
 *
 * @param st a struct
 */
static void vala_code_visitor_real_visit_struct (ValaCodeVisitor* self, ValaStruct* st) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
}


void vala_code_visitor_visit_struct (ValaCodeVisitor* self, ValaStruct* st) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_struct (self, st);
}


/**
 * Visit operation called for interfaces.
 *
 * @param iface an interface
 */
static void vala_code_visitor_real_visit_interface (ValaCodeVisitor* self, ValaInterface* iface) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (iface != NULL);
}


void vala_code_visitor_visit_interface (ValaCodeVisitor* self, ValaInterface* iface) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_interface (self, iface);
}


/**
 * Visit operation called for enums.
 *
 * @param en an enum
 */
static void vala_code_visitor_real_visit_enum (ValaCodeVisitor* self, ValaEnum* en) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (en != NULL);
}


void vala_code_visitor_visit_enum (ValaCodeVisitor* self, ValaEnum* en) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_enum (self, en);
}


/**
 * Visit operation called for enum values.
 *
 * @param ev an enum value
 */
static void vala_code_visitor_real_visit_enum_value (ValaCodeVisitor* self, ValaEnumValue* ev) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (ev != NULL);
}


void vala_code_visitor_visit_enum_value (ValaCodeVisitor* self, ValaEnumValue* ev) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_enum_value (self, ev);
}


/**
 * Visit operation called for error domains.
 *
 * @param edomain an error domain
 */
static void vala_code_visitor_real_visit_error_domain (ValaCodeVisitor* self, ValaErrorDomain* edomain) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (edomain != NULL);
}


void vala_code_visitor_visit_error_domain (ValaCodeVisitor* self, ValaErrorDomain* edomain) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_error_domain (self, edomain);
}


/**
 * Visit operation called for error codes.
 *
 * @param ecode an error code
 */
static void vala_code_visitor_real_visit_error_code (ValaCodeVisitor* self, ValaErrorCode* ecode) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (ecode != NULL);
}


void vala_code_visitor_visit_error_code (ValaCodeVisitor* self, ValaErrorCode* ecode) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_error_code (self, ecode);
}


/**
 * Visit operation called for delegates.
 *
 * @param d a delegate
 */
static void vala_code_visitor_real_visit_delegate (ValaCodeVisitor* self, ValaDelegate* d) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (d != NULL);
}


void vala_code_visitor_visit_delegate (ValaCodeVisitor* self, ValaDelegate* d) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_delegate (self, d);
}


/**
 * Visit operation called for members.
 *
 * @param m a member
 */
static void vala_code_visitor_real_visit_member (ValaCodeVisitor* self, ValaMember* m) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
}


void vala_code_visitor_visit_member (ValaCodeVisitor* self, ValaMember* m) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_member (self, m);
}


/**
 * Visit operation called for constants.
 *
 * @param c a constant
 */
static void vala_code_visitor_real_visit_constant (ValaCodeVisitor* self, ValaConstant* c) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (c != NULL);
}


void vala_code_visitor_visit_constant (ValaCodeVisitor* self, ValaConstant* c) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_constant (self, c);
}


/**
 * Visit operation called for fields.
 *
 * @param f a field
 */
static void vala_code_visitor_real_visit_field (ValaCodeVisitor* self, ValaField* f) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (f != NULL);
}


void vala_code_visitor_visit_field (ValaCodeVisitor* self, ValaField* f) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_field (self, f);
}


/**
 * Visit operation called for methods.
 *
 * @param m a method
 */
static void vala_code_visitor_real_visit_method (ValaCodeVisitor* self, ValaMethod* m) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
}


void vala_code_visitor_visit_method (ValaCodeVisitor* self, ValaMethod* m) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_method (self, m);
}


/**
 * Visit operation called for creation methods.
 *
 * @param m a method
 */
static void vala_code_visitor_real_visit_creation_method (ValaCodeVisitor* self, ValaCreationMethod* m) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
}


void vala_code_visitor_visit_creation_method (ValaCodeVisitor* self, ValaCreationMethod* m) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_creation_method (self, m);
}


/**
 * Visit operation called for formal parameters.
 *
 * @param p a formal parameter
 */
static void vala_code_visitor_real_visit_formal_parameter (ValaCodeVisitor* self, ValaFormalParameter* p) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (p != NULL);
}


void vala_code_visitor_visit_formal_parameter (ValaCodeVisitor* self, ValaFormalParameter* p) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_formal_parameter (self, p);
}


/**
 * Visit operation called for properties.
 *
 * @param prop a property
 */
static void vala_code_visitor_real_visit_property (ValaCodeVisitor* self, ValaProperty* prop) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (prop != NULL);
}


void vala_code_visitor_visit_property (ValaCodeVisitor* self, ValaProperty* prop) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_property (self, prop);
}


/**
 * Visit operation called for property accessors.
 *
 * @param acc a property accessor
 */
static void vala_code_visitor_real_visit_property_accessor (ValaCodeVisitor* self, ValaPropertyAccessor* acc) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (acc != NULL);
}


void vala_code_visitor_visit_property_accessor (ValaCodeVisitor* self, ValaPropertyAccessor* acc) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_property_accessor (self, acc);
}


/**
 * Visit operation called for signals.
 *
 * @param sig a signal
 */
static void vala_code_visitor_real_visit_signal (ValaCodeVisitor* self, ValaSignal* sig) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (sig != NULL);
}


void vala_code_visitor_visit_signal (ValaCodeVisitor* self, ValaSignal* sig) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_signal (self, sig);
}


/**
 * Visit operation called for constructors.
 *
 * @param c a constructor
 */
static void vala_code_visitor_real_visit_constructor (ValaCodeVisitor* self, ValaConstructor* c) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (c != NULL);
}


void vala_code_visitor_visit_constructor (ValaCodeVisitor* self, ValaConstructor* c) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_constructor (self, c);
}


/**
 * Visit operation called for destructors.
 *
 * @param d a destructor
 */
static void vala_code_visitor_real_visit_destructor (ValaCodeVisitor* self, ValaDestructor* d) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (d != NULL);
}


void vala_code_visitor_visit_destructor (ValaCodeVisitor* self, ValaDestructor* d) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_destructor (self, d);
}


/**
 * Visit operation called for type parameters.
 *
 * @param p a type parameter
 */
static void vala_code_visitor_real_visit_type_parameter (ValaCodeVisitor* self, ValaTypeParameter* p) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (p != NULL);
}


void vala_code_visitor_visit_type_parameter (ValaCodeVisitor* self, ValaTypeParameter* p) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_type_parameter (self, p);
}


/**
 * Visit operation called for using directives.
 *
 * @param ns a using directive
 */
static void vala_code_visitor_real_visit_using_directive (ValaCodeVisitor* self, ValaUsingDirective* ns) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (ns != NULL);
}


void vala_code_visitor_visit_using_directive (ValaCodeVisitor* self, ValaUsingDirective* ns) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_using_directive (self, ns);
}


/**
 * Visit operation called for type references.
 *
 * @param type a type reference
 */
static void vala_code_visitor_real_visit_data_type (ValaCodeVisitor* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
}


void vala_code_visitor_visit_data_type (ValaCodeVisitor* self, ValaDataType* type) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_data_type (self, type);
}


/**
 * Visit operation called for blocks.
 *
 * @param b a block
 */
static void vala_code_visitor_real_visit_block (ValaCodeVisitor* self, ValaBlock* b) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (b != NULL);
}


void vala_code_visitor_visit_block (ValaCodeVisitor* self, ValaBlock* b) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_block (self, b);
}


/**
 * Visit operation called for empty statements.
 *
 * @param stmt an empty statement
 */
static void vala_code_visitor_real_visit_empty_statement (ValaCodeVisitor* self, ValaEmptyStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_empty_statement (ValaCodeVisitor* self, ValaEmptyStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_empty_statement (self, stmt);
}


/**
 * Visit operation called for declaration statements.
 *
 * @param stmt a declaration statement
 */
static void vala_code_visitor_real_visit_declaration_statement (ValaCodeVisitor* self, ValaDeclarationStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_declaration_statement (ValaCodeVisitor* self, ValaDeclarationStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_declaration_statement (self, stmt);
}


/**
 * Visit operation called for local variables.
 *
 * @param local a local variable
 */
static void vala_code_visitor_real_visit_local_variable (ValaCodeVisitor* self, ValaLocalVariable* local) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (local != NULL);
}


void vala_code_visitor_visit_local_variable (ValaCodeVisitor* self, ValaLocalVariable* local) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_local_variable (self, local);
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_code_visitor_real_visit_initializer_list (ValaCodeVisitor* self, ValaInitializerList* list) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (list != NULL);
}


void vala_code_visitor_visit_initializer_list (ValaCodeVisitor* self, ValaInitializerList* list) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_initializer_list (self, list);
}


/**
 * Visit operation called for expression statements.
 *
 * @param stmt an expression statement
 */
static void vala_code_visitor_real_visit_expression_statement (ValaCodeVisitor* self, ValaExpressionStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_expression_statement (ValaCodeVisitor* self, ValaExpressionStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_expression_statement (self, stmt);
}


/**
 * Visit operation called for if statements.
 *
 * @param stmt an if statement
 */
static void vala_code_visitor_real_visit_if_statement (ValaCodeVisitor* self, ValaIfStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_if_statement (ValaCodeVisitor* self, ValaIfStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_if_statement (self, stmt);
}


/**
 * Visit operation called for switch statements.
 *
 * @param stmt a switch statement
 */
static void vala_code_visitor_real_visit_switch_statement (ValaCodeVisitor* self, ValaSwitchStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_switch_statement (ValaCodeVisitor* self, ValaSwitchStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_switch_statement (self, stmt);
}


/**
 * Visit operation called for switch sections.
 *
 * @param section a switch section
 */
static void vala_code_visitor_real_visit_switch_section (ValaCodeVisitor* self, ValaSwitchSection* section) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (section != NULL);
}


void vala_code_visitor_visit_switch_section (ValaCodeVisitor* self, ValaSwitchSection* section) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_switch_section (self, section);
}


/**
 * Visit operation called for switch label.
 *
 * @param label a switch label
 */
static void vala_code_visitor_real_visit_switch_label (ValaCodeVisitor* self, ValaSwitchLabel* label) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (label != NULL);
}


void vala_code_visitor_visit_switch_label (ValaCodeVisitor* self, ValaSwitchLabel* label) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_switch_label (self, label);
}


/**
 * Visit operation called for while statements.
 *
 * @param stmt an while statement
 */
static void vala_code_visitor_real_visit_while_statement (ValaCodeVisitor* self, ValaWhileStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_while_statement (ValaCodeVisitor* self, ValaWhileStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_while_statement (self, stmt);
}


/**
 * Visit operation called for do statements.
 *
 * @param stmt a do statement
 */
static void vala_code_visitor_real_visit_do_statement (ValaCodeVisitor* self, ValaDoStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_do_statement (ValaCodeVisitor* self, ValaDoStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_do_statement (self, stmt);
}


/**
 * Visit operation called for for statements.
 *
 * @param stmt a for statement
 */
static void vala_code_visitor_real_visit_for_statement (ValaCodeVisitor* self, ValaForStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_for_statement (ValaCodeVisitor* self, ValaForStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_for_statement (self, stmt);
}


/**
 * Visit operation called for foreach statements.
 *
 * @param stmt a foreach statement
 */
static void vala_code_visitor_real_visit_foreach_statement (ValaCodeVisitor* self, ValaForeachStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_foreach_statement (ValaCodeVisitor* self, ValaForeachStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_foreach_statement (self, stmt);
}


/**
 * Visit operation called for break statements.
 *
 * @param stmt a break statement
 */
static void vala_code_visitor_real_visit_break_statement (ValaCodeVisitor* self, ValaBreakStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_break_statement (ValaCodeVisitor* self, ValaBreakStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_break_statement (self, stmt);
}


/**
 * Visit operation called for continue statements.
 *
 * @param stmt a continue statement
 */
static void vala_code_visitor_real_visit_continue_statement (ValaCodeVisitor* self, ValaContinueStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_continue_statement (ValaCodeVisitor* self, ValaContinueStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_continue_statement (self, stmt);
}


/**
 * Visit operation called for return statements.
 *
 * @param stmt a return statement
 */
static void vala_code_visitor_real_visit_return_statement (ValaCodeVisitor* self, ValaReturnStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_return_statement (ValaCodeVisitor* self, ValaReturnStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_return_statement (self, stmt);
}


/**
 * Visit operation called for yield statement.
 *
 * @param y a yield statement
 */
static void vala_code_visitor_real_visit_yield_statement (ValaCodeVisitor* self, ValaYieldStatement* y) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (y != NULL);
}


void vala_code_visitor_visit_yield_statement (ValaCodeVisitor* self, ValaYieldStatement* y) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_yield_statement (self, y);
}


/**
 * Visit operation called for throw statements.
 *
 * @param stmt a throw statement
 */
static void vala_code_visitor_real_visit_throw_statement (ValaCodeVisitor* self, ValaThrowStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_throw_statement (ValaCodeVisitor* self, ValaThrowStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_throw_statement (self, stmt);
}


/**
 * Visit operation called for try statements.
 *
 * @param stmt a try statement
 */
static void vala_code_visitor_real_visit_try_statement (ValaCodeVisitor* self, ValaTryStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_try_statement (ValaCodeVisitor* self, ValaTryStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_try_statement (self, stmt);
}


/**
 * Visit operation called for catch clauses.
 *
 * @param clause a catch cluase
 */
static void vala_code_visitor_real_visit_catch_clause (ValaCodeVisitor* self, ValaCatchClause* clause) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (clause != NULL);
}


void vala_code_visitor_visit_catch_clause (ValaCodeVisitor* self, ValaCatchClause* clause) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_catch_clause (self, clause);
}


/**
 * Visit operation called for lock statements before the body has been visited.
 *
 * @param stmt a lock statement
 */
static void vala_code_visitor_real_visit_lock_statement (ValaCodeVisitor* self, ValaLockStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_lock_statement (ValaCodeVisitor* self, ValaLockStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_lock_statement (self, stmt);
}


/**
 * Visit operation called for delete statements.
 *
 * @param stmt a delete statement
 */
static void vala_code_visitor_real_visit_delete_statement (ValaCodeVisitor* self, ValaDeleteStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
}


void vala_code_visitor_visit_delete_statement (ValaCodeVisitor* self, ValaDeleteStatement* stmt) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_delete_statement (self, stmt);
}


/**
 * Visit operations called for expresions.
 *
 * @param expr an expression
 */
static void vala_code_visitor_real_visit_expression (ValaCodeVisitor* self, ValaExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_expression (ValaCodeVisitor* self, ValaExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_expression (self, expr);
}


/**
 * Visit operations called for array creation expresions.
 *
 * @param expr an array creation expression
 */
static void vala_code_visitor_real_visit_array_creation_expression (ValaCodeVisitor* self, ValaArrayCreationExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_array_creation_expression (ValaCodeVisitor* self, ValaArrayCreationExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_array_creation_expression (self, expr);
}


/**
 * Visit operation called for boolean literals.
 *
 * @param lit a boolean literal
 */
static void vala_code_visitor_real_visit_boolean_literal (ValaCodeVisitor* self, ValaBooleanLiteral* lit) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (lit != NULL);
}


void vala_code_visitor_visit_boolean_literal (ValaCodeVisitor* self, ValaBooleanLiteral* lit) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_boolean_literal (self, lit);
}


/**
 * Visit operation called for character literals.
 *
 * @param lit a character literal
 */
static void vala_code_visitor_real_visit_character_literal (ValaCodeVisitor* self, ValaCharacterLiteral* lit) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (lit != NULL);
}


void vala_code_visitor_visit_character_literal (ValaCodeVisitor* self, ValaCharacterLiteral* lit) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_character_literal (self, lit);
}


/**
 * Visit operation called for integer literals.
 *
 * @param lit an integer literal
 */
static void vala_code_visitor_real_visit_integer_literal (ValaCodeVisitor* self, ValaIntegerLiteral* lit) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (lit != NULL);
}


void vala_code_visitor_visit_integer_literal (ValaCodeVisitor* self, ValaIntegerLiteral* lit) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_integer_literal (self, lit);
}


/**
 * Visit operation called for real literals.
 *
 * @param lit a real literal
 */
static void vala_code_visitor_real_visit_real_literal (ValaCodeVisitor* self, ValaRealLiteral* lit) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (lit != NULL);
}


void vala_code_visitor_visit_real_literal (ValaCodeVisitor* self, ValaRealLiteral* lit) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_real_literal (self, lit);
}


/**
 * Visit operation called for string literals.
 *
 * @param lit a string literal
 */
static void vala_code_visitor_real_visit_string_literal (ValaCodeVisitor* self, ValaStringLiteral* lit) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (lit != NULL);
}


void vala_code_visitor_visit_string_literal (ValaCodeVisitor* self, ValaStringLiteral* lit) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_string_literal (self, lit);
}


/**
 * Visit operation called for null literals.
 *
 * @param lit a null literal
 */
static void vala_code_visitor_real_visit_null_literal (ValaCodeVisitor* self, ValaNullLiteral* lit) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (lit != NULL);
}


void vala_code_visitor_visit_null_literal (ValaCodeVisitor* self, ValaNullLiteral* lit) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_null_literal (self, lit);
}


/**
 * Visit operation called for member access expressions.
 *
 * @param expr a member access expression
 */
static void vala_code_visitor_real_visit_member_access (ValaCodeVisitor* self, ValaMemberAccess* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_member_access (ValaCodeVisitor* self, ValaMemberAccess* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_member_access (self, expr);
}


/**
 * Visit operation called for invocation expressions.
 *
 * @param expr an invocation expression
 */
static void vala_code_visitor_real_visit_method_call (ValaCodeVisitor* self, ValaMethodCall* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_method_call (ValaCodeVisitor* self, ValaMethodCall* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_method_call (self, expr);
}


/**
 * Visit operation called for element access expressions.
 *
 * @param expr an element access expression
 */
static void vala_code_visitor_real_visit_element_access (ValaCodeVisitor* self, ValaElementAccess* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_element_access (ValaCodeVisitor* self, ValaElementAccess* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_element_access (self, expr);
}


/**
 * Visit operation called for base access expressions.
 *
 * @param expr a base access expression
 */
static void vala_code_visitor_real_visit_base_access (ValaCodeVisitor* self, ValaBaseAccess* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_base_access (ValaCodeVisitor* self, ValaBaseAccess* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_base_access (self, expr);
}


/**
 * Visit operation called for postfix expressions.
 *
 * @param expr a postfix expression
 */
static void vala_code_visitor_real_visit_postfix_expression (ValaCodeVisitor* self, ValaPostfixExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_postfix_expression (ValaCodeVisitor* self, ValaPostfixExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_postfix_expression (self, expr);
}


/**
 * Visit operation called for object creation expressions.
 *
 * @param expr an object creation expression
 */
static void vala_code_visitor_real_visit_object_creation_expression (ValaCodeVisitor* self, ValaObjectCreationExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_object_creation_expression (ValaCodeVisitor* self, ValaObjectCreationExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_object_creation_expression (self, expr);
}


/**
 * Visit operation called for sizeof expressions.
 *
 * @param expr a sizeof expression
 */
static void vala_code_visitor_real_visit_sizeof_expression (ValaCodeVisitor* self, ValaSizeofExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_sizeof_expression (ValaCodeVisitor* self, ValaSizeofExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_sizeof_expression (self, expr);
}


/**
 * Visit operation called for typeof expressions.
 *
 * @param expr a typeof expression
 */
static void vala_code_visitor_real_visit_typeof_expression (ValaCodeVisitor* self, ValaTypeofExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_typeof_expression (ValaCodeVisitor* self, ValaTypeofExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_typeof_expression (self, expr);
}


/**
 * Visit operation called for unary expressions.
 *
 * @param expr an unary expression
 */
static void vala_code_visitor_real_visit_unary_expression (ValaCodeVisitor* self, ValaUnaryExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_unary_expression (ValaCodeVisitor* self, ValaUnaryExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_unary_expression (self, expr);
}


/**
 * Visit operation called for call expressions.
 *
 * @param expr a call expression
 */
static void vala_code_visitor_real_visit_cast_expression (ValaCodeVisitor* self, ValaCastExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_cast_expression (ValaCodeVisitor* self, ValaCastExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_cast_expression (self, expr);
}


/**
 * Visit operation called for pointer indirections.
 *
 * @param expr a pointer indirection
 */
static void vala_code_visitor_real_visit_pointer_indirection (ValaCodeVisitor* self, ValaPointerIndirection* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_pointer_indirection (ValaCodeVisitor* self, ValaPointerIndirection* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_pointer_indirection (self, expr);
}


/**
 * Visit operation called for address-of expressions.
 *
 * @param expr an address-of expression
 */
static void vala_code_visitor_real_visit_addressof_expression (ValaCodeVisitor* self, ValaAddressofExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_addressof_expression (ValaCodeVisitor* self, ValaAddressofExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_addressof_expression (self, expr);
}


/**
 * Visit operation called for reference transfer expressions.
 *
 * @param expr a reference transfer expression
 */
static void vala_code_visitor_real_visit_reference_transfer_expression (ValaCodeVisitor* self, ValaReferenceTransferExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_reference_transfer_expression (ValaCodeVisitor* self, ValaReferenceTransferExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_reference_transfer_expression (self, expr);
}


/**
 * Visit operation called for binary expressions.
 *
 * @param expr a binary expression
 */
static void vala_code_visitor_real_visit_binary_expression (ValaCodeVisitor* self, ValaBinaryExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_binary_expression (ValaCodeVisitor* self, ValaBinaryExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_binary_expression (self, expr);
}


/**
 * Visit operation called for type checks.
 *
 * @param expr a type check expression
 */
static void vala_code_visitor_real_visit_type_check (ValaCodeVisitor* self, ValaTypeCheck* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_type_check (ValaCodeVisitor* self, ValaTypeCheck* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_type_check (self, expr);
}


/**
 * Visit operation called for conditional expressions.
 *
 * @param expr a conditional expression
 */
static void vala_code_visitor_real_visit_conditional_expression (ValaCodeVisitor* self, ValaConditionalExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_conditional_expression (ValaCodeVisitor* self, ValaConditionalExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_conditional_expression (self, expr);
}


/**
 * Visit operation called for lambda expressions.
 *
 * @param expr a lambda expression
 */
static void vala_code_visitor_real_visit_lambda_expression (ValaCodeVisitor* self, ValaLambdaExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_lambda_expression (ValaCodeVisitor* self, ValaLambdaExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_lambda_expression (self, expr);
}


/**
 * Visit operation called for assignments.
 *
 * @param a an assignment
 */
static void vala_code_visitor_real_visit_assignment (ValaCodeVisitor* self, ValaAssignment* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
}


void vala_code_visitor_visit_assignment (ValaCodeVisitor* self, ValaAssignment* a) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_assignment (self, a);
}


/**
 * Visit operation called at end of full expressions.
 *
 * @param expr a full expression
 */
static void vala_code_visitor_real_visit_end_full_expression (ValaCodeVisitor* self, ValaExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
}


void vala_code_visitor_visit_end_full_expression (ValaCodeVisitor* self, ValaExpression* expr) {
	VALA_CODE_VISITOR_GET_CLASS (self)->visit_end_full_expression (self, expr);
}


static void vala_value_code_visitor_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_code_visitor_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_code_visitor_unref (value->data[0].v_pointer);
	}
}


static void vala_value_code_visitor_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_code_visitor_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_code_visitor_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_code_visitor_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaCodeVisitor* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_code_visitor_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_code_visitor_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaCodeVisitor** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_code_visitor_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecCodeVisitor* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_CODE_VISITOR), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_code_visitor (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_CODE_VISITOR), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_code_visitor (GValue* value, gpointer v_object) {
	ValaCodeVisitor* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_CODE_VISITOR));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_CODE_VISITOR));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_code_visitor_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_code_visitor_unref (old);
	}
}


static void vala_code_visitor_class_init (ValaCodeVisitorClass * klass) {
	vala_code_visitor_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_code_visitor_finalize;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_code_visitor_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_code_visitor_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_code_visitor_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_code_visitor_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_code_visitor_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_code_visitor_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum_value = vala_code_visitor_real_visit_enum_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_code_visitor_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_code = vala_code_visitor_real_visit_error_code;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_code_visitor_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member = vala_code_visitor_real_visit_member;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_code_visitor_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_code_visitor_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_code_visitor_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_code_visitor_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_code_visitor_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_code_visitor_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_code_visitor_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_code_visitor_real_visit_signal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_code_visitor_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_code_visitor_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_parameter = vala_code_visitor_real_visit_type_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_using_directive = vala_code_visitor_real_visit_using_directive;
	VALA_CODE_VISITOR_CLASS (klass)->visit_data_type = vala_code_visitor_real_visit_data_type;
	VALA_CODE_VISITOR_CLASS (klass)->visit_block = vala_code_visitor_real_visit_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_empty_statement = vala_code_visitor_real_visit_empty_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_declaration_statement = vala_code_visitor_real_visit_declaration_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_local_variable = vala_code_visitor_real_visit_local_variable;
	VALA_CODE_VISITOR_CLASS (klass)->visit_initializer_list = vala_code_visitor_real_visit_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_code_visitor_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_code_visitor_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_statement = vala_code_visitor_real_visit_switch_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_section = vala_code_visitor_real_visit_switch_section;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_label = vala_code_visitor_real_visit_switch_label;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_code_visitor_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_do_statement = vala_code_visitor_real_visit_do_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_code_visitor_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_foreach_statement = vala_code_visitor_real_visit_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_break_statement = vala_code_visitor_real_visit_break_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_continue_statement = vala_code_visitor_real_visit_continue_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_return_statement = vala_code_visitor_real_visit_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_yield_statement = vala_code_visitor_real_visit_yield_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_code_visitor_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_code_visitor_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_code_visitor_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_code_visitor_real_visit_lock_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delete_statement = vala_code_visitor_real_visit_delete_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression = vala_code_visitor_real_visit_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_array_creation_expression = vala_code_visitor_real_visit_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_boolean_literal = vala_code_visitor_real_visit_boolean_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_character_literal = vala_code_visitor_real_visit_character_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_integer_literal = vala_code_visitor_real_visit_integer_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_real_literal = vala_code_visitor_real_visit_real_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_string_literal = vala_code_visitor_real_visit_string_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_null_literal = vala_code_visitor_real_visit_null_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_code_visitor_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method_call = vala_code_visitor_real_visit_method_call;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_code_visitor_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_base_access = vala_code_visitor_real_visit_base_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_postfix_expression = vala_code_visitor_real_visit_postfix_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_object_creation_expression = vala_code_visitor_real_visit_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_sizeof_expression = vala_code_visitor_real_visit_sizeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_typeof_expression = vala_code_visitor_real_visit_typeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_code_visitor_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_cast_expression = vala_code_visitor_real_visit_cast_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_pointer_indirection = vala_code_visitor_real_visit_pointer_indirection;
	VALA_CODE_VISITOR_CLASS (klass)->visit_addressof_expression = vala_code_visitor_real_visit_addressof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_code_visitor_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_code_visitor_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_check = vala_code_visitor_real_visit_type_check;
	VALA_CODE_VISITOR_CLASS (klass)->visit_conditional_expression = vala_code_visitor_real_visit_conditional_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lambda_expression = vala_code_visitor_real_visit_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_assignment = vala_code_visitor_real_visit_assignment;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_full_expression = vala_code_visitor_real_visit_end_full_expression;
}


static void vala_code_visitor_instance_init (ValaCodeVisitor * self) {
	self->ref_count = 1;
}


static void vala_code_visitor_finalize (ValaCodeVisitor* obj) {
	ValaCodeVisitor * self;
	self = VALA_CODE_VISITOR (obj);
}


GType vala_code_visitor_get_type (void) {
	static GType vala_code_visitor_type_id = 0;
	if (vala_code_visitor_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_code_visitor_init, vala_value_code_visitor_free_value, vala_value_code_visitor_copy_value, vala_value_code_visitor_peek_pointer, "p", vala_value_code_visitor_collect_value, "p", vala_value_code_visitor_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaCodeVisitorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_code_visitor_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCodeVisitor), 0, (GInstanceInitFunc) vala_code_visitor_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_code_visitor_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaCodeVisitor", &g_define_type_info, &g_define_type_fundamental_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_code_visitor_type_id;
}


gpointer vala_code_visitor_ref (gpointer instance) {
	ValaCodeVisitor* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_code_visitor_unref (gpointer instance) {
	ValaCodeVisitor* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_CODE_VISITOR_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}




