/*
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include "galeon-embed-shell.h"
#include "galeon-marshal.h"

#if !defined (ENABLE_MOZILLA_EMBED) && !defined (ENABLE_GTKHTML_EMBED)
#error "At least one embed needs to be enabled"
#endif

#ifdef ENABLE_MOZILLA_EMBED
#include "mozilla-embed-shell.h"
#endif
#ifdef ENABLE_GTKHTML_EMBED
#include "gtkhtml-embed-shell.h"
#endif

#include <string.h>

enum
{
	NEW_WINDOW,
	LAST_SIGNAL
};

struct GaleonEmbedShellPrivate
{
	GlobalHistory *global_history;
	DownloaderView *downloader_view;
	GList *embeds;
};

static void
galeon_embed_shell_class_init (GaleonEmbedShellClass *klass);
static void
galeon_embed_shell_init (GaleonEmbedShell *ges);
static void
galeon_embed_shell_finalize (GObject *object);
static void
galeon_embed_shell_finalize (GObject *object);

static GlobalHistory *
impl_get_global_history (GaleonEmbedShell *shell);
static DownloaderView *
impl_get_downloader_view (GaleonEmbedShell *shell);

static GObjectClass *parent_class = NULL;
static guint galeon_embed_shell_signals[LAST_SIGNAL] = { 0 };

GaleonEmbedShell *embed_shell;

GType
galeon_embed_shell_get_type (void)
{
       static GType galeon_embed_shell_type = 0;

        if (galeon_embed_shell_type == 0)
        {
                static const GTypeInfo our_info =
                {
                        sizeof (GaleonEmbedShellClass),
                        NULL, /* base_init */
                        NULL, /* base_finalize */
                        (GClassInitFunc) galeon_embed_shell_class_init,
                        NULL, /* class_finalize */
                        NULL, /* class_data */
                        sizeof (GaleonEmbedShell),
                        0,    /* n_preallocs */
                        (GInstanceInitFunc) galeon_embed_shell_init
                };

                galeon_embed_shell_type = g_type_register_static (G_TYPE_OBJECT,
								  "GaleonEmbedShell",
								  &our_info, 0);
        }

        return galeon_embed_shell_type;
}

static void
galeon_embed_shell_class_init (GaleonEmbedShellClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = (GObjectClass *) g_type_class_peek_parent (klass);

        object_class->finalize = galeon_embed_shell_finalize;
	klass->get_downloader_view = impl_get_downloader_view;
	klass->get_global_history = impl_get_global_history;

	galeon_embed_shell_signals[NEW_WINDOW] =
                g_signal_new ("new_window_orphan",
                              G_OBJECT_CLASS_TYPE (object_class),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GaleonEmbedShellClass, new_window),
                              NULL, NULL,
                              galeon_marshal_VOID__POINTER_INT,
                              G_TYPE_NONE,
                              2,
                              G_TYPE_POINTER,
			      G_TYPE_INT);
}

static void
galeon_embed_shell_init (GaleonEmbedShell *ges)
{

	/* Singleton, globally accessible */
	embed_shell = ges;
	
	ges->priv = g_new0 (GaleonEmbedShellPrivate, 1);

	ges->priv->global_history = NULL;
	ges->priv->downloader_view = NULL;
	ges->priv->embeds = NULL;
}

static void
galeon_embed_shell_finalize (GObject *object)
{
	GaleonEmbedShell *ges;

        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_GALEON_EMBED_SHELL (object));

        ges = GALEON_EMBED_SHELL (object);

        g_return_if_fail (ges->priv != NULL);

	if (ges->priv->global_history)
	{
		global_history_save_if_needed (ges->priv->global_history);
		g_object_unref (ges->priv->global_history);
	}

	if (ges->priv->downloader_view)
	{
		g_object_remove_weak_pointer 
			(G_OBJECT(ges->priv->downloader_view),
			 (gpointer *)&ges->priv->downloader_view);
		g_object_unref (ges->priv->downloader_view);
	}

        g_free (ges->priv);
	
        G_OBJECT_CLASS (parent_class)->finalize (object);
}

GaleonEmbedShell *
galeon_embed_shell_new (const char *type)
{
#ifdef ENABLE_MOZILLA_EMBED
	if (strcmp (type, "mozilla") == 0)
	{
		return GALEON_EMBED_SHELL (g_object_new 
			(MOZILLA_EMBED_SHELL_TYPE, NULL));
	}
#endif
#ifdef ENABLE_GTKHTML_EMBED	
	if (strcmp (type, "gtkhtml") == 0)
	{
		return GALEON_EMBED_SHELL (g_object_new
			(GTKHTML_EMBED_SHELL_TYPE, NULL));
	}
#endif	
	g_assert_not_reached ();
	return NULL;
}

const char *supported_embeds [] = {
#ifdef ENABLE_MOZILLA_EMBED
		"mozilla",
#endif
#ifdef ENABLE_GTKHTML_EMBED
		"gtkhtml",
#endif
		NULL };

const char **
galeon_embed_shell_get_supported (void)
{
        return supported_embeds;
}

void              
galeon_embed_shell_add_embed (GaleonEmbedShell *ges,
                              GaleonEmbed *embed)
{
	ges->priv->embeds = g_list_append (ges->priv->embeds, embed);
}

void 
galeon_embed_shell_remove_embed (GaleonEmbedShell *ges,
                                 GaleonEmbed *embed)
{
	ges->priv->embeds = g_list_remove (ges->priv->embeds, embed);
}

GaleonEmbed *
galeon_embed_shell_get_active_embed (GaleonEmbedShell *ges)
{
	GList *list = ges->priv->embeds;

        g_return_val_if_fail (ges->priv->embeds != NULL, NULL);
        
        return GALEON_EMBED (list->data);
}

GList *
galeon_embed_shell_get_embeds (GaleonEmbedShell *ges)
{
	return ges->priv->embeds;
}

void              
galeon_embed_shell_get_capabilities (GaleonEmbedShell *shell,
				     EmbedShellCapabilities *caps)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->get_capabilities (shell, caps);
}

GlobalHistory *
galeon_embed_shell_get_global_history (GaleonEmbedShell *shell)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->get_global_history (shell);
}

DownloaderView *
galeon_embed_shell_get_downloader_view (GaleonEmbedShell *shell)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->get_downloader_view (shell);
}

gresult      
galeon_embed_shell_clear_cache (GaleonEmbedShell *shell,
				CacheType type)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->clear_cache (shell, type);
}

gresult          
galeon_embed_shell_set_offline_mode (GaleonEmbedShell *shell,
				     gboolean offline)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->set_offline_mode (shell, offline);
}

gresult           
galeon_embed_shell_load_proxy_autoconf (GaleonEmbedShell *shell,
					const char* url)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->load_proxy_autoconf (shell, url);
}

gresult           
galeon_embed_shell_show_java_console (GaleonEmbedShell *shell)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->show_java_console (shell);
}

gresult           
galeon_embed_shell_show_js_console (GaleonEmbedShell *shell)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->show_js_console (shell);
}

gresult           
galeon_embed_shell_get_charset_titles (GaleonEmbedShell *shell,
				       const char *group,
				       GList **charsets)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->get_charset_titles (shell, group, charsets);
}

gresult           
galeon_embed_shell_get_charset_groups (GaleonEmbedShell *shell,
			               GList **groups)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->get_charset_groups (shell, groups);
}

gresult
galeon_embed_shell_get_font_list (GaleonEmbedShell *shell,
				  const char *langGroup,
				  const char *fontType,
				  GList **fontList,
				  char **default_font)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->get_font_list (shell, langGroup, fontType, fontList,
				     default_font);
}

gresult           
galeon_embed_shell_set_permission (GaleonEmbedShell *shell,
				   const char *url, 
				   PermissionType type,
				   gboolean allow)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->set_permission (shell, url, type, allow);
}

gresult           
galeon_embed_shell_list_permissions (GaleonEmbedShell *shell,
				     PermissionType type, 
				     GList **permissions)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->list_permissions (shell, type, permissions);
}

gresult           
galeon_embed_shell_remove_permissions (GaleonEmbedShell *shell,
				       PermissionType type,
				       GList *permissions)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->remove_permissions (shell, type, permissions);
}

gresult
galeon_embed_shell_test_permission(GaleonEmbedShell *shell,
				   const char *url,
				   PermissionType type,
				   PermissionActionType *ret_action)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS(shell);
	return klass->test_permission(shell, url, type, ret_action);
}

gresult           
galeon_embed_shell_list_cookies (GaleonEmbedShell *shell,
				 GList **cookies)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->list_cookies (shell, cookies);
}

gresult           
galeon_embed_shell_remove_cookies (GaleonEmbedShell *shell,
				   GList *cookies)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->remove_cookies (shell, cookies);
}

gresult           
galeon_embed_shell_list_passwords (GaleonEmbedShell *shell,
				   PasswordType type, 
				   GList **passwords)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->list_passwords (shell, type, passwords);
}

gresult           
galeon_embed_shell_remove_passwords (GaleonEmbedShell *shell,
				     GList *passwords, 
				     PasswordType type)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->remove_passwords (shell, passwords, type);
}

/**
 * show_file_picker: Shows a file picker. Can be configured to select a
 * file or a directory.
 * @parentWidget: Parent Widget for file picker.
 * @title: Title for file picker.
 * @directory: Initial directory to start in. 
 * @file: Initial filename to show in filepicker.
 * @mode: Mode to run filepicker in (modeOpen, modeSave, modeGetFolder)
 * @ret_fullpath: On a successful return, will hold the full path to selected
 *		file or directory.
 * @file_formats: an array of FileFormat structures to fill the format chooser
 *              optionmenu. NULL if not needed. The last item must have 
 *              description == NULL.
 * @ret_file_format: where to store the index of the format selected (can be
 *              NULL)
 * returns: TRUE for success, FALSE for failure.
 */

gresult 
galeon_embed_shell_show_file_picker (GaleonEmbedShell *shell,
				     GtkWidget *parentWidget, 
		       		     const char *title,
		       		     const char *directory,
		                     const char *file, 
		       		     FilePickerMode mode,
                       		     char **ret_fullpath, 
		       		     gboolean *ret_save_content, 
                       		     FileFormat *file_formats, 
		       		     int *ret_file_format)
{
	GaleonEmbedShellClass *klass = GALEON_EMBED_SHELL_GET_CLASS (shell);
        return klass->show_file_picker (shell, parentWidget, title, 
					directory, file, mode,
					ret_fullpath, ret_save_content,
					file_formats, ret_file_format);	
}

static GlobalHistory *
impl_get_global_history (GaleonEmbedShell *shell)
{
	if (!shell->priv->global_history)
	{
		shell->priv->global_history = global_history_new ();
	}
	
	return shell->priv->global_history;
}

static DownloaderView *
impl_get_downloader_view (GaleonEmbedShell *shell)
{
	if (!shell->priv->downloader_view)
	{
		shell->priv->downloader_view = downloader_view_new ();
		g_object_add_weak_pointer 
			(G_OBJECT(shell->priv->downloader_view),
			 (gpointer *)&shell->priv->downloader_view);
	}
	
	return shell->priv->downloader_view;
}

gresult 
galeon_embed_shell_free_permissions (GaleonEmbedShell *shell,
				     GList *permissions)
{
	GList *l;
	
	for (l = permissions; l != NULL; l = l->next)
	{
		PermissionInfo *info = (PermissionInfo *)l->data;
		
		g_free (info->domain);
		g_free (info);
	}

	g_list_free (permissions);

	return G_OK;
}

gresult 
galeon_embed_shell_free_cookies (GaleonEmbedShell *shell,
				 GList *cookies)
{
	g_list_foreach (cookies, (GFunc)cookie_info_free, NULL);
	g_list_free (cookies);
	return G_OK;
}

gresult
galeon_embed_shell_free_passwords (GaleonEmbedShell *shell,
				   GList *passwords)
{
	GList *l;
	
	for (l = passwords; l != NULL; l = l->next)
	{
		PasswordInfo *info = (PasswordInfo *)l->data;
		g_free (info->host);
		g_free (info->username);
		g_free (info);
	}

	g_list_free (passwords);

	return G_OK;
}

