/* GStreamer
 *
 * v4lradio_calls.c: generic V4L calls
 *
 * Copyright (C) 2001-2002 Ronald Bultje <rbultje@ronald.bitfreak.net>
 * Copyright (C) 2005 Nickolay V. Shmyrev <nshmyrev@yandex.ru>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <gst/gst.h>
#include <gst/tuner/tuner.h>

#include "v4lradio_calls.h"
#include "gstv4lradiotuner.h"


GST_DEBUG_CATEGORY_EXTERN (v4lradio_debug);
#define GST_CAT_DEFAULT v4lradio_debug

G_GNUC_UNUSED static const char *audio_name[] = {
  "Volume",
  "Mute",
  "Mode",
  NULL
};

static gboolean
gst_v4lradio_fill_params (GstV4lRadioBin * v4lradio_bin)
{
  struct video_tuner tuner;

  tuner.tuner = 0;
  if (ioctl (v4lradio_bin->radio_fd, VIDIOCGTUNER, &tuner) < 0) {
    GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, SETTINGS, (NULL),
        ("Error getting tuner structure: %s", g_strerror (errno)));
    return FALSE;
  }

  v4lradio_bin->device_name = g_strdup (tuner.name);
  v4lradio_bin->channel = g_object_new (GST_TYPE_TUNER_CHANNEL, NULL);
  v4lradio_bin->channel->flags |= GST_TUNER_CHANNEL_FREQUENCY;
  v4lradio_bin->channel->freq_multiplicator =
      62.5 * ((tuner.flags & VIDEO_TUNER_LOW) ? 1 : 1000);
  v4lradio_bin->channel->min_frequency = tuner.rangelow;
  v4lradio_bin->channel->max_frequency = tuner.rangehigh;
  v4lradio_bin->channel->min_signal = 0;
  v4lradio_bin->channel->max_signal = 0xffff;

  return TRUE;
}

/******************************************************
 * gst_v4lradio_open():
 *   open the video device (v4lradio_bin->radiodev)
 * return value: TRUE on success, FALSE on error
 ******************************************************/

gboolean
gst_v4lradio_open (GstV4lRadioBin * v4lradio_bin)
{
  GST_DEBUG_OBJECT (v4lradio_bin, "opening device %s", v4lradio_bin->radiodev);
  GST_V4LRADIO_CHECK_NOT_OPEN (v4lradio_bin);

  /* be sure we have a device */
  if (!v4lradio_bin->radiodev) {
    GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, NOT_FOUND,
        (_("No device specified.")), (NULL));
    return FALSE;
  }

  /* open the device */
  v4lradio_bin->radio_fd = open (v4lradio_bin->radiodev, O_RDWR);
  if (!GST_V4LRADIO_IS_OPEN (v4lradio_bin)) {
    if (errno == ENODEV || errno == ENOENT) {
      GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, NOT_FOUND,
          (_("Device \"%s\" does not exist."), v4lradio_bin->radiodev), (NULL));
      return FALSE;
    }
    GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, OPEN_READ_WRITE,
        (_("Could not open device \"%s\" for reading and writing."),
            v4lradio_bin->radiodev), GST_ERROR_SYSTEM);
    return FALSE;
  }

  if (!gst_v4lradio_fill_params (v4lradio_bin)) {
    return FALSE;
  }

  GST_INFO_OBJECT (v4lradio_bin, "Opened device \'%s\' (\'%s\') successfully",
      v4lradio_bin->device_name, v4lradio_bin->radiodev);

  return TRUE;
}


/******************************************************
 * gst_v4lradio_close():
 *   close the video device (v4lradio_bin->radio_fd)
 * return value: TRUE on success, FALSE on error
 ******************************************************/

gboolean
gst_v4lradio_close (GstV4lRadioBin * v4lradio_bin)
{
  GST_DEBUG_OBJECT (v4lradio_bin, "closing device");
  GST_V4LRADIO_CHECK_OPEN (v4lradio_bin);

  close (v4lradio_bin->radio_fd);
  v4lradio_bin->radio_fd = -1;

  if (v4lradio_bin->channel != NULL) {
    g_object_unref (v4lradio_bin->channel);
    v4lradio_bin->channel = NULL;
  }

  return TRUE;
}

/******************************************************
 * gst_v4lradio_get_signal():
 *   get the current signal
 * return value: TRUE on success, FALSE on error
 ******************************************************/

gboolean
gst_v4lradio_get_signal (GstV4lRadioBin * v4lradio_bin, guint * signal)
{
  struct video_tuner tuner;

  GST_DEBUG_OBJECT (v4lradio_bin, "getting tuner signal");
  GST_V4LRADIO_CHECK_OPEN (v4lradio_bin);

  tuner.tuner = 0;
  if (ioctl (v4lradio_bin->radio_fd, VIDIOCGTUNER, &tuner) < 0) {
    GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, SETTINGS, (NULL),
        ("Error getting tuner signal: %s", g_strerror (errno)));
    return FALSE;
  }

  *signal = tuner.signal;

  return TRUE;
}


/******************************************************
 * gst_v4lradio_get_frequency():
 *   get the current frequency
 * return value: TRUE on success, FALSE on error
 ******************************************************/

gboolean
gst_v4lradio_get_frequency (GstV4lRadioBin * v4lradio_bin, gulong * frequency)
{
  struct video_tuner vtun;
  GstTunerChannel *channel;

  GST_DEBUG_OBJECT (v4lradio_bin, "getting tuner frequency");
  GST_V4LRADIO_CHECK_OPEN (v4lradio_bin);

  channel = gst_tuner_get_channel (GST_TUNER (v4lradio_bin));

  /* check that this is the current input */
  vtun.tuner = 0;
  if (ioctl (v4lradio_bin->radio_fd, VIDIOCGTUNER, &vtun) < 0)
    return FALSE;

  if (ioctl (v4lradio_bin->radio_fd, VIDIOCGFREQ, frequency) < 0) {
    GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, SETTINGS, (NULL),
        ("Error getting tuner frequency: %s", g_strerror (errno)));
    return FALSE;
  }

  *frequency = *frequency * channel->freq_multiplicator;

  return TRUE;
}


/******************************************************
 * gst_v4lradio_set_frequency():
 *   set frequency
 * return value: TRUE on success, FALSE on error
 ******************************************************/

gboolean
gst_v4lradio_set_frequency (GstV4lRadioBin * v4lradio_bin, gulong frequency)
{
  struct video_tuner vtun;
  GstTunerChannel *channel;

  GST_DEBUG_OBJECT (v4lradio_bin, "setting tuner frequency to %lu", frequency);
  GST_V4LRADIO_CHECK_OPEN (v4lradio_bin);

  channel = gst_tuner_get_channel (GST_TUNER (v4lradio_bin));

  /* check that this is the current input */
  vtun.tuner = 0;
  if (ioctl (v4lradio_bin->radio_fd, VIDIOCGTUNER, &vtun) < 0)
    return FALSE;

  frequency = frequency / channel->freq_multiplicator;

  if (ioctl (v4lradio_bin->radio_fd, VIDIOCSFREQ, &frequency) < 0) {
    GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, SETTINGS, (NULL),
        ("Error setting tuner frequency: %s", g_strerror (errno)));
    return FALSE;
  }

  return TRUE;
}



/******************************************************
 * gst_v4lradio_get_audio():
 *   get some audio value
 * return value: TRUE on success, FALSE on error
 ******************************************************/

gboolean
gst_v4lradio_get_audio (GstV4lRadioBin * v4lradio_bin,
    GstV4lRadioAudioType type, gint * value)
{
  struct video_audio vau;

  GST_DEBUG_OBJECT (v4lradio_bin, "getting audio parameter type %d (%s)", type,
      audio_name[type]);
  GST_V4LRADIO_CHECK_OPEN (v4lradio_bin);

  vau.audio = 0;
  if (ioctl (v4lradio_bin->radio_fd, VIDIOCGAUDIO, &vau) < 0) {
    GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, SETTINGS, (NULL),
        ("Error getting audio parameters: %s", g_strerror (errno)));
    return FALSE;
  }

  switch (type) {
    case V4LRADIO_AUDIO_MUTE:
      *value = (vau.flags & VIDEO_AUDIO_MUTE);
      break;
    case V4LRADIO_AUDIO_VOLUME:
      *value = vau.volume;
      break;
    case V4LRADIO_AUDIO_MODE:
      *value = vau.mode;
      break;
    default:
      GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, SETTINGS, (NULL),
          ("Error getting audio parameters: unknown type %d", type));
      return FALSE;
  }

  return TRUE;
}


/******************************************************
 * gst_v4lradio_set_audio():
 *   set some audio value
 * return value: TRUE on success, FALSE on error
 ******************************************************/

gboolean
gst_v4lradio_set_audio (GstV4lRadioBin * v4lradio_bin,
    GstV4lRadioAudioType type, gint value)
{
  struct video_audio vau;

  GST_DEBUG_OBJECT (v4lradio_bin,
      "setting audio parameter type %d (%s) to value %d", type,
      audio_name[type], value);
  GST_V4LRADIO_CHECK_OPEN (v4lradio_bin);

  vau.audio = 0;
  if (ioctl (v4lradio_bin->radio_fd, VIDIOCGAUDIO, &vau) < 0) {
    GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, SETTINGS, (NULL),
        ("Error getting audio parameters: %s", g_strerror (errno)));
    return FALSE;
  }

  switch (type) {
    case V4LRADIO_AUDIO_MUTE:
      if (!(vau.flags & VIDEO_AUDIO_MUTABLE)) {
        GST_ELEMENT_ERROR (v4lradio_bin, CORE, NOT_IMPLEMENTED, (NULL),
            ("Error setting audio mute: (un)setting mute is not supported"));
        return FALSE;
      }
      if (value)
        vau.flags |= VIDEO_AUDIO_MUTE;
      else
        vau.flags &= ~VIDEO_AUDIO_MUTE;
      break;
    case V4LRADIO_AUDIO_VOLUME:
      if (!(vau.flags & VIDEO_AUDIO_VOLUME)) {
        GST_ELEMENT_ERROR (v4lradio_bin, CORE, NOT_IMPLEMENTED, (NULL),
            ("Error setting audio volume: setting volume is not supported"));
        return FALSE;
      }
      vau.volume = value;
      break;
    case V4LRADIO_AUDIO_MODE:
      vau.mode = value;
      break;
    default:
      GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, SETTINGS, (NULL),
          ("Error setting audio parameters: unknown type %d", type));
      return FALSE;
  }

  if (ioctl (v4lradio_bin->radio_fd, VIDIOCSAUDIO, &vau) < 0) {
    GST_ELEMENT_ERROR (v4lradio_bin, RESOURCE, SETTINGS, (NULL),
        ("Error setting audio parameters: %s", g_strerror (errno)));
    return FALSE;
  }

  return TRUE;
}
