/* GStreamer
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "ximagesrc.h"

#include <string.h>
#include <stdlib.h>

#include "ximageutil.h"

#ifdef HAVE_XFIXES
#include <X11/extensions/Xfixes.h>
#endif
#ifdef HAVE_XDAMAGE
#include <X11/extensions/Xdamage.h>
#endif

GST_DEBUG_CATEGORY (ximagesrc_debug);
#define GST_CAT_DEFAULT ximagesrc_debug

/* elementfactory information */
static GstElementDetails ximagesrc_details =
GST_ELEMENT_DETAILS ("Ximage video source",
    "Source/Video",
    "Creates a screenshot video stream",
    "Lutz Mueller <lutz@users.sourceforge.net>");

static GstStaticPadTemplate gst_ximagesrc_src_template_factory =
GST_STATIC_PAD_TEMPLATE ("src", GST_PAD_SRC, GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-raw-rgb, "
        "framerate = (double) [ 1.0, 100.0 ], "
        "width = (int) [ 1, MAX ], " "height = (int) [ 1, MAX ]"));

#define DEFAULT_PROP_SYNC TRUE

enum
{
  PROP_0,
  PROP_DISPLAY_NAME,
  PROP_SCREEN_NUM,
  PROP_SYNC,
  PROP_SHOW_POINTER
};

static void gst_ximagesrc_set_clock (GstElement * element, GstClock * clock);
static void ximagesrc_imagepool_free (GstXimagesrc * src);

static GstElementClass *parent_class = NULL;

struct _GstXimagesrc
{
  GstElement element;

  /* Information on display */
  GstXContext *xcontext;
  guint screen_num;

  /* Information on window */
  Window xwindow;
  gint depth;
  gint width, height;

  /* State */
  gulong frame;
  gdouble framerate;

  GstClock *clock;
  gboolean sync;

  gchar *display;

  GMutex *imagepool_lock;
  GSList *imagepool;

  gboolean have_xfixes;
  gboolean have_xdamage;
  gboolean show_pointer;
#ifdef HAVE_XDAMAGE
  Damage damage;
  int damage_event_base;
  XserverRegion damage_region;
  GC damage_copy_gc;
#endif
};

struct _GstXimagesrcClass
{
  GstElementClass parent_class;
};

/* This function */
static GstXImage *
gst_ximagesrc_ximage_get (GstXimagesrc * ximagesrc)
{
  GstXImage *ximage = NULL;

  g_mutex_lock (ximagesrc->imagepool_lock);
  if (ximagesrc->imagepool != NULL) {

    ximage = ximagesrc->imagepool->data;

    if ((ximage->width != ximagesrc->width) ||
        (ximage->height != ximagesrc->height)) {

      ximageutil_ximage_destroy (ximagesrc->xcontext, ximage);
      ximage = NULL;
    }

    ximagesrc->imagepool = g_slist_delete_link (ximagesrc->imagepool,
        ximagesrc->imagepool);
  }

  g_mutex_unlock (ximagesrc->imagepool_lock);

  if (ximage == NULL) {
#ifdef HAVE_XSHM
    /* if i have xshm than allocate image.. */
    ximage =
        ximageutil_ximage_new (GST_ELEMENT (ximagesrc), ximagesrc->xcontext,
        ximagesrc->width, ximagesrc->height);
#else
    /* if i don't have it not! */
    ximage = g_new0 (GstXImage, 1);
    ximage->width = ximagesrc->width;
    ximage->height = ximagesrc->height;
    ximage->parent = GST_ELEMENT (ximagesrc);
#endif
  }

  g_return_val_if_fail (GST_IS_XIMAGESRC (ximagesrc), NULL);

#ifdef HAVE_XDAMAGE
  if (ximagesrc->have_xdamage) {
    XEvent ev;

    do {
      XNextEvent (ximagesrc->xcontext->disp, &ev);
      if (ev.type == ximagesrc->damage_event_base + XDamageNotify) {
        XDamageNotifyEvent *dev = (XDamageNotifyEvent *) & ev;

#ifdef HAVE_XSHM
        if (ximagesrc->xcontext->use_xshm &&
            dev->area.width == ximagesrc->width &&
            dev->area.height == ximagesrc->height) {
          XShmGetImage (ximagesrc->xcontext->disp, ximagesrc->xwindow,
              ximage->ximage, 0, 0, AllPlanes);
        } else
#endif

        {
          XGetSubImage (ximagesrc->xcontext->disp, ximagesrc->xwindow,
              dev->area.x, dev->area.y,
              dev->area.width, dev->area.height,
              AllPlanes, ZPixmap, ximage->ximage, dev->area.x, dev->area.y);
        }


      }
    } while (XPending (ximagesrc->xcontext->disp));
    XDamageSubtract (ximagesrc->xcontext->disp, ximagesrc->damage, None, None);

  } else {
#endif

#ifdef HAVE_XSHM

    if (ximagesrc->xcontext->use_xshm) {
      XShmGetImage (ximagesrc->xcontext->disp, ximagesrc->xwindow,
          ximage->ximage, 0, 0, AllPlanes);

    } else
#endif /* HAVE_XSHM */
    {
      ximage->ximage = XGetImage (ximagesrc->xcontext->disp, ximagesrc->xwindow,
          0, 0, ximagesrc->width, ximagesrc->height, AllPlanes, ZPixmap);
    }
#ifdef HAVE_XDAMAGE
  }
#endif
  return ximage;
}

/*********************************
 * Everyting relating to the pad *
 *********************************/

static GstCaps *
gst_ximagesrc_getcaps (GstPad * pad)
{
  GstXimagesrc *s = GST_XIMAGESRC (gst_pad_get_parent (pad));

  if (s->xcontext) {

    s->width = s->xcontext->width;
    s->height = s->xcontext->height;

    return gst_caps_new_simple ("video/x-raw-rgb",
        "bpp", G_TYPE_INT, s->xcontext->bpp,
        "endianness", G_TYPE_INT, s->xcontext->endianness,
        "red_mask", G_TYPE_INT, s->xcontext->visual->red_mask,
        "green_mask", G_TYPE_INT, s->xcontext->visual->green_mask,
        "blue_mask", G_TYPE_INT, s->xcontext->visual->blue_mask,
        "width", G_TYPE_INT, s->width,
        "height", G_TYPE_INT, s->height,
        "framerate", GST_TYPE_DOUBLE_RANGE, 1., 100., NULL);
  }

  return gst_caps_copy (gst_pad_template_get_caps (gst_static_pad_template_get
          (&gst_ximagesrc_src_template_factory)));
}

static GstPadLinkReturn
gst_ximagesrc_src_link (GstPad * pad, const GstCaps * caps)
{
  GstXimagesrc *src;
  const GstStructure *structure;
  GstPadLinkReturn ret;

  src = GST_XIMAGESRC (gst_pad_get_parent (pad));

  structure = gst_caps_get_structure (caps, 0);

  ret = gst_structure_get_double (structure, "framerate", &src->framerate);

  if ((!ret) || (src->framerate < 1))
    return GST_PAD_LINK_REFUSED;

  return GST_PAD_LINK_OK;
}

static void
gst_ximagesrc_free_data_func (GstBuffer * buf)
{
  GstXImage *ximage;
  GstXimagesrc *src;

  ximage = GST_BUFFER_PRIVATE (buf);

  src = GST_XIMAGESRC (ximage->parent);

  g_mutex_lock (src->imagepool_lock);
  src->imagepool = g_slist_prepend (src->imagepool, ximage);
  g_mutex_unlock (src->imagepool_lock);

}

/* ifdeff'ed to prevent warnings of not being used when xfixes not there */
#ifdef HAVE_XFIXES
static void
composite_pixel (GstXContext * xcontext, guchar * dest, guchar * src)
{
  guint8 r = src[2];
  guint8 g = src[1];
  guint8 b = src[0];
  guint8 a = src[3];
  gint dr, dg, db;
  guint32 color;
  gint r_shift, r_max;
  gint g_shift, g_max;
  gint b_shift, b_max;

  switch (xcontext->bpp) {
    case 8:
      color = *dest;
      break;
    case 16:
      color = GUINT16_FROM_LE (*(guint32 *) (dest));
      break;
    case 32:
      color = GUINT32_FROM_LE (*(guint32 *) (dest));
      break;
    default:
      g_warning ("bpp %d not supported\n", xcontext->bpp);
      color = 0;
  }

  /* FIXME: move the code that finds shift and max in the _link function */
  for (r_shift = 0; !(xcontext->visual->red_mask & (1 << r_shift)); r_shift++);
  for (g_shift = 0; !(xcontext->visual->green_mask & (1 << g_shift));
      g_shift++);
  for (b_shift = 0; !(xcontext->visual->blue_mask & (1 << b_shift)); b_shift++);

  r_max = (xcontext->visual->red_mask >> r_shift);
  b_max = (xcontext->visual->blue_mask >> b_shift);
  g_max = (xcontext->visual->green_mask >> g_shift);

#define RGBXXX_R(x)  (((x)>>r_shift) & (r_max))
#define RGBXXX_G(x)  (((x)>>g_shift) & (g_max))
#define RGBXXX_B(x)  (((x)>>b_shift) & (b_max))

  dr = (RGBXXX_R (color) * 255) / r_max;
  dg = (RGBXXX_G (color) * 255) / g_max;
  db = (RGBXXX_B (color) * 255) / b_max;

  dr = (r * a + (0xff - a) * dr) / 0xff;
  dg = (g * a + (0xff - a) * dg) / 0xff;
  db = (b * a + (0xff - a) * db) / 0xff;

  color = (((dr * r_max) / 255) << r_shift) +
      (((dg * g_max) / 255) << g_shift) + (((db * b_max) / 255) << b_shift);

  switch (xcontext->bpp) {
    case 8:
      *dest = color;
      break;
    case 16:
      *(guint16 *) (dest) = color;
      break;
    case 32:
      *(guint32 *) (dest) = color;
      break;
    default:
      g_warning ("bpp %d not supported\n", xcontext->bpp);
  }
}
#endif

static GstData *
gst_ximagesrc_get (GstPad * pad)
{
  GstXimagesrc *s = GST_XIMAGESRC (gst_pad_get_parent (pad));
  GstBuffer *buf;
  GstXImage *image;

  image = gst_ximagesrc_ximage_get (s);

#ifdef HAVE_XFIXES
  if (s->show_pointer && s->have_xfixes) {
    XFixesCursorImage *cursor_image;

    /* get cursor */
    cursor_image = XFixesGetCursorImage (s->xcontext->disp);
    if (cursor_image != NULL) {
      int cx, cy, i, j, count;

      cx = cursor_image->x - cursor_image->xhot;
      cy = cursor_image->y - cursor_image->yhot;
      //count = image->width * image->height;
      count = cursor_image->width * cursor_image->height;
      for (i = 0; i < count; i++)
        cursor_image->pixels[i] = GUINT_TO_LE (cursor_image->pixels[i]);

      /* copy those pixels across */
      for (j = cy; j < cy + cursor_image->height && j < s->height; j++) {
        for (i = cx; i < cx + cursor_image->width && i < s->width; i++) {
          guint8 *src, *dest;

          src =
              (guint8 *) & (cursor_image->pixels[((j -
                          cy) * cursor_image->width + (i - cx))]);
          dest =
              (guint8 *) & (image->ximage->data[(j * s->width +
                      i) * (s->xcontext->bpp / 8)]);

          composite_pixel (s->xcontext, (guint8 *) dest, (guint8 *) src);

/*          composite_pixel((guchar*)&(((guint32*)image->ximage->data)[j*s->width+i]), 
              (guchar*)&(((guint32*)cursor_image->pixels)[(j-cy)*cursor_image->width+(i-cx)]));*/
        }
      }
    }
  }
#endif
  buf = gst_buffer_new ();
  GST_BUFFER_FLAG_SET (buf, GST_BUFFER_DONTFREE);
  GST_BUFFER_FREE_DATA_FUNC (buf) = gst_ximagesrc_free_data_func;
  GST_BUFFER_PRIVATE (buf) = image;
  GST_BUFFER_DURATION (buf) = GST_SECOND / s->framerate;
  GST_BUFFER_TIMESTAMP (buf) = (gdouble) s->frame * GST_BUFFER_DURATION (buf);  //(gdouble) GST_SECOND *s->frame / s->framerate;

  GST_BUFFER_DATA (buf) = (guint8 *) image->ximage->data;
  GST_BUFFER_SIZE (buf) = s->width * s->height * 3;

  if ((s->sync) && (s->clock)) {
    gst_element_wait (GST_ELEMENT (s), GST_BUFFER_TIMESTAMP (buf));
  }

  s->frame++;

  return GST_DATA (buf);
}

static void
ximagesrc_imagepool_free (GstXimagesrc * src)
{

  g_mutex_lock (src->imagepool_lock);

  while (src->imagepool) {
    GstXImage *ximage = src->imagepool->data;

    src->imagepool = g_slist_delete_link (src->imagepool, src->imagepool);
    ximageutil_ximage_destroy (src->xcontext, ximage);
  }

  g_mutex_unlock (src->imagepool_lock);
}

/**************************************
 * Everything relating to the element *
 **************************************/

static GstElementStateReturn
gst_ximagesrc_change_state (GstElement * element)
{
  GstXimagesrc *s = GST_XIMAGESRC (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_PAUSED_TO_READY:
      s->frame = 0;
      break;
    case GST_STATE_NULL_TO_READY:
      if (!(s->xcontext =
              ximageutil_xcontext_get (GST_ELEMENT (s), s->display))) {
        GST_ELEMENT_ERROR (element, RESOURCE, OPEN_READ, (NULL), (NULL));
        return GST_STATE_FAILURE;
      }
      s->xwindow = RootWindow (s->xcontext->disp,
          MIN (s->screen_num, ScreenCount (s->xcontext->disp) - 1));
#ifdef HAVE_XFIXES
      /* check if xfixes supported */
      {
        int error_base;

        if (XFixesQueryExtension (s->xcontext->disp, &s->damage_event_base,
                &error_base))
          s->have_xfixes = TRUE;
      }
#endif
#ifdef HAVE_XDAMAGE
      /* check if xdamage is supported */
      {
        int error_base;
        long evmask = NoEventMask;

        if (XDamageQueryExtension (s->xcontext->disp, &s->damage_event_base,
                &error_base)) {
          s->damage =
              XDamageCreate (s->xcontext->disp, s->xwindow,
              XDamageReportRawRectangles);
          if (s->damage != None) {
            s->damage_region = XFixesCreateRegion (s->xcontext->disp, NULL, 0);
            if (s->damage_region != None) {
              XGCValues values;

              values.subwindow_mode = IncludeInferiors;
              s->damage_copy_gc = XCreateGC (s->xcontext->disp,
                  s->xwindow, GCSubwindowMode, &values);
              XSelectInput (s->xcontext->disp, s->xwindow, evmask);

              s->have_xdamage = TRUE;
            } else {
              XDamageDestroy (s->xcontext->disp, s->damage);
              s->damage = None;
            }
          }
        }
      }
#endif

      break;
    case GST_STATE_READY_TO_NULL:
      /* free all the imagepool */
      ximagesrc_imagepool_free (s);
      ximageutil_xcontext_clear (s->xcontext);
      s->xcontext = NULL;

      break;
    default:
      break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}

static void
gst_ximagesrc_init (GstXimagesrc * s)
{
  GstPad *pad;

  pad =
      gst_pad_new_from_template (gst_static_pad_template_get
      (&gst_ximagesrc_src_template_factory), "src");
  gst_pad_set_getcaps_function (pad, gst_ximagesrc_getcaps);
  gst_pad_set_link_function (pad, gst_ximagesrc_src_link);
  gst_pad_set_get_function (pad, gst_ximagesrc_get);
  gst_element_add_pad (GST_ELEMENT (s), pad);

  s->display = g_strdup (g_getenv ("DISPLAY"));
  s->xcontext = NULL;
  s->framerate = 10.;
  s->clock = NULL;
  s->sync = DEFAULT_PROP_SYNC;
  s->imagepool = NULL;
  s->imagepool_lock = g_mutex_new ();

  s->show_pointer = TRUE;
  s->have_xfixes = FALSE;
  s->have_xdamage = FALSE;
}

static void
gst_ximagesrc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstXimagesrc *src = GST_XIMAGESRC (object);

  switch (prop_id) {
    case PROP_DISPLAY_NAME:
      /* lock this property with chain!! */
      g_free (src->display);
      src->display = g_strdup (g_value_get_string (value));

      if (src->xcontext) {
        ximageutil_xcontext_clear (src->xcontext);
        src->xcontext = ximageutil_xcontext_get (GST_ELEMENT (src),
            (gchar *) g_value_get_string (value));
      }
      break;
    case PROP_SCREEN_NUM:
      src->screen_num = g_value_get_uint (value);
      break;
    case PROP_SYNC:
      src->sync = g_value_get_boolean (value);
      break;
    case PROP_SHOW_POINTER:
      src->show_pointer = g_value_get_boolean (value);
      break;
    default:
      break;
  }
}

static void
gst_ximagesrc_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstXimagesrc *src = GST_XIMAGESRC (object);

  switch (prop_id) {
    case PROP_DISPLAY_NAME:
      if (src->display) {
        g_value_set_string (value, src->display);
      } else {
        g_value_set_string (value, NULL);
      }
      break;
    case PROP_SCREEN_NUM:
      g_value_set_uint (value, src->screen_num);
      break;
    case PROP_SYNC:
      g_value_set_boolean (value, src->sync);
      break;
    case PROP_SHOW_POINTER:
      g_value_set_boolean (value, src->show_pointer);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_ximagesrc_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details (element_class, &ximagesrc_details);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_ximagesrc_src_template_factory));
}

static void
gst_ximagesrc_finalize (GObject * object)
{
  GstXimagesrc *src = GST_XIMAGESRC (object);

  g_free (src->display);
  g_mutex_free (src->imagepool_lock);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gst_ximagesrc_class_init (GstXimagesrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_DISPLAY_NAME,
      g_param_spec_string ("display_name", "Display", "X Display name", NULL,
          G_PARAM_READWRITE));
  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_SCREEN_NUM,
      g_param_spec_uint ("screen_num", "Screen number", "X Screen number",
          0, G_MAXINT, 0, G_PARAM_READWRITE));
  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_SHOW_POINTER,
      g_param_spec_boolean ("show_pointer", "Show Pointer",
          "Show Mouse Pointer (if available)", TRUE, G_PARAM_READWRITE));

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_SYNC,
      g_param_spec_boolean ("sync", "Sync", "Synchronize to clock",
          DEFAULT_PROP_SYNC, G_PARAM_READWRITE));


  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  gobject_class->set_property = gst_ximagesrc_set_property;
  gobject_class->get_property = gst_ximagesrc_get_property;
  gobject_class->finalize = gst_ximagesrc_finalize;

  gstelement_class->change_state = gst_ximagesrc_change_state;
  gstelement_class->set_clock = gst_ximagesrc_set_clock;

  GST_DEBUG_CATEGORY_INIT (ximagesrc_debug, "ximagesrc", 0,
      "X's framebuffer source");
}

static void
gst_ximagesrc_set_clock (GstElement * element, GstClock * clock)
{
  GstXimagesrc *x;

  x = GST_XIMAGESRC (element);

  gst_object_replace ((GstObject **) & x->clock, (GstObject *) clock);
}


GType
gst_ximagesrc_get_type (void)
{
  static GType ximagesrc_type = 0;

  if (!ximagesrc_type) {
    static const GTypeInfo ximagesrc_info = {
      sizeof (GstXimagesrcClass),
      gst_ximagesrc_base_init,
      NULL,
      (GClassInitFunc) gst_ximagesrc_class_init,
      NULL,
      NULL,
      sizeof (GstXimagesrc),
      0,
      (GInstanceInitFunc) gst_ximagesrc_init,
    };

    ximagesrc_type =
        g_type_register_static (GST_TYPE_ELEMENT, "GstXimagesrc",
        &ximagesrc_info, 0);
  }
  return ximagesrc_type;
}
