/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gnome-print-cups.c: A cups backend thingy
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useoful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:
 *    Dave Camp <dave@ximian.com>
 *    Chema Celorio <chema@celorio.com>
 *
 *  Copyright 2002  Ximian, Inc. and authors
 *
 */

#include <config.h>
#include <glib.h>
#include <gmodule.h>
#include <libgnomeprint/gnome-print-module.h>
#include <libgnomeprint/gpa/gpa-list.h>
#include <libgnomeprint/gpa/gpa-printer.h>
#include <libgnomeprint/gpa/gpa-value.h>
#include <libgnomeprint/gpa/gpa-settings.h>
#include <libgnomeprint/gpa/gpa-option.h>
#include <libgnomeprint/gpa/gpa-reference.h>
#include <libgnomeprint/gpa/gpa-model.h>

#include <cups/cups.h>
#include <cups/ppd.h>

/* Argument order: id, name, vendor */

xmlChar *model_xml_template = 
"<?xml version=\"1.0\"?>"
"<Model Id=\"%s\" Version=\"1.0\">"
"  <Name>%s</Name>"
"  <Vendor>%s</Vendor>"
"  <ModelVersion>0.0.1</ModelVersion>"
"  <Options>"
"    <Option Id=\"Transport\">"
"      <Option Id=\"Backend\" Type=\"List\" Default=\"CUPS\">"
"        <Item Id=\"CUPS\">"
"          <Name>CUPS</Name>"
"          <Key Id=\"Module\" Value=\"libgnomeprintcups.so\"/>"
"        </Item>"
"      </Option>"
"    </Option>"
"    <Option Id=\"Output\">"
"      <Option Id=\"Media\">"
"        <Option Id=\"PhysicalOrientation\" Type=\"List\" Default=\"R0\">"
"          <Fill Ref=\"Globals.Media.PhysicalOrientations\"/>"
"        </Option>"
"        <Key Id=\"Margins\">"
"          <Key Id=\"Left\" Value=\"0\"/>"
"          <Key Id=\"Right\" Value=\"0\"/>"
"          <Key Id=\"Top\" Value=\"0\"/>"
"          <Key Id=\"Bottom\" Value=\"0\"/>"
"        </Key>"
"      </Option>"
"    </Option>"
"    <Option Id=\"Resolution\" Type=\"List\" Default=\"360x360\">"
"       <Item Id=\"360x360\">"
"         <Name>360 x 360 DPI</Name>"
"         <Key Id=\"DPI\" Value=\"360\">"
"           <Key Id=\"X\" Value=\"360\"/>"
"           <Key Id=\"Y\" Value=\"360\"/>"
"         </Key>"
"       </Item>"
"    </Option>"
"    <Option Id=\"Document\">"
"      <Option Id=\"Page\">"
"        <Option Id=\"Layout\" Type=\"List\" Default=\"Plain\">"
"          <Fill Ref=\"Globals.Media.Layouts\"/>"
"        </Option>"
"        <Option Id=\"LogicalOrientation\" Type=\"List\" Default=\"R0\">"
"          <Fill Ref=\"Globals.Media.LogicalOrientations\"/>"
"        </Option>"
"        <Option Id=\"Margins\">"
"          <Option Id=\"Left\" Type=\"String\" Default=\"2 cm\"/>"
"          <Option Id=\"Right\" Type=\"String\" Default=\"2 cm\"/>"
"          <Option Id=\"Top\" Type=\"String\" Default=\"3 cm\"/>"
"          <Option Id=\"Bottom\" Type=\"String\" Default=\"3 cm\"/>"
"        </Option>"
"      </Option>"
"      <Option Id=\"PreferedUnit\" Type=\"String\" Default=\"cm\"/>"
"      <Option Id=\"Name\" Type=\"String\" Default=\"\"/>"
"    </Option>"
#if 0
"    <Option Id=\"Icon\">"
"      <Option Id=\"Filename\" Type=\"String\" Default=\"" DATADIR "/pixmaps/nautilus/default/i-printer.png\"/>"
"    </Option>"
#endif
"  </Options>"
"</Model>";

static GPANode *
gpa_model_new_from_xml (char *string)
{
	GPANode *model;
	xmlNodePtr root;
	xmlDocPtr model_xml_doc;
	
	model_xml_doc = xmlParseDoc (string);
	if (!model_xml_doc) {
		g_warning ("Could not parse model xml");
		return NULL;
	}
	
	root = model_xml_doc->xmlRootNode;
	model = gpa_model_new_from_tree (root);
	
	xmlFreeDoc (model_xml_doc);
	
	return model;
}

static char *
get_paper_text (ppd_file_t *ppd, ppd_size_t *size)
{
	/* This is dumb and slow and ugly and crappy and I hate myself. */
	int i;
	
	for (i = 0; i < ppd->num_groups; i++) {
		ppd_group_t *group = &ppd->groups[i];
		int j;
		for (j = 0; j < group->num_options; j++) {
			ppd_option_t *option = &group->options[j];
			if (!strcmp (option->keyword, "PageSize")) {
				int k;
				for (k = 0; k < option->num_choices; k++) {
					if (!strcmp (option->choices[k].choice,
						     size->name)) {
						return g_strdup (option->choices[k].text);
					}
				}
			}
		}
	}
	return g_strdup (size->name);
}

static GPANode *
get_physical_sizes (ppd_file_t *ppd)
{
	ppd_option_t *option;
	GPANode *node;
	int i;

	node = gpa_option_list_new ("PhysicalSize");
	
	option = ppdFindOption (ppd, "PageSize");
	if (option && option->defchoice)
		GPA_OPTION (node)->value = g_strdup (option->defchoice);
			
	for (i = 0; i < ppd->num_sizes; i++) {
		char *text;
		char *str;
		GPANode *size;
		GPANode *width;
		GPANode *height;

		text = get_paper_text (ppd, &ppd->sizes[i]);

		size = gpa_option_item_new (ppd->sizes[i].name, text);

		g_free (text);

		str = g_strdup_printf ("%d", (int)ppd->sizes[i].width);
		width = gpa_option_string_new ("Width", str);
		gpa_option_item_append_child (GPA_OPTION_ITEM (size), 
					      GPA_OPTION (width));
		gpa_node_unref (width);
		g_free (str);

		str = g_strdup_printf ("%d", (int)ppd->sizes[i].length);
		height = gpa_option_string_new ("Height", str);
		gpa_option_item_append_child (GPA_OPTION_ITEM (size), 
					      GPA_OPTION (height));
		gpa_node_unref (height);
		g_free (str);

		gpa_option_list_append_child (GPA_OPTION_LIST (node), 
					      GPA_OPTION (size));
		gpa_node_unref (size);
	}
	
	return node;
}

static GPANode *
get_sources (ppd_file_t *ppd)
{
	ppd_option_t *option;
	GPANode *node;
	int i;

	node = gpa_option_list_new ("PaperSource");
	
	option = ppdFindOption (ppd, "InputSlot");
	if (option && option->defchoice)
		GPA_OPTION (node)->value = g_strdup (option->defchoice);
			
	for (i = 0; i < option->num_choices; i++) {
		GPANode *source;

		source = gpa_option_item_new (option->choices[i].choice, option->choices[i].choice);
		gpa_option_list_append_child (GPA_OPTION_LIST (node), 
					      GPA_OPTION (source));
		gpa_node_unref (source);
	}
	
	return node;
}

static GPANode *
get_model (const gchar *printer, ppd_file_t *ppd)
{
	GPANode *media;
	GPANode *sizes;
	GPANode *model;
	GPANode *sources;
	GPANode *output;
	char *xml;
	char *id;

	id = g_strdup_printf ("Cups-%s-%s", ppd->manufacturer, ppd->nickname);

	model = gpa_model_get_by_id (id, TRUE);
	if (model)
		return model;

	xml = g_strdup_printf (model_xml_template, 
			       id, ppd->nickname, "GNOME"); /*FIXME: this should use a real vendor */
	model = gpa_model_new_from_xml (xml);

	media = gpa_node_lookup (model, "Options.Output.Media");
	sizes = get_physical_sizes (ppd);
	if (sizes) {
		gpa_option_node_append_child (GPA_OPTION_NODE (media),
					      GPA_OPTION (sizes));
		gpa_node_unref (sizes);
	}
	gpa_node_unref (media);

	output = gpa_node_lookup (model, "Options.Output");
	sources = get_sources (ppd);
	if (sources) {
		gpa_option_node_append_child (GPA_OPTION_NODE (output),
					      GPA_OPTION (sources));
		gpa_node_unref (sources);
	}
	gpa_node_unref (output);

	
	g_free (id);
	g_free (xml);
	
	return model;
}

static gboolean
append_printer (GPAList *printers_list, 
		const char *printer_name)
{
	GPAPrinter *gpa_printer;
	GPAList *settings_list;
	GPANode *settings;
	GPANode *model;
	const char *filename;
	ppd_file_t *ppd;

	filename = cupsGetPPD (printer_name);
	ppd = ppdOpenFile (filename);
	if (!ppd) {
		g_warning ("couldn't get ppd for %s\n", printer_name);
		return FALSE;
	}
	
	/* Printer */
	gpa_printer = gpa_printer_new_full (printer_name, printer_name);
	
	/* Model */
	model = get_model (printer_name, ppd);
	if (!model) {
		gpa_node_unref (GPA_NODE (gpa_printer));
		ppdClose (ppd);
		unlink (filename);
		return FALSE;
	}

	gpa_printer->model = gpa_node_attach (GPA_NODE (gpa_printer), gpa_reference_new (model));
	gpa_node_unref (GPA_NODE (model));
	
	/* Attach a Settings List */
	settings_list = gpa_list_new (GPA_TYPE_SETTINGS, "Settings", TRUE);
	gpa_printer->settings = gpa_node_attach (GPA_NODE (gpa_printer), 
						 GPA_NODE (settings_list));

	/* Attach a Settings and make them the default */
	settings = gpa_settings_new_from_model_full (GPA_NODE (model), "Default", "SettIdFromCups");
	gpa_list_prepend     (settings_list, settings);
	gpa_list_set_default (settings_list, settings);

	/* Verify the printer that we've just created */
	if (!gpa_node_verify (GPA_NODE (gpa_printer))) {
		g_warning ("Can't load printer from cups pluging, _verify failed\n");
		gpa_node_unref (GPA_NODE (gpa_printer));
		ppdClose (ppd);
		unlink (filename);		
		return FALSE;
	}

	gpa_list_prepend (printers_list, GPA_NODE (gpa_printer));

	ppdClose (ppd);
	unlink (filename);
	
	return TRUE;
}

static void
gnome_print_cups_printer_list_append (gpointer printers_list)
{
	gchar **printers;
	gint num_printers;
	gint i;
	
	g_return_if_fail (printers_list != NULL);
	g_return_if_fail (GPA_IS_LIST (printers_list));
	
	num_printers = cupsGetPrinters (&printers);
	if (num_printers < 1)
		return;
	
	for (i = 0; i < num_printers; i ++) {
		append_printer (GPA_LIST (printers_list), printers [i]);
	}
	
	for (i = 0; i < num_printers; i ++) {
		free (printers[i]);
	}

	free (printers);
	
	return;
}


/*  ------------- GPA init ------------- */
G_MODULE_EXPORT gboolean gpa_module_init (GpaModuleInfo *info);

G_MODULE_EXPORT gboolean
gpa_module_init (GpaModuleInfo *info)
{
	info->printer_list_append = gnome_print_cups_printer_list_append;
	return TRUE;
}
