/*******************************************************************************
 * Copyright (c) Contributors to the Eclipse Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0 
 *******************************************************************************/

package org.osgi.service.cdi.annotations;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.PARAMETER;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Documented;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import javax.enterprise.util.AnnotationLiteral;
import javax.inject.Qualifier;

/**
 * Annotation used in conjunction with {@link Reference} to specify the minimum
 * cardinality reference property.
 * <p>
 * Specifying the {@link MinimumCardinality} annotation with the value of
 * {@code 0} on a unary reference is a definition error.
 *
 * @author $Id: c84dcc45de25d3256e40e9b481fc4597b23178d3 $
 */
@Documented
@Qualifier
@Retention(RUNTIME)
@Target({
		FIELD, PARAMETER
})
public @interface MinimumCardinality {

	/**
	 * Support inline instantiation of the {@link MinimumCardinality} annotation.
	 */
	public static final class Literal extends AnnotationLiteral<MinimumCardinality>
			implements MinimumCardinality {

		private static final long serialVersionUID = 1L;

		/**
		 * @param value the minimum cardinality
		 * @return an instance of {@link MinimumCardinality}
		 */
		public static final Literal of(int value) {
			return new Literal(value);
		}

		private Literal(int value) {
			_value = value;
		}

		@Override
		public int value() {
			return _value;
		}

		private final int _value;

	}

	/**
	 * The minimum cardinality of the reference.
	 * <p>
	 * The value must be a positive integer.
	 * <p>
	 * For example:
	 *
	 * <pre>
	 * {@code @MinimumCardinality(3)}
	 * </pre>
	 */
	int value() default 1;

}
