/*
 *                           TERMS AND CONDITIONS
 *                                   FOR
 *                         OPEN SOURCE CODE LICENSE
 *                               Version 1.1
 * 
 * Japan Registry Services Co., Ltd. ("JPRS"), a Japanese corporation
 * having its head office at Chiyoda First Bldg. East 13F 3-8-1 Nishi-Kanda,
 * Chiyoda-ku, Tokyo 101-0065, Japan, grants you the license for open source
 * code specified in EXHIBIT A the "Code" subject to the following Terms and
 * Conditions ("OSCL").
 * 
 * 1. License Grant.
 *   JPRS hereby grants you a worldwide, royalty-free, non-exclusive
 *   license, subject to third party intellectual property claims:
 *   (a) under intellectual property rights (other than patent or
 *       trademark) licensable by JPRS to use, reproduce, modify, display,
 *       perform, sublicense and distribute the Code (or portions thereof)
 *       with or without modifications, and/or as part of a derivative work;
 *       or
 *   (b) under claims of the infringement through the making, using,
 *       offering to sell and/or otherwise disposing the JPRS Revised Code
 *       (or portions thereof);
 *   (c) the licenses granted in this Section 1(a) and (b) are effective on
 *       the date JPRS first distributes the Code to you under the terms of
 *       this OSCL;
 *   (d) Notwithstanding the above stated terms, no patent license is
 *       granted:
 *       1)  for a code that you delete from the Code;
 *       2)  separate from the Code; or
 *       3)  for infringements caused by:
 *            i) modification of the Code; or
 *           ii) combination of the Code with other software or devices.
 * 
 * 2. Consents.
 *   You agree that:
 *   (a) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of the Code you distribute;
 *   (b) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of binary form of the Code in the
 *       documentation and/or other materials provided with the distribution;
 *   (c) you may not offer or impose any terms on any source code version
 *       that alters or restricts the applicable version of this OSCL or
 *       the recipients' rights hereunder.
 *   (d) If the terms and conditions are set forth in EXHIBIT A, you must
 *       comply with those terms and conditions.
 * 
 * 3. Proprietary Information.
 *   All trademarks, service marks, patents, copyrights, trade secrets, and
 *   other proprietary rights in or related to the Code are and will remain
 *   the exclusive property of JPRS or its licensors, whether or not
 *   specifically recognized or perfected under local law except specified
 *   in this OSCL; provided however you agree and understand that the JPRS
 *   name may not be used to endorse or promote this Code without prior
 *   written approval of JPRS.
 * 
 * 4. WARRANTY DISCLAIMER.
 *   JPRS MAKES NO REPRESENTATIONS AND WARRANTIES REGARDING THE USE OF THE
 *   CODE, NOR DOES JPRS MAKE ANY REPRESENTATIONS THAT THE CODE WILL BECOME
 *   COMMERCIALLY AVAILABLE. JPRS, ITS AFFILIATES, AND ITS SUPPLIERS DO NOT
 *   WARRANT OR REPRESENT THAT THE CODE IS FREE OF ERRORS OR THAT THE CODE
 *   IS SUITABLE FOR TRANSLATION AND/OR LOCALIZATION. THE CODE IS PROVIDED
 *   ON AN "AS IS" BASIS AND JPRS AND ITS SUPPLIERS HAVE NO OBLIGATION TO
 *   CORRECT ERRORS OR TO SUPPORT THE CODE UNDER THIS OSCL FOR ANY REASON.
 *   TO THE FULL EXTENT PERMITTED BY LAW, ALL OBLIGATIONS ARE HEREBY
 *   EXCLUDED WHETHER EXPRESS, STATUTORY OR IMPLIED UNDER LAW, COURSE OF
 *   DEALING, CUSTOM, TRADE USAGE, ORAL OR WRITTEN STATEMENT OR OTHERWISE,
 *   INCLUDING BUT NOT LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
 *   OR FITNESS FOR A PARTICULAR PURPOSE CONCERNING THE CODE.
 * 
 * 5. NO LIABILITY.
 *   UNDER NO CIRCUMSTANCES SHALL JPRS AND/OR ITS AFFILIATES, LICENSORS, OR
 *   REPRESENTATIVES BE LIABLE FOR ANY DAMAGES INCLUDING BUT NOT LIMITED TO
 *   CONSEQUENTIAL, INDIRECT, SPECIAL, PUNITIVE OR INCIDENTAL DAMAGES,
 *   WHETHER FORESEEABLE OR UNFORESEEABLE, BASED ON YOUR CLAIMS, INCLUDING,
 *   BUT NOT LIMITED TO, CLAIMS FOR LOSS OF DATA, GOODWILL, PROFITS, USE OF
 *   MONEY, INTERRUPTION IN USE OR AVAILABILITY OF DATA, STOPPAGE, IMPLIED
 *   WARRANTY, BREACH OF CONTRACT, MISREPRESENTATION, NEGLIGENCE, STRICT
 *   LIABILITY IN TORT, OR OTHERWISE.
 * 
 * 6. Indemnification.
 *   You hereby agree to indemnify, defend, and hold harmless JPRS for any
 *   liability incurred by JRPS due to your terms of warranty, support,
 *   indemnity, or liability offered by you to any third party.
 * 
 * 7. Termination.
 * 7.1 This OSCL shall be automatically terminated in the events that:
 *   (a) You fail to comply with the terms herein and fail to cure such
 *       breach within 30 days of becoming aware of the breach;
 *   (b) You initiate patent or copyright infringement litigation against
 *       any party (including a cross-claim or counterclaim in a lawsuit)
 *       alleging that the Code constitutes a direct or indirect patent or
 *       copyright infringement, in such case, this OSCL to you shall
 *       terminate as of the date such litigation is filed;
 * 7.2 In the event of termination under Sections 7.1(a) or 7.1(b) above,
 *     all end user license agreements (excluding distributors and
 *     resellers) which have been validly granted by You or any distributor
 *     hereunder prior to termination shall survive termination.
 *
 * 
 * 8. General.
 *   This OSCL shall be governed by, and construed and enforced in
 *   accordance with, the laws of Japan. Any litigation or arbitration
 *   between the parties shall be conducted exclusively in Tokyo, Japan
 *   except written consent of JPRS provides other venue.
 * 
 * 
 *                                EXHIBIT A
 * 
 * The original open source code of idnkit-2 is idnkit-1.0 developed and
 * conceived by Japan Network Information Center ("JPNIC"), a Japanese
 * association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan, and JPRS modifies above original code
 * under following Terms and Conditions set forth by JPNIC.
 * 
 *                                  JPNIC
 * 
 * Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved.
 * 
 * By using this file, you agree to the terms and conditions set forth bellow.
 * 
 *                       LICENSE TERMS AND CONDITIONS
 * 
 * The following License Terms and Conditions apply, unless a different
 * license is obtained from Japan Network Information Center ("JPNIC"),
 * a Japanese association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan.
 * 
 * 1. Use, Modification and Redistribution (including distribution of any
 *    modified or derived work) in source and/or binary forms is permitted
 *    under this License Terms and Conditions.
 * 
 * 2. Redistribution of source code must retain the copyright notices as they
 *    appear in each source code file, this License Terms and Conditions.
 * 
 * 3. Redistribution in binary form must reproduce the Copyright Notice,
 *    this License Terms and Conditions, in the documentation and/or other
 *    materials provided with the distribution. For the purposes of binary
 *    distribution the "Copyright Notice" refers to the following language:
 *    "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * 
 * 4. The name of JPNIC may not be used to endorse or promote products
 *    derived from this Software without specific prior written approval of
 *    JPNIC.
 * 
 * 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
 *    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JPNIC BE LIABLE
 *    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * 
 *                        JPRS Public License Notice
 *                                   For
 *                                idnkit-2.
 * 
 * The contents of this file are subject to the Terms and Conditions for
 * the Open Source Code License (the "OSCL"). You may not use this file
 * except in compliance with above terms and conditions. A copy of the OSCL
 * is available at <http://jprs.co.jp/idn/>.
 * The JPRS Revised Code is idnkit-2.
 * The Initial Developer of the JPRS Revised Code is Japan Network
 * Information Center ("JPNIC"), a Japanese association,
 * Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda, Chiyoda-ku, Tokyo
 * 101-0047, Japan.
 * "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * "Copyright (c) 2010-2012 Japan Registry Services Co., Ltd.  All rights reserved."
 * Contributor(s): ______________________________________.
 * 
 * If you wish to allow use of your version of this file only under the
 * above License(s) and not to allow others to use your version of this
 * file, please indicate your decision by deleting the relevant provisions
 * above and replacing them with the notice and other provisions required
 * by the above License(s). If you do not delete the relevant provisions,
 * a recipient may use your version of this file under either the above
 * License(s).
 */

#include <config.h>

#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include <idn/assert.h>
#include <idn/debug.h>
#include <idn/logmacro.h>
#include <idn/result.h>
#include <idn/res.h>
#include <idn/res_internal.h>
#include <idn/labellist.h>
#include <idn/punycode.h>
#include <idn/utf32.h>
#include <idn/utf8.h>
#include <idn/util.h>

#define ENCODE_MASK \
	(IDN_UNICODECONV | IDN_MAP | IDN_ASCLOWER | IDN_RTCONV | \
	IDN_PROHCHECK | IDN_UNASCHECK | IDN_NFCCHECK | IDN_PREFCHECK | \
	IDN_HYPHCHECK | IDN_COMBCHECK | IDN_CTXJCHECK | IDN_CTXOCHECK | \
	IDN_CTXOLITECHECK | IDN_BIDICHECK | IDN_LOCALCHECK | IDN_IDNCONV | \
	IDN_LENCHECK | IDN_RTCHECK | IDN_LOCALCONV | IDN_LITEFORCEUTF8 | \
	IDN_UNDOIFERR)

#define DECODE_MASK \
	(IDN_UNICODECONV | IDN_MAP | IDN_ASCLOWER | IDN_IDNCONV | \
	IDN_PROHCHECK | IDN_UNASCHECK | IDN_NFCCHECK | IDN_PREFCHECK | \
	IDN_HYPHCHECK | IDN_COMBCHECK | IDN_CTXJCHECK | IDN_CTXOCHECK | \
	IDN_CTXOLITECHECK | IDN_BIDICHECK | IDN_LOCALCHECK | IDN_RTCHECK | \
	IDN_LOCALCONV |	IDN_LITEFORCEUTF8 | IDN_UNDOIFERR)

static int		have_rtl_label(idn_resconf_t ctx, 
				       idn__labellist_t labellist);
static int		is_ldh_label(idn__labellist_t label);

#ifdef LIBIDNKITLITE
static const int is_idnkitlite = 1;
#else
static const int is_idnkitlite = 0;
#endif

/*
 * Encode a domain name.
 */
idn_result_t
idn_res_encodename(idn_resconf_t ctx, idn_action_t actions, const char *from,
		   char *to, size_t tolen) {
	idn_result_t r = idn_success;
	char actions_string[ACTION_STRING_BUFSIE];
	char *utf8_from = NULL;
	unsigned long *utf32_from = NULL;
	unsigned long *utf32_to = NULL;
	char *utf8_to = NULL;
	idn__labellist_t labels = NULL, l;
	int have_rtl;

	assert(ctx != NULL && from != NULL && to != NULL);

	idn__res_actionstostring(actions, actions_string);
	TRACE(("idn_res_encodename(actions=%s, from=\"%s\", tolen=%d)\n",
		actions_string, idn__debug_xstring(from), (int)tolen));

	if (actions & ~ENCODE_MASK) {
		WARNING(("idn_res_encodename: invalid actions 0x%x\n",
			 actions));
		r = idn_invalid_action;
		goto ret;
	}

	if (is_idnkitlite && actions & IDN_LITEFORCEUTF8)
		actions &= ~(IDN_LOCALCONV | IDN_UNICODECONV);

	/*
	 * Convert `from' to UTF-32.
	 */
	if (actions & IDN_UNICODECONV)
		r = idn__res_unicodeconv(ctx, from, &utf8_from);
	else {
		utf8_from = idn__util_strdup(from);
		r = (utf8_from != NULL) ? idn_success : idn_nomemory;
	}
	if (r != idn_success)
		goto ret;

	r = idn__res_utf8toutf32(ctx, utf8_from, &utf32_from);
	if (r != idn_success)
		goto ret;

	/*
	 * Map.
	 */
	if (actions & IDN_MAP) {
		unsigned long *tmp_from;

		r = idn__res_map(ctx, utf32_from, &tmp_from);
		if (r != idn_success)
			goto ret;
		free(utf32_from);
		utf32_from = tmp_from;
	}

	/*
	 * Split the name into a list of labels.
	 */
	if ((utf32_from[0] == '\0') ||
	    (utf32_from[0] == '.' && utf32_from[1] == '\0')) {
		utf32_to = idn__utf32_strdup(utf32_from);
		if (utf32_to == NULL)
			r = idn_nomemory;
	} else {
		r = idn__labellist_create(utf32_from, &labels);
	}
	if (r != idn_success)
		goto ret;

	/*
	 * Perform conversions and tests.
	 */
	for (l = labels; l != NULL; l = idn__labellist_next(l)) {
		do {
			int is_xn;
			int is_ldh;

			if (actions & IDN_ASCLOWER) {
				r = idn__res_asclower(ctx, l);
				if (r != idn_success)
					break;
			}

			is_xn = idn__punycode_isacelabel(idn__labellist_getname(l));
			r = idn__labellist_setundoname(l);
			if (r != idn_success)
				break;
		
			if ((actions & IDN_RTCONV) && is_xn) {
				r = idn__labellist_setroundtripname(l);
				if (r != idn_success)
					break;
				r = idn__res_idnconv_decode(ctx, l);
				if (r != idn_success)
					break;
			}

			is_ldh = is_ldh_label(l);

			if ((actions & IDN_PROHCHECK) && !is_ldh) {
				r = idn__res_prohcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_UNASCHECK) && !is_ldh) {
				r = idn__res_unascheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_NFCCHECK) && !is_ldh) {
				r = idn__res_nfccheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_PREFCHECK) && !is_ldh) {
				r = idn__res_prefcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if (actions & IDN_HYPHCHECK) {
				r = idn__res_hyphcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_COMBCHECK) && !is_ldh) {
				r = idn__res_combcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_CTXJCHECK) && !is_ldh) {
				r = idn__res_ctxjcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_CTXOCHECK) && !is_ldh) {
				r = idn__res_ctxocheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_CTXOLITECHECK) && !is_ldh) {
				r = idn__res_ctxolitecheck(ctx, l);
				if (r != idn_success)
					break;
			}
		} while (0);

		if (r != idn_success && r != idn_nomemory &&
		    actions & IDN_UNDOIFERR) {
			r = idn__labellist_undo(l);
			if (r != idn_success)
				goto ret;
			TRACE(("idn_res_encodename(): undo (label=\"%s\")\n",
			       idn__debug_utf32xstring(idn__labellist_getname(l))));
			continue;
		}
		if (r != idn_success)
			goto ret;
	}

	have_rtl = have_rtl_label(ctx, labels);
	for (l = labels; l != NULL; l = idn__labellist_next(l)) {
		int is_xn = idn__punycode_isacelabel(idn__labellist_getroundtripname(l));
		int is_ldh = is_ldh_label(l);

		if (idn__labellist_getundocount(l) > 0)
			continue;

		do {
			if ((actions & IDN_BIDICHECK) && have_rtl) {
				r = idn__res_bidicheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if (actions & IDN_LOCALCHECK) {
				r = idn__res_localcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if (actions & IDN_IDNCONV && !is_ldh) {
				r = idn__res_idnconv_encode(ctx, l);
				if (r != idn_success)
					break;
			}
			if (actions & IDN_LENCHECK) {
				r = idn__res_lencheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_RTCHECK) && is_xn) {
				r = idn__res_rtcheck_encode(ctx, l);
				if (r != idn_success)
					break;
			}
		} while (0);

		if (r != idn_success && r != idn_nomemory &&
		    actions & IDN_UNDOIFERR) {
			r = idn__labellist_undo(l);
			if (r != idn_success)
				goto ret;
			TRACE(("idn_res_encodename(): undo (label=\"%s\")\n",
			       idn__debug_utf32xstring(idn__labellist_getname(l))));
			continue;
		}
		if (r != idn_success)
			goto ret;
	}

	/*
	 * Join labels.
	 */
	if (labels != NULL) {
		r = idn__res_joinlabels(ctx, labels, &utf32_to);
		if (r != idn_success)
			goto ret;
	}

	r = idn__res_utf32toutf8(ctx, utf32_to, &utf8_to);
	if (r != idn_success)
		goto ret;

	/*
	 * Perform IDN_LOCALCONV.
	 */
	if (actions & IDN_LOCALCONV)
		r = idn__res_localconv(ctx, utf8_to, to, tolen);
	else
		r = idn__util_strcpy(to, tolen, utf8_to);

ret:
	if (r == idn_success) {
		TRACE(("idn_res_encodename(): success (to=\"%s\")\n",
		       idn__debug_xstring(to)));
	} else {
		TRACE(("idn_res_encodename(): %s\n", idn_result_tostring(r)));
	}
	free(utf8_to);
	free(utf32_to);
	free(utf32_from);
	free(utf8_from);
	if (labels != NULL)
		idn__labellist_destroy(labels);
	return (r);
}

/*
 * Decode a domain name.
 */
idn_result_t
idn_res_decodename(idn_resconf_t ctx, idn_action_t actions, const char *from,
		    char *to, size_t tolen) {
	idn_result_t r = idn_success;
	char actions_string[ACTION_STRING_BUFSIE];
	char *utf8_from = NULL;
	unsigned long *utf32_from = NULL;
	unsigned long *utf32_to = NULL;
	char *utf8_to = NULL;
	idn__labellist_t labels = NULL, l;
	int have_rtl;

	assert(ctx != NULL && from != NULL && to != NULL);

	idn__res_actionstostring(actions, actions_string);
	TRACE(("idn_res_decodename(actions=%s, from=\"%s\", tolen=%d)\n",
		actions_string, idn__debug_xstring(from), (int)tolen));

	if (actions & ~DECODE_MASK) {
		WARNING(("idn_res_decodename: invalid actions 0x%x\n",
			 actions));
		r = idn_invalid_action;
		goto ret;
	}

	if (is_idnkitlite && actions & IDN_LITEFORCEUTF8)
		actions &= ~(IDN_LOCALCONV | IDN_UNICODECONV);

	/*
	 * Convert `from' to UTF-32.
	 */
	if (actions & IDN_UNICODECONV)
		r = idn__res_unicodeconv(ctx, from, &utf8_from);
	else {
		utf8_from = idn__util_strdup(from);
		r = (utf8_from != NULL) ? idn_success : idn_nomemory;
	}
	if (r != idn_success)
		goto ret;

	r = idn__res_utf8toutf32(ctx, utf8_from, &utf32_from);
	if (r != idn_success)
		goto ret;

	/*
	 * Map.
	 */
	if (actions & IDN_MAP) {
		unsigned long *tmp_from;

		r = idn__res_map(ctx, utf32_from, &tmp_from);
		if (r != idn_success)
			goto ret;
		free(utf32_from);
		utf32_from = tmp_from;
	}

	/*
	 * Split the name into a list of labels.
	 */
	if ((utf32_from[0] == '\0') ||
	    (utf32_from[0] == '.' && utf32_from[1] == '\0')) {
		utf32_to = idn__utf32_strdup(utf32_from);
		if (utf32_to == NULL)
			r = idn_nomemory;
	} else {
		r = idn__labellist_create(utf32_from, &labels);
	}
	if (r != idn_success)
		goto ret;

	/*
	 * Perform conversions and tests.
	 */
	for (l = labels; l != NULL; l = idn__labellist_next(l)) {
		do {
			int is_xn;
			int is_ldh;

			if (actions & IDN_ASCLOWER) {
				r = idn__res_asclower(ctx, l);
				if (r != idn_success)
					break;
			}

			is_xn  = idn__punycode_isacelabel(idn__labellist_getname(l));
			r = idn__labellist_setundoname(l);
			if (r != idn_success)
				break;

			if ((actions & IDN_IDNCONV) && is_xn) {
				r = idn__labellist_setroundtripname(l);
				if (r != idn_success)
					break;
				r = idn__res_idnconv_decode(ctx, l);
				if (r != idn_success)
					break;
			}

			is_ldh = is_ldh_label(l);

			if ((actions & IDN_PROHCHECK) && !is_ldh) {
				r = idn__res_prohcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_UNASCHECK) && !is_ldh) {
				r = idn__res_unascheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_NFCCHECK) && !is_ldh) {
				r = idn__res_nfccheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_PREFCHECK) && !is_ldh) {
				r = idn__res_prefcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if (actions & IDN_HYPHCHECK) {
				r = idn__res_hyphcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_COMBCHECK) && !is_ldh) {
				r = idn__res_combcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_CTXJCHECK) && !is_ldh) {
				r = idn__res_ctxjcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_CTXOCHECK) && !is_ldh) {
				r = idn__res_ctxocheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_CTXOLITECHECK) && !is_ldh) {
				r = idn__res_ctxolitecheck(ctx, l);
				if (r != idn_success)
					break;
			}
		} while (0);

		if (r != idn_success && r != idn_nomemory &&
		    actions & IDN_UNDOIFERR) {
			r = idn__labellist_undo(l);
			if (r != idn_success)
				goto ret;
			TRACE(("idn_res_decodename(): undo (label=\"%s\")\n",
			       idn__debug_utf32xstring(idn__labellist_getname(l))));
			continue;
		}
		if (r != idn_success)
			goto ret;
	}

	have_rtl = have_rtl_label(ctx, labels);
	for (l = labels; l != NULL; l = idn__labellist_next(l)) {
		int is_xn = idn__punycode_isacelabel(idn__labellist_getroundtripname(l));

		do {
			if ((actions & IDN_BIDICHECK) && have_rtl) {
				r = idn__res_bidicheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if (actions & IDN_LOCALCHECK) {
				r = idn__res_localcheck(ctx, l);
				if (r != idn_success)
					break;
			}
			if ((actions & IDN_RTCHECK) && is_xn) {
				r = idn__res_rtcheck_decode(ctx, l);
				if (r != idn_success)
					break;
			}
		} while (0);

		if (r != idn_success && r != idn_nomemory &&
		    actions & IDN_UNDOIFERR) {
			r = idn__labellist_undo(l);
			if (r != idn_success)
				goto ret;
			TRACE(("idn_res_decodename(): undo (label=\"%s\")\n",
			       idn__debug_utf32xstring(idn__labellist_getname(l))));
			continue;
		}
		if (r != idn_success)
			goto ret;
	}

	/*
	 * Join labels.
	 */
	if (labels != NULL) {
		r = idn__res_joinlabels(ctx, labels, &utf32_to);
		if (r != idn_success)
			goto ret;
	}

	r = idn__res_utf32toutf8(ctx, utf32_to, &utf8_to);
	if (r != idn_success)
		goto ret;

	/*
	 * Perform IDN_LOCALCONV.
	 */
	if (actions & IDN_LOCALCONV)
		r = idn__res_localconv(ctx, utf8_to, to, tolen);
	else
		r = idn__util_strcpy(to, tolen, utf8_to);

ret:
	if (r == idn_success) {
		TRACE(("idn_res_decodename(): success (to=\"%s\")\n",
		       idn__debug_xstring(to)));
	} else {
		TRACE(("idn_res_decodename(): %s\n", idn_result_tostring(r)));
	}
	free(utf8_to);
	free(utf32_to);
	free(utf32_from);
	free(utf8_from);
	if (labels != NULL)
		idn__labellist_destroy(labels);
	return (r);
}

/*
 * Decode a domain name auxiliary encoding support.
 */
idn_result_t
idn_res_decodename2(idn_resconf_t ctx, idn_action_t actions, const char *from,
		    char *to, size_t tolen, const char *auxencoding) {
	idn_result_t r = idn_success;
	char actions_string[ACTION_STRING_BUFSIE];
	idn_resconf_t auxctx = NULL;
	char *utf8_from = NULL;

	assert(ctx != NULL && from != NULL && to != NULL);

	idn__res_actionstostring(actions, actions_string);
	TRACE(("idn_res_decodename2(actions=%s, from=\"%s\", tolen=%d, "
		"auxencoding=\"%s\")\n",
		actions_string, idn__debug_xstring(from), (int)tolen,
		idn__debug_xstring(auxencoding)));

	if (auxencoding == NULL)
		auxencoding = IDN__UTF8_ENCODINGNAME;

	/*
	 * Convert `from' to UTF-8.
	 */
	r = idn_resconf_create(&auxctx);
	if (r != idn_success)
		goto ret;
	r = idn_resconf_setlocalencoding(auxctx, auxencoding);
	if (r != idn_success)
		goto ret;
	r = idn__res_unicodeconv(auxctx, from, &utf8_from);
	if (r != idn_success)
		goto ret;

	r = idn_res_decodename(ctx, actions & ~IDN_UNICODECONV, utf8_from,
			       to, tolen);

ret:
	if (r == idn_success) {
		TRACE(("idn_res_decodename2(): success (to=\"%s\")\n",
		       idn__debug_xstring(to)));
	} else {
		TRACE(("idn_res_decodename2(): %s\n", idn_result_tostring(r)));
	}

	free(utf8_from);
	if (auxctx != NULL)
		idn_resconf_destroy(auxctx);

	return (r);
}

/*
 * Compare two domain names.
 */
idn_result_t
idn_res_comparenames(idn_resconf_t ctx, idn_action_t actions,
		     const char *name1, const char *name2) {
	return (idn_res_comparenames2(ctx, actions, name1, actions, name2));
}

idn_result_t
idn_res_comparenames2(idn_resconf_t ctx,
		      idn_action_t actions1, const char *name1, 
		      idn_action_t actions2, const char *name2) {
	idn_result_t r = idn_success;
	char actions1_string[ACTION_STRING_BUFSIE];
	char actions2_string[ACTION_STRING_BUFSIE];
	char *to1 = NULL;
	char *to2 = NULL;
	size_t tolen1 = 256;  /* may be large enough. */
	size_t tolen2 = 256;  /* may be large enough. */

	assert(ctx != NULL && name1 != NULL && name2 != NULL);

	idn__res_actionstostring(actions1, actions1_string);
	idn__res_actionstostring(actions2, actions2_string);
	TRACE(("idn_res_comparenames2(actions1=%s, name1=\"%s\", "
	       "actions2=%s, name2=\"%s\")\n",
		actions1_string, idn__debug_xstring(name1),
		actions2_string, idn__debug_xstring(name2)));

	for (;;) {
		void *new_to1;

		new_to1 = realloc(to1, sizeof(*to1) * tolen1);
		if (new_to1 == NULL) {
			r = idn_nomemory;
			goto ret;
		}
		to1 = (char *)new_to1;
		r = idn_res_encodename(ctx, actions1, name1, to1, tolen1);
		if (r == idn_success)
			break;
		else if (r != idn_buffer_overflow)
			goto ret;
		tolen1 *= 2;
	}

	for (;;) {
		void *new_to2;

		new_to2 = realloc(to2, sizeof(*to2) * tolen2);
		if (new_to2 == NULL) {
			r = idn_nomemory;
			goto ret;
		}
		to2 = (char *)new_to2;
		r = idn_res_encodename(ctx, actions2, name2, to2, tolen2);
		if (r == idn_success)
			break;
		else if (r != idn_buffer_overflow)
			goto ret;
		tolen2 *= 2;
	}

	r = (idn__util_strcasecmp(to1, to2) == 0) ? idn_success : idn_neq;

ret:
	if (r == idn_success) {
		TRACE(("idn_res_comparenames2(): success "
		       "(name1=name2=\"%s\")\n",
		       idn__debug_xstring(to1)));
	} else if (r == idn_neq) {
		TRACE(("idn_res_comparenames2(): neq "
		       "(name1=\"%s\" <=> name2=\"%s\")\n",
		       idn__debug_xstring(to1),
		       idn__debug_xstring(to2)));
	} else {
		TRACE(("idn_res_comparenames2(): %s\n",
		       idn_result_tostring(r)));
	}
	free(to1);
	free(to2);

	return (r);
}

idn_result_t
idn_res_checkname(idn_resconf_t ctx, idn_action_t actions, const char *name) {
	idn_result_t r = idn_success;
	char actions_string[ACTION_STRING_BUFSIE];
	char *to = NULL;
	size_t tolen = 256;  /* may be large enough. */

	assert(ctx != NULL && name != NULL);

	idn__res_actionstostring(actions, actions_string);
	TRACE(("idn_res_checkname(actions=%s, name=\"%s\")\n",
		actions_string, idn__debug_xstring(name)));

	for (;;) {
		void *new_to;

		new_to = realloc(to, sizeof(*to) * tolen);
		if (new_to == NULL) {
			r = idn_nomemory;
			goto ret;
		}
		to = (char *)new_to;
		r = idn_res_encodename(ctx, actions, name, to, tolen);
		if (r == idn_success)
			break;
		else if (r != idn_buffer_overflow)
			goto ret;
		tolen *= 2;
	}

ret:
	if (r == idn_success) {
		TRACE(("idn_res_checkname(): success (name=\"%s\")\n",
		       idn__debug_xstring(to)));
	} else {
		TRACE(("idn_res_checkname(): %s\n", idn_result_tostring(r)));
	}
	free(to);

	return (r);
}

/*
 * Check if 'labellist' has at least one RTL label.
 */
static int
have_rtl_label(idn_resconf_t ctx, idn__labellist_t labellist) {
	idn__labellist_t l;

	for (l = labellist; l != NULL; l = idn__labellist_next(l)) {
		if (idn__res_isrtllabel(ctx, l))
			return (1);
	}

	return (0);
}

/*
 * Check if 'label' is an LDH label.
 */
static int
is_ldh_label(idn__labellist_t label) {
	const unsigned long *l;

	l = idn__labellist_getname(label);
	if (l == NULL)
		return (1);

	while (*l != '\0') {
		if (!('A' <= *l && *l <= 'Z') &&
		    !('a' <= *l && *l <= 'z') &&
		    !('0' <= *l && *l <= '9') &&
		    *l != '-') {
			return (0);
		}
		l++;
	}

	return (1);
}

/*
 * Convert 'actions' to a string.
 */
struct action_name_table {
	idn_action_t action;
	char *name;
};

static const struct action_name_table
combined_action_name_table[] = {
	{IDN_DECODE_REGIST,	"DECODE_REGIST"},
	{IDN_DECODE_LOOKUP,	"DECODE_LOOKUP"},
	{IDN_ENCODE_REGIST,	"ENCODE_REGIST"},
	{IDN_ENCODE_LOOKUP,	"ENCODE_LOOKUP"},
	{IDN_COMPARE_REGIST,	"COMPARE_REGIST"},
	{IDN_COMPARE_LOOKUP,	"COMPARE_LOOKUP"},
	{IDN_CHECK_REGIST,	"CHECK_REGIST"},
	{IDN_CHECK_LOOKUP,	"CHECK_LOOKUP"},
	{0,			NULL}
};

static const struct action_name_table
single_action_name_table[] = {
	{IDN_UNICODECONV,	"UNICODECONV"},
	{IDN_MAP,		"MAP"},
	{IDN_ASCLOWER,		"ASCLOWER"},
	{IDN_RTCONV,		"RTCONV"},
	{IDN_PROHCHECK,		"PROHCHECK"},
	{IDN_UNASCHECK,		"UNASCHECK"},
	{IDN_NFCCHECK,		"NFCCHECK"},
	{IDN_PREFCHECK,		"PREFCHECK"},
	{IDN_HYPHCHECK,		"HYPHCHECK"},
	{IDN_COMBCHECK,		"COMBCHECK"},
	{IDN_CTXJCHECK,		"CTXJCHECK"},
	{IDN_CTXOCHECK,		"CTXOCHECK"},
	{IDN_CTXOLITECHECK,	"CTXOLITECHECK"},
	{IDN_BIDICHECK,		"BIDICHECK"},
	{IDN_LOCALCHECK,	"LOCALCHECK"},
	{IDN_IDNCONV,		"IDNCONV"},
	{IDN_LENCHECK,		"LENCHECK"},
	{IDN_RTCHECK,		"RTCHECK"},
	{IDN_LOCALCONV,		"LOCALCONV"},
	{IDN_UNDOIFERR,		"UNDOIFERR"},
	{IDN_LITEFORCEUTF8,	"LITEFORCEUTF8"},
	{0,			NULL}
};

void
idn__res_actionstostring(idn_action_t actions, char *string) {
	const struct action_name_table *t;
	const idn_action_t aux_actions =
		IDN_LOCALCONV | IDN_LOCALCHECK | IDN_UNDOIFERR;

	*string = '\0';
	for (t = combined_action_name_table; t->name != NULL; t++) {
		if ((actions | aux_actions) == (t->action | aux_actions)) {
			strcpy(string, t->name);
			actions &= ~t->action;
			break;
		}
	}

	for (t = single_action_name_table; t->name != NULL; t++) {
		if (actions & t->action) {
			if (*string != '\0')
				strcat(string, "|");
			strcat(string, t->name);
		}
	}
}

/*
 * Obsolete functions.
 */
void
idn_res_enable(int on_off) {
	/* no operation. */
}
