/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.persistence.JoinColumn;
import javax.persistence.JoinColumns;
import javax.persistence.ManyToOne;
import javax.persistence.OneToOne;
import javax.persistence.Transient;

import org.eclipse.jsr220orm.core.internal.options.IntOption;
import org.eclipse.jsr220orm.generic.GenericEntityModelManager;
import org.eclipse.jsr220orm.generic.Utils;
import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;
import org.eclipse.jsr220orm.generic.reflect.RClass;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.Join;
import org.eclipse.jsr220orm.metadata.JoinPair;
import org.eclipse.jsr220orm.metadata.MetadataPackage;
import org.eclipse.jsr220orm.metadata.ReferenceAttribute;
import org.eclipse.jsr220orm.metadata.TypeMetaData;

/**
 * Handles mapping for reference attributes (ManyToOne and OneToOne). 
 */
public class ReferenceAttributeIO extends AttributeIO {

	protected ReferenceAttribute amd;
	
	public static final IntOption MAPPING_MANY_TO_ONE = new IntOption(1,
			"Many to one", "Foreign key in table for entity",
			Utils.getImage("ManyToOne16"));
			
	public static final IntOption MAPPING_ONE_TO_ONE = new IntOption(2,
			"One to one", "Unique foreign key in table for entity",
			Utils.getImage("OneToOne16"));			
	
	public ReferenceAttributeIO(EntityIO entityIO) {
		super(entityIO);
	}
	
	public AttributeMetaData getAttributeMetaData() {
		return amd;
	}
	
    public void getPossibleMappings(List ans) {
        ans.add(MAPPING_MANY_TO_ONE);
        ans.add(MAPPING_ONE_TO_ONE);
        ans.add(MAPPING_NOT_PERSISTENT);
    }
    
	public IntOption getMapping() {
		if (amd == null || amd.isNonPersistent()) {
			return MAPPING_NOT_PERSISTENT;
		}
		if (amd.isOneToOne()) {
			return MAPPING_ONE_TO_ONE;
		} else {
			return MAPPING_MANY_TO_ONE;
		}
	}
	
	public void setMapping(IntOption mapping) {
		if (MAPPING_NOT_PERSISTENT.equals(mapping)) {
			setPersistent(false);		
		} else {
			setPersistent(true);
			boolean oneToOne = MAPPING_ONE_TO_ONE.equals(mapping);
			amd.setOneToOne(oneToOne);
			if (!oneToOne) {
				amd.setMappedBy(null);
			}
		}
	}

	public boolean updateModelFromMetaData(RClass cls,
			RAnnotatedElement attribute, boolean metaDataChanged) {
		
		amd = (ReferenceAttribute)initAttributeMetaData(amd, attribute, 
				MetadataPackage.eINSTANCE.getReferenceAttribute());
		if (amd.isNonPersistent()) {
			return true;
		}
		
		EntityMetaData emd = entityIO.getEntityMetaData();
		if (!emd.isEntity()) {
			updateModelCleanup();
			entityIO.addProblem(getTypeName(emd) + " classes " +
					"may only have basic attributes", 
					attribute.getLocation());
			return true;
		}		
		
		TypeMetaData tmd = amd.getJavaType();
		if (!(tmd instanceof EntityMetaData)) {
			updateModelCleanup();
			addNullJavaTypeProblem(attribute);
			return true;
		}
		
		EntityMetaData target = (EntityMetaData)tmd;
		if (target.getEntityType() != EntityMetaData.TYPE_ENTITY) {
			updateModelCleanup();
			entityIO.addProblem(
					target.getShortName() + " is " +
					getTypeName(target) + " so it may not be the " +
					"target of a persistent relationship", 
					attribute.getLocation());
			entityIO.addDependencyOn(target);
			return true;			
		}

		if (!hasTableAndPrimaryKey(target)) {
			return false;
		}
		List targetPk = target.getTable().getPrimaryKeyList();
		
		entityIO.addDependencyOn(target);

		GenericEntityModelManager mm = entityIO.getModelManager();

		OneToOne oneToOne = attribute.getAnnotation(OneToOne.class);
		ManyToOne manyToOne = attribute.getAnnotation(ManyToOne.class);
		boolean usingMappedBy = false;
		amd.setOneToOne(oneToOne != null);
		if (oneToOne != null) {
			setFetchType(amd, oneToOne.fetch());
			amd.setOptional(oneToOne.optional());
			if (manyToOne != null) {
				entityIO.addProblem("ManyToOne and OneToOne may not be " +
						"used together", 
						((AnnotationEx)manyToOne).getLocation(null));
			}
			amd.setCascadeType(getCascadeBits(oneToOne.cascade()));
			usingMappedBy = updateModelMappedBy(oneToOne, target);
		} else {
			if (manyToOne == null) {
				manyToOne = mm.getAnnotationRegistry().getDefaultProxy(
						ManyToOne.class);
			}
			setFetchType(amd, manyToOne.fetch());
			amd.setOptional(manyToOne.optional());
			amd.setMappedBy(null);
			amd.setCascadeType(getCascadeBits(manyToOne.cascade()));
		}
	
		if (usingMappedBy) {
			deleteJoinAndSrcCols();
		} else {
			AnnotationEx jcs = (AnnotationEx)attribute.getAnnotation(
					JoinColumns.class);
			AnnotationEx jc = (AnnotationEx)attribute.getAnnotation(
					JoinColumn.class);
			Join join = entityIO.getJoinIO().updateModelJoin(amd.getJoin(), 
					jc,
					jcs == null ? null : (Object[])jcs.get("value"),
					jcs == null ? null : jcs.getLocation(null),
					metaDataChanged, emd.getTable(), target.getTable(),
					mm.getReferenceCNS(amd),
					getComment(amd), amd.isOptional(), 
					getRelativePosition(attribute));
			if (amd.getJoin() != join) {
				amd.setJoin(join);
			}
		}
				
		return true;
	}

	/**
	 * Get rid of anything for our attribute currently in the model. 
	 */
	protected void updateModelCleanup() {
		deleteJoinAndSrcCols();
		amd.setMappedBy(null);
	}

	/**
	 * If our attribute has a join then delete it and its source columns
	 * and set it to null.
	 */
	protected void deleteJoinAndSrcCols() {
		if (Utils.deleteJoinAndSrcCols(amd.getJoin())) {
			amd.setJoin(null);
		}
	}

	/**
	 * Update our model for any mappedBy inverse information.
	 */
	protected boolean updateModelMappedBy(OneToOne oneToOne, 
			EntityMetaData target) {
		String mappedBy = oneToOne.mappedBy();
		AttributeMetaData mb = null;
		boolean usingMappedBy;
		if (usingMappedBy = (mappedBy.length() > 0)) {
			mb = getMappedByAttribute((AnnotationEx)oneToOne, mappedBy, target);
			if (mb instanceof ReferenceAttribute) {
				ReferenceAttribute ra = (ReferenceAttribute)mb;
				if (ra.getJavaType() != entityIO.getEntityMetaData()) {
					entityIO.addProblem(
							"Attribute '" + mappedBy + "' has incorrect type: " +
							ra.getJavaType().getClassName(),
							((AnnotationEx)oneToOne).getLocation("mappedBy"));	
					mb = null;						
				}
			} else if (mb != null) {
				entityIO.addProblem(
						"Attribute '" + mappedBy + "' is not a reference",
						((AnnotationEx)oneToOne).getLocation("mappedBy"));	
				mb = null;
			}
		}
		amd.setMappedBy(mb);
		return usingMappedBy;
	}
	
	public void updateMetaDataFromModel(RClass cls, RAnnotatedElement attribute) {
		GenericEntityModelManager mm = getModelManager();
		
		if (amd.isNonPersistent()) {
			Utils.removeAnnotation(attribute, OneToOne.class);			
			Utils.removeAnnotation(attribute, ManyToOne.class);			
			Utils.removeAnnotation(attribute, JoinColumn.class);			
			Utils.removeAnnotation(attribute, JoinColumns.class);
			ensureTransient(attribute);
			return;
		} else {
			Utils.removeAnnotation(attribute, Transient.class);			
		}

		if (amd.getJavaType() == null) {
			return;
		}
		
		AnnotationEx main = (AnnotationEx)attribute.getAnnotation(
				amd.isOneToOne() ? OneToOne.class : ManyToOne.class, true);
		if (main.getValueCount() == 0) {
			// if we will be creating a OneToOne or ManyToMany annotation then
			// get rid of any existing opposite one
			AnnotationEx toDelete = (AnnotationEx)attribute.getAnnotation(
					amd.isOneToOne() ? ManyToOne.class : OneToOne.class);
			if (toDelete != null) {
				toDelete.delete();
			}
		}

		AttributeMetaData mb = amd.getMappedBy();
		main.set("mappedBy", mb == null ? null : mb.getName());
		if (mb == null) {
			main.setMarker(amd.isOneToOne() || mm.isUseMarkerAnnotations());
		}

		Utils.setIfNotNull(main, "fetch", getFetchType(amd));
		main.set("optional", amd.isOptional());
		main.set("cascade", getCascadeTypes(amd.getCascadeType()));
		
		Join join = amd.getJoin();
		if (join == null) {
			// we were non-persistent previously - the join will be created
			// during the re-read with defaults so we have nothing more todo
			return;
		}
		
		if (join.getPairList().size() == 1) {
			Utils.removeAnnotation(attribute, JoinColumns.class);	
			AnnotationEx jc = (AnnotationEx)attribute.getAnnotation(
						JoinColumn.class, true);
			entityIO.getJoinIO().updateMetaDataJoinColumn(jc, join, 
					(JoinPair)join.getPairList().get(0),
					mm.getReferenceCNS(amd), amd.isOptional());
		} else {
			Utils.removeAnnotation(attribute, JoinColumn.class);	
			List pairList = join.getPairList();
			AnnotationEx joinColumns = (AnnotationEx)attribute.getAnnotation(
					JoinColumns.class, true);
			int n = pairList.size();
			if (joinColumns.setArraySize("value", n)) {
				JoinColumn[] a = ((JoinColumns)joinColumns).value();
				for (int i = 0; i < n; i++) {
					entityIO.getJoinIO().updateMetaDataJoinColumn(
							(AnnotationEx)a[i], join, 
							(JoinPair)pairList.get(i), 
							mm.getReferenceCNS(amd), amd.isOptional());
				}
			}
		}
	}
	
	public List getValidMappedByAttributes() {
		TypeMetaData tmd = amd.getJavaType();
		if (!(tmd instanceof EntityMetaData) || !amd.isOneToOne()) {
			return Collections.EMPTY_LIST;
		}
		// the emd on amd will be null if it is non-persistent so dont use that
		EntityMetaData emd = entityIO.getEntityMetaData();
		EntityMetaData target = (EntityMetaData)tmd;
		ArrayList ans = new ArrayList();
		for (Iterator i = target.getAttributeList().iterator(); i.hasNext(); ) {
			AttributeMetaData a = (AttributeMetaData)i.next();
			if (a.getMappedBy() == null 
					&& a instanceof ReferenceAttribute 
					&& a.getJavaType() == emd) {
				ans.add(a);
			}
		}
		return ans;
	}			

}
