/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.persistence.Embeddable;
import javax.persistence.EmbeddableSuperclass;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.IBuffer;
import org.eclipse.jdt.core.IClassFile;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IOpenable;
import org.eclipse.jdt.core.IParent;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.core.JarPackageFragmentRoot;
import org.eclipse.jsr220Orm.xml.EntityManagerDocument;
import org.eclipse.jsr220orm.core.OrmPlugin;
import org.eclipse.jsr220orm.core.nature.PersistenceProperties;
import org.eclipse.jsr220orm.generic.io.EntityIO;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.EntityModel;
import org.eclipse.jsr220orm.metadata.TypeMetaData;

import com.sun.org.apache.xerces.internal.impl.XMLEntityManager.Entity;

/**
 * Manages access to the persistence.xml deployment descriptor keeping it and
 * the entity model in sync. This class is also responsible for discovering
 * all of the entities in the model.
 */
public class PersistenceXmlManager implements IResourceChangeListener {

	protected final GenericEntityModelManager mm;
	protected final EntityModel model;
	
	protected EntityManagerDocument doc;
	protected EntityManagerDocument.EntityManager emElement;
    protected IFile xmlFile;
    
    protected boolean ignoreEvents;

    protected static final char[] JAVAX_PERSISTENCE_CHARS = 
    	"javax.persistence.".toCharArray();
    
    protected static final char[] ENTITY_CHARS = 
    	Entity.class.getSimpleName().toCharArray();
    
    protected static final char[] EMBEDDABLE_CHARS = 
    	Embeddable.class.getSimpleName().toCharArray();
    
    protected static final char[] EMBEDDABLE_SUPERCLASS_CHARS = 
    	EmbeddableSuperclass.class.getSimpleName().toCharArray();
	
	public PersistenceXmlManager(GenericEntityModelManager manager) {
		mm = manager;
		model = manager.getEntityModel();
        ResourcesPlugin.getWorkspace().addResourceChangeListener(this, IResourceChangeEvent.POST_CHANGE);
	}

    public void dispose(){
        ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
    }

    /**
     * Same as {@link #updateModelFromPersistenceXml(int)} but the type of 
     * newly created entities is {@link EntityMetaData#TYPE_ENTITY}.
     */
	public boolean updateModelFromPersistenceXml() throws Exception {
		return updateModelFromPersistenceXml(EntityMetaData.TYPE_ENTITY);
	}
    
	/**
	 * <p>Synchronize the model with persistence.xml. This will create a new
	 * persistence.xml in the project if none exists. Otherwise the current
	 * descriptor is read. The list of class elements is combined with the
	 * list of entities discovered by scanning the project. Entities not in 
	 * the model are created. Entities in the model but not in persistence.xml 
	 * are removed. An EntityIO is created to manage each newly created entity.
	 * </p>
	 * 
	 * <p>Returns true if we changed persistence.xml and it should be resaved.
	 * </p>
	 * 
	 * <p>The type parameter specifies the type
     * of entities created ({@link EntityMetaData#TYPE_ENTITY} etc.) from
     * class elements. If the actual type of the entity is different it will
     * change itself when the model is updated.</p>
     */
	public boolean updateModelFromPersistenceXml(int entityType) 
			throws Exception {
		boolean changed = false;
		
		boolean updateModel = false;
		
		Set classes = new HashSet();
		classes.addAll(Arrays.asList(emElement.getClass1Array()));
		
		// search the project for anything with @Entity on it and add class 
		// elements for each to persistence.xml if they are not already
		// there
		List<String> all = findEntitiesInProject();
		for (String name : all) {
			if (!classes.contains(name)) {
				emElement.addClass1(name);
				classes.add(name);
				changed = true;
			}
		}
		
		// create missing entities
		List classList = new ArrayList(classes);
		Collections.sort(classList);
    	List typeList = model.getTypeList();
		for (Iterator i = classList.iterator(); i.hasNext(); ) {
			String className = (String)i.next();
			TypeMetaData tmd = model.findTypeByClassName(className);
			if (tmd != null && !(tmd instanceof EntityMetaData)) {
				// TODO need to raise a problem about this case
				continue;
			}
			EntityMetaData emd = (EntityMetaData)tmd;
			if (emd == null) {
				IType type = mm.getProject().findType(className);
				if (type == null) {
					
				} else {
					emd = mm.getFactory().createEntityMetaData();
					emd.setEntityModel(model);
					emd.setClassName(className);
					emd.setEntityType(entityType);
					typeList.add(emd);
					updateModel = true;
					new EntityIO(mm, emd, type);
					// the EntityIO associates itself with emd
				}
			}
		}
		
		// get rid of entities no longer in persistence.xml
		for (Iterator i = typeList.iterator(); i.hasNext(); ) {
			TypeMetaData tmd = (TypeMetaData)i.next();
			if (tmd instanceof EntityMetaData 
					&& !classes.contains(tmd.getClassName())) {
				i.remove();
		        updateModel = true;
		        EntityIO entityIO = (EntityIO)tmd.adapt(EntityIO.class);
		        if (entityIO != null) {
		        	entityIO.dispose();
		        }
			}
		}

		if (updateModel) {
			ArrayList<EntityIO> a = new ArrayList();
			for (Iterator i = typeList.iterator(); i.hasNext(); ) {
				Object o = i.next();
				if (o instanceof EntityMetaData) {
					EntityIO entityIO = EntityIO.get((EntityMetaData)o);
					if (entityIO != null) {
						a.add(entityIO);
					}
				}
			}
			mm.updateModelFromMetaData(a);
		}
		
		return changed;
	}

	/**
	 * Create our XmlBeans graph from the contents of persistence.xml. If
	 * this resource does not exist in the project then create a new
	 * persistence.xml.
	 * 
	 * TODO search the project for a persistence.xml
	 * TODO search the project for entities when creating new persistence.xml
	 */
	protected void loadPersistenceXml() throws Exception {
        doc = Utils.loadPersistenceXml(mm.getProject().getProject());
        emElement = doc.getEntityManager();
	}

	/**
	 * Overwrite persistence.xml from our XmlBeans graph. 
	 */
	public void savePersistenceXml() throws Exception {
		try {
			ignoreEvents = true;
			Utils.savePersistenceXml(mm.getProject().getProject(), doc);
		} finally {
			ignoreEvents = false;
		}
	}	
	
	/**
	 * Search the project for all Java files annotated with Entity and return
	 * the class names in alphabetical order.
	 */
	public List<String> findEntitiesInProject() {
		List<String> ans = new ArrayList();
		try {
			findEntitiesInProject(mm.getProject(), ans);
		} catch (JavaModelException e) {
			OrmPlugin.log(e);
		}
		Collections.sort(ans);
		return ans;
	}
	
    protected void findEntitiesInProject(IJavaElement root, List<String> ans)
    		throws JavaModelException {
		if (root instanceof IClassFile 
				|| root instanceof JarPackageFragmentRoot) {
		    return;
		}
    	if (root instanceof ICompilationUnit) {
    		ICompilationUnit cu = (ICompilationUnit)root;
    		IType[] types = cu.getTypes();
    		for (int i = 0; i < types.length; i++) {
    			IType t = types[i];
    			if (t.isBinary()) {
    				return;
    			}
    			if (t.isClass() && Flags.isPublic(t.getFlags())) {
					if (getEntityType(t) != 0) {
    					ans.add(t.getFullyQualifiedName());
    				}
				}
    		}
    	} else if (root instanceof IParent) {
		    IJavaElement[] children = ((IParent)root).getChildren();
		    for (int i = 0; i < children.length; i++) {
		    	findEntitiesInProject(children[i], ans);
		    }
		}
	}
    
    /**
     * Figure out what type of entity t is or return 0 if not an entity.
     * This scans its source code skipping comments looking for at Entity, 
     * at Embeddable and at EmbeddableSuperclass. It returns constants 
     * {@link EntityMetaData#TYPE_ENTITY} and so on or 0 if no annotation
     * was found.
     */
    protected int getEntityType(IType t) throws JavaModelException {
		int first = t.getSourceRange().getOffset();
		int last = t.getNameRange().getOffset();
    	IOpenable openable = t.getOpenable();
    	IBuffer buffer = openable.getBuffer();
    	if (buffer == null) {
    		return 0;
    	}
    	int state = 0;
    	char[] ann = new char[64];
    	int annLength = 0;
    	for (int i = first; i < last; ) {
    		char c = buffer.getChar(i++);
    		switch (state) {
    		case 0:
    			switch (c) {
    			case '/':	
    				state = '/';	
    				break;
    			case '@':	
    				state = '@';	
    				annLength = 0;
    				break;
    			}
    			break;
    		case '/':
    			switch (c) {
    			case '/':	state = -2;		break;
    			case '*':	state = -3;		break;
    			}
    			break;
    		case -2:	// single line comment
    			if (c == '\n') {
    				state = 0;
    			}
    			break;
    		case -3:	// block comment
    			if (c == '*' && i < last && buffer.getChar(i) == '/') {
    				state = 0;
    				++i;
    			}
    			break;
    		case '@':	// annotation - collect characters of name
    			if (c >= 'a' && c <= 'z' || c >= 'A' && c <= 'Z' || c == '.') {
    				if (annLength == ann.length) { // too long - discard
    					state = 0;
    				} else {
    					ann[annLength++] = c;
    				}
    			} else { // end of the annotation name so see what we have
    				int offset;
    				if (equal(ann, 0, annLength, JAVAX_PERSISTENCE_CHARS)) {
    					offset = JAVAX_PERSISTENCE_CHARS.length;
    					annLength -= offset;
    				} else {
    					offset = 0;
    				}
    				if (equal(ann, offset, annLength, ENTITY_CHARS)) {
    					return EntityMetaData.TYPE_ENTITY;
    				}
    				if (equal(ann, offset, annLength, EMBEDDABLE_CHARS)) {
    					return EntityMetaData.TYPE_EMBEDDABLE;
    				}
    				if (equal(ann, offset, annLength, EMBEDDABLE_SUPERCLASS_CHARS)) {
    					return EntityMetaData.TYPE_EMBEDDABLE_SUPERCLASS;
    				}
    			}
    			break;
    		}
    	}
    	return 0;
    }

    /**
     * Are the characters of a starting at offset the same as those of b?
     * The len parameter indicates the total number of characters in a
     * from offset onwards.
     */
    protected static boolean equal(char[] a, int offset, int len, char[] b) {
    	int n = b.length;
		if (len < n) {
    		return false;
    	}
    	for (int i = 0; i < n; ) {
    		if (a[offset++] != b[i++]) {
    			return false;
    		}
    	}
    	return true;
    }
    
    /**
     * Make all the classes entities in the model. The collection contains
     * their fully qualified names. The type parameter specifies the type
     * of entities created ({@link EntityMetaData#TYPE_ENTITY} etc.).
     */
    public void makePersistent(Collection<String> names, int entityType) 
    		throws Exception {
    	// TODO use the type parameter
    	for (String fullName : names) {
            emElement.addClass1(fullName);			
		}
        savePersistenceXml();
        updateModelFromPersistenceXml(entityType);
    }

    /**
     * Remove the entities from the model.
     */
    public void makeNotPersistent(Collection<EntityMetaData> col) 
    		throws Exception {
    	for (EntityMetaData emd : col) {
    		String className = emd.getClassName();
    		String[] all = emElement.getClass1Array();
    		for (int i = all.length - 1; i >= 0; i--) {
    			if (all[i].equals(className)) {
    				emElement.removeClass1(i);
    			}
    		}
		}
    	savePersistenceXml();
    	updateModelFromPersistenceXml();
    }
    
    public void resourceChanged(IResourceChangeEvent changeEvent) {
    	if (ignoreEvents) {
    		return;
    	}
        IResourceDelta delta = changeEvent.getDelta();
        if(xmlFile == null){
            IProject project = mm.getProject().getProject();
            try {
                PersistenceProperties props = new PersistenceProperties(project);
                String name = props.getPersistenceFileName();
                if (name != null) {
                    xmlFile = project.getFile(name);
                }
            } catch (Exception e) {
                //Ok if no props
            }
            if(xmlFile == null){ //Not in catch because prop may be null
                return;
            }
        }
        if (delta.findMember(xmlFile.getFullPath()) != null) {
            GenericPlugin.getStandardDisplay().asyncExec(new Runnable() {
                public void run() {
                    try{
                        if(mm.getEntityModel() == model && model != null){
                            loadPersistenceXml();
                            updateModelFromPersistenceXml();
                        }
                    } catch(Exception x){
                        GenericPlugin.logException(x);
                    }
                }
            });
        }
    }	
}
