/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.ui.internal.selection;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.jface.viewers.IPostSelectionProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jsr220orm.core.IEntityModelManager;
import org.eclipse.jsr220orm.core.internal.product.OrmProject;
import org.eclipse.jsr220orm.core.nature.OrmNature;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.EntityModel;
import org.eclipse.jsr220orm.metadata.MetadataElement;
import org.eclipse.jsr220orm.metadata.TypeMetaData;
import org.eclipse.jsr220orm.ui.OrmUiPlugin;
import org.eclipse.ui.texteditor.ITextEditor;

public class ITextEditorSelectionProvider implements  ISelectionChangedListener, 
            PersistenceSelectionProvider {

    private ITextEditor textEditor;
    private List listeners = new ArrayList(5);
    private ICompilationUnit unit;
    private Object lastSelection;

    public ITextEditorSelectionProvider() {
//        JavaCore.addElementChangedListener(this, ElementChangedEvent.POST_RECONCILE);
    }

    public void installITextEditor(ITextEditor textEditor, ICompilationUnit unit) {
        this.textEditor = textEditor;
        this.unit = unit;
        ISelectionProvider selectionProvider = textEditor.getSelectionProvider();
        install(selectionProvider);
//        setSelection(getSelection());
    }

    public void uninstallITextEditor() {
        if (textEditor != null) {
            ISelectionProvider selectionProvider = textEditor.getSelectionProvider();
            if (selectionProvider != null) {
                uninstall(selectionProvider);
            }
            textEditor = null;
        }
    }

    /**
     * Installs this selection changed listener with the given selection
     * provider. If the selection provider is a post selection provider, post
     * selection changed events are the preferred choice, otherwise normal
     * selection changed events are requested.
     * 
     * @param selectionProvider
     */
    private void install(ISelectionProvider selectionProvider) {
        if (selectionProvider == null) {
            return;
        }
        if (selectionProvider instanceof IPostSelectionProvider) {
            IPostSelectionProvider provider = (IPostSelectionProvider) selectionProvider;
            provider.addPostSelectionChangedListener(this);
        } else {
            selectionProvider.addSelectionChangedListener(this);
        }
    }

    /**
     * Removes this selection changed listener from the given selection
     * provider.
     * 
     * @param selectionProvider
     *            the selection provider
     */
    private void uninstall(ISelectionProvider selectionProvider) {
        if (selectionProvider == null) {
            return;
        }
        if (selectionProvider instanceof IPostSelectionProvider) {
            IPostSelectionProvider provider = (IPostSelectionProvider) selectionProvider;
            provider.removePostSelectionChangedListener(this);
        } else {
            selectionProvider.removeSelectionChangedListener(this);
        }
    }

    /*
     * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
     */
    public void selectionChanged(SelectionChangedEvent event) {
        ISelection selection = getNewSelection(event.getSelection());
        if (selection != null && !selection.isEmpty()) {
            setSelection(selection);
        }
    }

    /*
     * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
     */
    public PersistenceSelection getNewSelection(ISelection selection) {
        if (selection instanceof IStructuredSelection) {
            IStructuredSelection structSel = (IStructuredSelection) selection;
            Object firstElement = structSel.getFirstElement();
            if (firstElement instanceof IJavaElement) {
                IJavaElement element = (IJavaElement) firstElement;
                try {
                    return getSelectionForJavaElement(element);
                } catch (CoreException e) {
                    OrmUiPlugin.log(e);
                }
            }
        } else if (selection instanceof ITextSelection) {
            ITextSelection textSelection = (ITextSelection) selection;
            int offset = textSelection.getOffset();
//            if (textEditor != null) {
//                IEditorInput editorInput = textEditor.getEditorInput();
//                IWorkingCopyManager manager = JavaUI.getWorkingCopyManager();
//                ICompilationUnit unit = manager.getWorkingCopy(editorInput);
//                if (unit == null) {
//                    try {
//                        manager.connect(editorInput);
//                        unit = manager.getWorkingCopy(editorInput);
//                        return getSelectionForJavaElement(getJavaElement(unit, offset));
//                    } catch (CoreException e1) {
//                        OrmUiPlugin.log(null, null, e1);
//                        return null;
//                    }finally{
//                        manager.disconnect(editorInput);
//                    }
//                }else{
                    try {
//                        if(!unit.isConsistent()){
//                            unit.reconcile(AST.JLS3, false, unit.getOwner(), null);
//                        }
                        return getSelectionForJavaElement(getJavaElement(unit, offset));
                    } catch (CoreException e1) {
                        OrmUiPlugin.log(e1);
                        return null;
                    }
//                }
//            }
        }
        return null;
    }

    public IJavaElement getJavaElement(ICompilationUnit unit, int offset){
        if (unit != null) {
            try {
                IJavaElement element = unit.getElementAt(offset);
                if(element == null){
                    return unit;
                }
                return  element;
            } catch (JavaModelException x) {
                if (!x.isDoesNotExist())
                    OrmUiPlugin.log(x);
                // nothing found, be tolerant and go on
            }
        }
        return null;
    }
        
    private PersistenceSelection getSelectionForJavaElement(IJavaElement element)
            throws CoreException {
        if (element == null) {
            return null;
        }
        String className = null;
        String fieldName = null;
        String methodName = null;
        IType classElement = (IType) element.getAncestor(IJavaElement.TYPE);
        IJavaElement fieldElement = element.getAncestor(IJavaElement.FIELD);
        IJavaElement methodElement = element.getAncestor(IJavaElement.METHOD);
        if (classElement == null) {
            ICompilationUnit unit = (ICompilationUnit) element
                    .getAncestor(IJavaElement.COMPILATION_UNIT);
            if (unit != null) {
                classElement = unit.findPrimaryType();
            }
        }
        if (classElement != null) {
            className = classElement.getFullyQualifiedName();
        }
        if (fieldElement != null) {
            fieldName = fieldElement.getElementName();
        }
        if (methodElement != null) {
            methodName = methodElement.getElementName();
        }
        if (className != null) {
            IProject project = element.getJavaProject().getProject();
            OrmNature nature = (OrmNature) project.getNature(OrmNature.ID);
            if(nature == null){
                return null;
            }
            OrmProject ormProject = nature.getActiveOrmProject();
            if(ormProject == null){
                return null;
            }
            EntityModel model = ormProject.getModel();
            if(model == null){
                return null;
            }
            TypeMetaData typeMetaData = model
                    .findTypeByClassName(className);
            if (typeMetaData instanceof EntityMetaData) {
                EntityMetaData entityMetaData = (EntityMetaData) typeMetaData;
                IEntityModelManager modelManager = ormProject.getModelManager();
                AttributeMetaData attributeMetaData = null;
                if (fieldName != null) {
                    attributeMetaData = 
                        modelManager.getAttributeMetaData(entityMetaData, fieldName, false);
                }else if(methodName != null){
                    attributeMetaData = 
                        modelManager.getAttributeMetaData(entityMetaData, methodName, true);
                }
                if (attributeMetaData != null) {
                    return createSelection(attributeMetaData);
                }
                return createSelection(typeMetaData);
            }
        }
        return null;
    }

    private PersistenceSelection createSelection(final MetadataElement element) {
        PersistenceSelection selection = new PersistenceSelection(){
            public boolean isEmpty() {
                return element == null;
            }
            public MetadataElement getSelectedElement() {
                return element;
            }
        };
        return selection;
    }

    public void addSelectionChangedListener(ISelectionChangedListener listener) {
        listeners.add(listener);
    }

    public ISelection getSelection() {
        ISelectionProvider selectionProvider = textEditor.getSelectionProvider();
        if (selectionProvider != null) {
            PersistenceSelection selection = getNewSelection(selectionProvider
                    .getSelection());
            if (selection != null && !selection.isEmpty()) {
                return selection;
            }
        }
        return null;
    }

    public void removeSelectionChangedListener(
            ISelectionChangedListener listener) {
        listeners.remove(listener);
    }

    public void setSelection(ISelection selection) {
        if(!(selection instanceof PersistenceSelection)){
            return;
        }
        MetadataElement selectedElement = ((PersistenceSelection)selection).getSelectedElement();
//        if(lastSelection == selectedElement){
//            return;
//        }
//        lastSelection = selectedElement;
        final SelectionChangedEvent event = new SelectionChangedEvent(this,
                selection);
        fireSelectionChanged(event);
    }

    private void fireSelectionChanged(final SelectionChangedEvent event) {
        Object[] listenersArray = listeners.toArray();
        for (int i = 0; i < listenersArray.length; i++) {
            final ISelectionChangedListener l = (ISelectionChangedListener) listenersArray[i];
            Platform.run(new SafeRunnable() {
                public void run() {
                    l.selectionChanged(event);
                }
            });
        }
    }
}