/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.core;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.jsr220orm.core.nature.OrmNature;
import org.eclipse.jsr220orm.core.options.IIntOption;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.CollectionAttribute;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.EntityModel;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.models.dbdefinition.PredefinedDataTypeDefinition;

/**
 * This class provides services for GUI tools related to the model.
 * 
 * @see EntityModel#getEntityModelManager()
 */
public interface IEntityModelManager {

    /**
     * This will be the first method called after construction
     * abd should initialize the model manager.
     */
    void init(IProject project, OrmNature nature, DatabaseDefinition definition) throws Exception;

    /**
     * The EntityModel associated with this model manager.
     * This method will only be called after init and should 
     * never return null.
     */
    EntityModel getEntityModel();

    /**
     * This model manager is not longer used so clean up.
     */
    void dispose();

	/**
	 * All the possible options of how to map this entity. If emd is null
	 * then the entity is currently not persistent.
	 */
	List getPossibleEntityMappings(EntityMetaData emd);
	
    /**
     * The current mapping option of this entity. If emd is null then the
     * entity is currently not persistent.
     */
	IIntOption getEntityMapping(EntityMetaData emd);

    /**
     * Use this mapping option to map emd (will never be null). Note that
     * {@link #makePersistent(Collection, IIntOption)} is used to make
     * a non-persistent class persistent.
     */
	void setEntityMapping(EntityMetaData emd, IIntOption intOption);

    /**
     * Make all the classes entities in the model. The collection contains
     * their fully qualified names. The option parameter is one of the
     * values returned by a call to {@link #getEntityMapping(EntityMetaData)} 
     * with a null parameter.
     * 
     * @see #setEntityMapping(EntityMetaData, IIntOption)
     */
    void makePersistent(Collection names, IIntOption option);	
	
	/**
	 * All the possible options of how to map this attribute.
	 */
	List getPossibleAttributeMappings(AttributeMetaData amd);
	
	/**
	 * The current mapping option of this attribute.
	 */
	IIntOption getAttributeMapping(AttributeMetaData amd);
	
	/**
	 * Use this mapping option to map this attribute.
	 */
	void setAttributeMapping(AttributeMetaData amd, IIntOption intOption);
	
    /**
     * The valid id generator types for this entity
     * @see EntityMetaData.ID_GENERATOR_TYPE_NONE
     * @see EntityMetaData.ID_GENERATOR_TYPE_TABLE
     * @see EntityMetaData.ID_GENERATOR_TYPE_SEQUENCE
     * @see EntityMetaData.ID_GENERATOR_TYPE_IDENTITY
     * @see EntityMetaData.ID_GENERATOR_TYPE_AUTO
     */
    int[] getValidIdGeneratorTypes(EntityMetaData entityMetaData);

    /**
     * The valid access types for this entity
     * @see EntityMetaData.ACCESS_TYPE_FIELD
     * @see EntityMetaData.FETCH_TYPE_LAZY
     */
    int[] getValidAccessTypes(EntityMetaData entityMetaData);
    
    /**
     * The valid fetch types for this attribute
     * @see AttributeMetaData.FETCH_TYPE_EAGERE
     * @see AttributeMetaData.ACCESS_TYPE_PROPERTY
     */
    int[] getValidFetchTypes(AttributeMetaData entityMetaData);

    /**
     * All the valid JDBC type that this column can map to.
     * @see java.sql.Types
     */
    int[] getValidJdbcTypes(OrmColumn column);

    /**
     * The user has selected this database as the target.
     */
    void setDatabaseDefinition(DatabaseDefinition definition);

    /**
     * This helper method will return the AttributeMetaData 
     * corresponding to the field or method name passed. This might return
     * a persistent attribute, a non-persistent attribute or null.
     * 
     * @see AttributeMetaData#isNonPersistent()
     * 
     * @param entityMetaData the EntityMetaData to search in.
     * @param name the name of the field or method (eg. person or getPerson)
     * @param isMethod true if this is a method false if it is a field
     * 
     * @return the AttributeMetaData representing the field or method
     */
    AttributeMetaData getAttributeMetaData(EntityMetaData entityMetaData, 
    		String name, boolean isMethod);
    
    /**
     * Get the PredefinedDataTypeDefinition (RDB) for a column.
     */
    PredefinedDataTypeDefinition getColumnDataTypeDefinition(OrmColumn c);

    /**
     * A list of possible TypeMetadata's that are valid element types for this 
     * collection.
     */
    List getPossibleElementTypes(CollectionAttribute collectionAttribute);

    /**
     * A list of possible AttributeMetaData's that attributeMD can set as it's
     * mapped by attribute.
     */
    List getValidMappedByAttribute(AttributeMetaData attributeMD);

    /**
     * Can this entity have a discriminator column.
     */
    boolean isUseDiscriminatorColumnEnabled(EntityMetaData entity);

    /**
     * Should this Entity have a discriminator column.
     */
    void setUseDiscriminatorColumn(EntityMetaData entity, boolean selection);

    /**
     * Executed runnable.run(), batch all model events fired while it is 
     * running and then process the events in one go. 
     */
    void executeModelChanges(Runnable runnable);
    
    /**
     * Can the entities idGeneratorName and idGeneratorType properties be
     * changed?
     */
	boolean isIdGeneratorEnabled(EntityMetaData emd);

    /**
     * The Model manager can add extra sql objects not defined in the model 
     * to be created by the ddl generator.
     */
    Set getExtraSqlObjects();
    
}
