/*
	Copyright (c) 2004-2009, The Dojo Foundation All Rights Reserved.
	Available via Academic Free License >= 2.1 OR the modified BSD license.
	see: http://dojotoolkit.org/license for details
*/


if(!dojo._hasResource["dojo.DeferredList"]){ //_hasResource checks added by build. Do not use _hasResource directly in your code.
dojo._hasResource["dojo.DeferredList"] = true;
dojo.provide("dojo.DeferredList");
dojo.declare("dojo.DeferredList", dojo.Deferred, {
	constructor: function(/*Array*/ list, /*Boolean?*/ fireOnOneCallback, /*Boolean?*/ fireOnOneErrback, /*Boolean?*/ consumeErrors, /*Function?*/ canceller){
		// summary:
		//		Provides event handling for a group of Deferred objects.
		// description:
		//		DeferredList takes an array of existing deferreds and returns a new deferred of its own
		//		this new deferred will typically have its callback fired when all of the deferreds in
		//		the given list have fired their own deferreds.  The parameters `fireOnOneCallback` and
		//		fireOnOneErrback, will fire before all the deferreds as appropriate
		//
		//	list:
		//		The list of deferreds to be synchronizied with this DeferredList
		//	fireOnOneCallback:
		//		Will cause the DeferredLists callback to be fired as soon as any
		//		of the deferreds in its list have been fired instead of waiting until
		//		the entire list has finished
		//	fireonOneErrback:
		//		Will cause the errback to fire upon any of the deferreds errback
		//	canceller:
		//		A deferred canceller function, see dojo.Deferred
		this.list = list;
		this.resultList = new Array(this.list.length);

		// Deferred init
		this.chain = [];
		this.id = this._nextId();
		this.fired = -1;
		this.paused = 0;
		this.results = [null, null];
		this.canceller = canceller;
		this.silentlyCancelled = false;

		if(this.list.length === 0 && !fireOnOneCallback){
			this.callback(this.resultList);
		}

		this.finishedCount = 0;
		this.fireOnOneCallback = fireOnOneCallback;
		this.fireOnOneErrback = fireOnOneErrback;
		this.consumeErrors = consumeErrors;

		dojo.forEach(this.list, function(d, index){
			d.addCallback(this, function(r){ this._cbDeferred(index, true, r); return r; });
			d.addErrback(this, function(r){ this._cbDeferred(index, false, r); return r; });
		}, this);
	},

	_cbDeferred: function(index, succeeded, result){
		// summary:
		//	The DeferredLists' callback handler

		this.resultList[index] = [succeeded, result]; this.finishedCount += 1;
		if(this.fired !== 0){
			if(succeeded && this.fireOnOneCallback){
				this.callback([index, result]);
			}else if(!succeeded && this.fireOnOneErrback){
				this.errback(result);
			}else if(this.finishedCount == this.list.length){
				this.callback(this.resultList);
			}
		}
		if(!succeeded && this.consumeErrors){
			result = null;
		}
		return result;
	},

	gatherResults: function(deferredList){
		// summary:	
		//	Gathers the results of the deferreds for packaging
		//	as the parameters to the Deferred Lists' callback

		var d = new dojo.DeferredList(deferredList, false, true, false);
		d.addCallback(function(results){
			var ret = [];
			dojo.forEach(results, function(result){
				ret.push(result[1]);
			});
			return ret;
		});
		return d;
	}
});

}
