"use strict";
/*********************************************************************
 * Copyright (c) 2023 Kichwa Coders Canada Inc. and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const tmp = require("tmp");
const fs = require("fs");
const utils_1 = require("./utils");
describe('config', function () {
    const emptyProgram = path.join(utils_1.testProgramsDir, 'empty');
    const emptySrc = path.join(utils_1.testProgramsDir, 'empty.c');
    function verifyLaunchWorks(test, commandLine, requestArgs) {
        return __awaiter(this, void 0, void 0, function* () {
            if (utils_1.debugServerPort) {
                // This test requires launching the adapter to work
                test.skip();
            }
            const dc = yield (0, utils_1.standardBeforeEach)(utils_1.defaultAdapter, commandLine);
            try {
                yield dc.hitBreakpoint((0, utils_1.fillDefaults)(test.test, requestArgs), {
                    path: emptySrc,
                    line: 3,
                });
            }
            finally {
                yield dc.stop();
            }
        });
    }
    it('can specify program via --config=', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const config = { program: emptyProgram };
            yield verifyLaunchWorks(this, [`--config=${JSON.stringify(config)}`], {});
        });
    });
    it('program via --config= can be overridden', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const config = { program: '/program/that/does/not/exist' };
            yield verifyLaunchWorks(this, [`--config=${JSON.stringify(config)}`], {
                program: emptyProgram,
            });
        });
    });
    it('can specify program via --config-frozen=', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const config = { program: emptyProgram };
            yield verifyLaunchWorks(this, [`--config-frozen=${JSON.stringify(config)}`], {});
        });
    });
    it('program via --config-frozen= can not be overridden', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const config = { program: emptyProgram };
            yield verifyLaunchWorks(this, [`--config-frozen=${JSON.stringify(config)}`], {
                program: '/program/that/does/not/exist',
            });
        });
    });
    it('can specify program via --config= using response file', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const config = { program: emptyProgram };
            const json = JSON.stringify(config);
            const jsonFile = tmp.fileSync();
            fs.writeFileSync(jsonFile.fd, json);
            fs.closeSync(jsonFile.fd);
            yield verifyLaunchWorks(this, [`--config=@${jsonFile.name}`], {});
        });
    });
    it('can specify program via --config-frozen= using response file', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const config = { program: emptyProgram };
            const json = JSON.stringify(config);
            const jsonFile = tmp.fileSync();
            fs.writeFileSync(jsonFile.fd, json);
            fs.closeSync(jsonFile.fd);
            yield verifyLaunchWorks(this, [`--config-frozen=@${jsonFile.name}`], {});
        });
    });
    // This test most closely models the original design goal
    // for the change that added --config and --config-frozen
    // as discussed in #227 and #228
    // In summary we force a launch request for the given program,
    // but the user does not specify the program and specifies
    // an attach request
    it('config frozen forces specific launch type', function () {
        return __awaiter(this, void 0, void 0, function* () {
            if (utils_1.debugServerPort) {
                // This test requires launching the adapter to work
                this.skip();
            }
            const config = { request: 'launch', program: emptyProgram };
            // Launch the adapter with the frozen config
            const dc = yield (0, utils_1.standardBeforeEach)(utils_1.defaultAdapter, [
                `--config-frozen=${JSON.stringify(config)}`,
            ]);
            try {
                yield Promise.all([
                    // Do an attach request omitting the program that we want
                    // the adapter to force into a launch request
                    dc.attachRequest((0, utils_1.fillDefaults)(this.test, {})),
                    // The rest of this code is to ensure we launcher properly by verifying
                    // we can run to a breakpoint
                    dc.waitForEvent('initialized').then((_event) => {
                        return dc
                            .setBreakpointsRequest({
                            lines: [3],
                            breakpoints: [{ line: 3 }],
                            source: { path: emptySrc },
                        })
                            .then((_response) => {
                            return dc.configurationDoneRequest();
                        });
                    }),
                    dc.assertStoppedLocation('breakpoint', {
                        path: emptySrc,
                        line: 3,
                    }),
                ]);
            }
            finally {
                yield dc.stop();
            }
        });
    });
});
//# sourceMappingURL=config.spec.js.map