"use strict";
// *****************************************************************************
// Copyright (C) 2018 Google and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
Object.defineProperty(exports, "__esModule", { value: true });
exports.bindCorePreferences = exports.createCorePreferences = exports.CorePreferences = exports.CorePreferenceContribution = exports.corePreferenceSchema = void 0;
const environment_1 = require("@theia/application-package/lib/environment");
const preferences_1 = require("./preferences");
const supported_encodings_1 = require("./supported-encodings");
const frontend_application_config_provider_1 = require("./frontend-application-config-provider");
const os_1 = require("../common/os");
const nls_1 = require("../common/nls");
const application_props_1 = require("@theia/application-package/lib/application-props");
/* eslint-disable max-len */
const windowTitleDescription = [
    'Controls the window title based on the active editor. Variables are substituted based on the context:',
    '`${activeEditorShort}`: the file name (e.g. myFile.txt).',
    '`${activeEditorMedium}`: the path of the file relative to the workspace folder (e.g. myFolder/myFileFolder/myFile.txt).',
    '`${activeEditorLong}`: the full path of the file (e.g. /Users/Development/myFolder/myFileFolder/myFile.txt).',
    '`${activeFolderShort}`: the name of the folder the file is contained in (e.g. myFileFolder).',
    '`${activeFolderMedium}`: the path of the folder the file is contained in, relative to the workspace folder (e.g. myFolder/myFileFolder).',
    '`${activeFolderLong}`: the full path of the folder the file is contained in (e.g. /Users/Development/myFolder/myFileFolder).',
    '`${folderName}`: name of the workspace folder the file is contained in (e.g. myFolder).',
    '`${folderPath}`: file path of the workspace folder the file is contained in (e.g. /Users/Development/myFolder).',
    '`${rootName}`: name of the opened workspace or folder (e.g. myFolder or myWorkspace).',
    '`${rootPath}`: file path of the opened workspace or folder (e.g. /Users/Development/myWorkspace).',
    '`${appName}`: e.g. VS Code.',
    '`${remoteName}`: e.g. SSH',
    '`${dirty}`: an indicator for when the active editor has unsaved changes.',
    '`${separator}`: a conditional separator (" - ") that only shows when surrounded by variables with values or static text.'
].map(e => nls_1.nls.localizeByDefault(e)).join('\n- ');
exports.corePreferenceSchema = {
    'type': 'object',
    properties: {
        'application.confirmExit': {
            type: 'string',
            enum: [
                'never',
                'ifRequired',
                'always',
            ],
            default: 'ifRequired',
            description: nls_1.nls.localizeByDefault('Controls whether to show a confirmation dialog before closing the browser tab or window. Note that even if enabled, browsers may still decide to close a tab or window without confirmation and that this setting is only a hint that may not work in all cases.'),
        },
        'breadcrumbs.enabled': {
            'type': 'boolean',
            'default': true,
            'description': nls_1.nls.localizeByDefault('Enable/disable navigation breadcrumbs.'),
            'scope': 'application'
        },
        'files.encoding': {
            'type': 'string',
            'enum': Object.keys(supported_encodings_1.SUPPORTED_ENCODINGS),
            'default': 'utf8',
            'description': nls_1.nls.localizeByDefault('The default character set encoding to use when reading and writing files. This setting can also be configured per language.'),
            'scope': 'language-overridable',
            'enumDescriptions': Object.keys(supported_encodings_1.SUPPORTED_ENCODINGS).map(key => supported_encodings_1.SUPPORTED_ENCODINGS[key].labelLong),
            'included': Object.keys(supported_encodings_1.SUPPORTED_ENCODINGS).length > 1
        },
        'keyboard.dispatch': {
            type: 'string',
            enum: [
                'code',
                'keyCode',
            ],
            default: 'code',
            markdownDescription: nls_1.nls.localizeByDefault('Controls the dispatching logic for key presses to use either `code` (recommended) or `keyCode`.')
        },
        'window.tabbar.enhancedPreview': {
            type: 'boolean',
            default: false,
            description: nls_1.nls.localize('theia/core/enhancedPreview', 'Controls whether more information about the tab should be displayed in horizontal tab bars.')
        },
        'window.menuBarVisibility': {
            type: 'string',
            enum: ['classic', 'visible', 'hidden', 'compact'],
            markdownEnumDescriptions: [
                nls_1.nls.localizeByDefault('Menu is displayed at the top of the window and only hidden in full screen mode.'),
                nls_1.nls.localizeByDefault('Menu is always visible at the top of the window even in full screen mode.'),
                nls_1.nls.localizeByDefault('Menu is always hidden.'),
                nls_1.nls.localizeByDefault('Menu is displayed as a compact button in the side bar. This value is ignored when {0} is {1}.', '`#window.titleBarStyle#`', '`native`')
            ],
            default: 'classic',
            scope: 'application',
            markdownDescription: nls_1.nls.localizeByDefault("Control the visibility of the menu bar. A setting of 'toggle' means that the menu bar is hidden and a single press of the Alt key will show it. A setting of 'compact' will move the menu into the side bar."),
            included: !(os_1.isOSX && environment_1.environment.electron.is())
        },
        'window.title': {
            type: 'string',
            default: os_1.isOSX
                ? '${activeEditorShort}${separator}${rootName}'
                : '${dirty} ${activeEditorShort}${separator}${rootName}${separator}${appName}',
            scope: 'application',
            markdownDescription: windowTitleDescription
        },
        'window.titleSeparator': {
            type: 'string',
            default: ' - ',
            scope: 'application',
            markdownDescription: nls_1.nls.localizeByDefault('Separator used by {0}.', '`#window.title#`')
        },
        'http.proxy': {
            type: 'string',
            pattern: '^https?://([^:]*(:[^@]*)?@)?([^:]+|\\[[:0-9a-fA-F]+\\])(:\\d+)?/?$|^$',
            markdownDescription: nls_1.nls.localizeByDefault('The proxy setting to use. If not set, will be inherited from the `http_proxy` and `https_proxy` environment variables.'),
            scope: 'application'
        },
        'http.proxyStrictSSL': {
            type: 'boolean',
            default: true,
            description: nls_1.nls.localizeByDefault('Controls whether the proxy server certificate should be verified against the list of supplied CAs.'),
            scope: 'application'
        },
        'http.proxyAuthorization': {
            type: 'string',
            markdownDescription: nls_1.nls.localizeByDefault('The value to send as the `Proxy-Authorization` header for every network request.'),
            scope: 'application'
        },
        'http.proxySupport': {
            type: 'string',
            enum: ['off', 'on', 'fallback', 'override'],
            enumDescriptions: [
                nls_1.nls.localizeByDefault('Disable proxy support for extensions.'),
                nls_1.nls.localizeByDefault('Enable proxy support for extensions.'),
                nls_1.nls.localizeByDefault('Enable proxy support for extensions, fall back to request options, when no proxy found.'),
                nls_1.nls.localizeByDefault('Enable proxy support for extensions, override request options.'),
            ],
            default: 'override',
            description: nls_1.nls.localizeByDefault('Use the proxy support for extensions.'),
            scope: 'application'
        },
        'http.systemCertificates': {
            type: 'boolean',
            default: true,
            description: nls_1.nls.localizeByDefault('Controls whether CA certificates should be loaded from the OS. (On Windows and macOS, a reload of the window is required after turning this off.)'),
            scope: 'application'
        },
        'workbench.list.openMode': {
            type: 'string',
            enum: [
                'singleClick',
                'doubleClick'
            ],
            default: 'singleClick',
            description: nls_1.nls.localizeByDefault('Controls how to open items in trees and lists using the mouse (if supported). Note that some trees and lists might choose to ignore this setting if it is not applicable.')
        },
        'workbench.editor.highlightModifiedTabs': {
            'type': 'boolean',
            'markdownDescription': nls_1.nls.localize('theia/core/highlightModifiedTabs', 'Controls whether a top border is drawn on modified (dirty) editor tabs or not.'),
            'default': false
        },
        'workbench.editor.closeOnFileDelete': {
            'type': 'boolean',
            'description': nls_1.nls.localizeByDefault('Controls whether editors showing a file that was opened during the session should close automatically when getting deleted or renamed by some other process. Disabling this will keep the editor open  on such an event. Note that deleting from within the application will always close the editor and that editors with unsaved changes will never close to preserve your data.'),
            'default': false
        },
        'workbench.editor.mouseBackForwardToNavigate': {
            'type': 'boolean',
            'description': nls_1.nls.localizeByDefault("Enables the use of mouse buttons four and five for commands 'Go Back' and 'Go Forward'."),
            'default': true
        },
        'workbench.editor.revealIfOpen': {
            'type': 'boolean',
            'description': nls_1.nls.localizeByDefault('Controls whether an editor is revealed in any of the visible groups if opened. If disabled, an editor will prefer to open in the currently active editor group. If enabled, an already opened editor will be revealed instead of opened again in the currently active editor group. Note that there are some cases where this setting is ignored, e.g. when forcing an editor to open in a specific group or to the side of the currently active group.'),
            'default': false
        },
        'workbench.commandPalette.history': {
            type: 'number',
            default: 50,
            minimum: 0,
            description: nls_1.nls.localizeByDefault('Controls the number of recently used commands to keep in history for the command palette. Set to 0 to disable command history.')
        },
        'workbench.colorTheme': {
            type: 'string',
            enum: ['dark', 'light', 'hc-theia'],
            enumItemLabels: ['Dark (Theia)', 'Light (Theia)', 'High Contrast (Theia)'],
            default: application_props_1.DefaultTheme.defaultForOSTheme(frontend_application_config_provider_1.FrontendApplicationConfigProvider.get().defaultTheme),
            description: nls_1.nls.localizeByDefault('Specifies the color theme used in the workbench.')
        },
        'workbench.iconTheme': {
            type: ['string'],
            enum: ['none', 'theia-file-icons'],
            enumItemLabels: [nls_1.nls.localizeByDefault('None'), 'File Icons (Theia)'],
            default: frontend_application_config_provider_1.FrontendApplicationConfigProvider.get().defaultIconTheme,
            description: nls_1.nls.localizeByDefault("Specifies the file icon theme used in the workbench or 'null' to not show any file icons.")
        },
        'workbench.silentNotifications': {
            type: 'boolean',
            default: false,
            description: nls_1.nls.localize('theia/core/silentNotifications', 'Controls whether to suppress notification popups.')
        },
        'workbench.statusBar.visible': {
            type: 'boolean',
            default: true,
            description: nls_1.nls.localizeByDefault('Controls the visibility of the status bar at the bottom of the workbench.')
        },
        'workbench.tree.renderIndentGuides': {
            type: 'string',
            enum: ['onHover', 'none', 'always'],
            default: 'onHover',
            description: nls_1.nls.localizeByDefault('Controls whether the tree should render indent guides.')
        },
        'workbench.hover.delay': {
            type: 'number',
            default: os_1.isOSX ? 1500 : 500,
            description: nls_1.nls.localizeByDefault('Controls the delay in milliseconds after which the hover is shown.')
        },
        'workbench.sash.hoverDelay': {
            type: 'number',
            default: 300,
            minimum: 0,
            maximum: 2000,
            description: nls_1.nls.localizeByDefault('Controls the hover feedback delay in milliseconds of the dragging area in between views/editors.')
        },
        'workbench.sash.size': {
            type: 'number',
            default: 4,
            minimum: 1,
            maximum: 20,
            description: nls_1.nls.localizeByDefault('Controls the feedback area size in pixels of the dragging area in between views/editors. Set it to a larger value if you feel it\'s hard to resize views using the mouse.')
        },
        'workbench.tab.maximize': {
            type: 'boolean',
            default: false,
            description: nls_1.nls.localize('theia/core/tabMaximize', 'Controls whether to maximize tabs on double click.')
        },
        'workbench.tab.shrinkToFit.enabled': {
            type: 'boolean',
            default: false,
            description: nls_1.nls.localize('theia/core/tabShrinkToFit', 'Shrink tabs to fit available space.')
        },
        'workbench.tab.shrinkToFit.minimumSize': {
            type: 'number',
            default: 50,
            minimum: 10,
            description: nls_1.nls.localize('theia/core/tabMinimumSize', 'Specifies the minimum size for tabs.')
        },
        'workbench.tab.shrinkToFit.defaultSize': {
            type: 'number',
            default: 200,
            minimum: 10,
            description: nls_1.nls.localize('theia/core/tabDefaultSize', 'Specifies the default size for tabs.')
        }
    }
};
exports.CorePreferenceContribution = Symbol('CorePreferenceContribution');
exports.CorePreferences = Symbol('CorePreferences');
function createCorePreferences(preferences, schema = exports.corePreferenceSchema) {
    return (0, preferences_1.createPreferenceProxy)(preferences, schema);
}
exports.createCorePreferences = createCorePreferences;
function bindCorePreferences(bind) {
    bind(exports.CorePreferences).toDynamicValue(ctx => {
        const preferences = ctx.container.get(preferences_1.PreferenceService);
        const contribution = ctx.container.get(exports.CorePreferenceContribution);
        return createCorePreferences(preferences, contribution.schema);
    }).inSingletonScope();
    bind(exports.CorePreferenceContribution).toConstantValue({ schema: exports.corePreferenceSchema });
    bind(preferences_1.PreferenceContribution).toService(exports.CorePreferenceContribution);
}
exports.bindCorePreferences = bindCorePreferences;
//# sourceMappingURL=core-preferences.js.map