"use strict";
/********************************************************************************
 * Copyright (C) 2019 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigurationModel = exports.Configuration = void 0;
const types_1 = require("../../common/types");
const cloneDeep = require("lodash.clonedeep");
/* eslint-disable @typescript-eslint/no-explicit-any */
class Configuration {
    constructor(defaultConfiguration, userConfiguration, workspaceConfiguration = new ConfigurationModel(), folderConfigurations = {}) {
        this.defaultConfiguration = defaultConfiguration;
        this.userConfiguration = userConfiguration;
        this.workspaceConfiguration = workspaceConfiguration;
        this.folderConfigurations = folderConfigurations;
        this.folderCombinedConfigs = {};
    }
    getValue(section, workspace, resource) {
        return this.getCombinedResourceConfig(workspace, resource).getValue(section);
    }
    inspect(key, workspace, resource) {
        const combinedConfiguration = this.getCombinedResourceConfig(workspace, resource);
        const folderConfiguration = this.getFolderResourceConfig(workspace, resource);
        return {
            default: this.defaultConfiguration.getValue(key),
            user: this.userConfiguration.getValue(key),
            workspace: workspace ? this.workspaceConfiguration.getValue(key) : undefined,
            workspaceFolder: folderConfiguration ? folderConfiguration.getValue(key) : undefined,
            value: combinedConfiguration.getValue(key)
        };
    }
    getCombinedResourceConfig(workspace, resource) {
        const combinedConfig = this.getCombinedConfig();
        if (!workspace || !resource) {
            return combinedConfig;
        }
        const workspaceFolder = workspace.getWorkspaceFolder(resource);
        if (!workspaceFolder) {
            return combinedConfig;
        }
        return this.getFolderCombinedConfig(workspaceFolder.uri.toString()) || combinedConfig;
    }
    getCombinedConfig() {
        if (!this.combinedConfig) {
            this.combinedConfig = this.defaultConfiguration.merge(this.userConfiguration, this.workspaceConfiguration);
        }
        return this.combinedConfig;
    }
    getFolderCombinedConfig(folder) {
        if (this.folderCombinedConfigs[folder]) {
            return this.folderCombinedConfigs[folder];
        }
        const combinedConfig = this.getCombinedConfig();
        const folderConfig = this.folderConfigurations[folder];
        if (!folderConfig) {
            return combinedConfig;
        }
        const folderCombinedConfig = combinedConfig.merge(folderConfig);
        this.folderCombinedConfigs[folder] = folderCombinedConfig;
        return folderCombinedConfig;
    }
    getFolderResourceConfig(workspace, resource) {
        if (!workspace || !resource) {
            return;
        }
        const workspaceFolder = workspace.getWorkspaceFolder(resource);
        if (!workspaceFolder) {
            return;
        }
        return this.folderConfigurations[workspaceFolder.uri.toString()];
    }
}
exports.Configuration = Configuration;
class ConfigurationModel {
    constructor(contents = Object.create(null), keys = []) {
        this.contents = contents;
        this.keys = keys;
    }
    getValue(section) {
        if (!section) {
            return this.contents;
        }
        const path = section.split('.');
        let current = this.contents;
        for (let i = 0; i < path.length; i++) {
            if (typeof current !== 'object' || current === null) {
                return undefined;
            }
            current = current[path[i]];
        }
        return current;
    }
    merge(...others) {
        const contents = cloneDeep(this.contents);
        const allKeys = [...this.keys];
        for (const other of others) {
            this.mergeContents(contents, other.contents);
            this.mergeKeys(allKeys, other.keys);
        }
        return new ConfigurationModel(contents, allKeys);
    }
    mergeContents(source, target) {
        for (const key of Object.keys(target)) {
            if (key === '__proto__') {
                continue;
            }
            if (key in source) {
                if (types_1.isObject(source[key]) && types_1.isObject(target[key])) {
                    this.mergeContents(source[key], target[key]);
                    continue;
                }
            }
            source[key] = cloneDeep(target[key]);
        }
    }
    mergeKeys(source, target) {
        for (const key of target) {
            if (source.indexOf(key) === -1) {
                source.push(key);
            }
        }
    }
}
exports.ConfigurationModel = ConfigurationModel;
//# sourceMappingURL=configuration.js.map