/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import * as dom from '../../../base/browser/dom';
import * as browser from '../../../base/browser/browser';
import { Selection } from '../../common/core/selection';
import { createFastDomNode } from '../../../base/browser/fastDomNode';
import { onUnexpectedError } from '../../../base/common/errors';
import { PointerHandler } from '../controller/pointerHandler';
import { TextAreaHandler } from '../controller/textAreaHandler';
import { ViewController } from './viewController';
import { ViewUserInputEvents } from './viewUserInputEvents';
import { ContentViewOverlays, MarginViewOverlays } from './viewOverlays';
import { PartFingerprints } from './viewPart';
import { ViewContentWidgets } from '../viewParts/contentWidgets/contentWidgets';
import { CurrentLineHighlightOverlay, CurrentLineMarginHighlightOverlay } from '../viewParts/currentLineHighlight/currentLineHighlight';
import { DecorationsOverlay } from '../viewParts/decorations/decorations';
import { EditorScrollbar } from '../viewParts/editorScrollbar/editorScrollbar';
import { GlyphMarginOverlay } from '../viewParts/glyphMargin/glyphMargin';
import { IndentGuidesOverlay } from '../viewParts/indentGuides/indentGuides';
import { LineNumbersOverlay } from '../viewParts/lineNumbers/lineNumbers';
import { ViewLines } from '../viewParts/lines/viewLines';
import { LinesDecorationsOverlay } from '../viewParts/linesDecorations/linesDecorations';
import { Margin } from '../viewParts/margin/margin';
import { MarginViewLineDecorationsOverlay } from '../viewParts/marginDecorations/marginDecorations';
import { Minimap } from '../viewParts/minimap/minimap';
import { ViewOverlayWidgets } from '../viewParts/overlayWidgets/overlayWidgets';
import { DecorationsOverviewRuler } from '../viewParts/overviewRuler/decorationsOverviewRuler';
import { OverviewRuler } from '../viewParts/overviewRuler/overviewRuler';
import { Rulers } from '../viewParts/rulers/rulers';
import { ScrollDecorationViewPart } from '../viewParts/scrollDecoration/scrollDecoration';
import { SelectionsOverlay } from '../viewParts/selections/selections';
import { ViewCursors } from '../viewParts/viewCursors/viewCursors';
import { ViewZones } from '../viewParts/viewZones/viewZones';
import { Position } from '../../common/core/position';
import { Range } from '../../common/core/range';
import { RenderingContext } from '../../common/view/renderingContext';
import { ViewContext } from '../../common/view/viewContext';
import { ViewportData } from '../../common/viewLayout/viewLinesViewportData';
import { ViewEventHandler } from '../../common/viewModel/viewEventHandler';
import { getThemeTypeSelector } from '../../../platform/theme/common/themeService';
import { PointerHandlerLastRenderData } from '../controller/mouseTarget';
var View = /** @class */ (function (_super) {
    __extends(View, _super);
    function View(commandDelegate, configuration, themeService, model, userInputEvents, overflowWidgetsDomNode) {
        var _this = _super.call(this) || this;
        _this._selections = [new Selection(1, 1, 1, 1)];
        _this._renderAnimationFrame = null;
        var viewController = new ViewController(configuration, model, userInputEvents, commandDelegate);
        // The view context is passed on to most classes (basically to reduce param. counts in ctors)
        _this._context = new ViewContext(configuration, themeService.getColorTheme(), model);
        _this._configPixelRatio = _this._configPixelRatio = _this._context.configuration.options.get(122 /* pixelRatio */);
        // Ensure the view is the first event handler in order to update the layout
        _this._context.addEventHandler(_this);
        _this._register(themeService.onDidColorThemeChange(function (theme) {
            _this._context.theme.update(theme);
            _this._context.model.onDidColorThemeChange();
            _this.render(true, false);
        }));
        _this._viewParts = [];
        // Keyboard handler
        _this._textAreaHandler = new TextAreaHandler(_this._context, viewController, _this._createTextAreaHandlerHelper());
        _this._viewParts.push(_this._textAreaHandler);
        // These two dom nodes must be constructed up front, since references are needed in the layout provider (scrolling & co.)
        _this._linesContent = createFastDomNode(document.createElement('div'));
        _this._linesContent.setClassName('lines-content' + ' monaco-editor-background');
        _this._linesContent.setPosition('absolute');
        _this.domNode = createFastDomNode(document.createElement('div'));
        _this.domNode.setClassName(_this._getEditorClassName());
        // Set role 'code' for better screen reader support https://github.com/microsoft/vscode/issues/93438
        _this.domNode.setAttribute('role', 'code');
        _this._overflowGuardContainer = createFastDomNode(document.createElement('div'));
        PartFingerprints.write(_this._overflowGuardContainer, 3 /* OverflowGuard */);
        _this._overflowGuardContainer.setClassName('overflow-guard');
        _this._scrollbar = new EditorScrollbar(_this._context, _this._linesContent, _this.domNode, _this._overflowGuardContainer);
        _this._viewParts.push(_this._scrollbar);
        // View Lines
        _this._viewLines = new ViewLines(_this._context, _this._linesContent);
        // View Zones
        _this._viewZones = new ViewZones(_this._context);
        _this._viewParts.push(_this._viewZones);
        // Decorations overview ruler
        var decorationsOverviewRuler = new DecorationsOverviewRuler(_this._context);
        _this._viewParts.push(decorationsOverviewRuler);
        var scrollDecoration = new ScrollDecorationViewPart(_this._context);
        _this._viewParts.push(scrollDecoration);
        var contentViewOverlays = new ContentViewOverlays(_this._context);
        _this._viewParts.push(contentViewOverlays);
        contentViewOverlays.addDynamicOverlay(new CurrentLineHighlightOverlay(_this._context));
        contentViewOverlays.addDynamicOverlay(new SelectionsOverlay(_this._context));
        contentViewOverlays.addDynamicOverlay(new IndentGuidesOverlay(_this._context));
        contentViewOverlays.addDynamicOverlay(new DecorationsOverlay(_this._context));
        var marginViewOverlays = new MarginViewOverlays(_this._context);
        _this._viewParts.push(marginViewOverlays);
        marginViewOverlays.addDynamicOverlay(new CurrentLineMarginHighlightOverlay(_this._context));
        marginViewOverlays.addDynamicOverlay(new GlyphMarginOverlay(_this._context));
        marginViewOverlays.addDynamicOverlay(new MarginViewLineDecorationsOverlay(_this._context));
        marginViewOverlays.addDynamicOverlay(new LinesDecorationsOverlay(_this._context));
        marginViewOverlays.addDynamicOverlay(new LineNumbersOverlay(_this._context));
        var margin = new Margin(_this._context);
        margin.getDomNode().appendChild(_this._viewZones.marginDomNode);
        margin.getDomNode().appendChild(marginViewOverlays.getDomNode());
        _this._viewParts.push(margin);
        // Content widgets
        _this._contentWidgets = new ViewContentWidgets(_this._context, _this.domNode);
        _this._viewParts.push(_this._contentWidgets);
        _this._viewCursors = new ViewCursors(_this._context);
        _this._viewParts.push(_this._viewCursors);
        // Overlay widgets
        _this._overlayWidgets = new ViewOverlayWidgets(_this._context);
        _this._viewParts.push(_this._overlayWidgets);
        var rulers = new Rulers(_this._context);
        _this._viewParts.push(rulers);
        var minimap = new Minimap(_this._context);
        _this._viewParts.push(minimap);
        // -------------- Wire dom nodes up
        if (decorationsOverviewRuler) {
            var overviewRulerData = _this._scrollbar.getOverviewRulerLayoutInfo();
            overviewRulerData.parent.insertBefore(decorationsOverviewRuler.getDomNode(), overviewRulerData.insertBefore);
        }
        _this._linesContent.appendChild(contentViewOverlays.getDomNode());
        _this._linesContent.appendChild(rulers.domNode);
        _this._linesContent.appendChild(_this._viewZones.domNode);
        _this._linesContent.appendChild(_this._viewLines.getDomNode());
        _this._linesContent.appendChild(_this._contentWidgets.domNode);
        _this._linesContent.appendChild(_this._viewCursors.getDomNode());
        _this._overflowGuardContainer.appendChild(margin.getDomNode());
        _this._overflowGuardContainer.appendChild(_this._scrollbar.getDomNode());
        _this._overflowGuardContainer.appendChild(scrollDecoration.getDomNode());
        _this._overflowGuardContainer.appendChild(_this._textAreaHandler.textArea);
        _this._overflowGuardContainer.appendChild(_this._textAreaHandler.textAreaCover);
        _this._overflowGuardContainer.appendChild(_this._overlayWidgets.getDomNode());
        _this._overflowGuardContainer.appendChild(minimap.getDomNode());
        _this.domNode.appendChild(_this._overflowGuardContainer);
        if (overflowWidgetsDomNode) {
            overflowWidgetsDomNode.appendChild(_this._contentWidgets.overflowingContentWidgetsDomNode.domNode);
        }
        else {
            _this.domNode.appendChild(_this._contentWidgets.overflowingContentWidgetsDomNode);
        }
        _this._applyLayout();
        // Pointer handler
        _this._pointerHandler = _this._register(new PointerHandler(_this._context, viewController, _this._createPointerHandlerHelper()));
        return _this;
    }
    View.prototype._flushAccumulatedAndRenderNow = function () {
        this._renderNow();
    };
    View.prototype._createPointerHandlerHelper = function () {
        var _this = this;
        return {
            viewDomNode: this.domNode.domNode,
            linesContentDomNode: this._linesContent.domNode,
            focusTextArea: function () {
                _this.focus();
            },
            dispatchTextAreaEvent: function (event) {
                _this._textAreaHandler.textArea.domNode.dispatchEvent(event);
            },
            getLastRenderData: function () {
                var lastViewCursorsRenderData = _this._viewCursors.getLastRenderData() || [];
                var lastTextareaPosition = _this._textAreaHandler.getLastRenderData();
                return new PointerHandlerLastRenderData(lastViewCursorsRenderData, lastTextareaPosition);
            },
            shouldSuppressMouseDownOnViewZone: function (viewZoneId) {
                return _this._viewZones.shouldSuppressMouseDownOnViewZone(viewZoneId);
            },
            shouldSuppressMouseDownOnWidget: function (widgetId) {
                return _this._contentWidgets.shouldSuppressMouseDownOnWidget(widgetId);
            },
            getPositionFromDOMInfo: function (spanNode, offset) {
                _this._flushAccumulatedAndRenderNow();
                return _this._viewLines.getPositionFromDOMInfo(spanNode, offset);
            },
            visibleRangeForPosition: function (lineNumber, column) {
                _this._flushAccumulatedAndRenderNow();
                return _this._viewLines.visibleRangeForPosition(new Position(lineNumber, column));
            },
            getLineWidth: function (lineNumber) {
                _this._flushAccumulatedAndRenderNow();
                return _this._viewLines.getLineWidth(lineNumber);
            }
        };
    };
    View.prototype._createTextAreaHandlerHelper = function () {
        var _this = this;
        return {
            visibleRangeForPositionRelativeToEditor: function (lineNumber, column) {
                _this._flushAccumulatedAndRenderNow();
                return _this._viewLines.visibleRangeForPosition(new Position(lineNumber, column));
            }
        };
    };
    View.prototype._applyLayout = function () {
        var options = this._context.configuration.options;
        var layoutInfo = options.get(124 /* layoutInfo */);
        this.domNode.setWidth(layoutInfo.width);
        this.domNode.setHeight(layoutInfo.height);
        this._overflowGuardContainer.setWidth(layoutInfo.width);
        this._overflowGuardContainer.setHeight(layoutInfo.height);
        this._linesContent.setWidth(1000000);
        this._linesContent.setHeight(1000000);
    };
    View.prototype._getEditorClassName = function () {
        var focused = this._textAreaHandler.isFocused() ? ' focused' : '';
        return this._context.configuration.options.get(121 /* editorClassName */) + ' ' + getThemeTypeSelector(this._context.theme.type) + focused;
    };
    // --- begin event handlers
    View.prototype.handleEvents = function (events) {
        _super.prototype.handleEvents.call(this, events);
        this._scheduleRender();
    };
    View.prototype.onConfigurationChanged = function (e) {
        this._configPixelRatio = this._context.configuration.options.get(122 /* pixelRatio */);
        this.domNode.setClassName(this._getEditorClassName());
        this._applyLayout();
        return false;
    };
    View.prototype.onCursorStateChanged = function (e) {
        this._selections = e.selections;
        return false;
    };
    View.prototype.onFocusChanged = function (e) {
        this.domNode.setClassName(this._getEditorClassName());
        return false;
    };
    View.prototype.onThemeChanged = function (e) {
        this.domNode.setClassName(this._getEditorClassName());
        return false;
    };
    // --- end event handlers
    View.prototype.dispose = function () {
        var e_1, _a;
        if (this._renderAnimationFrame !== null) {
            this._renderAnimationFrame.dispose();
            this._renderAnimationFrame = null;
        }
        this._contentWidgets.overflowingContentWidgetsDomNode.domNode.remove();
        this._context.removeEventHandler(this);
        this._viewLines.dispose();
        try {
            // Destroy view parts
            for (var _b = __values(this._viewParts), _c = _b.next(); !_c.done; _c = _b.next()) {
                var viewPart = _c.value;
                viewPart.dispose();
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_1) throw e_1.error; }
        }
        _super.prototype.dispose.call(this);
    };
    View.prototype._scheduleRender = function () {
        if (this._renderAnimationFrame === null) {
            this._renderAnimationFrame = dom.runAtThisOrScheduleAtNextAnimationFrame(this._onRenderScheduled.bind(this), 100);
        }
    };
    View.prototype._onRenderScheduled = function () {
        this._renderAnimationFrame = null;
        this._flushAccumulatedAndRenderNow();
    };
    View.prototype._renderNow = function () {
        var _this = this;
        safeInvokeNoArg(function () { return _this._actualRender(); });
    };
    View.prototype._getViewPartsToRender = function () {
        var e_2, _a;
        var result = [], resultLen = 0;
        try {
            for (var _b = __values(this._viewParts), _c = _b.next(); !_c.done; _c = _b.next()) {
                var viewPart = _c.value;
                if (viewPart.shouldRender()) {
                    result[resultLen++] = viewPart;
                }
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_2) throw e_2.error; }
        }
        return result;
    };
    View.prototype._actualRender = function () {
        var e_3, _a, e_4, _b;
        if (!dom.isInDOM(this.domNode.domNode)) {
            return;
        }
        var viewPartsToRender = this._getViewPartsToRender();
        if (!this._viewLines.shouldRender() && viewPartsToRender.length === 0) {
            // Nothing to render
            return;
        }
        var partialViewportData = this._context.viewLayout.getLinesViewportData();
        this._context.model.setViewport(partialViewportData.startLineNumber, partialViewportData.endLineNumber, partialViewportData.centeredLineNumber);
        var viewportData = new ViewportData(this._selections, partialViewportData, this._context.viewLayout.getWhitespaceViewportData(), this._context.model);
        if (this._contentWidgets.shouldRender()) {
            // Give the content widgets a chance to set their max width before a possible synchronous layout
            this._contentWidgets.onBeforeRender(viewportData);
        }
        if (this._viewLines.shouldRender()) {
            this._viewLines.renderText(viewportData);
            this._viewLines.onDidRender();
            // Rendering of viewLines might cause scroll events to occur, so collect view parts to render again
            viewPartsToRender = this._getViewPartsToRender();
        }
        var renderingContext = new RenderingContext(this._context.viewLayout, viewportData, this._viewLines);
        try {
            // Render the rest of the parts
            for (var viewPartsToRender_1 = __values(viewPartsToRender), viewPartsToRender_1_1 = viewPartsToRender_1.next(); !viewPartsToRender_1_1.done; viewPartsToRender_1_1 = viewPartsToRender_1.next()) {
                var viewPart = viewPartsToRender_1_1.value;
                viewPart.prepareRender(renderingContext);
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (viewPartsToRender_1_1 && !viewPartsToRender_1_1.done && (_a = viewPartsToRender_1.return)) _a.call(viewPartsToRender_1);
            }
            finally { if (e_3) throw e_3.error; }
        }
        try {
            for (var viewPartsToRender_2 = __values(viewPartsToRender), viewPartsToRender_2_1 = viewPartsToRender_2.next(); !viewPartsToRender_2_1.done; viewPartsToRender_2_1 = viewPartsToRender_2.next()) {
                var viewPart = viewPartsToRender_2_1.value;
                viewPart.render(renderingContext);
                viewPart.onDidRender();
            }
        }
        catch (e_4_1) { e_4 = { error: e_4_1 }; }
        finally {
            try {
                if (viewPartsToRender_2_1 && !viewPartsToRender_2_1.done && (_b = viewPartsToRender_2.return)) _b.call(viewPartsToRender_2);
            }
            finally { if (e_4) throw e_4.error; }
        }
        // Try to detect browser zooming and paint again if necessary
        if (Math.abs(browser.getPixelRatio() - this._configPixelRatio) > 0.001) {
            // looks like the pixel ratio has changed
            this._context.configuration.updatePixelRatio();
        }
    };
    // --- BEGIN CodeEditor helpers
    View.prototype.delegateVerticalScrollbarMouseDown = function (browserEvent) {
        this._scrollbar.delegateVerticalScrollbarMouseDown(browserEvent);
    };
    View.prototype.restoreState = function (scrollPosition) {
        this._context.model.setScrollPosition({ scrollTop: scrollPosition.scrollTop }, 1 /* Immediate */);
        this._context.model.tokenizeViewport();
        this._renderNow();
        this._viewLines.updateLineWidths();
        this._context.model.setScrollPosition({ scrollLeft: scrollPosition.scrollLeft }, 1 /* Immediate */);
    };
    View.prototype.getOffsetForColumn = function (modelLineNumber, modelColumn) {
        var modelPosition = this._context.model.validateModelPosition({
            lineNumber: modelLineNumber,
            column: modelColumn
        });
        var viewPosition = this._context.model.coordinatesConverter.convertModelPositionToViewPosition(modelPosition);
        this._flushAccumulatedAndRenderNow();
        var visibleRange = this._viewLines.visibleRangeForPosition(new Position(viewPosition.lineNumber, viewPosition.column));
        if (!visibleRange) {
            return -1;
        }
        return visibleRange.left;
    };
    View.prototype.getTargetAtClientPoint = function (clientX, clientY) {
        var mouseTarget = this._pointerHandler.getTargetAtClientPoint(clientX, clientY);
        if (!mouseTarget) {
            return null;
        }
        return ViewUserInputEvents.convertViewToModelMouseTarget(mouseTarget, this._context.model.coordinatesConverter);
    };
    View.prototype.createOverviewRuler = function (cssClassName) {
        return new OverviewRuler(this._context, cssClassName);
    };
    View.prototype.change = function (callback) {
        this._viewZones.changeViewZones(callback);
        this._scheduleRender();
    };
    View.prototype.render = function (now, everything) {
        var e_5, _a;
        if (everything) {
            // Force everything to render...
            this._viewLines.forceShouldRender();
            try {
                for (var _b = __values(this._viewParts), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var viewPart = _c.value;
                    viewPart.forceShouldRender();
                }
            }
            catch (e_5_1) { e_5 = { error: e_5_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_5) throw e_5.error; }
            }
        }
        if (now) {
            this._flushAccumulatedAndRenderNow();
        }
        else {
            this._scheduleRender();
        }
    };
    View.prototype.focus = function () {
        this._textAreaHandler.focusTextArea();
    };
    View.prototype.isFocused = function () {
        return this._textAreaHandler.isFocused();
    };
    View.prototype.refreshFocusState = function () {
        this._textAreaHandler.refreshFocusState();
    };
    View.prototype.setAriaOptions = function (options) {
        this._textAreaHandler.setAriaOptions(options);
    };
    View.prototype.addContentWidget = function (widgetData) {
        this._contentWidgets.addWidget(widgetData.widget);
        this.layoutContentWidget(widgetData);
        this._scheduleRender();
    };
    View.prototype.layoutContentWidget = function (widgetData) {
        var newRange = widgetData.position ? widgetData.position.range || null : null;
        if (newRange === null) {
            var newPosition = widgetData.position ? widgetData.position.position : null;
            if (newPosition !== null) {
                newRange = new Range(newPosition.lineNumber, newPosition.column, newPosition.lineNumber, newPosition.column);
            }
        }
        var newPreference = widgetData.position ? widgetData.position.preference : null;
        this._contentWidgets.setWidgetPosition(widgetData.widget, newRange, newPreference);
        this._scheduleRender();
    };
    View.prototype.removeContentWidget = function (widgetData) {
        this._contentWidgets.removeWidget(widgetData.widget);
        this._scheduleRender();
    };
    View.prototype.addOverlayWidget = function (widgetData) {
        this._overlayWidgets.addWidget(widgetData.widget);
        this.layoutOverlayWidget(widgetData);
        this._scheduleRender();
    };
    View.prototype.layoutOverlayWidget = function (widgetData) {
        var newPreference = widgetData.position ? widgetData.position.preference : null;
        var shouldRender = this._overlayWidgets.setWidgetPosition(widgetData.widget, newPreference);
        if (shouldRender) {
            this._scheduleRender();
        }
    };
    View.prototype.removeOverlayWidget = function (widgetData) {
        this._overlayWidgets.removeWidget(widgetData.widget);
        this._scheduleRender();
    };
    return View;
}(ViewEventHandler));
export { View };
function safeInvokeNoArg(func) {
    try {
        return func();
    }
    catch (e) {
        onUnexpectedError(e);
    }
}
