/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { isFalsyOrEmpty, isNonEmptyArray } from '../../../base/common/arrays';
import { Schemas } from '../../../base/common/network';
import { URI } from '../../../base/common/uri';
import { Event, Emitter } from '../../../base/common/event';
import { MarkerSeverity } from './markers';
import { ResourceMap } from '../../../base/common/map';
import { Iterable } from '../../../base/common/iterator';
var DoubleResourceMap = /** @class */ (function () {
    function DoubleResourceMap() {
        this._byResource = new ResourceMap();
        this._byOwner = new Map();
    }
    DoubleResourceMap.prototype.set = function (resource, owner, value) {
        var ownerMap = this._byResource.get(resource);
        if (!ownerMap) {
            ownerMap = new Map();
            this._byResource.set(resource, ownerMap);
        }
        ownerMap.set(owner, value);
        var resourceMap = this._byOwner.get(owner);
        if (!resourceMap) {
            resourceMap = new ResourceMap();
            this._byOwner.set(owner, resourceMap);
        }
        resourceMap.set(resource, value);
    };
    DoubleResourceMap.prototype.get = function (resource, owner) {
        var ownerMap = this._byResource.get(resource);
        return ownerMap === null || ownerMap === void 0 ? void 0 : ownerMap.get(owner);
    };
    DoubleResourceMap.prototype.delete = function (resource, owner) {
        var removedA = false;
        var removedB = false;
        var ownerMap = this._byResource.get(resource);
        if (ownerMap) {
            removedA = ownerMap.delete(owner);
        }
        var resourceMap = this._byOwner.get(owner);
        if (resourceMap) {
            removedB = resourceMap.delete(resource);
        }
        if (removedA !== removedB) {
            throw new Error('illegal state');
        }
        return removedA && removedB;
    };
    DoubleResourceMap.prototype.values = function (key) {
        var _a, _b, _c, _d;
        if (typeof key === 'string') {
            return (_b = (_a = this._byOwner.get(key)) === null || _a === void 0 ? void 0 : _a.values()) !== null && _b !== void 0 ? _b : Iterable.empty();
        }
        if (URI.isUri(key)) {
            return (_d = (_c = this._byResource.get(key)) === null || _c === void 0 ? void 0 : _c.values()) !== null && _d !== void 0 ? _d : Iterable.empty();
        }
        return Iterable.map(Iterable.concat.apply(Iterable, __spreadArray([], __read(this._byOwner.values()))), function (map) { return map[1]; });
    };
    return DoubleResourceMap;
}());
var MarkerStats = /** @class */ (function () {
    function MarkerStats(service) {
        this.errors = 0;
        this.infos = 0;
        this.warnings = 0;
        this.unknowns = 0;
        this._data = new ResourceMap();
        this._service = service;
        this._subscription = service.onMarkerChanged(this._update, this);
    }
    MarkerStats.prototype.dispose = function () {
        this._subscription.dispose();
    };
    MarkerStats.prototype._update = function (resources) {
        var e_1, _a;
        try {
            for (var resources_1 = __values(resources), resources_1_1 = resources_1.next(); !resources_1_1.done; resources_1_1 = resources_1.next()) {
                var resource = resources_1_1.value;
                var oldStats = this._data.get(resource);
                if (oldStats) {
                    this._substract(oldStats);
                }
                var newStats = this._resourceStats(resource);
                this._add(newStats);
                this._data.set(resource, newStats);
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (resources_1_1 && !resources_1_1.done && (_a = resources_1.return)) _a.call(resources_1);
            }
            finally { if (e_1) throw e_1.error; }
        }
    };
    MarkerStats.prototype._resourceStats = function (resource) {
        var e_2, _a;
        var result = { errors: 0, warnings: 0, infos: 0, unknowns: 0 };
        // TODO this is a hack
        if (resource.scheme === Schemas.inMemory || resource.scheme === Schemas.walkThrough || resource.scheme === Schemas.walkThroughSnippet) {
            return result;
        }
        try {
            for (var _b = __values(this._service.read({ resource: resource })), _c = _b.next(); !_c.done; _c = _b.next()) {
                var severity = _c.value.severity;
                if (severity === MarkerSeverity.Error) {
                    result.errors += 1;
                }
                else if (severity === MarkerSeverity.Warning) {
                    result.warnings += 1;
                }
                else if (severity === MarkerSeverity.Info) {
                    result.infos += 1;
                }
                else {
                    result.unknowns += 1;
                }
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_2) throw e_2.error; }
        }
        return result;
    };
    MarkerStats.prototype._substract = function (op) {
        this.errors -= op.errors;
        this.warnings -= op.warnings;
        this.infos -= op.infos;
        this.unknowns -= op.unknowns;
    };
    MarkerStats.prototype._add = function (op) {
        this.errors += op.errors;
        this.warnings += op.warnings;
        this.infos += op.infos;
        this.unknowns += op.unknowns;
    };
    return MarkerStats;
}());
var MarkerService = /** @class */ (function () {
    function MarkerService() {
        this._onMarkerChanged = new Emitter();
        this.onMarkerChanged = Event.debounce(this._onMarkerChanged.event, MarkerService._debouncer, 0);
        this._data = new DoubleResourceMap();
        this._stats = new MarkerStats(this);
    }
    MarkerService.prototype.dispose = function () {
        this._stats.dispose();
        this._onMarkerChanged.dispose();
    };
    MarkerService.prototype.getStatistics = function () {
        return this._stats;
    };
    MarkerService.prototype.remove = function (owner, resources) {
        var e_3, _a;
        try {
            for (var _b = __values(resources || []), _c = _b.next(); !_c.done; _c = _b.next()) {
                var resource = _c.value;
                this.changeOne(owner, resource, []);
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_3) throw e_3.error; }
        }
    };
    MarkerService.prototype.changeOne = function (owner, resource, markerData) {
        var e_4, _a;
        if (isFalsyOrEmpty(markerData)) {
            // remove marker for this (owner,resource)-tuple
            var removed = this._data.delete(resource, owner);
            if (removed) {
                this._onMarkerChanged.fire([resource]);
            }
        }
        else {
            // insert marker for this (owner,resource)-tuple
            var markers = [];
            try {
                for (var markerData_1 = __values(markerData), markerData_1_1 = markerData_1.next(); !markerData_1_1.done; markerData_1_1 = markerData_1.next()) {
                    var data = markerData_1_1.value;
                    var marker = MarkerService._toMarker(owner, resource, data);
                    if (marker) {
                        markers.push(marker);
                    }
                }
            }
            catch (e_4_1) { e_4 = { error: e_4_1 }; }
            finally {
                try {
                    if (markerData_1_1 && !markerData_1_1.done && (_a = markerData_1.return)) _a.call(markerData_1);
                }
                finally { if (e_4) throw e_4.error; }
            }
            this._data.set(resource, owner, markers);
            this._onMarkerChanged.fire([resource]);
        }
    };
    MarkerService._toMarker = function (owner, resource, data) {
        var code = data.code, severity = data.severity, message = data.message, source = data.source, startLineNumber = data.startLineNumber, startColumn = data.startColumn, endLineNumber = data.endLineNumber, endColumn = data.endColumn, relatedInformation = data.relatedInformation, tags = data.tags;
        if (!message) {
            return undefined;
        }
        // santize data
        startLineNumber = startLineNumber > 0 ? startLineNumber : 1;
        startColumn = startColumn > 0 ? startColumn : 1;
        endLineNumber = endLineNumber >= startLineNumber ? endLineNumber : startLineNumber;
        endColumn = endColumn > 0 ? endColumn : startColumn;
        return {
            resource: resource,
            owner: owner,
            code: code,
            severity: severity,
            message: message,
            source: source,
            startLineNumber: startLineNumber,
            startColumn: startColumn,
            endLineNumber: endLineNumber,
            endColumn: endColumn,
            relatedInformation: relatedInformation,
            tags: tags,
        };
    };
    MarkerService.prototype.changeAll = function (owner, data) {
        var e_5, _a, e_6, _b, e_7, _c;
        var changes = [];
        // remove old marker
        var existing = this._data.values(owner);
        if (existing) {
            try {
                for (var existing_1 = __values(existing), existing_1_1 = existing_1.next(); !existing_1_1.done; existing_1_1 = existing_1.next()) {
                    var data_2 = existing_1_1.value;
                    var first = Iterable.first(data_2);
                    if (first) {
                        changes.push(first.resource);
                        this._data.delete(first.resource, owner);
                    }
                }
            }
            catch (e_5_1) { e_5 = { error: e_5_1 }; }
            finally {
                try {
                    if (existing_1_1 && !existing_1_1.done && (_a = existing_1.return)) _a.call(existing_1);
                }
                finally { if (e_5) throw e_5.error; }
            }
        }
        // add new markers
        if (isNonEmptyArray(data)) {
            // group by resource
            var groups = new ResourceMap();
            try {
                for (var data_1 = __values(data), data_1_1 = data_1.next(); !data_1_1.done; data_1_1 = data_1.next()) {
                    var _d = data_1_1.value, resource = _d.resource, markerData = _d.marker;
                    var marker = MarkerService._toMarker(owner, resource, markerData);
                    if (!marker) {
                        // filter bad markers
                        continue;
                    }
                    var array = groups.get(resource);
                    if (!array) {
                        groups.set(resource, [marker]);
                        changes.push(resource);
                    }
                    else {
                        array.push(marker);
                    }
                }
            }
            catch (e_6_1) { e_6 = { error: e_6_1 }; }
            finally {
                try {
                    if (data_1_1 && !data_1_1.done && (_b = data_1.return)) _b.call(data_1);
                }
                finally { if (e_6) throw e_6.error; }
            }
            try {
                // insert all
                for (var groups_1 = __values(groups), groups_1_1 = groups_1.next(); !groups_1_1.done; groups_1_1 = groups_1.next()) {
                    var _e = __read(groups_1_1.value, 2), resource = _e[0], value = _e[1];
                    this._data.set(resource, owner, value);
                }
            }
            catch (e_7_1) { e_7 = { error: e_7_1 }; }
            finally {
                try {
                    if (groups_1_1 && !groups_1_1.done && (_c = groups_1.return)) _c.call(groups_1);
                }
                finally { if (e_7) throw e_7.error; }
            }
        }
        if (changes.length > 0) {
            this._onMarkerChanged.fire(changes);
        }
    };
    MarkerService.prototype.read = function (filter) {
        var e_8, _a, e_9, _b, e_10, _c, e_11, _d, e_12, _e;
        if (filter === void 0) { filter = Object.create(null); }
        var owner = filter.owner, resource = filter.resource, severities = filter.severities, take = filter.take;
        if (!take || take < 0) {
            take = -1;
        }
        if (owner && resource) {
            // exactly one owner AND resource
            var data = this._data.get(resource, owner);
            if (!data) {
                return [];
            }
            else {
                var result = [];
                try {
                    for (var data_3 = __values(data), data_3_1 = data_3.next(); !data_3_1.done; data_3_1 = data_3.next()) {
                        var marker = data_3_1.value;
                        if (MarkerService._accept(marker, severities)) {
                            var newLen = result.push(marker);
                            if (take > 0 && newLen === take) {
                                break;
                            }
                        }
                    }
                }
                catch (e_8_1) { e_8 = { error: e_8_1 }; }
                finally {
                    try {
                        if (data_3_1 && !data_3_1.done && (_a = data_3.return)) _a.call(data_3);
                    }
                    finally { if (e_8) throw e_8.error; }
                }
                return result;
            }
        }
        else if (!owner && !resource) {
            // all
            var result = [];
            try {
                for (var _f = __values(this._data.values()), _g = _f.next(); !_g.done; _g = _f.next()) {
                    var markers = _g.value;
                    try {
                        for (var markers_1 = (e_10 = void 0, __values(markers)), markers_1_1 = markers_1.next(); !markers_1_1.done; markers_1_1 = markers_1.next()) {
                            var data = markers_1_1.value;
                            if (MarkerService._accept(data, severities)) {
                                var newLen = result.push(data);
                                if (take > 0 && newLen === take) {
                                    return result;
                                }
                            }
                        }
                    }
                    catch (e_10_1) { e_10 = { error: e_10_1 }; }
                    finally {
                        try {
                            if (markers_1_1 && !markers_1_1.done && (_c = markers_1.return)) _c.call(markers_1);
                        }
                        finally { if (e_10) throw e_10.error; }
                    }
                }
            }
            catch (e_9_1) { e_9 = { error: e_9_1 }; }
            finally {
                try {
                    if (_g && !_g.done && (_b = _f.return)) _b.call(_f);
                }
                finally { if (e_9) throw e_9.error; }
            }
            return result;
        }
        else {
            // of one resource OR owner
            var iterable = this._data.values(resource !== null && resource !== void 0 ? resource : owner);
            var result = [];
            try {
                for (var iterable_1 = __values(iterable), iterable_1_1 = iterable_1.next(); !iterable_1_1.done; iterable_1_1 = iterable_1.next()) {
                    var markers = iterable_1_1.value;
                    try {
                        for (var markers_2 = (e_12 = void 0, __values(markers)), markers_2_1 = markers_2.next(); !markers_2_1.done; markers_2_1 = markers_2.next()) {
                            var data = markers_2_1.value;
                            if (MarkerService._accept(data, severities)) {
                                var newLen = result.push(data);
                                if (take > 0 && newLen === take) {
                                    return result;
                                }
                            }
                        }
                    }
                    catch (e_12_1) { e_12 = { error: e_12_1 }; }
                    finally {
                        try {
                            if (markers_2_1 && !markers_2_1.done && (_e = markers_2.return)) _e.call(markers_2);
                        }
                        finally { if (e_12) throw e_12.error; }
                    }
                }
            }
            catch (e_11_1) { e_11 = { error: e_11_1 }; }
            finally {
                try {
                    if (iterable_1_1 && !iterable_1_1.done && (_d = iterable_1.return)) _d.call(iterable_1);
                }
                finally { if (e_11) throw e_11.error; }
            }
            return result;
        }
    };
    MarkerService._accept = function (marker, severities) {
        return severities === undefined || (severities & marker.severity) === marker.severity;
    };
    MarkerService._debouncer = function (last, event) {
        var e_13, _a;
        if (!last) {
            MarkerService._dedupeMap = new ResourceMap();
            last = [];
        }
        try {
            for (var event_1 = __values(event), event_1_1 = event_1.next(); !event_1_1.done; event_1_1 = event_1.next()) {
                var uri = event_1_1.value;
                if (!MarkerService._dedupeMap.has(uri)) {
                    MarkerService._dedupeMap.set(uri, true);
                    last.push(uri);
                }
            }
        }
        catch (e_13_1) { e_13 = { error: e_13_1 }; }
        finally {
            try {
                if (event_1_1 && !event_1_1.done && (_a = event_1.return)) _a.call(event_1);
            }
            finally { if (e_13) throw e_13.error; }
        }
        return last;
    };
    return MarkerService;
}());
export { MarkerService };
