/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
import './progressbar.css';
import { Disposable } from '../../../common/lifecycle';
import { Color } from '../../../common/color';
import { mixin } from '../../../common/objects';
import { hide, show } from '../../dom';
import { RunOnceScheduler } from '../../../common/async';
import { isNumber } from '../../../common/types';
var CSS_DONE = 'done';
var CSS_ACTIVE = 'active';
var CSS_INFINITE = 'infinite';
var CSS_DISCRETE = 'discrete';
var defaultOpts = {
    progressBarBackground: Color.fromHex('#0E70C0')
};
/**
 * A progress bar with support for infinite or discrete progress.
 */
var ProgressBar = /** @class */ (function (_super) {
    __extends(ProgressBar, _super);
    function ProgressBar(container, options) {
        var _this = _super.call(this) || this;
        _this.options = options || Object.create(null);
        mixin(_this.options, defaultOpts, false);
        _this.workedVal = 0;
        _this.progressBarBackground = _this.options.progressBarBackground;
        _this._register(_this.showDelayedScheduler = new RunOnceScheduler(function () { return show(_this.element); }, 0));
        _this.create(container);
        return _this;
    }
    ProgressBar.prototype.create = function (container) {
        this.element = document.createElement('div');
        this.element.classList.add('monaco-progress-container');
        this.element.setAttribute('role', 'progressbar');
        this.element.setAttribute('aria-valuemin', '0');
        container.appendChild(this.element);
        this.bit = document.createElement('div');
        this.bit.classList.add('progress-bit');
        this.element.appendChild(this.bit);
        this.applyStyles();
    };
    ProgressBar.prototype.off = function () {
        this.bit.style.width = 'inherit';
        this.bit.style.opacity = '1';
        this.element.classList.remove(CSS_ACTIVE, CSS_INFINITE, CSS_DISCRETE);
        this.workedVal = 0;
        this.totalWork = undefined;
    };
    /**
     * Indicates to the progress bar that all work is done.
     */
    ProgressBar.prototype.done = function () {
        return this.doDone(true);
    };
    /**
     * Stops the progressbar from showing any progress instantly without fading out.
     */
    ProgressBar.prototype.stop = function () {
        return this.doDone(false);
    };
    ProgressBar.prototype.doDone = function (delayed) {
        var _this = this;
        this.element.classList.add(CSS_DONE);
        // let it grow to 100% width and hide afterwards
        if (!this.element.classList.contains(CSS_INFINITE)) {
            this.bit.style.width = 'inherit';
            if (delayed) {
                setTimeout(function () { return _this.off(); }, 200);
            }
            else {
                this.off();
            }
        }
        // let it fade out and hide afterwards
        else {
            this.bit.style.opacity = '0';
            if (delayed) {
                setTimeout(function () { return _this.off(); }, 200);
            }
            else {
                this.off();
            }
        }
        return this;
    };
    /**
     * Use this mode to indicate progress that has no total number of work units.
     */
    ProgressBar.prototype.infinite = function () {
        this.bit.style.width = '2%';
        this.bit.style.opacity = '1';
        this.element.classList.remove(CSS_DISCRETE, CSS_DONE);
        this.element.classList.add(CSS_ACTIVE, CSS_INFINITE);
        return this;
    };
    /**
     * Tells the progress bar the total number of work. Use in combination with workedVal() to let
     * the progress bar show the actual progress based on the work that is done.
     */
    ProgressBar.prototype.total = function (value) {
        this.workedVal = 0;
        this.totalWork = value;
        this.element.setAttribute('aria-valuemax', value.toString());
        return this;
    };
    /**
     * Finds out if this progress bar is configured with total work
     */
    ProgressBar.prototype.hasTotal = function () {
        return isNumber(this.totalWork);
    };
    /**
     * Tells the progress bar that an increment of work has been completed.
     */
    ProgressBar.prototype.worked = function (value) {
        value = Math.max(1, Number(value));
        return this.doSetWorked(this.workedVal + value);
    };
    /**
     * Tells the progress bar the total amount of work that has been completed.
     */
    ProgressBar.prototype.setWorked = function (value) {
        value = Math.max(1, Number(value));
        return this.doSetWorked(value);
    };
    ProgressBar.prototype.doSetWorked = function (value) {
        var totalWork = this.totalWork || 100;
        this.workedVal = value;
        this.workedVal = Math.min(totalWork, this.workedVal);
        this.element.classList.remove(CSS_INFINITE, CSS_DONE);
        this.element.classList.add(CSS_ACTIVE, CSS_DISCRETE);
        this.element.setAttribute('aria-valuenow', value.toString());
        this.bit.style.width = 100 * (this.workedVal / (totalWork)) + '%';
        return this;
    };
    ProgressBar.prototype.getContainer = function () {
        return this.element;
    };
    ProgressBar.prototype.show = function (delay) {
        this.showDelayedScheduler.cancel();
        if (typeof delay === 'number') {
            this.showDelayedScheduler.schedule(delay);
        }
        else {
            show(this.element);
        }
    };
    ProgressBar.prototype.hide = function () {
        hide(this.element);
        this.showDelayedScheduler.cancel();
    };
    ProgressBar.prototype.style = function (styles) {
        this.progressBarBackground = styles.progressBarBackground;
        this.applyStyles();
    };
    ProgressBar.prototype.applyStyles = function () {
        if (this.bit) {
            var background = this.progressBarBackground ? this.progressBarBackground.toString() : '';
            this.bit.style.backgroundColor = background;
        }
    };
    return ProgressBar;
}(Disposable));
export { ProgressBar };
