"use strict";
/********************************************************************************
 * Copyright (C) 2018 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var BlameDecorator_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppliedBlameDecorations = exports.BlameDecorator = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const browser_1 = require("@theia/editor/lib/browser");
const core_1 = require("@theia/core");
const moment = require("moment");
const uri_1 = require("@theia/core/lib/common/uri");
let BlameDecorator = BlameDecorator_1 = class BlameDecorator {
    constructor() {
        this.emptyHover = {
            contents: [{
                    value: ''
                }]
        };
        this.appliedDecorations = new Map();
        this.now = moment();
    }
    registerHoverProvider(uri) {
        return monaco.languages.registerHoverProvider([{ pattern: new uri_1.default(uri).path.toString() }], this);
    }
    async provideHover(model, position, token) {
        const line = position.lineNumber - 1;
        const uri = model.uri.toString();
        const applications = this.appliedDecorations.get(uri);
        if (!applications) {
            return this.emptyHover;
        }
        const blame = applications.blame;
        if (!blame) {
            return this.emptyHover;
        }
        const commitLine = blame.lines.find(l => l.line === line);
        if (!commitLine) {
            return this.emptyHover;
        }
        const sha = commitLine.sha;
        const commit = blame.commits.find(c => c.sha === sha);
        const date = new Date(commit.author.timestamp);
        let commitMessage = commit.summary + '\n' + (commit.body || '');
        commitMessage = commitMessage.replace(/[`\>\#\*\_\-\+]/g, '\\$&').replace(/\n/g, '  \n');
        const value = `${commit.sha}\n \n ${commit.author.name}, ${date.toString()}\n \n> ${commitMessage}`;
        const hover = {
            contents: [{ value }],
            range: monaco.Range.fromPositions(new monaco.Position(position.lineNumber, 1), new monaco.Position(position.lineNumber, 10 ^ 10))
        };
        return hover;
    }
    decorate(blame, editor, highlightLine) {
        const uri = editor.uri.toString();
        let applications = this.appliedDecorations.get(uri);
        if (!applications) {
            const that = applications = new AppliedBlameDecorations();
            this.appliedDecorations.set(uri, applications);
            applications.toDispose.push(this.registerHoverProvider(uri));
            applications.toDispose.push(core_1.Disposable.create(() => {
                this.appliedDecorations.delete(uri);
            }));
            applications.toDispose.push(core_1.Disposable.create(() => {
                editor.deltaDecorations({ oldDecorations: that.previousDecorations, newDecorations: [] });
            }));
        }
        if (applications.highlightedSha) {
            const sha = this.getShaForLine(blame, highlightLine);
            if (applications.highlightedSha === sha) {
                return applications;
            }
            applications.highlightedSha = sha;
        }
        const blameDecorations = this.toDecorations(blame, highlightLine);
        applications.previousStyles.dispose();
        applications.previousStyles.pushAll(blameDecorations.styles);
        const newDecorations = blameDecorations.editorDecorations;
        const oldDecorations = applications.previousDecorations;
        const appliedDecorations = editor.deltaDecorations({ oldDecorations, newDecorations });
        applications.previousDecorations.length = 0;
        applications.previousDecorations.push(...appliedDecorations);
        applications.blame = blame;
        return applications;
    }
    getShaForLine(blame, line) {
        const commitLines = blame.lines;
        const commitLine = commitLines.find(c => c.line === line);
        return commitLine ? commitLine.sha : undefined;
    }
    toDecorations(blame, highlightLine) {
        const beforeContentStyles = new Map();
        const commits = blame.commits;
        for (const commit of commits) {
            const sha = commit.sha;
            const commitTime = moment(commit.author.timestamp);
            const heat = this.getHeatColor(commitTime);
            const content = this.formatContentLine(commit, commitTime);
            const short = sha.substr(0, 7);
            const selector = 'git-' + short + '::before';
            beforeContentStyles.set(sha, new browser_1.EditorDecorationStyle(selector, style => {
                browser_1.EditorDecorationStyle.copyStyle(BlameDecorator_1.defaultGutterStyles, style);
                style.content = `'${content}'`;
                style.borderColor = heat;
            }));
        }
        const commitLines = blame.lines;
        const highlightedSha = this.getShaForLine(blame, highlightLine) || '';
        let previousLineSha = '';
        const editorDecorations = [];
        for (const commitLine of commitLines) {
            const { line, sha } = commitLine;
            const beforeContentClassName = beforeContentStyles.get(sha).className;
            const options = {
                beforeContentClassName,
            };
            if (sha === highlightedSha) {
                options.beforeContentClassName += ' ' + BlameDecorator_1.highlightStyle.className;
            }
            if (sha === previousLineSha) {
                options.beforeContentClassName += ' ' + BlameDecorator_1.continuationStyle.className;
            }
            previousLineSha = sha;
            const range = browser_1.Range.create(browser_1.Position.create(line, 0), browser_1.Position.create(line, 0));
            editorDecorations.push({ range, options });
        }
        const styles = [...beforeContentStyles.values()];
        return { editorDecorations, styles };
    }
    formatContentLine(commit, commitTime) {
        const when = commitTime.fromNow();
        const contentWidth = BlameDecorator_1.maxWidth - when.length - 2;
        let content = commit.summary.substring(0, contentWidth + 1);
        content = content.replace('\n', '↩︎').replace(/'/g, "\\'");
        if (content.length > contentWidth) {
            let cropAt = content.lastIndexOf(' ', contentWidth - 4);
            if (cropAt < contentWidth / 2) {
                cropAt = contentWidth - 3;
            }
            content = content.substring(0, cropAt) + '...';
        }
        if (content.length < contentWidth) {
            content = content + '\u2007'.repeat(contentWidth - content.length); // fill up with blanks
        }
        return `${content} ${when}`;
    }
    getHeatColor(commitTime) {
        const daysFromNow = this.now.diff(commitTime, 'days');
        if (daysFromNow <= 2) {
            return 'var(--md-orange-50)';
        }
        if (daysFromNow <= 5) {
            return 'var(--md-orange-100)';
        }
        if (daysFromNow <= 10) {
            return 'var(--md-orange-200)';
        }
        if (daysFromNow <= 15) {
            return 'var(--md-orange-300)';
        }
        if (daysFromNow <= 60) {
            return 'var(--md-orange-400)';
        }
        if (daysFromNow <= 180) {
            return 'var(--md-deep-orange-600)';
        }
        if (daysFromNow <= 365) {
            return 'var(--md-deep-orange-700)';
        }
        if (daysFromNow <= 720) {
            return 'var(--md-deep-orange-800)';
        }
        return 'var(--md-deep-orange-900)';
    }
};
__decorate([
    inversify_1.inject(browser_1.EditorManager),
    __metadata("design:type", browser_1.EditorManager)
], BlameDecorator.prototype, "editorManager", void 0);
BlameDecorator = BlameDecorator_1 = __decorate([
    inversify_1.injectable()
], BlameDecorator);
exports.BlameDecorator = BlameDecorator;
(function (BlameDecorator) {
    BlameDecorator.maxWidth = 50; // character
    BlameDecorator.defaultGutterStyles = {
        width: `${BlameDecorator.maxWidth}ch`,
        color: 'var(--theia-gitlens-gutterForegroundColor)',
        backgroundColor: 'var(--theia-gitlens-gutterBackgroundColor)',
        height: '100%',
        margin: '0 26px -1px 0',
        display: 'inline-block',
        borderRight: '2px solid',
    };
    BlameDecorator.continuationStyle = new browser_1.EditorDecorationStyle('git-blame-continuation-line::before', style => {
        style.content = "'\u2007'"; // blank
    });
    BlameDecorator.highlightStyle = new browser_1.EditorDecorationStyle('git-blame-highlight::before', style => {
        style.backgroundColor = 'var(--theia-gitlens-lineHighlightBackgroundColor)';
    });
})(BlameDecorator = exports.BlameDecorator || (exports.BlameDecorator = {}));
exports.BlameDecorator = BlameDecorator;
class AppliedBlameDecorations {
    constructor() {
        this.toDispose = new core_1.DisposableCollection();
        this.previousStyles = new core_1.DisposableCollection();
        this.previousDecorations = [];
    }
    dispose() {
        this.previousStyles.dispose();
        this.toDispose.dispose();
        this.blame = undefined;
    }
}
exports.AppliedBlameDecorations = AppliedBlameDecorations;
//# sourceMappingURL=blame-decorator.js.map