"use strict";
/********************************************************************************
 * Copyright (C) 2018 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var BreakpointManager_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BreakpointManager = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const common_1 = require("@theia/core/lib/common");
const browser_1 = require("@theia/core/lib/browser");
const marker_manager_1 = require("@theia/markers/lib/browser/marker-manager");
const uri_1 = require("@theia/core/lib/common/uri");
const breakpoint_marker_1 = require("./breakpoint-marker");
let BreakpointManager = BreakpointManager_1 = class BreakpointManager extends marker_manager_1.MarkerManager {
    constructor() {
        super(...arguments);
        this.owner = 'breakpoint';
        this.onDidChangeBreakpointsEmitter = new common_1.Emitter();
        this.onDidChangeBreakpoints = this.onDidChangeBreakpointsEmitter.event;
        this.onDidChangeFunctionBreakpointsEmitter = new common_1.Emitter();
        this.onDidChangeFunctionBreakpoints = this.onDidChangeFunctionBreakpointsEmitter.event;
        this._breakpointsEnabled = true;
        this.exceptionBreakpoints = new Map();
        this.functionBreakpoints = [];
    }
    getKind() {
        return breakpoint_marker_1.BREAKPOINT_KIND;
    }
    setMarkers(uri, owner, newMarkers) {
        const result = super.setMarkers(uri, owner, newMarkers);
        const added = [];
        const removed = [];
        const changed = [];
        const oldMarkers = new Map(result.map(({ data }) => [data.id, data]));
        const ids = new Set();
        for (const newMarker of newMarkers) {
            ids.add(newMarker.id);
            if (oldMarkers.has(newMarker.id)) {
                changed.push(newMarker);
            }
            else {
                added.push(newMarker);
            }
        }
        for (const [id, data] of oldMarkers.entries()) {
            if (!ids.has(id)) {
                removed.push(data);
            }
        }
        this.onDidChangeBreakpointsEmitter.fire({ uri, added, removed, changed });
        return result;
    }
    getLineBreakpoints(uri, line) {
        return this.findMarkers({
            uri,
            dataFilter: breakpoint => breakpoint.raw.line === line
        }).map(({ data }) => data);
    }
    getInlineBreakpoint(uri, line, column) {
        const marker = this.findMarkers({
            uri,
            dataFilter: breakpoint => breakpoint.raw.line === line && breakpoint.raw.column === column
        })[0];
        return marker && marker.data;
    }
    getBreakpoints(uri) {
        return this.findMarkers({ uri }).map(marker => marker.data);
    }
    setBreakpoints(uri, breakpoints) {
        this.setMarkers(uri, this.owner, breakpoints.sort((a, b) => (a.raw.line - b.raw.line) || ((a.raw.column || 0) - (b.raw.column || 0))));
    }
    addBreakpoint(breakpoint) {
        const uri = new uri_1.default(breakpoint.uri);
        const breakpoints = this.getBreakpoints(uri);
        const newBreakpoints = breakpoints.filter(({ raw }) => !(raw.line === breakpoint.raw.line && raw.column === breakpoint.raw.column));
        if (breakpoints.length === newBreakpoints.length) {
            newBreakpoints.push(breakpoint);
            this.setBreakpoints(uri, newBreakpoints);
            return true;
        }
        return false;
    }
    enableAllBreakpoints(enabled) {
        for (const uriString of this.getUris()) {
            let didChange = false;
            const uri = new uri_1.default(uriString);
            const markers = this.findMarkers({ uri });
            for (const marker of markers) {
                if (marker.data.enabled !== enabled) {
                    marker.data.enabled = enabled;
                    didChange = true;
                }
            }
            if (didChange) {
                this.fireOnDidChangeMarkers(uri);
            }
        }
        let didChangeFunction = false;
        for (const breakpoint of this.getFunctionBreakpoints()) {
            if (breakpoint.enabled !== enabled) {
                breakpoint.enabled = enabled;
                didChangeFunction = true;
            }
        }
        if (didChangeFunction) {
            this.fireOnDidChangeMarkers(BreakpointManager_1.FUNCTION_URI);
        }
    }
    get breakpointsEnabled() {
        return this._breakpointsEnabled;
    }
    set breakpointsEnabled(breakpointsEnabled) {
        if (this._breakpointsEnabled !== breakpointsEnabled) {
            this._breakpointsEnabled = breakpointsEnabled;
            for (const uri of this.getUris()) {
                this.fireOnDidChangeMarkers(new uri_1.default(uri));
            }
            this.fireOnDidChangeMarkers(BreakpointManager_1.FUNCTION_URI);
        }
    }
    getExceptionBreakpoint(filter) {
        return this.exceptionBreakpoints.get(filter);
    }
    getExceptionBreakpoints() {
        return this.exceptionBreakpoints.values();
    }
    setExceptionBreakpoints(exceptionBreakpoints) {
        const toRemove = new Set(this.exceptionBreakpoints.keys());
        for (const exceptionBreakpoint of exceptionBreakpoints) {
            const filter = exceptionBreakpoint.raw.filter;
            toRemove.delete(filter);
            this.exceptionBreakpoints.set(filter, exceptionBreakpoint);
        }
        for (const filter of toRemove) {
            this.exceptionBreakpoints.delete(filter);
        }
        if (toRemove.size || exceptionBreakpoints.length) {
            this.fireOnDidChangeMarkers(BreakpointManager_1.EXCEPTION_URI);
        }
    }
    toggleExceptionBreakpoint(filter) {
        const breakpoint = this.getExceptionBreakpoint(filter);
        if (breakpoint) {
            breakpoint.enabled = !breakpoint.enabled;
            this.fireOnDidChangeMarkers(BreakpointManager_1.EXCEPTION_URI);
        }
    }
    getFunctionBreakpoints() {
        return this.functionBreakpoints;
    }
    setFunctionBreakpoints(functionBreakpoints) {
        const oldBreakpoints = new Map(this.functionBreakpoints.map(b => [b.id, b]));
        this.functionBreakpoints = functionBreakpoints;
        this.fireOnDidChangeMarkers(BreakpointManager_1.FUNCTION_URI);
        const added = [];
        const removed = [];
        const changed = [];
        const ids = new Set();
        for (const newBreakpoint of functionBreakpoints) {
            ids.add(newBreakpoint.id);
            if (oldBreakpoints.has(newBreakpoint.id)) {
                changed.push(newBreakpoint);
            }
            else {
                added.push(newBreakpoint);
            }
        }
        for (const [id, breakpoint] of oldBreakpoints.entries()) {
            if (!ids.has(id)) {
                removed.push(breakpoint);
            }
        }
        this.onDidChangeFunctionBreakpointsEmitter.fire({ uri: BreakpointManager_1.FUNCTION_URI, added, removed, changed });
    }
    hasBreakpoints() {
        return !!this.getUris().next().value || !!this.functionBreakpoints.length;
    }
    removeBreakpoints() {
        this.cleanAllMarkers();
        this.setFunctionBreakpoints([]);
    }
    async load() {
        const data = await this.storage.getData('breakpoints', {
            breakpointsEnabled: true,
            breakpoints: {}
        });
        this._breakpointsEnabled = data.breakpointsEnabled;
        // eslint-disable-next-line guard-for-in
        for (const uri in data.breakpoints) {
            this.setBreakpoints(new uri_1.default(uri), data.breakpoints[uri]);
        }
        if (data.functionBreakpoints) {
            this.setFunctionBreakpoints(data.functionBreakpoints);
        }
        if (data.exceptionBreakpoints) {
            this.setExceptionBreakpoints(data.exceptionBreakpoints);
        }
    }
    save() {
        const data = {
            breakpointsEnabled: this._breakpointsEnabled,
            breakpoints: {}
        };
        const uris = this.getUris();
        for (const uri of uris) {
            data.breakpoints[uri] = this.findMarkers({ uri: new uri_1.default(uri) }).map(marker => marker.data);
        }
        if (this.functionBreakpoints.length) {
            data.functionBreakpoints = this.functionBreakpoints;
        }
        if (this.exceptionBreakpoints.size) {
            data.exceptionBreakpoints = [...this.exceptionBreakpoints.values()];
        }
        this.storage.setData('breakpoints', data);
    }
};
BreakpointManager.EXCEPTION_URI = new uri_1.default('debug:exception://');
BreakpointManager.FUNCTION_URI = new uri_1.default('debug:function://');
__decorate([
    inversify_1.inject(browser_1.StorageService),
    __metadata("design:type", Object)
], BreakpointManager.prototype, "storage", void 0);
BreakpointManager = BreakpointManager_1 = __decorate([
    inversify_1.injectable()
], BreakpointManager);
exports.BreakpointManager = BreakpointManager;
//# sourceMappingURL=breakpoint-manager.js.map